from collections import Counter
from project.src.webapp_utils import (generate_quasar_table_column_settings,
                                      prettify_proba,
                                      flag_proba_for_webapp)

def load_user_recommendation_legend_data(user_id,
                                                user_proba_labels,
                                                sorted_probability_labels,
                                                probability_label_colors,
                                                probability_label_helpers
                                               ):
    user_recommendation_legend_section_ids = []
    user_recommendation_legend_section_information = {}
    user_proba_labels_counts = dict(Counter(user_proba_labels))
    for probability_label_index, probability_label in enumerate(sorted_probability_labels):
        if probability_label in user_proba_labels_counts.keys():
            user_probability_label_occurences = user_proba_labels_counts[probability_label]
            section_id = "user_{}_section_{}".format(user_id, probability_label_index)
            user_recommendation_legend_section_ids.append(section_id)
            user_recommendation_legend_section_information[section_id] = {
                "section_id": section_id
            }
            user_recommendation_legend_section_information[section_id]["section_label"] = "{} ({})".format(probability_label, user_probability_label_occurences)
            user_recommendation_legend_section_information[section_id]["color"] = probability_label_colors[probability_label]
            user_recommendation_legend_section_information[section_id]["has_tooltip"] = True
            user_recommendation_legend_section_information[section_id]["tooltip_text"] = probability_label_helpers[probability_label]
    
    return user_recommendation_legend_section_ids, user_recommendation_legend_section_information


def load_user_recommendation_data(results_display_style, user_id, list_of_item_ids, leverage_item_metadata,
                        item_metadata_columns, item_metadata_mapping, user_recommendations_data, 
                        deployed_model_threshold, items_scope,
                        probability_labels, probability_label_colors, probability_label_helpers):
    if results_display_style == "Pictures view":
        user_item_recommendation_data, user_recommendation_legend_section_ids, user_recommendation_legend_section_information = load_user_recommendation_data_for_pictures(
                                            user_id,
                                            list_of_item_ids,
                                            leverage_item_metadata,
                                            item_metadata_columns,
                                            item_metadata_mapping,
                                            user_recommendations_data,
                                            deployed_model_threshold,
                                            items_scope,
                                            probability_labels,
                                            probability_label_colors,
                                            probability_label_helpers)
    else:
        user_item_recommendation_data, user_recommendation_legend_section_ids, user_recommendation_legend_section_information = load_user_recommendation_data_for_tables(user_id,
                                           list_of_item_ids,
                                           leverage_item_metadata,
                                           item_metadata_columns,
                                           item_metadata_mapping,
                                           user_recommendations_data,
                                           deployed_model_threshold,
                                           items_scope,
                                           probability_labels,
                                            probability_label_colors,
                                            probability_label_helpers)
    return user_item_recommendation_data, user_recommendation_legend_section_ids, user_recommendation_legend_section_information


def load_user_recommendation_data_for_tables(
    user_id,
    list_of_item_ids,
    leverage_item_metadata,
    item_metadata_columns,
    item_metadata_mapping,
    user_recommendations_data,
    deployed_model_threshold,
    items_scope,
    probability_labels,
    probability_label_colors,
    probability_label_helpers
):
    user_item_recommendation_data = {}
    user_proba_labels = []
    table_columns_data = [generate_quasar_table_column_settings("item_id")]
    if items_scope == "recommendation_items":
        table_columns_data.append(
            generate_quasar_table_column_settings("recommendation_proba")
        )
    if leverage_item_metadata:
        for column_name in item_metadata_columns:
            table_columns_data.append(
                generate_quasar_table_column_settings(column_name)
            )
            pass
    
    table_rows_data = []
    for item_id in list_of_item_ids:
        item_data = {
            "item_id": item_id,
        }

        if leverage_item_metadata:
            for column_name in item_metadata_columns:
                item_data[column_name] = item_metadata_mapping[item_id][column_name]

        if items_scope == "recommendation_items":
            item_data["has_label"] = True
            item_data["has_gauge"] = True
            recommendation_proba = user_recommendations_data[user_id]["recommendation_probas"][item_id]
            item_data["recommendation_proba"] = prettify_proba(recommendation_proba)
            webapp_quasar_color, proba_label = flag_proba_for_webapp(recommendation_proba, deployed_model_threshold)
            user_proba_labels.append(proba_label)
            user_recommendation_legend_section_ids, user_recommendation_legend_section_information = load_user_recommendation_legend_data(user_id,
                                                user_proba_labels,
                                                probability_labels,
                                                probability_label_colors,
                                                probability_label_helpers
                                               )
            item_data["item_label_color"] = webapp_quasar_color
        table_rows_data.append(item_data)
    user_item_recommendation_data["columns"] = table_columns_data
    user_item_recommendation_data["rows"] = table_rows_data

    if items_scope == "past_interaction_items":
        user_recommendation_legend_section_ids = None
        user_recommendation_legend_section_information = None
    return user_item_recommendation_data, user_recommendation_legend_section_ids, user_recommendation_legend_section_information


def load_user_recommendation_data_for_pictures(
    user_id,
    list_of_item_ids,
    leverage_item_metadata,
    item_metadata_columns,
    item_metadata_mapping,
    user_recommendations_data,
    deployed_model_threshold,
    items_scope,
    probability_labels,
    probability_label_colors,
    probability_label_helpers):
    user_item_recommendation_data = []
    user_proba_labels = []
    for item_id in list_of_item_ids:
        item_data = {"picture_id": item_id, "metadata": {}}
        if leverage_item_metadata:
            item_data["metadata"]["variables"] = item_metadata_columns
            item_data["metadata"]["content"] = item_metadata_mapping[item_id]

        else:
            item_data["metadata"]["variables"] = []
            item_data["metadata"]["content"] = {}

        if items_scope == "past_interaction_items":
            item_data["item_has_proba"] = False

        elif items_scope == "recommendation_items":
            item_data["picture_has_label"] = True
            item_data["picture_has_gauge"] = True
            recommendation_proba = user_recommendations_data[user_id]["recommendation_probas"][item_id]
            item_data["picture_label"] = prettify_proba(recommendation_proba)
            webapp_quasar_color, proba_label = flag_proba_for_webapp(recommendation_proba, deployed_model_threshold)
            user_proba_labels.append(proba_label)
            user_recommendation_legend_section_ids, user_recommendation_legend_section_information = load_user_recommendation_legend_data(user_id,
                                                user_proba_labels,
                                                probability_labels,
                                                probability_label_colors,
                                                probability_label_helpers
                                               )
            item_data["picture_label_color"] = webapp_quasar_color
            item_data["picture_gauge_value"] = recommendation_proba
            item_data["picture_gauge_color"] = webapp_quasar_color

        user_item_recommendation_data.append(item_data)
    if items_scope == "past_interaction_items":
        user_recommendation_legend_section_ids = None
        user_recommendation_legend_section_information = None
    return user_item_recommendation_data, user_recommendation_legend_section_ids, user_recommendation_legend_section_information