import re
import dataikuapi
from ..flow.constants import (STARTER_SCENARIOS_TO_BUILD_ALL_FLOW,
                              STARTER_SCENARIOS_TO_PREPROCESS_INPUTS,
                              STARTER_SCENARIOS_FOR_ITEMS_ANALYSIS_AND_SELECTION)
from ....dku_utils.datasets.dataset_commons import (get_dataset_schema,
                                                    extract_dataset_schema_information)
from ....dku_utils.flow.flow_commons import move_recipe_in_flow_zone, get_all_flow_recipe_names

########################################################################################################################
############################################ DATA INGESTION ############################################################
########################################################################################################################
def get_list_of_datasets_to_load(variables):
    app_variables = variables["standard"]
    DATASETS_TO_LOAD = ["interactions_history"]
    leverage_user_metadata = app_variables["leverage_user_metadata_app"]
    if leverage_user_metadata:
        DATASETS_TO_LOAD.append("user_metadata")
    leverage_item_metadata = app_variables["leverage_item_metadata_app"]
    if leverage_item_metadata:
        DATASETS_TO_LOAD.append("item_metadata")
    return DATASETS_TO_LOAD


def get_list_of_folders_to_load(variables):
    app_variables = variables["standard"]
    FOLDERS_TO_LOAD = []
    use_product_pictures = app_variables["use_product_pictures_app"]
    if use_product_pictures:
        FOLDERS_TO_LOAD.append("item_pictures")
    return FOLDERS_TO_LOAD


########################################################################################################################
############################################## SCENARIOS ###############################################################
########################################################################################################################
def load_scenarios_to_build_all_flow(variables):
    app_variables = variables["standard"]
    scenarios_to_build_all_flow = STARTER_SCENARIOS_TO_BUILD_ALL_FLOW
    leverage_user_metadata = app_variables["leverage_user_metadata_app"]
    user_age_available = app_variables["user_age_available_app"]
    if leverage_user_metadata:
        if user_age_available:
            scenarios_to_build_all_flow.append("BUILD_AGES_CLUSTERING")
            pass
        pass
    scenarios_to_build_all_flow.append("BUILD_RECOMMENDATION_SPLIT")
    scenarios_to_build_all_flow.append("APPLY_FLOW_FEATURE_ENGINEERING")
    scenarios_to_build_all_flow.append("BUILD_FEATURES_GATHERING")
    scenarios_to_build_all_flow.append("BUILD_NEGATIVE_SAMPLING")
    scenarios_to_build_all_flow.append("BUILD_PRODUCT_RECOMMENDATIONS")
    scenarios_to_build_all_flow.append("BUILD_RECOMMENDATION_FUNNELS")
    scenarios_to_build_all_flow.append("BUILD_WEBAPP_ZONE")
    scenarios_to_build_all_flow.append("REFRESH_DASHBOARD_AND_WEBAPP")
    return scenarios_to_build_all_flow
    

def load_optional_feature_engineering_scenarios(variables):
    app_variables = variables["standard"]
    feature_engineering_scenarios = []
    # Handling scenario depending on 'age' variable:
    leverage_user_metadata = app_variables["leverage_user_metadata_app"]
    user_age_available = app_variables["user_age_available_app"]
    if leverage_user_metadata:
        if user_age_available:
            feature_engineering_scenarios.append("BUILD_AGE_FEATURE_ENGINEERING")
            pass
        pass
    # Handling scenario depending on contextual collaborative filtering settings:
    leverage_item_metadata = app_variables["leverage_item_metadata_app"]
    contextual_collaborative_filtering_columns = app_variables["contextual_collaborative_filtering_columns_app"]
    item_metadata_handling = app_variables["item_metadata_handling_app"]
    contextual_collaborative_filtering_is_required = \
    (leverage_item_metadata) and ("collaborative_filtering" in item_metadata_handling)\
    and (len(contextual_collaborative_filtering_columns) > 0)
    if contextual_collaborative_filtering_is_required:
        feature_engineering_scenarios.append("BUILD_COLLABORATIVE_FILTERING_ITEM_CHARACTERISTICS")
        pass
    # Main collaborative filtering scenario:
    feature_engineering_scenarios.append("REMOVE_NOT_USED_COLLABORATIVE_FILTERING_ITEM_CHARACTERISTICS")
    feature_engineering_scenarios.append("BUILD_COLLABORATIVE_FILTERING_ITEMS")
    return feature_engineering_scenarios


def load_inputs_preprocessing_scenarios(variables):
    app_variables = variables["standard"]
    leverage_user_metadata = app_variables["leverage_user_metadata_app"]
    leverage_item_metadata = app_variables["leverage_item_metadata_app"]
    use_product_pictures = app_variables["use_product_pictures_app"]
    inputs_preprocessing_scenarios = STARTER_SCENARIOS_TO_PREPROCESS_INPUTS
    if leverage_user_metadata:
        inputs_preprocessing_scenarios.append("PREPROCESS_USER_METADATA")
    if leverage_item_metadata:
        inputs_preprocessing_scenarios.append("PREPROCESS_ITEM_METADATA")
    if use_product_pictures:
        inputs_preprocessing_scenarios.append("PREPROCESS_ITEM_PICTURES")
    return inputs_preprocessing_scenarios


def load_items_analysis_and_selection_scenarios(variables):
    app_variables = variables["standard"]
    items_revenue_available = app_variables["items_revenue_available_app"]
    items_analysis_and_selection_scenarios = STARTER_SCENARIOS_FOR_ITEMS_ANALYSIS_AND_SELECTION
    if items_revenue_available:
        items_analysis_and_selection_scenarios.append("COMPUTE_ITEM_REVENUE_DATA")
    return items_analysis_and_selection_scenarios


########################################################################################################################
######################### FLOW/FEATURES GATHERING/ RECIPE 'compute_all_affinities_and_features' ########################
########################################################################################################################
def handle_join_recipe_compute_all_affinities_and_features(project, variables):
    RECIPE_NAME = "compute_all_affinities_and_features"
    RECIPE_INPUT_DATASET = "user_item_metadata_and_affinities"
    RECIPE_OUTPUT_DATASET = "all_affinities_and_features"
    RECIPE_BELONGING_FLOW_ZONE = "features_gathering"
    
    app_variables = variables["standard"]
    main_connection = app_variables["main_connection_app"]
    leverage_item_metadata = app_variables["leverage_item_metadata_app"]
    leverage_user_metadata = app_variables["leverage_user_metadata_app"]
    
    all_flow_recipe_names = get_all_flow_recipe_names(project)
    recipe_exists = (RECIPE_NAME in all_flow_recipe_names)
    
    if leverage_item_metadata or leverage_user_metadata:
        if not recipe_exists:
            builder = dataikuapi.JoinRecipeCreator(RECIPE_NAME, project)
            builder.with_input(RECIPE_INPUT_DATASET)
            builder.with_output(RECIPE_OUTPUT_DATASET)
            builder.build()
        move_recipe_in_flow_zone(project, RECIPE_NAME, RECIPE_BELONGING_FLOW_ZONE)
    else:
        if recipe_exists:
            project.get_recipe(RECIPE_NAME).delete()
    pass


########################################################################################################################
#################################### FLOW/COLLABORATIVE FILTERING ######################################################
########################################################################################################################
def load_contextual_collaborative_filtering_feature_datasets(contextual_collaborative_filtering_feature_name):
    contextual_collaborative_filtering_flow_branch_dataset_names = {}
    all_interactions_output = "_{}_affinities".format(contextual_collaborative_filtering_feature_name)
    engineering_output = "_{}_affinities_for_learning".format(contextual_collaborative_filtering_feature_name)
    stack_recipe_output = "_all_{}_affinities".format(contextual_collaborative_filtering_feature_name)
    group_recipe_output = "_{}_min_max_scores".format(contextual_collaborative_filtering_feature_name) 
    join_recipe_output = "_all_{}_affinities_enriched".format(contextual_collaborative_filtering_feature_name)
    feature_datasets = [all_interactions_output, engineering_output, stack_recipe_output, group_recipe_output, join_recipe_output]
    return feature_datasets


def load_computed_collaborative_filtering_flow_branches(project):
    project_datasets = [dataset_information["name"] for dataset_information in project.list_datasets()]
    computed_collaborative_filtering_flow_branches = []
    for dataset_name in project_datasets:
        if ("_all_" in dataset_name) and ("_affinities_enriched" in dataset_name):
            dataset_contextual_collaborative_filtering_flow_branch =\
            re.sub("_all_|_affinities_enriched", "", dataset_name)
            computed_collaborative_filtering_flow_branches.append(dataset_contextual_collaborative_filtering_flow_branch)
    return computed_collaborative_filtering_flow_branches


def load_collaborative_filtering_optional_scores(project, variables):
    app_variables = variables["standard"]
    collaborative_filtering_optional_scores = []
    contextual_collaborative_filtering_columns = app_variables["contextual_collaborative_filtering_columns_app"]
    for feature_name in contextual_collaborative_filtering_columns:
        collaborative_filtering_optional_scores.append("item_{}_score".format(feature_name))
        collaborative_filtering_optional_scores.append("item_{}_score_max".format(feature_name))
        collaborative_filtering_optional_scores.append("item_{}_score_min".format(feature_name))
        collaborative_filtering_optional_scores.append("item_{}_score_scaled".format(feature_name))
    leverage_user_metadata = app_variables["leverage_user_metadata_app"]
    user_age_available = app_variables["user_age_available_app"]
    if leverage_user_metadata:
        if user_age_available:
            user_age_columns = ["user_age"]
            all_affinities_and_features_prepared_columns_data =\
            project.get_dataset("all_affinities_and_features_prepared")\
            .get_settings().settings["schema"]["columns"]
            all_affinities_and_features_prepared_columns = [column_data["name"] 
                                                            for column_data in
                                                            all_affinities_and_features_prepared_columns_data]
            for column_name in all_affinities_and_features_prepared_columns:
                if "item_user_age" in column_name:
                    user_age_columns.append(column_name)
                    pass
                pass
            user_age_columns.append("delta_user_age_to_average")
            collaborative_filtering_optional_scores += user_age_columns
    return collaborative_filtering_optional_scores


########################################################################################################################
############################################### AUTO ML ################################################################
########################################################################################################################
def load_model_features(project, variables):
    model_enrichment_features_to_select = ["item_score_scaled"]
    app_variables = variables["standard"]
    leverage_item_metadata = app_variables["leverage_item_metadata_app"]
    item_metadata_handling = app_variables["item_metadata_handling_app"]
    contextual_collaborative_filtering_columns = app_variables["contextual_collaborative_filtering_columns_app"]
    contextual_collaborative_filtering_columns_renamed =\
    ["item_{}".format(column_name) for column_name in contextual_collaborative_filtering_columns]
    leverage_user_metadata = app_variables["leverage_user_metadata_app"]
    available_user_metadata = app_variables["available_user_metadata_app"]
    if leverage_item_metadata:
        item_metadata_dataset_schema = get_dataset_schema(project, "item_metadata_prepared")
        item_metadata_dataset_columns, __ = extract_dataset_schema_information(item_metadata_dataset_schema)
        for feature_name in item_metadata_dataset_columns:
            if feature_name not in ["item_id", "item_picture"]:
                if "contextual_collaborative_filtering" in item_metadata_handling:
                    if feature_name in contextual_collaborative_filtering_columns_renamed:
                        model_enrichment_features_to_select.append("{}_score_scaled".format(feature_name))
                        pass
                if "model_features_enrichment" in item_metadata_handling:
                    if feature_name not in model_enrichment_features_to_select:
                        model_enrichment_features_to_select.append(feature_name)
    if leverage_user_metadata: 
        user_metadata_dataset_schema = get_dataset_schema(project, "user_metadata_prepared")
        user_metadata_dataset_columns, __ = extract_dataset_schema_information(user_metadata_dataset_schema)
        for feature_name in user_metadata_dataset_columns:
            if feature_name not in ["user_id", "user_age"]:
                if "other_user_metadata" in available_user_metadata:
                    model_enrichment_features_to_select.append(feature_name)
                    pass
                pass
            pass
        pass
        if "user_age" in available_user_metadata:
            for feature_name in ["user_age", "delta_user_age_to_average", "user_age_cluster"]:
                model_enrichment_features_to_select.append(feature_name)
            recommendation_modeling_dataset_schema = get_dataset_schema(project, "final_machine_learning_set_prepared")
            recommendation_modeling_dataset_columns, __ =\
            extract_dataset_schema_information(recommendation_modeling_dataset_schema)
            for feature_name in recommendation_modeling_dataset_columns:
                if "item_user_age" in feature_name:
                    model_enrichment_features_to_select.append(feature_name)
                    pass
                pass
            pass
        pass
    return model_enrichment_features_to_select


def load_sub_population_analysis_features(dataset_columns, dataset_column_datatypes, forbidden_features):
    sub_population_analysis_features = []
    for column_name, column_datatype in zip(dataset_columns, dataset_column_datatypes):
        if (column_name not in forbidden_features) and (column_datatype == 'string'):
            sub_population_analysis_features.append(column_name)         
    return sub_population_analysis_features


def load_partial_dependencies_features(dataset_columns, forbidden_features):
    partial_dependencies_features = []
    for column_name in dataset_columns:
        if column_name not in forbidden_features:
            partial_dependencies_features.append(column_name)
    return partial_dependencies_features