import numpy as np
from ..flow.constants import (N_NEIGHBORS_STATIONS_TO_SEARCH,
                              N_NEIGHBORS_STATIONS_TO_RETRIEVE,
                              SORT_NEIGHBORS_BY_GEODESIC_DISTANCES)
from ..geographic_handling.formating.points import read_geo_point
from ..geographic_handling.feature_engineering.information_extraction import get_geodesic_distance


def concatenate_list_values(list_):
    return ",".join(str(value) for value in list_)


def get_station_feature_information(feature, stations_metadata, neighbor_stations_indexes):
    return [stations_metadata[index][feature] for index in neighbor_stations_indexes]


def get_average_distance_from_list_of_geo_points(reference_list_geo_point, list_of_dku_geo_points):
    distances_from_reference_geo_point = []
    for dku_geo_point in list_of_dku_geo_points:
        list_geo_point = read_geo_point(dku_geo_point, True)
        distance_from_reference_geo_point = get_geodesic_distance(reference_list_geo_point,
                                                                  list_geo_point,
                                                                  False)
        distances_from_reference_geo_point.append(distance_from_reference_geo_point)
    return np.mean(distances_from_reference_geo_point)


def lookup_stations_features(reference_geo_point, stations_metadata, subway_stations_indexer):
    addresses_neighboring_stations = \
    subway_stations_indexer.search_single_geo_point_neighbors(reference_geo_point, N_NEIGHBORS_STATIONS_TO_SEARCH,
                                                              N_NEIGHBORS_STATIONS_TO_RETRIEVE, SORT_NEIGHBORS_BY_GEODESIC_DISTANCES)
    neighbor_stations_information = {}
    STATIONS_COLUMNS_TO_RETRIEVE = ["is_train_station", "line_terminus", "line_transfers", "station_closeness_centrality",
                                    "station_degree", "station_eigenvector_centrality", "station_geo_point",
                                    "station_name_normalized", "station_pagerank"]
    for column in STATIONS_COLUMNS_TO_RETRIEVE:
        neighbor_stations_information[column] = get_station_feature_information(column,
                                                                                stations_metadata,
                                                                                addresses_neighboring_stations)
    
    station_features = {}
    # Closest station features:
    station_features["closest_station_station_degree"] = neighbor_stations_information["station_degree"][0]
    station_features["closest_station_station_closeness_centrality"] = neighbor_stations_information["station_closeness_centrality"][0]
    station_features["closest_station_station_pagerank"] = neighbor_stations_information["station_pagerank"][0]
    station_features["closest_station_station_eigenvector_centrality"] = neighbor_stations_information["station_eigenvector_centrality"][0]
    station_features["closest_station_name"] = neighbor_stations_information["station_name_normalized"][0]
    station_0_geo_point = read_geo_point(neighbor_stations_information["station_geo_point"][0], True)
    station_features["distance_from_station_0"] = get_geodesic_distance(reference_geo_point, station_0_geo_point, False)


    # Averages station features:
    station_features["close_stations_average_station_degree"] = np.mean(neighbor_stations_information["station_degree"])
    station_features["close_stations_average_station_closeness_centrality"] = np.mean(neighbor_stations_information["station_closeness_centrality"])
    station_features["close_stations_average_station_pagerank"] = np.mean(neighbor_stations_information["station_pagerank"])
    station_features["close_stations_average_station_eigenvector_centrality"] = np.mean(neighbor_stations_information["station_eigenvector_centrality"])

    # Geospatial computation station features: 
    station_features["average_distance_from_closest_stations"] = get_average_distance_from_list_of_geo_points(reference_geo_point, neighbor_stations_information["station_geo_point"])

    # Concatenations station features:
    station_features["close_stations"] = concatenate_list_values(neighbor_stations_information["station_name_normalized"])
    station_features["close_train_station"] = concatenate_list_values(neighbor_stations_information["is_train_station"])
    station_features["close_lines"] = concatenate_list_values(neighbor_stations_information["line_transfers"])
    station_features["close_line_terminus"] = concatenate_list_values(neighbor_stations_information["line_terminus"])
    return station_features