# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
import dataiku
from dataiku import pandasutils as pdu
import pandas as pd
import requests
import json

# custom function that can be found within Libraries tab > G+L
from census_api_functions import get_project_variables, get_query_text, state_name_list, get_tracts_code_table

# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
# variable specifications
census_api_key = get_project_variables('standard','api_key')
# census API variables specifications
census_code = 'S2503'
year = 2022

# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
# url path to call census API services
url_path = f"https://api.census.gov/data/{year}/acs/acs5/profile?get=NAME&for=state:*&key={census_api_key}"

# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
# numerical list of US states
state_nums_list = state_name_list(url_path)

# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
# API request to gather the dataset from U.S. Census Bureau
all_tracts_df = pd.DataFrame()

for state in state_nums_list:
    print(f"Processing state: {state}")
    try:
        # Construct API URL
        state_all_tracts_query_url = (
            f"https://api.census.gov/data/{year}/acs/acs5/subject"
            f"?get=NAME,group({census_code})&for=tract:*&in=state:{state}&key={census_api_key}"
        )
        # Fetch the data using requests
        response = requests.get(state_all_tracts_query_url)
        response.raise_for_status()  # Raise exception for HTTP errors
        
        # Parse the JSON response
        state_all_tract_names_query_result_list = response.json()
        
        # Convert JSON to DataFrame
        state_all_tract_names_df = pd.DataFrame(
            state_all_tract_names_query_result_list[1:], 
            columns=state_all_tract_names_query_result_list[0]
        )
        
        # Exclude NAME column (first column)
        df = state_all_tract_names_df.iloc[:, 1:]
        
        # Concatenate to main DataFrame
        all_tracts_df = pd.concat([all_tracts_df, df], ignore_index=True)
        
    except requests.exceptions.RequestException as req_err:
        print(f"HTTP error for state {state}: {req_err}")
    except json.JSONDecodeError as json_err:
        print(f"JSON decode error for state {state}: {json_err}")
    except Exception as e:
        print(f"Unexpected error for state {state}: {e}")


# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
# data preprocessing: column mapping, feature generation
all_tracts_df = all_tracts_df[['GEO_ID', 'S2503_C01_028E', 'S2503_C01_032E', 'S2503_C01_036E', 'S2503_C01_040E', 'S2503_C01_001E']]
all_tracts_df[['S2503_C01_028E', 'S2503_C01_032E', 'S2503_C01_036E', 'S2503_C01_040E', 'S2503_C01_001E']] = all_tracts_df[['S2503_C01_028E', 'S2503_C01_032E', 'S2503_C01_036E', 'S2503_C01_040E', 'S2503_C01_001E']].astype(float)
all_tracts_df['E_HBURD'] = all_tracts_df['S2503_C01_028E'] + all_tracts_df['S2503_C01_032E'] + all_tracts_df['S2503_C01_036E'] + all_tracts_df['S2503_C01_040E']

# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
try:
    all_tracts_df['EP_HBURD'] = (all_tracts_df['E_HBURD'] / all_tracts_df['S2503_C01_001E']) * 100
except ZeroDivisionError:
    all_tracts_df['EP_HBURD'] = 0

# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
all_tracts_df = all_tracts_df.drop(['S2503_C01_028E', 'S2503_C01_032E', 'S2503_C01_036E', 'S2503_C01_040E', 'S2503_C01_001E'], axis = 1)

# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
# Write recipe outputs

S2503_all_tracts = dataiku.Dataset("S2503_svi_tracts")
S2503_all_tracts.write_with_schema(all_tracts_df)