import sys, json, os, socket, time, logging, traceback, signal, tarfile, glob, random, string, zipfile, re, subprocess, base64, tempfile, shutil, hashlib
import numpy as np
import threading
import requests
from dataiku.core.intercom import backend_api_post_call, backend_json_call, backend_stream_call, backend_void_call
from dataiku.base import remoterun
from dataiku.base.utils import get_json_friendly_error

COMMITID_HIDDEN_FILE_NAME = ".dku_current_commit_id"
CHANGES_HIDDEN_FILE_NAME = ".dku_changes_lists"
FETCH_FILE_NAME = "_dku_fetch.zip"

# utility to replicate the SmartLogTail we have in java, but in python. The point is
# to not have to forward logs to the java backend and do the log tail there
class TailerThread(threading.Thread):
    def __init__(self, input, output):
        super(TailerThread, self).__init__()
        self.input = input
        if hasattr(output, 'buffer'):
            self.output = output.buffer # for python3
        else:
            self.output = output
        self.tail = b''
        self.tail_size = 4196
        
    def run(self):
        while True:
            b = self.input.read(self.tail_size)
            if b is None:
                continue # non-blocking and nothing to read
            if len(b) == 0:
                break # eof
            # replicate on output
            self.output.write(b)
            self.output.flush()
            # tail the stream (in bytes, will convert to text at the end)
            b = b''.join([self.tail, b])
            if len(b) > self.tail_size:
                b = b[-self.tail_size:]
            self.tail = b
                
    def get_tail(self): 
        lines = self.tail.split(b'\n')
        if len(self.tail) >= self.tail_size:
            lines = lines[1:] # skip first line because it's likely incomplete
        return b'\n'.join(lines).decode('utf8')
        
# handling of the sync exclusion patterns, which are using the gitignore syntax (more or
# less, the "rule-negation" thing notably isn't done)
def translate_gitignore_to_pattern(s):
    match_dir = False
    while s.endswith('/'):
        match_dir = True
        s = s[:-1]
    anchored = '/' in s
    while s.startswith('/'):
        s = s[1:]
    starts_from_anywhere = False
    if s.startswith('**/'):
        starts_from_anywhere = True
        s = s[3:]
    s = s.replace('/**/', '[_dku_any_number_of_subdirs_]')
    if s.endswith('/**'):
        s = s[:-3] + '[_dku_anything_under_]'
    s = s.replace('**', '[_dku_anything_anything_]') # you shouldn't have these guys
    s = s.replace('*', '[_dku_any_whatever_but_slash_]')
    s = s.replace('?', '[_dku_one_whatever_but_slash_]')
    
    s = s.replace('[_dku_one_whatever_but_slash_]', '[^/]')
    s = s.replace('[_dku_any_whatever_but_slash_]', '[^/]*')
    s = s.replace('[_dku_any_number_of_subdirs_]', '/([^/]+/)*')
    s = s.replace('[_dku_anything_anything_]', '.*')
    s = s.replace('[_dku_anything_under_]', '/.*')
    s = s.replace('[_dku_anywhere_under_]', '.*/')
    if anchored and not starts_from_anywhere:
        s = '^/?' + s
    else:
        s = '^([^/]*/)*' + s
    if match_dir:
        s = s + '(/.*)?'
    return s + '$'

class GitignorePatternMatch(object):
    def __init__(self, excluded_from_sync=[]):
        self.patterns = [re.compile(translate_gitignore_to_pattern(e)) for e in excluded_from_sync]
    
    def matches(self, path):
        for p in self.patterns:
            if p.match(path) is not None:
                return True
        return False

# utility to copy a directory over an existing one, without replacing
# unchanged files/folders. The reason being that IDEs will notice if a file
# was overwritten (even if the contents are the same), so you risk seeing
# alerts of 'uh oh this file was changed, want to reload it' everywhere
# after a synchro with the DSS files
def get_sha256_of_file(file_path):
    h = hashlib.sha256()
    with open(file_path, 'rb') as fd:
        while True:
            c = fd.read(h.block_size)
            if not c:
                break
            h.update(c)
    return h.hexdigest()

def get_dir_contents(path):
    dirs = set()
    files = set()
    for n in os.listdir(path):
        e = os.path.join(path, n)
        if os.path.islink(e):
            continue
        elif os.path.isfile(e):
            files.add(n)
        elif os.path.isdir(e):
            dirs.add(n)     
    return dirs, files

def delete_file_or_dir(path):
    if os.path.exists(path):
        if os.path.isdir(path):
            shutil.rmtree(path)
        else:
            os.remove(path)
            
def copy_unless_same(src, dst):
    logging.info("Copy dir %s -> %s" % (src, dst))
    if os.path.exists(src):
        # make sure it's directories on both sides
        if not os.path.exists(dst):
            logging.info("New folder %s" % dst)
            os.makedirs(dst)
        # list contents
        src_dirs, src_files = get_dir_contents(src)
        dst_dirs, dst_files = get_dir_contents(dst)
        # delete what is gone or has changed type
        for name in (dst_files - src_files):
            if name in src_dirs:
                logging.info("Changed file to dir %s" % name)
            else:
                logging.info("Deleted file %s" % name)
            delete_file_or_dir(os.path.join(dst, name))
        for name in (dst_dirs - src_dirs):
            if name in src_files:
                logging.info("Changed dir to file %s" % name)
            else:
                logging.info("Deleted dir %s" % name)
            delete_file_or_dir(os.path.join(dst, name))
        # compare files
        for name in src_files:
            src_file = os.path.join(src, name)
            dst_file = os.path.join(dst, name)

            is_same = False
            if name in dst_files and os.path.getsize(src_file) == os.path.getsize(dst_file):
                # check if the files are the same, the lazy way, with a hash
                is_same = get_sha256_of_file(src_file) == get_sha256_of_file(dst_file)
            if is_same:
                logging.info("Unchanged file %s" % dst_file)
            else:
                logging.info("New or modified file %s" % dst_file)
                shutil.copy2(src_file, dst_file) # note: follow_symlinks=False is python3 only, but we don't need it here
            
        # recurse in directories
        for name in src_dirs:
            src_dir = os.path.join(src, name)
            dst_dir = os.path.join(dst, name)
            copy_unless_same(src_dir, dst_dir)
            
    else:
        logging.info("Delete dir %s" % dst)
        delete_file_or_dir(dst)
        
# dumb 'summary' of folder, to get a feeling of how big it is
def summarize_contents(path):
    file_count = 0
    file_size = 0
    for root, dirs, files in os.walk(path):
        for file in files:
            s = os.path.getsize(os.path.join(root, file))
            file_count += 1
            file_size += s
    return {"count":file_count, "size":file_size}
    
def run_adjustment_script_on_folder(zone, direction, script, folder):
    # catch errors because adjustments should be a "nice-to-have" at most, but not
    # a reason to fail the file sync
    logging.info("Adjust files in %s" % folder)
    try:
        subprocess.check_call("%s %s %s %s" % (script, zone, direction, folder), shell=True)
    except:
        logging.warning("Unable to adjust files in zone %s" % zone)
    # nothing special to return

# DSS -> CodeStudio
# returns False if no files were changed on DSS, True otherwise
def fetch_dir(execution_id, project_key, code_studio_id, call_path, file_path, commit_id, file_adjustment_script, zone):
    logging.info("Fetching dir for %s.%s" % (project_key, code_studio_id))
    start_time = time.time()
    resp = backend_api_post_call(call_path, stream=True, data={'executionId': execution_id, 'commitId': commit_id})

    if resp.status_code == 200:
        logging.info("Got changes from DSS for %s, merging them" % file_path)
        with open(FETCH_FILE_NAME, 'wb') as fd:
            for chunk in resp.iter_content(chunk_size=4096):
                fd.write(chunk)
        fetch_file_size = os.stat(FETCH_FILE_NAME).st_size
        logging.info("Changes from DSS for %s zipped size is %s bytes" % (file_path, fetch_file_size))
        # create target folder if needed
        if not os.path.exists(file_path):
            os.makedirs(file_path)
        # create temp folder
        tmp_dir = tempfile.mkdtemp()
        try:
            # extract in temp folder
            with zipfile.ZipFile(FETCH_FILE_NAME, mode='r') as zip:
                zip.extractall(tmp_dir)
            # do adjustments before we start tracking changes, because we
            # don't want adjustements to be confused with actual user changes
            if file_adjustment_script is not None and len(file_adjustment_script) > 0:
                run_adjustment_script_on_folder(zone, "pull", file_adjustment_script, tmp_dir)
            # recursive copy, unless file is the same, to avoid
            # jupyterlab/rstudio/vscode... noticing the file's timestamp 
            # was modified
            copy_unless_same(tmp_dir, file_path)
        finally:
            # cleanup temp folder
            if tmp_dir and os.path.isdir(tmp_dir):
                shutil.rmtree(tmp_dir)
        # cleanup zip        
        os.remove(FETCH_FILE_NAME)
        elapsed_time = int(time.time() - start_time)
        logging.info("Merge changes from DSS for %s took %.3fs" % (file_path, elapsed_time))
    elif resp.status_code == 204:
        logging.info("Nothing changed on DSS for %s" % file_path)
        return False  # nothing has changed.
    elif resp.status_code != 404:
        raise Exception("Error fetching dir %s, HTTP status %d: %s" % (file_path, resp.status_code, resp.text))
    else:
        logging.warning("Could not fetch dir")
    return True  # some changes on DSS

# CodeStudio -> DSS
def send_dir(execution_id, command_no, project_key, code_studio_id, call_path, file_path, excluded_from_sync, changes=None, commit_id=None):
    global CHANGES_HIDDEN_FILE_NAME
    logging.info("Sending dir %s for %s.%s" % (file_path, project_key, code_studio_id))
    start_time = time.time()
    tmp_filename = '_dku_send.%s.zip' % ''.join(random.choice(string.ascii_lowercase + string.digits) for _ in range(8))
    tmp_archive = os.path.join(os.environ.get('TMPDIR', '/tmp'), tmp_filename)

    if not os.path.exists(file_path):
        logging.warning("Folder to send doesn't exist")
        return
        
    exclusions = GitignorePatternMatch(excluded_from_sync)
        
    # for some libs, only send what has been touched, typically when the CodeStudio
    # doesn't "own" the lib. Here we build a filter of stuff to send back
    restrict_to_files = None
    if changes is not None:
        # deleted files cannot be sent in the zip (obviously) so the list of
        # deletions is sent separately as the list of paths to delete in a
        # hidden file
        restrict_to_files = changes['added'] + changes['modified']
        # the paths in the changes computed from snapshots all start with a '.'
        # so we need to convert them back to something containing the folder
        # root, so that we can filter on the full path
        def expand_dot_to_folder(p):
            if p == '.':
                return file_path
            elif p.startswith('./'):
                return file_path + p[1:]
            else:
                return p
        restrict_to_files = [expand_dot_to_folder(p) for p in restrict_to_files]
        # at this stage the filter only contains leaf files. Here we add their
        # parents folders
        paths = set()
        for p in restrict_to_files:
            while len(p) >= len(file_path) and p != '.':
                paths.add(p)
                p = os.path.dirname(p)
        restrict_to_files = list(paths)
                    
    logging.info("Files sent will be restricted by %s" % json.dumps(restrict_to_files))
    
    relroot = os.path.abspath(file_path)
    nb_files = 0
    with zipfile.ZipFile(tmp_archive, "w", zipfile.ZIP_DEFLATED) as zip:
        if changes is not None:
            zip.writestr("dku_send/%s" % CHANGES_HIDDEN_FILE_NAME, json.dumps(changes))
            
        for root, dirs, files in os.walk(file_path):
            root_inside_location = os.path.relpath(root, relroot)
            if exclusions.matches(root_inside_location):
                logging.info("Skipping excluded folder %s" % root_inside_location)
                continue                
            if restrict_to_files is not None and root not in restrict_to_files:
                logging.info("%s unchanged, not sent" % root_inside_location)
                continue
            
            arcroot = os.path.join('dku_send', root_inside_location) # DSS complains if the contents of the zip are at its root
            zip.write(root, arcroot) # add directory (needed for empty dirs)
            for file in files:
                file_inside_location = os.path.join(root_inside_location, file)
                if exclusions.matches(file_inside_location):
                    logging.info("Skipping excluded file %s" % file_inside_location)
                    continue
                filename = os.path.join(root, file)
                if restrict_to_files is not None and filename not in restrict_to_files:
                    logging.info("Unchanged, not sent")
                    continue
                if os.path.isfile(filename): # regular files only
                    nb_files += 1
                    arcname = os.path.join(arcroot, file)
                    zip.write(filename, arcname)

    fetch_file_size = os.stat(tmp_archive).st_size
    logging.info("%d files to send, in a zip of %d bytes" % (nb_files, fetch_file_size))

    with open(tmp_archive, 'rb') as f:
        ret = backend_json_call(call_path,
            params={'executionId': execution_id, "commandNo": command_no, 'commitId': commit_id},
            files={'file':('_dku_send.zip', f, 'application/zip')})
    os.remove(tmp_archive)

    elapsed_time = int(time.time() - start_time)
    logging.info("%s sent in %.3fs" % (file_path, elapsed_time))
    return ret

# run one shell command inside the CodeStudio
def run_command_line(execution_id, project_key, code_studio_id, command_no, params):
    use_shell = params.get("useShell", True)
    process = None
    cmd_str = None
    cmd_array = None
    if use_shell:
        cmd_str = params.get("commandString", "")
        if cmd_str.startswith('test_harness_'):
            logging.info("running test command %s" % cmd_str)
            process = subprocess.Popen("echo 'this is just a test'", shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        else:
            logging.info("Execute %s" % cmd_str)
            if cmd_str is None or len(cmd_str) == 0:
                # don't fail, it's too late for that
                process = subprocess.Popen("echo 'No command defined!'", shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            else:
                process = subprocess.Popen(cmd_str, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    else:
        cmd_array = params.get("commandArray", [])
        logging.info("Execute %s" % json.dumps(cmd_array))
        process = subprocess.Popen(cmd_array, shell=False, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    # tail stdout and stderr
    tail_out = TailerThread(process.stdout, sys.stdout)
    tail_err = TailerThread(process.stderr, sys.stderr)
    tail_out.start()
    tail_err.start()
    # it's running, wait for the end asynchronously
    def wait_for_end():
        try:
            if cmd_str == 'test_harness_fail_before_wait':
                raise Exception("fail before wait")
            rv = process.wait()
            logging.info("process ended with %s" % rv)
            if cmd_str == 'test_harness_fail_before_join':
                raise Exception("fail before join")
            tail_out.join()
            tail_err.join()
            logging.info("process out/err joined")
            if cmd_str == 'test_harness_fail_after_join':
                raise Exception("fail after join")
            ret = {'rv':rv, 'tailOut':tail_out.get_tail(), 'tailErr':tail_err.get_tail()}
        except:
            logging.info("Failed to wait on process")
            ret = {'error': get_json_friendly_error()}
        if cmd_str == 'test_harness_fail_before_answer':
            raise Exception("fail before answer")
        backend_void_call("code-studios/answer-command/%s/%s" % (execution_id, command_no), data=json.dumps(ret))
    t = threading.Thread(target=wait_for_end)
    t.start()
    
# for the files tab of the CodeStudio
def list_files(execution_id, project_key, code_studio_id, command_no, params):
    listing_path = params.get("path", '.')
    # make absolute, so that the display in UI is anchored at root
    listing_path = os.path.abspath(listing_path)
    # if path is a file or a link, go one step up to the parent dir
    if os.path.exists(listing_path) and not os.path.isdir(listing_path):
        listing_path = os.path.dirname(listing_path)
        
    ret = {'path': listing_path, 'children':[]}
    ret['exists'] = os.path.exists(listing_path)
    if ret['exists']:
        try:
            for n in os.listdir(listing_path):
                full_path = os.path.join(listing_path, n)
                try:
                    entry = {'name': n}
                    entry['isDirectory'] = os.path.isdir(full_path)
                    entry['isLink'] = os.path.islink(full_path)
                    entry['size'] = os.path.getsize(full_path)
                    entry['lastModified'] = int(os.path.getmtime(full_path) * 1000)
                    ret['children'].append(entry)
                except:
                    logging.warning("File %s couldn't be added" % full_path)
        except Exception as e:
            logging.warning("Can't list %s" % listing_path)
            ret['error'] = str(e)
    backend_void_call("code-studios/answer-command/%s/%s" % (execution_id, command_no), data=json.dumps(ret))
    
def preview_file(execution_id, project_key, code_studio_id, command_no, params):
    file_path = params.get("path", '.')
    # make absolute, so that the display in UI is anchored at root
    file_path = os.path.abspath(file_path)
        
    ret = {'path': file_path}
    ret['exists'] = os.path.exists(file_path)
    ret['isFile'] = os.path.isfile(file_path)
    ret['size'] = os.path.getsize(file_path)
    ret['lastModified'] = int(os.path.getmtime(file_path) * 1000)
    if ret['exists'] and ret['isFile'] and ret['size'] < 10240: # 10k max
        try:
            with open(file_path, 'rb') as fp:
                ret['content'] = base64.b64encode(fp.read()).decode('utf8')
        except Exception as e:
            logging.warning("Can't preview %s" % file_path)
            ret['error'] = str(e)
    backend_void_call("code-studios/answer-command/%s/%s" % (execution_id, command_no), data=json.dumps(ret))
    
# utility to get the commit id of that CodeStudio zone
def find_commit_id(folder, file_name):
    full_path = os.path.join(folder, file_name)
    if os.path.exists(full_path):
        with open(full_path, 'r') as f:
            commit_id = f.read().strip()
        # remove the file to avoid temptation of editing it
        os.remove(full_path)
        return commit_id
    else:
        return None
        
# build a pandas dataframe of the files in a folder, with path+size+mtime
def make_track_changes_snapshot(folder, excluded_from_sync):
    import pandas as pd  # imported inline to allow loading dataiku.core without pandas
    if folder.endswith('/') and len(folder) > 1:
        folder = folder[:-1]

    exclusions = GitignorePatternMatch(excluded_from_sync)
    relroot = os.path.abspath(folder)

    contents = []
    for root, dirs, files in os.walk(folder):
        root_inside_location = os.path.relpath(root, relroot)
        if exclusions.matches(root_inside_location):
            logging.info("Skipping excluded folder %s" % root_inside_location)
            continue
        contents.append({"p":root, "d":True, "s":0, "m":os.path.getmtime(root)})
        for file in files:
            p = os.path.join(root, file)
            if exclusions.matches(p):
                logging.info("Skipping excluded file %s" % p)
                continue
            contents.append({"p":p, "d":False, "s":os.path.getsize(p), "m":os.path.getmtime(p)})
    repl = re.compile("^%s(.*)$" % re.escape(folder))
    def repl_folder_by_dot(f):
        m = repl.match(f['p'])
        if m is not None:
            f['p'] = '.' + m.group(1)
        return f
    contents = [repl_folder_by_dot(f) for f in contents]
    return pd.DataFrame.from_records(contents)
    
# compare 2 snapshots to see what changed (modified/deleted/added)
def diff_track_changes_snapshots(a, b):
    a = a.set_index('p')
    b = b.set_index('p')
    ab = a.join(b, how='outer', lsuffix="_a", rsuffix="_b")
    added = ab[ab['d_a'].isnull()].index.values
    deleted = ab[ab['d_b'].isnull()].index.values
    unmoved = ab[~ab['d_a'].isnull() & ~ab['d_b'].isnull()]
    changed_d = unmoved['d_a'] != unmoved['d_b']
    changed_s = unmoved['s_a'] != unmoved['s_b']
    changed_m = unmoved['m_a'] != unmoved['m_b']
    modified = unmoved[changed_d | changed_s | changed_m].index.values
    return deleted.tolist(), modified.tolist(), added.tolist()
    
class FileZone(object):
    def __init__(self, zone, location, one_way, execution_id, project_key, code_studio_id, excluded_from_sync, file_adjustment_script):
        self.zone = zone
        self.location = location
        self.one_way = one_way
        self.execution_id = execution_id
        self.project_key = project_key
        self.code_studio_id = code_studio_id
        self.excluded_from_sync = excluded_from_sync
        self.file_adjustment_script = file_adjustment_script
        self.ready = False
        self.commitid = None
        self.snapshot = None
        self.zone_id = zone["id"]
        
    def _send_from_code_studio_to_dss(self, command_no, changes):
        log = send_dir(self.execution_id, command_no, self.project_key, self.code_studio_id, 'code-studios/push/%s' % self.zone_id, self.location, self.excluded_from_sync, changes, self.commitid)
        # check the log of git that was returned
        # - if only 1 commit, then that means there is only one new commit since self.projectlib_commitid : the new one
        # - if more than 1, then some commits happened between self.projectlib_commitid and the new one, so there may
        #   be changes that the CodeStudio doesn't have (yet)
        log_entries = log.get("logEntries", [])
        push_committed = log.get("pushCommitted", True)
        if push_committed and len(log_entries) == 1:
            logging.info("Nothing was changed on DSS.")
            self.commitid = log_entries[0]['commitId']
            # refresh the snapshot of the dir, so that previous changes are not sent again
            self.snapshot = make_track_changes_snapshot(self.location, self.excluded_from_sync)
            return summarize_contents(self.location)
        else:
            if len(log_entries) == 0:
                logging.error("Can't get last commit from DSS, future syncs may be slow.")  # still do the sync, hoping for eventual recovery
            # fetch anew
            logging.info("Things may have changed on DSS, getting them.")
            return self.load_from_dss_to_code_studio(-1, {})
        
    def send_from_code_studio_to_dss(self, command_no, command_params):
        if self.location is None or len(self.location) == 0:
            raise Exception("No location set for %s files" % self.zone_id)
        if not self.ready:
            raise Exception("%s files were never pulled into the Code Studio, not pushing back" % self.zone_id)
        # snapshot the dir, to find out what changes and not shuffle everything back and forth
        snapshot = make_track_changes_snapshot(self.location, self.excluded_from_sync)
        deleted, modified, added = diff_track_changes_snapshots(self.snapshot, snapshot)
        if len(deleted) == 0 and len(modified) == 0 and len(added) == 0:
            logging.info("No changes to files, nothing to push back, nothing could be conflicting, fetch DSS version")
            return self.load_from_dss_to_code_studio(-1, {})
        if self.one_way:
            logging.info("Zone is one-way, fetch DSS version")
            return self.load_from_dss_to_code_studio(-1, {})
        changes = {'added': added, 'modified': modified, 'deleted': deleted}

        if command_params.get('force', False):
            logging.info("Send %s files regardless of potential conflicts" % self.zone_id)
            return self._send_from_code_studio_to_dss(command_no, changes)
        else:
            conflicts = backend_json_call("code-studios/check-conflicts/%s/%s/%s" % (self.zone_id, self.execution_id, command_no), data={'commitId': self.commitid, 'changes': json.dumps(changes)})
            conflict_count = 0
            conflict_count += len(conflicts.get('added', []))
            conflict_count += len(conflicts.get('modified', []))
            conflict_count += len(conflicts.get('deleted', []))
            commit_from = conflicts.get('commitFrom')  # beginning of the range of commits from commit_id to HEAD
            commit_to = conflicts.get('commitTo')  # end of the range of commits from commit_id to HEAD
            if conflict_count > 0:
                logging.info("Conflicts found, requesting confirmation : %s" % json.dumps(conflicts))
                return conflicts
            else:
                logging.info("No conflicts found, sending %s files" % self.zone_id)
                return self._send_from_code_studio_to_dss(command_no, changes)

    def load_from_dss_to_code_studio(self, command_no, command_params):
        global COMMITID_HIDDEN_FILE_NAME
        if self.location is None or len(self.location) == 0:
            raise Exception("No location set for %s files" % self.name)
        had_changes = fetch_dir(self.execution_id, self.project_key, self.code_studio_id, 'code-studios/pull/%s' % self.zone_id, self.location, self.commitid, self.file_adjustment_script, self.zone.get("zone"))
        self.ready = True
        if had_changes:
            # find commit id if it's there
            self.commitid = find_commit_id(self.location, COMMITID_HIDDEN_FILE_NAME)
            # snapshot the dir, to find out what changes and not shuffle everything back and forth
            self.snapshot = make_track_changes_snapshot(self.location, self.excluded_from_sync)
        return summarize_contents(self.location)

    def get_changes(self):
        if self.location is None or len(self.location) == 0:
            return {}
        if not self.ready:
            return {}
        if self.one_way:
            logging.info("Zone is one-way, cannot conflict")
            return {}
        # snapshot the dir, to find out what changes and not shuffle everything back and forth
        snapshot = make_track_changes_snapshot(self.location, self.excluded_from_sync)
        deleted, modified, added = diff_track_changes_snapshots(self.snapshot, snapshot)
        if len(deleted) == 0 and len(modified) == 0 and len(added) == 0:
            logging.info("No changes to files for %s" % self.zone_id)
            return {}
        logging.info("%i added, %i modified, %i deleted for %s" % (len(added), len(modified), len(deleted), self.zone_id))
        changes = {'added': added, 'modified': modified, 'deleted': deleted}
        return changes

    def get_conflicts(self, command_no):
        changes = self.get_changes()
        logging.info("Checking conflicts on %s : %s" % (self.zone_id, json.dumps(changes)))
        return backend_json_call("code-studios/check-conflicts/%s/%s/%s" % (self.zone_id, self.execution_id, command_no), data={'commitId': self.commitid, 'changes':json.dumps(changes)})

# the code running in a thread spawned from the runner.py in parallel of the CodeStudio entrypoints
class CommandServerThread(object):
    def __init__(self, execution_id, project_key, code_studio_id, synced_zones, excluded_from_sync, file_adjustment_script, reattach_on_restart):
        self.execution_id = execution_id
        self.code_studio_id = code_studio_id
        self.project_key = project_key
        self.projectlib_ready = False
        self.synced_zones = [FileZone(synced_zone, synced_zone['pathInContainer'], synced_zone.get('oneWay', False), execution_id, project_key, code_studio_id, excluded_from_sync, file_adjustment_script) for synced_zone in synced_zones]
        self.excluded_from_sync = excluded_from_sync
        self.file_adjustment_script = file_adjustment_script
        self.reattach_on_restart = reattach_on_restart
        
    def get_synced_zone(self, zone_id):
        for synced_zone in self.synced_zones:
            if synced_zone.zone_id == zone_id:
                return synced_zone
        raise Exception("No synced zone handles %s" % zone_id)
        
    def project_key_path_replacements(self):
        replacements = {}
        for synced_zone in self.synced_zones:
            if synced_zone.zone['zone'] == 'project_lib_versioned' and synced_zone.ready:
                path_in_zone = synced_zone.zone.get('pathInZone', '')
                if path_in_zone.startswith('/'):
                    path_in_zone = path_in_zone[1:]
                if len(path_in_zone) > 0 and not path_in_zone.endswith('/'):
                    path_in_zone = path_in_zone + '/'
                target_folder = synced_zone.location
                if not target_folder.endswith('/'):
                    target_folder = target_folder + '/'
                replacements[path_in_zone] = target_folder
        return replacements
        
    def poll_commands(self):
        while True:
            time.sleep(1)
            try:
                # timeout has to be larger than the polling in the backend
                commands = backend_json_call("code-studios/poll-commands", data={'executionId':self.execution_id}, timeout=40)
                logging.info('received %s' % json.dumps(commands))
                logging.info('received %s commands' % len(commands))
                for command in commands:
                    command_no = command.get('no')
                    try:
                        result = self.handle_command(command.get('type'), command_no, command.get('params'))
                        if result is not None: # None means "handle_command already took care of the answer"
                            backend_void_call("code-studios/answer-command/%s/%s" % (self.execution_id, command_no), data=json.dumps(result))
                    except Exception as e:
                        logging.error("Cannot execute command: %s" % e)
                        traceback.print_exc()
                        backend_void_call("code-studios/answer-command/%s/%s" % (self.execution_id, command_no), data=json.dumps({'error': get_json_friendly_error()}))
            except Exception as e:
                logging.error("Failed to poll commands: %s" % e)
                # in case this is a disconnect and not a refusal to talk from the backend, ie when it's a
                # network/restart issue but not that the backend wants to kill the CSio, check the error
                if 'ConnectionError' in str(e) or 'RemoteDisconnected' in str(e):
                    logging.warning("Backend seems unreachable, wait and retry")
                # note: this is the message that the backend spits out when you try to use an
                # API ticket that has expired or was valid in another DSS backend run
                if 'Ticket not given or unrecognized' in str(e) and (self.reattach_on_restart is None or self.reattach_on_restart == False):
                    logging.info("Backend has restarted, won't come back, aborting loop")
                    return
                time.sleep(10)
        logging.info("Done polling commands")
                  
    def start_polling_commands(self):                
        t = threading.Thread(target=self.poll_commands)
        t.daemon = True
        t.start()
        
    def handle_command(self, command_type, command_no, command_params):
        global COMMITID_HIDDEN_FILE_NAME
        logging.info('Handle command %s (params=%s)' % (command_type, json.dumps(command_params)))
        _dku_test = {} 
        if os.path.exists('_dku-test.json'): # test harness stuff
            try:
                with open('_dku-test.json') as fd:
                    _dku_test = json.loads(fd.read())
            except Exception as e:
                logging.error("Cannot handle the contents of _dku-test.json : %s" % str(e))
        # basic test stuff: make it slow
        if 'throttleCommands' in _dku_test:
            time.sleep(_dku_test['throttleCommands'])
        # more basic stuff: make it fail
        if command_type in _dku_test.get("fails", {}):
            raise Exception(_dku_test.get("fails", {}).get(command_type, "Fail for test"))
        if command_type in _dku_test.get("answerBullshit", {}):
            backend_void_call("code-studios/answer-command/%s/%s" % (self.execution_id, command_no), data=_dku_test.get("answerBullshit", {}).get(command_type))
            return None
        # actual command handling
        if command_type == 'run_command_line':
            run_command_line(self.execution_id, self.project_key, self.code_studio_id, command_no, command_params)
        elif command_type == 'list_files':
            list_files(self.execution_id, self.project_key, self.code_studio_id, command_no, command_params)
        elif command_type == 'preview_file':
            preview_file(self.execution_id, self.project_key, self.code_studio_id, command_no, command_params)
        elif command_type == 'pull_bundle_from_code_studio':  # User clicked on Sync. Force = true means : no need to check conflicts, we already did it
            zone_id = command_params.get("zone")
            if zone_id is None or len(zone_id) == 0:
                raise Exception("Undefined zone to pull")
            return self.get_synced_zone(zone_id).send_from_code_studio_to_dss(command_no, command_params)
        elif command_type == 'push_bundle_to_code_studio':  # User clicked on Reset.
            zone_id = command_params.get("zone")
            if zone_id is None or len(zone_id) == 0:
                raise Exception("Undefined zone to push")
            zone = self.get_synced_zone(zone_id)
            zone.commitid = None  # To trigger the reset
            return zone.load_from_dss_to_code_studio(command_no, command_params)
        elif command_type == 'check_conflicts':
            conflicts = {}
            for k in command_params:
                synced_zone = self.get_synced_zone(k)
                conflicts[k] = synced_zone.get_conflicts(command_no)
            return conflicts
        elif command_type == 'get_changes':
            changes = {}
            for k in command_params:
                synced_zone = self.get_synced_zone(k)
                changes[synced_zone.location] = synced_zone.get_changes()
            return changes
        else:
            logging.warning("Unknown command %s, ignoring" % command_type)
            return {}
        