(function() {
'use strict';

const app = angular.module('dataiku.bundles.automation',[]);


app.controller("AutomationBundleDetailsModalController", function($scope, $stateParams, Assert, DataikuAPI) {
    $scope.uiState = {
        activeTab : "content"
    };

    // We don't have access to the original git on the automation node, and don't store a per-commit diff in the bundle's changelog.json
    $scope.noCommitDiff = true;

    function fetch() {
        Assert.inScope($scope, 'bundleId');
        DataikuAPI.projects.automation.getBundleDetails($stateParams.projectKey, $scope.bundleId).success(function(data) {
            $scope.bundleDetails = data;
        }).error(setErrorInScope.bind($scope))
    }

    $scope.$watch("bundleId", function(nv, ov) {
        if (!nv) return;
        fetch();
    });

    $scope.modalActivate = function() {
        Assert.inScope($scope, 'bundleId');
        $scope.startActivate($scope.bundleId);
        $scope.dismiss();
    };

    $scope.modalPreload = function() {
        Assert.inScope($scope, 'bundleId');
        $scope.preloadBundle($scope.bundleId);
        $scope.dismiss();
    };
});


app.controller("AutomationBundlesSettingsController", function($scope, $stateParams, Assert, DataikuAPI, $state, TopNav, ActivityIndicator, AutomationUtils) {
    TopNav.setLocation(TopNav.TOP_HOME, "bundlesautomation", TopNav.TABS_NONE, "settings");
    TopNav.setNoItem();

    $scope.uiState = {
        settingsPane: 'connections'
    };
    $scope.AutomationUtils = AutomationUtils;

    var savedSettings;
    $scope.dirtySettings = function() {
        return !angular.equals($scope.settings, savedSettings);
    };
    checkChangesBeforeLeaving($scope, $scope.dirtySettings);

    function load() {
        DataikuAPI.projects.automation.getBundleActivationSettingsExt($stateParams.projectKey).success(function(data) {
            $scope.settings = data.settings;
            $scope.availableConnections = data.availableConnections;
            $scope.connectionsUsedByLastBundle = data.usedByLastBundle;
            $scope.availableContainerExecConfs = data.availableContainerExecConfs;
            $scope.containerExecConfsUsedByLastBundle = data.containerExecConfsUsedByLastBundle;
            savedSettings = angular.copy($scope.settings);
            updateConnectionsCoverage();
            updateContainerExecCoverage();
        }).error(setErrorInScope.bind($scope));
    }

    $scope.$watch("settings.remapping.connections", updateConnectionsCoverage, true);

    function updateConnectionsCoverage() {
        if(!$scope.connectionsUsedByLastBundle) {
            return;
        }
        $scope.connectionTypeAheadValues = [];
        $scope.connectionsUsedByLastBundle.forEach(function(connection) {
            var mapped = $scope.settings.remapping.connections.filter(function(v) { return v.source === connection.name; });
            var original = $scope.availableConnections.filter(function(v) { return v.name === connection.name; });
            if (mapped.length) {
                connection.mapsTo = $scope.availableConnections.filter(function(v) { return v.name === mapped[0].target; })[0];
                connection.clickable = false;
            } else if (original.length) {
                connection.mapsTo = original[0];
                connection.clickable = true;
                $scope.connectionTypeAheadValues.push(connection.name);
            } else {
                connection.mapsTo = null;
                connection.clickable = true;
                $scope.connectionTypeAheadValues.push(connection.name);
            }
        });
    }

    function updateContainerExecCoverage() {
        if(!$scope.containerExecConfsUsedByLastBundle) {
            return;
        }
        $scope.containerExecTypeAheadValues = [];
        $scope.containerExecConfsUsedByLastBundle.forEach(function(config) {
            var mapped = $scope.settings.remapping.containerExecs.filter(function(v) { return v.source === config.name; });
            var original = $scope.availableContainerExecConfs.filter(function(v) { return v.name === config.name; });
            if (mapped.length) {
                config.mapsTo = $scope.availableContainerExecConfs.filter(function(v) { return v.name === mapped[0].target; })[0];
            } else if (original.length) {
                config.mapsTo = original[0];
                $scope.containerExecTypeAheadValues.push(config.name);
            } else {
                config.mapsTo = null;
                $scope.containerExecTypeAheadValues.push(config.name);
            }
        });
    }

    $scope.addConnectionRemapping = function(name) {
        Assert.inScope($scope, 'settings');
        const newConnectionRemapping = $scope.prepareConnectionRemapping(name);
        const connectionsCopy = angular.copy($scope.settings.remapping.connections);
        connectionsCopy.push(newConnectionRemapping);
        $scope.settings.remapping.connections = connectionsCopy; // we use a copy in order to trigger the $watch('ngModel') in editableList directive
    };

    $scope.addContainerExecRemapping = function(name) {
        Assert.inScope($scope, 'settings');
        const newContainerExecRemapping = $scope.prepareContainerExecRemapping(name);
        const containerExecsCopy = angular.copy($scope.settings.remapping.containerExecs);
        containerExecsCopy.push(newContainerExecRemapping);
        $scope.settings.remapping.containerExecs = containerExecsCopy; // we use a copy in order to trigger the $watch('ngModel') in editableList directive
    };

    $scope.prepareConnectionRemapping = function(name) {
        return {
            source: name,
            target: null
        };
    };

    $scope.prepareContainerExecRemapping = function(name) {
        return {
            source: name,
            target: null
        };
    };

    $scope.save = function() {
        DataikuAPI.projects.automation.saveBundleActivationSettings($stateParams.projectKey, $scope.settings).success(function(data) {
            ActivityIndicator.success("Saved");
            savedSettings = angular.copy($scope.settings);
        }).error(setErrorInScope.bind($scope));
    };

    load();
});


app.controller("AutomationBundlesListController", function($scope, $controller, $stateParams, DataikuAPI, Dialogs, $state, $q, TopNav, Fn, CreateModalFromTemplate, FutureProgressModal) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    TopNav.setLocation(TopNav.TOP_HOME, "bundlesautomation", TopNav.TABS_NONE, "list");
    TopNav.setNoItem();

    $scope.noTags = true;
    $scope.noWatch = true;
    $scope.noStar = true;
    $scope.sortBy = [
        {label: 'Imported On', value: 'importState.importedOn'},
        {label: 'Exported On', value: 'exportManifest.exportUserInfo.exportedOn'},
        {label: 'Name', value: 'bundleId'},
    ];
    $scope.sortCookieKey = 'bundlesautomation';
    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
        },
        filterParams: {userQueryTargets: "bundleId"},
        orderQuery: 'importState.importedOn',
        orderReversed: false,
    }, $scope.selection || {});
    $scope.noTags = true;
    $scope.maxItems = 20;

    $scope.list = function() {
        DataikuAPI.projects.automation.listBundles($stateParams.projectKey).success(function(data) {
            $scope.listItems = data.bundles;
            $scope.$broadcast('clearMultiSelect');
        }).error(setErrorInScope.bind($scope));
    };

    $scope.list();

    $scope.goToItem = function(data) {
        $scope.showBundleDetails(data);
    };

    $scope.showBundleDetails = function(data) {
        CreateModalFromTemplate("/templates/bundles/automation/details-modal.html", $scope, null, function(modalScope) {
            modalScope.bundleId = data.bundleId;
            modalScope.$apply();
        });
    };

    $scope.importBundle = function() {
        CreateModalFromTemplate("/templates/bundles/automation/import-bundle-modal.html", $scope);
    };

    $scope.startActivate = function(bundleId) {
        DataikuAPI.projects.automation.checkBundleActivation($stateParams.projectKey, bundleId).success(function(data) {
            CreateModalFromTemplate("/templates/bundles/automation/activation-check-result.html", $scope, null, function(modalScope) {
                modalScope.checkResult = data;
            })
        }).error(setErrorInScope.bind($scope));
    };

    $scope.preloadBundle = function(bundleId) {
        DataikuAPI.projects.automation.preloadBundle($stateParams.projectKey, bundleId).success(function(data) {
            FutureProgressModal.show($scope, data, "Preloading bundle").then(function(preloadResult) {
                if (preloadResult.anyMessage) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Preload report", preloadResult, preloadResult.futureLog);
                }
                $scope.list() // TODO do we really need this?
                $scope.refreshProjectData();
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.deleteBundle = function(bundle) {
        Dialogs.confirmSimple($scope, "Delete bundle <strong>" + bundle.bundleId +"</strong>?").then(function() {
            DataikuAPI.projects.automation.deleteBundle($stateParams.projectKey, bundle.bundleId)
                .success($scope.list.bind(null))
                .error(setErrorInScope.bind($scope));
        });
    };

    $scope.deleteSelected = function() {
        if ($scope.selection.none) {
            return;
        } else if ($scope.selection.single) {
            $scope.deleteBundle($scope.selection.selectedObject);
        } else {
            Dialogs.confirm($scope, "Confirm deletion", "Are you sure you want to delete the selected bundles?").then(function() {
                $q.all($scope.selection.selectedObjects.map(Fn.prop('bundleId'))
                    .map(DataikuAPI.projects.automation.deleteBundle.bind(null, $stateParams.projectKey))
                ).then($scope.list.bind(null), setErrorInScope.bind($scope));
            });
        }
    };
});

// You MUST call setCurrentProjectFolderId with the appropriate folderId in the link method of CreateModal...
app.controller("AutomationBundleNewProjectModalController", function($scope, $stateParams, $state, Assert, DataikuAPI, ProjectFolderService, PathUtils, PromiseService, translate, $rootScope) {
    $scope.newProject = {}
    $scope.phase = "READY_TO_UPLOAD";

    $scope.targetProjectFolder = null; // used in the project folder service
    $scope.defaultProjectFolderId = null;

    $scope.setCurrentProjectFolderId = (folderId) => {
        ProjectFolderService.getDefaultFolderForNewProject(folderId).then((folder) => {
            $scope.targetProjectFolder = folder;
            $scope.defaultProjectFolderId = folder.id;
        }).catch(setErrorInScope.bind($scope));
    }

    $scope.browse = folderIds => {
        // Use last id in path
        $scope.destination = PathUtils.makeNLNT(folderIds).split('/').pop();
        // browse-path expects a success-error promise so we need to wrap with qToHttp for now (catch() does not return a monkey-patched promise)
        return PromiseService.qToHttp(ProjectFolderService.getBrowseNode($scope.destination).catch(setErrorInScope.bind($scope)));
    };

    $scope.getProjectFolderName = item => item.name;
    $scope.canSelectFolder = item => item.canWriteContents;

    $scope.importButtonTooltip = function() {
        if ($scope.newProjectForm.$invalid) {
            return translate("HOME.PROJECTS.NEW.BUNDLE.HELP", "Select a bundle to install");
        }

        if (!$scope.mayCreateProjects) {
            return translate("HOME.PROJECTS.NEW.BUNDLE.NO_PERM.HELP", "You do not have permission to create projects");
        }

        if ($scope.targetProjectFolder && !$scope.targetProjectFolder.canWriteContents) {
            return translate("HOME.NEW_PROJECT.FOLDER.CANNOT_WRITE.HELP", "You cannot write to this folder");
        }
        return "";
    };

    $scope.mayCreateProjects = $rootScope.appConfig && $rootScope.appConfig.globalPermissions.mayCreateProjects;

    $scope.create = function() {
        Assert.trueish($scope.newProject.file, "No file for new project");
        $scope.phase = "UPLOADING";
        DataikuAPI.projects.automation.createWithInitialBundle(
            $scope.newProject.file,
            $scope.targetProjectFolder.id,
            null,
            function(e){
                if (e.lengthComputable) {
                    $scope.$apply(function () {
                        $scope.uploadProgress = Math.round(e.loaded * 100 / e.total);
                    });
                }
            }
        ).then(function(data) {
            $scope.resolveModal();
            $state.go("projects.project.home.regular", {projectKey: JSON.parse(data).projectKey});
        }, function(payload) {
            $scope.phase = "READY_TO_UPLOAD";
            setErrorInScope.bind($scope)(JSON.parse(payload.response), payload.status, function(h) {return payload.getResponseHeader(h)});
        });
    };
});


app.controller("AutomationBundleImportBundleModalController", function($scope, $stateParams, Assert, DataikuAPI) {
    $scope.newBundleImport = {}

    $scope.newBundle = {}

    $scope.import = function() {
        Assert.trueish($scope.newBundle.file, "No file for new bundle");
        DataikuAPI.projects.automation.importBundle($stateParams.projectKey, $scope.newBundle.file).then(function(data) {
            $scope.$parent.$parent.list();
            $scope.dismiss();
        }, function(payload) {
            setErrorInScope.bind($scope)(JSON.parse(payload.response), payload.status, function(h) {return payload.getResponseHeader(h)});
        });
    }
});


app.controller("AutomationBundleCheckResultModalController", function($scope, DataikuAPI, $state, $stateParams, Assert, FutureProgressModal, Dialogs, DKUConstants) {
    $scope.doActivate = function() {
        Assert.inScope($scope, 'checkResult');
        DataikuAPI.projects.automation.activateBundle($stateParams.projectKey, $scope.checkResult.bundleId).success(function(data) {
            var parentScope = $scope.$parent.$parent; // Ugly

            $scope.dismiss();

            FutureProgressModal.show(parentScope, data, "Activating bundle").then(function(activateResult) {
                if (activateResult.anyMessage) {
                    Dialogs.infoMessagesDisplayOnly(parentScope, "Activation report", activateResult);
                }
                parentScope.list() // TODO do we really need this?
                parentScope.refreshProjectData();
            });
        }).error(setErrorInScope.bind($scope));
    };
});

})();
