(function() {
'use strict';

const app = angular.module('dataiku.apideployer');


app.constant('INFRA_TYPES', {
    STATIC: "static",
    K8S: "Kubernetes",
    DATABRICKS: "Databricks",
    SAGEMAKER: "Amazon SageMaker",
    SNOWPARK: "Snowflake Snowpark",
    VERTEX_AI: "Google Vertex AI",
    AZURE_ML: "Azure Machine Learning"
});


app.controller('APIDeployerInfrasListController', function($scope, $controller, DataikuAPI, DeployerUtils) {
    $controller('_DeployerInfrasListController', {$scope: $scope});

    if ($scope.isFeatureLocked) return;

    $scope.$watch("infraStatusList", function(nv) {
        if (!nv) return;
        nv.forEach(function(infraStatus) {
            const infraId = infraStatus.infraBasicInfo.id;
            infraStatus.enabledDeploymentCount = DeployerUtils.enabledDeploymentCount(infraStatus.deployments, true);
            DataikuAPI.apideployer.infras.checkStatus(infraId)
                .success(function(healthStatus) {
                    infraStatus.infraHealthError = healthStatus.messages.find(function(msg) {
                        return msg.severity === healthStatus.maxSeverity;
                    }) || {};
                })
                .error(setErrorInScope.bind($scope));
        });
    });
});


app.controller('APIDeployerInfraController', function($controller, $scope, $rootScope) {
    $controller('_DeployerInfraController', {$scope: $scope});
    $scope.uiState = {};

    const deregisterInfra = $scope.$watch("infraStatus", function(nv, ov) {
        if (!nv) return;

        let canDeleteInfraWhenInCloud;

        if (!$scope.isCloud) {
            canDeleteInfraWhenInCloud = true;
        } else {
            canDeleteInfraWhenInCloud = (nv.infraBasicInfo.type !== 'STATIC') || $rootScope.appConfig.canAccessCloudDataikerAdminCapabilities;
        }

        $scope.uiState.canDeleteInfraWhenInCloud = canDeleteInfraWhenInCloud;

        deregisterInfra();
    }, false);

});


app.controller('APIDeployerInfraSetupModalController', function($scope, $controller, $rootScope) {
    $controller('_DeployerInfraSetupModalController', {$scope: $scope});
    $scope.newInfra.type = 'STATIC';
    $scope.appConfig = $rootScope.appConfig;
});


app.controller('APIDeployerInfraStatusController', function($scope, $controller, DataikuAPI, APIDeployerAsyncHeavyStatusLoader, DeployerDeploymentTileService) {
    $controller("_DeployerInfraStatusController", {$scope: $scope});

    let loader;
    $scope.$watch("infraStatus", function(nv) {
        if (nv) {
            let infraId = $scope.infraStatus.infraBasicInfo.id;
            DataikuAPI.apideployer.infras.checkStatus(infraId)
                .success(function(healthStatus) {
                    $scope.infraHealthError = healthStatus.messages.find(function(msg) {
                        return msg.severity === healthStatus.maxSeverity;
                    }) || {};
                })
                .error(setErrorInScope.bind($scope));

            $scope.healthMap = {};
            const heavyStatusByDeploymentId = {};
            loader = APIDeployerAsyncHeavyStatusLoader.newLoader(heavyStatusByDeploymentId);
            loader.loadFromInfraLightStatus(nv);

            const deregister = $scope.$watch(function(){
                return loader.stillRefreshing();
            }, function(nv, ov) {
                if (nv || ov === nv) return;
                $scope.healthMap = DeployerDeploymentTileService.getDeploymentHealthMap($scope.infraStatus.deployments, heavyStatusByDeploymentId);
                deregister();
            });

            $scope.$on('$destroy', function() {
                loader && loader.stopLoading();
            });
        }
    });
});

app.controller('APIDeployerInfraHistoryController', function($scope, $controller) {
    $controller('_DeployerInfraHistoryController', {$scope: $scope});
});


app.controller('APIDeployerInfraSettingsController', function($scope, $controller, $rootScope, $filter, DataikuAPI, ClipboardUtils, ActivityIndicator, CreateModalFromTemplate, APIDeploymentInfraHelper, FutureWatcher) {
    $controller('_DeployerInfraSettingsController', {$scope: $scope});

    $scope.uiState.showInstanceTypeList = false;

    let inlineContainerConfig = {
        name: "inline",
        type: "KUBERNETES",
        baseImageType: "EXEC",
        properties: [],
    };

    $scope.getInlineContainerConfig = function() {
        if ($scope.infra) {
            inlineContainerConfig.kubernetesNamespace = $scope.infra.k8sNamespace;
            inlineContainerConfig.kubeCtlContext = $scope.infra.k8sContext;
            inlineContainerConfig.kubeConfigPath = $scope.infra.k8sConfigPath;
            inlineContainerConfig.properties = $scope.infra.k8sProperties;
            inlineContainerConfig.baseImage = $scope.infra.baseImageTag;
            inlineContainerConfig.repositoryURL = $scope.infra.registryHost;
            inlineContainerConfig.prePushMode = $scope.infra.prePushMode;
            inlineContainerConfig.prePushScript = $scope.infra.prePushScript;
        }
        return inlineContainerConfig; // return the same object to avoid never-ending $digest() issues
    };

    if ($rootScope.appConfig.admin) {
        DataikuAPI.admin.connections.list().success(function(allConnections) {
            $scope.allConnections = allConnections;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.autoFillEventServer = function() {
        $scope.deployerAPIBase.infras.getDkuEventServerUrlAndKeys($scope.infra.type === 'K8S').success(function (urlAndKeys) {
            if (urlAndKeys && urlAndKeys.isLocalEventServerEnabled) {
                $scope.infra.defaultApiNodeLogging.eventServerURL = urlAndKeys.eventServerExtUrl ? urlAndKeys.eventServerExtUrl : urlWithProtocolAndHost();
                $scope.infra.defaultApiNodeLogging.eventServerAuthKey = (urlAndKeys.authKeys && urlAndKeys.authKeys.length)? urlAndKeys.authKeys[0].key : "";
            } else {
                ActivityIndicator.warning("No local event server configured");
            }
        }).error(setErrorInScope.bind($scope));
    };

    /******** cluster *******/
    DataikuAPI.admin.clusters.listAccessible('KUBERNETES').success(function(data){
        $scope.k8sClusterIds = data.map(function(c) {return c.id;});
    }).error(setErrorInScope.bind($scope));


    /******** actions *******/
    $scope.onLocalConnectionChanged = function() {
        if ($scope.uiState.selectedLocalConnectionName) {
            ClipboardUtils.copyToClipboard(JSON.stringify($scope.allConnections[$scope.uiState.selectedLocalConnectionName], null, 2));
        }
    };

    $scope.deleteRemappedConnection = function(pckConId) {
        if ($scope.infra && $scope.infra.remappedConnections) {
            delete $scope.infra.remappedConnections[pckConId];
        }
    };

    $scope.hasRemappedConnections = function() {
        return $scope.infra && Object.keys($scope.infra.remappedConnections).length;
    };

    const requestSageMakerInstanceTypes = function() {
        $scope.deployerAPIBase.infras.getSageMakerInstanceTypes().success(function (data) {
            $scope.instanceTypes = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.requestAzureMLInstanceTypes = function(infraId, authConnection, azWorkspace, azResourceGroup, azSubscription) {
        $scope.deployerAPIBase.infras.getAzureMLInstanceTypes(infraId, authConnection, azWorkspace, azResourceGroup, azSubscription).success(function (data) {
            $scope.instanceTypes = data;
            $scope.uiState.showInstanceTypeList = true;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.requestVertexAIInstanceTypes = function(infraId, authConnection, gcpProjectId, gcpRegion) {
        $scope.deployerAPIBase.infras.getVertexAIInstanceTypes(infraId, authConnection, gcpProjectId, gcpRegion).success(function (data) {
            $scope.instanceTypes = data;
            $scope.uiState.showInstanceTypeList = true;
        }).error(setErrorInScope.bind($scope));
    }

    const requestVertexAIAcceleratorTypes = function() {
        $scope.deployerAPIBase.infras.getVertexAIAcceleratorTypes().success(function (data) {
            $scope.acceleratorTypes = data.map(d => {
                return { value: d, name: d === "ACCELERATOR_TYPE_UNSPECIFIED" ? "None" : d };
            });
        }).error(setErrorInScope.bind($scope));
    }

    const requestDatabricksWorkloadTypes = function() {
        $scope.deployerAPIBase.infras.getDatabricksWorkloadTypes().success(function (data) {
            $scope.workloadTypes = data;
        }).error(setErrorInScope.bind($scope));
    }

    const requestDatabricksWorkloadSizeTypes = function() {
        $scope.deployerAPIBase.infras.getDatabricksWorkloadSizeTypes().success(function (data) {
            $scope.workloadSizeTypes = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.requestDatabricksExperimentLocationSuggestions = function() {
        if ($scope.uiState.isExperimentLocationsLoadingInProgress) {
            return;
        }
        $scope.deployerAPIBase.infras.getDatabricksExperimentLocationSuggestions($scope.infra.authConnection)
            .success(function (futureResp) {
                $scope.uiState.isExperimentLocationsLoadingInProgress = true;
                FutureWatcher.watchJobId(futureResp.jobId)
                    .success(function (resp) {
                        $scope.experimentLocationSuggestions = resp.result.map(s => s.path);
                    })
                    .error(setErrorInScope.bind($scope))
                    .finally(() => {
                        $scope.uiState.isExperimentLocationsLoadingInProgress = false;
                    });
            })
            .error(setErrorInScope.bind($scope));
    }

    $scope.removeDatabricksExperimentLocationSuggestions = function() {
        $scope.experimentLocationSuggestions = null;
    }

    const canAccessStaticCloudManagedSettings = function(infraType) {
        if (infraType !== 'STATIC') {
            return false;
        }

        return $scope.onlyForDataikerAdminWhenInCloud;
    };

    const canAccessAuditing = function(infraType) {
        if (infraType === 'DATABRICKS') {
            return false;
        }

        return $scope.onlyForDataikerAdminWhenInCloud;
    };

    const showWhenDataikerAdminOrNonStaticWhenInCloud = function(infraType) {
        if (!$scope.isCloud) {
            return true;
        }

        return infraType !== 'STATIC' || $rootScope.appConfig.canAccessCloudDataikerAdminCapabilities;
    };

    const deregisterInfra = $scope.$watch("infra", function(nv, ov) {
        if (!nv) return;

        $scope.uiState.showApiNodesTab = canAccessStaticCloudManagedSettings(nv.type);
        $scope.uiState.showAuditingTab = canAccessAuditing(nv.type);
        $scope.uiState.showPermissionsTab = showWhenDataikerAdminOrNonStaticWhenInCloud(nv.type);
        $scope.uiState.showLifecycleField = showWhenDataikerAdminOrNonStaticWhenInCloud(nv.type);
        $scope.uiState.showTrustAllCertificates = canAccessStaticCloudManagedSettings(nv.type);

        if (APIDeploymentInfraHelper.isFullyManagedInfra(nv.type)) {
            let connectionType;
            let cloudStorageConnectionType;

            if (nv.type === 'SAGEMAKER') {
                connectionType = 'SageMaker';
                cloudStorageConnectionType = 'EC2';
                requestSageMakerInstanceTypes();
            } else if (nv.type === 'SNOWPARK') {
                connectionType = 'Snowflake';
            } else if (nv.type === 'AZURE_ML') {
                connectionType = 'AzureML';
                // Do not ask for instance types since they are retrieved from an external resource and can take a long time.
                // The user can initiate the retrieval manually.
            } else if (nv.type === 'DATABRICKS') {
                connectionType = 'DatabricksModelDeployment';
                requestDatabricksWorkloadTypes();
                requestDatabricksWorkloadSizeTypes();
                $scope.experimentLocationSuggestions = null;
            } else if (nv.type === 'VERTEX_AI') {
                connectionType = 'VertexAIModelDeployment';
                // Do not ask for instance types since they are retrieved from an external resource and can take a long time.
                // The user can initiate the retrieval manually.
                requestVertexAIAcceleratorTypes();
            }

            if (connectionType) {
                DataikuAPI.connections.getNames(connectionType)
                    .success(function (data) {
                        $scope.connections = data.map(n => {
                            return {name: n, label: n}
                        });
                        $scope.connectionsWithEnvironment = $scope.isCloud ? $scope.connections : [{name: null, label: "Environment"}].concat($scope.connections);
                    })
                    .error(setErrorInScope.bind($scope));
            }

            if (cloudStorageConnectionType) {
                DataikuAPI.connections.getNames(cloudStorageConnectionType)
                    .success(function (data) {
                        $scope.cloudStorageConnections = data;
                    })
                    .error(setErrorInScope.bind($scope));
            }
        }

        deregisterInfra();
    }, false);
});


app.filter('infraTypeToName', function(INFRA_TYPES) {
    return function(type) {
        if (!type) {
            return;
        }
        return INFRA_TYPES[type] || type;
    };
});

app.filter('infraTypeToIcon', function() {
    return function(infraType, size) {
        if (!size) {
            size = 16;
        }
        if (!infraType) {
            return;
        }
        if (infraType === 'SAGEMAKER') {
            return 'dku-icon-amazon-sagemaker-' + size;
        } else if (infraType === 'AZURE_ML') {
            return 'dku-icon-microsoft-azure-ml-' + size;
        } else if (infraType === 'VERTEX_AI') {
            return 'dku-icon-google-vertex-' + size;
        } else if (infraType === 'SNOWPARK') {
            return 'dku-icon-snowflake-brand-' + size;
        }  else if (infraType === 'DATABRICKS') {
            return "dku-icon-databricks-" + size;
        } else {
            return 'dku-icon-dataiku-' + size;
        }
    };
});

app.component("activityMonitoringSettings", {
    bindings: {
        activityMonitoringSettings: '=',
        infra: '<'
    },
    template: `
    <div class="activity-monitoring-settings">
        <h2 class="settings-section-title mtop0">Activity server</h2>
        <div class="alert alert--warning mtop4 mbot4"
             ng-if="$ctrl.infra.endpointType === 'SERVERLESS' && $ctrl.activityMonitoringSettings.mode === 'PUSH_TO_DEPLOYER'">
            Pushing Activity Metrics to Deployer is not supported in SERVERLESS mode.
        </div>

        <div class="alert alert-info" ng-if="['SAGEMAKER', 'VERTEX_AI', 'AZURE_ML', 'DATABRICKS'].includes($ctrl.infra.type)">
            These settings control how the activity statistics are collected.
        </div>
        <div class="alert alert-info" ng-if="['STATIC', 'K8S'].includes($ctrl.infra.type)">
            These settings control the activity server to which the API nodes will send statistics for monitoring. Usually, this is the Deployer.
        </div>
        <div class="alert" ng-if="['STATIC', 'K8S'].includes($ctrl.infra.type)">
            Changing these settings requires updating the deployments handled by this infrastructure.
        </div>

        <div class="control-group">
            <label class="control-label">Mode</label>
            <div class="controls"
                toggle="tooltip-left">
                <select dku-bs-select
                        class="form-control"
                        ng-model="$ctrl.activityMonitoringSettings.mode"
                        layout="list"
                        options-descriptions="modeDescriptions">
                    <option value="PULL_FROM_API_NODES" ng-if="$ctrl.infra.type=='STATIC'">Deployer pulls activity from API nodes</option>
                    <option value="PULL_FROM_API_NODES" ng-if="$ctrl.infra.type=='SAGEMAKER'">Pull activity from CloudWatch</option>
                    <option value="PULL_FROM_API_NODES" ng-if="$ctrl.infra.type=='VERTEX_AI'">Pull activity from Google Cloud Monitoring</option>
                    <option value="PULL_FROM_API_NODES" ng-if="$ctrl.infra.type=='AZURE_ML'">Pull activity from Azure Monitor</option>
                    <option value="PULL_FROM_API_NODES" ng-if="$ctrl.infra.type=='DATABRICKS'">Pull activity from Databricks</option>
                    <option value="PUSH_TO_DEPLOYER" ng-if="$ctrl.infra.type != 'DATABRICKS' && onlyForDataikerAdminWhenInCloud">API nodes push activity to Deployer</option>
                    <option value="PUSH_TO_DEPLOYER_AUTO" ng-if="deployerAndNodesDirectoryEnabled && $ctrl.infra.type != 'DATABRICKS' && !isCloud">API nodes push activity to Deployer - Auto</option>
                    <option value="DISABLED">Don't get activity</option>
                </select>
                <span class="help-inline"
                        ng-if="$ctrl.activityMonitoringSettings.mode=='PULL_FROM_API_NODES' && $ctrl.infra.type === 'SAGEMAKER'">
                    CloudWatch will be queried using the credentials defined in the Basic parameters tab.
                </span>
                <span class="help-inline"
                        ng-if="$ctrl.activityMonitoringSettings.mode=='PULL_FROM_API_NODES' && $ctrl.infra.type === 'VERTEX_AI'">
                    Google Cloud Monitoring will be queried using the credentials defined in the Basic parameters tab.
                </span>
                <span class="help-inline"
                        ng-if="$ctrl.activityMonitoringSettings.mode=='PULL_FROM_API_NODES' && $ctrl.infra.type=='AZURE_ML'">
                    Azure Monitor will be queried using the credentials defined in the Basic parameters tab.
                </span>
                <span class="help-inline"
                        ng-if="$ctrl.activityMonitoringSettings.mode=='PULL_FROM_API_NODES' && $ctrl.infra.type=='DATABRICKS'">
                    Databricks will be queried using the credentials defined in the Basic parameters tab.
                </span>
            </div>
        </div>
        <div class="control-group" ng-if="$ctrl.activityMonitoringSettings.mode === 'PUSH_TO_DEPLOYER'">
            <label for="endpoint" class="control-label">Deployer activity server endpoint *</label>
            <div class="controls"
                toggle="tooltip-left">
                <input id="endpoint"
                    type="text"
                    ng-model="$ctrl.activityMonitoringSettings.activityServerAPIURL"
                    placeholder="http://host:port/public/api/admin/monitoring/activity-metrics"/>
                <button class="btn btn--secondary"
                        title="Use current URL"
                        ng-click="autofillDeployerActivityEndpointUrl()">
                    <i class="icon-magic" ></i>
                </button>
                <span class="help-inline">URL (http:// form) of the activity server endpoint (usually, this Deployer). Make sure it is reachable from the API nodes.</span>
            </div>
        </div>
        <div class="control-group" ng-if="$ctrl.activityMonitoringSettings.mode === 'PUSH_TO_DEPLOYER'">
            <label for="apikey" class="control-label">Deployer API key *</label>
            <div class="controls">
                <global-api-key-input
                    api-key="$ctrl.activityMonitoringSettings.apiKey",
                    new-key-label="newKeyLabel",
                    new-key-description="newKeyDescription">
                </global-api-key-input>
                <span class="help-inline">An admin API key valid on the remote Activity server (usually, this Deployer)</span>
            </div>
        </div>
        <div class="control-group" ng-if="$ctrl.activityMonitoringSettings.mode === 'PUSH_TO_DEPLOYER'">
            <label for="allSsl" class="control-label">Trust all certificates</label>
            <div class="controls"
                toggle="tooltip-left">
                <input id="allSsl" type="checkbox"
                    ng-model="$ctrl.activityMonitoringSettings.trustAllSSLCertificates"/>
            </div>
        </div>
    </div>
    `,

    controller: function($scope, $controller, $rootScope, DataikuAPI, INFRA_TYPES) {
        const $ctrl = this;

        angular.extend($ctrl, $controller('_DeployerBaseController', {
            $scope: $scope
        }));

        // Hackish : to check for the existence of a nodes directory, we test to see if the deployer is registered in the nodes directory.
        // only use this in a context where you already know that the deployer is active.
        $scope.deployerAndNodesDirectoryEnabled = $rootScope.appConfig.nodesDirectoryManagedDeployerServer || $rootScope.appConfig.nodesDirectoryManagedDeployerClient;

        $ctrl.$onInit = function () {
            $scope.modeDescriptions = getModeDescriptions($ctrl.infra.type, $scope.deployerAndNodesDirectoryEnabled, $scope.isCloud);
            $scope.newKeyLabel = "Monitoring-" + $ctrl.infra.id;
            $scope.newKeyDescription = "Used by " + INFRA_TYPES[$ctrl.infra.type] + " nodes of infrastructure " + $ctrl.infra.id + " to send activity metrics to this Deployer";
        }

        const getModeDescriptions = function(infraType, deployerAndNodesDirectoryEnabled, isCloud) {
            const modeDescriptions = [];

            /* PULL_FROM_API_NODES description */
            if (infraType == 'STATIC') {
                modeDescriptions.push('The Deployer directly queries the API nodes');
            } else if (['SAGEMAKER', 'VERTEX_AI', 'AZURE_ML', 'DATABRICKS'].includes((infraType))) {
                modeDescriptions.push('The Deployer directly fetches the clouds metrics');
            }

            /* PUSH_TO_DEPLOYER and PUSH_TO_DEPLOYER_AUTO description */
            if (infraType != 'DATABRICKS' && !isCloud) {
                modeDescriptions.push('The API nodes send metrics to the deployer. Manual configuration of the deployer\'s address and access');
                if (deployerAndNodesDirectoryEnabled) {
                    modeDescriptions.push('The API nodes send metrics to the deployer. Deployer\'s address and access auto-configured by Fleet Manager');
                }
            }

            /* DISABLED description */
            modeDescriptions.push('Activity won\'t be available for this infrastructure\'s deployments');

            return modeDescriptions;
        }

        $scope.autofillDeployerActivityEndpointUrl = function() {
            DataikuAPI.apideployer.infras.getDkuBackendExtUrl($ctrl.infra.type === 'K8S').success(function (url) {
                const urlBase = url ? url : urlWithProtocolAndHost();
                $ctrl.activityMonitoringSettings.activityServerAPIURL = urlBase + '/public/api/admin/monitoring/activity-metrics';
            }).error(setErrorInScope.bind($scope));
        };
    }
});

})();