(function(){
    'use strict';

    var app = angular.module('dataiku.flow.runtime');

    /** Directive job regular status */
    app.directive('jobStatusGraph', function ($stateParams, DataikuAPI, ContextualMenu, FlowGraph, FlowGraphSelection, JobFlowGraphSelection, FlowGraphHighlighting, StateUtils, GraphZoomTrackerService, FlowZoneMoveService) {
        return {
            controller: function ($scope, $element, $attrs) {
                const isPreview = $attrs.preview !== undefined;
                // the number of nodes in the flow is greater than BIG_FLOW_THRESHOLD defined in the backend
                $scope.isBigFlow = false;
                // force draw regardless of optimizations (e.g. not drawing big flows)
                $scope.forceDraw = false;

                let graphData, jobGraph, jobId;
                let collapsedZones = [];

                // Returns the targets of the given activity
                function getTargetsByRecipeName(recipeName) {
                    let targets = [];
                    let activities = $scope.getJobAllActivities();
                    Object.keys(activities).forEach(function(activityKey) {
                        let activity = activities[activityKey];
                        if (graphVizEscape('recipe_' + activity.recipeName) === recipeName) {
                            targets = targets.concat(activity.targets);
                        }
                    });
                    return targets;
                }

                function fadeOutActivity(recipeName, node) {
                    d3.select(node).classed("fade-out--no-filter", true);
                    getTargetsByRecipeName(recipeName).forEach(target => {
                        const datasetId = graphVizEscape(`dataset_${target.projectKey}.${target.datasetName}`);
                        const managedFolderId = graphVizEscape(`managedfolder_${target.projectKey}.${target.datasetName}`);
                        const savedModelId = graphVizEscape(`savedmodel_${target.projectKey}.${target.datasetName}`);
                        d3.selectAll(`
                            #flow-graph [data-id="${datasetId}"], #flow-graph [data-node-id="${datasetId}"],
                            #flow-graph [data-id="${managedFolderId}"], #flow-graph [data-node-id="${managedFolderId}"],
                            #flow-graph [data-id="${savedModelId}"], #flow-graph [data-node-id="${savedModelId}"]
                        `).classed("fade-out--no-filter", true);
                    });
                }

                function setStatus(element, classes, value) {
                    let statusValue = document.createElement('span');
                    let statusIcon =  document.createElement('i');
                    statusIcon.className = 'node-status__icon ' + classes;
                    statusValue.textContent = value;
                    statusValue.className = 'node-status__value';
                    element.appendChild(statusIcon);
                    element.appendChild(statusValue);
                }

                function setGraphData(serializedGraph) {
                    if (!serializedGraph) return;
                    $scope.nodesGraph = serializedGraph;
                    if ($scope.zonesManualPositioning) {
                        FlowZoneMoveService.updateZonesDimensions($scope.nodesGraph, collapsedZones);
                    }
                    FlowGraph.set($scope.nodesGraph);
                    $scope.nodesGraph.nodesOnGraphCount = Object.keys(serializedGraph.nodes).length;
                    $scope.isFlowEmpty = !$scope.nodesGraph.nodesOnGraphCount;
                }

                function getGraph() {
                    let jobElements = [];
                    let jobAllActivities = $scope.getJobAllActivities();

                    Object.keys(jobAllActivities).forEach(function(activityKey) {
                        let activity = jobAllActivities[activityKey];
                        activity.recipes.forEach(recipe => jobElements.push(graphVizEscape('recipe_' + recipe.name)));
                        activity.sources.forEach(function(source) {
                            jobElements.push(graphVizEscape('dataset_' + source.projectKey + "." + source.datasetName));
                            jobElements.push(graphVizEscape('managedfolder_' + source.projectKey + "." + source.datasetName));
                            jobElements.push(graphVizEscape('savedmodel_' + source.projectKey + "." + source.datasetName));
                            jobElements.push(graphVizEscape('modelevaluationstore_' + source.projectKey + "." + source.datasetName));
                            jobElements.push(graphVizEscape('retrievableknowledge_' + source.projectKey + "." + source.datasetName));
                        });
                        activity.targets.forEach(function (target) {
                            jobElements.push(graphVizEscape('dataset_' + target.projectKey + "." + target.datasetName));
                            jobElements.push(graphVizEscape('managedfolder_' + target.projectKey + "." + target.datasetName));
                            jobElements.push(graphVizEscape('savedmodel_' + target.projectKey + "." + target.datasetName));
                            jobElements.push(graphVizEscape('modelevaluationstore_' + target.projectKey + "." + target.datasetName));
                            jobElements.push(graphVizEscape('retrievableknowledge_' + target.projectKey + "." + target.datasetName));
                        });
                        // For backward compatibility with jobs created by a version of DSS before 4.1.2
                        if (activity.recipeName) {
                            jobElements.push(graphVizEscape('recipe_' + activity.recipeName));
                        }
                    });

                    return jobElements;
                }

                $scope.$emit('setResizeStrategy', 'highlight'); // resizing around highlighted nodes

                function setStatusOnGraph() {
                    $scope.svg = $element.find('svg');
                    $scope.jobStatus && $scope.svg && $.each($scope.jobStatus.stateByGraphNodeId, function (nodeRealId, jobStatusState) {
                        const nodeId = FlowGraph.nodesIdFromRealId(nodeRealId)[0]; // recipes belong to at most one zone
                        let svgDOM = $($scope.svg).get(0);
                        let node = svgDOM && svgDOM.querySelector(`g[data-id="${nodeId}"]`);
                        let flowNode = FlowGraph.node(nodeId);
                        let nodeTotem = node && node.querySelector('.node-totem span');
                        let hasMultipleStatus = jobStatusState.done + jobStatusState.running + jobStatusState.failed + jobStatusState.notStarted + jobStatusState.aborted + jobStatusState.warning + jobStatusState.skipped > 1;
                        let isJobFinished = $scope.jobStatus && $scope.jobStatus.baseStatus
                            && $scope.jobStatus.baseStatus.state != 'RUNNING'
                            && $scope.jobStatus.baseStatus.state != 'NOT_STARTED'
                            && $scope.jobStatus.baseStatus.state != 'WAITING_CONFIRMATION';

                        if (!node) {
                            return;
                        }

                        // Dynamically remove "never-built-computable" style where needed (successor of done recipes)
                        if (['DONE', 'WARNING'].includes(jobStatusState.state)) {
                            node.setAttribute('data-state', jobStatusState.state);
                            if (flowNode.successors) {
                                flowNode.successors.forEach(s => {
                                    let nodeElt = FlowGraph.d3NodeWithId(s);
                                    window.nodeElt = nodeElt;
                                    if (nodeElt) {
                                        $(nodeElt[0]).find('.never-built-computable').removeClass('never-built-computable')
                                    }
                                })
                            }
                        }

                        // Add global state and fade non-builded stuff if not in preview mode
                        switch (jobStatusState.state) {
                            case 'DONE':
                                nodeTotem.className += ' node-totem__status-icon icon-dku-success text-success';
                                nodeTotem.setAttribute('title', 'DONE');
                                break;
                            case 'WARNING':
                                nodeTotem.className += ' node-totem__status-icon icon-dku-warning text-warning';
                                nodeTotem.setAttribute('title', 'WARNING');
                                break;
                            case 'FAILED':
                                nodeTotem.className += ' node-totem__status-icon icon-dku-error text-error';
                                nodeTotem.setAttribute('title', 'FAILED');
                                break;
                            case 'NOT_STARTED':
                                nodeTotem.className += ' node-totem__status-icon icon-dku-queued text-weak';
                                nodeTotem.setAttribute('title', 'NOT STARTED');
                                if (!isPreview) fadeOutActivity(nodeRealId, node);
                                break;
                            case 'RUNNING': {
                                nodeTotem.className += ' node-totem__status-icon';
                                // Make the icon spin only if the job is not finished. If job is finished do not spin because it means the activity has only partially run.
                                if (!isJobFinished) {
                                    // nodeTotem.className += ' icon-spin'; DISABLED for now, nasty animation bug in chrome svg, does not work at all
                                    nodeTotem.setAttribute('title', 'RUNNING');
                                } else {
                                    nodeTotem.setAttribute('title', 'PARTIALLY RAN');
                                }
                                let iconBar = document.createElement('span');
                                iconBar.className = 'dku-loader';
                                nodeTotem.appendChild(iconBar);
                                break;
                            }
                            case 'ABORTED':
                                nodeTotem.className += ' node-totem__status-icon icon-dku-pause text-debug';
                                nodeTotem.setAttribute('title', 'ABORTED');
                                if (!isPreview) fadeOutActivity(nodeRealId, node);
                                break;
                            case 'SKIPPED':
                                nodeTotem.className += ' node-totem__status-icon icon-step-forward text-debug';
                                nodeTotem.setAttribute('title', 'SKIPPED');
                                if (!isPreview) fadeOutActivity(nodeRealId, node);
                                break;
                        }

                        // Add status counts
                        if (hasMultipleStatus) {
                            let nodeStatus = makeSVG('foreignObject', {
                                x: -100,
                                y: 110,
                                width: 300,
                                height: 42,
                                class: 'node-status'
                            });

                            nodeTotem.parentElement.parentElement.appendChild(nodeStatus);

                            if (jobStatusState.failed > 0) {
                                setStatus(nodeStatus, 'icon-dku-error text-error', jobStatusState.failed);
                            }

                            if (jobStatusState.warning > 0) {
                                setStatus(nodeStatus, 'icon-dku-warning text-warning', jobStatusState.warning);
                            }

                            if (jobStatusState.notStarted > 0) {
                                setStatus(nodeStatus, 'icon-dku-queued text-weak', jobStatusState.notStarted);
                            }

                            if (jobStatusState.skipped > 0) {
                                setStatus(nodeStatus, 'icon-step-forward text-debug', jobStatusState.skipped);
                            }

                            if (jobStatusState.aborted > 0) {
                                setStatus(nodeStatus, 'icon-dku-pause text-debug', jobStatusState.aborted);
                            }

                            if (jobStatusState.done > 0) {
                                setStatus(nodeStatus, 'icon-dku-success text-success', jobStatusState.done);
                            }
                        }
                    });
                }

                function reset() {
                    d3.select('#flow-graph:not(.disable-has-selection)').classed('has-selection', true);
                    d3.selectAll('#flow-graph .selected').classed('selected', false);
                    d3.selectAll('#flow-graph .fade-out').classed('fade-out', false);
                    d3.selectAll('#flow-graph .filter-remove').classed('filter-remove', false).classed('fade-out--no-filter', false);
                    d3.selectAll('#flow-graph .highlight').classed('highlight', false).classed('fade-out--no-filter', false); // removes fade-out on built nodes in a running job preview
                    d3.selectAll('#flow-graph .zone_cluster.clusterHighlight').classed('clusterHighlight', false);
                    d3.selectAll('#flow-graph .zone_cluster .zone_header [ng-click]:not([ng-click^="toggleZoneCollapse("])').remove();
                    d3.selectAll('#flow-graph .node.out-of-focus').classed('out-of-focus', false);
                    d3.selectAll('#flow-graph .node-totem span').attr('class', null);
                    d3.selectAll('#flow-graph .node-status').remove();
                }

                /**
                 * Highlight given nodes in the flow graph.
                 * Works both on graphs with and without zones.
                 * @param {string[]} nodesRealId
                 */
                function highlight(nodesRealId) {
                    if (!nodesRealId) return;
                    nodesRealId.forEach(nodeRealId => {
                        const zonesId = FlowGraph.zonesIdFromRealId(nodeRealId);
                        zonesId.forEach(zoneId => {
                            FlowGraphHighlighting.highlightZoneCluster(d3.select(`#flow-graph .zone_cluster[data-id="zone_${zoneId}"]`).classed('selected', true).node());
                            zonesId.filter(zId => zoneId !== zId).forEach(zId => d3.select(`#flow-graph .edge[data-from="zone_${zoneId}"][data-to="zone_${zId}"]`)
                                .classed('disable-fade-out', true)); // highlighting zone edges break highlight resize strategy - using a filter class instead to disable fade out
                        });
                        const nodesId = FlowGraph.nodesIdFromRealId(nodeRealId);
                        nodesId.forEach(nodeId => d3.selectAll(`#flow-graph .node[data-id="${nodeId}"], #flow-graph .edge[data-to="${nodeId}"]`)
                            .classed('highlight', true).classed('fade-out', false));
                    });
                    d3.selectAll('#flow-graph .zone_cluster:not(.clusterHighlight)')
                        .attr('style', null).select('.zone_header').attr('style', null);
                    d3.selectAll('#flow-graph .node:not(.highlight), #flow-graph .edge:not(.highlight):not(.disable-fade-out)')
                        .classed('filter-remove', true).classed('out-of-focus', true);
                    d3.selectAll('#flow-graph .edge:not(.highlight):not(.disable-fade-out)')
                        .classed('fade-out--no-filter', true);
                    d3.selectAll('#flow-graph .edge.disable-fade-out')
                        .classed('disable-fade-out', false);
                }

                function draw(forceRedrawGraph = false) {
                    if (!graphData) return;
                    if (!$scope.forceDraw && ($scope.isBigFlow || !jobGraph)) return;
                    if (forceRedrawGraph) $scope.$emit('drawGraph', true, true);
                    reset();
                    highlight(jobGraph);
                    setStatusOnGraph();
                    $scope.$emit('setResizeStrategy', 'highlight'); // resizing around highlighted nodes
                }

                function fetchGraph(forceRedrawGraph = false) {
                    DataikuAPI.flow.recipes.getGraph($stateParams.projectKey, true, $scope.projectSummary.zonesGraphForJobs, null, collapsedZones).success(data => {
                        $scope.zonesManualPositioning = data.zonesManualPositioning.projectSettingsValue;
                        $scope.canMoveZones = false;

                        const firstDraw = !graphData;
                        graphData = data.serializedFilteredGraph.serializedGraph;
                        setGraphData(graphData);
                        jobGraph = getGraph();
                        draw(forceRedrawGraph || firstDraw);
                    }).error(setErrorInScope.bind($scope));
                }

                $scope.drawJobGraph = function (shouldFetchGraph = false, forceDrawBigGraph = false, forceRedrawGraph = false) {
                    if (forceDrawBigGraph) $scope.forceDraw = true;
                    if (!shouldFetchGraph) return draw(forceRedrawGraph);
                    if ($scope.forceDraw) return fetchGraph(forceRedrawGraph);
                    // quick call to check the number of potential nodes on the graph
                    DataikuAPI.taggableObjects.countAccessibleObjects($stateParams.projectKey).success(accessibleObjectsCount => {
                        $scope.isBigFlow = accessibleObjectsCount.isBigFlow;
                        // not drawing big flow (et Oli... wait what?) by default but the user can force drawing with a button
                        if (!$scope.isBigFlow) fetchGraph(forceRedrawGraph);
                    }).error(setErrorInScope.bind($scope));
                };

                $scope.toggleZoneCollapse = (zones) => {
                    zones.forEach(zone => {
                        const collapsedZoneIndex = collapsedZones.indexOf(zone.id);
                        if (collapsedZoneIndex === -1) collapsedZones.push(zone.id);
                        else collapsedZones.splice(collapsedZoneIndex, 1);
                    })
                    $scope.$emit('disableNextFlowResize'); // cancels next resizing around highlighted nodes
                    $scope.drawJobGraph(true, false, true);
                }

                $scope.$watch("jobStatus", function (jobStatus) {
                    // Disable flow zoom remembering while navigating on jobs flows
                    GraphZoomTrackerService.disable();
                    // Do not re-compute job graph if already known (for running jobs)
                    const shouldFetchGraph = jobId !== jobStatus.baseStatus.def.id;
                    jobId = jobStatus.baseStatus.def.id;
                    // Temporarily disable resizing while a job is running to let the user navigate the job flow
                    if (jobStatus.baseStatus.state === 'RUNNING') {
                        $scope.$emit('disableNextFlowResize');
                    }
                    $scope.drawJobGraph(shouldFetchGraph);
                    FlowGraphSelection.setSingleSelectionStrategy(JobFlowGraphSelection.singleSelectionStrategy($scope));
                    FlowGraphSelection.setMultiSelectionStrategy(JobFlowGraphSelection.multiSelectionStrategy($scope));
                });

                // Come back to default flow selection strategy when
                $scope.$on('$destroy', _ => {
                    FlowGraphSelection.resetSelectionStrategies();
                    $scope.$emit('setResizeStrategy', '');
                    reset();
                    // too soon to re-enable zoom tracking (zoom would be persisted)
                });
            },

            link: function($scope) {
                $scope.onItemDblClick = function(item, evt){
                    let destUrl = StateUtils.href.node(item);
                    fakeClickOnLink(destUrl, evt);
                };
                $scope.onItemContextualMenu = function(){
                    return false;
                };
                $scope.onContextualMenu = function(item, evt) {
                    ContextualMenu.prototype.closeAny();
                    return true;
                };
                $scope.$watch("rightColumnItem", function(nv, ov) {
                    if (!nv) return;
                    $scope.selectedItemData = $scope.jobStatus.stateByGraphNodeId[nv.id];
                });
                $scope.$watch("jobStatus", function(nv, ov){
                    if (nv && $scope.rightColumnItem) {
                        $scope.selectedItemData = $scope.jobStatus.stateByGraphNodeId[$scope.rightColumnItem.id];
                    }
                });
            }
        };
    });
})();
