(function() {
'use strict';

const app = angular.module('dataiku.controllers');

app.controller('ProjectVersionControlController', function($scope, TopNav, DataikuAPI, Dialogs, FullGitSupportService, 
                                                           $stateParams, $state, $filter, WT1, CreateModalFromTemplate, 
                                                           ActivityIndicator, LocalStorage){
    const PAGE_SIZE = 20;
    TopNav.setLocation(TopNav.TOP_MORE, "version-control", "NONE", null);
    TopNav.setItem(TopNav.ITEM_PROJECT, $stateParams.projectKey);

    function warnBeforeGlobalOperation(opName, opText){
        return Dialogs.confirm($scope, opName,
            opText + " will not have any impact on any data. "+
            " This could lead to some data becoming orphan, or to some datasets pointing to stale data");
    }

    function getCountUntilText(countUntil) {
        let text = `${countUntil} commit`;
        if (countUntil > 1) {
            text += 's';
        }
        text += ' after'
        return text;
    }

    $scope.getGitFullStatus = function() {
        return FullGitSupportService.getFullStatus($scope, 
            DataikuAPI.projects.git.getFullStatus($stateParams.projectKey));
    };

    $scope.gitBranchesByType = {};
    $scope.gitBranchesByTypeCount = 0;
    $scope.gitBranchesByTypeFiltered = {};
    $scope.gitBranchesByTypeFilteredCount = 0;
    $scope.getGitBranches = function() {
         DataikuAPI.projects.git.listBranchesByType($stateParams.projectKey)
            .then(function(resp) {
                $scope.gitBranchesByTypeCount = 0;
                // we keep branches sorted by type
                $scope.gitBranchesByType = Object.keys(resp.data).reduce((branchesByType, type) => {
                    resp.data[type].sort();
                    branchesByType[type] = resp.data[type];
                    $scope.gitBranchesByTypeCount += branchesByType[type].length;
                    return branchesByType;
                }, {});
                $scope.gitBranchesByTypeFiltered = $scope.gitBranchesByType;
                $scope.gitBranchesByTypeFilteredCount = $scope.gitBranchesByTypeCount;
            }, setErrorInScope.bind($scope));
    };

    $scope.filterBranches = function(query) {
        $scope.gitBranchesByTypeFilteredCount = 0;
        $scope.gitBranchesByTypeFiltered = Object.keys($scope.gitBranchesByType).reduce((filteredBranchesByType, type) => {
            filteredBranchesByType[type] =  $filter("filter")($scope.gitBranchesByType[type], query);
            $scope.gitBranchesByTypeFilteredCount += filteredBranchesByType[type].length;
            return filteredBranchesByType;
        }, {});
    };

    $scope.formatTrackingCount = function(count) {
        return count != null ? count : "-";
    };

    $scope.fmtProjectBranch = function(projectBranch) {
        if (!projectBranch) return '';
        const [,, project, branch] = projectBranch.split('/');
        return branch + ' (' + project + ')';
    };

    $scope.isCurrentProjectBranch = function(projectBranch) {
        if (!projectBranch) return false;
        const [,, project] = projectBranch.split('/');
        return project && project === $stateParams.projectKey;
    }


    $scope.switchToBranch = function(branchName, branchType) {
        let toProject, toBranch, checkoutMode;
        if (branchType === 'LOCAL') {
            toBranch = branchName;
            checkoutMode = 'USE_CURRENT_PROJECT';
        } else if (branchType === 'PROJECT') {
            [toProject, toBranch] = branchName.split('/').slice(2);
            checkoutMode = 'GO_TO_ALTERNATE_PROJECT';
        } else { //branchType === 'REMOTE'
            //by default duplicate the project
            toBranch = branchName.split('/').pop();
            checkoutMode = "DUPLICATE_PROJECT";
            const dupProjectBranch = $scope.gitBranchesByType.PROJECT.find(b => b.endsWith('/' + toBranch));
            if (dupProjectBranch) {
                //found a duplicated project on this branch
                [toProject, toBranch] = dupProjectBranch.split('/').slice(2);
                checkoutMode = "GO_TO_ALTERNATE_PROJECT";
            } else if ($scope.gitBranchesByType.LOCAL && $scope.gitBranchesByType.LOCAL.includes(toBranch)) {
                //found a local matching branch
                checkoutMode = 'USE_CURRENT_PROJECT';
            }
        }
        CreateModalFromTemplate("/templates/projects/git/checkout-branch-modal.html", $scope, "ProjectCheckoutBranchController", (newScope) => {
            newScope.checkoutModel.createBranch = false;
            newScope.checkoutModel.targetBranchName = toBranch;
            newScope.checkoutModel.alternateProjectKey = toProject;
            newScope.checkoutModel.checkoutMode = checkoutMode;
        });
        WT1.event("projects-git-switch-branch");
    };

    $scope.modalCreateBranch = function(wantedBranch) {
        CreateModalFromTemplate("/templates/projects/git/checkout-branch-modal.html", $scope, "ProjectCheckoutBranchController", function (newScope) {
            newScope.checkoutModel.createBranch = true;
            newScope.checkoutModel.targetBranchName = wantedBranch || "";
            newScope.checkoutModel.checkoutMode = "DUPLICATE_PROJECT";
        });
        WT1.event("projects-git-create-branch", {source: "branch"});
    };

    $scope.createBranchFromCommit = function(commitId) {
        DataikuAPI.projects.git.getDSSVersionForASpecificCommit($stateParams.projectKey, commitId).success(function(dssVersionForCommitResponse){ 
            CreateModalFromTemplate("/templates/projects/git/checkout-branch-modal.html", $scope, "ProjectCheckoutBranchController", function (newScope) {
                newScope.checkoutModel.createBranch = true;
                newScope.dssVersionForCommitResponse = dssVersionForCommitResponse;
                newScope.checkoutModel.targetBranchName = "";
                newScope.checkoutModel.commitId = commitId;
                newScope.checkoutModel.checkoutMode = "DUPLICATE_PROJECT";
            });
            WT1.event("projects-git-create-branch", {source: "commit"});
        }).error(setErrorInScope.bind($scope));
      
    };

    $scope.modalDeleteLocalBranches = function() {
        const callback = function(modalScope, branchesToDelete, deleteOptions) {
            DataikuAPI.projects.git.deleteBranches($stateParams.projectKey, branchesToDelete, deleteOptions).then(function() {
                $scope.getGitBranches();
                modalScope.dismiss();
                ActivityIndicator.success(`${branchesToDelete.length} Branch${branchesToDelete.length>1?'es':''} deleted`, 5000);
                WT1.event("projects-git-delete-branches");
            }, setErrorInScope.bind(modalScope));
        };
        $scope.gitBranches = $scope.gitBranchesByType.LOCAL;
        FullGitSupportService.deleteBranches($scope, callback);
    };

    $scope.modalFetch = function () {
        FullGitSupportService.fetch($scope, DataikuAPI.projects.git.fetch($stateParams.projectKey));
        WT1.event("projects-git-fetch");
    };

    $scope.modalPull = function() {
        warnBeforeGlobalOperation("Pulling project", "Pulling updates from remote").then(function(){
            FullGitSupportService.pull($scope, DataikuAPI.projects.git.pull($stateParams.projectKey, $scope.gitStatus.remoteOrigin.name, $scope.gitStatus.currentBranch));
            WT1.event("projects-git-pull");
        });
    };

    $scope.modalPush = function() {
        FullGitSupportService.push($scope, DataikuAPI.projects.git.push($stateParams.projectKey));
        WT1.event("projects-git-push");
    };

    $scope.modalReset = function() {
        warnBeforeGlobalOperation("Dropping changes", "Dropping changes").then(function(){
            CreateModalFromTemplate("/templates/plugins/development/git/reset-modal.html", $scope, "ProjectGitResetController");
        });
    };

    $scope.modalAddOrEditRemote = function() {
        const callback = function(remoteName, newURL) {
            DataikuAPI.projects.git.setRemote($stateParams.projectKey, remoteName, newURL).then(function() {
                $scope.getGitFullStatus();
                ActivityIndicator.success("Remote saved", 5000);
                WT1.event("projects-git-set-remote");
            }, setErrorInScope.bind($scope));
        };
        FullGitSupportService.editRemote($scope, callback);
    };

    $scope.modalRemoveRemote = function() {
        const callback = function(remoteName) {
            DataikuAPI.projects.git.removeRemote($stateParams.projectKey, remoteName).then(function() {
                $scope.getGitFullStatus();
                $scope.getGitBranches();
                ActivityIndicator.success("Remote removed", 5000);
                WT1.event("projects-git-remove-remote");
            }, setErrorInScope.bind($scope));
        };
        FullGitSupportService.removeRemote($scope, callback);
    };

    $scope.modalCommit = function() {
        CreateModalFromTemplate("/templates/plugins/development/git/commit-modal.html", $scope, "ProjectGitCommitController");
    };

    $scope.addHEADTag = function() {
        FullGitSupportService.addTagModal($scope, 'HEAD', tag => DataikuAPI.git.addTag($stateParams.projectKey, 'PROJECT', $stateParams.projectKey, tag.reference, tag.name, tag.message))
                             .then(() => $scope.loadLogFromStart());
    };

    $scope.getResetModes = function() {
        let modes = [];

        if ($scope.projectSummary && $scope.projectSummary.commitMode !== 'AUTO') {
            modes.push('HEAD');
        }

        if ($scope.gitStatus.hasRemoteOrigin && $scope.gitStatus.hasTrackingCount) {
            modes.push('UPSTREAM');
        }

        return modes;
    };

    $scope.needsExplicitCommit = function(){
        return $scope.projectSummary && $scope.projectSummary.commitMode !== 'AUTO';
    };

    $scope.gitBranchesLoaded = false;
    const updatePermissions = function() {
        if ($scope.projectSummary) {
            $scope.canChangeRemote = $scope.projectSummary.isProjectAdmin;
            $scope.canChangeBranch = $scope.projectSummary.isProjectAdmin;
            $scope.canUpdateContent = $scope.projectSummary.isProjectAdmin;
            if (!$scope.gitBranchesLoaded && $scope.canChangeBranch) {
                $scope.getGitBranches();
                $scope.gitBranchesLoaded = true;
            }
        } else {
            $scope.canChangeRemote = false;
            $scope.canChangeBranch = false;
            $scope.canUpdateContent = false;
        }
    };

    $scope.getGitFullStatus();
    updatePermissions();
    $scope.$watch('projectSummary', updatePermissions);

    $scope.projectKey = $stateParams.projectKey;

    $scope.loadMoreUntil = function (doneCallback) {
        if ($scope.hasMoreUntil && !$scope.loading) {
            $scope.loading = true;
            $scope.scrollToCommitId = null;
            if (!$scope.logEntries || $scope.logEntries.length === 0) {
                $scope.hasMoreUntil = false;
                $scope.countUntil = 0;
                $scope.countToLoadUntil = 0;
                $scope.countUntilText = null;
                $scope.loading = false;
                return;
            }
            const until = $scope.logEntries[0].commitId;
            DataikuAPI.git.getObjectLogUntil($stateParams.projectKey, 'PROJECT', $stateParams.projectKey, $scope.branch, until, PAGE_SIZE, 0).success(function (data) {
                $scope.logEntries = data.logEntries.concat($scope.logEntries);
                $scope.countUntil = data.countUntil;
                $scope.countToLoadUntil = $scope.countUntil > PAGE_SIZE ? PAGE_SIZE : $scope.countUntil;
                $scope.hasMoreUntil = $scope.countUntil ? true : false;
                $scope.countUntilText = getCountUntilText($scope.countUntil);
                $scope.loading = false;
                if (doneCallback) {
                    doneCallback();
                }
            }).error(function(e) {
                $scope.loading = false;
                setErrorInScope.bind($scope);
            });
        }
    };

    $scope.loadMoreSince = function (doneCallback) {
        if ($scope.hasMoreSince && !$scope.loading) {
            $scope.scrollToCommitId = null;
            $scope.loading = true;
            DataikuAPI.git.getObjectLogSince($stateParams.projectKey, 'PROJECT', $stateParams.projectKey, $scope.branch, $scope.nextCommit, PAGE_SIZE).success(function (data) {
                $scope.logEntries = $scope.logEntries.concat(data.logEntries);
                $scope.nextCommit = data.nextCommit;
                if (!$scope.nextCommit) {
                    $scope.hasMoreSince = false;
                }
                $scope.loading = false;
                if (doneCallback) {
                    doneCallback();
                }
            }).error(function(e) {
                $scope.loading = false;
                setErrorInScope.bind($scope);
            });
        }
    };

    $scope.loadLogFromStart = function() {
        $scope.nextCommit = null;
        $scope.countUntil = null;
        $scope.logEntries = [];
        $scope.hasMoreSince = true;
        $scope.hasMoreUntil = false;
        $scope.scrollToCommitId = null;
        $scope.highlightCommitId = null;
        $scope.branch = null;
        $scope.loadMoreSince();
    };

    $scope.loadLogFromStartWithCommitAndBranch = function() {
        $scope.logEntries = [];
        $scope.hasMoreSince = true;
        $scope.hasMoreUntil = true;
        // If we have a commit id, load all commits up to this commit (plus the next 20 ones, as usual)
        $scope.nextCommit = $stateParams.commitId;
        $scope.highlightCommitId = $stateParams.commitId;
        $scope.commitId = $stateParams.commitId;
        $scope.requestedBranch = $stateParams.branch;

        $scope.branch = null;
        // try to load log from HEAD
        $scope.loadMoreSince(() => {
            if ($scope.logEntries.length > 0) {
                $scope.loadMoreUntil(() => {
                    $scope.scrollToCommitId = $stateParams.commitId;
                });
            } else if ($scope.requestedBranch) {
                // If unsuccessful, try again, this time from $stateParams.branch
                $scope.branch = $scope.requestedBranch;
                $scope.nextCommit = $stateParams.commitId;
                $scope.hasMoreSince = true;
                $scope.hasMoreUntil = true;
                $scope.loadMoreSince(() => {
                    $scope.loadMoreUntil(() => {
                        $scope.scrollToCommitId = $stateParams.commitId;
                    });
                });
            }
        });
    }

    $scope.openMergeRequest = mergeRequest => {
        $state.go('projects.project.version-control-merge', {mergeRequestId: mergeRequest.id});
    };

    $scope.openCreateMergeRequestModal = (defaultTitle, branchToMerge) => {
        FullGitSupportService
            .createMergeRequestModal($scope, 'PROJECT', $scope.projectKey, $scope.gitStatus.currentBranch, defaultTitle, branchToMerge)
            .then(
                $scope.openMergeRequest,
                angular.noop //ignore error
            );
    };

    $scope.mergeRequests = [];
    $scope.filteredMergeRequests = [];
    $scope.uiState = {
        filterQuery: '',
        filterWithMerged: !!LocalStorage.get('merge-requests-with-merged'),
        maxMergeRequests: 0 
    };
    const MAX_MERGE_REQUESTS_DISPLAYED = 8; 
    const comparableString = str => (str || '').trim().toLowerCase().normalize('NFD').replace(/\p{Diacritic}/gu, '');

    $scope.filterMergeRequests = () => {
        //filter merged
        let mergeRequests = $scope.uiState.filterWithMerged
            ? [].concat($scope.mergeRequests)
            : $scope.mergeRequests.filter(mr => mr.status !== 'MERGED');
        $scope.uiState.maxMergeRequests = mergeRequests.length;
        //filter query
        const qry = comparableString($scope.uiState.filterQuery);
        if (qry) {
            mergeRequests = mergeRequests.filter(mr => comparableString(mr.title).indexOf(qry) !== -1);
        }
        //compute last update
        mergeRequests.forEach(mr => mr.lastUpdate = mr.events.at(-1).creationDate);
        //sort by last update desc
        mergeRequests.sort((a,b) => b.lastUpdate - a.lastUpdate);
        $scope.filteredMergeRequests = mergeRequests.slice(0, MAX_MERGE_REQUESTS_DISPLAYED);
    };

    $scope.switchFilterWithMerged = () => {
        LocalStorage.set('merge-requests-with-merged', $scope.uiState.filterWithMerged);
        $scope.filterMergeRequests();
    };
    
    const statusOrder = {
        'OPENED': 1,
        'MERGED': 2,
        'DELETED': 3
      };
      
    $scope.loadMergeRequests = () => {
        DataikuAPI.git.mergeRequests
            .list($scope.projectKey)
            .then(
                resp => {
                    $scope.mergeRequests = resp.data.sort((a, b) => {
                        // First, compare by status
                        const statusComparison = statusOrder[a.status] - statusOrder[b.status];
                        if (statusComparison !== 0) {
                          return statusComparison;
                        }
                        // If statuses are the same, sort by ID in descending order
                        return b.id - a.id;
                      });
                    $scope.mergeRequests.sort((a, b) => a.title.localeCompare(b.title));
                    $scope.filterMergeRequests();
                },
                setErrorInScope.bind($scope)
            );
    };
    $scope.loadMergeRequests();

    if ($stateParams.commitId) {
        $scope.loadLogFromStartWithCommitAndBranch();
    } else {
        $scope.loadLogFromStart();
    }
});

app.controller("ProjectGitCommitController", function($scope, $stateParams, $filter, DataikuAPI, ActivityIndicator, $timeout, WT1) {
    DataikuAPI.git.prepareObjectCommit($stateParams.projectKey, 'PROJECT', $stateParams.projectKey).success(function (data) {
        $scope.preparationData = data;
    }).error(setErrorInScope.bind($scope));

    $scope.uiState = {
        activeTab: 'message',
        message: ''
    };

    $timeout(() => {
        // Magic happens here: if commitEditorOptions is defined too early, the textarea won't properly autofocus
        $scope.commitEditorOptions = {
            mode : 'text/plain',
            lineNumbers : false,
            matchBrackets : false,
            autofocus: true,
            onLoad : function(cm) {$scope.codeMirror = cm;}
        };
    }, 100);


    $scope.gitCommit = function() {
        DataikuAPI.git.commitObject($stateParams.projectKey, 'PROJECT', $stateParams.projectKey, $scope.uiState.message)
            .success(function () {
                ActivityIndicator.success('Changes successfully committed.');
                $scope.dismiss();
                $scope.getGitFullStatus();
                $scope.loadLogFromStart();
                WT1.event("projects-git-commit");
            }).error(setErrorInScope.bind($scope));
    };
});

app.controller("ProjectCheckoutBranchController", function($scope, $state, $stateParams, DataikuAPI, FutureWatcher, PathUtils, ProgressStackMessageBuilder, WT1, PromiseService, Dialogs, $q, ProjectFolderService) {
    function isProjectKeyAlreadyUsed(projectKey) {
        return projectKey && $scope.allProjectKeys && $scope.allProjectKeys.indexOf(projectKey) >= 0;
    }

    function isBranchNameAlreadyUsed(branchName) {
        return branchName && $scope.gitBranches && $scope.gitBranches.indexOf(branchName) >= 0;
    }

    function doesBranchNameContainsInvalidCharacters(branchName) {
        // must not have ASCII control characters (00-40 + DEL=177), space, tilde, caret, colon, at, star, question mark, left square bracket.
        return /[\000-\037\177 ~^:@*?[\\]/g.test(branchName);
    }

    function isBranchNameValid(branchName) {
        return branchName
            && !doesBranchNameContainsInvalidCharacters(branchName)
            // must not start with slash (/)
            && !/^\//g.test(branchName)
            // must not end with slash (/), dot (.) or ".lock"
            && !/(\/|\.|\.lock)$/g.test(branchName)
            // must not contain "/." or ".." or "//"
            && !/(\/\.|\.\.|\/\/)/g.test(branchName);
    }

    function sanitizeProjectKey(projectKey) {
        return projectKey.toUpperCase().replace(/\W+/g, "_").replace(/_+$/,'');
    }

    function fixProjectKey(newProjectKey) {
        if (!newProjectKey || !$scope.allProjectKeys) { // Not initialized or all project keys not loaded yet
            return;
        }
        const slug = sanitizeProjectKey(newProjectKey);
        let cur = slug;
        let i = 0;
        while (isProjectKeyAlreadyUsed(cur)) {
            cur = slug + "_" + (++i);
        }
        $scope.checkoutModel.projectKey = cur;
    }

    var abortHook = null;

    // Initialize scope variables

    DataikuAPI.projects.getExposedObjects($stateParams.projectKey).then((resp) => {
        $scope.sharedObjectsCount = resp.data.objects.filter(e => e.rules.length > 0).length;
    });

    if (!$scope.uiState) $scope.uiState = {};
    $scope.uiState.showInitialAdvancedOptions = false;
    $scope.uiState.showDuplicateAdvancedOptions = false;

    $scope.phase = 'INITIAL';
    $scope.checkoutModel = {
        // Initial phase
        createBranch: false,
        targetBranchName: undefined,
        checkoutMode: "USE_CURRENT_PROJECT",
        branchNameError: false, // true if the branch name already exists or is invalid (only used when creating a new branch)

        // Switch phase
        clearOutputDatasets: false,

        // Duplicate phase
        projectKey: sanitizeProjectKey($scope.projectSummary.projectKey),
        projectName: $scope.projectSummary.name,
        projectKeyAlreadyUsed: false
    };
    $scope.checkoutModel.projectKeyAlreadyUsed = isProjectKeyAlreadyUsed($scope.checkoutModel.projectKey);
    $scope.dupOptions = {
        exportAnalysisModels: true,
        exportSavedModels: true,
        exportModelEvaluationStores: false,
        exportGitRepository: $scope.projectSummary.canExportGitRepository,
        exportInsightsData: true,
        duplicationMode: 'UPLOADS_ONLY',
        exportUploads: true,
        exportEditableDatasets: true,
        exportAllInputDatasets: false,
        exportAllInputManagedFolders: false,
        exportAllDatasets: false,
        exportManagedFolders: false,
        exportKnowledgeBanks: false,
        exportPromptStudioHistories: false,
        exportNotebooksWithOutputs: true,
        targetProjectFolder: null, // will be initialized asynchronously
        targetProjectFolderId: '', // will be initialized asynchronously
    };

    // Define scope functions
    $scope.setDuplicationMode = function(mode) {
        $scope.dupOptions.duplicationMode = mode;
    };

    $scope.validateInitialPhase = function() {
        if ($scope.checkoutModel.createBranch && !isBranchNameValid($scope.checkoutModel.targetBranchName)) {
            $scope.checkoutModel.branchNameError = true;
            $scope.checkoutModel.branchNameErrorMessage = "This branch name is invalid.";
            return;
        }
        if ($scope.checkoutModel.checkoutMode === "GO_TO_ALTERNATE_PROJECT") {
            $scope.dismiss();
            $state.transitionTo("projects.project.home.regular", {projectKey : $scope.checkoutModel.alternateProjectKey});
        } else if ($scope.checkoutModel.checkoutMode === "DUPLICATE_PROJECT") {
            $scope.moveToDuplicatePhase();
        } else if (!$scope.checkoutModel.createBranch) {
            $scope.moveToSwitchPhase();
        } else {
            $scope.createBranch();
        }
    };

    $scope.moveToInitialPhase = function() {
        $scope.phase = "INITIAL";
        $scope.fatalAPIError = null;
        $scope.duplicateResponse = null;
    };

    $scope.moveToDuplicatePhase = function() {
        $scope.phase = "READY_TO_DUPLICATE";
        $scope.checkoutModel.projectKey = sanitizeProjectKey($scope.projectSummary.projectKey + "_" + $scope.checkoutModel.targetBranchName);
        $scope.checkoutModel.projectName = $scope.projectSummary.name + " (" + $scope.checkoutModel.targetBranchName + ")";
        fixProjectKey($scope.checkoutModel.projectKey);
    };

    $scope.moveToSwitchPhase = function() {
        $scope.phase = "READY_TO_SWITCH";
        DataikuAPI.projects.checkDeletability($stateParams.projectKey).success(function(data) {
            if (data.anyMessage) {
                $scope.hasDependencyWarnings = true;
                $scope.dependencyWarnings = { messages : data.messages };
            }
        });
     };

    ProjectFolderService.getDefaultFolderForNewProject().then((folder) => {
        $scope.dupOptions.targetProjectFolder = folder;
        $scope.dupOptions.defaultProjectFolderId = folder.id;
    }).catch(setErrorInScope.bind($scope));

    $scope.browse = folderIds => {
        // Use last id in path
        $scope.destination = PathUtils.makeNLNT(folderIds).split('/').pop();
        // browse-path expects a success-error promise so we need to wrap with qToHttp for now (catch() does not return a monkey-patched promise)
        return PromiseService.qToHttp(ProjectFolderService.getBrowseNode($scope.destination).catch(setErrorInScope.bind($scope)));
    };

    $scope.getProjectFolderName = item => item.name;
    $scope.canSelectFolder = item => item.canWriteContents;

    $scope.createBranch = function() {
        DataikuAPI.projects.git.createBranch($stateParams.projectKey, $scope.checkoutModel.targetBranchName, $scope.checkoutModel.commitId).then(function () {
            $state.reload();
            $scope.dismiss();
        }, setErrorInScope.bind($scope));
    };

    $scope.switchBranch = function () {
        $scope.duplicateResponse = null;
        DataikuAPI.projects.git.switchBranch($stateParams.projectKey, $scope.checkoutModel.targetBranchName, $scope.checkoutModel.clearOutputDatasets).then(function (result) {
            var parentScope = $scope.$parent;
            $scope.dismiss();
            const success = result.data.commandSucceeded;
            const hasWarnOrErrorMessage = result.data.messages && (result.data.messages.warning || result.data.messages.error);
            const hasInfoMessages = result.data.messages && result.data.messages.messages.length > 1;
            const hasLogMessages = result.data.log && result.data.log.lines && result.data.log.lines.length > 2; // If everything went smoothly, we get 2 lines of logs.
            if (!success || hasWarnOrErrorMessage || hasInfoMessages || hasLogMessages) {
                Dialogs.infoMessagesDisplayOnly(parentScope, "Switch branch result", result.data.messages, result.data.log, true).then(function() {
                    if (success) {
                        $state.reload();
                    }
                }, null);
            } else {
                $state.reload();
            }
        }, setErrorInScope.bind($scope));
    };

    $scope.duplicate = function() {
        $scope.duplicateResponse = null;
        $scope.phase = 'DUPLICATING';
        $scope.dupOptions.targetProjectKey = $scope.checkoutModel.projectKey;
        $scope.dupOptions.targetProjectName = $scope.checkoutModel.projectName;
        $scope.dupOptions.createBranch = $scope.checkoutModel.createBranch;
        $scope.dupOptions.targetBranchName = $scope.checkoutModel.targetBranchName;
        $scope.dupOptions.commitId = $scope.checkoutModel.commitId;
        $scope.dupOptions.targetProjectFolderId = $scope.dupOptions.targetProjectFolder ? $scope.dupOptions.targetProjectFolder.id : '';
        DataikuAPI.projects.startProjectDuplication(
            $scope.projectSummary.projectKey,
            $scope.dupOptions
        ).success(function(initialResponse){
            abortHook = function() {
                DataikuAPI.futures.abort(initialResponse.jobId).error(setErrorInScope.bind($scope));
            };
            FutureWatcher.watchJobId(initialResponse.jobId).success(function(data){
                abortHook = null;
                $scope.futureResponse = null;
                $scope.duplicateResponse = data.result;
                if (!data.aborted && (data.result.success || data.result.messages == null || data.result.messages.length === 0)) {
                    $scope.gotoResult();
                } else if ((data.result.warning || data.result.error) && !data.result.fatal) {
                    $scope.phase = "SHOW_WARNINGS";
                } else {
                    $scope.phase = "READY_TO_DUPLICATE";
                }
            }).update(function(data){
                $scope.percentage = ProgressStackMessageBuilder.getPercentage(data.progress);
                $scope.futureResponse = data;
                $scope.stateLabels = ProgressStackMessageBuilder.build($scope.futureResponse.progress, true);
            }).error(function(data, status, headers) {
                abortHook = null;
                $scope.futureResponse = null;
                $scope.duplicateResponse = null;
                $scope.phase = "READY_TO_DUPLICATE";
                setErrorInScope.bind($scope)(data, status, headers);
            })
        }).error(function(a,b,c){
            $scope.phase = 'READY_TO_DUPLICATE';
            setErrorInScope.bind($scope)(a,b,c);
            $scope.duplicateResponse = null;
        });
        WT1.event("projects-git-create-branch-and-duplicate", {
            duplicationMode: $scope.dupOptions.duplicationMode,
            exportAnalysisModels: $scope.dupOptions.exportAnalysisModels,
            exportPromptStudioHistories: $scope.dupOptions.exportPromptStudioHistories,
            exportKnowledgeBanks: $scope.dupOptions.exportKnowledgeBanks,
            exportSavedModels: $scope.dupOptions.exportSavedModels,
            exportModelEvaluationStores: $scope.dupOptions.exportModelEvaluationStores,
            exportProjectResources: $scope.dupOptions.exportProjectResources,
            exportInsightsData: $scope.dupOptions.exportInsightsData,
            isInSuggestedFolder: $scope.dupOptions.targetProjectFolder && $scope.dupOptions.targetProjectFolder.id === $scope.dupOptions.defaultProjectFolderId,
            isInRoot: $scope.dupOptions.targetProjectFolder && ProjectFolderService.isInRoot($scope.dupOptions.targetProjectFolder.id),
            isUnderSandbox: ProjectFolderService.isUnderSandbox($scope.dupOptions.targetProjectFolder)
        });
    };

    $scope.gotoResult = function() {
        $scope.dismiss();
        $state.transitionTo("projects.project.home.regular", {projectKey : $scope.checkoutModel.projectKey});
    };

    // Add watches
    $scope.$on("$destroy", function() {
        // cancel import if modal dismissed
        if (abortHook) {
            abortHook();
        }
    });
    $scope.$watch("checkoutModel.targetBranchName", function(newBranchName) {
        if ($scope.checkoutModel.createBranch) {
            if (doesBranchNameContainsInvalidCharacters(newBranchName)) {
                $scope.checkoutModel.branchNameError = true;
                $scope.checkoutModel.branchNameErrorMessage = "This branch name contains invalid characters.";
            } else if (isBranchNameAlreadyUsed(newBranchName)) {
                $scope.checkoutModel.branchNameError = true;
                $scope.checkoutModel.branchNameErrorMessage = "This branch name already exists.";
            } else {
                $scope.checkoutModel.branchNameError = false;
            }
        }
    });
    $scope.$watch("checkoutModel.projectName", fixProjectKey);
    $scope.$watch("checkoutModel.projectKey", function(newProjectKey) {
        $scope.checkoutModel.projectKeyAlreadyUsed = isProjectKeyAlreadyUsed(newProjectKey);
    });

    // Call backend to initialize form
    DataikuAPI.projects.listAllKeys()
        .success(function(data) {
            $scope.allProjectKeys = data;
            fixProjectKey($scope.checkoutModel.projectKey);
        })
        .error(setErrorInScope.bind($scope));
});

app.controller("ProjectGitResetController", function($scope, $filter, $stateParams, DataikuAPI, ActivityIndicator, Dialogs, $state, WT1) {
    $scope.resetStrategy = $scope.getResetModes()[0];

    $scope.setStrategy = function(strategy) {
        if ($scope.getResetModes().includes(strategy)) {
            $scope.resetStrategy = strategy;
        }
    };

    $scope.gitReset = function() {
        const resetToUpstream = () => DataikuAPI.projects.git.resetToUpstream($stateParams.projectKey);
        const resetToHead = () => DataikuAPI.projects.git.resetToHead($stateParams.projectKey);
        const resetAPICall = $scope.resetStrategy === 'HEAD' ? resetToHead : resetToUpstream;

        resetAPICall().then(function () {
                ActivityIndicator.success('Reset succeeded.');
                $state.reload();
                $scope.dismiss();
                WT1.event("projects-git-reset", {resetStrategy: $scope.resetStrategy});
            },
            setErrorInScope.bind($scope));
    };
});



}());