(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('ChartLegendUtils', ChartLegendUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/common/legends.js
     */
    function ChartLegendUtils(CreateCustomElementFromTemplate, $q, $timeout, ColorUtils, ChartColorUtils, ChartLabels, ChartDimension, ChartDataUtils, ChartFeatures, ChartFormatting, ChartUADimension, CHART_TYPES, CHART_AXIS_TYPES, ChartIconUtils) {
        const that = {
            initLegend: function(chartDef, chartData, legendsWrapper, colorScale, hideLegend, ignoreLabels) {
                if (!colorScale || !ChartFeatures.shouldComputeLegend(chartDef.type)) {
                    legendsWrapper.deleteLegends();
                    return;
                }

                // when using color groups should be called once per colorGroup
                if (chartDef.colorMode === 'COLOR_GROUPS' && ChartFeatures.canHaveConditionalFormatting(chartDef.type)) {
                    const legend = that.createContinuousLegend(ChartColorUtils.getColorDimensionOrMeasure(chartDef), colorScale);
                    legendsWrapper.pushLegend(legend);
                    return; // return because currently charts with color groups don't have legends outside groups with scale mode
                }

                switch (colorScale.type) {
                    case CHART_AXIS_TYPES.DIMENSION:
                        return that.initDimensionLegend(chartDef, chartData, legendsWrapper, colorScale, hideLegend, ignoreLabels);
                    case CHART_AXIS_TYPES.MEASURE:
                        return that.initMeasureLegend(chartDef, legendsWrapper, colorScale);
                    case CHART_AXIS_TYPES.UNAGGREGATED:
                        if (colorScale.isContinuous) {
                            return that.initMeasureLegend(chartDef, legendsWrapper, colorScale);
                        }
                        return;
                    case 'CUSTOM':
                        return that.initCustomLegend(chartDef, chartData, legendsWrapper, colorScale, hideLegend, ignoreLabels);
                    default:
                        throw new Error('Unknown scale type: ' + colorScale.type);
                }
            },

            initDimensionLegend: function(chartDef, chartData, legendsWrapper, colorScale, hideLegend, ignoreLabels = new Set()) {
                const colorDimensionLabels = chartData.getAxisLabels('color');
                const defaultLegendDimension = ChartColorUtils.getDefaultLegendDimension(chartDef);
                const colorLabels = colorDimensionLabels || defaultLegendDimension;
                const colorDimension = ChartColorUtils.getColorDimensionOrMeasure(chartDef);

                const items = colorLabels.filter(colorLabel => !ignoreLabels.has(colorLabel.label)).map(function(colorOrMeasure, c) {
                    const color = colorScale(c);

                    return {
                        label: {
                            colorId: ChartColorUtils.getColorId(defaultLegendDimension, chartData, c, undefined, ignoreLabels),
                            label: colorOrMeasure.label || chartData.getColorMeasureOrDimensionLabel(colorOrMeasure, chartDef.uaDimensionPair),
                            min: colorOrMeasure.min,
                            max: colorOrMeasure.max,
                            sortValue: colorOrMeasure.sortValue,
                            tsValue: colorOrMeasure.tsValue
                        },
                        color,
                        desaturatedColor: ColorUtils.desaturate(color),
                        rgbaColor: ColorUtils.toRgba(colorScale(c), ChartColorUtils.getTransparency(chartDef)),
                        focused: false,
                        unfocusFn: function() { /* focus/unfocus on mouseover */ },
                        focusFn: function() { /* focus/unfocus on mouseover */ },
                        id: c,
                        elements: d3.select()
                    };
                });


                legendsWrapper.deleteLegends();
                legendsWrapper.pushLegend(
                    {
                        type: 'COLOR_DISCRETE',
                        minValue: chartData.getMinValue('color'),
                        maxValue: chartData.getMaxValue('color'),
                        numValues: chartData.getNumValues('color'),
                        numberFormattingOptions: colorDimension && ChartDimension.getNumberFormattingOptions(colorDimension),
                        items,
                        hideLegend
                    }
                );
            },

            initMeasureLegend: function(chartDef, legendsWrapper, colorScale) {
                legendsWrapper.deleteLegends();
                const legend = that.createContinuousLegend(ChartColorUtils.getColorDimensionOrMeasure(chartDef), colorScale);
                legendsWrapper.pushLegend(legend);
            },

            initCustomLegend: function(chartDef, chartData, legendsWrapper, colorScale, hideLegend, ignoreLabels = new Set()) {
                const colorDimensionLabels = chartData.getAxisLabels('color', ignoreLabels) || [];
                const genericMeasures = chartDef.genericMeasures.filter(colorScale.customHandle);
                const colorLabels = colorDimensionLabels.concat(genericMeasures.filter(colorLabel => !ignoreLabels.has(colorLabel.label)));
                const colorDimension = ChartColorUtils.getColorDimensionOrMeasure(chartDef);
                const items = colorLabels.map(function(colorOrMeasure, c) {
                    const color = colorScale(c);

                    return {
                        label: {
                            colorId: ChartColorUtils.getColorId(colorLabels, chartData, c, undefined, ignoreLabels, colorOrMeasure.isA === 'measure'),
                            label: colorOrMeasure.label || ChartLabels.getLongMeasureLabel(colorOrMeasure),
                            min: colorOrMeasure.min,
                            max: colorOrMeasure.max,
                            sortValue: colorOrMeasure.sortValue,
                            tsValue: colorOrMeasure.tsValue
                        },
                        color,
                        desaturatedColor: ColorUtils.desaturate(color),
                        rgbaColor: ColorUtils.toRgba(colorScale(c), chartDef.colorOptions.transparency),
                        focused: false,
                        unfocusFn: function() { /* focus/unfocus on mouseover */ },
                        focusFn: function() { /* focus/unfocus on mouseover */ },
                        id: c,
                        elements: d3.select()
                    };
                });

                legendsWrapper.deleteLegends();
                legendsWrapper.pushLegend({
                    type: 'COLOR_DISCRETE',
                    minValue: chartData.getMinValue('color'),
                    maxValue: chartData.getMaxValue('color'),
                    numValues: items.length,
                    numberFormattingOptions: colorDimension && ChartDimension.getNumberFormattingOptions(colorDimension),
                    items,
                    hideLegend
                });
            },

            drawLegend: function(chartDef, chartHandler, $container) {
                const deferred = $q.defer();
                const $legendZone = $container.find('.legend-zone');

                if (chartDef.legendPlacement === 'SIDEBAR') {
                    // remove the potential previously drawn legend
                    $legendZone.empty();
                    $container.attr('legend-placement', chartDef.legendPlacement);
                    deferred.resolve();
                    return deferred.promise;
                }

                CreateCustomElementFromTemplate('/templates/simple_report/legend/legend-zone.html', chartHandler, null, function() {
                    $timeout(deferred.resolve);
                }, function($el) {
                    $container.attr('legend-placement', chartDef.legendPlacement);

                    //  Used to avoid flickering when redrawing legends
                    if ($legendZone.length) {
                        //  Hack to set same style if legend zone is the same
                        $el.css({
                            'min-width': $legendZone.outerWidth(),
                            'overflow': 'hidden'
                        });

                        //  Removes style attribute to avoid conflict with current legend zone styles
                        $timeout(() => {
                            $legendZone.replaceWith($el);
                            $el.removeAttr('style');
                        });
                    } else {
                        $el.appendTo($container);
                    }
                });

                return deferred.promise;
            },

            /**
             * Create legends, init + draw legends
             * @param {Element} $container
             * @param {ChartDef.java} chartDef
             * @param {Object} chartData
             * @param {chart_views scope} chartHandler
             * @param {Object} colorScale
             * @param {Set} ignoreLabels (Optional), labels that should not be included in the legend
             * @returns A promise
             */
            createLegend: ($container, chartDef, chartData, chartHandler, colorSpec, colorScale, hideLegend, ignoreLabels) => {
                // We need to draw the legend first, because (if it's set to be outside of the chart), its size will influence the remaining available width & height, that we need to know for the following
                that.initLegend(chartDef, chartData, chartHandler.legendsWrapper, colorScale, hideLegend, ignoreLabels);

                if (chartDef.type === CHART_TYPES.SCATTER) {
                    that.createScatterLegend(chartDef, chartHandler, chartData, colorScale);
                }

                // Create measure formatter for color legend
                if (chartHandler.legendsWrapper.hasLegends() && chartHandler.legendsWrapper.getLegend(0).type === 'COLOR_CONTINUOUS' && ChartFeatures.canDisplayLegend(chartDef.type)) {
                    // Depending on the chart type, the color dimension or measure is stored at different places in the chartDef.
                    const colorUaOrMeasure = chartDef.type === CHART_TYPES.SCATTER ? chartDef.uaColor[0] : chartDef.colorMeasure[0];
                    if (colorSpec.type === 'UNAGGREGATED') {
                        if (ChartUADimension.isAlphanumLike(colorUaOrMeasure) || ChartUADimension.isDiscreteDate(colorUaOrMeasure)) {
                            // chartHandler.legendsWrapper.getLegend(0).type should not be COLOR_CONTINUOUS ?
                        } else if (ChartUADimension.isDateRange(colorUaOrMeasure)) {
                            chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForDate();
                        } else {
                            chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForLegend(colorUaOrMeasure, colorScale.innerScale.domain());
                        }
                    } else {
                        const extent = colorSpec.extent || ChartDataUtils.getMeasureExtent(chartData, colorSpec.measureIdx, true);
                        chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForLegend(colorUaOrMeasure, extent);
                    }
                }
                return that.drawLegend(chartDef, chartHandler, $container);
            },


            adjustLegendPlacement: function(chartDef, $container, margins) {
                const $legendZone = $container.find('.legend-zone');

                const getEffectiveLeftMargin = function() {
                    if (chartDef.facetDimension.length) {
                        return $('.facet-info').width() + margins.left;
                    } else {
                        return margins.left;
                    }
                };

                const setMaxSize = function() {
                    $legendZone
                        .css('max-height', 'calc(100% - ' + (margins.top + margins.bottom) + 'px)')
                        .css('max-width', '25%')
                        .css('visibility', 'visible');
                };

                switch (chartDef.legendPlacement) {
                    case 'INNER_TOP_LEFT':
                        $legendZone.css('left', getEffectiveLeftMargin()).css('top', margins.top);
                        setMaxSize();
                        break;
                    case 'INNER_TOP_RIGHT':
                        $legendZone.css('right', margins.right).css('top', margins.top);
                        setMaxSize();
                        break;
                    case 'INNER_BOTTOM_LEFT':
                        $legendZone.css('left', getEffectiveLeftMargin()).css('bottom', margins.bottom);
                        setMaxSize();
                        break;
                    case 'INNER_BOTTOM_RIGHT':
                        $legendZone.css('right', margins.right).css('bottom', margins.bottom);
                        setMaxSize();
                        break;
                    default:
                        break;
                }
            },

            createContinuousLegend: function(colorDimension, colorScale) {
                const numberFormattingOptions = colorDimension && ChartDimension.getNumberFormattingOptions(colorDimension);
                colorScale.type = 'MEASURE';
                return {
                    type: 'COLOR_CONTINUOUS',
                    scale: colorScale,
                    numberFormattingOptions
                };
            },

            getSingleColorLegend: function(singleColor, label) {
                return {
                    type: 'COLOR_DISCRETE',
                    label: label,
                    items: [{ color: singleColor }]
                };
            },

            createScatterLegend: function(chartDef, chartHandler, chartData, colorScale) {
                const hasUAColor = chartData.hasUAColor(chartDef);
                const hasUAShape = chartData.hasUAShape(chartDef);
                const shapeScale = chartData.makeShapeScale();
                const hasColorLegend = (hasUAColor && (ChartUADimension.isAlphanumLike(chartDef.uaColor[0]) || ChartUADimension.isDiscreteDate(chartDef.uaColor[0])));

                if (hasUAShape || hasColorLegend) {
                    const legend = {
                        type: 'COLOR_DISCRETE',
                        items: []
                    };

                    if (hasColorLegend) {
                        colorScale.domain().forEach(v => {
                            const item = {
                                label: {
                                    colorId: ChartColorUtils.getColorId(chartDef.genericMeasures, chartData, v),
                                    ...chartData.data.values.color.str.sortedMapping[v]
                                },
                                color: colorScale(v),
                                focused: false
                            };
                            legend.items.push(item);
                        });
                    }

                    if (hasUAShape && hasColorLegend) {
                        legend.items.push({ separator: true });
                    }

                    if (hasUAShape) {
                        Object.values(chartData.data.values.shape.str.sortedMapping).forEach((v, index) => {
                            const shapeType = shapeScale(index).type;
                            const item = {
                                label: v,
                                shape: {
                                    svgSrc: ChartIconUtils.computeScatterLegendIcon(shapeType)
                                },
                                focused: false
                            };
                            legend.items.push(item);
                        });
                    }

                    chartHandler.legendsWrapper.deleteLegends();
                    chartHandler.legendsWrapper.pushLegend(legend);
                } else if (hasUAColor) {
                    // Done in initChart
                } else {
                    chartHandler.legendsWrapper.deleteLegends();
                }
            }
        };

        return that;
    }

})();
