(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('ScatterPlotChartDrawerWebGl', ScatterPlotChartDrawerWebGl);

    function ScatterPlotChartDrawerWebGl(UaChartsCommon, ColorUtils, ChartDataUtils, ChartDimension, ChartColorUtils, ChartFeatures, ChartUADimension, D3ChartAxes, ReferenceLines, RegressionLine, ChartZoomControlAdapter, CHART_ZOOM_CONTROL_TYPES, CHART_TYPES, ChartAxesUtils, ChartUsableColumns, ChartsStaticData, ChartCustomMeasures, ScatterUtils, $timeout, ChartIconUtils, ConnectPointsSplitOptions) {
        return function(container, chartDef, chartHandler, chartData, chartBase, uiDisplayState) {
            const g = d3.select(chartBase.$svgs.get(0));
            const multiple = chartDef.type === CHART_TYPES.SCATTER_MULTIPLE_PAIRS;
            const visibleCountMap = new Map();
            const axisPairs = chartData.prepareAxisPairs(chartDef, chartHandler.getChartTheme());
            const kdbushs = new Array(axisPairs.length);
            const foreignObject = g.append('foreignObject')
                .attr('x', chartBase.margins.left)
                .attr('y', chartBase.margins.top)
                .attr('width', chartBase.vizWidth)
                .attr('height', chartBase.vizHeight);
            const $body = $('<div>').css('height', '100%').css('width', '100%').appendTo(foreignObject.node());
            const loading = document.createElement('p');
            loading.className = 'scatter-loading-indicator';
            loading.setAttribute('style', 'position: relative; top: 50%; text-align: center');
            loading.innerText = 'Loading points, please wait';
            $body.append(loading);
            const thumbnailState = {
                shouldUpdate: false
            };

            const series = [];
            // for highlighting hovered point when tooltip is shown
            let highlightPointSerie;
            let tooltipHighlight = false;

            function isTooltipsReady() {
                return kdbushs.filter(index => index).length === axisPairs.length;
            }

            function updateSamplingInfo() {
                series.forEach((serie, index) => {
                    const xDomain = serie.xScale().domain();
                    const yDomain = serie.yScale().domain();
                    visibleCountMap.set(index, kdbushs[index].range(xDomain[0], yDomain[0], xDomain[xDomain.length - 1], yDomain[yDomain.length - 1]).length);
                });
                uiDisplayState.chartTopRightLabel = ChartDataUtils.computeRecordsStatusLabel(
                    chartData.getBeforeFilterRecords(),
                    chartData.getAfterFilterRecords(),
                    ChartDimension.getComputedMainAutomaticBinningModeLabel(chartData.getData(), chartDef, chartBase.zoomUtils && chartBase.zoomUtils.disableChartInteractivityGlobally),
                    visibleCountMap,
                    chartDef.type
                );
                uiDisplayState.chartRecordsFinalCountTooltip = ChartDataUtils.getRecordsFinalCountTooltip(
                    chartDef.type,
                    chartData.getAfterFilterRecords(),
                    chartData.data.axesPairs,
                    visibleCountMap
                );

                uiDisplayState.samplingSummaryMessage = ChartDataUtils.getSamplingSummaryMessage(
                    chartData.getData(),
                    chartDef.type,
                    undefined,
                    visibleCountMap
                );
            }

            /*
             * Create a worker and index the points in spatial index
             * to be able to get the points from the mouse coords.
             * The indexes will be used for the records tooltip
             */
            const getIndexBuilder = () => {
                const worker = new Worker('/static/dataiku/js/scatterplot-tooltips.worker.js');
                return worker;
            };

            const buildTooltipsIndex = (axesDef, axisPairs) => {
                const workers = [];
                axisPairs.forEach((pair, pairIndex) => {
                    chartHandler.uiDisplayState.isBuildingTooltips = true;
                    const axisDef = axesDef[pairIndex];
                    const xType = ChartUADimension.getUnaggregatedDimensionType(axisDef.xDim);
                    const yType = ChartUADimension.getUnaggregatedDimensionType(axisDef.yDim);

                    const worker = getIndexBuilder();
                    worker.postMessage({ pair, pairDef: { x: xType, y: yType } });
                    worker.onmessage = e => {
                        kdbushs[pairIndex] = KDBush.from(e.data);
                        if (isTooltipsReady()) {
                            chartHandler.uiDisplayState.isBuildingTooltips = false;
                            updateSamplingInfo();
                        }
                        worker.terminate();
                    };
                    workers.push(worker);
                });
            };

            const getDataAxesPairs = data => {
                if (multiple) {
                    return data.axesPairs;
                }
                return [{
                    afterFilterRecords: data.afterFilterRecords,
                    values: data.values,
                    xAxis: data.xAxis,
                    yAxis: data.yAxis
                }];
            };

            const createWebGLCanvas = (id, className) => {
                const canvas = document.createElement('d3fc-canvas');
                canvas.setAttribute('id', id);
                canvas.setAttribute('use-device-pixel-ratio', '');
                canvas.setAttribute('set-webgl-viewport', '');
                canvas.className = typedPlotArea(className);
                $(canvas).css('height', '100%');
                $(canvas).css('width', '100%');

                return canvas;
            };

            buildTooltipsIndex(axisPairs, getDataAxesPairs(chartData.data));

            const xScale = ScatterUtils.convertScale(chartBase.xAxis);
            const yScale = ScatterUtils.convertScale(chartBase.yAxes[0]);

            const xRangeSize = Math.abs(xScale.range()[0] - xScale.range()[1]);
            const yRangeSize = Math.abs(yScale.range()[0] - yScale.range()[1]);

            const typedPlotArea = type => `${type}-plot-area plot-area`;
            const group = document.createElement('d3fc-group');
            group.setAttribute('auto-resize', true);
            group.setAttribute('id', 'd3fc-group-scatter');
            //canvas has to match axes lengths when the scales are equal, otherwise the points are drawn out of range
            if (chartDef.scatterOptions.equalScales) {
                $(group).css('width', `${xRangeSize}px`);
                $(group).css('height', `${yRangeSize}px`);
            }

            const exportCanvas = document.createElement('canvas');
            const webglCanvas = createWebGLCanvas('scatter-canvas-gl', 'webgl');
            // canvas on which highlighted points are drawn on hover
            const webglHighlightCanvas = createWebGLCanvas('scatter-hlight-canvas-gl', 'webgl-hlight');
            const canvas = document.createElement('d3fc-canvas');
            const svg = document.createElement('d3fc-svg');

            canvas.setAttribute('id', 'scatter-canvas');
            canvas.setAttribute('use-device-pixel-ratio', '');
            canvas.className = typedPlotArea('canvas');

            svg.setAttribute('id', 'scatter-svg');
            svg.className = typedPlotArea('svg');

            $(canvas).css('height', '100%');
            $(canvas).css('width', '100%');

            $(svg).css('height', '100%');
            $(svg).css('width', '100%');

            $(exportCanvas).css('display', 'none');
            $(exportCanvas).css('height', '100%');
            $(exportCanvas).css('width', '100%');
            group.append(exportCanvas);
            group.append(webglCanvas);
            group.append(webglHighlightCanvas);
            group.append(canvas);
            group.append(svg);
            $body.append(group);
            const d3WebglCanvas = g.select('#scatter-canvas-gl');
            const d3WebglCanvasHLight = g.select('#scatter-hlight-canvas-gl');
            const glCanvas = d3WebglCanvas.node().querySelector('canvas');
            const glCanvasHLight = d3WebglCanvasHLight.node().querySelector('canvas');
            const scatterCanvas = g.select('#scatter-canvas');
            const canvad2d = scatterCanvas.node().querySelector('canvas');
            const scatterSvg = g.select('#scatter-svg > svg');
            const d3fcGroup = g.select('#d3fc-group-scatter');
            // Use a flag to deactivate regression line during panning because the current implementation is costly
            let shouldDrawRegressionLine = ChartFeatures.shouldDrawRegressionLine(chartDef);

            $timeout(() => {
                const pxlr = window.devicePixelRatio; // pixel ratio
                let context = null;
                const colorCache = {};
                let modifiedMargins = chartBase.margins;
                let drawTooltipHighlight = false;
                let isChartRendered = false;
                const pairColorOptions = chartDef.scatterMPOptions.pairColorOptions;

                let sizeScale, colorScale, shapeScale, resultingColor, resultingColors, cacheKey, sizeAvg;
                let computedRegression;
                const scatterOptions = multiple ? chartDef.scatterMPOptions : chartDef.scatterOptions;
                const hasBlending = scatterOptions.optimizeRendering !== true;
                const connectPoints = scatterOptions.connectPoints;
                const hasUAColor = chartData.hasUAColor(chartDef);
                const hasUASize = chartData.hasUASize(chartDef);
                if (hasUASize) {
                    sizeScale = UaChartsCommon.makeSizeScale(5, chartDef.uaSize[0], chartData.getData());
                    sizeAvg = chartData.getSizeAvg(chartDef.uaSize[0]);
                }
                const hasUAShape = chartData.hasUAShape(chartDef);
                if (hasUAShape) {
                    shapeScale = chartData.makeShapeScale();
                }
                if (hasUAColor) {
                    colorScale = chartBase.colorScale;
                } else if (multiple) {
                    resultingColors = pairColorOptions;
                } else {
                    resultingColor = UaChartsCommon.makeSingleColor(chartDef); // No color scale, compute the single color
                }
                const getSize = dataIndex => {
                    const radius = chartData.hasUASize(chartDef) ? UaChartsCommon.makeSize(chartDef, chartData.getData(), dataIndex, sizeScale) : (chartDef.bubblesOptions.defaultRadius * pxlr) / 2;

                    if (hasUAShape) {
                        return Math.max(1, Math.round(Math.PI * Math.pow(radius, 1.5)));
                    }
                    return Math.max(1, Math.round(Math.PI * Math.pow(radius, 2)));
                };
                let firstDraw = true;
                const seriesColorCache = new Map();

                const xPositionScale = function(d) {
                    return chartBase.xAxis.scale()(d) * pxlr;
                };

                const yPositionScale = function(d, pairIndex = 0) {
                    return chartBase.yAxes[pairIndex].scale()(d) * pxlr;
                };

                const getColorForSerie = pairIndex => index => {
                    const c = getColor(hasUAColor, chartData, chartDef, index, colorCache, colorScale, multiple, resultingColors, axisPairs[pairIndex], resultingColor, cacheKey, chartHandler.getChartTheme());
                    return c;
                };

                const getWebGLContext = (canvas) => {
                    const context = canvas.getContext('webgl');
                    if (hasBlending) {
                        context.enable(context.BLEND);
                        context.blendEquation(context.FUNC_ADD);
                        context.blendFunc(context.ONE, context.ONE_MINUS_SRC_ALPHA);
                    }
                    return context;
                };

                const gl = getWebGLContext(glCanvas);
                const pointSizeRange = gl.getParameter(gl.ALIASED_POINT_SIZE_RANGE);
                uiDisplayState.lowPointSizeRangeWarning = pointSizeRange[1] < 255 && (chartData.hasUASize(chartDef) || chartDef.bubblesOptions.defaultRadius > 25);

                context = canvad2d.getContext('2d');
                context.scale(pxlr, pxlr);
                const reshapeDataPerPairs = [];
                axisPairs.forEach(pair => {
                    const reshapedData = chartData.hasDiscreteUAColor(chartDef) || chartData.hasUAShape(chartDef) ? ScatterUtils.reshapeData(chartData, pair, chartDef, connectPoints) : null;
                    reshapeDataPerPairs.push(reshapedData);
                    // Create property length so the draw in d3fc is called
                    pair.length = pair.afterFilterRecords;
                    visibleCountMap.set(pair.pairIndex, pair.afterFilterRecords);
                    const yScale = ScatterUtils.convertScale(chartBase.yAxes[pair.pairIndex]);
                    let serie = ScatterUtils.createSerie(getSize, chartData, pair, chartDef, getColorForSerie(pair.pairIndex), seriesColorCache, reshapedData, hasBlending, chartDef.xAxisFormatting.isLogScale, ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, pair.id), connectPoints);
                    if (connectPoints.enabled) {
                        serie = ScatterUtils.addLineSerie(serie, chartData, pair, chartDef, reshapedData, connectPoints, getColorForSerie(pair.pairIndex), seriesColorCache, chartDef.xAxisFormatting.isLogScale, ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, pair.id));
                    }
                    serie.xScale(xScale);
                    serie.yScale(yScale);
                    serie.context(gl);
                    series.push(serie);
                    serie.$highlighting = true;
                });


                glCanvas.addEventListener('webglcontextlost', event => {
                    event.preventDefault();
                    series.forEach(serie => serie.context(null));
                }, false);
                glCanvas.addEventListener('webglcontextrestored', () => {
                    series.forEach(serie => {
                        serie.context(getWebGLContext(glCanvas));
                        serie(chartData.getData());
                    });
                }, false);

                d3fcGroup.on('draw', () => {
                    drawChart(modifiedMargins);
                });

                function getMousePos(rect, evt) {
                    return {
                        x: evt.clientX - rect.left,
                        y: evt.clientY - rect.top
                    };
                }

                svg.addEventListener('mousemove', onMouseMove);
                canvas.addEventListener('mousemove', onMouseMove);

                svg.addEventListener('mouseup', onMouseUp);
                canvas.addEventListener('mouseup', onMouseUp);

                svg.addEventListener('mousedown', onMouseDown);
                canvas.addEventListener('mousedown', onMouseDown);

                const cachedSortedMapping = {
                };

                function getValueForMousePos(axis, scale, position, axisData) {
                    if (axis.scaleType === D3ChartAxes.scaleTypes.TIME) {
                        return scale.invert(position).getTime();
                    }
                    if (axis.scaleType === D3ChartAxes.scaleTypes.ORDINAL) {
                        const rangePoints = axis.scale().range();
                        if (rangePoints[rangePoints.length - 1] < rangePoints[0]) {
                            rangePoints.reverse();
                        }
                        let indexInRange = d3.bisect(rangePoints, position);

                        const radius = chartData.hasUASize(chartDef) ? sizeAvg : (chartDef.bubblesOptions.defaultRadius / 2);
                        const isHoveringAtIndex = (index) => {
                            return position >= rangePoints[index] - radius && position <= rangePoints[index] + radius;
                        };

                        let hoveredIndex;
                        if (indexInRange === 0 || indexInRange === rangePoints.length) {
                            indexInRange = indexInRange === rangePoints.length ? indexInRange - 1 : indexInRange;
                            if (isHoveringAtIndex(indexInRange)) {
                                hoveredIndex = indexInRange;
                            } else {
                                return -1;
                            }
                        } else {
                            const indexBefore = indexInRange - 1;
                            if (isHoveringAtIndex(indexBefore)) {
                                hoveredIndex = indexBefore;
                            } else if (isHoveringAtIndex(indexInRange)) {
                                hoveredIndex = indexInRange;
                            } else {
                                return -1;
                            }
                        }
                        let sortedMapping;
                        if (cachedSortedMapping[axis.id]) {
                            sortedMapping = cachedSortedMapping[axis.id];
                        } else {
                            sortedMapping = axisData.str.sortedMapping.toSorted((a, b) => a.sortOrder - b.sortOrder);
                            const range = scale.range();
                            if (range[1] < range[0]) {
                                sortedMapping.reverse();
                            }
                            cachedSortedMapping[axis.id] = sortedMapping;
                        }
                        return sortedMapping[hoveredIndex].sortOrder;
                    }
                    return scale.invert(position);
                }

                function getRadiusInDomainFromMousePos(axis, value, scale, position) {
                    if (axis.scaleType === D3ChartAxes.scaleTypes.ORDINAL) {
                        return 0;
                    }
                    let radius;
                    if (chartData.hasUASize(chartDef)) {
                        radius = sizeScale(sizeAvg);
                    } else {
                        radius = (chartDef.bubblesOptions.defaultRadius / 2);
                    }
                    // apply some adjustments (numbers are random but look good)
                    if (chartData.hasUAShape(chartDef) && radius >= 10) {
                        radius *= 0.5;
                    }
                    return Math.abs(value - scale.invert(position - radius));
                }

                function onMouseMove(evt) {
                    if (!chartHandler.noTooltips) {
                        if (!isTooltipsReady()) {
                            return;
                        }
                        if (ChartZoomControlAdapter.isZooming(chartDef.$zoomControlInstanceId) || ChartZoomControlAdapter.isBrushing(chartDef.$zoomControlInstanceId)) {
                            hideTooltip();
                        } else {
                            const rect = canvas.getBoundingClientRect();
                            const mousePos = getMousePos(rect, evt);
                            const bests = [];
                            series.forEach((serie, serieIdx) => {
                                const serieXscale = serie.xScale();
                                const serieYscale = serie.yScale();
                                // From x,y on screen get x,y in on the domain
                                const x = getValueForMousePos(chartBase.xAxis, serieXscale, mousePos.x, multiple ? chartData.data.axesPairs[serieIdx].xAxis : chartData.data.xAxis);
                                const y = getValueForMousePos(chartBase.yAxes[serieIdx], serieYscale, mousePos.y, multiple ? chartData.data.axesPairs[serieIdx].yAxis : chartData.data.yAxis);
                                if (x !== -1 && y !== -1) {
                                    // Add some radius in domain values so we have margins to select in the spatial index
                                    const radiusX = getRadiusInDomainFromMousePos(chartBase.xAxis, x, serieXscale, mousePos.x);
                                    const radiusY = getRadiusInDomainFromMousePos(chartBase.yAxes[serieIdx], y, serieYscale, mousePos.y);
                                    bests.push(kdbushs[serieIdx].range(x - radiusX, y - radiusY, x + radiusX, y + radiusY));
                                }
                            });
                            if (bests.some(best => best.length)) {
                                const pairIndex = bests.findIndex(best => best.length);
                                const i = bests[pairIndex][0];
                                const color = getColorForSerie(pairIndex)(i);
                                chartBase.tooltips.showForCoords(0, { x: i, y: i, color: i, extras: { shape: i, size: i, pair: pairIndex } }, { pageX: mousePos.x + rect.left, pageY: mousePos.y + rect.top, target: canvas }, color);

                                updateTooltipHighlightSerie(pairIndex, i);
                                tooltipHighlight = true;
                                redrawTooltipHighlight();
                            } else {
                                hideTooltip();
                            }
                        }
                    }
                }

                function onMouseUp() {
                    canvas.classList.remove('panning');
                }


                function onMouseDown() {
                    if (ChartZoomControlAdapter.isEnabled(chartDef.$zoomControlInstanceId)) {
                        canvas.classList.add('panning');
                    }
                }

                function hideTooltip() {
                    chartBase.tooltips.hide();
                    if (tooltipHighlight) {
                        tooltipHighlight = false;
                        redrawTooltipHighlight();
                    }
                }

                function updateTooltipHighlightSerie(pairIndex, dataIndex) {
                    const color = getColorForSerie(pairIndex)(dataIndex);
                    highlightPointSerie = ScatterUtils.createHighlightSerie(dataIndex, chartData, axisPairs[pairIndex], getSize, color, multiple ? null : chartData.getShapeVal(chartDef.uaShape[0], dataIndex));
                    const xScale = ScatterUtils.convertScale(chartBase.xAxis);
                    const yScale = ScatterUtils.convertScale(chartBase.yAxes[pairIndex]);
                    highlightPointSerie.xScale(xScale);
                    highlightPointSerie.yScale(yScale);
                    const glContext = getWebGLContext(glCanvasHLight);
                    highlightPointSerie.context(glContext);
                }

                function redrawChart() {
                    drawTooltipHighlight = false;
                    isChartRendered = false;
                    group.requestRedraw();
                }

                function redrawTooltipHighlight() {
                    drawTooltipHighlight = true;
                    group.requestRedraw();
                }

                function clearCanvas() {
                    if (context) {
                        context.clearRect(0, 0, canvas.width, canvas.height);
                    }
                    /*
                     *  If some points reach the top of the canvas when zooming,
                     *  they left an artifact which it not clean by `clearRect`,
                     *  resetting width and height seems to do the trick.
                     */
                    const width = canvas.width;
                    const height = canvas.height;
                    canvas.width = width;
                    canvas.height = height;
                }

                function drawIdentityLine() {
                    if (ChartUADimension.areAllNumericalOrDate(chartDef) && chartDef.scatterOptions && chartDef.scatterOptions.identityLine && !multiple) {
                        const xAxisExtent = D3ChartAxes.getCurrentAxisExtent(chartBase.xAxis);
                        const yAxisExtent = D3ChartAxes.getCurrentAxisExtent(chartBase.yAxes[0]);

                        const start = Math.min(xAxisExtent[0], yAxisExtent[0]);
                        const end = Math.max(xAxisExtent[1], yAxisExtent[1]);

                        if (end < start) {
                            return;
                        }

                        context.strokeStyle = '#777';
                        context.beginPath();
                        context.moveTo(xPositionScale(start), yPositionScale(start));
                        context.lineTo(xPositionScale(end), yPositionScale(end));
                        context.closePath();
                        context.stroke();
                    }
                }

                const dataSpec = chartHandler.getDataSpec();
                const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
                const allDimensions = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
                const displayablePairs = ChartUADimension.getDisplayableDimensionPairs(chartDef.uaDimensionPair);
                const allUsedDimensions = multiple ? displayablePairs.flatMap(pair => {
                    const uaXDimension = ChartUADimension.getPairUaXDimension(displayablePairs, pair);
                    return [uaXDimension, pair.uaYDimension[0]];
                }) : [chartDef.uaXDimension[0], chartDef.uaYDimension[0]];

                const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, chartBase.xSpec, chartBase.ySpecs),
                    referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartBase.chartData, allDimensions, allUsedDimensions, customMeasures, false);

                function *getPoints(chartData) {
                    for (const pair of axisPairs) {
                        for (let i = 0; i < pair.afterFilterRecords; i++) {
                            yield [chartData.uaRawAxisVal(pair.xDim, 'x', i, pair.pairIndex), chartData.uaRawAxisVal(pair.yDim, 'y', i, pair.pairIndex)];
                        }
                    }
                }

                function drawChart(margins) {
                    if (!drawTooltipHighlight || !isChartRendered) {
                        axisPairs.forEach(pair => {
                            const serie = series[pair.pairIndex];
                            const reshapedData = reshapeDataPerPairs[pair.pairIndex];
                            if (serie.$highlighting) {
                                if (!_.isNil(serie.$highlightingColorIndex) || !_.isNil(serie.$highlightingShapeIndex)) {
                                    const splitBy = connectPoints.enabled ? connectPoints.splitBy : ConnectPointsSplitOptions.COLOR;
                                    const highlightingInnerSeries = chartData.hasDiscreteUAColor(chartDef) && chartData.hasUAShape(chartDef) && ((!_.isNil(serie.$highlightingColorIndex) && splitBy !== ConnectPointsSplitOptions.COLOR) || (!_.isNil(serie.$highlightingShapeIndex) && splitBy !== ConnectPointsSplitOptions.SHAPE));
                                    const highlightingIndex = !_.isNil(serie.$highlightingColorIndex) ? serie.$highlightingColorIndex : serie.$highlightingShapeIndex;
                                    const colorOrShapeSeries = serie.series();
                                    // if we highlight the inner series, they are not connected with a line so we don't display it
                                    const highlightedLineSerie = highlightingInnerSeries ? null : ScatterUtils.getLineSeries(colorOrShapeSeries[highlightingIndex]);

                                    if (highlightingInnerSeries) {
                                        colorOrShapeSeries.forEach((outerSerie, outerIndex) => {
                                            const highlightedInnerSerie = (connectPoints.enabled ? ScatterUtils.getMultiSeries(outerSerie) : outerSerie).series()[highlightingIndex];
                                            highlightedInnerSerie && highlightedInnerSerie(reshapedData.get(outerIndex).innerValues[highlightingIndex]);
                                        });
                                    } else if (chartData.hasDiscreteUAColor(chartDef) && chartData.hasUAShape(chartDef)) {
                                        const highlighedMultiSeries = connectPoints.enabled ? ScatterUtils.getMultiSeries(colorOrShapeSeries[highlightingIndex]) : colorOrShapeSeries[highlightingIndex];
                                        highlighedMultiSeries.series().forEach((innerSerie, innerIndex) => {
                                            innerSerie(reshapedData.get(highlightingIndex).innerValues[innerIndex]);
                                        });
                                    } else {
                                        const highlightedPointSeries = connectPoints.enabled ? ScatterUtils.getPointSeries(colorOrShapeSeries[highlightingIndex]) : colorOrShapeSeries[highlightingIndex];
                                        highlightedPointSeries(reshapedData.get(highlightingIndex).values);
                                    }

                                    highlightedLineSerie && highlightedLineSerie(reshapedData.get(highlightingIndex).values);
                                } else {
                                    serie(pair);
                                }
                            }
                            highlightPointSerie && highlightPointSerie({ length: 0 });
                        });
                        isChartRendered = true;
                        const xAxis = chartBase.xAxis ? { ...chartBase.xAxis, formattingOptions: chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting } : null;
                        const yAxes = [];
                        if (chartBase.yAxes) {
                            chartBase.yAxes.forEach(axis => {
                                if (axis) {
                                    const formatting = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, axis.id);
                                    yAxes.push({ ...axis, formattingOptions: formatting.axisValuesFormatting.numberFormatting });
                                }
                            });
                        }

                        ReferenceLines.drawReferenceLines(
                            g,
                            chartBase.vizWidth,
                            chartBase.vizHeight,
                            xAxis,
                            yAxes,
                            displayedReferenceLines,
                            referenceLinesValues,
                            { left: margins.left, top: margins.top }
                        );
                    } else {
                        if (tooltipHighlight) {
                            highlightPointSerie({ length: 1 });
                        } else {
                            highlightPointSerie && highlightPointSerie({ length: 0 });
                        }
                    }
                    if (ChartFeatures.canDrawIdentityLine(chartDef)) {
                        drawIdentityLine();
                    }
                    if (shouldDrawRegressionLine) {
                        if (!computedRegression) {
                            computedRegression = RegressionLine.computeRegression(() => getPoints(chartData), chartDef.scatterOptions.regression, chartDef.xAxisFormatting.customExtent, ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, chartBase.yAxes[0].id));
                        }

                        RegressionLine.drawRegression(computedRegression, context, chartDef.scatterOptions.regression, xPositionScale, yPositionScale, chartHandler.getChartTheme());
                    }
                    if (firstDraw) {
                        loading.remove();
                        firstDraw = false;
                    }
                    if (thumbnailState.shouldUpdate) {
                        const glOrigCanvas = glCanvas;
                        exportCanvas.width = glOrigCanvas.width;
                        exportCanvas.height = glOrigCanvas.height;
                        const ctx = exportCanvas.getContext('2d');
                        ctx.drawImage(glOrigCanvas, 0, 0);
                        ctx.drawImage(canvad2d, 0, 0);
                        thumbnailState.shouldUpdate = false;
                        chartHandler.updateThumbnail();
                    }
                }

                function clearChart() {
                    clearCanvas();
                    ReferenceLines.removeReferenceLines(container);
                }

                function toggleZoomableClass(canvas, isEnabled) {
                    if (isEnabled) {
                        canvas.classList.add('zoomable');
                    } else {
                        canvas.classList.remove('zoomable');
                    }
                }

                function updateAxisWithZoom(isLogScale, chartData, spec, axis, axisOptions, axisNumberFormatting, min, max, pairIndex) {
                    // When chart is in log scale, the tick values are set manually. They need to be updated.
                    if (isLogScale) {
                        D3ChartAxes.setLogAxisTicks(axis, min, max);
                        // we have to update the ticks in chartBase, as we use them to redraw axes later
                        axisOptions.tickValues = axis.tickValues();
                    } else if (ChartUADimension.isTrueNumerical(spec.dimension)) {
                        // if the axis is not in log scale, we need to rescale it and reinitialise axis formatting to be able to have increased precision as we zoom
                        axis.scale().domain([min, max]);
                        D3ChartAxes.addNumberFormatterToAxis(axis, axisNumberFormatting);
                        // we have to update the ticks format in chartBase, as we use it to redraw axes later
                        axisOptions.tickFormat = axis.tickFormat();
                    } else if (ChartUADimension.isDate(spec.dimension) && axis.scaleType === D3ChartAxes.scaleTypes.TIME) {
                        axis.scale().domain([min, max]);
                        const computedDateDisplayUnit = ChartDataUtils.computeDateDisplayUnit(min, max);
                        // update the tick formatting to have the right precision for the date on axis
                        axisOptions.tickFormat = date => computedDateDisplayUnit.formatDateFn(date, computedDateDisplayUnit.dateFormat);
                        // update min/max on data wrapper as well
                        chartData.setExtent(pairIndex, min, max);
                    }
                }


                function configureZoom(d3canvas, d3svg) {
                    //  Create new instance
                    chartDef.$zoomControlInstanceId = ChartZoomControlAdapter.create(CHART_ZOOM_CONTROL_TYPES.D3, d3, d3canvas, chartBase.xAxis, chartBase.yAxes, g.node().clientWidth, d3svg);

                    if (ChartZoomControlAdapter.isActivated(chartDef.$zoomControlInstanceId)) {
                        modifiedMargins = chartBase.margins;

                        const onZoom = () => {
                            D3ChartAxes.clearAxes(g);
                            clearChart();
                            if (ChartFeatures.shouldDrawRegressionLine(chartDef)) {
                                shouldDrawRegressionLine = false;
                            }

                            const xExtent = D3ChartAxes.getCurrentAxisExtent(chartBase.xAxis);
                            const xMin = xExtent[0];
                            const xMax = xExtent[1];
                            updateAxisWithZoom(chartDef.xAxisFormatting.isLogScale, chartBase.chartData, chartBase.xSpec, chartBase.xAxis, chartBase.axisOptions.x, chartDef.xAxisFormatting.axisValuesFormatting && chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting ? chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting : {}, xMin, xMax);

                            axisPairs.forEach(pair => {
                                const yAxis = chartBase.yAxes.find(v => v.id === (pair.id || ChartsStaticData.LEFT_AXIS_ID));
                                const yAxisSpec = chartBase.ySpecs[pair.id || ChartsStaticData.LEFT_AXIS_ID];
                                const yExtent = D3ChartAxes.getCurrentAxisExtent(yAxis);
                                const yMin = yExtent[0];
                                const yMax = yExtent[1];
                                const yAxisOpts = chartBase.axisOptions.y.find(v => v.id === yAxis.id);
                                const isLogScale = ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, pair.id);
                                updateAxisWithZoom(isLogScale, chartBase.chartData, yAxisSpec, yAxis, yAxisOpts, ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, pair.id), yMin, yMax, pair.pairIndex);
                            });

                            const yAxesColors = chartData.getYAxesColors(chartBase.ySpecs, chartDef, chartHandler.getChartTheme());
                            const d3DrawContext = {
                                $svgs: chartBase.$svgs,
                                chartDef,
                                chartHandler,
                                ySpecs: chartBase.ySpecs,
                                xAxis: chartBase.xAxis,
                                yAxes: chartBase.yAxes,
                                axisOptions: chartBase.axisOptions,
                                yAxesColors
                            };
                            const { updatedMargins, vizWidth, vizHeight } = D3ChartAxes.drawAxes(d3DrawContext);
                            modifiedMargins = updatedMargins;
                            chartBase.vizWidth = vizWidth;
                            chartBase.vizHeight = vizHeight;

                            const newXScale = ScatterUtils.convertScale(chartBase.xAxis);
                            series.forEach((serie, index) => {
                                const newYScale = ScatterUtils.convertScale(chartBase.yAxes[index]);
                                serie.yScale(newYScale);
                                serie.xScale(newXScale);
                            });

                            redrawChart();
                        };

                        const onZoomEnd = () => {
                            /*
                             * zooming can change chart margins and chart size, chart needs to be retranslated and resized
                             */
                            g.select('foreignObject')
                                .attr('width', chartBase.vizWidth)
                                .attr('height', chartBase.vizHeight);

                            g.select('foreignObject')
                                .attr('x', modifiedMargins.left)
                                .attr('y', modifiedMargins.top);

                            if (isTooltipsReady()) {
                                updateSamplingInfo();
                            }
                            shouldDrawRegressionLine = ChartFeatures.shouldDrawRegressionLine(chartDef);
                            thumbnailState.shouldUpdate = true;
                            clearChart();
                            redrawChart();
                        };

                        const onEnabledChange = (value) => {
                            toggleZoomableClass(d3canvas.node(), value);
                        };
                        ChartZoomControlAdapter.init(CHART_ZOOM_CONTROL_TYPES.D3, chartDef.$zoomControlInstanceId, { hasRectangleSelection: true, ...chartDef.scatterZoomOptions }, { onZoom, onZoomEnd, onEnabledChange });
                    } else {
                        thumbnailState.shouldUpdate = true;
                        d3svg[0][0].parentElement.remove();
                        redrawChart();
                    }
                }

                const hasColorLegend = (hasUAColor && (ChartUADimension.isAlphanumLike(chartDef.uaColor[0]) || ChartUADimension.isDiscreteDate(chartDef.uaColor[0])));

                if (hasUAShape || hasColorLegend) {
                    const legend = {
                        type: 'COLOR_DISCRETE',
                        items: []
                    };

                    if (hasColorLegend) {
                        colorScale.domain().forEach(function(v, idx) {
                            const item = {
                                label: {
                                    colorId: ChartColorUtils.getColorId(chartDef.genericMeasures, chartData, v),
                                    ...chartData.getColorData().str.sortedMapping[v]
                                },
                                color: colorScale(v),
                                focusFn: function() {
                                    clearChart();
                                    series.forEach((serie) => {
                                        serie.$highlightingColorIndex = idx;
                                    });
                                    redrawChart();
                                },
                                unfocusFn: function() {
                                    clearCanvas();
                                    series.forEach((serie) => {
                                        serie.$highlightingColorIndex = null;
                                    });
                                    redrawChart();
                                }
                            };
                            legend.items.push(item);
                        });
                    }

                    if (hasUAShape && hasColorLegend) {
                        legend.items.push({ separator: true });
                    }

                    if (hasUAShape) {
                        chartData.getShapeData().str.sortedMapping.map((v, idx) => {
                            const shapeType = shapeScale(idx).type;
                            const item = {
                                label: chartData.getShapeData().str.sortedMapping[idx],
                                shape: {
                                    svgSrc: ChartIconUtils.computeScatterLegendIcon(shapeType)
                                },
                                focusFn: function() {
                                    clearChart();
                                    series.forEach((serie) => {
                                        serie.$highlightingShapeIndex = idx;
                                    });
                                    redrawChart();
                                },
                                unfocusFn: function() {
                                    clearCanvas();
                                    series.forEach((serie) => {
                                        serie.$highlightingShapeIndex = null;
                                    });
                                    redrawChart();
                                }
                            };
                            legend.items.push(item);
                        });
                    }

                    chartHandler.legendsWrapper.deleteLegends();
                    chartHandler.legendsWrapper.pushLegend(legend);
                } else if (multiple && chartHandler.legendsWrapper.hasLegends()) {
                    chartHandler.legendsWrapper.getLegend(0).items.forEach((item, index) => {
                        item.focusFn = function() {
                            clearChart();
                            series.forEach((serie, serieIdx) => {
                                serie.$highlighting = index === serieIdx;
                            });
                            redrawChart();
                        };
                        item.unfocusFn = function() {
                            clearChart();
                            chartData.$highlighting = true;
                            series.forEach(serie => {
                                serie.$highlighting = true;
                            });
                            redrawChart();
                        };
                    });
                } else if (!hasUAColor && !hasUAShape) {
                    chartHandler.legendsWrapper.deleteLegends();
                }
                configureZoom(scatterCanvas, scatterSvg);
            });
        };

        function getColor(hasUAColor, chartData, chartDef, i, colorCache, colorScale, multiple, resultingColors, pair, resultingColor, cacheKey, theme) {
            let c;
            if (hasUAColor) {
                let rgb;
                const colorData = chartData.getColorData();
                if (chartDef.uaColor[0].type == 'NUMERICAL' && !chartDef.uaColor[0].treatAsAlphanum) {
                    cacheKey = colorData.num.data[i];
                } else if (chartDef.uaColor[0].type == 'DATE' && chartDef.uaColor[0].dateMode == 'RANGE') {
                    cacheKey = colorData.ts.data[i];
                } else {
                    cacheKey = colorData.str.data[i];
                }
                if (colorCache[cacheKey]) {
                    rgb = colorCache[cacheKey];
                } else {
                    rgb = colorScale(cacheKey);
                    colorCache[cacheKey] = rgb;
                }
                c = rgb;
            } else if (multiple) {
                const colors = ChartColorUtils.getColors(resultingColors.colorPalette, resultingColors.customPalette, theme);
                const assignedColor = ChartColorUtils.getColor(resultingColors, colors, pair.colorInfo.id, pair.pairIndex);
                if (colorCache[assignedColor]) {
                    return colorCache[assignedColor];
                }
                c = colorCache[assignedColor] = ColorUtils.toRgba(assignedColor, chartDef.scatterMPOptions.pairColorOptions.transparency);
            } else {
                c = resultingColor;
            }
            return c;
        }
    }
})();
