param (
    $d,
    $p,
    $l,
    [switch] $u,
    $py,
    [switch] $h,
    [switch] $r,
    [switch] $f,
    [switch] $y
)

# Version of the python binary
function GetPythonVersion()
{
    param(
        [Parameter(Mandatory=$true)] [string] $PythonBin
    )

    & $PythonBin -c "import sysconfig;print(sysconfig.get_python_version())"
}

# This function takes output from the pipeline, and write it in a log file, with UTF8 without BOM encoding
# By default with Powershell version lower than 6, ">" writes with UTF8 with BOM, which is really weird
function Write-Log {
    [CmdletBinding()]
    param (
        [Parameter(Mandatory = $true, Position = 0)]
        $Path,
        [Parameter(ValueFromPipeline = $true)] # Not mandatory as it can cause crashes
        [Collections.Generic.List[String]] # Need to convert to a list because AppendAllLines requires it
        # The conversion is safe, as we will only have strings and string arrays (and the cast works on those two types)
        $Value,
        [switch]
        $Print,
        [switch]
        $Append
    )

    Begin {
        # Delete log file if it already exists, and we don't want to append
        if ((Test-Path -Path $Path) -and (-not $Append)) {
            Remove-Item -Path $Path
        }
    }

    Process {
        $Utf8NoBomEncoding = New-Object System.Text.UTF8Encoding $False
        [System.IO.File]::AppendAllLines($Path, $Value, $Utf8NoBomEncoding)
        # This function will open and close the file itself, no need to worry
        if ($Print) {
            Write-Output ($Value -Join '`n')
        }
    }

    End {}
}

Write-Output "*********************************************"
Write-Output "*           Dataiku DSS installer           *"
Write-Output "*********************************************"

if ($PSVersionTable.PSVersion.Major -lt 4) {
    Write-Error "Sorry, but DSS requires PowerShell 4.0 to work properly."
    Exit 1
}

# Get absolute path of kit
$myDir=$PSScriptRoot

function usage() {
    $prog = $PSCommandPath
    $c = @"
*** Usage:
New installation:    $prog -d DATADIR -p BASE_PORT [-l LICENSE_FILE] [-u]
Upgrade:             $prog -u -d DATADIR
Print this help:     $prog -h

    -d DATADIR : specify data directory
    -p BASE_PORT : specify base port
    -l LICENSE_FILE : specify license file
    -u : upgrade an existing data directory
    -py BASE_PYTHON : Python binary on which to build DSS default virtual environment
    -h : prints this help
"@
    Write-Output "$c"
    Exit 1
}

if ($h) {
    usage
}

$DIP_HOME=$d
$PORT=$p
$NODETYPE="design"
$INSTALLSIZE="auto"
$LICENSE=$l

if ($y) {
    $env:DKU_MIGRATE_YES=1
}

if ($u -and ($p -or $l)) {
    Write-Output "[-] Bad usage: cannot specify port or license file while upgrading"
    usage
}

if (-not $DIP_HOME) {
    Write-Output "[-] Bad usage : -d DATADIR argument is mandatory"
	usage
}

if ($u) {
    # Check for valid dss folder
    if ((Test-Path -Path "$DIP_HOME") -and (Test-Path -Path "$DIP_HOME/dss-version.json")) {
        Write-Output "[+] $DIP_HOME already exists and is a DSS installation"
    } else {
        Write-Output "[-] $DIP_HOME does not appear to be a valid DSS installation"
        Exit 1
    }
} else {
    if (-not $p) {
        Write-Output "Bad usage : -p BASE_PORT argument is mandatory"
        usage
    }
}


# TODO Check if folder already exists, is empty, etc...
# Creating folder, and getting absolute path
mkdir -Force $DIP_HOME | Out-Null
$DIP_HOME=$(Resolve-Path -Path $DIP_HOME).Path.Trim("\").Replace("\", "/")
$env:DIP_HOME=$DIP_HOME

New-Item -Path "$DIP_HOME/DSS-INITIAL-INSTALL-IN-PROGRESS.txt" -Type File -ErrorAction Ignore

New-Item -Path "$DIP_HOME/run" -Type Directory -ErrorAction Ignore
$logFile="$DIP_HOME/run/install.log"
New-Item -Path $logfile -Type File -ErrorAction Ignore

Write-Output "[+] Saving installation log to $logFile"
@"
*********************************************************
Data Science Studio installer: $(Get-Date -Format "yyyy/MM/dd HH:mm:ss K")
Command line: $($MyInvocation.Line)
Version: $(Get-Content -Path "$MYDIR/dss-version.json")
DIP_HOME: $DIP_HOME
"@ | Write-Log -Path $logFile

Invoke-Command -ScriptBlock {
# TODO: Check for running instance
# Check for DATADIR inside INSTALLDIR
if ($DIP_HOME.Replace("\", "/") -like "$myDir/*".Replace("\", "/")) {
	Write-Output "[!] *********************************************************"
	Write-Output "[!] Warning: the data directory you specified:"
	Write-Output "[!]     $DIP_HOME"
	Write-Output "[!] appears to be a subdirectory of the installation directory:"
	Write-Output "[!]     $myDir"
	Write-Output "[!] This is NOT RECOMMENDED for production environment as it complexifies subsequent Studio upgrades."
    Write-Output "[!] Press Enter to continue, Ctrl+C to abort"
    Pause
}

# Check system dependencies
if (-not $env:DKUJAVABIN -and -not (Get-Command java -errorAction SilentlyContinue)) {
    Write-Output "[-] Java is not installed on the system, can't continue"
    Exit 1
}

. "$MYDIR/scripts/_py-utils.ps1"

$expectedPythonVersion = "3.11"

if ($py) {
    $pythonBin = $py
    $pythonVersion = GetPythonVersion $pythonBin
    if (-Not ($pythonVersion -eq $expectedPythonVersion)) {
        Write-Output "[-] Wrong Python version: this DSS version only works for Python $expectedPythonVersion."
        Exit 1
    }
} elseif (-not $u) {
    if (!(Test-PythonInstallation -Version $expectedPythonVersion)) {
        Write-Output "[-] Python $expectedPythonVersion is not installed on the system, can't continue"
        Exit 1
    }

    $pythonBin = Get-PythonExecutablePath -Version $expectedPythonVersion
}

$env:DKUINSTALLDIR = "$myDir"
if (-not $env:DKUJAVABIN) {
    $env:DKUJAVABIN=(Get-Command java).Path
}

# install-deps -check

function precompile_python() {
    $pythonBin = $args[0]

    Write-Output "[+] Precompiling Dataiku Python code"
    & "$pythonBin" -m compileall -q "$myDir/python"

    Write-Output "[+] Precompiling Jupyter Python code"
    & "$pythonBin" -m compileall -q "$myDir/dku-jupyter"

    # Need to change if we need to support multiple Python versions
    Write-Output "[+] Precompiling third-party Python code"
    & "$pythonBin" -m compileall -q "$myDir/pythonwin.packages"
}

function InitializePyEnv() {
	Write-Output "[+] Initializing Python environment using '$pythonBin'"
    & "$MYDIR/scripts/_install-pyenv.ps1" -PythonBin $pythonBin -DATADIR $DIP_HOME
}

$env:PYTHONUTF8=1

if ($u) {
	######################################################################################
	#                             Upgrade
	######################################################################################
    if (Test-Path -Path "$DIP_HOME/pyenv") {
        Write-Output "[+] Migrating Python environment"
        $installPyenvOptArgs = @{
            "Upgrade" = $true
            "ForceRebuildPyenv" = $f
            "RebuildIfNeeded" = $r
            "DATADIR" = $DIP_HOME
        }

        if ($py)
        {
            $installPyenvOptArgs['PythonBin'] = $pythonBin
        }

        & "$MYDIR/scripts/_install-pyenv.ps1" @installPyenvOptArgs
        $pythonBin="$DIP_HOME/pyenv/Scripts/python.exe"

        precompile_python $pythonBin
    } else {
        InitializePyEnv
    }

    # Perform migration
    Write-Output "[+] Migrating data directory"
    $env:PYTHONIOENCODING = "UTF-8"
    $env:PYTHONUNBUFFERED = 1

    & $pythonBin "$myDir/scripts/dkuinstall/migrate_auto.py" "$DIP_HOME"
} else {
    ######################################################################################
    #                             Fresh install
    ######################################################################################
    Write-Output "[+] Installation starting"
    Copy-Item -Path "$myDir/dss-version.json" -Destination "$DIP_HOME"

    # For each loop syntax
    "bin", "config" | ForEach-Object {New-Item -Path "$DIP_HOME/$_" -Type Directory -ErrorAction Ignore *>$null}

    New-Item -Path "$DIP_HOME/bin/env-site.ps1" -ItemType File *>$null
    Set-Content -Path "$DIP_HOME/bin/env-site.ps1" -Value "# This file is sourced last by DSS startup scripts`n# You can add local customizations to it"

    # License etc
    if ($LICENSE) {
        Write-Output "[+] Installing license file"
        Copy-Item -Path $LICENSE -Destination "$DIP_HOME/config/license.json"
    }

    # Pyenv setup
    InitializePyEnv

    $pythonBin="$DIP_HOME/pyenv/Scripts/python.exe"

    precompile_python $pythonBin

    $params ="$myDir/scripts/dkuinstall/install_dss.py",
    $DIP_HOME,
    $PORT,
    $NODETYPE,
    $INSTALLSIZE
    & $pythonBin $params

    Write-Output "[+] Preparing data directory initial data"

    & "$DIP_HOME/bin/dku.cmd" "-s" "__initial-setup-home" "`"$DIP_HOME`"" "`"$NODETYPE`"" *>&1
}

# Install native IPython kernel
$jupyterData="$DIP_HOME\jupyter-run\jupyter"
$env:JUPYTER_DATA_DIR="$jupyterData"
& $pythonBin -m ipykernel install --user

# Install widgetsnbextension
$env:JUPYTER_CONFIG_DIR = $jupyterData
$env:JUPYTER_DATA_DIR = $jupyterData
$env:PYTHONPATH = "$MYDIR/dku-jupyter/packages/"
& $pythonBin -c 'from notebook.nbextensions import main;main()' install --py widgetsnbextension --user

$jupyterNbconfig="$jupyterData/config/nbconfig"
if (-Not (Test-Path("$jupyterNbconfig/notebook.json"))) {
    New-Item -Path "$jupyterNbconfig" -Type Directory -ErrorAction Ignore *>$null
    New-Item -Path "$jupyterNbconfig/notebook.json" -ItemType File *>$null
    $content=@"
{
    "load_extensions": {
        "collapsible_headings/main": true,
        "codefolding/main": true,
        "toggle_all_line_numbers/main": true,
        "hide_input_all/main": true,
        "addbefore/main": true
    }
}
"@
    Set-Content -Path "$jupyterNbconfig/notebook.json" -Value $content
}

# Probably need to check if the link already exists, to replace it
# ln -s
# New-Item with type junction is only supported in PS 5.1.
if ($PSVersionTable.PSVersion.Major -ge 5) {
    New-Item -Path "$jupyterData/nbextensions" -ItemType Junction -Target "$myDir/dku-jupyter/nbextensions" *>$null
} else {
    cmd.exe /c mklink /J "$jupyterData\nbextensions" "$myDir\dku-jupyter\nbextensions" *> $null
}

& "$DIP_HOME/bin/dssadmin" regenerate-config

# END

"$DIP_HOME/lib/jdbc", "$DIP_HOME/lib/java", "$DIP_HOME/lib/python" | ForEach-Object {New-Item -Path "$_" -Type Directory -ErrorAction Ignore *>$null}

Remove-Item -Force -Path "$DIP_HOME/DSS-INITIAL-INSTALL-IN-PROGRESS.txt"

New-Item -Path "$DIP_HOME/install-support/expected-dip-home.txt" -ItemType File *>$null
Set-Content -Path "$DIP_HOME/install-support/expected-dip-home.txt" -Value "$DIP_HOME"

Write-Output "***************************************************************"
Write-Output "* Installation complete (DSS node type: $NODETYPE)"
Write-Output "* Next, start DSS using:"
Write-Output "*            '$DIP_HOME\bin\dss.cmd start'"
Write-Output "***************************************************************"

} | Write-Log -Path $logFile -Print -Append