# -*- coding: utf-8 -*-
from dataiku.doctor.docgen.renderer.document_handler import DocumentHandler
from dataiku.doctor.docgen.extractor.docx_parser import DocxParser
from dataiku.doctor.docgen.renderer.widget import Table, Text, PuppeteerContent
from dataiku.doctor.docgen.common.placeholder import BlockPlaceholder, FooterPlaceholder, HeaderPlaceholder
import logging

logger = logging.getLogger(__name__)


class RendererHelpers(object):

    def __init__(self):
        self.docx_parser = DocxParser()

    def __duplicates_before_run__(self, input_runs, output_paragraph, until_run, until_char):
        """
        Copy any run located before the run number "until_run" and the character number "until_char"
        from the input_paragraph to the output_paragraph.placeholders
        :param Paragraph input_runs: runs of the input paragraph containing data to copy
        :param Paragraph output_paragraph: the output paragraph when copied data will be transferred
        :param int until_run: the run index where we need to stop copying data
        :param int until_char: the character index inside the run where we need to stop copying data
        """
        if until_run == 0:
            if until_char != 0:
                # We are on the first run, but there is stuff before the current placeholder,
                # extract data on their own run.
                logger.debug("> Adding first part run before: data='%s' cutted='%s'" %
                              (input_runs[0].text[0:until_char], input_runs[0].text[until_char:]))
                run = output_paragraph.add_run(input_runs[0].text[0:until_char], input_runs[0].style)
                DocumentHandler.copy_font(input_runs[0], run)
        else:
            # The placeholder is not present on the first run, duplicate the runs.
            if len(input_runs) > 0:
                for i in range(0, until_run):
                    logger.debug("> Adding full run %d before: data='%s'" % (i, input_runs[i].text))
                    run = output_paragraph.add_run(input_runs[i].text, input_runs[i].style)
                    DocumentHandler.copy_font(input_runs[i], run)
            # And we reach the run with the placeholder
            if until_char != 0:
                logger.debug("> Adding part run before: data='%s' cutted='%s'" %
                              (input_runs[until_run].text[0:until_char], input_runs[until_run].text[until_char:]))
                run = output_paragraph.add_run(input_runs[until_run].text[0:until_char], input_runs[until_run].style)
                DocumentHandler.copy_font(input_runs[until_run], run)

    def __duplicates_after_run__(self, input_runs, output_paragraph, after_run, after_char):
        """
        Copy any run located after the run number "after_run" and the character number "after_char" from the
        input_paragraph to the output_paragraph.
        :param Paragraph input_runs: runs of the input paragraph containing data to copy
        :param Paragraph output_paragraph: the output paragraph when copied data will be transferred
        :param int after_run: the run index where we need to start copying data
        :param int after_char: the character index inside the run where we need to start copying data
        """
        if len(input_runs) == 0:
            return

        input_run = input_runs[after_run]
        # after_char is the position of the last "}" so we want to start the completion from the next character
        first_char_to_add = after_char + 1

        # Finish the current run in a separate run
        if first_char_to_add < len(input_run.text):
            logger.debug("< Adding part run after: data='%s'" % (input_run.text[first_char_to_add:]))
            output_run = output_paragraph.add_run(input_run.text[first_char_to_add:], input_run.style)
            DocumentHandler.copy_font(input_run, output_run)

        # Add the runs left in the paragraph
        for i in range(after_run + 1, len(input_runs)):
            logger.debug("< Adding full run %d after: data='%s'" % (i, input_runs[i].text))
            output_run = output_paragraph.add_run(input_runs[i].text, input_runs[i].style)
            DocumentHandler.copy_font(input_runs[i], output_run)

    def replace_simple_placeholder(self, document, placeholder, widget, font_reference, table_style = None):
        """
        Locate and replace the Paragraph with the placeholder inside the document by a new paragraph where the
        placeholder is replaced by its data.
        :param Union[Document, _Cell] document: the current document (or sub-document) that contains paragraph with placeholders
        :param Placeholder placeholder: data representing the location of a placeholder
        :param Union[Text, Table, PuppeteerContent] widget: the data associated to the placeholder
        :param Document font_reference: the input docx it will be use as a reference for font and table insertion
        """

        # "start_run_index" is the index of the run containing the first character of the placeholder opening delimiter
        # "start" is the index in the run of the first character of the placeholder opening delimiter
        (start_run_index, start) = placeholder.start
        # "end_run_index" is the index of the run containing the last character of the placeholder closing delimiter
        # "end" is the index in the run of the last character of the placeholder closing delimiter
        (end_run_index, end) = placeholder.end

        # The next variable is the paragraph where the placeholder is located.
        # We want to extract everything from the paragraph except the placeholder
        input_paragraph = DocumentHandler.get_paragraph(document, placeholder.paragraph._p)

        # Check if the paragraph was not deleted previously by a conditional placeholder
        if input_paragraph:
            # the output paragraph will change as soon as you insert a table
            output_paragraph = input_paragraph

            # copy the runs from the input paragraph and purge it
            input_runs = DocxParser.get_safe_runs(input_paragraph)
            input_paragraph.clear()

            # extract data inside the paragraph before the placeholder
            self.__duplicates_before_run__(input_runs, output_paragraph, start_run_index, start)

            # Apply the data from the widget on the new run
            # If we are on a table widget, a new output paragraph will be created
            # if widget = None, it will just remove the placeholder
            if widget != None:
                output_paragraph = widget.apply(output_paragraph, font_reference, input_runs[start_run_index], table_style)

            # extract data inside the paragraph after the placeholder
            self.__duplicates_after_run__(input_runs, output_paragraph, end_run_index, end)


    def replace_block_placeholder(self, document, block, widget):
        """
        Replace pair of placeholders inside the document by their content
        :param Document document: the current document that contains tables which will contains paragraph with placeholders
        :param BlockPlaceholder block: the BlockPlaceholder to replace
        :param Union[Table, Text, PuppeteerContent, None] widget: what should replace the placeholder
        """
        # Extracted contains everything between the blocks placeholders
        extracted = DocumentHandler.between(document, block)
        if isinstance(widget, Table) or isinstance(widget, PuppeteerContent):
            filtered = [e for e in extracted if e.__class__.__name__ == "Table"]
            logger.debug("Filtered Table %s", filtered)
            if len(filtered) > 0:
                self.replace_simple_placeholder(document, block.start, widget, document, filtered[0])
            else:
                # Not a table, run a basic transformation on the starting placeholder
                self.replace_simple_placeholder(document, block.start, widget, document)
        elif isinstance(widget, Text):
            # Not a table, run a basic transformation on the starting placeholder
            self.replace_simple_placeholder(document, block.start, widget, document)

        else: # the widget is None, we just delete the starting placeholder
            # delete the starting placeholder
            if block.start.paragraph and block.start.paragraph._element.getparent():
                DocumentHandler.delete_paragraph(block.start.paragraph)
        
        # Delete data between the placeholders
        for e in extracted:
            DocumentHandler.delete_element(e)
        
        # in any cases, delete the ending placeholder
        if block.end.paragraph and block.end.paragraph._element.getparent():
            DocumentHandler.delete_paragraph(block.end.paragraph)

    def remove_block_placeholder(self, document, block):
        """
        Remove a pair of placeholders inside an everything in between
        :param Document document: the current document that contains tables which will contains paragraph with placeholders
        :param BlockPlaceholder block: the BlockPlaceholder to remove
        """
        # Extracted contains everything between the blocks placeholders
        extracted = DocumentHandler.between(document, block)

        if block.start.paragraph and block.start.paragraph._element.getparent():
            DocumentHandler.delete_paragraph(block.start.paragraph)
        
        # Delete data between the placeholders
        for e in extracted:
            DocumentHandler.delete_element(e)
        
        # delete the ending placeholder
        if block.end.paragraph and block.end.paragraph._element.getparent():
            DocumentHandler.delete_paragraph(block.end.paragraph)

    def strip_block_placeholder(self, block):
        """
        Remove a block placeholder, but keep its content in place. Used for a fulfilled if
        :param Document document: the current document that contains tables which will contains paragraph with placeholders
        :param BlockPlaceholder block: the BlockPlaceholder to strip
        """
        # delete the starting placeholder
        if block.start.paragraph is not None:
            DocumentHandler.delete_paragraph(block.start.paragraph)

        # delete the ending placeholder
        if block.end.paragraph is not None:
            DocumentHandler.delete_paragraph(block.end.paragraph)

    def replace_placeholder(self, document, placeholder, widget, font_reference):
        """
        Replace a placeholders inside the document / a table cell by their content
        :param Document document: the current document / table cell
        :param Union[BlockPlaceholder, Placeholder, HeaderPlaceholder, FooterPlaceholder] block: the placeholder to replace
        :param Union[Table, Text, PuppeteerContent, None] widget: what should replace the placeholder
        """
        if isinstance(placeholder, BlockPlaceholder):
            self.replace_block_placeholder(document, placeholder, widget)
        elif isinstance(placeholder, HeaderPlaceholder):
            self.replace_simple_placeholder(document.sections[placeholder.section_number].header, placeholder, widget, font_reference)
        elif isinstance(placeholder, FooterPlaceholder):
            self.replace_simple_placeholder(document.sections[placeholder.section_number].footer, placeholder, widget, font_reference)
        else:
            self.replace_simple_placeholder(document, placeholder, widget, font_reference)
