(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('BinnedXYChart', BinnedXYChart);

    // (!) This service previously was in static/dataiku/js/simple_report/scatter/aggr_scatter.js
    function BinnedXYChart(ChartManager, ChartDataWrapperFactory, BinnedXYUtils, d3Utils, D3ChartAxes, ChartDataUtils, ChartAxesUtils, SVGUtils, ChartYAxisPosition) {
        return function($container, chartDef, chartHandler, axesDef, data) {
            const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef),
                facetLabels = chartData.getAxisLabels('facet') || [null],
                pointsData = BinnedXYUtils.prepareData(chartDef, chartData),
                yAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT),
                yCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, yAxisID);

            const sizeMeasure = (chartDef.sizeMeasure.length) ? 0 : -1,
                colorMeasure = (chartDef.colorMeasure.length && chartDef.sizeMeasure.length) ? 1 : (chartDef.colorMeasure.length ? 0 : -1);
            let sizeScale = null,
                colorScale = null;

            if (sizeMeasure >= 0) {
                if (chartDef.variant == 'binned_xy_hex') {
                    sizeScale = d3.scale.pow().exponent(.5).domain(ChartDataUtils.getMeasureExtent(chartData, sizeMeasure, true))
                        .range([10, chartDef.hexbinRadius]);
                } else {
                    sizeScale = d3.scale.sqrt().domain(ChartDataUtils.getMeasureExtent(chartData, sizeMeasure, true));
                }
                chartDef.sizeMeasure.$mIdx = sizeMeasure;
            }

            if (colorMeasure >= 0) {
                chartDef.colorMeasure.$mIdx = colorMeasure;
            }

            if (chartDef.variant === 'binned_xy_hex') {
                // no range modifications available for hex charts
                ChartAxesUtils.resetCustomExtents([chartDef.xAxisFormatting]);
                ChartAxesUtils.resetCustomExtents(chartDef.yAxesFormatting);
            }

            const drawFrame = function(frameIdx, chartBase) {
                chartData.fixAxis('animation', frameIdx);
                facetLabels.forEach(function(facetLabel, f) {
                    const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                    if (chartDef.variant == 'binned_xy_hex') {
                        HexBinnedXYChartDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', f), chartBase, pointsData, f);
                    } else if (chartDef.variant === 'binned_xy_rect') {
                        RectBinnedXYChartDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', f), chartBase, pointsData, f);
                    } else {
                        BinnedXYChartDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', f), chartBase, pointsData, f);
                    }
                });
            };


            ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                {
                    x: { type: 'DIMENSION', mode: 'POINTS', padding: 1, dimension: chartDef.xDimension[0], name: 'x', customExtent: chartDef.xAxisFormatting.customExtent },
                    [yAxisID]: { id: yAxisID, type: 'DIMENSION', mode: 'POINTS', padding: 1, dimension: chartDef.yDimension[0], name: 'y', customExtent: yCustomExtent, position: ChartYAxisPosition.LEFT }
                },
                { type: 'MEASURE', measureIdx: colorMeasure });

            function getPointColor(d) {
                if (colorScale) {
                    return colorScale(chartData.aggr(colorMeasure).get(d));
                } else {
                    return chartDef.colorOptions.singleColor;
                }
            }

            function getCustomExtentFraction(chartDef, yAxisID) {
                // Adjust the radius with the smallest fraction of y or x custom extent
                const yFraction = ChartAxesUtils.getCustomExtentRatio(ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, yAxisID));
                const xFraction = ChartAxesUtils.getCustomExtentRatio(chartDef.xAxisFormatting.customExtent);
                return Math.min(yFraction, xFraction);
            }

            /**
             * Get the size of one step within the axis range.
             * Useful for computing the size of elements in chart (e.g bubbles or rectangles)
             * @param {string} axisName - 'x' or 'y'
             * @param {d3 axis} axis
             * @param {ChartTensorDataWrapper} chartData
             * @returns the step size
             */
            function getRangeStep(axisName, axis, chartData) {
                if (axis.scaleType !== D3ChartAxes.scaleTypes.ORDINAL) {
                    const labels = chartData.getAxisLabels(axisName);
                    if (labels.length) {
                        // Use the last step as this will be the smallest in case of log scale
                        const label = axis.scaleType === D3ChartAxes.scaleTypes.LOG ? labels[labels.length - 1] : labels[0];
                        if (label && label.min !== label.max) {
                            const scale = axis.scale();
                            return Math.abs(scale(label.min) - scale(label.max));
                        }
                    }
                }
                return d3Utils.getOrdinalScaleRangeStep(axis.ordinalScale);
            }

            function getPosition(axisName, axis, chartData, value) {
                if (axis.scaleType !== D3ChartAxes.scaleTypes.ORDINAL) {
                    const axisLabel = chartData.getAxisLabels(axisName)[value];
                    if (axisLabel.tsValue) {
                        return axis.scale()((axisLabel.min + axisLabel.max) / 2);
                    }
                    return axis.scale()(axisLabel.sortValue);
                }
                return axis.ordinalScale(value);
            }

            function BinnedXYChartDrawer(g, chartDef, chartHandler, chartData, chartBase, pointsData, f) {

                const xStep = getRangeStep('x', chartBase.xAxis, chartData);
                const yStep = getRangeStep('y', chartBase.yAxes[0], chartData);
                const radius = Math.min(xStep / 3.5, yStep / 3.5);

                if (sizeMeasure >= 0) {
                    sizeScale.range([1, Math.min(xStep / 2, yStep / 2)]);
                }
                if (colorMeasure >= 0) {
                    colorScale = chartBase.colorScale;
                }

                // Wrapper to contain all points (used to apply clip-path)
                let wrapper = g.selectAll('g.points-wrapper');
                if (wrapper.empty()) {
                    g.append('g').attr('class', 'points-wrapper');
                    wrapper = g.selectAll('g.points-wrapper');
                }

                const points = wrapper.selectAll('circle.point').data(pointsData, function(d) {
                    return d.x + '-' + d.y;
                });
                points.enter().append('circle')
                    .attr('class', 'point')
                    .attr('transform', function(d) {
                        return 'translate('
                            + getPosition('x', chartBase.xAxis, chartData, d.x) + ','
                            + getPosition('y', chartBase.yAxes[0], chartData, d.y)
                            + ')';
                    })
                    .attr('fill', getPointColor)
                    .attr('opacity', chartDef.colorOptions.transparency)
                    .each(function(d) {
                        chartBase.tooltips.addTooltipHandlers(this, angular.extend({}, d, { facet: f }), getPointColor(d));
                        chartBase.contextualMenu.addContextualMenuHandler(this, angular.extend({}, d, { facet: f }));
                    });

                points.transition()
                    .attr('r', function(d) {
                        if (chartData.getCount(d) > 0) {
                            if (sizeScale) {
                                // Note: circle radius cannot be negative
                                return Math.max(0, Math.ceil(sizeScale(chartData.aggr(sizeMeasure).get(d)) * getCustomExtentFraction(chartDef, chartBase.yAxes[0].id)));
                            } else {
                                return radius;
                            }
                        } else {
                            return 0;
                        }
                    })
                    .attr('fill', getPointColor);

                // Clip paths to prevent points from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
                ChartAxesUtils.isCroppedChart(chartDef) && SVGUtils.clipPaths(chartBase, g, wrapper);
            }

            function RectBinnedXYChartDrawer(g, chartDef, chartHandler, chartData, chartBase, pointsData, f) {

                const xStep = getRangeStep('x', chartBase.xAxis, chartData);
                const yStep = getRangeStep('y', chartBase.yAxes[0], chartData);
                let widthScale, heightScale;

                if (sizeMeasure >= 0) {
                    widthScale = sizeScale.copy().range([0.1 * xStep, xStep]);
                    heightScale = sizeScale.copy().range([0.1 * yStep, yStep]);
                }
                if (colorMeasure >= 0) {
                    colorScale = chartBase.colorScale;
                }

                const rectWidth = function(d) {
                    if (chartData.getCount(d) > 0) {
                        if (widthScale) {
                            // Note: rect width attribute cannot be negative
                            return Math.max(0, Math.ceil(widthScale(chartData.aggr(sizeMeasure).get(d)) * getCustomExtentFraction(chartDef, chartBase.yAxes[0].id)));
                        } else {
                            return xStep;
                        }
                    } else {
                        return 0;
                    }
                };

                const rectHeight = function(d) {
                    if (chartData.getCount(d) > 0) {
                        if (heightScale) {
                            // Note: rect height attribute cannot be negative
                            return Math.max(0, Math.ceil(heightScale(chartData.aggr(sizeMeasure).get(d)) * getCustomExtentFraction(chartDef, chartBase.yAxes[0].id)));
                        } else {
                            return yStep;
                        }
                    } else {
                        return 0;
                    }
                };

                const points = g.selectAll('rect.point').data(pointsData, function(d) {
                    return d.x + '-' + d.y;
                });
                points.enter().append('rect')
                    .attr('class', 'point')
                    .attr('fill', getPointColor)
                    .attr('opacity', chartDef.colorOptions.transparency)
                    .each(function(d) {
                        chartBase.tooltips.addTooltipHandlers(this, angular.extend({}, d, { facet: f }), getPointColor(d));
                        chartBase.contextualMenu.addContextualMenuHandler(this, angular.extend({}, d, { facet: f }));
                    });

                points.transition()
                    .attr('x', function(d) {
                        const x = getPosition('x', chartBase.xAxis, chartData, d.x);
                        return x - rectWidth(d) / 2;
                    })
                    .attr('y', function(d) {
                        const y = getPosition('y', chartBase.yAxes[0], chartData, d.y);
                        return y - rectHeight(d) / 2;
                    })
                    .attr('width', rectWidth)
                    .attr('height', rectHeight)
                    .attr('fill', getPointColor);

                // Clip paths to prevent points from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
                ChartAxesUtils.isCroppedChart(chartDef) && SVGUtils.clipPaths(chartBase, g, points);
            }

            function HexBinnedXYChartDrawer(g, chartDef, chartHandler, chartData, chartBase, pointsData, f) {
                if (colorMeasure >= 0) {
                    colorScale = chartBase.colorScale;
                }

                const radius = BinnedXYUtils.getRadius(chartDef, chartHandler.request.$expectedVizWidth, chartHandler.request.$expectedVizHeight);
                if (sizeScale) {
                    sizeScale.range([3, radius]);
                }

                function hexagon(radius) {
                    const d3_hexbinAngles = d3.range(0, 2 * Math.PI, Math.PI / 3);
                    let x0 = 0,
                        y0 = 0;
                    return d3_hexbinAngles.map(function(angle) {
                        const x1 = Math.sin(angle) * radius,
                            y1 = -Math.cos(angle) * radius,
                            dx = x1 - x0,
                            dy = y1 - y0;
                        x0 = x1;
                        y0 = y1;
                        return [dx, dy];
                    });
                }

                function hexagonPath(radius) {
                    return 'm' + hexagon(radius).join('l') + 'z';
                }

                /*
                 * Approximate chart dimensions are in ChartRequestComputer to determine the number of X & Y hexagons to be generated by the backend
                 * These dimensions may be changed by initChart (margins adjustements to fit axis labels or space used by the OUTER_* legends placement)
                 * We rescale the mainzone to compensate for these changes and still show all generated hexagons
                 */

                const mainZone = {
                    g: g.selectAll('g.mainzone').data([null]),
                    width: chartBase.vizWidth,
                    height: chartBase.vizHeight,
                    x: 0,
                    y: chartHandler.request.$expectedVizHeight - chartBase.vizHeight,
                    getScaleX() {
                        return chartBase.vizWidth / chartHandler.request.$expectedVizWidth;
                    },
                    getScaleY() {
                        return this.height / chartHandler.request.$expectedVizHeight;
                    }
                };

                mainZone.g.enter().append('g').attr('class', 'mainzone');
                mainZone.g.attr('transform', 'scale(' + mainZone.getScaleX() + ', ' + mainZone.getScaleY() + ') translate(' + mainZone.x + ', ' + mainZone.y + ')');

                const hexagons = mainZone.g.selectAll('path.hexagon').data(pointsData, function(d) {
                    return d.x + '-' + d.y;
                });

                function formatHexagons(sel) {
                    return sel.attr('transform', function(d) {
                        let hexagonX = d.x * radius * 2 * Math.sin(Math.PI / 3);
                        const hexagonY = mainZone.height - (d.y * radius * 1.5);
                        if (d.y % 2 == 1) { // To position hexagons in diagonal
                            hexagonX += Math.sin(Math.PI / 3) * radius;
                        }
                        return 'translate(' + (hexagonX + parseInt(radius)) + ',' + (hexagonY - parseInt(radius)) + ')';
                    }).attr('d', function(d) {
                        if (chartData.getCount(d) > 0) {
                            if (sizeScale) {
                                return hexagonPath(sizeScale(chartData.aggr(sizeMeasure).get(d)));
                            } else {
                                return hexagonPath(radius);
                            }
                        } else {
                            return 'm0,0z';
                        }
                    }).attr('fill', getPointColor);
                };

                hexagons.enter().append('path').attr('class', 'hexagon')
                    .call(formatHexagons)
                    .attr('opacity', chartDef.colorOptions.transparency)
                    .each(function(d) {
                        chartBase.tooltips.addTooltipHandlers(this, angular.extend({}, d, { facet: f }), getPointColor(d));
                        chartBase.contextualMenu.addContextualMenuHandler(this, angular.extend({}, d, { facet: f }));
                    });
                hexagons.exit().remove();

                hexagons.transition().call(formatHexagons);
            }
        };
    }
})();
