(function () {
  "use strict";

  angular.module("dataiku.opals").factory("OpalsMessageService", OpalsMessageService);

  function OpalsMessageService(
    $rootScope,
    $state,
    $http,
    Logger,
    Assert,
    WT1,
    DataikuAPI,
    UserImageUrl,
    TopbarDrawersService,
    QuestionnaireService,
    TOPBAR_DRAWER_IDS,
    HomeBannerService,
    DataikuCloudService
  ) {
    const FRAME_IDS = {
      HELP_CENTER: "help-center",
      HOME_BANNER: "home-banner",
      HEADLESS: "headless",
    };

    const HOME_BANNER_STATE = {
      OPEN: "open",
      CLOSE: "closed",
    };

    const AUTHORIZED_MESSAGE_ORIGINS = {
      DSS: "DSS-MESSAGE",
      OPALS: "OPALS-MESSAGE",
    };

    const PAGE_SPECIFIC_TOURS_RECOMMENDATIONS = {
        FLOW: "FLOW",
        EXPLORE: "EXPLORE",
        PREPARE: "PREPARE",
      }

    const DSS_MESSAGE_TYPES = {
      OPALS_DSS_CONTEXT_CHANGE: "OPALS_DSS_CONTEXT_CHANGE",
      OPALS_DSS_STATE_CHANGE: "OPALS_DSS_STATE_CHANGE",
      OPALS_HELP_DRAWER_DOCKED: "OPALS_HELP_DRAWER_DOCKED",
      OPALS_HELP_DRAWER_MINIMIZED: "OPALS_HELP_DRAWER_MINIMIZED",
      OPALS_NAVIGATE_TO: "OPALS_NAVIGATE_TO",
      OPALS_HOME_BANNER_CLOSED: "OPALS_HOME_BANNER_CLOSED",
      OPALS_SET_LOCAL_STORAGE: "OPALS_SET_LOCAL_STORAGE",
      OPALS_GET_LOCAL_STORAGE: "OPALS_GET_LOCAL_STORAGE",
      OPALS_RECOMMEND_PAGE_SPECIFIC_TOUR: "OPALS_RECOMMEND_PAGE_SPECIFIC_TOUR",
    };

    const OPALS_MESSAGE_TYPES = {
      DSS_DOCK_OPALS_HELP_DRAWER: "DSS_DOCK_OPALS_HELP_DRAWER",
      DSS_GET_ADDITIONAL_HELP_MENU_ITEMS: "DSS_GET_ADDITIONAL_HELP_MENU_ITEMS",
      DSS_GET_CURRENT_PROJECT_ACTIVITY_SUMMARY: "DSS_GET_CURRENT_PROJECT_ACTIVITY_SUMMARY",
      DSS_GET_CURRENT_PROJECT_OPALS_METADATA: "DSS_GET_CURRENT_PROJECT_OPALS_METADATA",
      DSS_GET_CURRENT_PROJECT_SUMMARY: "DSS_GET_CURRENT_PROJECT_SUMMARY",
      DSS_GET_CURRENT_URL: "DSS_GET_CURRENT_URL",
      DSS_GET_GIVE_FEEDBACK_STATUS: "DSS_GET_GIVE_FEEDBACK_STATUS",
      DSS_GET_INSTANCE_PLUGINS: "DSS_GET_INSTANCE_PLUGINS",
      DSS_GET_INTERCOM_STATUS: "DSS_GET_INTERCOM_STATUS",
      DSS_GET_NPS_SURVEY_STATUS: "DSS_GET_NPS_SURVEY_STATUS",
      DSS_GET_STUDIO_ADMIN_CONTACT: "DSS_GET_STUDIO_ADMIN_CONTACT",
      DSS_GET_SUPPORT_LINKS: "DSS_GET_SUPPORT_LINKS",
      DSS_GET_USERS: "DSS_GET_USERS",
      DSS_GET_USER_IMAGE: "DSS_GET_USER_IMAGE",
      DSS_GET_VERSION: "DSS_GET_VERSION",
      DSS_GET_SPACE_ID: "DSS_GET_SPACE_ID",
      DSS_GET_WIKI_ARTICLES: "DSS_GET_WIKI_ARTICLES",
      DSS_HIDE_OPALS_HELP_DRAWER: "DSS_HIDE_OPALS_HELP_DRAWER",
      DSS_HIDE_OPALS_HOME_BANNER: "DSS_HIDE_OPALS_HOME_BANNER",
      DSS_SHOW_OPALS_HOME_BANNER: "DSS_SHOW_OPALS_HOME_BANNER",
      DSS_LAUNCH_INTERCOM: "DSS_LAUNCH_INTERCOM",
      DSS_MINIMIZE_OPALS_HELP_DRAWER: "DSS_MINIMIZE_OPALS_HELP_DRAWER",
      DSS_OPALS_READY: "DSS_OPALS_READY",
      DSS_OPEN_GIVE_FEEDBACK_MODAL: "DSS_OPEN_GIVE_FEEDBACK_MODAL",
      DSS_OPEN_KEYBOARD_SHORTCUTS_MODAL: "DSS_OPEN_KEYBOARD_SHORTCUTS_MODAL",
      DSS_OPEN_NPS: "DSS_OPEN_NPS",
      DSS_OPEN_PLUGINS_STORE: "DSS_OPEN_PLUGINS_STORE",
      DSS_SEND_WT1_EVENT: "DSS_SEND_WT1_EVENT",
      DSS_SET_MOVE_OVERLAY_GEOMETRY: "DSS_SET_MOVE_OVERLAY_GEOMETRY",
      DSS_GET_SETTINGS_ALLOW_RAISING_TICKETS: "DSS_GET_SETTINGS_ALLOW_RAISING_TICKETS",
      DSS_OPEN_HELP_DRAWER: "DSS_OPEN_HELP_DRAWER",
      DSS_GET_LOGGED_IN_USER_INFO: "DSS_GET_LOGGED_IN_USER_INFO",
      DSS_GET_HOME_BANNER_INFO: "DSS_GET_HOME_BANNER_INFO",
      DSS_GET_THEME_INFO: "DSS_GET_THEME_INFO",
      DSS_GET_CODE_ENV_PERMISSIONS: "DSS_GET_CODE_ENV_PERMISSIONS",
      DSS_START_PAGE_SPECIFIC_TOUR: "DSS_START_PAGE_SPECIFIC_TOUR",
      DSS_GET_IS_ASSISTANT_ENABLED: "DSS_GET_IS_ASSISTANT_ENABLED",
    };

    const OPALS_EVENT_CALLBACKS = [
      // Events with a response to Opals
      { type: OPALS_MESSAGE_TYPES.DSS_GET_ADDITIONAL_HELP_MENU_ITEMS, callback: handleGetAdditionalHelpMenuItems },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_CURRENT_PROJECT_ACTIVITY_SUMMARY, callback: handleGetCurrentProjectActivitySummary },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_CURRENT_PROJECT_OPALS_METADATA, callback: handleGetCurrentProjectMetadata },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_CURRENT_PROJECT_SUMMARY, callback: handleGetCurrentProjectSummary },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_CURRENT_URL, callback: handleGetCurrentUrl },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_GIVE_FEEDBACK_STATUS, callback: handleGetGiveFeedbackStatus },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_INSTANCE_PLUGINS, callback: handleGetInstancePlugins },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_INTERCOM_STATUS, callback: handleGetIntercomStatus },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_NPS_SURVEY_STATUS, callback: handleGetNpsSurveyStatus },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_STUDIO_ADMIN_CONTACT, callback: handleGetStudioAdminContact },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_SUPPORT_LINKS, callback: handleGetSupportLinks },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_USERS, callback: handleGetUsers },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_USER_IMAGE, callback: handleGetUserImage },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_VERSION, callback: handleGetVersion },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_SPACE_ID, callback: handleGetSpaceId },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_WIKI_ARTICLES, callback: handleGetWikiArticles },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_SETTINGS_ALLOW_RAISING_TICKETS, callback: handleGetAllowTicketsSetting },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_LOGGED_IN_USER_INFO, callback: handleGetUserInfo },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_HOME_BANNER_INFO, callback: handleGetBannerInfo },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_THEME_INFO, callback: handleGetThemeInfo },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_CODE_ENV_PERMISSIONS, callback: handleGetCodeEnvPermissions },
      { type: OPALS_MESSAGE_TYPES.DSS_GET_IS_ASSISTANT_ENABLED, callback: handleGetIsAssistantEnabled },

      // Events without any response to Opals
      { type: OPALS_MESSAGE_TYPES.DSS_DOCK_OPALS_HELP_DRAWER, callback: handleDockHelpDrawer },
      { type: OPALS_MESSAGE_TYPES.DSS_HIDE_OPALS_HELP_DRAWER, callback: handleHideHelpDrawer },
      { type: OPALS_MESSAGE_TYPES.DSS_HIDE_OPALS_HOME_BANNER, callback: handleHideHomeBanner },
      { type: OPALS_MESSAGE_TYPES.DSS_SHOW_OPALS_HOME_BANNER, callback: handleShowHomeBanner },
      { type: OPALS_MESSAGE_TYPES.DSS_LAUNCH_INTERCOM, callback: handleLaunchIntercom },
      { type: OPALS_MESSAGE_TYPES.DSS_MINIMIZE_OPALS_HELP_DRAWER, callback: handleMinimizeHelpDrawer },
      { type: OPALS_MESSAGE_TYPES.DSS_OPALS_READY, callback: handleOpalsReady },
      { type: OPALS_MESSAGE_TYPES.DSS_OPEN_GIVE_FEEDBACK_MODAL, callback: handleOpenGiveFeedbackModal },
      { type: OPALS_MESSAGE_TYPES.DSS_OPEN_KEYBOARD_SHORTCUTS_MODAL, callback: handleOpenKeyboardShortcutsModal },
      { type: OPALS_MESSAGE_TYPES.DSS_OPEN_NPS, callback: handleOpenNps },
      { type: OPALS_MESSAGE_TYPES.DSS_OPEN_PLUGINS_STORE, callback: handleOpenPluginsStore },
      { type: OPALS_MESSAGE_TYPES.DSS_SET_MOVE_OVERLAY_GEOMETRY, callback: handleSetMoveOverlayGeometry },
      { type: OPALS_MESSAGE_TYPES.DSS_SEND_WT1_EVENT, callback: handleSendWT1Event },
      { type: OPALS_MESSAGE_TYPES.DSS_OPEN_HELP_DRAWER, callback: handleOpenHelpDrawer },
      { type: OPALS_MESSAGE_TYPES.DSS_START_PAGE_SPECIFIC_TOUR, callback: handleStartPageSpecificTour },
    ];

    // this map is used to handle the responses to the request messages thank to callbacks
    const requestPromiseMap = new Map(); 

    const DEFAULT_THEME_TEXT_COLOR = "rgb(51, 51, 51)"; // #333333

    const contentWindows = new Map();
    const frameIds = new Map();
    const readyContentWindows = new Set();

    const messageBuffersByFrameId = new Map();
    const messageBuffersByContentWindow = new Map();

    let dssState = null;
    let wt1EventBuffer = [];

    function getAppConfig() {
      return new Promise((resolve) => {
        if ($rootScope.appConfig) {
          resolve($rootScope.appConfig);
          return;
        }
        const unregister = $rootScope.$watch("appConfig", (newVal) => {
          if (newVal) {
            resolve(newVal);
            unregister();
          }
        });
      });
    }

    function getBaseUrl() {
      return getAppConfig().then((appConfig) => {
        return appConfig.opalsUrl ? new URL(appConfig.opalsUrl) : null;
      });
    }

    function setFrame(frameId, frame) {
      contentWindows.set(frameId, frame.contentWindow);
      frameIds.set(frame.contentWindow, frameId);
      if (readyContentWindows.has(frame.contentWindow)) {
        sendBufferedMessages(frame.contentWindow);
      }
    }

    function unregisterFrame(frameId, frame) {
      contentWindows.delete(frameId);
      frameIds.delete(frame.contentWindow);
      readyContentWindows.delete(frame.contentWindow);
      messageBuffersByFrameId.delete(frameId);
      messageBuffersByContentWindow.delete(frame.contentWindow);
    }

    function sendBufferedMessages(contentWindow) {
      sendMessagesInBuffer(messageBuffersByFrameId, frameIds.get(contentWindow), contentWindow);
      sendMessagesInBuffer(messageBuffersByContentWindow, contentWindow, contentWindow);
    }

    function sendMessagesInBuffer(buffers, key, contentWindow) {
      const buffer = buffers.get(key);
      if (buffer) {
        getBaseUrl().then((baseUrl) => {
          buffer.forEach((message) => contentWindow.postMessage(message, baseUrl.origin))
          buffers.delete(key);
        });
      }
    }

    function setDssState(data) {
      dssState = data;
      sendBufferedWT1Events();
    }

    function sendBufferedWT1Events() {
      wt1EventBuffer.forEach((e) => WT1.event(e.type, enrichWT1EventParams(e.params)));
      wt1EventBuffer = [];
    }

    function enrichWT1EventParams(params) {
      if (!params) {
        params = {};
      }
      params.context = dssState.context ? dssState.context.join("|") : "";
      return params;
    }

    function handleMessageEvent(event) {
      getBaseUrl().then((baseUrl) => {
        if (event.data.origin !== AUTHORIZED_MESSAGE_ORIGINS.OPALS || event.origin !== baseUrl.origin) {
          return;
        }
        Logger.debug("OPALS message:", event.data);
        handleOpalsMessageEvent(event);
      });
    }

    async function handleOpalsMessageEvent(opalsMessageEvent) {
      let response;
      let ok = true;
      const opalsEvent = OPALS_EVENT_CALLBACKS.find((event) => event.type === opalsMessageEvent.data.type);
      if (opalsEvent && opalsMessageEvent.data.kind !== "response") {
        try {
          response = await opalsEvent.callback(opalsMessageEvent.data, opalsMessageEvent);
          $rootScope.$apply();
        } catch (error) {
          response = "Error while handling request message";
          ok = false;
          Logger.error(error);
        }
      } else if (opalsMessageEvent.data.kind === "response") {
        const promise = requestPromiseMap.get(opalsMessageEvent.data.id);
        if (promise) {
          requestPromiseMap.delete(opalsMessageEvent.data.id);
          clearTimeout(promise.timeout);
          if (opalsMessageEvent.data.ok) {
            promise.resolve(opalsMessageEvent.data);
          } else {
            promise.reject(`Request failed: ${JSON.stringify(opalsMessageEvent.data)}`);
          }
        }
      } else {
        response = "Unhandled request message";
        ok = false;
        if (window.devInstance) {
          Logger.info("OPALS message not handled:", opalsMessageEvent.data);
        }
      }
      if (opalsMessageEvent.data.kind === "request") {
        createResponseMessage(opalsMessageEvent.data.id, opalsMessageEvent.data.type, response, ok).then((message) => {
          if (!sendMessageObject(null, opalsMessageEvent.source, message)) {
            bufferMessage(messageBuffersByContentWindow, opalsMessageEvent.source, message);
          }
        });
      }
    }

    function createResponseMessage(id, type, payload, ok) {
      return createMessage(type, payload).then((message) => {
        return { ...message, id, ok, kind: "response" }
      });
    }

    function createRequestMessage(id, type, payload) {
      return createMessage(type, payload).then((message) => {
        return { ...message, id, kind: "request" }
      });
    }

    function createMessage(type, payload) {
      return getAppConfig().then((appConfig) => {
        let hlicenseId;
        let instanceId;
        const licenseContent = appConfig.licensing.licenseContent;
        if (licenseContent) {
          hlicenseId = licenseContent.licenseId ? md5(licenseContent.licenseId) : null;
          instanceId = licenseContent.instanceId ? licenseContent.instanceId : null;
        } else {
          hlicenseId = null;
          instanceId = null;
        }
        return {
          metadata: {
            hhost: md5(location.host),
            hlicenseId: hlicenseId,
            instanceId: instanceId,
            version: 3,
          },
          origin: AUTHORIZED_MESSAGE_ORIGINS.DSS,
          type,
          payload,
        };
      });
    }

    function bufferMessage(buffers, key, message) {
      let buffer = buffers.get(key);
      if (!buffer) {
        buffer = [];
        buffers.set(key, buffer);
      }
      buffer.push(message);
    }

    function sendMessage(frameId, type, payload) {
      createMessage(type, payload).then((message) => {
        if (!sendMessageObject(frameId, null, message)) {
          bufferMessage(messageBuffersByFrameId, frameId, message);
        }
      });
    }

    function sendRequestMessage(frameId, type, payload) {
      const messageId = generateUniqueId();

      const promise = new Promise((resolve, reject) => {
        const timeout = setTimeout(() => {
          reject(new Error("Request timed out: " + type));
          requestPromiseMap.delete(messageId);
        }, 60000);
        requestPromiseMap.set(messageId, { resolve, reject, timeout });
      });

      createRequestMessage(messageId, type, payload).then((message) => {
        if (!sendMessageObject(frameId, null, message)) {
          bufferMessage(messageBuffersByFrameId, frameId, message);
        }
      });

      return promise;
    }

    function sendMessageObject(frameId, contentWindow, message) {
      Assert.trueish(frameId && !contentWindow || !frameId && contentWindow,
        "Either both frameId and contentWindow specified or neither specified");
      if (!contentWindow) {
        contentWindow = contentWindows.get(frameId);
      }
      if (contentWindow && readyContentWindows.has(contentWindow)) {
        getBaseUrl().then((baseUrl) => {
          contentWindow.postMessage(message, baseUrl.origin);
        });
        return true;
      } else {
        return false;
      }
    }

    function sendWT1Event(type, params) {
      if (dssState) {
        WT1.event(type, enrichWT1EventParams(params));
      } else {
        wt1EventBuffer.push({ type, params });
      }
    }

    function getLocalStorage(key) {
      return sendRequestMessage(FRAME_IDS.HEADLESS, DSS_MESSAGE_TYPES.OPALS_GET_LOCAL_STORAGE,key)
      .then((data)=>{
        return data.payload
      })
      .catch((err)=>{
        Logger.error(`Error retrieving local storage value key '${key}':`, err);
      })
    }

    function setLocalStorage(key, value) {
      return sendRequestMessage(FRAME_IDS.HEADLESS, DSS_MESSAGE_TYPES.OPALS_SET_LOCAL_STORAGE,{'key':key, 'value': value})
      .catch((err)=>{
        Logger.error(`Error setting local storage key '${key}' to '${value}' :`, err);
      })
    }

    // Event handlers ---------------------------------------------------------

    function handleGetAdditionalHelpMenuItems() {
      return $rootScope.wl.additionalHelpMenuItems;
    }

    function handleGetIsAssistantEnabled() {
      return {
        isEnabled: $rootScope.appConfig.askDataikuEnabled,
        isTelemetryEnabled: $rootScope.appConfig.askDataikuTelemetryEnabled,
      };
    }

    let projectActivitySummaries = new Map();

    async function handleGetCurrentProjectActivitySummary() {
      const projectKey = $state.params.projectKey;
      if (!projectActivitySummaries.has(projectKey)) {
        const summary = (await DataikuAPI.projects.activity.getActivitySummary(projectKey, "all")).data;
        projectActivitySummaries.set(projectKey, summary);
      }
      return projectActivitySummaries.get(projectKey);
    }

    async function handleGetCurrentProjectMetadata() {
      const projectKey = $state.params.projectKey;
      return (await DataikuAPI.projects.getProjectFeaturesUsage(projectKey)).data;
    }

    async function handleGetCurrentProjectSummary() {
      const projectKey = $state.params.projectKey;
      return (await DataikuAPI.projects.getSummary(projectKey)).data;
    }

    function handleGetCurrentUrl() {
      return window.location.href;
    }

    function handleGetGiveFeedbackStatus() {
      return !!$rootScope.wl.giveFeedbackModalEnabled;
    }

    let plugins = null;

    async function handleGetInstancePlugins() {
      if (!plugins) {
        plugins = (await DataikuAPI.plugins.list()).data;
      }
      return plugins;
    }

    function isIntercomEnabled() {
      // Intercom should always be enabled on Cloud but intercomEnabled might be False since intercomAppId is not set for Cloud licenses
        return $rootScope.intercomEnabled || DataikuCloudService.isDataikuCloud();
    }

    function handleGetIntercomStatus() {
      return isIntercomEnabled();
    }

    function handleGetNpsSurveyStatus() {
      Assert.inScope($rootScope, "appConfig");
      return $rootScope.appConfig.npsSurveyEnabled;
    }

    function handleGetStudioAdminContact() {
      Assert.inScope($rootScope, "appConfig");
      return {
        "label": $rootScope.appConfig.studioAdminContactLabel ? $rootScope.appConfig.studioAdminContactLabel : "",
        "info": $rootScope.appConfig.studioAdminContact ? $rootScope.appConfig.studioAdminContact : "",
      };
    }

    function handleGetSupportLinks() {
      Assert.inScope($rootScope, "appConfig");
      return $rootScope.appConfig.helpCenterSupportLinks;
    }

    let users = null;

    async function handleGetUsers() {
      if (!users) {
        users = (await DataikuAPI.security.listUsers()).data;
      }
      return users;
    }

    async function handleGetUserImage(data) {
      const imageSize = 24;
      const imageUrl = UserImageUrl(data.payload, imageSize);
      const httpConfig = { cache: true, responseType: "blob" };
      // We are not streaming the image data from the HTTP call to the
      // serializer since it's just a tiny icon
      const imageBlob = (await $http.get(imageUrl, httpConfig)).data;
      return await convertBlobToDataUrl(imageBlob);
    }

    function convertBlobToDataUrl(blob) {
      return new Promise((resolve, reject) => {
        const fileReader = new FileReader();
        const timeoutId = setTimeout(() => {
          fileReader.abort();
          reject("Timeout converting a Blob to a data: URL");
        }, 3000);
        fileReader.onloadend = () => {
          clearTimeout(timeoutId);
          if (!fileReader.error) {
            resolve(fileReader.result);
          } else {
            reject(fileReader.error);
          }
        };
        fileReader.readAsDataURL(blob);
      });
    }

    let user = null;

    async function handleGetUserInfo() {
      if (!user) {
        user = (await DataikuAPI.profile.get()).data;
      }
      const userProfile = $rootScope.appConfig.userProfile;
      return {
        "mayPython": userProfile.mayRegularCode || false,
        "mayJulia": userProfile.mayRegularCode || false,
        "maySQL": userProfile.maySQL || false,
        "mayR": userProfile.mayRegularCode || false,
        "mayScala": userProfile.mayRegularCode || false,
        "mayWriteProjectContent": userProfile.mayWriteProjectContent || false,
        "mayWriteDashboard": userProfile.mayWriteDashboards || false,
        "mayVisualML": userProfile.mayVisualML || false,
        "creationDate": user.creationDate,
      };
    }

    function handleGetBannerInfo() {
      return {
        "recoBannerHomeMessage": $rootScope.appConfig.recoBannerHomeMessage || "",
        "recoBannerMustReads": ($rootScope.appConfig.recoBannerMustReads  && $rootScope.appConfig.recoBannerMustReads instanceof Array)? $rootScope.appConfig.recoBannerMustReads.filter(elt => elt.label && elt.url) : [],
      };
    }

    function handleGetThemeInfo() {
      let foregroundColor;
      const element = angular.element("div.opals-home-banner-container__spinner");
      if (element.length === 0) {
        console.warn("Could not find the spinner element to detect the theme text color. " +
          `Falling back to the default value of '${DEFAULT_THEME_TEXT_COLOR}'.`);
        foregroundColor = DEFAULT_THEME_TEXT_COLOR;
      } else {
        foregroundColor = window.getComputedStyle(element[0]).getPropertyValue("color");
      }
      return { foregroundColor };
    }

    function handleGetCodeEnvPermissions() {
      return {
        "isAdmin": $rootScope.appConfig.admin,
        "mayCreateCodeEnvs": $rootScope.appConfig.globalPermissions.mayCreateCodeEnvs,
        "mayManageCodeEnvs": $rootScope.appConfig.globalPermissions.mayManageCodeEnvs,
      }
    }
    
    function sendHomeBannerState(bannerState) {
      sendMessage(
        FRAME_IDS.HOME_BANNER,
        DSS_MESSAGE_TYPES.OPALS_HOME_BANNER_CLOSED,
        bannerState
      );
    }

    function handleGetVersion() {
      Assert.inScope($rootScope, "appConfig");
      return $rootScope.appConfig.version.product_version;
    }

    function handleGetSpaceId() {
      Assert.inScope($rootScope, "appConfig");
      return $rootScope.appConfig.dkuCloudSpaceId;
    }

    let wikiArticles = null;

    async function handleGetWikiArticles() {
      if (!wikiArticles) {
        wikiArticles = (await DataikuAPI.getHomeArticles(true)).data;
      }
      return wikiArticles;
    }

    function handleGetAllowTicketsSetting() {
      return getAppConfig().then((appConfig) => {
        return appConfig.openTicketsFromOpalsEnabled;
      });
    }

    function handleDockHelpDrawer(data) {
      TopbarDrawersService.getDrawer(TOPBAR_DRAWER_IDS.OPALS_HELP).dock(data.payload);
    }

    function handleHideHelpDrawer() {
      hideHelpDrawer();
    }

    function navigateToAndShowDrawer(page, searchParams) {
      let pageWithSearchParams = page;
      if (searchParams && Object.keys(searchParams).length > 0) {
        const formattedSearchParams = [];
        for (let p in searchParams) {
          formattedSearchParams.push(`${p}=${encodeURIComponent(searchParams[p])}`);
        }
        pageWithSearchParams += "?" + formattedSearchParams.join("&");
      }
      sendMessage(
        FRAME_IDS.HELP_CENTER,
        DSS_MESSAGE_TYPES.OPALS_NAVIGATE_TO,
        pageWithSearchParams
      );
      TopbarDrawersService.getDrawer(TOPBAR_DRAWER_IDS.OPALS_HELP).show();
    }

    function handleOpenHelpDrawer(data) {
        let url = data.payload.pathname;
        if (data.payload.search) {
            url += data.payload.search;
        }
        navigateToAndShowDrawer(url);
    }

    function hideHelpDrawer() {
      TopbarDrawersService.getDrawer(TOPBAR_DRAWER_IDS.OPALS_HELP).hide();
    }

    function handleHideHomeBanner() {
      const isImplicit = false;
      HomeBannerService.hideHomeBanner(isImplicit);
      sendHomeBannerState(HOME_BANNER_STATE.CLOSE);
    }

    function handleShowHomeBanner() {
      const isImplicit = false;
      HomeBannerService.showHomeBanner(isImplicit);
      sendHomeBannerState(HOME_BANNER_STATE.OPEN);
    }

    function handleLaunchIntercom() {
      if (isIntercomEnabled()) {
        hideHelpDrawer();
        $rootScope.forceShowIntercom();
      }
    }

    function handleMinimizeHelpDrawer(data) {
      TopbarDrawersService.getDrawer(TOPBAR_DRAWER_IDS.OPALS_HELP).minimize(data.payload);
    }

    function handleOpalsReady(_, messageEvent) {
      readyContentWindows.add(messageEvent.source);
      if (frameIds.has(messageEvent.source)) {
        sendBufferedMessages(messageEvent.source);
      }
    }

    function handleOpenGiveFeedbackModal() {
      hideHelpDrawer();
      $rootScope.showFeedbackModal();
    }

    function handleOpenKeyboardShortcutsModal() {
      hideHelpDrawer();
      $rootScope.showKeyboardShortcuts();
    }

    function handleOpenNps() {
      hideHelpDrawer();
      $rootScope.showNpsSurvey();
    }

    function handleOpenPluginsStore(data) {
      hideHelpDrawer();
      let params;
      if (data.payload) {
        params = { pluginid: data.payload };
      }
      $state.go("plugins.store", params);
    }

    function handleSetMoveOverlayGeometry(data) {
      const drawer = TopbarDrawersService.getDrawer(TOPBAR_DRAWER_IDS.OPALS_HELP);
      drawer.setMoveOverlayGeometry(data.payload);
    }

    function handleSendWT1Event(data) {
      sendWT1Event(data.payload.type, data.payload.params);
    }

    function handleStartPageSpecificTour(data) {
      QuestionnaireService.setIsFromQuestionnaire(false); // in case we are triggering the tour while doing the tutorial
      hideHelpDrawer();
      switch (data.payload) {
        case PAGE_SPECIFIC_TOURS_RECOMMENDATIONS.FLOW:
          $rootScope.$emit("startFlowTour");
          break;
        case PAGE_SPECIFIC_TOURS_RECOMMENDATIONS.EXPLORE:
          $rootScope.$emit("startExploreTour");
          break;
        case PAGE_SPECIFIC_TOURS_RECOMMENDATIONS.PREPARE:
          $rootScope.$emit("startPrepareTour");
          break;
        default:
          break;
      }
    }


    // End of event handlers --------------------------------------------------

    return {
      getAppConfig,
      getBaseUrl,
      setFrame,
      setDssState,
      handleMessageEvent,
      sendMessage,
      sendWT1Event,
      sendHomeBannerState,
      unregisterFrame,
      FRAME_IDS,
      HOME_BANNER_STATE,
      DSS_MESSAGE_TYPES,
      PAGE_SPECIFIC_TOURS_RECOMMENDATIONS,
      navigateToAndShowDrawer,
      sendRequestMessage,
      getLocalStorage,
      setLocalStorage,
    };
  }
})();
