(function(){
    "use strict";

    const searchNotebook = {
        bindings : {
            notebook: "<"
        },
        templateUrl: "/static/dataiku/search-notebooks/search-notebook/search-notebook.component.html",
        controller: function(CreateModalFromComponent, searchNotebookInitModalDirective, DataikuAPI, $scope,
                             $stateParams, Dialogs, Logger, $q, searchNotebookCreateDatasetModalDirective,
                             searchNotebookEditCellNameModalDirective, WT1) {
            const $ctrl = this;
            $ctrl.selectedCell = null;
            $ctrl.mappingStats = null;
            $ctrl.schema = null;
            $ctrl.selectedField = null;
            $ctrl.leftPaneTab = "Fields";

            const resolveIndexForDatasets = (datasets) => datasets.map(d => d.params.index).join(",");

            const resolveIndexForScope = (searchScope, alreadyResolvedIndex) => {
                const deferred = $q.defer(); // use a promise to have the same return interface for datasets and the others
                if (alreadyResolvedIndex) {
                    deferred.resolve(alreadyResolvedIndex);
                    return deferred.promise;
                }
                switch (searchScope.scope) {
                    case "INDICES":
                        deferred.resolve(searchScope.indices.join(","));
                        return deferred.promise;
                    case "INDEX_PATTERN":
                        deferred.resolve(searchScope.indexPattern);
                        return deferred.promise;
                    case "DATASETS":
                        return DataikuAPI.searchNotebooks.resolveDatasets($stateParams.projectKey, searchScope.datasets)
                                .then(({data}) => resolveIndexForDatasets(data));
                    default:
                        throw new Error("Unknown scope: " + $ctrl.selectedScope);
                }
            }

            $ctrl.$onChanges = (changes) => {
                if (changes.notebook && changes.notebook.currentValue) { // initial notebook load
                    if ($ctrl.notebook.cells.length === 0) { // add a new cell if non exist
                        $ctrl.selectIndices();
                    } else {
                        const cell = $ctrl.notebook.cells[$ctrl.notebook.cells.length - 1]; // load last cell
                        $ctrl.onSelectCell(cell);
                    }
                }
            };

            const findCell = (cell) => {
                for (let i = 0; i < $ctrl.notebook.cells.length; i++) {
                    if ($ctrl.notebook.cells[i].id === cell.id) {
                        return i;
                    }
                }
                return -1;
            };

            $ctrl.onSelectedIndices = (cell, mappingStats, schema, updateNotebook) => {
                $ctrl.selectedCell = cell;
                if (updateNotebook) {
                    const index = findCell(cell);
                    if (index >= 0) {
                        $ctrl.notebook.cells[index] = cell;
                    } else {
                        cell.id = generateUniqueId();
                        $ctrl.notebook.cells.push(cell);
                    }
                }
                $ctrl.mappingStats = mappingStats;
                $ctrl.selectedCell.$mappingStats = mappingStats;
                $ctrl.schema = schema;
            };

            $ctrl.onSelectField = (fieldName) => {
                $ctrl.selectedField = {name: fieldName}; // Use an object to trigger change detection
            };

            $ctrl.onSelectCell = (cell) => {
                resolveIndexForScope(cell.searchScope, cell.$resolvedIndex).then((resolvedIndex) => {
                    cell.$resolvedIndex = resolvedIndex;
                })
                .then(() => DataikuAPI.searchNotebooks.inferSchema($stateParams.projectKey, $ctrl.notebook.connection, cell.$resolvedIndex))
                .then(({data}) => {
                    $ctrl.onSelectedIndices(cell, data.mappingStats, data.inferredSchema, false);
                })
                .catch((err) => {
                    $ctrl.onSelectedIndices(cell, null, null, false);
                    setErrorInScope.bind($scope)(err);
                });
            };

            $ctrl.cellMenu = {
                add: () => {
                    $ctrl.selectIndices(false)
                        .then(() => WT1.event("search-notebook-action-new-cell"));
                },
                editScope: () => {
                    $ctrl.selectIndices(true)
                        .then(() => WT1.event("search-notebook-action-edit-cell-scope"));
                },
                editName: (cell) => {
                    CreateModalFromComponent(searchNotebookEditCellNameModalDirective, {
                        cell: cell
                    })
                    .then(() => WT1.event("search-notebook-action-edit-name"));
                },
                copy: (cell) => {
                    WT1.event("search-notebook-action-copy-cell");
                    const newCell = angular.copy(cell);
                    newCell.id = generateUniqueId();
                    if (newCell.name) {
                        newCell.name = "Copy of " + newCell.name;
                    }
                    $ctrl.notebook.cells.push(newCell);
                    $ctrl.onSelectCell(newCell);
                },
                delete: (cell) => {
                    Dialogs.confirm($scope, "Confirm deletion", "Are you sure you want to remove this query ?").then(() => {
                        WT1.event("search-notebook-action-delete-cell");
                        const index = findCell(cell);
                        $ctrl.notebook.cells.splice(index, 1);
                        if (cell == $ctrl.selectedCell) {
                            $ctrl.selectedCell = null;
                            if ($ctrl.notebook.cells.length == 0) {
                                $ctrl.selectIndices(false);
                            } else {
                                $ctrl.onSelectCell($ctrl.notebook.cells[$ctrl.notebook.cells.length - 1]); // last cell as new
                            }
                        }
                    });
                },
                createDataset: (fakeDataset, elasticSearchQuery) => {
                   CreateModalFromComponent(searchNotebookCreateDatasetModalDirective, {
                       schema: $ctrl.schema,
                       index: $ctrl.selectedCell.$resolvedIndex,
                       connection: $ctrl.notebook.connection,
                       elasticSearchQuery: elasticSearchQuery
                   })
                   .then(() => WT1.event("search-notebook-action-create-dataset"));
                }
            };

            $ctrl.selectIndices = (edit) => {
                return CreateModalFromComponent(searchNotebookInitModalDirective, {
                    connection: $ctrl.notebook.connection,
                    onSelectedIndices: $ctrl.onSelectedIndices,
                    cell: edit ?  $ctrl.selectedCell : null
                });
            };
        }
    }

    angular.module("dataiku.searchNotebooks").component("searchNotebook", searchNotebook);
})();