import json
import os


def _get_code_env_resources_path():
    """
    Get the path of the code environment resources directory if we are in a
    code environment
    :return str: the code environment resources directory path
    """

    code_env_resources_path = os.environ.get("DKU_CODE_ENV_RESOURCES_PATH", "")
    if code_env_resources_path:
        return code_env_resources_path

    raise EnvironmentError("Resources directory is only available in a code environment")


def _get_env_vars_path():
    resources_env_variables_path = os.environ.get("DKU_CODE_ENV_RESOURCES_ENV_VARS_PATH", "")
    if not resources_env_variables_path:
        raise EnvironmentError("This method must be called from a code env resources initialization script")

    return resources_env_variables_path


def set_env_var(name, value, is_relative_path=False):
    """
    Set an environment variable to be loaded at runtime in code environment
    :param str name: Name of the environment variable
    :param str value: Value of the environment variable
    :param bool is_relative_path: True if the value is a relative path with respect to the
        code env resources directory
    """

    resources_env_variables_path = _get_env_vars_path()

    if not os.path.exists(resources_env_variables_path):
        env_variables = {"variables": {}}
    else:
        with open(resources_env_variables_path, 'r') as f:
            env_variables = json.load(f)

    if "variables" not in env_variables:
        env_variables["variables"] = {}

    env_variables["variables"][name] = {
        "value": value,
        "type": "RELATIVE_PATH" if is_relative_path else "RAW_STRING"
    }

    if is_relative_path:
        os.environ[name] = os.path.join(_get_code_env_resources_path(), value)
    else:
        os.environ[name] = value

    with open(resources_env_variables_path, 'w') as f:
        json.dump(env_variables, f)


def set_env_path(name, value):
    """
    Set an environment variable to be loaded at runtime in code environment (relative path with respect
    to the code env resources directory)
    :param str name: Name of the environment variable
    :param str value: Value of the environment variable
    """
    set_env_var(name, value, is_relative_path=True)


def delete_env_var(name):
    """
    Delete an environment variable from the code environment runtime
    :param str name: Name of the environment variable
    """

    resources_env_variables_path = _get_env_vars_path()

    if not os.path.exists(resources_env_variables_path):
        return
    else:
        with open(resources_env_variables_path, 'r') as f:
            env_variables = json.load(f)

        if name in env_variables.get("variables", {}):
            del env_variables["variables"][name]
            with open(resources_env_variables_path, 'w') as f:
                json.dump(env_variables, f)

    if name in os.environ:
        del os.environ[name]


def get_env_var(name):
    """
    Get an environment variable from the code environment runtime
    :param str name: Name of the environment variable
    :return dict: Dictionary containing the value and the type (RAW_STRING or RELATIVE_PATH) of
                  the environment variable
    """

    resources_env_variables_path = _get_env_vars_path()

    if not os.path.exists(resources_env_variables_path):
        raise FileNotFoundError("No environment variables set for the code environment runtime")
    else:
        with open(resources_env_variables_path, 'r') as f:
            env_variables = json.load(f)

        return env_variables.get("variables", {})[name]


def clear_all_env_vars():
    """
    Delete all environment variables from the code environment runtime
    """

    resources_env_variables_path = _get_env_vars_path()

    if not os.path.exists(resources_env_variables_path):
        return
    else:
        with open(resources_env_variables_path, 'r') as f:
            env_variables = json.load(f)

        for name in env_variables.get("variables", {}).keys():
            if name in os.environ:
                del os.environ[name]

        with open(resources_env_variables_path, 'w') as f:
            json.dump({}, f)
