import sys
import json
import calendar, datetime, time
import traceback, logging

from dataiku.base.utils import watch_stdin, get_clazz_in_code, get_json_friendly_error, get_argspec
from dataiku.base.socket_block_link import JavaLink, parse_javalink_args
from .code_studio_block import CodeStudioBlock
from .code_studio_template import CodeStudioTemplate
from ..runnables.progress_utils import get_progress_callback, send_result_json, send_error

logging.basicConfig(level=logging.INFO, format='[%(asctime)s] [%(process)s/%(threadName)s] [%(levelname)s] [%(name)s] %(message)s')

# socket-based connection to backend
def serve(port, secret, server_cert=None):
    link = JavaLink(port, secret, server_cert=server_cert)
    # initiate connection
    link.connect()
    # get work to do
    command = link.read_json()
    try:
        config = command.get("config", {})
        plugin_config = command.get("pluginConfig", {})
        code = command["code"]
        
        code_defines_block = command.get("codeDefinesBlock", False)
        
        logging.info("Run %s as %s" % (command["type"], code_defines_block))
        
        # get the object
        if code_defines_block:
            clazz = get_clazz_in_code(code, CodeStudioBlock)
            arg_count = len(get_argspec(clazz.__init__).args)
            block = None
            if arg_count == 3:
                block = clazz(config, plugin_config)
            else:
                raise Exception("Wrong signature of the CodeStudioBlock subclass: %i args but expected 3 (self, config, plugin_config)" % arg_count)
        else:
            clazz = get_clazz_in_code(code, CodeStudioTemplate)
            arg_count = len(get_argspec(clazz.__init__).args)
            code_studio_template = None
            if arg_count == 3:
                code_studio_template = clazz(config, plugin_config)
            else:
                raise Exception("Wrong signature of the CodeStudioTemplate subclass: %i args but expected 3 (self, config, plugin_config)" % arg_count)
                    
        # work
        call_name = command["type"]
        if call_name == 'build_spec':
            spec_in = command["in"]
            build_env = command["env"]
            template = command["template"]

            if code_defines_block:
                arg_count = len(get_argspec(block.build_spec).args)
                if arg_count == 4:
                    result = block.build_spec(spec_in, build_env, template)
                elif arg_count == 3:
                    result = block.build_spec(spec_in, build_env)
                else:
                    raise Exception("Wrong signature of the build_spec method: %i args but expected 3 (self, in, env)" % arg_count)
            else:
                arg_count = len(get_argspec(code_studio_template.build_spec).args)
                if arg_count == 2:
                    result = code_studio_template.build_spec(build_env)
                else:
                    raise Exception("Wrong signature of the build_spec method: %i args but expected 2 (self, env)" % arg_count)
                        
            link.send_string(json.dumps(result))
        elif call_name == 'build_launch':
            spec_in = command["in"]
            build_env = command["env"]

            if code_defines_block:
                arg_count = len(get_argspec(block.build_launch).args)
                if arg_count == 3:
                    result = block.build_launch(spec_in, build_env)
                else:
                    raise Exception("Wrong signature of the build_launch method: %i args but expected 3 (self, in, env)" % arg_count)
            else:
                arg_count = len(get_argspec(code_studio_template.build_launch).args)
                if arg_count == 2:
                    result = code_studio_template.build_launch(build_env)
                else:
                    raise Exception("Wrong signature of the build_launch method: %i args but expected 2 (self, env)" % arg_count)
                        
            link.send_string(json.dumps(result))
        elif call_name == 'postprocess_launch':
            spec_in = command["in"]
            build_env = command["env"]
            
            if hasattr(block, 'postprocess_launch'):
                arg_count = len(get_argspec(block.postprocess_launch).args)
                if arg_count == 3:
                    result = block.postprocess_launch(spec_in, build_env)
                else:
                    raise Exception("Wrong signature of the postprocess_launch method: %i args but expected 3 (self, in, env)" % arg_count)
            else:
                # undefined on this block, just no-op
                result = spec_in
                                        
            link.send_string(json.dumps(result))
        elif call_name == 'build_creation':
            spec_in = command["in"]
            creation_env = command["env"]

            if code_defines_block:
                arg_count = len(get_argspec(block.build_creation).args)
                if arg_count == 3:
                    result = block.build_creation(spec_in, creation_env)
                else:
                    raise Exception("Wrong signature of the build_creation method: %i args but expected 3 (self, in, env)" % arg_count)
            else:
                arg_count = len(get_argspec(code_studio_template.build_creation).args)
                if arg_count == 2:
                    result = code_studio_template.build_creation(creation_env)
                else:
                    raise Exception("Wrong signature of the build_creation method: %i args but expected 2 (self, env)" % arg_count)
                        
            link.send_string(json.dumps(result))
        elif call_name == 'build_block_list':
            build_env = command.get("buildEnv")
            launch_env = command.get("launchEnv")
            creation_env = command.get("creationEnv")

            arg_count = len(get_argspec(code_studio_template.build_block_list).args)
            if arg_count == 4:
                result = code_studio_template.build_block_list(build_env, launch_env, creation_env)
            else:
                raise Exception("Wrong signature of the build_block_list method: %i args but expected 4 (self, build_end, launch_env, creation_env)" % arg_count)
                        
            link.send_string(json.dumps(result))
        else:
            raise Exception("Wrong call type : %s" % call_name)
        # send end of stream (data is expected as a stream)
        link.send_string('')

    except:
        traceback.print_exc()
        link.send_string('') # send null to mark failure
        link.send_json(get_json_friendly_error())
    finally:
        # done
        link.close()
    

if __name__ == "__main__":
    watch_stdin()
    port, secret, server_cert = parse_javalink_args()
    serve(port, secret, server_cert=server_cert)
        
