#!/usr/bin/env python
# encoding: utf-8
"""
dataset.py : Interaction with DSS datasets
Copyright (c) 2013-2014 Dataiku SAS. All rights reserved.
"""

import numpy as np
from dataiku.base.utils import check_base_package_version
import logging
import uuid


try:
    import pandas as pd
    check_base_package_version(pd, 'pandas', '0.23.0', None, "DSS requires version 0.23 or above") # keep the version number in sync with install-python-packages.sh
except ImportError as e:
    logging.warning("Could not import pandas. Pandas support will be disabled. To enable get_dataframe and other methods, please install the pandas package")

from dataiku.core import default_project_key
from dataiku.core import dku_pandas_csv
from dataiku.core.dku_pandas_csv import pandas_date_parser_compat
from dataiku.base import remoterun
import json, os, sys, csv, time
from os import path as osp

import atexit
import warnings
import threading
import struct
import threading, logging
from datetime import datetime

# Module code
from dataiku.core import flow, base, schema_handling, dkuio
from dataiku.core.platform_exec import read_dku_json
from dataiku.core.dkujson import dump_to_filepath, load_from_filepath
from dataiku.core import intercom, metrics, dataset_write, continuous_write

FULL_SAMPLING = {"samplingMethod": "FULL"}
CSV_SEP = '\t'
CSV_DOUBLE_QUOTE = True
CSV_QUOTE_CHAR = '"'

# Loads the export button in IPython.
try:
    from ..notebook import export
    export.setup()
except:
    pass

DEFAULT_TIMEOUT = 30
if flow.FLOW is not None:
    # Timeout has been introduced to cope with ipython leaks.
    # As a default value, we have an infinite timeout when in flow.
    DEFAULT_TIMEOUT = -1

# We want to stderr something on DeprecationWarning
# But don't reset everything because pandas has set up some filters
warnings.filterwarnings("default", category=DeprecationWarning)

def unique(g):
    vals = set()
    for val in g:
        if val not in vals:
            yield val
            vals.add(val)



def none_if_throws(f):
    def aux(*args, **kargs):
        try:
            return f(*args, **kargs)
        except:
            return None
    return aux


(GENERATING,              # underlying  generator is currently working
 IDLE,                    # waiting for the generator user to call .next()
 TIMEOUT_REACHED,         # timeout has been reached
 END_OF_ITERATOR,
 TERMINATED,) = range(5)  # we reached the generator last element.


class IteratorWithTimeOut(object):

    __slots__ = ('generator', 'state', 'timeout',
                 'wake_me_up', 'touched', 'iterator', )

    def __init__(self, iterator, timeout=-1):
        self.iterator = iterator
        self.state = IDLE
        self.timeout = timeout
        self.touched = True
        self.wake_me_up = threading.Event()

    def check_timeout(self,):
        while self.state != TERMINATED:
            if self.touched is False and self.state == IDLE:
                # reached timeout !
                self.state = TIMEOUT_REACHED
                # closing underlying iterator right away
                self.iterator.close()
                # terminating the thread
                break
            if self.state == IDLE:
                self.touched = False
            self.wake_me_up.wait(self.timeout)

    def get_generator(self,):
        if self.timeout > 0:
            timeout_thread = threading.Thread(target=self.check_timeout)
            timeout_thread.daemon = True
            timeout_thread.start()
        try:
            while True:
                self.state = GENERATING
                try:
                    val = next(self.iterator)
                except StopIteration:
                    return
                self.state = IDLE
                yield val
                self.touched = True
                if self.state == TIMEOUT_REACHED:
                    # we didn't reach the end of the file
                    # we returned because of the timeout.
                    return
        finally:
            if self.state != TIMEOUT_REACHED:
                # we are here, either because
                # we reached the end of the stream
                # or the stream rose an exception.
                self.state = TERMINATED
                self.wake_me_up.set()


class Schema(list):
    """
    List of the definitions of the columns in a dataset.

    Each column definition is a dict with at least a **name** field and a **type**
    field. Available columns types include:

    +----------+-------------+------------------------------------+
    | type     | note        | sample value                       |
    +==========+=============+====================================+
    | string   |             | b'foobar'                          |
    +----------+-------------+------------------------------------+
    | bigint   | 64 bits     | 9223372036854775807                |
    +----------+-------------+------------------------------------+
    | int      | 32 bits     | 2147483647                         |
    +----------+-------------+------------------------------------+
    | smallint | 16 bits     | 32767                              |
    +----------+-------------+------------------------------------+
    | tinyint  |  8 bits     | 127                                |
    +----------+-------------+------------------------------------+
    | double   | 64 bits     | 3.1415                             |
    +----------+-------------+------------------------------------+
    | float    | 32 bits     | 3.14                               |
    +----------+-------------+------------------------------------+
    | boolean  | 32 bits     | true                               |
    +----------+-------------+------------------------------------+
    | date     | string      | "2020-12-31T00:00:00.101Z"         |
    +----------+-------------+------------------------------------+
    | array    | json string | '["foo","bar"]'                    |
    +----------+-------------+------------------------------------+
    | map      | json string | '{"foo":"bar"}'                    |
    +----------+-------------+------------------------------------+
    | object   | json string | '{"foo":{"bar":[1,2,3]}}'          |
    +----------+-------------+------------------------------------+
    | geopoint | string      | "POINT(12 24)"                     |
    +----------+-------------+------------------------------------+
    | geometry | string      | "POLYGON((1.1 1.1, 2.1 0, 0.1 0))" |
    +----------+-------------+------------------------------------+

    Each column definition has fields:

        * **name**: name of the column as string
        * **type**: type of the column as string
        * **maxLength**: maximum length of values (when applicable, typically for string)
        * **comment**: user comment on the column
        * **timestampNoTzAsDate**: for columns of type "date" in non-managed datasets, whether the actual type in the underlying SQL database or file bears timezone information
        * **originalType** and **originalSQLType**: for columns in non-managed datasets, the name of the column type in the underlying SQL database or file
        * **arrayContent**: for array-typed columns, a column definition that applies to all elements in the array
        * **mapKeys** and **mapValues**: for map-types columns, a column definition that applies to all keys (resp. values) in the map
        * **objectFields**: for object-typed columns, a list of column definitions for the sub-fields in the object
    """
    def __init__(self, data):
        list.__init__(self, data)

    def _repr_html_(self,):
        s = "<table>"
        s += "<tr><th>Column</th><th>Type</th></tr>"
        for col in self:
            s += "<tr><td>%s</td><td>%s</td></tr>" % (col["name"], col["type"])
        s += "</table>"
        return s

def create_sampling_argument(sampling='head',
                             sampling_column=None,
                             limit=None,
                             ratio=None,
                             ascending=True):
    """
    Generate sampling parameters.
    Please see https://doc.dataiku.com/dss/latest/explore/sampling.html#sampling-methods for more information.

    :param sampling: sampling method, see :meth:`dataiku.core.dataset.create_sampling_argument`. Defaults to 'head'.
    :type sampling: str, optional
    :param sampling_column: select the column used for "random-column" and "sort-column" sampling, defaults to None
    :type sampling_column: string, optional
    :param limit: set the sampling max rows count, defaults to None
    :type limit: int, optional
    :param ratio: define the max row count as a ratio (between 0 and 1) of the dataset's total row count
    :type ratio: float, optional
    :param ascending: sort in ascending order the selected column of the "sort-column" sampling, defaults to True
    :type ascending: boolean, optional
    :returns: sampling parameters
    :rtype: dict
    """
    if type(sampling) == dict:
        # HACK : in the doctor we happen to have
        # the sampling in the java format already.
        # Rather than convert them twice, we
        # use this loophole and return the sampling dictionary
        # directly.
        return sampling
    if sampling_column is not None and sampling not in ["random-column", "sort-column"]:
        raise ValueError("sampling_column argument does not make sense with %s sampling method." % sampling)
    if sampling == "head":
        if ratio is not None:
            raise ValueError("target_ratio parameter is not supported by the head sampling method.")
        if limit is None:
            return FULL_SAMPLING
        else:
            return {
                "samplingMethod": "HEAD_SEQUENTIAL",
                "maxRecords": limit
            }
    elif sampling == "random":
        if ratio is not None:
            if limit is not None:
                raise ValueError("Cannot set both ratio and limit.")
            return {
                "samplingMethod": "RANDOM_FIXED_RATIO",
                "targetRatio": ratio
            }
        elif limit is not None:
            return {
                "samplingMethod": "RANDOM_FIXED_NB",
                "maxRecords": limit
            }
        else:
            raise ValueError("Sampling method random requires either a parameter limit or ratio")
    elif sampling == "random-column":
        if sampling_column is None:
            raise ValueError("random-column sampling method requires a sampling_column argument.")
        if ratio is not None:
            raise ValueError("ratio parameter is not handled by sampling column method.")
        if limit is None:
            raise ValueError("random-column requires a limit parameter")
        return {
            "samplingMethod": "COLUMN_BASED",
            "maxRecords": limit,
            "column": sampling_column
        }
    elif sampling == "sort-column":
        if sampling_column is None:
            raise ValueError("sort-column sampling method requires a sampling_column argument.")
        if limit is None:
            raise ValueError("sort-column sampling requires a limit parameter")
        return {
            "samplingMethod": "COLUMN_ORDERED",
            "maxRecords": limit,
            "column": sampling_column,
            "ascending": ascending
        }
    else:
        raise ValueError("Sampling %s is unsupported" % sampling)


class Dataset:
    """
    Provides a handle to obtain readers and writers on a dataiku Dataset.
    From this Dataset class, you can:

        * Read a dataset as a Pandas dataframe
        * Read a dataset as a chunked Pandas dataframe
        * Read a dataset row-by-row
        * Write a pandas dataframe to a dataset
        * Write a series of chunked Pandas dataframes to a dataset
        * Write to a dataset row-by-row
        * Edit the schema of a dataset

    :param str name: The name of the dataset.
    :param str project_key: The key of the project in which the dataset is located (current project key if none is specified)
    :param boolean ignore_flow: this parameter is only relevant for recipes, not for notebooks or code in metrics or scenario steps.
                                when in a recipe, if it's left to False, then DSS also checks whether the dataset is part
                                of the inputs or outputs of the recipe and raises an error if it's not, defaults to False

    :returns: a handle to interact with the dataset
    :rtype: :class:`Dataset`
    """

    @staticmethod
    def list(project_key=None):
        """
        List the names of datasets of a given project.

        Usage example:

        .. code-block:: python

            import dataiku

            # current project datasets
            current_project_datasets = dataiku.Dataset.list()

            # given project datasets
            my_project_datasets =  dataiku.Dataset.list("my_project")

        :param str project_key: the optional key of the project to retrieve the datasets from, defaults to current project
        :returns: a list of a dataset names
        :rtype: list[str]
        """
        project_key = project_key or default_project_key()

        return intercom.jek_or_backend_json_call("datasets/list", data={
            "projectKey": project_key
        })

    def __init__(self, name, project_key=None, ignore_flow=False):
        self.name = name
        self.cols = None
        self.partitions = None
        self.read_partitions = None
        self.writePartition = None
        self.writable = False
        self.readable = False
        self.preparation_steps = None
        self.preparation_requested_output_schema = None
        self.preparation_context_project_key = None
        self.ignore_flow = ignore_flow

        # Flow mode, initialize partitions to read and write and read/write flags
        if flow.FLOW is not None and ignore_flow == False:
            for input_dataset in flow.FLOW["in"]:
                if input_dataset["smartName"] == self.name or input_dataset["fullName"] == self.name:
                    self.readable = True
                    self.name = input_dataset["fullName"]
                    if "partitions" in input_dataset:
                        self.read_partitions = input_dataset["partitions"]
            for output_dataset in flow.FLOW["out"]:
                if output_dataset["smartName"] == self.name or output_dataset["fullName"] == self.name:
                    self.name = output_dataset["fullName"]
                    self.writable = True
                    self.spec_item = output_dataset
                    if "partition" in output_dataset:
                        self.writePartition = output_dataset["partition"]
            if not self.readable and not self.writable:
                raise Exception("Dataset %s cannot be used : declare it as input or output of your recipe" % self.name)
            (self.project_key, self.short_name) = self.name.split(".", 1)

        else:
            if "." not in name:
                try:
                    self.project_key = project_key or default_project_key()
                    self.short_name = name
                    self.name = self.project_key + "." + name
                except:
                    logging.exception("Failure happened")
                    raise Exception("Dataset %s is specified with a relative name, "
                                    "but no default project was found. Please use complete name" % self.name)
            else:
                # use gave a full name
                (self.project_key, self.short_name) = self.name.split(".", 1)
                if project_key is not None and self.project_key != project_key:
                    raise ValueError("Project key %s incompatible with fullname dataset %s." % (project_key, name))
            self.readable = True
            self.writable = True
            self.spec_item = {"appendMode" : False} # notebook always overwrites

    @property
    def full_name(self):
        """
        Get the fully-qualified identifier of the dataset on the DSS instance.

        :returns: a fully qualified identifier for the dataset in the form "project_key.dataset_name"
        :rtype: str
        """
        return self.project_key + "." + self.short_name

    def get_location_info(self, sensitive_info=False):
        """
        Retrieve the location information of the dataset.

        Usage example

        .. code-block:: python

            # save a dataframe to csv with fixed name to S3
            dataset = dataiku.Dataset("my_target_dataset")
            location_info = dataset.get_location_info(True)

            s3_folder = location_info["info"]["path"] # get URI of the dataset
            import re
            # extract the bucket from the URI
            s3_bucket = re.match("^s3://([^/]+)/.*$", s3_folder).group(1)
            # extract path inside bucket
            s3_path_in_bucket = re.match("^s3://[^/]+/(.*)$", s3_folder).group(1)

            # save to S3 using boto
            from io import StringIO
            import boto3
            csv_buffer = StringIO()
            df.to_csv(csv_buffer)
            s3_resource = boto3.resource('s3')
            s3_resource.Object(s3_bucket, s3_path_in_bucket + '/myfile.csv').put(Body=csv_buffer.getvalue())


        :param boolean sensitive_info: whether or not to provide sensitive infos such as passwords,
                                       conditioned on the user being allowed to read details of the
                                       connection on which this dataset is defined
        :returns: a dict with the location info, with as notable fields:

                     * **locationInfoType**: type of location. Possible values are 'FS', 'HDFS', 'UPLOADED', 'SQL'
                     * **info** : a dict whose structure depends on the type of connection

                         * **connectionName**: connection name, if any
                         * **connectionParams** : parameters of the connection on which the dataset is defined, as a dic, if any. The actual fields depend on the connection type. For S3 dataset, this will for example contain the bucket and credentials.
                         * **path** : the URI of the dataset, if any

        :rtype: dict
        """
        return intercom.jek_or_backend_json_call("datasets/get-location-info/", data={
                "projectKey": self.project_key,
                "datasetName" : self.short_name,
                "sensitiveInfo" : sensitive_info
            }, err_msg="Failed to get the dataset location info")

    def get_files_info(self, partitions=[]):
        """
        Get information on the files of the dataset, with details per partition.

        :param partitions: list of partition identifiers, defaults to all partitions
        :type partitions: list[str], optional
        :returns: global files information and per partitions

                    * **globalPaths**: list of files of the dataset.

                        * **path**: file path
                        * **lastModified**: timestamp of last file update, in milliseconds
                        * **size**: size of the file, in bytes

                    * **pathsByPartition**: files grouped per partition, as a dict of partition identifier to list of files (same structure as **globalPaths**)

        :rtype: dict
        """
        return intercom.jek_or_backend_json_call("datasets/get-files-info/", data={
            "projectKey": self.project_key,
            "datasetName": self.short_name,
            "partitions": json.dumps(partitions)
        }, err_msg="Failed to get the dataset files info")

    def _repr_html_(self,):
        s = "Dataset[   <b>%s</b>   ]</br>" % self.name
        s += self.read_schema()._repr_html_()
        return s

    def set_write_partition(self, spec):
        """
        Set which partition of the dataset gets written to when
        you create a DatasetWriter.

        .. caution::
           Setting the write partition is not allowed in Python recipes,
           where write is controlled by the Flow.

        :param string spec: partition identifier

        """
        if flow.FLOW is not None and self.ignore_flow == False:
            raise Exception("You cannot explicitly set partitions when "
                            "running within Dataiku Flow")
        self.writePartition = spec

    def add_read_partitions(self, spec):
        """
        Add a partition or range of partitions to read.

        .. caution::
           You cannot manually add read partitions when running inside a Python recipe.
           They are automatically computed according to the partition dependencies defined
           on the recipe's Input/Output tab.

        :param string spec: partition spec, or partition identifier
        """
        if flow.FLOW is not None and self.ignore_flow == False:
            raise Exception("You cannot explicitly set partitions when "
                            "running within Dataiku Flow")
        if self.read_partitions is None:
            self.read_partitions = []
        self.read_partitions.append(spec)

    def read_schema(self, raise_if_empty=True):
        """
        Get the schema of this dataset, as an array of column definition.

        :param raise_if_empty: raise an exception if there is no column, defaults to True
        :type raise_if_empty: bool, optional

        :returns: list of column definitions
        :rtype: :class:`dataiku.core.dataset.Schema`
        """
        if self.cols is None:

            if os.getenv("FLOW_FORCED_SCHEMAS") is not None:
                ffs = json.loads(os.getenv("FLOW_FORCED_SCHEMAS"))
                if self.full_name in ffs:
                    logging.info("Forcing schema: %s"  % ffs[self.full_name])
                    return ffs[self.full_name]["columns"]

            self.cols = intercom.jek_or_backend_json_call("datasets/get-schema/", data={
                "fullDatasetName": self.full_name
            }, err_msg='Unable to fetch schema for %s'%(self.name)).get("columns")

        if raise_if_empty and len(self.cols) == 0:
            raise Exception(
                "No column in schema of %s."
                " Have you set up the schema for this dataset?" % self.name)
        return Schema(self.cols,)

    def list_partitions(self, raise_if_empty=True):
        """
        List the partitions of this dataset, as an array of partition identifiers.

        Usage example

        .. code-block:: python

            # build a list of partitions for use in a build/train step in a scenario
            dataset = dataiku.Dataset("some_input_dataset")
            partitions = dataset.list_partitions()
            variable_value = ','.join(partitions)

            # set as a variable, to use in steps after this one
            Scenario().set_scenario_variables(som_variable_name=variable_value)

        :param raise_if_empty: raise an exception if there is no partition, defaults to True
        :type raise_if_empty: bool, optional

        :returns: list of partitions identifiers
        :rtype: list[string]
        """
        if self.partitions is None:
            self.partitions = intercom.jek_or_backend_json_call("datasets/list-partitions/", data={
                "fullDatasetName": self.full_name
            }, err_msg='Unable to list partitions for %s'%(self.name))

        if raise_if_empty and len(self.partitions) == 0:
            raise Exception("No partition in %s." % self.name)
        return self.partitions

    def set_preparation_steps(self, steps, requested_output_schema, context_project_key=None):
        """
        Set preparation steps.

        .. caution:: for internal use

        :param list steps: list of steps

        :param dict requested_output_schema: output schema with a key **columns** containing a list of columns definition (name, type, ...)

        :param context_project_key: context project key, defaults to None
        :type context_project_key: string, optional
        """
        self.preparation_steps = steps
        self.preparation_requested_output_schema = requested_output_schema
        self.preparation_context_project_key = context_project_key

    def get_fast_path_dataframe(self,
                                auto_fallback=False,
                                columns=None,
                                pandas_read_kwargs=None,
                                print_deep_memory_usage=True):

        """
        Reads the dataset as a Pandas dataframe, using fast-path access (without going through DSS), if possible.

        Pandas dataframes are fully in-memory, so you need to make sure that your dataset will fit in RAM before using this.

        The fast path method provides better performance than the usual :meth:`get_dataframe` method, but is only compatible with
        some dataset types and formats.

        Fast path requires the "permission details readable" to be granted on the connection.

        Dataframes obtained using this method may differ from those using :meth:`get_dataframe`, notably around schemas and data.
        :meth:`get_dataframe` provides a unified API with the same schema and data for all connections. On the other hand, this method
        uses dataset-specific access patterns that may yield different results.

        At the moment, this fast path is available for:

        * S3 datasets using Parquet. This requires the additional `s3fs` package, as well as `fastparquet` or `pyarrow`
        * Snowflake datasets. This requires the additional `snowflake-connector-python[pandas]` package

        :param list columns: List of columns to read, or None for all columns
        :param boolean auto_fallback: If fast path is impossible and auto_fallback is True, then a regular :meth:`get_dataframe` call will be used.
                                If auto_fallback is False, this method will fail
        :param bool print_deep_memory_usage: After reading the dataframe, Dataiku prints the memory usage of the dataframe. When this is enabled,
                                             this will provide the accurate memory usage, including for string columns. This can have a small
                                             performance impact. Defaults to True
        :param dict pandas_read_kwargs: For the case where the read is mediated by a call to pd.read_parquet, arguments to pass to the read_parquet function
        """

        from dataiku.core import dataset_fast_path

        logging.info("Reading dataset %s as dataframe using fast-path" % (self.full_name))

        was_fast_path = True

        try:
            df = dataset_fast_path.try_read_dataframe(self, columns, pandas_read_kwargs)
        except dataset_fast_path.DataikuReadFastPathNotCompatibleException as e:
            if auto_fallback:
                was_fast_path = False
                df = self.get_dataframe(columns)
            else:
                raise e

        logging.debug("Dataframe read of dataset %s complete, getting memory usage" % (self.full_name))
        memory_usage_bytes = df.memory_usage(index=True, deep=print_deep_memory_usage).sum()
        memory_usage_str = "%.2fMB%s" % (memory_usage_bytes/(1024*1024), "" if print_deep_memory_usage else "+")
        shape = df.shape
        logging.info("Done reading dataset %s as dataframe %s rows=%s cols=%s mem=%s" % (self.full_name, was_fast_path and "using fast path" or "(fallbacked from fast path)", shape[0], shape[1], memory_usage_str))

        return df

    def get_dataframe(self,

                      # Filtering
                      columns=None,

                      # Sampling
                      sampling='head',
                      sampling_column=None,
                      limit=None,
                      ratio=None,
                      ascending=True,

                      # Types management
                      infer_with_pandas=True,
                      parse_dates=True,
                      bool_as_str=False,
                      int_as_float=False,
                      use_nullable_integers=False,
                      categoricals=None,

                      # Other data management
                      float_precision=None,
                      na_values=None,
                      keep_default_na=True,

                      # Performance
                      print_deep_memory_usage=True,
                      skip_additional_data_checks=False,

                      # Misc
                      date_parser=None,
                      override_dtypes=None,
                      pandas_read_kwargs=None):

        """
        Read the dataset (or its selected partitions, if applicable) as a Pandas dataframe.

        Pandas dataframes are fully in-memory, so you need to make sure that your dataset will fit in RAM before using this.

        .. code-block:: python

            # read some dataset and print its shape
            dataset = dataiku.Dataset("the_dataset_name")
            df = dataset.get_dataframe()
            print("Number of rows: %s" df.shape[0])
            print("Number of columns: %s" df.shape[1])

        :param list columns: when not None, returns only columns from the given list. defaults to None

        :param integer limit: limits the number of rows returned, defaults to None
        :param sampling: sampling method, see :meth:`dataiku.core.dataset.create_sampling_argument`. Defaults to 'head'.
        :param string sampling_column: column used for "random-column" and "sort-column" sampling, defaults to None
        :param float ratio: define the max row count as a ratio (between 0 and 1) of the dataset's total row count
        :param ascending boolean: sort in ascending order the selected column of the "sort-column" sampling, defaults to True

        :param bool infer_with_pandas: uses the types detected by pandas rather than the dataset schema as detected in DSS, defaults to True
        :param bool parse_dates: Only used when infer_with_pandas is False. Parses date column in DSS schema. Defaults to True
        :param bool bool_as_str: Only used when infer_with_pandas is False. Leaves boolean values as string. Defaults to False
        :param bool int_as_float: Only used when infer_with_pandas is False. Leaves int values as floats. Defaults to False
        :param bool use_nullable_integers: Only used when infer_with_pandas is False. Use pandas nullable integer types, which allows missing
                                        values in integer columns
        :param categoricals: Only used when infer_with_pandas is False.
                             What columns to read as categoricals. This is particularly efficient for columns with low cardinality.
                             Can be either "all_strings" to read all string columns as categorical, or a list of column names to read as categoricals

        :param string float_precision: set Pandas converter, can be None, 'high', 'legacy' or 'round_trip', defaults to None.
                                       see `Pandas.read_table documentation <https://pandas.pydata.org/>`_ for more information
        :param string/list/dict na_values: additional strings to recognize as NA/NaN, defaults to None.
                                           see `Pandas.read_table documentation <https://pandas.pydata.org/>`_ for more information
        :param bool keep_default_na: whether or not to include the default NaN values when parsing the data, defaults to True.
                                     see `Pandas.read_table documentation <https://pandas.pydata.org/>`_ for more information

        :param function date_parser: function to use for converting a sequence of string columns to an array of datetime instances, defaults to None.
                                     see `Pandas.read_table documentation <https://pandas.pydata.org/>`_ for more information

        :param bool skip_additional_data_checks: Skip some data type checks. Enabling this can lead to strongly increased performance (up to x3).
                                                 It is usually safe to enable this. Default to False
        :param bool print_deep_memory_usage: After reading the dataframe, Dataiku prints the memory usage of the dataframe. When this is enabled,
                                             this will provide the accurate memory usage, including for string columns. This can have a small
                                             performnace impact. Defaults to True


        :param dict override_dtypes: If not None, overrides dtypes computed from schema. Defaults to None
        :param dict pandas_read_kwargs: If not None, additional kwargs passed to pd.read_table. Defaults to None

        :returns: a Pandas dataframe object
        :rtype: :class:`pandas.core.frame.DataFrame`
        """

        logging.info("Reading dataset %s as dataframe" % (self.full_name))

        (names, dtypes, parse_date_columns) = self._get_dataframe_schema(
            columns=columns,
            parse_dates=parse_dates,
            infer_with_pandas=infer_with_pandas,
            bool_as_str=bool_as_str,
            int_as_float=int_as_float,
            use_nullable_integers=use_nullable_integers,
            categoricals=categoricals) # see df_from_split_desc

        read_session_id = str(uuid.uuid4())
        with self._stream(infer_with_pandas=infer_with_pandas,
                          sampling=sampling,
                          sampling_column=sampling_column,
                          limit=limit,
                          ratio=ratio,
                          columns=columns,
                          read_session_id=read_session_id,
                          skip_additional_data_checks=skip_additional_data_checks,
                          ascending=ascending) as dku_output:

            if pandas_read_kwargs is None:
                pandas_read_kwargs = {}

            if override_dtypes is not None:
                dtypes = override_dtypes

            with warnings.catch_warnings() as w:
                warnings.filterwarnings("ignore", category=FutureWarning) # Remove date_parser warning

                results = pd.read_table(dku_output,
                                    names=names,
                                    dtype=dtypes,
                                    header=None,
                                    sep=CSV_SEP,
                                    doublequote=CSV_DOUBLE_QUOTE,
                                    quotechar=CSV_QUOTE_CHAR,
                                    parse_dates=parse_date_columns,
                                    float_precision=float_precision,
                                    na_values=na_values,
                                    keep_default_na=keep_default_na,
                                    **pandas_read_kwargs)
                results = pandas_date_parser_compat(results, parse_date_columns, date_parser)

        # stream seems to have run fine. 'Seems'. Verify that.
        # note to self: this call has to be made after the dataframe creation, because it is streamed
        self._verify_read(read_session_id)

        logging.debug("Dataframe read of dataset %s complete, getting memory usage" % (self.full_name))

        memory_usage_bytes = results.memory_usage(index=True, deep=print_deep_memory_usage).sum()
        memory_usage_str = "%.2fMB%s" % (memory_usage_bytes/(1024*1024), "" if print_deep_memory_usage else "+")
        shape = results.shape

        logging.info("Done reading dataset %s as dataframe rows=%s cols=%s mem=%s" % (self.full_name, shape[0], shape[1], memory_usage_str))

        return results

    def to_html(self,
                columns=None,
                sampling='head',
                sampling_column=None,
                limit=None,
                ratio=None,
                apply_conditional_formatting=True,
                header=True,
                classes="",
                border=0,
                null_string="",
                indent_string=None,
                filter_expression=None):
        """
        Render the dataset as an html table.

        HTML tables are fully in-memory, so you need to make
        sure that your dataset will fit in RAM before using
        this, or pass a value to the limit parameter.

        .. code-block:: python

            # read some dataset and displays the first 50 rows
            dataset = dataiku.Dataset("the_dataset_name")
            df = dataset.to_html(limit=50)

        :param columns: when not None, returns only columns from the given list. Defaults to None
        :type columns: list[str]
        :param sampling: sampling method, see :meth:`dataiku.core.dataset.create_sampling_argument`. Defaults to 'head'.
        :param string sampling_column: column used for "random-column" and "sort-column" sampling, defaults to None
        :param integer limit: limits the number of rows returned, defaults to None
        :param float ratio: define the max row count as a ratio (between 0 and 1) of the dataset's total row count
        :param bool apply_conditional_formatting: true to apply conditional formatting as it has been defined in DSS Explore view
        :param bool header: Whether to print column labels, default True.
        :param classes: Name of the CSS class attached to TABLE tag in the generated HTML (or multiple classes as a list).
        :type classes: str or list[str]
        :param int border: A border attribute of the specified size is included in the opening <table> tag. Default to 0
        :param str null_string: string to represent null values. Defaults to an empty string.
        :param str indent_string: characters to use to indent the formatted HTML. If None or empty string, no indentation and no carriage return line feed. Defaults to None
        :param str filter_expression: expression used to filter data using formula language, defaults to None. Not supported on datasets with preparation steps.
        :returns: an HTML representation of the dataset
        :rtype: str
        """
        if self.preparation_steps is not None:
            raise Exception("to_html cannot be used on datasets with preparation steps")

        read_session_id = str(uuid.uuid4())
        with self._stream(sampling=sampling,
                          sampling_column=sampling_column,
                          limit=limit,
                          ratio=ratio,
                          filter_expression=filter_expression,
                          columns=columns,
                          read_session_id=read_session_id,
                          ascending=True,
                          format="html",
                          format_params={
                              "applyColoring": apply_conditional_formatting,
                              "cssClasses": '' if classes is None else classes if isinstance(classes, base.dku_basestring_type) else  ' '.join(classes),
                              "indentString": indent_string,
                              "nullString": null_string,
                              "border": border,
                              "header": header
                          }) as dku_output:

            if dku_output.supports_chunked_reads:
                result = ""
                for chunk in dku_output.read_chunked():
                    result = result + chunk.decode("utf-8")
            else:
                result = dku_output.data.decode("utf-8")

        # stream seems to have run fine. 'Seems'. Verify that.
        # note to self: this call has to be made after the HTML creation, because it is streamed
        self._verify_read(read_session_id)

        return result

    def _stream(self,
                infer_with_pandas=True,
                sampling="head",
                sampling_column=None,
                limit=None,
                ratio=None,
                columns=None,
                skip_additional_data_checks=False,
                read_session_id=None,
                ascending=True,
                format=None,
                format_params=None,
                filter_expression=None):
        if not self.readable:
            raise Exception("You cannot read dataset %s, "
                            "it is not declared as an input" % self.name)
        if flow.FLOW is not None:
            add_env = {"DKU_FLOW": "1"}
        else:
            add_env = {}

        sampling_params = create_sampling_argument(
            sampling=sampling,
            sampling_column=sampling_column,
            limit=limit,
            ratio=ratio,
            ascending=ascending)

        if self.preparation_steps is not None:
            if filter_expression is not None:
                raise Exception("Reading a dataset with preparation steps is not supported when filter_expression parameter is set")

            data = {
                "fullDatasetName": self.full_name,
                "script" :  json.dumps({ "steps" : self.preparation_steps }),
                "requestedOutputSchema" : json.dumps(self.preparation_requested_output_schema),
                "contextProjectKey": self.preparation_context_project_key,
                "sampling" : json.dumps(sampling_params),
                "readSessionId": read_session_id
            }
            if self.read_partitions is not None:
                data["partitions"] = json.dumps(self.read_partitions)

            return intercom.jek_or_backend_stream_call("datasets/stream-prepared-dataset/",
                        data=data, err_msg="Failed to read prepared data")

        else:
            data = {
                       "projectKey" : self.project_key,
                       "datasetName" : self.short_name,
                       "sampling" : json.dumps(sampling_params) if sampling_params is not None else None,
                       "columns" : ','.join(columns) if columns is not None else None,
                       "format" : ("tsv-excel-noheader%s" % ("-fast-unsafe" if skip_additional_data_checks else "")) if format is None else format,
                       "formatParams": json.dumps(format_params) if format_params is not None else None,
                       "partitions" : ",".join(self.read_partitions) if self.read_partitions is not None else None,
                       "filter" : filter_expression,
                       "readSessionId": read_session_id
                   }

            return intercom.jek_or_backend_stream_call("datasets/read-data/", data=data, err_msg="Failed to read dataset stream data")

    def _verify_read(self, read_session_id):
        intercom.jek_or_backend_void_call("datasets/verify-read/",
                                          data={"readSessionId": read_session_id}, err_msg="Reading dataset failed")

    @staticmethod
    def get_dataframe_schema_st(schema,
                                columns=None,
                                parse_dates=True,
                                infer_with_pandas=False,
                                bool_as_str=False,
                                int_as_float=False,
                                use_nullable_integers=False,
                                categoricals=None):
        """
        Extract information for Pandas from a schema.

        See :meth:`get_dataframe` for explanation of the other parameters

        :param schema: a schema definition as returned by :meth:`read_schema`
        :type schema: list[dict]

        :returns: a list of 3 items:

                    * a list columns names
                    * a dict of columns Numpy data types by names
                    * a list of the indexes of the dates columns or False

        :rtype: tuple[list,dict,list]
        """
        names = []
        dtypes = {}

        if categoricals is not None and isinstance(categoricals, list):
            categoricals_set = set(categoricals)
        else:
            categoricals_set = None

        for col in schema:
            n = col["name"]
            t = col["type"]
            if bool_as_str and t == "boolean":
                dtypes[n] = "str"  # see df_from_split_desc
            elif int_as_float and t in ["tinyint", "smallint", "int", "bigint"]:
                dtypes[n] = "float64"
            elif use_nullable_integers and t in ["tinyint", "smallint", "int", "bigint"]:
                if t == "tinyint":
                    dtypes[n] = pd.Int8Dtype()
                elif t == "smallint":
                    dtypes[n] = pd.Int16Dtype()
                elif t == "int":
                    dtypes[n] = pd.Int32Dtype()
                elif t == "bigint":
                    dtypes[n] = pd.Int64Dtype()
            elif t in schema_handling.DKU_RICHER_PANDAS_TYPES_MAP:
                dtypes[n] = schema_handling.DKU_RICHER_PANDAS_TYPES_MAP[t]
            elif categoricals == "all_strings" and t == "string":
                dtypes[n] = "category"
            elif categoricals_set is not None and n in categoricals_set:
                dtypes[n] = "category"
            else:
                dtypes[n] = np.object_

            names.append(n)
        if columns is not None:
            columns = list(unique(columns))
            names = columns
            dtypes = {
                column_name: dtypes[column_name]
                for column_name in dtypes
                if column_name in columns
            }


        # if parse_dates is set to True,
        # list up the index of the columns set up as dates by DSS
        # and forward them to pandas.
        if parse_dates is True:
            parse_dates = [
                col_id
                for (col_id, col_schema) in enumerate(schema)
                if (col_schema["type"] == "date" or col_schema["type"] == "dateonly" or col_schema["type"] == "datetimenotz") and (columns is None or col_schema["name"] in columns)
            ]
            # must not force dtype on date column as dtype now has precedence over parse_dates (since pandas 2.2)
            for col_id in parse_dates:
                dtypes.pop(schema[col_id]["name"])

            if len(parse_dates) == 0:
                parse_dates = False
        if infer_with_pandas:
            if bool_as_str:
                dtypes = dict((c["name"], "str") for c in schema if c["type"] == "boolean")
            else:
                dtypes = None
        return (names, dtypes, parse_dates)

    def _get_dataframe_schema(self,
                              columns=None,
                              parse_dates=True,
                              infer_with_pandas=False,
                              bool_as_str=False,
                              int_as_float=False,
                              use_nullable_integers=False,
                              categoricals=None):

        if self.preparation_steps is not None:
            return Dataset.get_dataframe_schema_st(self.preparation_requested_output_schema["columns"],
                                                   columns, parse_dates, infer_with_pandas, bool_as_str, int_as_float=int_as_float,
                                                   use_nullable_integers=use_nullable_integers, categoricals=categoricals)
        else:
            return Dataset.get_dataframe_schema_st(self.read_schema(),
                                                   columns, parse_dates, infer_with_pandas, bool_as_str, int_as_float=int_as_float,
                                                   use_nullable_integers=use_nullable_integers, categoricals=categoricals)


    def iter_dataframes_forced_types(self,
                        names, dtypes, parse_date_columns,
                        chunksize=10000,
                        sampling="head",
                        sampling_column=None,
                        limit=None,
                        ratio=None,
                        float_precision=None,
                        na_values=None,
                        keep_default_na=True,
                        date_parser=None,
                        ascending=True,
                        pandas_read_kwargs=None):
        """
        Read the dataset to Pandas dataframes by chunks of fixed size with given data types.

        .. code-block:: python

            import dataiku

            dataset = dataiku.Dataset("my_dataset")
            [names, dtypes, parse_date_columns] = dataiku.Dataset.get_dataframe_schema_st(dataset.read_schema())
            chunk = 0
            chunksize = 1000
            headsize = 5
            for df in dataset.iter_dataframes_forced_types(names, dtypes, parse_date_columns, chunksize = chunksize):
                print("> chunk #", chunk, "- first", headsize, "rows of", df.shape[0])
                chunk += 1
                print(df.head(headsize))


        :param names: list of column names
        :type names: list[string]
        :param dtypes: dict of data types by columns name
        :type dtypes: dict
        :param parse_date_columns: a list of the indexes of the dates columns or False
        :type parse_date_columns: list
        :param chunksize: chunk size, defaults to 10000
        :type chunksize: int, optional
        :param integer limit: limits the number of rows returned, defaults to None
        :param sampling: sampling method, see :meth:`dataiku.core.dataset.create_sampling_argument`. Defaults to 'head'.
        :type sampling: str, optional
        :param sampling_column: select the column used for "random-column" and "sort-column" sampling, defaults to None
        :type sampling_column: string, optional
        :param ratio: define the max row count as a ratio (between 0 and 1) of the dataset's total row count
        :type ratio: float, optional

        :param float_precision: set Pandas converter, can be None, 'high', 'legacy' or 'round_trip', defaults to None.
                                see `Pandas.read_table documentation <https://pandas.pydata.org/>`_ for more information
        :type float_precision: string, optional
        :param na_values: additional strings to recognize as NA/NaN, defaults to None.
                          see `Pandas.read_table documentation <https://pandas.pydata.org/>`_ for more information
        :type na_values: string/list/dict, optional
        :param keep_default_na: whether or not to include the default NaN values when parsing the data, defaults to True.
                                see `Pandas.read_table documentation <https://pandas.pydata.org/>`_ for more information
        :type keep_default_na: bool, optional
        :param date_parser: function to use for converting a sequence of string columns to an array of datetime instances, defaults to None.
                                see `Pandas.read_table documentation <https://pandas.pydata.org/>`_ for more information
        :type date_parser: function, optional
        :param ascending: sort in ascending order the selected column of the "sort-column" sampling, defaults to True
        :type ascending: boolean, optional
        :param dict pandas_read_kwargs: If not None, additional kwargs passed to pd.read_table. Defaults to None
        :yield: :class:`pandas.core.frame.DataFrame`
        :rtype: generator
        """
        if pandas_read_kwargs is None:
            pandas_read_kwargs = {}

        read_session_id = str(uuid.uuid4())
        with self._stream(
                          sampling=sampling,
                          sampling_column=sampling_column,
                          limit=limit,
                          ratio=ratio,
                          columns=names,
                          read_session_id=read_session_id,
                          ascending=ascending) as dku_output:
            df_it = pd.read_table(
                dku_output,
                dtype=dtypes,
                names=names,
                low_memory=True,
                header=None,
                sep=CSV_SEP,
                doublequote=CSV_DOUBLE_QUOTE,
                chunksize=chunksize,
                iterator=True,
                parse_dates=parse_date_columns,
                float_precision=float_precision,
                na_values=na_values,
                keep_default_na=keep_default_na,
                **pandas_read_kwargs)

            logging.info("Starting dataframes iterator")
            for df in df_it:
                df = pandas_date_parser_compat(df, parse_date_columns, date_parser)
                yield df

        # stream seems to have run fine. 'Seems'. Verify that.
        # note to self: this call has to be made after the dataframe creation, because it is streamed
        self._verify_read(read_session_id)

    def iter_dataframes(self,
                        chunksize=10000,
                        infer_with_pandas=True,
                        sampling="head",
                        sampling_column=None,
                        parse_dates=True,
                        limit=None,
                        ratio=None,
                        columns=None,
                        bool_as_str=False,
                        int_as_float=False,
                        use_nullable_integers=False,
                        categoricals=None,
                        float_precision=None,
                        na_values=None,
                        keep_default_na=True,
                        ascending=True,
                        pandas_read_kwargs=None):
        """
        Read the dataset to Pandas dataframes by chunks of fixed size.

        .. tip:: Useful is the dataset doesn't fit in RAM

        .. code-block:: python

            import dataiku

            dataset = dataiku.Dataset("my_dataset")
            for df in dataset.iter_dataframes(chunksize = 5000):
                print("> chunk of", df.shape[0], "rows")
                print(df.head(headsize))


        :param chunksize: chunk size, defaults to 10000
        :type chunksize: int, optional
        :param infer_with_pandas: use the types detected by pandas rather than the dataset schema as detected in DSS, defaults to True
        :type infer_with_pandas: bool, optional
        :param integer limit: limits the number of rows returned, defaults to None
        :param sampling: sampling method, see :meth:`dataiku.core.dataset.create_sampling_argument`. Defaults to 'head'.
        :type sampling: str, optional
        :param sampling_column: select the column used for "random-column" and "sort-column" sampling, defaults to None
        :type sampling_column: string, optional
        :param parse_dates: date column in DSS's dataset schema are parsed, defaults to True
        :type parse_dates: bool, optional
        :param limit: set the sampling max rows count, defaults to None
        :type limit: int, optional
        :param ratio: define the max row count as a ratio (between 0 and 1) of the dataset's total row count
        :type ratio: float, optional
        :param columns: specify the desired columns, defaults to None (all columns)
        :type columns: list[str], optional
        :param bool_as_str: Only used when infer_with_pandas is False. Leaves boolean values as strings, defaults to False
        :type bool_as_str: bool, optional
        :param int_as_float: Only used when infer_with_pandas is False. Leaves int values as floats. Defaults to False
        :type int_as_float: bool, optional
        :param use_nullable_integers: Only used when infer_with_pandas is False.
                                      Use pandas nullable integer types, which allows missing values in integer columns.
                                      Defaults to False
        :type use_nullable_integers: bool, optional
        :param categoricals: Only used when infer_with_pandas is False.
                             What columns to read as categoricals. This is particularly efficient for columns with low cardinality.
                             Can be either "all_strings" to read all string columns as categorical, or a list of column names to read as categoricals
        :type categoricals: string/list, optional

        :param float_precision: set Pandas converter, can be None, 'high', 'legacy' or 'round_trip', defaults to None.
                                see `Pandas.read_table documentation <https://pandas.pydata.org/>`_ for more information
        :type float_precision: string, optional
        :param na_values: additional strings to recognize as NA/NaN, defaults to None.
                          see `Pandas.read_table documentation <https://pandas.pydata.org/>`_ for more information
        :type na_values: string/list/dict, optional
        :param keep_default_na: whether or not to include the default NaN values when parsing the data, defaults to True.
                                see `Pandas.read_table documentation <https://pandas.pydata.org/>`_ for more information
        :type keep_default_na: bool, optional
        :param ascending: sort in ascending order the selected column of the "sort-column" sampling, defaults to True
        :type ascending: boolean, optional
        :param dict pandas_read_kwargs: If not None, additional kwargs passed to pd.read_table. Defaults to None
        :yield: :class:`pandas.core.frame.DataFrame`
        :rtype: generator
        """
        if pandas_read_kwargs is None:
            pandas_read_kwargs = {}

        if not self.readable:
            raise Exception("You cannot read dataset %s, "
                            "it is not declared as an input" % self.name)
        (names, dtypes, parse_date_columns) = self._get_dataframe_schema(
            columns=columns,
            parse_dates=parse_dates,
            infer_with_pandas=infer_with_pandas,
            bool_as_str=bool_as_str,
            int_as_float=int_as_float,
            use_nullable_integers=use_nullable_integers,
            categoricals=categoricals)
        read_session_id = str(uuid.uuid4())
        with self._stream(infer_with_pandas=infer_with_pandas,
                          sampling=sampling,
                          sampling_column=sampling_column,
                          limit=limit,
                          ratio=ratio,
                          columns=columns,
                          read_session_id=read_session_id,
                          ascending=ascending) as dku_output:
            df_it = pd.read_table(
                dku_output,
                dtype=dtypes,
                names=names,
                low_memory=True,
                header=None,
                sep=CSV_SEP,
                doublequote=CSV_DOUBLE_QUOTE,
                chunksize=chunksize,
                iterator=True,
                parse_dates=parse_date_columns,
                float_precision=float_precision,
                na_values=na_values,
                keep_default_na=keep_default_na,
                **pandas_read_kwargs)
            logging.info("Starting dataframes iterator")
            for df in df_it:
                yield df

        # stream seems to have run fine. 'Seems'. Verify that.
        # note to self: this call has to be made after the dataframe creation, because it is streamed
        self._verify_read(read_session_id)

    def write_with_schema(self, df, drop_and_create=False, **kwargs):
        """
        Write a pandas dataframe to this dataset (or its target partition, if applicable).

        This variant replaces the schema of the output dataset with the schema
        of the dataframe.

        .. caution:: strings MUST be in the dataframe as UTF-8 encoded str objects.
                     Using unicode objects will fail.

        .. note:: the dataset must be *writable*, ie declared as an output,
                  except if you instantiated the dataset with `ignore_flow=True`

        .. code-block:: python

            import dataiku
            from dataiku import recipe

            # simply copy the first recipe input dataset
            # to the first recipe output dataset, with the schema

            ds_input = recipe.get_inputs()[0]
            df_input = ds_input.get_dataframe()
            ds_output = recipe.get_outputs()[0]
            ds_output.write_with_schema(df_input, True)

        :param df: a panda dataframe
        :type df: :class:`pandas.core.frame.DataFrame`
        :param drop_and_create: whether to drop and recreate the dataset, defaults to False
        :type drop_and_create: bool, optional
        :param dropAndCreate: **deprecated**, use ``drop_and_create``
        :type dropAndCreate: bool, optional
        """
        for k in kwargs:  # for backward compat
            if k == "dropAndCreate":
                drop_and_create = kwargs.get("dropAndCreate")
            else:
                raise Exception("Unknown argument '{}'".format(k))
        if not hasattr(df, "to_csv"):
            raise ValueError("Method write_with_schema expects a "
                             "dataframe as argument. You gave a %s" %
                             (df is None and "None" or df.__class__))
        self.write_dataframe(df, True, drop_and_create)

    def write_dataframe(self, df, infer_schema=False, drop_and_create=False, **kwargs):
        """
        Write a pandas dataframe to this dataset (or its target partition, if applicable).

        This variant only edits the schema if infer_schema is True, otherwise you need
        to only write dataframes that have a compatible schema.
        Also see :meth:`write_with_schema`.

        .. caution:: strings MUST be in the dataframe as UTF-8 encoded str objects.
                     Using unicode objects will fail.

        .. note:: the dataset must be *writable*, ie declared as an output,
                  except if you instantiated the dataset with `ignore_flow=True`

        :param df: a pandas dataframe
        :type df: :class:`pandas.core.frame.DataFrame`
        :param infer_schema: whether to infer the schema from the dataframe, defaults to False
        :type infer_schema: bool, optional
        :param drop_and_create: whether to drop and recreate the dataset, defaults to False
        :type drop_and_create: bool, optional
        :param dropAndCreate: **deprecated**, use ``drop_and_create``
        :type dropAndCreate: bool, optional
        """
        for k in kwargs:  # for backward compat
            if k == "dropAndCreate":
                drop_and_create = kwargs.get("dropAndCreate")
            else:
                raise Exception("Unknown argument '{}'".format(k))
        if not hasattr(df, "to_csv"):
            raise ValueError("Method write_dataframe expects a "
                             "dataframe as argument. You gave a %s" %
                             (df is None and "None" or df.__class__))
        if not self.writable:
            raise Exception("You cannot write dataset %s, "
                            "it is not declared as an output" % self.name)
        try:
            if infer_schema:
                self.write_schema_from_dataframe(df, drop_and_create)

            with self.get_writer() as writer:
                writer.write_dataframe(df)

        except AttributeError as e:
            raise TypeError("write_dataframe is a expecting a "
                            "DataFrame object. You provided a " +
                            df.__class__.__name__, e)

    def write_from_dataframe(self, df, infer_schema=False, write_direct=False, drop_and_create=False, **kwargs):
        """
        Write a pandas dataframe to this dataset (or its target partition, if applicable).

        .. caution:: Deprecated. Please use :meth:`write_dataframe` instead.
        """
        for k in kwargs:  # for backward compat
            if k == "dropAndCreate":
                drop_and_create = kwargs.get("dropAndCreate")
            else:
                raise Exception("Unknown argument '{}'".format(k))
        self.write_dataframe(df, infer_schema, drop_and_create)

    def iter_rows(self,
                  sampling='head',
                  sampling_column=None,
                  limit=None,
                  ratio=None,
                  log_every=-1,
                  timeout=DEFAULT_TIMEOUT,
                  columns=None,
                  ascending=True):
        """
        Get a generator of rows (as a dict-like object) in the
        data (or its selected partitions, if applicable).

        Values are cast according to their types.
        String are parsed into "unicode" values.

        :param integer limit: limits the number of rows returned, defaults to None
        :param sampling: sampling method, see :meth:`dataiku.core.dataset.create_sampling_argument`. Defaults to 'head'.
        :type sampling: str, optional
        :param sampling_column: select the column used for "random-column" and "sort-column" sampling, defaults to None
        :type sampling_column: string, optional
        :param limit: maximum number of rows to be emitted, defaults to None
        :type limit: int, optional
        :param ratio: define the max row count as a ratio (between 0 and 1) of the dataset's total row count
        :type ratio: float, optional
        :param log_every: print out the number of rows read on stdout, defaults to -1 (no log)
        :type log_every: int, optional
        :param timeout: set a timeout in seconds, defaults to 30
        :type timeout: int, optional
        :param columns: specify the desired columns, defaults to None (all columns)
        :type columns: list[str], optional
        :param ascending: sort in ascending order the selected column of the "sort-column" sampling, defaults to True
        :type ascending: boolean, optional

        :yield: :class:`dataiku.core.dataset.DatasetCursor`
        :rtype: generator
        """
        schema = self._read_filtered_schema(columns)
        col_names = [col["name"] for col in schema]
        col_idx = {
            col_name: col_id
            for (col_id, col_name) in enumerate(col_names)
        }
        for row_tuple in self.iter_tuples(sampling=sampling,
                                          sampling_column=sampling_column,
                                          limit=limit,
                                          ratio=ratio,
                                          log_every=log_every,
                                          timeout=timeout,
                                          columns=columns,
                                          ascending=ascending):
            yield DatasetCursor(row_tuple, col_names, col_idx)

    def _read_filtered_schema(self, columns=None):
        schema = self.read_schema()

        # if needed, filter the schema columns to the requested columns
        if columns is not None:
            # index columns in a map by name
            cols_by_name = {}
            for col in schema:
                cols_by_name[col['name']] = col
            # make a truncated schema with just the requested columns ( in the right order)
            cols = []
            for column in columns:
                if column not in cols_by_name:
                    raise Exception("Column '%s' not in dataset." % column)
                cols.append(cols_by_name[column])
            schema = Schema(cols)

        return schema

    def _iter_tuples_no_timeout(self,
                                sampling=None,
                                log_every=-1,
                                columns=None):
        """
        Same as iter_tuples but without the timeout.
        """
        if not self.readable:
            raise Exception("You cannot read dataset %s, it is "
                            "not declared as an input" % self.name)
        schema = self._read_filtered_schema(columns)

        casters = [
            schema_handling.CASTERS.get(col["type"], lambda s:s)
            for col in schema
        ]

        read_session_id = str(uuid.uuid4())

        with intercom.jek_or_backend_stream_call("datasets/read-data", data = {
                    "projectKey" : self.project_key,
                    "datasetName" : self.short_name,
                    "sampling" : json.dumps(sampling) if sampling is not None else None,
                    "columns" : ','.join(columns) if columns is not None else None,
                    "format" : "tsv-excel-noheader",
                    "partitions" : ",".join(self.read_partitions) if self.read_partitions is not None else None,
                    "readSessionId": read_session_id
                }) as stream:

            count = 0

            csv_reader = dkuio.new_utf8_csv_reader(stream,
                                           delimiter=CSV_SEP,
                                           quotechar=CSV_QUOTE_CHAR,
                                           doublequote=CSV_DOUBLE_QUOTE)

            for row_tuple in csv_reader:
                yield [none_if_throws(caster)(val)
                       for (caster, val) in base.dku_zip_longest(casters, row_tuple)]
                count += 1
                if log_every > 0 and count % log_every == 0:
                    sys.stderr.write ("Dataset<%s> - read %i lines" % (self.name, count))
                    sys.stderr.write("\n")

        # stream seems to have run fine. 'Seems'. Verify that.
        self._verify_read(read_session_id)

    def raw_formatted_data(self, sampling=None, columns=None, format="tsv-excel-noheader", format_params=None,
                           read_session_id=None, filter_expression=None):
        """
        Get a stream of raw bytes from a dataset as a file-like object, formatted in a supported
        DSS output format.

        .. caution:: You MUST close the file handle. Failure to do so will result in resource leaks.

        After closing, you can also call :meth:`verify_read` to check for any errors
        that occurred while reading the dataset data.

        .. code-block:: python

            import uuid
            import dataiku
            from dataiku.core.dataset import create_sampling_argument

            dataset = dataiku.Dataset("customers_partitioned")
            read_session_id = str(uuid.uuid4())
            sampling = create_sampling_argument(sampling='head', limit=5)
            resp = dataset.raw_formatted_data(sampling=sampling, format="json", read_session_id=read_session_id)
            print(resp.data)
            resp.close()
            dataset.verify_read(read_session_id) #throw an exception if the read hasn't been fully completed
            print("read completed successfully")

        :param sampling: a dict of sampling specs, see :meth:`dataiku.core.dataset.create_sampling_argument`, defaults to None
        :type sampling: dict, optional
        :param columns: list of desired columns, defaults to None (all columns)
        :type columns: list[str], optional
        :param format: output format, defaults to "tsv-excel-noheader". Supported formats are : "json",
                       "tsv-excel-header" (tab-separated with header) and "tsv-excel-noheader" (tab-separated without header)
        :type format: str, optional
        :param format_params: dict of output format parameters, defaults to None
        :type format_params: dict, optional
        :param read_session_id: identifier of the read session, used to check at the end if the read was successful, defaults to None
        :type read_session_id: str, optional
        :param filter_expression: expression used to filter data using formula language, defaults to None
        :type filter_expression: str, optional
        :returns: an HTTP response
        :rtype: :class:`urllib3.response.HTTPResponse`
        """
        if not self.readable:
            raise Exception("You cannot read dataset %s, it is "
                            "not declared as an input" % self.name)

        # Build the query
        req_settings = {
            "format" : format
        }
        if format_params is not None:
            req_settings["formatParams"] = format_params
        if sampling is not None:
            req_settings["sampling"] = sampling
        if columns is not None:
            req_settings["columns"] = columns
        if self.read_partitions is not None:
            req_settings["partitions"] = ",".join(self.read_partitions)
        if read_session_id is not None:
            req_settings["readSessionId"] = read_session_id
        if filter_expression is not None:
            req_settings["filterExpression"] = filter_expression

        # Send
        return intercom.backend_stream_call("datasets/read-data2", data={
            "projectKey" : self.project_key,
            "datasetName" : self.short_name,
            "settings" : json.dumps(req_settings)
        }, err_msg="Failed to read dataset")

    def verify_read(self, read_session_id):
        """
        Verify that no error occurred when using :meth:`raw_formatted_data` to read a dataset.

        Use the same `read_session_id` that you passed to the call to :meth:`raw_formatted_data`.

        :param read_session_id: identifier of the read session
        :type read_session_id: str
        :raises Exception: if an error occured while the read
        """
        intercom.backend_void_call("datasets/verify-read/",
                                   data={"readSessionId": read_session_id}, err_msg="Reading dataset failed")

    def iter_tuples(self,
                    sampling='head',
                    sampling_column=None,
                    limit=None,
                    ratio=None,
                    log_every=-1,
                    timeout=DEFAULT_TIMEOUT,
                    columns=None,
                    ascending=True):
        """
        Get the rows of the dataset as tuples.

        The order and type of the values are the same are matching
        the dataset's parameter

        Values are cast according to their types.
        String are parsed into "unicode" values.

        :param integer limit: limits the number of rows returned, defaults to None
        :param sampling: sampling method, see :meth:`dataiku.core.dataset.create_sampling_argument`. Defaults to 'head'.
        :type sampling: str, optional
        :param sampling_column: select the column used for "random-column" and "sort-column" sampling, defaults to None
        :type sampling_column: string, optional
        :param limit: maximum number of rows to be emitted, defaults to None (all)
        :type limit: int, optional
        :param ratio: define the max row count as a ratio (between 0 and 1) of the dataset's total row count
        :type ratio: float, optional
        :param log_every: print out the number of rows read on stdout, defaults to -1 (no log)
        :type log_every: int, optional
        :param timeout: time (in seconds) of inactivity after which we want to close the generator if nothing has been read.
                        Without it notebooks typically tend to leak "DKU" processes, defaults to 30
        :type timeout: int, optional
        :param columns: list of desired columns, defaults to None (all columns)
        :type columns: list[str], optional
        :param ascending: sort in ascending order the selected column of the "sort-column" sampling, defaults to True
        :type ascending: boolean, optional

        :yield: a tuples of columns values
        :rtype: generator
        """
        sampling_params = create_sampling_argument(
            sampling=sampling,
            sampling_column=sampling_column,
            limit=limit,
            ratio=ratio,
            ascending=ascending)
        generator = self._iter_tuples_no_timeout(sampling=sampling_params,
                                                 log_every=log_every,
                                                 columns=columns)
        generator_with_timeout = IteratorWithTimeOut(iterator=generator,
                                     timeout=timeout).get_generator()
        for value in generator_with_timeout:
            yield value

    def get_writer(self):
        """
        Get a stream writer for this dataset (or its target partition, if applicable).

        .. caution:: The writer must be closed as soon as you don't need it.

        :returns: a stream writer
        :rtype: :class:`dataiku.core.dataset_write.DatasetWriter`
        """

        if os.getenv("FLOW_FAKE_WRITER") is not None:
            return dataset_write.FakeDatasetWriter(self,)
        else:
            return dataset_write.DatasetWriter(self,)


    def get_continuous_writer(self, source_id, split_id=0):
        """
        Get a stream writer for this dataset (or its target partition, if applicable).

        .. code-block:: python

            dataset = dataiku.Dataset("wikipedia_dataset")
            dataset.write_schema([{"name":"data", "type":"string"}, ...])
            with dataset.get_continuous_writer(...) as writer:
                for msg in message_iterator:
                    writer.write_row_dict({"data":msg.data, ...})
                    writer.checkpoint("this_recipe", "some state")

        :param source_id: identifier of the source of the stream
        :type source_id: str
        :param split_id: split id in the output (for concurrent usage), defaults to 0
        :type split_id: int, optional


        :returns: a stream writer
        :rtype: :class:`dataiku.core.continuous_write.DatasetContinuousWriter`
        """
        return continuous_write.DatasetContinuousWriter(self, source_id, split_id=split_id)

    def write_schema(self, columns, drop_and_create=False, **kwargs):
        """
        Write the dataset schema into the dataset JSON definition file.

        Sometimes, the schema of a dataset being written is
        known only by the code of the Python script itself.
        In that case, it can be useful for the Python script
        to actually modify the schema of the dataset.

        .. caution:: Obviously, this must be used with caution.

        :param list columns: see :meth:`read_schema`
        :param drop_and_create: whether to drop and recreate the dataset, defaults to False
        :type drop_and_create: bool, optional
        :param dropAndCreate: **deprecated**, use ``drop_and_create``
        :type dropAndCreate: bool, optional
        """
        origin = "python"
        for k in kwargs:
            if k == "dropAndCreate":  # for backward compat
                drop_and_create = kwargs.get("dropAndCreate")
            elif k == "origin":  # for hiding technical crap added for backward compat too
                origin = kwargs.get("origin", origin)
            else:
                raise Exception("Unknown argument '{}'".format(k))
        if not self.writable:
            raise Exception("You cannot write the schema for the dataset %s, "
                            "as it is not declared as an output" % self.name)
        for column in columns:
            if "type" not in column:
                raise Exception("Columns %s has no attribute type"
                                % str(column))
            if "name" not in column:
                raise Exception("Columns %s has no attribute name"
                                % str(column))
            if not isinstance(column['name'], base.dku_basestring_type):
                raise Exception("Columns %s name attribute is not a string"
                                % str(column))
            if not isinstance(column['type'], base.dku_basestring_type):
                raise Exception("Columns %s type attribute is not a string"
                                % str(column))

        intercom.jek_or_backend_void_call("datasets/set-schema/", data={
            "fullDatasetName": self.full_name,
            "schemaData": json.dumps({
                "userModified": False,
                "columns": columns
            }),
            "origin": origin,
            "dropAndCreate" : drop_and_create
        })
        # trash the current cached schema, it's probably not valid anymore
        self.cols = None

    def write_schema_from_dataframe(self, df, drop_and_create=False, **kwargs):
        """
        Set the schema of this dataset to the schema of a Pandas dataframe.

        .. code-block:: py

            import dataiku

            input_ds = dataiku.Dataset("input_dataset")
            my_input_dataframe = input_ds.get_dataframe()
            output_ds = dataiku.Dataset("output_dataset")

            # Set the schema of "output_ds" to match the columns of "my_input_dataframe"
            output_ds.write_schema_from_dataframe(my_input_dataframe)

        :param df: a Pandas dataframe
        :type df: :class:`pandas.core.frame.DataFrame`
        :param drop_and_create: whether drop and recreate the dataset, defaults to False
        :type drop_and_create: bool, optional
        :param dropAndCreate: **deprecated**, use ``drop_and_create``
        :type dropAndCreate: bool, optional
        """
        for k in kwargs:  # for backward compat
            if k == "dropAndCreate":
                drop_and_create = kwargs.get("dropAndCreate")
            else:
                raise Exception("Unknown argument '{}'".format(k))
        self.write_schema(schema_handling.get_schema_from_df(df), drop_and_create)


    def read_metadata(self):
        """
        Get the metadata attached to this dataset.

        The metadata contains label, description checklists, tags and custom metadata of the dataset

        :return: the metadata as a dict, with fields:

                    * **label** : label of the object (not defined for recipes)
                    * **description** : description of the object (not defined for recipes)
                    * **checklists** : checklists of the object, as a dict with a **checklists** field, which is a list of checklists, each a dict of fields:

                        * **id** : identifier of the checklist
                        * **title** : label of the checklist
                        * **createdBy** : user who created the checklist
                        * **createdOn** : timestamp of creation, in milliseconds
                        * **items** : list of the items in the checklist, each a dict of

                            * **done** : True if the item has been done
                            * **text** : label of the item
                            * **createdBy** : who created the item
                            * **createdOn** : when the item was created, as a timestamp in milliseconds
                            * **stateChangedBy** : who ticked the item as done (or not done)
                            * **stateChangedOn** : when the item was last changed to done (or not done), as a timestamp in milliseconds

                    * **tags** : list of tags, each a string
                    * **custom** : custom metadata, as a dict with a **kv** field, which is a dict with any contents the user wishes
                    * **customFields** : dict of custom field info (not defined for recipes)

        :rtype: dict
        """
        return intercom.jek_or_backend_json_call("datasets/get-metadata", data={
            "fullDatasetName" : self.full_name
        })

    def write_metadata(self, meta):
        """
        Write the metadata to the dataset.

        .. note:: you should set a metadata that you obtained via :meth:`read_metadata` then modified.

        :param dict meta: metadata specifications as dict, see :meth:`read_metadata`
        """

        #if not self.writable:
        #    raise Exception("You cannot write the metadata for the dataset %s, "
        #                    "as it is not declared as an output" % self.name)

        if "checklists" not in meta:
            raise Exception("'checklists' is missing")
        if "custom" not in meta:
            raise Exception("'custom' is missing")
        if "tags" not in meta:
            raise Exception("'tags' is missing")

        intercom.jek_or_backend_void_call("datasets/write-metadata", data={
            "fullDatasetName" : self.full_name,
            "metadata" : json.dumps(meta)
        })


    def get_config(self):
        """
        Get the dataset config.

        :returns: all dataset settings, with many relative to its type. main settings keys are:

                        * **type**: type of the dataset such as "PostgreSQL", "Filesystem", etc...
                        * **name**: name of the dataset
                        * **projectKey**: project hosting the dataset
                        * **schema**: dataset schema as dict with 'columns' definition
                        * **partitioning**: partitions settings as dict
                        * **managed**: True if the dataset is managed
                        * **readWriteOptions**: dict of read or write options
                        * **versionTag**: version info as dict with 'versionNumber', 'lastModifiedBy', and 'lastModifiedOn'
                        * **creationTag**: creation info as dict with 'versionNumber', 'lastModifiedBy', and 'lastModifiedOn'
                        * **tags**: list of tags
                        * **metrics**: dict with a list of `probes`, see :doc:`/api-reference/python/metrics`

        :rtype: dict
        """
        return intercom.backend_json_call("datasets/read-config/", data={
            "projectKey" : self.project_key,
            "datasetName" : self.short_name
        })


    # ################################### Metrics #############################

    def get_last_metric_values(self, partition=''):
        """
        Get the set of last values of the metrics on this dataset.

        :param string partition: (optional), the partition for which to fetch the values. On partitioned datasets,
                                 the partition value to use for accessing metrics on the whole dataset (ie. all
                                 partitions) is ALL

        :rtype: :class:`dataiku.core.metrics.ComputedMetrics`
        """
        return metrics.ComputedMetrics(intercom.backend_json_call("metrics/datasets/get-last-values", data = {
             "projectKey": self.project_key,
             "datasetName" : self.short_name,
             "partition" : partition
        }))

    def get_metric_history(self, metric_lookup, partition=''):
        """
        Get the set of all values a given metric took on this dataset.

        :param string metric_lookup: metric name or unique identifier
        :param string partition: (optional), the partition for which to fetch the values. On partitioned datasets,
                                 the partition value to use for accessing metrics on the whole dataset (ie. all
                                 partitions) is ALL
        :rtype: dict
        """
        return intercom.backend_json_call("metrics/datasets/get-metric-history", data = {
            "projectKey": self.project_key,
            "datasetName" : self.short_name,
            "partition" : partition,
            "metricLookup" : metric_lookup if isinstance(metric_lookup, str) or isinstance(metric_lookup, unicode) else json.dumps(metric_lookup)
        }, err_msg="Failed to get metric history")

    def save_external_metric_values(self, values_dict, partition=''):
        """
        Save metrics on this dataset.

        The metrics are saved with the type "external".

        :param dict values_dict: the values to save, as a dict. The keys of the dict are used as metric names
        :param string partition: (optional), the partition for which to save the values. On partitioned datasets,
                                 the partition value to use for accessing metrics on the whole dataset (ie. all
                                 partitions) is ALL

        :rtype: dict
        """
        return intercom.backend_json_call("metrics/datasets/save-external-values", data = {
            "projectKey": self.project_key,
            "datasetName" : self.short_name,
            "partitionId" : partition,
            "data" : json.dumps(values_dict)
        }, err_msg="Failed to save external metric values")

    def get_last_check_values(self, partition=''):
        """
        Get the set of last values of the checks on this dataset.

        :param string partition: (optional), the partition for which to fetch the values. On partitioned datasets,
                                 the partition value to use for accessing metrics on the whole dataset (ie. all
                                 partitions) is ALL

        :rtype: :class:`dataiku.core.metrics.ComputedChecks`
        """
        return metrics.ComputedChecks(intercom.backend_json_call("checks/datasets/get-last-values", data = {
             "projectKey": self.project_key,
             "datasetName" : self.short_name,
             "partition" : partition
        }))

    def save_external_check_values(self, values_dict, partition=''):
        """
        Save checks on this dataset.

        The checks are saved with the type "external"

        :param dict values_dict: the values to save, as a dict. The keys of the dict are used as check names
        :param string partition: (optional), the partition for which to save the values. On partitioned datasets,
                                 the partition value to use for accessing metrics on the whole dataset (ie. all
                                 partitions) is ALL

        :rtype: dict
        """
        return intercom.backend_json_call("checks/datasets/save-external-values", data = {
            "projectKey": self.project_key,
            "datasetName" : self.short_name,
            "partitionId" : partition,
            "data" : json.dumps(values_dict)
        }, err_msg="Failed to save external check values")

class DatasetCursor(object):
    """
    A dataset cursor iterating on the rows.

    .. caution:: you should not instantiate it manually, see :meth:`dataiku.Dataset.iter_rows`
    """

    __slots__ = ('_col_idx', '_col_names', '_val')

    def __init__(self, val, col_names, col_idx):
        self._col_idx = col_idx
        self._col_names = col_names
        self._val = val

    def __getitem__(self, col_name):
        try:
            col_id = self._col_idx.get(col_name)
            return self._val[col_id]
        except KeyError:
            raise KeyError("Column '%s' is not declared in the schema"
                           % col_name)
        except IndexError:
            raise KeyError("CSV file number of column does not match. Expected"
                           " %i, got %i" %
                           (len(self._col_names, len(self._val))))

    def __len__(self,):
        return len(self._col_idx)

    def __iter__(self,):
        return iter(self._col_names)

    def __contains__(self, k):
        return k in self._col_idx

    def column_id(self, name,):
        """
        Get a column index from its name.

        :param name: column name
        :type name: str
        :returns: the column index
        :rtype: int
        """
        return self._col_idx.get(name)

    def keys(self,):
        """
        Get the set of column names.

        :returns: list of columns name
        :rtype: list[str]
        """
        return self._col_names

    def items(self,):
        """
        Get the full row.

        :returns: a list of tuple (column, value)
        :rtype: list[tuple]
        """
        return zip(self._col_names, self._val)

    def values(self,):
        """
        Get values in the row.

        :returns: list of columns values
        :rtype: list
        """
        return self._val

    def __repr__(self,):
        return repr(dict(self.items()))

    def get(self, col_name, default_value=None):
        """
        Get a value by its column name.

        :param col_name: a column name
        :type col_name: str
        :param default_value: value to return if the column is not present, defaults to None
        :type default_value: str, optional
        :returns: the value of the column
        :rtype: depends on the column's type
        """
        if col_name in self._col_idx:
            col_id = self._col_idx.get(col_name)
            return self._val[col_id]
        else:
            return default_value


def _dataset_writer_atexit_handler():
    dataset_write.DatasetWriter.atexit_handler()


