import random
import shutil
import string
import tempfile
from contextlib import contextmanager
import time
import copy
import json
import os
import logging
import math
import numpy as np
import six

###########################
# IMPORTANT NOTE
# You must NEVER import dataiku.doctor here
# Because this file is imported by "import dataiku"
# and dataiku.doctor imports sklearn
#
# And the tableau plugin fails if sklearn is imported
# (known bug in Tableau SDK)
# So all imports of dataiku.doctor must be inline in functions
###########################

from dataiku.base import remoterun
from dataiku.base.utils import safe_unicode_str
from dataiku.base.folder_context import build_saved_model_folder_context
from dataiku.base.folder_context import get_partitions_fmi_folder_contexts
from dataiku.core import base, flow, metrics, default_project_key
from dataiku.core import doctor_constants
from dataiku.core.base import PartitionEscaper
from dataiku.core.dataset import Dataset
from dataiku.core.intercom import backend_json_call, backend_void_call, backend_stream_call

logger = logging.getLogger(__name__)

try:
    import pandas as pd
except:
    logger.warning("Pandas not available, certain saved models features are unavailable")


class Model(base.Computable):
    """
    Handle to interact with a saved model.

    .. note::

        This class is also available as ``dataiku.Model``

    :param string lookup: name or identifier of the saved model
    :param string project_key: project key of the saved model, if it is not in the current project.
        (defaults to **None**, i.e. current project)
    :param boolean ignore_flow: if True, create the handle regardless of whether the saved model is an input
        or output of the recipe (defaults to **False**)
    """
    def __init__(self, lookup, project_key=None, ignore_flow=False):
        self.lookup = lookup
        self.versions = None
        self.definition = None
        self.info = None
        self._predictors = {}
        self._dku_scoring_predictors = {}
        self.ignore_flow = ignore_flow
        self.read_partitions = None
        self.writePartition = None

        if flow.FLOW is not None and not ignore_flow:
            self._init_data_from_flow(obj_type="Saved model", project_key=project_key)
        else:
            if "." not in lookup:
                self.project_key = project_key or default_project_key()
                self.short_name = lookup
                self.name = self.project_key + "." + lookup
            else:
                self.project_key = lookup.split(".")[0]
                self.short_name = lookup.split(".")[1]
                self.name = lookup
                # except:
                #    raise Exception("Managed folder %s is specified with a relative name, "
                #                    "but no default project was found. Please use complete name" % id)

    def _repr_html_(self, ):
        s = "Model[   <b>%s</b>   ]" % (self.name)
        return s

    @staticmethod
    def list_models(project_key=None):
        """
        Retrieves the list of saved models of the given project.

        :param str project_key: key of the project from which to list models. (defaults to **None**, i.e. current project)

        :return: a list of the saved models of the project, as dict.
            Each dict contains at least the following fields:

            * **id**: identifier of the saved model
            * **name**: name of the saved model
            * **type**: type of saved model (CLUSTERING / PREDICTION)
            * **backendType**: backend type of the saved model (PY_MEMORY / KERAS / MLLIB / H2O / DEEP_HUB)
            * **versionsCount**: number of versions in the saved model

        :rtype: list[dict]
        """

        project_key = project_key or default_project_key()
        return backend_json_call("savedmodels/list", data={
            "projectKey": project_key
        })

    def get_info(self):
        """
        Gets the model information.

        :return: the model information.
            Fields are:

            * **id** : identifier of the saved model
            * **projectKey** : project key of the saved model
            * **name** : name of the saved model
            * **type**: type of saved model (CLUSTERING / PREDICTION)

        :rtype: dict
        """
        if self.info is None:
            self.info = backend_json_call("savedmodels/get-info", data={
                "projectKey": self.project_key,
                "lookup": self.short_name
            })["info"]

        return self.info

    def get_id(self):
        """
        Gets the identifier of the model.

        :rtype: str
        """
        return self.get_info()["id"]

    def get_name(self):
        """
        Gets the name of the model

        :rtype: str
        """
        return self.get_info()["name"]

    def get_type(self):
        """
        Gets the type of the model.

        :return: the model type (PREDICTION / CLUSTERING)
        :rtype: str
        """
        return self.get_info()["type"]

    def get_definition(self):
        """
        Gets the model definition.

        :rtype: dict
        """
        if self.definition is None:
            self.definition = backend_json_call("savedmodels/get", data={
                "projectKey": self.project_key,
                "savedModelId": self.get_id()
            })

        return self.definition

    def list_versions(self):
        """
        Lists the model versions.

        .. note::

            The ``versionId`` field can be used to call the :meth:`activate_version` method.

        :return: Information about versions of the saved model, as a list of dict.
            Fields are:

            * **versionId**: identifier of the model version
            * **active**: whether this version is active or not
            * **snippet**: detailed dict containing version information

        :rtype: list[dict]
        """
        if self.versions is None:
            self.versions = backend_json_call("savedmodels/list-versions", data={
                "projectKey": self.project_key,
                "savedModelId": self.get_id()
            })

        return self.versions

    def activate_version(self, version_id):
        """
        Activates the given version of the model.

        :param str version_id: the identifier of the version to activate
        """
        backend_void_call("savedmodels/set-active", data={
            "projectKey": self.project_key,
            "smId": self.get_id(),
            "versionId": version_id
        })

    def get_version_metrics(self, version_id):
        """
        Gets the training metrics of a given version of the model.

        :param str version_id: the identifier of the version from which to retrieve metrics
        :rtype: :class:`dataiku.core.saved_model.SavedModelVersionMetrics`
        """
        return SavedModelVersionMetrics(
            metrics.ComputedMetrics(backend_json_call("metrics/saved-models/get-values-for-version", data={
                "projectKey": self.project_key,
                "modelId": self.get_id(),
                "modelVersionId": version_id
            })), version_id)

    def get_version_checks(self, version_id):
        """
        Gets the training checks of the given version of the model.

        :param str version_id: the identifier of the version from which to retrieve checks
        :rtype: :class:`dataiku.core.metrics.ComputedChecks`
        """
        return metrics.ComputedChecks(backend_json_call("checks/saved-models/get-values-for-version", data={
                "projectKey": self.project_key,
                "modelId": self.get_id(),
                "modelVersionId": version_id
            }))

    def save_external_check_values(self, values_dict, version_id):
        """
        Saves checks on the model, the checks are saved with the type "external".

        :param dict values_dict: the values to save, as a dict. The keys of the dict are used as check names
        :param str version_id: the identifier of the version for which checks should be saved
        :rtype: dict
        """
        return backend_json_call("checks/saved-models/save-external-values", data={
            "projectKey": self.project_key,
            "modelId": self.get_id(),
            "modelVersionId": version_id,
            "data": json.dumps(values_dict)
        }, err_msg="Failed to save external check values")

    def get_predictor(self, version_id=None, optimize="BATCH"):
        """
        Returns a ``Predictor`` for the given version of the model.

        .. note::

            This predictor can then be used to preprocess and make predictions on a dataframe.

        :param optimize: If set to ``LATENCY``, attempts (if the model is compatible) to build a predictor optimized for latency (leveraging the ``dataikuscoring`` package).
            In this case, beware that the returned ``Predictor`` only supports the ``predictor.predict(df, with_input_cols=False, with_prediction=True, with_probas=False)`` method, with no other arguments passed.

        :param str version_id: the identifier of the version from which to build the predictor
            (defaults to **None**, current active version)

        :return: The predictor built from the given version of this model
        :rtype: Union[:class:`dataiku.core.saved_model.BasePredictor`, :class:`dataiku.core.saved_model.DkuScoringPredictor`]
        """

        versions = self.list_versions()

        if version_id is None:
            active_version = next(x for x in versions if x["active"])
            version_id = active_version["versionId"]
        else:
            try:
                active_version = next(x for x in versions if x['versionId'] == version_id)
            except StopIteration:
                raise Exception(u"No version found with version id {}.".format(version_id))

        if optimize == "LATENCY":
            try:
                if version_id not in self._dku_scoring_predictors:
                    import dataiku
                    from dataikuapi.dss.future import DSSFuture
                    python_export_future = DSSFuture.from_resp(dataiku.api_client(), backend_json_call("ml/prediction/export-python", data={
                        "fmi": active_version['snippet']['fullModelId']
                    }))
                    export_id = python_export_future.wait_for_result().get('exportId')
                    self._dku_scoring_predictors[version_id] = DkuScoringPredictor(export_id)
                return self._dku_scoring_predictors[version_id]
            except Exception as e:
                logger.warning("Could not use the python export for this saved model. Using the predictor optimized for batch instead. Reason: %s" % str(e))

        if version_id not in self._predictors:
            sm_id = self.get_id()
            res = backend_json_call("savedmodels/get-model-details", data={
                "projectKey": self.project_key,
                "smId": sm_id,
                "versionId": version_id
            })
            fmi = res["fmi"]
            model_folder = res["model_folder"]
            sm = res["saved_model"]
            backend_void_call("ml/grant-fs-read-acls", data={"fmi": fmi})
            model_folder_context = build_saved_model_folder_context(model_folder, self.project_key, self.get_id())
            self._predictors[version_id] = build_predictor_for_saved_model(model_folder_context, sm.get('savedModelType'), sm.get("conditionalOutputs", []), fmi)
        return self._predictors[version_id]

    @contextmanager
    def create_finetuned_llm_version(self, connection_name, quantization=None, set_active=True):
        """ Creates a new fine-tuned LLM version, using a context manager (experimental)

        Upon exit of the context manager, the new model version is made available with the content of the working directory.
        The model weights must use the safetensors format. This model will be loaded at inference time with trust_remote_code=False.

        Simple usage:

        .. code-block:: python

            with saved_model.create_finetuned_llm_version("MyLocalHuggingfaceConnection") as finetuned_llm_version:
                # write model files to finetuned_llm_version.working_directory
            # the new version is now available

        :param str connection_name: name of the connection to link this version
        :param str quantization: quantization mode, must be one of [None, "Q_4BIT", "Q_8BIT"] (default: None)
        :param bool set_active: if True, set the new version as active for this saved model (default: True)

        :return: yields a :class:`FinetunedLLMVersionTrainingParameters` object
        """
        def now_timestamp():
            try:
                return int(time.time_ns() / 1000000)
            except AttributeError:
                # Python 3.6 and lower
                return int(time.time() * 1000)

        # create working directory for model files
        model_folder_name = ''.join(random.choice(string.ascii_uppercase + string.digits) for _ in range(8))
        working_dir = os.path.join(os.getcwd(), model_folder_name)
        os.makedirs(working_dir, exist_ok=True)
        logger.info("Starting fine-tuning for version in %s", working_dir)

        # this is yielded to allow two-way communication with user code
        # they want to know about working_dir, we want to know about training parameters like dataset names, hyperparameters and metrics
        finetuned_llm_training_parameters = FinetunedLLMVersionTrainingParameters(self, working_dir)
        finetuned_llm_training_parameters.config['quantization'] = quantization

        start_time = now_timestamp()

        yield finetuned_llm_training_parameters

        if not os.path.isdir(working_dir):
            error = "fine-tuning failed, the working directory doesn't exist anymore"
            logger.warning(error)
            raise Exception(error)

        filenames = os.listdir(working_dir)
        # we could check files to make sure required files are present
        # we could filter files to exclude some for security or other reasons

        # create fine-tuned version in the back-end
        res = backend_json_call("savedmodels/llm-generic/create-finetuned-version", data={
            "projectKey": self.project_key,
            "smId": self.get_id(),
        })
        version_id = res["smVersionId"]

        # copy model files to the back-end
        try:
            res = backend_json_call("savedmodels/get-model-details", data={
                "projectKey": self.project_key,
                "smId": self.get_id(),
                "versionId": version_id
            })
            model_folder_context = build_saved_model_folder_context(res["model_folder"],
                                                                    self.project_key, self.get_id())

            with model_folder_context.get_folder_path_to_write() as saved_model_dir:
                for filename in filenames:
                    shutil.move(os.path.join(working_dir, filename), saved_model_dir)

                logger.info("Ending fine-tuning for version %s in %s", version_id, working_dir)
                end_time = now_timestamp()
                model_folder_context.write_json("train_info.json", {
                    "state": "DONE",
                    "startTime": start_time,
                    "endTime": end_time,
                    "trainingTime": end_time - start_time,
                })

                # write metrics file from event log if any
                fmp = finetuned_llm_training_parameters
                if "eventLog" in fmp.config:
                    from dataiku.llm.finetuning.metrics import read_training_metrics
                    llm_step_wise_training_metrics, nb_epochs, total_steps = read_training_metrics(fmp.config["eventLog"])
                    if not fmp.config.get("nbEpochs"):
                        fmp.config["nbEpochs"] = nb_epochs
                    if not fmp.config.get("totalSteps"):
                        fmp.config["totalSteps"] = total_steps

                    logger.info("Writing metrics file for version %s of model %s.%s", version_id, self.project_key, self.get_id())
                    model_folder_context.write_json("llm_stepwise_training_metrics.json", llm_step_wise_training_metrics)

                # exiting the folder context manager triggers uploading model files to the back-end in container exec
                logger.info("Synchronizing working directory")
        except Exception as exc:
            logger.exception("failed to store model files")
            backend_void_call("savedmodels/llm-generic/delete-finetuned-version", data={
                "projectKey": self.project_key,
                "smId": self.get_id(),
                "versionId": version_id
            })
            raise exc

        # save the new version
        logger.info("Saving new fine-tuned LLM version %s of model %s.%s", version_id, self.project_key, self.get_id())
        backend_json_call("savedmodels/llm-generic/save-finetuned-version", data={
            "projectKey": self.project_key,
            "smId": self.get_id(),
            "versionId": version_id,
            "connectionName": connection_name,
            "finetuningConfig": json.dumps(fmp.config),
        })

        # increment last trained version number
        logger.info("Incrementing the last trained version number for model %s.%s", self.project_key, self.get_id())
        backend_void_call("savedmodels/increment-last-train", data={
            "projectKey": self.project_key,
            "smId": self.get_id(),
            "jobId": version_id,  # this only has to be unique for this saved model so using version_id is ok
        })

        # set the new version as active if requested
        if set_active:
            self.activate_version(version_id)


def _parse_dates(df, model_params):
    """Parse dates and convert them to UTC"""

    df = df.copy(deep=False)
    (names, dtypes, parse_dates) = Dataset.get_dataframe_schema_st(model_params.schema["columns"], infer_with_pandas=False, bool_as_str=True)
    # For columns for which preparation output schema says date, parse it,
    # because the Pandas CSV parser does not do it
    if parse_dates is not False:
        for col_idx in parse_dates:
            col = model_params.schema["columns"][col_idx]["name"]
            if col in df:
                # date columns must always be converted into UTC
                df[col] = pd.to_datetime(df[col], utc=True)

    return df


def _preformat(df, model_params):
    from dataiku.doctor.utils import normalize_dataframe

    parsed_df = _parse_dates(df, model_params)

    normalize_dataframe(parsed_df, model_params.preprocessing_params["per_feature"])
    for col in parsed_df:
        logger.info("Normalized column: %s -> %s" % (col, parsed_df[col].dtype))

    per_feature_params = model_params.preprocessing_params["per_feature"]
    # don't cast to integer types, because astype() will fail on NaN, instead behave like ml_dtype_from_dss_column()
    types_map = {
        'int': np.float64,
        'bigint': np.float64,
        'float': np.float64,
        'double': np.float64,
        'boolean': bool
    }

    column_types = {}
    for c in model_params.schema['columns']:
        c_name = c['name']
        if c_name in df.columns and c_name in per_feature_params and per_feature_params[c_name]["role"] not in ["REJECT", "TIME", "TIMESERIES_IDENTIFIER", "TREATMENT"]:
            t = str(c['type'])
            # cast boolean features to string so that dummyfication works properly
            if t == 'boolean':
                t = 'string'
            # cast numerical date features to float because they are EPOCH (categorical date features are casted to object below)
            elif t in ['date', 'dateonly', 'datetimenotz']:
                t = 'bigint'
            column_types[six.text_type(c_name)] = types_map[t] if t in types_map else t

    # replace "string" by object because astype behaves differently for both
    # also replace any type by object for categorical variables, otherwise this will break the dummyfier
    for k in column_types:
        if column_types[k] in {"string", "str"} or (k in per_feature_params and per_feature_params[k]["type"] == "CATEGORY"):
            column_types[k] = object
    return parsed_df.astype(column_types)


def is_model_prediction(model_type):
    return model_type == "PREDICTION"


class ModelPartitioningInfo(object):

    class PartitionInfo(object):
        def __init__(self, name, fmi, model_folder_context, preprocessing_folder_context, split_folder_context, split_desc):
            """
            :type name: str
            :type fmi: str or None
            :type model_folder_context: dataiku.base.folder_context.FolderContext
            :type preprocessing_folder_context: dataiku.base.folder_context.FolderContext
            :type split_folder_context:dataiku.base.folder_context.FolderContext
            :type split_desc: dict
            """
            self.name = name
            self.fmi = fmi
            self.model_folder_context = model_folder_context
            self.preprocessing_folder_context = preprocessing_folder_context
            self.split_folder_context = split_folder_context
            self.split_desc = split_desc

    def __init__(self, core_params, model_folder_context, fmi):
        self._partitioned = core_params is not None and core_params.get('partitionedModel', {}).get('enabled', False)
        self.core_params = core_params
        self.partitions_info = []
        self._is_base_model = False

        if self._partitioned:
            if (os.environ.get("DKU_NODE_TYPE") == "api" or os.environ.get("DKU_LAMBDA_DEVSERVER")) and model_folder_context.isdir("parts"):
                # On the API node, only one version for each partition
                # files hierarchy: {sm_id}/parts/{partition_id}/{ model partition files }
                self._is_base_model = True
                parts_folder_context = model_folder_context.get_subfolder_context("parts")
                with parts_folder_context.get_folder_path_to_read() as parts_folder_path:
                    partition_folder_names = os.listdir(parts_folder_path)
                    for part_folder_name in partition_folder_names:
                        part_fmi = None  # We don't care about the full model id for API node
                        part_folder_context = parts_folder_context.get_subfolder_context(part_folder_name)
                        split_folder_context = part_folder_context.get_subfolder_context("split")
                        split_desc = split_folder_context.read_json("split.json")
                        self.partitions_info.append(
                            ModelPartitioningInfo.PartitionInfo(
                                PartitionEscaper.unescape(part_folder_name), part_fmi, part_folder_context, part_folder_context, split_folder_context,
                                split_desc
                            )
                        )

            elif fmi is not None:
                # fmi should always be passed for analysis/saved models so that we can fetch partitions and folders from the backend
                partitions_fmi_folder_contexts = get_partitions_fmi_folder_contexts(fmi)
                self._is_base_model = partitions_fmi_folder_contexts is not None
                if self._is_base_model:
                    for partition_name, partition_fmi_folder_contexts in partitions_fmi_folder_contexts.items():
                        partition_fmi = partition_fmi_folder_contexts.fmi
                        part_model_folder_context = partition_fmi_folder_contexts.model_folder_context
                        part_preprocessing_folder_context = partition_fmi_folder_contexts.preprocessing_folder_context
                        part_split_folder_context = partition_fmi_folder_contexts.split_folder_context
                        part_split_desc = part_split_folder_context.read_json(partition_fmi_folder_contexts.split_desc_filename)

                        self.partitions_info.append(
                            ModelPartitioningInfo.PartitionInfo(
                                partition_name, partition_fmi, part_model_folder_context, part_preprocessing_folder_context, part_split_folder_context,
                                part_split_desc
                            )
                        )

    def is_partitioned(self):
        """
        Checks whether the current model is partitioned

        :return: True if the partitioning is activated for the current model, False otherwise
        :rtype: bool
        """
        return self._partitioned

    def is_base_model(self):
        return self._is_base_model


def build_predictor_for_saved_model(model_folder_context, saved_model_type, conditional_outputs, fmi=None):
    """
    :param model_folder_context:
    :param saved_model_type:
    :param conditional_outputs:
    :param fmi: Saved model FMI or None.
                Mostly used to build the ModelPartitioningInfo object.
                None only for API node scoring
                # TODO @predictor unentangle build_predictor and ModelPartitioningInfo to remove fmi argument

    :return: predictor for the saved model
    """
    if saved_model_type == "MLFLOW_PYFUNC" or saved_model_type == "PROXY_MODEL":
        from dataiku.external_ml.mlflow.predictor import MLflowPredictor
        return MLflowPredictor(model_folder_context)
    elif saved_model_type == "DSS_MANAGED":
        split_folder_context = model_folder_context.get_subfolder_context("split")
        split_desc = split_folder_context.read_json("split.json")

        core_params = model_folder_context.read_json("core_params.json")
        model_type = core_params.get("taskType")
        return build_predictor(model_type, model_folder_context, model_folder_context, split_folder_context,
                               conditional_outputs, core_params, split_desc, fmi=fmi)
    else:
        raise Exception("DSS doesn't support get_predictor for Saved Model type: %s." % saved_model_type)


def build_predictor(model_type, model_folder_context, preprocessing_folder_context, split_folder_context,
                    conditional_outputs, core_params, split_desc, train_split_desc=None,
                    train_split_folder_context=None, fmi=None):

    # Handle base partitioned model
    model_part_info = ModelPartitioningInfo(core_params, model_folder_context, fmi)
    if model_part_info.is_partitioned() and model_part_info.is_base_model():
        return PartitionedModelPredictor(core_params, model_folder_context, model_type,
                                         conditional_outputs, model_part_info, preprocessing_folder_context)

    is_prediction = is_model_prediction(model_type)

    # import various parameters
    preprocessing_params = preprocessing_folder_context.read_json("rpreprocessing_params.json")
    modeling_params = model_folder_context.read_json("rmodeling_params.json")
    resolved_params = model_folder_context.read_json("actual_params.json")["resolved"]

    user_meta = model_folder_context.read_json("user_meta.json") if model_folder_context.isfile("user_meta.json") else {}

    if model_folder_context.isfile("perf.json.gz"):
        model_perf = model_folder_context.read_json("perf.json.gz")
    elif model_folder_context.isfile("perf.json"):
        model_perf = model_folder_context.read_json("perf.json")
    else:
        model_perf = {}
    if is_prediction:
        cluster_name_map = None
    else:
        cluster_name_map = {}
        if "clusterMetas" in user_meta:
            for cluster_id, cluster_data in user_meta["clusterMetas"].items():
                cluster_name_map[cluster_id] = cluster_data["name"]

    model_params = ModelParams(model_type, modeling_params, preprocessing_params, core_params, split_desc,
                               user_meta, model_perf, conditional_outputs, cluster_name_map, preprocessing_folder_context,
                               model_folder_context, split_folder_context, resolved_params, train_split_desc,
                               train_split_folder_context)

    if core_params["backendType"] == "DEEP_HUB":
        return DeepHubPredictor(model_params)

    is_keras_backend = modeling_params.get("algorithm") == "KERAS_CODE"
    is_timeseries = core_params.get(doctor_constants.PREDICTION_TYPE) == doctor_constants.TIMESERIES_FORECAST
    is_causal = core_params.get(doctor_constants.PREDICTION_TYPE) in doctor_constants.CAUSAL_PREDICTION_TYPES

    from dataiku.doctor.utils.gpu_execution import get_gpu_config_from_core_params, log_nvidia_smi_if_use_gpu

    gpu_config = get_gpu_config_from_core_params(core_params)
    log_nvidia_smi_if_use_gpu(gpu_config=gpu_config)
    if is_timeseries:
        _set_ts_predictor_gpu(gpu_config, model_folder_context)
    else:
        _set_predictor_gpu(modeling_params, is_keras_backend, preprocessing_params, gpu_config)

    # load model
    if is_keras_backend:
        try:
            from dataiku.doctor.deep_learning import keras_model_io_utils
            model = keras_model_io_utils.load_model(model_folder_context)
        except IOError as exc:
            raise NotImplementedError("Failed to load saved model to predict, the model should be trained using the Keras engine: %s" % exc)
    elif not is_causal:
        try:
            from dataiku.doctor.utils.model_io import load_model_from_folder
            clf = load_model_from_folder(model_folder_context, is_prediction=is_prediction)
            if hasattr(clf, "post_process") and callable(clf.post_process):
                # known post-processable estimators include DkuLassoLarsRegressor, DkuLassoLarsClassifier and TwoStepClustering
                logger.info("Post-processing model")
                clf.post_process(user_meta)
        except IOError as exc:
            raise NotImplementedError("Failed to load saved model to predict, the model should be trained using the python engine: %s" % exc)

    if is_timeseries:
        # initialize is required for models that were not serialized with all parameters from release 11.2 to 12.0
        clf.initialize(core_params, resolved_params)
        return TimeseriesPredictor(model_params, clf, preprocessing_folder_context)

    if is_causal:
        from dataiku.doctor.utils.model_io import from_pkl
        dku_causal_model = from_pkl(model_folder_context, "causal_model.pkl")
        propensity_model_file_name = "propensity_model.pkl"
        if model_folder_context.isfile(propensity_model_file_name):
            propensity_model = from_pkl(model_folder_context, propensity_model_file_name)
        else:
            propensity_model = None
        return CausalPredictor(model_params, dku_causal_model, propensity_model, preprocessing_folder_context)

    # create preprocessing
    from dataiku.doctor.preprocessing_handler import PredictionPreprocessingHandler
    from dataiku.doctor.preprocessing_handler import ClusteringPreprocessingHandler
    if is_prediction:
        from dataiku.doctor.prediction.common import PredictionAlgorithmNaNSupport
        nan_support = PredictionAlgorithmNaNSupport(modeling_params, preprocessing_params)
        preprocessing_handler = PredictionPreprocessingHandler.build(core_params, preprocessing_params, preprocessing_folder_context,
                                                                     nan_support=nan_support)
    else:
        preprocessing_handler = ClusteringPreprocessingHandler({}, preprocessing_params, preprocessing_folder_context)

    collector_data = preprocessing_folder_context.read_json("collector_data.json")
    preprocessing_handler.collector_data = collector_data

    if modeling_params["algorithm"] == "PYTHON_ENSEMBLE":
        return EnsemblePredictor(model_params, clf)
    elif is_keras_backend:
        from dataiku.doctor.deep_learning.keras_utils import tag_special_features
        per_feature = preprocessing_params["per_feature"]
        tag_special_features(per_feature)
        preprocessing = KerasPreprocessing(preprocessing_handler, modeling_params, per_feature)
        return KerasPredictor(model_params, preprocessing, model, batch_size=100)
    else:
        preprocessing = Preprocessing(preprocessing_handler, modeling_params)
        _init_pipeline(collector_data, preprocessing.pipeline)
        feature_names = _get_or_compute_feature_names(model_params.model_perf, collector_data, preprocessing.pipeline)
        return Predictor(model_params, preprocessing, feature_names, clf)


def _set_ts_predictor_gpu(gpu_config, model_folder_context):
    version_info = model_folder_context.read_json("version_info.json")
    source_dss_version = int(version_info.get("trainedWithDSSConfVersion", 0))

    # timeseries models could previously be trained with a mxnet deviceid != 0
    # in this case, setting cuda visible devices to 0 will cause the model unpickling to fail
    # therefore, we don't override the gpu selection for timeseries models when deployed
    if source_dss_version >= 12400:
        from dataiku.doctor.utils.gpu_execution import GluonTSMXNetGPUCapability
        is_api_node = os.environ.get("DKU_NODE_TYPE") == "api" or os.environ.get("DKU_LAMBDA_DEVSERVER") is not None
        GluonTSMXNetGPUCapability.configure_predictor_gpu_env(gpu_config, is_api_node)


def _set_predictor_gpu(modeling_params, is_keras_backend, preprocessing_params, gpu_config):
    from dataiku.doctor.utils.gpu_execution import KerasGPUCapability, XGBOOSTGpuCapability, DeepNNGpuCapability, SentenceEmbeddingGpuCapability

    is_api_node = os.environ.get("DKU_NODE_TYPE") == "api" or os.environ.get("DKU_LAMBDA_DEVSERVER") is not None
    algorithm = modeling_params.get("algorithm", "")

    # api node, set gpu 0
    if is_keras_backend:
        if is_api_node:
            KerasGPUCapability.force_gpu_0(gpu_config)

        use_gpu = KerasGPUCapability.should_use_gpu(gpu_config)
        from dataiku.doctor.deep_learning import gpu
        if use_gpu:
            gpu.load_gpu_options_only_allow_growth()
        else:
            gpu.deactivate_gpu()

    elif algorithm in ["XGBOOST_REGRESSION", "XGBOOST_CLASSIFICATION"]:
        XGBOOSTGpuCapability.configure_predictor_gpu_env(gpu_config, is_api_node)

    elif algorithm in ["DEEP_NEURAL_NETWORK_REGRESSION", "DEEP_NEURAL_NETWORK_CLASSIFICATION"]:
        DeepNNGpuCapability.configure_predictor_gpu_env(gpu_config, is_api_node)

    else:
        is_sentence_embedding = False

        for feature in preprocessing_params["per_feature"].values():
            if feature.get("role", "") == "INPUT" and feature.get("type", "") == "TEXT" and feature.get('text_handling', "") == "SENTENCE_EMBEDDING":
                is_sentence_embedding = True
                break

        if is_sentence_embedding:
            SentenceEmbeddingGpuCapability.configure_predictor_gpu_env(gpu_config, is_api_node)


def _init_pipeline(collector_data, pipeline, return_feature_names=False):
    df = _generate_empty_df(collector_data)
    train_block = pipeline.process(df)["TRAIN"]
    if return_feature_names:
        return train_block.columns()

def _generate_empty_df(collector_data):
    dat = {}
    for f in collector_data["per_feature"]:
        par = collector_data["per_feature"][f]
        if "category_possible_values" in par and len(par["category_possible_values"]) > 0:
            val = par["category_possible_values"][0]
        elif "stats" in par and "average" in par["stats"]:
            val = par["stats"]["average"]
        elif par.get('is_vector', False):
            val = "[" + ",".join(["0"] * par.get('vector_length', 0)) + "]"
        else:
            val = ""
        dat[f] = [val]
    df = pd.DataFrame(dat)
    return df


def _get_or_compute_feature_names(model_perf, collector_data, pipeline):

    feature_names = model_perf.get("processed_feature_names")

    if feature_names is None:
        logger.info("processed_feature_names missing from model Perf, running the pipeline on empty data to regenerate that list")
        # Kfolds cross-test models trained prior to v12.1 do not have the list of processed_feature_names
        # stored in their model perf file. (see https://app.shortcut.com/dataiku/story/95140)
        # the following code is a workaround to run the whole pipeline on artificial empty data to get the generated column names.
        feature_names = _init_pipeline(collector_data, pipeline, return_feature_names=True)
    return feature_names



class Preprocessing:
    def __init__(self, preprocessing_handler, modeling_params, with_prediction=False):
        self.pipeline = preprocessing_handler.build_preprocessing_pipeline()
        self.pipeline_with_target = preprocessing_handler.build_preprocessing_pipeline(with_target=True,
                                                                                       allow_empty_mf=True,
                                                                                       with_prediction=with_prediction)
        self.modeling_params = modeling_params
        self.debug_options = {}

    def _transform(self, df, with_target, with_sample_weights):
        if with_target or with_sample_weights:
            pipeline = self.pipeline_with_target
        else:
            pipeline = self.pipeline

        return pipeline.process(df), pipeline

    def _enrich_preprocess(self, preprocessed_result, transformed, with_target, with_sample_weights, with_unprocessed, with_transformed):
        if with_target:
            target = transformed.get("target", None)
            preprocessed_result = preprocessed_result + (target,)

        if with_sample_weights:
            sample_weights = transformed.get("weight", None)
            preprocessed_result = preprocessed_result + (sample_weights,)

        if with_unprocessed:
            unprocessed_df = transformed.get("UNPROCESSED", None)
            preprocessed_result = preprocessed_result + (unprocessed_df,)

        if with_transformed:
            preprocessed_result = preprocessed_result + (transformed,)

        return preprocessed_result

    def preprocess(self, df, with_target=False, with_sample_weights=False, with_unprocessed=False, with_transformed=False):
        from dataiku.doctor.prediction.common import prepare_multiframe
        transformed, _ = self._transform(df, with_target, with_sample_weights)
        transformed_train = transformed["TRAIN"]
        input_mf_index = transformed_train.index

        if self.debug_options.get("dumpPreprocessedFirstLine", False):
            fail_na = self.debug_options.get("failOnNADummy", False)
            fail_others = self.debug_options.get("failOnOthersDummy", False)
            features_X_df = transformed_train.as_dataframe()
            logger.info("Dumping first line of preprocessed data")
            for col in features_X_df:
                logger.info("F %s = %s" % (col, features_X_df[col].iloc[0]))

                val = features_X_df[col].iloc[0]
                if val == 1 and fail_na and col.startswith("dummy:") and col.endswith(":N/A"):
                    raise Exception("Unexpected N/A dummy: %s  = %s" % (col, val))
                if val == 1 and fail_others and col.startswith("dummy:") and col.endswith(":__Others__"):
                    raise Exception("Unexpected Others dummy: %s  = %s" % (col, val))

        X = prepare_multiframe(transformed_train, self.modeling_params)[0]
        is_empty = X.shape[0] == 0

        ret = (X, input_mf_index, is_empty)
        ret = self._enrich_preprocess(ret, transformed, with_target, with_sample_weights, with_unprocessed, with_transformed)
        return ret


class KerasPreprocessing(Preprocessing):

    def __init__(self, preprocessing_handler, modeling_params, per_feature):
        Preprocessing.__init__(self, preprocessing_handler, modeling_params)
        self.per_feature = per_feature

    def preprocess(self, df, with_target=False, with_sample_weights=False, with_unprocessed=False, with_transformed=False):
        from dataiku.doctor.deep_learning.keras_utils import split_train_per_input
        transformed, pipeline = self._transform(df, with_target, with_sample_weights)
        transformed_train = transformed["TRAIN"]
        input_mf_index = transformed_train.index
        is_empty = transformed_train.shape()[0] == 0
        X = split_train_per_input(transformed_train, self.per_feature,
                                  pipeline.generated_features_mapping)

        ret = (X, input_mf_index, is_empty)
        ret = self._enrich_preprocess(ret, transformed, with_target, with_sample_weights, with_unprocessed, with_transformed)
        return ret


class ModelParams:
    def __init__(self, model_type, modeling_params, preprocessing_params, core_params, split_desc, user_meta, model_perf,
                 conditional_outputs, cluster_name_map, preprocessing_folder_context, model_folder_context,
                 split_folder_context, resolved_params, train_split_desc=None, train_split_folder_context=None):
        self.modeling_params = modeling_params
        self.preprocessing_params = preprocessing_params
        self.core_params = core_params
        self.user_meta = user_meta
        self.schema = split_desc["schema"]
        self.split_desc = split_desc
        self.train_split_desc = train_split_desc or split_desc
        self.train_split_folder_context = train_split_folder_context or split_folder_context
        self.model_perf = model_perf
        self.model_type = model_type
        if preprocessing_params.get("target_remapping", None) is not None:
            self.target_map = {t["mappedValue"]: t["sourceValue"] for t in preprocessing_params["target_remapping"]}
        self.conditional_outputs = conditional_outputs
        self.cluster_name_map = cluster_name_map
        self.model_folder = model_folder_context.get_absolute_folder_path()  # for legacy reason
        self.model_folder_context = model_folder_context
        self._preprocessing_folder_context = preprocessing_folder_context
        self.split_folder_context = split_folder_context
        self.resolved_params = resolved_params

        from dataiku.doctor.prediction.overrides.ml_overrides_params import ml_overrides_params_from_model_folder
        self.ml_overrides_params = ml_overrides_params_from_model_folder(model_folder_context)


def _add_proba_percentiles(pred_df, model_perf, target_map):
    percentile = pd.Series(model_perf["probaPercentiles"])
    proba_1 = "proba_" + target_map[1]
    pred_df["proba_percentile"] = pred_df[proba_1].apply(
        lambda p: percentile.where(percentile <= p).count() + 1)


def _add_conditional_output(pred_df, co):
    inp = pred_df[co["input"]]
    acc = inp.notnull()  # condition accumulator
    for r in co["rules"]:
        if r["operation"] == 'GT':
            cond = inp > r["operand"]
        elif r["operation"] == 'GE':
            cond = inp >= r["operand"]
        elif r["operation"] == 'LT':
            cond = inp < r["operand"]
        elif r["operation"] == 'LE':
            cond = inp <= r["operand"]
        pred_df.loc[acc & cond, co["name"]] = r["output"]
        acc &= ~cond
    pred_df.loc[acc, co["name"]] = co.get("defaultOutput", None)


class BasePredictor:
    """
    Object allowing to preprocess and make predictions on a dataframe.
    """

    def __init__(self, params, clf):
        from dataiku.doctor.prediction.scorable_model import ScorableModel
        from dataiku.doctor.prediction.prediction_interval_model import PredictionIntervalsModel
        self.params = params
        # Leaving _clf here as it can be used externally
        self._clf = clf
        self._prediction_type = self.params.core_params.get("prediction_type")
        model_type = self.params.core_params.get("taskType")
        pred_interval_model = PredictionIntervalsModel.load_or_none(self.params.model_folder_context,
                                                                    self.params.core_params)
        self._model = ScorableModel.build(clf, model_type, self._prediction_type,
                                          self.params.modeling_params["algorithm"],
                                          self.params.preprocessing_params,
                                          self.params.ml_overrides_params,
                                          pred_interval_model)
        if self._prediction_type == doctor_constants.BINARY_CLASSIFICATION:
            self._model.set_threshold(self.params.user_meta.get("activeClassifierThreshold", 0.5))

        if self.params.conditional_outputs is not None:
            self.conditional_output_names = [co["name"] for co in self.params.conditional_outputs]
        else:
            self.conditional_output_names = []
        if self._model.is_classification():
            self.classes = self._model.get_classes()
        else:
            self.classes = None

        if model_type == "PREDICTION":
            from dataiku.doctor.individual_explainer import IndividualExplainer
            self._individual_explainer = IndividualExplainer(self,
                                                             self.params.model_folder_context,
                                                             self.params._preprocessing_folder_context,
                                                             self.params.train_split_desc,
                                                             self.params.train_split_folder_context,
                                                             self.params.preprocessing_params["per_feature"],
                                                             self.params.modeling_params.get("algorithm",
                                                                                             None) == "PYTHON_ENSEMBLE",
                                                             self.params.core_params["prediction_type"],
                                                             self.params.core_params["weight"].get(
                                                                "sampleWeightVariable", None))
        else:
            self._individual_explainer = None

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        pass

    def get_classes(self):
        """
        Returns the classes from which this model will predict if a classifier, None if a regressor
        """
        return self.classes

    def get_proba_columns(self):
        """
        Returns the names of the probability columns if a classifier, None if a regressor
        """
        if self.classes is None:
            return None
        else:
            return ["proba_%s" % c for c in self.classes]

    def get_conditional_output_names(self):
        """
        Returns the name of all conditional outputs defined for this model (note: limited to binary classifiers)
        """
        return self.conditional_output_names

    def ready_explainer(self):
        """
        Preload the necessary components to compute explanations
        :param df: data from which the random sample have to be drawn
        :type df: pd.DataFrame
        """
        self._check_can_compute_explanations()
        self._individual_explainer.make_ready()

    @staticmethod
    def _check_can_compute_explanations():
        pass

    def _set_debug_options(self, debug_options):
        raise NotImplementedError()

    def _compute_explanations(self, df, method, n_explanations, mc_steps):
        if not self._individual_explainer.is_ready():
            self.ready_explainer()

        explanations_df = self._individual_explainer.explain(df, n_explanations, method,
                                                             shapley_background_size=mc_steps)

        explanations_df = self._individual_explainer.format_explanations(explanations_df, n_explanations)
        explanations_df.columns = [u"explanations_{}".format(safe_unicode_str(col)) for col in explanations_df.columns]
        return explanations_df


class Predictor(BasePredictor):
    """
    Object allowing to preprocess and make predictions on a dataframe.
    """

    def __init__(self, params, preprocessing, features, clf):
        self.preprocessing = preprocessing
        BasePredictor.__init__(self, params, clf)
        self.features = features
        self.debug_options = {}

    def _get_prediction_dataframe(self, input_df, with_prediction, with_probas,
                                  with_conditional_outputs, with_proba_percentile):
        if self.params.model_type == "PREDICTION":
            pred_df = self._prediction_type_dataframe(input_df, with_prediction, with_probas)
            self._add_percentiles_and_condoutputs(pred_df, with_proba_percentile, with_conditional_outputs)
            return pred_df
        else:
            if not with_prediction:
                raise ValueError("Predicting a clustering model with with_prediction=False. Oops.")
            return self._clustering_type_dataframe(input_df)

    def _add_percentiles_and_condoutputs(self, pred_df, with_proba_percentile, with_conditional_outputs):

        # Only compute percentiles and conditional outputs if there are actual predictions
        if pred_df.shape[0] == 0:
            return

        # percentiles and conditional outputs if applicable
        if "probaPercentiles" in self.params.model_perf and self.params.model_perf["probaPercentiles"] \
                and with_proba_percentile:
            _add_proba_percentiles(pred_df, self.params.model_perf, self.params.target_map)
        if with_conditional_outputs and self.params.conditional_outputs is not None:
            for co in self.params.conditional_outputs:
                _add_conditional_output(pred_df, co)

    def _prediction_type_dataframe(self, input_df, with_prediction, with_probas):
        X, input_mf_index, is_empty, unprocessed_df = self.preprocessing.preprocess(input_df, with_unprocessed=True)
        if is_empty:
            # special case for all empty scoring data : scikit-learn models reject them
            pred_df = pd.DataFrame(columns=["prediction"], index=input_df.index)
            return pred_df

        return self._predict_preprocessed(X, unprocessed_df, input_mf_index, with_prediction, with_probas)

    def _predict_preprocessed(self, X, unprocessed_df, input_mf_index, with_prediction, with_probas):
        if self._prediction_type == doctor_constants.REGRESSION:
            if with_prediction:
                prediction_result = self._model.compute_predictions(X, unprocessed_df)
                pred_df = prediction_result.as_dataframe()
            else:
                raise ValueError("Predicting a regression model with with_prediction=False. Oops.")
        else:
            proba_df = None
            prediction_result = self._model.compute_predictions(X, unprocessed_df)
            if with_probas and self.classes is not None and prediction_result.has_probas():
                proba_columns = self.get_proba_columns()
                proba_df = pd.DataFrame(data=prediction_result.probas, columns=proba_columns)

            pred_df = pd.DataFrame({"prediction": prediction_result.preds})
            pred_df = pd.concat([pred_df, proba_df], axis=1)  # `concat` won't do anything if proba_df is None

        # put back the original index (i.e. account for dropped rows)
        pred_df.index = input_mf_index

        from dataiku.doctor.prediction.overrides.ml_overrides_results import OverridesResultsMixin
        from dataiku.doctor.prediction.overrides.ml_overrides_params import OVERRIDE_INFO_COL
        if isinstance(prediction_result, OverridesResultsMixin):
            pred_df[OVERRIDE_INFO_COL] = prediction_result.compute_and_return_info_column()

        return pred_df

    def _clustering_type_dataframe(self, input_df):
        from dataiku.doctor.clustering.clustering_fit import clustering_predict
        try:
            custom_labels = self._clf.get_cluster_labels()

            def map_fun_custom(i):
                name = custom_labels[i]
                return self.params.cluster_name_map.get(name, name)

            naming = map_fun_custom
        except AttributeError:
            def map_fun(i):
                name = "cluster_%i" % i
                return self.params.cluster_name_map.get(name, name)

            naming = map_fun
        transformed = self.preprocessing.pipeline.process(input_df)
        if transformed["TRAIN"].as_dataframe().shape[0] == 0:
            # special case for all empty scoring data : scikit-learn models reject them
            result = pd.DataFrame(columns=["cluster"], index=input_df.index)
        else:
            labels_arr, anomaly_scores_arr = clustering_predict(self.params.modeling_params, self._clf, transformed)
            cluster_labels = pd.Series(labels_arr, name="cluster_labels").map(naming)
            cluster_labels.index = transformed["TRAIN"].index
            labels_df = pd.DataFrame({"cluster": cluster_labels}).reindex(input_df.index)
            if anomaly_scores_arr is not None:
                anomaly_scores_series = pd.Series(anomaly_scores_arr, name="anomaly_score", index=transformed["TRAIN"].index)
                anomaly_scores_series.reindex(input_df.index)
                result = pd.concat([labels_df, anomaly_scores_series], axis=1)
            else:
                result = labels_df
        if self.params.preprocessing_params["outliers"]["method"] == "CLUSTER":
            outlier_name = self.params.cluster_name_map.get(doctor_constants.CLUSTER_OUTLIERS, doctor_constants.CLUSTER_OUTLIERS)
            result['cluster'].fillna(outlier_name, inplace=True)
        return result

    def _get_preprocessing_folder_context(self):
        return self.preprocessing.get_folder_context()

    def get_features(self):
        """
        Returns the feature names generated by this predictor's preprocessing
        """
        return self.features

    def predict(self, df, with_input_cols=False, with_prediction=True, with_probas=True, with_conditional_outputs=False,
                with_proba_percentile=False, with_explanations=False,
                explanation_method="ICE",
                n_explanations=3,
                n_explanations_mc_steps=100, **kwargs):
        """
        Predict a dataframe.
        The results are returned as a dataframe with columns corresponding to the various prediction information.

        :param with_input_cols: whether the input columns should also be present in the output
        :param with_prediction: whether the prediction column should be present
        :param with_probas: whether the probability columns should be present
        :param with_conditional_outputs: whether the conditional outputs for this model should be present (binary classif)
        :param with_proba_percentile: whether the percentile of the probability should be present (binary classif)
        :param with_explanations: whether explanations should be computed for each prediction
        :param explanation_method: method to compute the explanations
        :param n_explanations: number of explanations to output for each prediction
        :param n_explanations_mc_steps: number of Monte Carlo steps for SHAPLEY method (higher means more precise but slower)
        """
        if with_explanations:
            self._check_can_compute_explanations()

        final_df = self.preformat(df)
        pred_df = self._get_prediction_dataframe(final_df, with_prediction, with_probas, with_conditional_outputs,
                                                 with_proba_percentile)

        results = []
        if with_input_cols:
            results.append(df)
        results.append(pred_df)
        if with_explanations:
            explanations_df = self._compute_explanations(final_df, explanation_method,
                                                         n_explanations, n_explanations_mc_steps)
            results.append(explanations_df)

        results_df = pd.concat(results, axis=1)

        # Whether the output dataframe is aligned with the input dataframe.
        # Not used in our api node code, but might be used in user custom code, also doubles as a reminder / semi-comment.
        results_df._aligned = True

        return results_df

    def preformat(self, df):
        """
        Formats data originating from json (api node, interactive scoring) so that it's compatible with `preprocess`
        """
        return _preformat(df, self.params)

    def preprocess(self, df):
        """
        Preprocess a dataframe.
        The results are returned as a numpy 2-dimensional matrix (which may be sparse). The columns of this matrix
        correspond to the generated features, which can be listed by the `get_features` property of this Predictor.
        """
        return self.preprocessing.preprocess(df)

    def get_preprocessing(self):
        return self.preprocessing

    def _set_debug_options(self, debug_options):
        self.preprocessing.debug_options = debug_options


class KerasPredictor(Predictor):

    def __init__(self, params, preprocessing, model, batch_size=100):
        Predictor.__init__(self, params, preprocessing, features=None, clf=model)
        self.batch_size = batch_size

    def _get_prediction_dataframe(self, input_df, with_prediction, with_probas,
                                  with_conditional_outputs, with_proba_percentile):
        if self.params.model_type == "PREDICTION":

            pred_df_list = []
            num_rows = input_df.shape[0]
            nb_batches = int(math.ceil(num_rows * 1.0 / self.batch_size))

            for num_batch in range(nb_batches):

                input_df_batch = input_df.iloc[num_batch * self.batch_size: (num_batch + 1) * self.batch_size, :]
                pred_df_list.append(self._prediction_type_dataframe(input_df_batch, with_prediction, with_probas))

            pred_df = pd.concat(pred_df_list)
            self._add_percentiles_and_condoutputs(pred_df, with_proba_percentile, with_conditional_outputs)
            return pred_df

        else:
            raise ValueError("Clustering problems are not implemented with Keras Backend.")

    @staticmethod
    def _check_can_compute_explanations():
        raise ValueError("Deep learning models are not compatible with explanations")


class CausalPredictor(object):
    def __init__(self, params, dku_causal_model, propensity_model, preprocessing_folder_context):
        # Must do a local import because CausalPredictionScoringHandler requires econml library not available by default
        from dataiku.doctor.causal.score.scoring_handler import CausalPredictionScoringHandler

        self.params = params
        collector_data = preprocessing_folder_context.read_json("collector_data.json")
        self.scoring_handler = CausalPredictionScoringHandler(params.core_params,
                                                              params.preprocessing_params,
                                                              params.resolved_params,
                                                              dku_causal_model,
                                                              propensity_model,
                                                              collector_data,
                                                              preprocessing_folder_context)

    @property
    def with_propensity(self):
        return self.scoring_handler.propensity_scoring_enabled

    @with_propensity.setter
    def with_propensity(self, value):
        if value is True and self.scoring_handler.propensity_model is None:
            logger.warning("Could not find trained propensity model, causal predictor won't compute propensity")
        else:
            self.scoring_handler.propensity_scoring_enabled = value

    def predict(self, data, *args, **kwargs):
        data = _preformat(data, self.params)

        pred_df = self.scoring_handler.score(data)
        # Whether the output dataframe is aligned with the input dataframe.
        # Not used in our api node code, but might be used in user custom code, also doubles as a reminder / semi-comment.
        pred_df._aligned = True
        return pred_df

    def _set_debug_options(self, debug_options):
        pass


class TimeseriesPredictor(object):
    def __init__(self, params, clf, preprocessing_folder):
        # Must do a local imports because it requires libraries not available everywhere (e.g. `gluonts`)
        from dataiku.doctor.timeseries.score.scoring_handler import TimeseriesScoringHandler
        from dataiku.doctor.timeseries.models import TimeseriesForecastingAlgorithm
        from dataiku.doctor.timeseries.models import SeasonalTrendLoess

        is_partitioned = params.core_params.get("partitionedModel", {}).get("enabled", False)
        self.partition_columns = params.core_params.get("partitionedModel", {}).get("dimensionNames") if is_partitioned else None
        self.quantiles = params.core_params[doctor_constants.QUANTILES]

        # We only refit the model for STL, otherwise we score without refit
        algorithm = TimeseriesForecastingAlgorithm.build(params.modeling_params["algorithm"])
        self.refit = isinstance(algorithm, SeasonalTrendLoess)

        self.scoring_handler = TimeseriesScoringHandler(
            params.core_params,
            params.preprocessing_params,
            params.modeling_params,
            params.resolved_params,
            clf,
            preprocessing_folder,
        )

        self.params = params

    def predict(self, data, *args, **kwargs):
        data = _preformat(data, self.params)

        pred_df = self.scoring_handler.score(
            data,
            self.params.schema,
            self.quantiles,
            past_time_steps_to_include=0,
            partition_columns=self.partition_columns,
            refit=self.refit,
        )
        # Whether the output dataframe is aligned with the input dataframe.
        # Not used in our api node code, but might be used in user custom code, also doubles as a reminder / semi-comment.
        pred_df._aligned = False

        return pred_df

    def _set_debug_options(self, debug_options):
        pass

    def preprocess(self, df):
        """
        Preprocess a dataframe for a time series forecasting model.

        Note that this method has a different return type from the preprocess method of classical models.
        For time series, preprocessing consists of both resampling and preprocessing external features if any.
        Resampling involves interpolating missing time steps to match the desired sampling frequency,
        and/or extrapolating to align the start and end points of multiple timeseries.

        :param pd.Dataframe df: The dataframe to preprocess.
        :return: past_preprocessed_df, future_preprocessed_df, past_resampled_df, future_resampled_df
            - past_preprocessed_df: The resampled **and** preprocessed dataframe containing the time steps before the forecast horizon.
            - future_preprocessed_df: The resampled **and** preprocessed dataframe containing the time steps within the forecast horizon. None if no external features are used.
            - past_resampled_df: The resampled-only dataframe containing the time steps before the forecast horizon.
            - future_resampled_df: The resampled-only dataframe containing the time steps within the forecast horizon.
        """
        # time column is first converted to UTC and then timezone is removed because gluon does not support timezone
        df[self.scoring_handler.time_variable] = pd.to_datetime(df[self.scoring_handler.time_variable], utc=True).dt.tz_localize(tz=None)

        past_preprocessed_df, future_preprocessed_df, past_resampled_df, future_resampled_df = self.scoring_handler._prepare_data_for_scoring(df, self.params.schema)

        return past_preprocessed_df, future_preprocessed_df, past_resampled_df, future_resampled_df

    def get_features(self):
        raise NotImplementedError("Time series predictors do not need the getFeatures() method."
                                  " The dataframes returned by TimseriesPredictor.preprocess() already contain the feature names as column headers.")


class DeepHubPredictor(BasePredictor):

    def __init__(self, params):
        """
        :type params: ModelParams
        """

        # Explicitly not calling super init because it does unsupported actions for deephub models, we should probably
        # move those at the Predictor level in a future refactoring

        # Must do a local imports because it requires libraries not available everywhere (e.g. `torch`)
        from dataiku.doctor.deephub import builtins
        from dataiku.doctor.deephub.deephub_params import DeepHubScoringParams
        from dataiku.doctor.deephub.deephub_scoring import DeepHubScoringHandler
        from dataiku.doctor.deephub.utils.file_utils import Base64FilesReader

        builtins.load()
        self.params = params

        deephub_params = DeepHubScoringParams.build_for_predictor(params.modeling_params,
                                                                  params.preprocessing_params,
                                                                  params.core_params,
                                                                  params.model_folder_context,
                                                                  os.getcwd(),
                                                                  params.user_meta)
        deephub_params.init_deephub_context(allow_cpu_fallback=True)

        # No need to close the files_reader, as this one has no side effect
        self.files_reader = Base64FilesReader()

        self.scoring_handler = DeepHubScoringHandler(deephub_params)
        self.classes = deephub_params.target_remapping.list_categories()

    def get_proba_columns(self):
        """
        Deephub has no "probas" column at the moment. Object Detection has confidence score
        """
        return None

    def ready_explainer(self):
        """
        Deephub does not support explanations for now
        """
        pass

    def predict(self, df, **kwargs):
        pred_df = self.scoring_handler.score(df, self.files_reader,
                                             with_explanations=kwargs.get("with_explanations", False),
                                             n_explanations=kwargs.get("n_explanations", 1))

        # Whether the output dataframe is aligned with the input dataframe.
        # Not used in our api node code, but might be used in user custom code, also doubles as a reminder / semi-comment.
        pred_df._aligned = False

        return pred_df

    def _set_debug_options(self, debug_options):
        pass

    def _compute_explanations(self, df, method, n_explanations, mc_steps):
        raise ValueError("Not supported for " + self.params.core_params["prediction_type"])


class PartitionedModelPredictor(BasePredictor):

    def __init__(self, core_params, model_folder, model_type, conditional_outputs, model_part_info, preprocessing_folder_context):
        self.conditional_outputs = conditional_outputs
        self.model_type = model_type
        self.model_folder = model_folder
        self.core_params = core_params
        self._preprocessing_folder_context = preprocessing_folder_context

        self.predictors = {}
        self.any_predictor = None
        self.params = None
        self.partition = None

        self.model_part_info = model_part_info

        self._build_predictors()

    def _build_predictors(self):
        # /!\ Does not support container exec

        for partition_info in self.model_part_info.partitions_info:
            predictor = build_predictor(self.model_type, partition_info.model_folder_context,
                                        partition_info.preprocessing_folder_context,
                                        partition_info.split_folder_context,
                                        self.conditional_outputs, self.model_part_info.core_params,
                                        partition_info.split_desc, fmi=partition_info.fmi)
            self._set_predictor(partition_info.name, predictor)

    def _set_predictor(self, partition_name, predictor):
        self.predictors[partition_name] = predictor

        if self.params is None and predictor.params.preprocessing_params is not None:
            self.params = copy.deepcopy(self.predictors[partition_name].params)

        if self.any_predictor is None:
            self.any_predictor = predictor

    def get_classes(self):
        return self.any_predictor.get_classes()

    def get_proba_columns(self):
        return self.any_predictor.get_proba_columns()

    def get_conditional_output_names(self):
        return self.any_predictor.get_conditional_output_names()

    def set_partition(self, partition):
        self.partition = partition

    def _predict_partition(self, df, partition_id, with_input_cols, with_prediction, with_probas,
                           with_conditional_outputs, with_proba_percentile, with_explanations,
                           explanation_method, n_explanations, n_explanations_mc_steps):
        predictor = self.predictors.get(partition_id, None)
        if predictor is not None:
            return predictor.predict(df, with_input_cols, with_prediction, with_probas,
                                     with_conditional_outputs, with_proba_percentile, with_explanations,
                                     explanation_method, n_explanations, n_explanations_mc_steps)
        elif with_input_cols:
            return df
        else:
            return pd.DataFrame()

    def predict(self, df, with_input_cols=False, with_prediction=True, with_probas=True, with_conditional_outputs=False,
                with_proba_percentile=False, with_explanations=False, explanation_method="ICE",
                n_explanations=3, n_explanations_mc_steps=100, **kwargs):

        if self.partition is not None:
            return self._predict_partition(df, self.partition, with_input_cols, with_prediction, with_probas,
                                           with_conditional_outputs, with_proba_percentile, with_explanations,
                                           explanation_method, n_explanations, n_explanations_mc_steps)

        needs_reindex = not(isinstance(self.any_predictor, TimeseriesPredictor))

        if needs_reindex:
            origin_index = df.index.copy()

        partitioning_params = self.params.core_params["partitionedModel"]

        part_dfs = []
        for partition, part_df in df.groupby(partitioning_params["dimensionNames"], sort=False):
            partition_id = PartitionEscaper.build_partition_id(partition, partitioning_params["dimensionNames"], partitioning_params["dimensionTypes"])
            part_dfs.append(self._predict_partition(part_df, partition_id, with_input_cols, with_prediction,
                                                    with_probas, with_conditional_outputs, with_proba_percentile,
                                                    with_explanations, explanation_method, n_explanations,
                                                    n_explanations_mc_steps))

        pred_df = pd.concat(part_dfs, axis=0, sort=False)
        if needs_reindex:
            origin_index = origin_index[np.isin(origin_index, pred_df.index)]  # remove index dropped by PP
            pred_df = pred_df.reindex(origin_index)

        # Whether the output dataframe is aligned with the input dataframe.
        # Not used in our api node code, but might be used in user custom code, also doubles as a reminder / semi-comment.
        pred_df._aligned = needs_reindex

        return pred_df

    def _set_debug_options(self, debug_options):
        for predictor in self.predictors.values():
            predictor._set_debug_options(debug_options)

    def ready_explainer(self):
        if self.partition is not None:
            self._get_explainer(self.partition).make_ready()
        else:
            partitions = [(part_name, None) for part_name in self.predictors.keys()]
            partitioning_params = self.params.core_params["partitionedModel"]
            for part_name, part_df in partitions:
                partition_id = PartitionEscaper.build_partition_id(part_name, partitioning_params["dimensionNames"], partitioning_params["dimensionTypes"])
                self._get_explainer(partition_id).make_ready(part_df)
                logger.info("Explanations background for partition '{}' successfully loaded".format(partition_id))

    def _compute_explanations(self, df, method, n_explanations, mc_steps):
        if self.partition is not None:
            return self._get_predictor(self.partition)._compute_explanations(df, method, n_explanations,
                                                                             mc_steps)
        else:
            explanations_dfs = []
            partitioning_params = self.params.core_params["partitionedModel"]
            for partition, part_df in df.groupby(partitioning_params["dimensionNames"], sort=False):
                partition_id = PartitionEscaper.build_partition_id(partition, partitioning_params["dimensionNames"], partitioning_params["dimensionTypes"])
                explanations_dfs.append(self._get_predictor(partition_id)._compute_explanations(df, method,
                                                                                                n_explanations,
                                                                                                mc_steps))

        return pd.concat(explanations_dfs, axis=0)

    def _get_explainer(self, partition_id):
        return self._get_predictor(partition_id)._individual_explainer

    def _get_predictor(self, partition_id):
        """
        :rtype: Predictor
        """
        predictor = self.predictors.get(partition_id, None)
        if predictor is None:
            raise ValueError("The model for partition {} does not exist".format(partition_id))
        return predictor

    def _get_preprocessing_folder_context(self):
        return self._preprocessing_folder_context


class EnsemblePredictor(BasePredictor):
    """
    A predictor for Ensemble models.
    Unlike regular models, they do not have a preprocessing and do not have feature names
    (various models use different features and preprocessings).
    Attempted calls to preprocess, get_preprocessing and get_features will therefore raise an AttributeError
    """

    def __init__(self, params, clf):
        BasePredictor.__init__(self, params, clf)
        self.features = None # Features are unknown for ensemble model

    def get_prediction_dataframe(self, input_df, with_prediction, with_probas,
                                 with_conditional_outputs, with_proba_percentile):
        prediction_type = self.params.core_params["prediction_type"]
        if prediction_type == doctor_constants.REGRESSION:
            pred_df = pd.DataFrame({"prediction": self._clf.predict(input_df)})
        else:
            probas = self._clf.predict_proba(input_df)
            if probas.size == 0:
                return pd.DataFrame(columns=["prediction"], index=input_df.index)
            proba_columns = self.get_proba_columns()
            proba_df = pd.DataFrame(data=probas, columns=proba_columns)
            pred_df = None
            if prediction_type == doctor_constants.BINARY_CLASSIFICATION:
                threshold = self.params.user_meta.get("activeClassifierThreshold", 0.5)
                pred_raw = (probas[:, 1] > threshold).astype(int)
            else:
                pred_raw = self._clf.predict(input_df)
            if with_prediction:
                pred_df = pd.DataFrame({"prediction": pd.Series(pred_raw).map(self.params.target_map)})
            if with_probas or with_proba_percentile or with_conditional_outputs:
                pred_df = pd.concat([pred_df, proba_df], axis=1) if pred_df is not None else proba_df
            # percentiles and conditional outputs if applicable
            if with_proba_percentile and "probaPercentiles" in self.params.model_perf and self.params.model_perf["probaPercentiles"]:
                _add_proba_percentiles(pred_df, self.params.model_perf, self.params.target_map)
            if with_conditional_outputs and self.params.conditional_outputs is not None:
                for co in self.params.conditional_outputs:
                    _add_conditional_output(pred_df, co)

        if input_df.index.size != 0:
            pred_df.index = input_df.index
        return pred_df

    def predict(self, df, with_input_cols=False, with_prediction=True, with_probas=True, with_conditional_outputs=False,
                with_proba_percentile=False, with_explanations=False,
                explanation_method="ICE",
                n_explanations=3,
                n_explanations_mc_steps=100, **kwargs):
        """
        Predict a dataframe.
        The results are returned as a dataframe with prediction columns added.
        """
        df = _preformat(df, self.params)

        pred_df = self.get_prediction_dataframe(df, with_prediction, with_probas, with_conditional_outputs, with_proba_percentile)

        results = []
        if with_input_cols:
            results.append(df)
        results.append(pred_df)
        if with_explanations:
            explanations_df = self._compute_explanations(df, explanation_method,
                                                         n_explanations, n_explanations_mc_steps)
            results.append(explanations_df)

        results_df = pd.concat(results, axis=1)

        # Whether the output dataframe is aligned with the input dataframe.
        # Not used in our api node code, but might be used in user custom code, also doubles as a reminder / semi-comment.
        results_df._aligned = df.index.size != 0

        return results_df

    def _set_debug_options(self, debug_options):
        if debug_options.get("dumpPreprocessedFirstLine", False):
            logger.warn("Preprocessing advanced debugging options are not available for ensembling models")


class SavedModelVersionMetrics(object):
    """
    Handle to the metrics of a version of a saved model
    """

    def __init__(self, metrics, version_id):
        self.metrics = metrics
        self.version_id = version_id

    def get_performance_values(self):
        """
        Retrieve the metrics as a dict.

        :rtype: dict
        """
        ret = {}
        for metric_id in self.metrics.get_all_ids():
            if metric_id.startswith("model_perf:"):
                data = self.metrics.get_partition_data_for_version(metric_id, self.version_id)
                clean_id = metric_id.replace("model_perf:", "")
                ret[clean_id] = metrics.ComputedMetrics.get_value_from_data(data)
        return ret

    def get_computed(self):
        """
        Get the underlying metrics object.

        :rtype: :class:`dataiku.core.metrics.ComputedMetrics`
        """
        return self.metrics


class FinetunedLLMVersionTrainingParameters(object):
    """

    Attributes:
        saved_model: (SavedModel) saved model this version belongs to
        working_directory: (str) local filepath where the model files are stored
        config: (dict) record informative fields related to fine-tuning
            {
                "trainingDataset": (str) name of the training dataset (default: None)
                "validationDataset": (str) name of the validation dataset if any (default: None)
                "promptColumn": (str) name of the prompt column in the training dataset, for instruct models (default: None)
                "completionColumn": (str) name of the completion column in the training dataset, for instruct models (default: None)
                "userMessageColumn": (str) name of the user message column in the training dataset, for conversational models (default: None)
                "assistantMessageColumn": (str) name of the assistant message column in the training dataset, for conversational models (default: None)
                "systemMessageColumn": (str) name of the system message column in the training dataset, for conversational models (default: None)
                "staticSystemMessage": (str) static prompt to use for all rows of the training dataset, for conversational models (default: None)
                "promptHandlingMode": (str) type of prompt handling mode of the base model, must be one of ["auto", "dolly", "falcon", "gemma", "llama_2", "llama_3", "mistral", "mpt", "zephyr"] (default: "auto")
                "codeEnvName": (str) name of the code env used to fine-tune the model, informational use only, the code env used to run the model is the one from the connection (default: code env of the running recipe or notebook or None)
                "batchSize": (int) the size of each batch of samples (default: None)
                "neftuneNoiseAlpha": (float) magnitude of noise added to the embeddings during fine-tuning (default: None)
                "initialLearningRate": (float) step size in updating model weights during training (default: None)
                "nbEpochs": (int) the total number of epochs (default: None, computed from the event log if any)
                "totalSteps": (int) the number of steps (default: None, computed from the event log if any)
                "eventLog": (list) a log of events created during fine-tuning, used to extract metrics (default: None)
            }
    """
    def __init__(self, saved_model, working_directory):
        """ Fine-tuning parameters for a fine-tuned LLM version

        :param SavedModel saved_model: saved model this version belongs to
        :param str working_directory: local filepath where the model files are stored
        """
        self.saved_model = saved_model
        self.working_directory = working_directory
        self.config = {
            "codeEnvName": remoterun.get_env_var("DKU_CODE_ENV_NAME"),
            "promptHandlingMode": "auto",
        }


class DkuScoringPredictor(object):
    def __init__(self, export_id):
        self._export_id = export_id
        self._temp_dir = None
        self._model = None
        self._in_context_manager = False

    def __enter__(self):
        self._in_context_manager = True
        assert self._export_id is not None

        try:
            from dataikuscoring import load_model

            self._temp_dir = tempfile.mkdtemp()
            export_path = os.path.join(self._temp_dir, "model-python-export.zip")
            logger.info("Starting the download of the python export")
            http_response = backend_stream_call("ml/prediction/download-python-export", data={
                "exportId": self._export_id
            })
            logger.info("Download of the python export completed")

            if http_response.status == 200:
                with open(export_path, 'wb') as file:
                    file.write(http_response.data)
            else:
                raise Exception(u"The download of the python export of the model failed with status code {} with reason {}".format(http_response.status_code, http_response.reason))

            self._model = load_model(export_path)
        except Exception:
            if self._temp_dir and os.path.exists(self._temp_dir):
                shutil.rmtree(self._temp_dir)
            raise

        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self._temp_dir and os.path.exists(self._temp_dir):
            shutil.rmtree(self._temp_dir)

    def predict(self, df, with_input_cols=False, with_prediction=True, with_probas=False, **kwargs):
        """
        Predict a dataframe.
        The results are returned as a dataframe with columns corresponding to the various prediction information.

        :param df: list, or array or DataFrame containing the rows to predict
        :param with_input_cols: whether the input columns should also be present in the output
        :param with_prediction: whether the prediction column should be present
        :param with_probas: whether the probability columns should be present
        No other argument is supported for this latency-optimized predictor.
        """
        if not self._in_context_manager:
            raise Exception("The predictor optimized for latency must be used inside a context manager.")

        if kwargs:
            raise Exception("Latency-optimized predictors only support `df`, `with_input_cols`, `with_prediction` and `with_probas` arguments. Unsupported arguments: " + str(kwargs))

        results = []

        if with_input_cols:
            results.append(df)

        if with_probas:
            from dataikuscoring.models.common import ProbabilisticModelMixin
            if isinstance(self._model, ProbabilisticModelMixin):
                probas = {"proba_" + key: value for (key, value) in self._model.predict_proba(df).items()}
                results.append(pd.DataFrame(probas))
            else:
                raise ValueError("Prediction a non-probabilistic model with with_probas=True.")
        elif not with_prediction:
            raise ValueError("Cannot predict with neither probabilities nor predictions.")

        if with_prediction:
            results.append(pd.DataFrame({"prediction": self._model.predict(df)}))

        results_df = pd.concat(results, axis=1)

        # Whether the output dataframe is aligned with the input dataframe.
        # Not used in our api node code, but might be used in user custom code, also doubles as a reminder / semi-comment.
        results_df._aligned = True

        return results_df
