import dataiku
from dataiku.runnables import Runnable, ResultTable
from dataikuapi.dss.project import DSSProject
from dataikuapi.dss.app import DSSApp
from dataikuapi.dss.workspace import DSSWorkspace
from dataikuapi.dss.admin import DSSCodeEnv

class MyRunnable(Runnable):
    def __init__(self, project_key, config, plugin_config):
        self.config = config
        self.client = dataiku.api_client()

    def get_progress_target(self):
        return (100, 'NONE')

    def delete_object(self, dss_object):
        if dss_object["objectType"] in ['project', 'application instance']:
            dss_project = DSSProject(self.client, dss_object["objectParams"]["objectId"])
            dss_project.delete()
        elif dss_object["objectType"] == 'workspace':
            dss_workspace = DSSWorkspace(self.client, dss_object["objectParams"]["objectId"])
            dss_workspace.delete()
        elif dss_object["objectType"] == 'code environment':
            dss_code_env = DSSCodeEnv(self.client, dss_object["objectParams"]['envLang'], dss_object["objectParams"]['envName'])
            dss_code_env.delete()
        else:
            raise ValueError("The deletion of an object of type '" + str(dss_object["objectType"]) + "' is not supported")

    def run(self, progress_callback):
        perform_deletion = self.config.get("performDeletion", False)
        to_delete = []
        expired_users_logins = [user["login"] for user in self.client.list_expired_trial_users() if user["userProfile"] == "NONE"]
        projects = []
        applications_instances = []
        workspaces = []
        code_envs = []
        if len(expired_users_logins) > 0:
            if self.config.get("clearProjects", False):
                projects = self.client.list_projects()
            elif self.config.get("clearAppInstances", False):
                # when you delete projects, you automatically delete the application instances as they are projects
                for app in self.client.list_apps():
                    applications_instances.extend(DSSApp(self.client, app["appId"]).list_instances())
            if self.config.get("clearWorkspaces", False):
                workspaces = self.client.list_workspaces()
            if self.config.get("clearCodeEnvs", False):
                code_envs = self.client.list_code_envs()

        for user_login in expired_users_logins:
            if (self.config.get("clearProjects", False)):
                for project in projects:
                    if project["ownerLogin"] == user_login:
                        to_delete.append({'objectType': 'project', 'objectParams': {'objectId': project["projectKey"], 'owner': user_login}})

            elif self.config.get("clearAppInstances", False):
                for instance in applications_instances:
                    if instance["ownerLogin"] == user_login:
                        to_delete.append({'objectType': 'application instance', 'objectParams': {'objectId': instance["projectKey"], 'owner': user_login}})

            if (self.config.get("clearWorkspaces", False)):
                for workspace in workspaces:
                    admin_permissions = [perm for perm in workspace["permissions"] if perm["admin"]]
                    if (len(admin_permissions)==1 and admin_permissions[0]["user"] == user_login):
                        to_delete.append({'objectType': 'workspace', 'objectParams': {'objectId': workspace["workspaceKey"], 'owner': user_login}})

            if (self.config.get("clearCodeEnvs", False)):
                for code_env in code_envs:
                    if code_env["owner"] == user_login:
                        to_delete.append({'objectType': 'code environment', 'objectParams': {'envLang': code_env['envLang'], 'owner': user_login, 'envName': code_env['envName']}})

        if perform_deletion:
            for i, dss_object in enumerate(to_delete):
                self.delete_object(dss_object)
                progress_callback(100*(i+1)/len(to_delete))


        rt = ResultTable()
        if perform_deletion:
            rt.set_name("Deleted objects")
        else:
            rt.set_name("Would be deleted")

        rt.add_column("objectType", "Object type", "STRING")
        rt.add_column("objectId", "Object ID", "STRING")
        rt.add_column("owner", "Owner", "STRING")
        rt.add_column("envLang", "Environment language (if applicable)", "STRING")

        for elt in to_delete:
            if elt['objectType']=="code environment":
                rt.add_record([elt['objectType'], elt['objectParams']['envName'], elt['objectParams']['owner'], elt['objectParams']['envLang']])
            else:
                rt.add_record([elt['objectType'], elt['objectParams']['objectId'], elt['objectParams']['owner']])
        return rt
