#' Copies the whole content of a DSS folder to a local path, recursively
#'
#' @param folder_id the id of the DSS folder
#' @param local_base_path the local path (directory) to download to. If this path does not exist, it is created. 
#'
#' Files that are both in the source and destination are overwritten.
#' Files that exist in the destination but not in the DSS folder are preserved.
#' Changes from directories to files or vice-versa are not supported 
#'
#' @export
dkuManagedFolderCopyToLocal <- function(folder_id, local_base_path) {
    if (!dir.exists(local_base_path)) {
        dir.create(local_base_path, recursive=TRUE)
    }

    folder_paths <- dkuManagedFolderPartitionPaths(folder_id)

    for (folder_path in folder_paths) {
        parent_dir=dirname(paste0(local_base_path, folder_path))
        if (!dir.exists(parent_dir)) {
            dir.create(parent_dir, recursive = TRUE)
        }
        local_path = paste0(local_base_path, folder_path)
        local_file = file(local_path, "wb")
        print(paste("Copying", folder_path, "to", local_path))
        data = dkuManagedFolderDownloadPath(folder_id, folder_path, as="raw")
        writeBin(data, local_file)
        close(local_file)
    }
    print("Done copying")
}


#' Copies the whole content of a local folder to a DSS folder, recursively
#'
#' @param folder_id the id of the DSS folder
#' @param source_base_path the local path (directory) to copy from. This must exist and be a directory
#'
#' Files that are both in the source and destination are overwritten.
#' Files that already exist in the destination DSS folder but not in the source local path are preserved
#'
#' @export
dkuManagedFolderCopyFromLocal <- function(folder_id, source_base_path) {
	local_paths <- list.files(source_base_path, recursive=TRUE)

	for (local_path in local_paths) {
		print(paste("Uploading", local_path))
		complete_path <- paste0(source_base_path, "/", local_path)
		local_file <- file(complete_path, "rb")
		dkuManagedFolderUploadPath(folder_id, local_path, local_file)
	}
}

#' Gets details about a path of a managed folder
#' @param folder_id the id of the DSS folder
#' @param path the path within the folder
#'
#' Returns a list of various information about the path
#'
#' @export
dkuManagedFolderPathDetails <- function(folder_id, path) {
    getDetailsURL = dku_intercom__get_jek_or_backend_url("/managed-folders/get-path-details")
    ref = dku__resolve_smart_name(folder_id)

    resp = POST(getDetailsURL,body = list(projectKey=dku__ref_to_pkey(ref), lookup=dku__ref_to_name(ref), path=path),
        encode="form",  dku__get_auth_headers(), dku_intercom__get_httr_config())

    dku__check_api_error(resp, "Failed to get Folder information")
    content(resp)
}