#' Writes a data.frame to Dataiku's Data Science Studio
#'
#' @param df The data.frame to write to Data Science Studio
#' @param name The name of the dataset in data science studio to write to
#' @param partition The name of the partition to write to. Optional
#' @param schema Whether to overwrite the schema of the target dataset
#' @param convertLists Whether to convert list-columns to JSON
#' @return None
#'
#' @details
#'
#' This function writes a dataset to Dataiku's Data Science Studio.
#'
#' Please see Dataiku's online documentation for
#' a description of how to use partitions in combination with R.
#'
#' @importFrom RJSONIO toJSON
#' @importFrom RJSONIO fromJSON
#'
#' @examples
#' \dontrun{
#' # write the iris dataset to Data Science Studio
#' dkuWriteDataset(iris, "irisDataiku")
#'
#' # do not copy schema to Data Science Studio
#' dkuWriteDataset(iris, "irisDataiku", schema=F)
#'
#' # write dataset to a partition of the 'counts' dataset
#' df = data.frame(x=c(1,2,3,4))
#' dkuWriteDataset(df, "counts", partition="2015-01-01")
#' }
#' @export

dkuWriteDataset <- function (df, name, partition="", schema=TRUE, convertLists=TRUE) {
  if (dku_remoterun__get_env_var("RSTUDIO") == "1" && dku_remoterun__get_env_var("DKU_IS_CODE_STUDIO") != "1") {
    warning("Writing datasets is not supported in RStudio mode")
    warning(paste("Ignoring dkuWriteDataset for ", name))
    return()
  }

  if(!is.data.frame(df)) {
    stop("The first argument of dkuWriteDataset must be a data.frame")
  }

  # coerce to data.frame
  # fixes problems with data.tables and grouped data.frames
  df = as.data.frame(df)

  dipHome = dku_remoterun__get_env_var("DIP_HOME")
  if(is.null(dipHome) || dipHome=="") {
    stop("DIP_HOME isn't defined")
  }

  appendMode <- FALSE
  parsedRSpec <- getDkuFlowSpec()
  if (is.null(parsedRSpec)) {
    # Notebook mode
    name <- dku__resolve_smart_name(name)
    tmpFilePath <- tempfile(pattern='r-data-')
    currentActivityId <- ""
    # Need to stream for MUS compatibility
    streamFile <- TRUE
  } else {
    # Flow mode
    sharedTmpDir <- parsedRSpec$tmpDir
    if (is.na(file.info(sharedTmpDir)$isdir) || !file.info(sharedTmpDir)$isdir) {
      # E.g. executing in Docker container
      name <- dku__resolve_smart_name(name)
      tmpFilePath <- tempfile(pattern='r-data-')
      streamFile <- TRUE
    } else {
      tmpFilePath <- tempfile(tmpdir=sharedTmpDir)
      streamFile <- FALSE
    }
    if (partition!="") {
      stop("You cannot explicitly set partitions when running within Dataiku Flow")
    }
    found = FALSE
    for(out in parsedRSpec[["out"]]) {
      if(out[["fullName"]] == name || out[["smartName"]] == name) {
        if(!("partition" %in% names(out))) {
          partition = ""
        } else if(is.null(out[["partition"]])) {
          partition = ""
        } else {
          partition = out[["partition"]]
        }
        name = out[["fullName"]]
        if(!(is.null(out[["appendMode"]]))) {
            appendMode <- out[["appendMode"]]
        }
        found = TRUE
      }
    }
    if(!found) {
      stop(paste0("The dataset ",name," is not declared as output"))
    }
    currentActivityId <- parsedRSpec$currentActivityId
  }
  initSessionUrl = dku_intercom__get_jek_or_backend_url("/datasets/init-write-session/")
  waitSessionUrl = dku_intercom__get_jek_or_backend_url("/datasets/wait-write-session/")

  if(convertLists){
    listCols = sapply(df, function(c) {return (is.list(c) && !is(c,"POSIXct") && !is(c,"POSIXlt"))})
    if(any(listCols)){
      cat("Attempting to coerce the following columns to JSON: ")
      cat(paste(colnames(df)[listCols]), sep=", ")
      cat("\n")
      df[listCols] = lapply(df[listCols], RJSONIO::toJSON)
    }
  }

  # Write schema
  if(schema) dkuWriteDatasetSchema(df,name)

  # Write a temp file
  file.create(tmpFilePath)
  tmpFileConn <- file(tmpFilePath, "wb")
  cat("Start writing table to file ...\n")
  write.table(df, tmpFileConn, sep=",", qmethod="double", row.names=FALSE, col.names=FALSE,na="")
  cat("Done writing table to file ...\n")
  close(tmpFileConn)

  # Prepare write request
  writeRequest <- {}
  writeRequest$format <- "CSV"
  writeRequest$dataSchema <- dataframe_to_dss_schema(df)
  writeRequest$partitionSpec <- partition
  writeRequest$fullDatasetName <- name
  writeRequest$activityId <- currentActivityId
  if (streamFile) {
    writeRequest$method <- 'STREAM'
  } else {
    writeRequest$method <- 'FILE'
    writeRequest$dataFilePath <- tmpFilePath
  }
  if (appendMode) {
    writeRequest$writeMode <- 'APPEND'
  } else {
    writeRequest$writeMode <- 'OVERWRITE'
  }
  jsonWriteRequest = RJSONIO::toJSON(writeRequest)

  # Post the request to init writing the dataset
  init_resp = POST(initSessionUrl,body = list(request=jsonWriteRequest),
                   encode="form", dku__get_auth_headers(), dku_intercom__get_httr_config())
  if(init_resp$status != 200) {
    parsed_init_resp = content(init_resp)
    stop(paste0("Unable to initialize write session : ",parsed_init_resp$message))
  }

  parsed_init_resp = content(init_resp)
  session_id = parsed_init_resp$id

  if (streamFile) {
    # Actually write the data
    push_resp = POST(paste0(dku_intercom__get_jek_or_backend_url("/datasets/push-data/"),'?id=',session_id),
      body=upload_file(tmpFilePath), dku__get_auth_headers(), dku_intercom__get_httr_config())
  }

  # Wait until the writing session ends
  wait_resp = GET(paste0(waitSessionUrl,'?id=',session_id), dku__get_auth_headers(), dku_intercom__get_httr_config())

  # Drop the temp file
  unlink(tmpFilePath)
  if(wait_resp$status != 200) {
    stop("Unable to get write session status")
  }

  # Parse response
  parsed_wait_resp = content(wait_resp)
  if (!parsed_wait_resp$ok) {
    stop(paste("Unable to write dataset ",name,":",parsed_wait_resp$message))
  } else {
    cat(paste(parsed_wait_resp$writtenRows,"rows successfully written into ",name, "\n"))
  }
}
