(function(){
'use strict';
/**
 * Services for building computables
 */

var app = angular.module('dataiku.common.build', []);

app.service('PartitionSelection', ['LocalStorage', '$stateParams', 'Logger', function(LocalStorage, $stateParams, Logger) {
    function get() {
        if ($stateParams.projectKey) {
            var stored = LocalStorage.get($stateParams.projectKey + '.partitionSettings') || {};
            Logger.info("reading partitionSettings", stored);
            return stored;
        }
        throw new Error("projectKey not defined");
    }
    function set(partitionSettings) {
        //TODO cleanup mechanism to avoid growing partitionSettings too much
        Logger.info("saving partitionSettings", partitionSettings);
        LocalStorage.set($stateParams.projectKey + '.partitionSettings', partitionSettings);
    }
    function save(dimName, value) {
        var partitionSettings = get();
        if (value != null && (value.start !== undefined || value.end !== undefined || value.explicit !== undefined)) {
            var startKey = dimName+"_start_"+value.format;
            partitionSettings[startKey] = value.start;
            var endKey = dimName+"_end_"+value.format;
            partitionSettings[endKey] = value.end;
            var explicitKey = dimName+"_explicit_"+value.format;
            partitionSettings[explicitKey] = value.explicit;
            var useExplicitKey = dimName+"_useExplicit_"+value.format;
            partitionSettings[useExplicitKey] = value.useExplicit;
        } else {
            partitionSettings[dimName] = value;
        }
        set(partitionSettings);
    }
    function getTimeFormat(dimType, params) {
        var format;
        if (dimType == 'time' && params.period != 'YEAR') {
            format = 'YYYY';
            if(params.period == 'MONTH'){
                format = 'YYYY-MM';
            } else if(params.period == 'DAY'){
                format = 'YYYY-MM-DD';
            } else if(params.period == 'HOUR'){
                format = 'YYYY-MM-DD-HH';
            }
        }
        return format;
    }

    //Legacy
    function getPartitionsFromCookie(dimName, timeFormat) {
        var dimName2 = 'partition_' + dimName;
        var ret, found;
        if (timeFormat) {
            var start = getCookie(dimName2 + "_start_" + timeFormat);
            var end = getCookie(dimName2 + "_end_" + timeFormat);
            var explicit = getCookie(dimName2 + "_explicit" + timeFormat);
            var useExplicit = getCookie(dimName2 + "_useExplicit" + timeFormat);
            ret = {
                'start': start || moment().format(timeFormat),
                'end': end || moment().format(timeFormat),
                'explicit': explicit || moment().format(timeFormat),
                'useExplicit': useExplicit || false,
                'format': timeFormat
            };
            found = start !== undefined || end !== undefined;
        } else {
            ret = getCookie(dimName2) || undefined;;
            found = ret !== undefined;
        }
        return found ? ret : undefined;
    }

    function getPartitions(dimName, timeFormat) {
        var partitionSettings = get();
        var ret, found;
        if (timeFormat) {
            var start = partitionSettings[dimName + "_start_" + timeFormat];
            var end = partitionSettings[dimName + "_end_" + timeFormat];
            var explicit = partitionSettings[dimName + "_explicit_" + timeFormat];
            var useExplicit = partitionSettings[dimName + "_useExplicit_" + timeFormat];
            ret = {
                'start': start || moment().format(timeFormat),
                'end': end || moment().format(timeFormat),
                'explicit': explicit || moment().format(timeFormat),
                'useExplicit': useExplicit || false,
                'format': timeFormat
            };
            found = start !== undefined || end !== undefined;
        } else {
            ret = partitionSettings[dimName];
            found = ret !== undefined;
        }
        if (!found) {
            var fromCookie = getPartitionsFromCookie(dimName, timeFormat);
            if (fromCookie !== undefined) {
                ret = fromCookie;
                save(dimName, fromCookie);
            }
        }
        return ret;
    }

    return {
        getBuildPartitions: function(partitioning) {
            const buildPartitions = {};
            if (partitioning) {
                angular.forEach(partitioning.dimensions, function(dim) {
                    const timeFormat = getTimeFormat(dim.type, dim.params);
                    buildPartitions[dim.name] = getPartitions(dim.name, timeFormat);
                });
            }
            return buildPartitions;
        },
        saveBuildPartitions: function(partitioning, buildPartitions) {
            if (partitioning) {
                angular.forEach(partitioning.dimensions, function(dim) {
                    save(dim.name, buildPartitions[dim.name]);
                });
            }
        },
    }
}]);


app.service("JobDefinitionComputer", function(AnyLoc, $stateParams){
    var svc = {
        computeTargetPartition : function(partitioning, buildPartitions = {}) {
            if (partitioning && partitioning.dimensions.length > 0) {
                var dimensionValuesList = []; // will do a cartesian product of these
                angular.forEach(partitioning.dimensions, function(dimension, index){
                    var bp = buildPartitions[dimension.name];
                    if (!bp) {
                        return;
                    }
                    if (dimension.type == 'time') {
                         if (dimension.params.period == "YEAR") {
                             dimensionValuesList.push([bp]);
                         } else if (bp.useExplicit) {
                             if (bp.explicit.indexOf(',') >= 0) {
                                 dimensionValuesList.push(bp.explicit.split(','))
                             } else {
                                 dimensionValuesList.push([bp.explicit]);
                             }
                         } else {
                             if(bp.start != bp.end) {
                                 dimensionValuesList.push([bp.start + '/' + bp.end]);
                             } else {
                                 dimensionValuesList.push([bp.start]);
                             }
                         }
                    } else {
                        dimensionValuesList.push([bp]);
                    }
                });
                var partitionIds = [''];
                dimensionValuesList.forEach(function(dimensionValues) {
                    var newPartitionIds = [];
                    dimensionValues.forEach(function(dimensionValue) {
                        partitionIds.forEach(function(partitionId) {
                            newPartitionIds.push(partitionId + (partitionId ? '|' : '') + dimensionValue);
                        });
                    });
                    Array.prototype.splice.apply(partitionIds, [0, partitionIds.length].concat(newPartitionIds));
                });
                return partitionIds.join(',');
            } else {
                return null;
            }
        },

        /* Build output parts of job definitions */

        computeOutputForDataset : function(dataset, buildPartitions) {
            var output =  {
                targetDataset : dataset.name,
                targetDatasetProjectKey : dataset.projectKey,
                type : 'DATASET',
                targetPartition : svc.computeTargetPartition(dataset.partitioning, buildPartitions)
            };
            return output;
        },

        computeOutputForSavedModel : function(model, buildPartitions) {
            return {
                targetDataset : model.id,
                targetDatasetProjectKey : model.projectKey,
                type : 'SAVED_MODEL',
                targetPartition : svc.computeTargetPartition(model.partitioning, buildPartitions)
            };
        },

        computeOutputForBox : function(box, buildPartitions) {
           return {
               targetDataset : box.id,
               targetDatasetProjectKey : box.projectKey,
               type : 'MANAGED_FOLDER',
               targetPartition : svc.computeTargetPartition(box.partitioning, buildPartitions)
           };
        },

        computeOutputForModelEvaluationStore : function(store, buildPartitions) {
           return {
               targetDataset : store.id,
               targetDatasetProjectKey : store.projectKey,
               type : 'MODEL_EVALUATION_STORE',
               targetPartition : svc.computeTargetPartition(store.partitioning, buildPartitions)
           };
        },

        computeOutputForRetrievableKnowledge : function(retrievableKnowledge, buildPartitions) {
           return {
               targetDataset : retrievableKnowledge.id,
               targetDatasetProjectKey : retrievableKnowledge.projectKey,
               type : 'RETRIEVABLE_KNOWLEDGE',
               targetPartition : svc.computeTargetPartition(retrievableKnowledge.partitioning, buildPartitions)
           };
        },

        computeOutputForStreamingEndpoint : function(projectKey, streamingEndpointId) {
            return {
                targetDataset : streamingEndpointId,
                targetDatasetProjectKey : projectKey,
                type : 'STREAMING_ENDPOINT'
            };
        },


        /* Build job definitions */

       computeJobDefForSingleDataset: function(projectKey, mode, dataset, buildPartitions, triggeredFrom, recipe, options){
           var svc = this;
           var jd = {
               type: mode,
               refreshHiveMetastore: true,
               projectKey: projectKey,
               outputs: [svc.computeOutputForDataset(dataset, buildPartitions)]
           }
           if (triggeredFrom) {
               jd.triggeredFrom = triggeredFrom;
           }
           if (recipe) {
               jd.recipe = recipe;
           }
           if (options) {
                jd.autoUpdateSchemaBeforeEachRecipeRun = options.autoUpdateSchemaBeforeEachRecipeRun;
                jd.stopAtFlowZoneBoundary = options.stopAtFlowZoneBoundary;
           }
           return jd;
       },

        computeJobDefForSavedModel : function(projectKey, mode, model, buildPartitions, triggeredFrom, recipe, options) {
            var jd = {};
            jd.type = mode;
            jd.refreshHiveMetastore = true;
            jd.projectKey = projectKey;
            jd.outputs = [svc.computeOutputForSavedModel(model, buildPartitions)];
            if (recipe) {
                jd.recipe = recipe;
            }
            if (triggeredFrom) {
                jd.triggeredFrom = triggeredFrom;
            }
            if (options) {
              jd.autoUpdateSchemaBeforeEachRecipeRun = options.autoUpdateSchemaBeforeEachRecipeRun;
              jd.stopAtFlowZoneBoundary = options.stopAtFlowZoneBoundary;
            }
            return jd;
        },


       computeJobDefForBox : function(projectKey, mode, box, buildPartitions, triggeredFrom, recipe, options) {
           var jd = {};
           jd.type = mode;
           jd.refreshHiveMetastore = true;
           jd.projectKey = projectKey;
           jd.outputs = [svc.computeOutputForBox(box, buildPartitions)];
           if (recipe) {
               jd.recipe = recipe;
           }
           if (triggeredFrom) {
                jd.triggeredFrom = triggeredFrom;
           }
           if (options) {
                jd.autoUpdateSchemaBeforeEachRecipeRun = options.autoUpdateSchemaBeforeEachRecipeRun;
                jd.stopAtFlowZoneBoundary = options.stopAtFlowZoneBoundary;
           }
           return jd;
        },
       computeJobDefForModelEvaluationStore : function(projectKey, mode, store, buildPartitions, triggeredFrom, recipe, options) {
           var jd = {};
           jd.type = mode;
           jd.refreshHiveMetastore = true;
           jd.projectKey = projectKey;
           jd.outputs = [svc.computeOutputForModelEvaluationStore(store, buildPartitions)];
           if (recipe) {
               jd.recipe = recipe;
           }
           if (triggeredFrom) {
                jd.triggeredFrom = triggeredFrom;
           }
           if (options) {
                jd.autoUpdateSchemaBeforeEachRecipeRun = options.autoUpdateSchemaBeforeEachRecipeRun;
                jd.stopAtFlowZoneBoundary = options.stopAtFlowZoneBoundary;
           }
           return jd;
        },
        computeJobDefForRetrievableKnowledge : function(projectKey, mode, retrievableKnowledge, buildPartitions, triggeredFrom, recipe, options) {
           var jd = {};
           jd.type = mode;
           jd.refreshHiveMetastore = true;
           jd.projectKey = projectKey;
           jd.outputs = [svc.computeOutputForRetrievableKnowledge(retrievableKnowledge, buildPartitions)];
           if (recipe) {
               jd.recipe = recipe;
           }
           if (triggeredFrom) {
                jd.triggeredFrom = triggeredFrom;
           }
           if (options) {
                jd.autoUpdateSchemaBeforeEachRecipeRun = options.autoUpdateSchemaBeforeEachRecipeRun;
                jd.stopAtFlowZoneBoundary = options.stopAtFlowZoneBoundary;
           }
           return jd;
        },

        computeJobDefForStreamingEndpoint : function(projectKey, mode, streamingEndpoint, buildPartitions, triggeredFrom, recipe, options) {
           var jd = {};
           jd.type = mode;
           jd.refreshHiveMetastore = true;
           jd.projectKey = projectKey;
           jd.outputs = [svc.computeOutputForStreamingEndpoint(projectKey, streamingEndpoint.id)];
           if (recipe) {
               jd.recipe = recipe;
           }
           if (triggeredFrom) {
               jd.triggeredFrom = triggeredFrom;
           }
          if (options) {
                jd.autoUpdateSchemaBeforeEachRecipeRun = options.autoUpdateSchemaBeforeEachRecipeRun;
                jd.stopAtFlowZoneBoundary = options.stopAtFlowZoneBoundary;
           }
           return jd;
        },

        computeJobDefForComputable: function(projectKey, mode, computable, buildPartitions, triggeredFrom, recipe, options) {
            let output = null;
            if (computable.type == "SAVED_MODEL") {
                output = svc.computeOutputForSavedModel(computable.model, buildPartitions);
            } else if (computable.type == "MANAGED_FOLDER") {
                output = svc.computeOutputForBox(computable.box, buildPartitions);
            } else if (computable.type == "STREAMING_ENDPOINT") {
                output = svc.computeOutputForStreamingEndpoint(projectKey, computable.streamingEndpoint.id);
            } else if (computable.type == "MODEL_EVALUATION_STORE") {
                output = svc.computeOutputForModelEvaluationStore(computable.mes, buildPartitions);
            } else if (computable.type == "RETRIEVABLE_KNOWLEDGE") {
                output = svc.computeOutputForRetrievableKnowledge(computable.retrievableKnowledge, buildPartitions);
            } else {
                output = svc.computeOutputForDataset(computable.dataset, buildPartitions);
            }

            var jd = {
                type: mode,
                refreshHiveMetastore: true,
                projectKey: projectKey,
                outputs: [output]
            };
            if (triggeredFrom) {
                jd.triggeredFrom = triggeredFrom;
            }
            if (options) {
                jd.autoUpdateSchemaBeforeEachRecipeRun = options.autoUpdateSchemaBeforeEachRecipeRun;
                jd.stopAtFlowZoneBoundary = options.stopAtFlowZoneBoundary;
            }
            return jd;
        },

        computeJobDefForComputables: function(projectKey, mode, computables, buildPartitions, triggeredFrom, recipe, options) {
            let outputs = computables.map(computable => {
                if (computable.type == "SAVED_MODEL") {
                    return svc.computeOutputForSavedModel(computable.model, buildPartitions);
                } else if (computable.type == "MANAGED_FOLDER") {
                    return svc.computeOutputForBox(computable.box, buildPartitions);
                } else if (computable.type == "STREAMING_ENDPOINT") {
                    return svc.computeOutputForStreamingEndpoint(projectKey, computable.streamingEndpoint.id);
                } else if (computable.type == "MODEL_EVALUATION_STORE") {
                    return svc.computeOutputForModelEvaluationStore(computable.mes, buildPartitions);
                } else if (computable.type == "RETRIEVABLE_KNOWLEDGE") {
                    return svc.computeOutputForRetrievableKnowledge(computable.retrievableKnowledge, buildPartitions);
                } else {
                    return svc.computeOutputForDataset(computable.dataset, buildPartitions);
                }
            });


            var jd = {};
            jd.type = mode;
            jd.refreshHiveMetastore = true;
            jd.projectKey = projectKey;
            jd.outputs = outputs;
            if (recipe) {
                jd.recipe = recipe;
            }
            if (triggeredFrom) {
                jd.triggeredFrom = triggeredFrom;
            }
            if (options) {
                jd.autoUpdateSchemaBeforeEachRecipeRun = options.autoUpdateSchemaBeforeEachRecipeRun;
                jd.stopAtFlowZoneBoundary = options.stopAtFlowZoneBoundary;
            }
            return jd;
         }
    }
    return svc;
});


app.controller('BuildSingleFlowComputableController', function($scope, $state, $stateParams, DataikuAPI, Logger,
                                    PartitionSelection, JobDefinitionComputer, $q, CreateModalFromTemplate, Dialogs, WT1, translate) {
    $scope.uiState = {
        topLevelMode: "BUILD_SINGLE_COMPUTABLE",
        recursiveUpstreamBuildMode: "RECURSIVE_BUILD",
        recursiveDownstreamBuildMode: "REVERSE_BUILD",
        updateOutputSchemasPerMode: {
            "BUILD_SINGLE_COMPUTABLE": false,
            "RECURSIVE_UPSTREAM": false,
            // For downstream, it's better to default to propagate
            "RECURSIVE_DOWNSTREAM": true,
        },
        advancedMode: false
    };

    // $scope.computeMode = undefined;
    $scope.buildPartitions = {};
    $scope.jobOptions = {
        autoUpdateSchemaBeforeEachRecipeRun: $scope.uiState.updateOutputSchemasPerMode[$scope.uiState.topLevelMode],
        stopAtFlowZoneBoundary: false
    }

    const WT1BuildEvents = {
        Start: 'flow-build-computable-start',
        StartFailed : 'flow-build-computable-start-failed',
        Preview: 'flow-build-computable-preview',
        Next: 'flow-build-computable-find-outputs-next'
    }

    // Watch settings changes to update UI and default parameters, based on target type and both levels of build mode
    $scope.$watch("uiState", function (nv) {
        if (!nv) return;
        $scope.noNeedForOptions = ($scope.uiState.topLevelMode == 'BUILD_SINGLE_COMPUTABLE' && 
        (($scope.partitioning && $scope.partitioning.dimensions.length > 0) ||
        $scope.targetType === 'model'));
        $scope.jobOptions.autoUpdateSchemaBeforeEachRecipeRun = $scope.uiState.updateOutputSchemasPerMode[$scope.uiState.topLevelMode]
        switch (nv.topLevelMode) {
            case "BUILD_SINGLE_COMPUTABLE":
                $scope.computableBuildModeExplanation = translate(`FLOW.BUILD_SINGLE.BUILD_MODE_EXPLANATION.BUILD_SINGLE_COMPUTABLE.${$scope.computable.type}`, 
                    "Build only this {{targetType}} by running the parent recipe.", { targetType : $scope.targetType});  //Note - targetType is only used to parameterise the *default* expression in english here
                switch ($scope.computable.type) {
                    case "DATASET":
                    case "MANAGED_FOLDER":
                    case "RETRIEVABLE_KNOWLEDGE":
                        $scope.buildModelAnimationData = "/static/dataiku/images/flow/build-computable-single-computable.json";
                        break;
                    case "SAVED_MODEL":
                        // override default string as it is "train" not "build"
                        $scope.computableBuildModeExplanation = translate("FLOW.BUILD_SINGLE.BUILD_MODE_EXPLANATION.BUILD_SINGLE_COMPUTABLE.SAVED_MODEL", 
                            "Train only this saved model by running the parent recipe.");
                    case "MODEL_EVALUATION_STORE":
                        $scope.buildModelAnimationData = "/static/dataiku/images/flow/build-ml-computable-single-computable.json";
                        break;
                }
                break;
            case "RECURSIVE_UPSTREAM":
                switch (nv.recursiveUpstreamBuildMode) {
                    case "RECURSIVE_BUILD":
                        $scope.computableBuildModeExplanation = translate(`FLOW.BUILD_SINGLE.BUILD_MODE_EXPLANATION.RECURSIVE_UPSTREAM.RECURSIVE_BUILD.${$scope.computable.type}`,
                           "Build out-of-date or modified upstream dependencies (e.g. predecessors, recipes) and this {{targetType}}. This option significantly reduces the time required for building.",
                           { targetType : $scope.targetType});  //Note - targetType is only used to parameterise the *default* expression in english here
                        switch ($scope.computable.type) {
                            case "DATASET":
                            case "MANAGED_FOLDER":
                            case "RETRIEVABLE_KNOWLEDGE":
                                $scope.buildModelAnimationData = "/static/dataiku/images/flow/build-computable-recursive-upstream.json";
                                break;
                            case "SAVED_MODEL":
                            case "MODEL_EVALUATION_STORE":
                                $scope.buildModelAnimationData = "/static/dataiku/images/flow/build-ml-computable-recursive-upstream.json";
                                break;
                        }
                        break;
                    case "RECURSIVE_FORCED_BUILD":
                        $scope.computableBuildModeExplanation = translate(`FLOW.BUILD_SINGLE.BUILD_MODE_EXPLANATION.RECURSIVE_UPSTREAM.RECURSIVE_FORCED_BUILD.${$scope.computable.type}`,
                            "Rebuild all datasets and run all recipes that lead to the selected {{targetType}}.", //Note - targetType is only used to parameterise the *default* expression in english here
                            { targetType : $scope.targetType});
                        switch ($scope.computable.type) {
                            case "DATASET":
                            case "MANAGED_FOLDER":
                            case "RETRIEVABLE_KNOWLEDGE":
                                $scope.buildModelAnimationData = "/static/dataiku/images/flow/build-computable-recursive-forced-upstream.json";
                                break;
                            case "SAVED_MODEL":
                            case "MODEL_EVALUATION_STORE":
                                $scope.buildModelAnimationData = "/static/dataiku/images/flow/build-ml-computable-recursive-forced-upstream.json";
                                break;
                        }
                        break;
                }
                break;
            case "RECURSIVE_DOWNSTREAM":
                switch (nv.recursiveDownstreamBuildMode) {
                    case "REVERSE_BUILD":
                        $scope.computableBuildModeExplanation = translate(`FLOW.BUILD_SINGLE.BUILD_MODE_EXPLANATION.RECURSIVE_DOWNSTREAM.REVERSE_BUILD.${$scope.computable.type}`,
                            "From this {{targetType}}, run all recipes and build all datasets from left to right until the end of the Flow.",  //Note - targetType is only used to parameterise the *default* expression in english here
                            { targetType : $scope.targetType});
                        switch ($scope.computable.type) {
                            case "DATASET":
                            case "MANAGED_FOLDER":
                            case "RETRIEVABLE_KNOWLEDGE":
                                $scope.buildModelAnimationData = "/static/dataiku/images/flow/build-computable-reverse.json";
                                break;
                            case "SAVED_MODEL":
                            case "MODEL_EVALUATION_STORE":
                                $scope.buildModelAnimationData = "/static/dataiku/images/flow/build-ml-computable-reverse.json";
                                break;
                        }
                        break;
                    case "FIND_OUTPUTS":
                        $scope.computableBuildModeExplanation = translate(`FLOW.BUILD_SINGLE.BUILD_MODE_EXPLANATION.RECURSIVE_DOWNSTREAM.FIND_OUTPUTS.${$scope.computable.type}`,
                            "Find Flow final outputs reachable from here and build them recursively. This process may build branches of the Flow that are not directly linked to this {{targetType}}.", //Note - targetType is only used to parameterise the *default* expression in english here
                            { targetType : $scope.targetType});
                        switch ($scope.computable.type) {
                            case "DATASET":
                            case "MANAGED_FOLDER":
                            case "RETRIEVABLE_KNOWLEDGE":
                                $scope.buildModelAnimationData = "/static/dataiku/images/flow/build-computable-find-outputs.json";
                                break;
                            case "SAVED_MODEL":
                            case "MODEL_EVALUATION_STORE":
                                $scope.buildModelAnimationData = "/static/dataiku/images/flow/build-ml-computable-find-outputs.json";
                                break;
                        }
                        break;
                }
                break;
        }
    }, true);

    /**
     * @typedef {Object} BuildModalOptions
     * @property {boolean=} [upstreamBuildable=true] - Modal option to display upstream build options
     * @property {boolean=} [downstreamBuildable=true] - Modal option to display downstream build options
     * @property {boolean=} [redirectToJobPage=false] - Modal option to redirect to job page
     * @property {boolean=} selectPartitions - Modal option to display partition selection options
     */

    /**
     * Initialize single flow computable build modal.
     * 
     * @param {string} computableType - The computable type ("DATASET", "MANAGED_FOLDER", ...)
     * @param {Object} object - The computable to build full info
     * @param {Object} computableLoc - A location on this computable
     * @param {BuildModalOptions} [options] - Options used to initialize the modal
     */
    $scope.initModal = function (computableType, computableLoc, objectFullInfo, options) {
        const { upstreamBuildable = true, downstreamBuildable = true, redirectToJobPage = false, selectPartitions } = options || {};

        $scope.computable = {
            type: computableType
        };
        $scope.computableLoc = computableLoc;
        let object;

        switch (computableType) {
        case "DATASET": 
            object = objectFullInfo.dataset;
            $scope.computable.dataset = object;
            $scope.partitioning = object.partitioning;
            $scope.modalTitle = translate('FLOW.BUILD_SINGLE.TITLE.DATASET', "From dataset \"{{name}}\"", {name : object.name});
            $scope.targetType = "dataset";
            break;
        case "MANAGED_FOLDER": 
            object = objectFullInfo.folder;
            $scope.computable.box = object;
            $scope.partitioning = object.partitioning;
            $scope.modalTitle = translate('FLOW.BUILD_SINGLE.TITLE.MANAGED_FOLDER', "From managed folder \"{{name}}\"", {name : object.name});
            $scope.targetType = "managed folder";
            break;
        case "SAVED_MODEL": 
            object = objectFullInfo.model;
            $scope.computable.model = object;
            $scope.partitioning = object.partitioning;
            $scope.modalTitle = translate('FLOW.BUILD_SINGLE.TITLE.SAVED_MODEL', "Train model \"{{name}}\"", {name : object.name});
            $scope.targetType = "model";
            break;
        case "MODEL_EVALUATION_STORE":
            object = objectFullInfo.evaluationStore;
            $scope.computable.mes = object;
            $scope.partitioning = object.partitioning;
            $scope.modalTitle = translate('FLOW.BUILD_SINGLE.TITLE.MODEL_EVALUATION_STORE', "Update Model Evaluation Store \"{{name}}\"", {name : object.name});
            $scope.targetType = "model evaluation store";
            break;
        case "RETRIEVABLE_KNOWLEDGE":
            object = objectFullInfo.retrievableKnowledge;
            $scope.computable.retrievableKnowledge = object;
            $scope.partitioning = object.partitioning;
            $scope.modalTitle = translate('FLOW.BUILD_SINGLE.TITLE.RETRIEVABLE_KNOWLEDGE', "From Knowledge Bank \"{{name}}\"", {name : object.name});
            $scope.targetType = "knowledge bank";
            break;
        // TODO others
        }

        $scope.upstreamBuildable = upstreamBuildable;
        $scope.downstreamBuildable = downstreamBuildable;

        if (!upstreamBuildable) {
            // In that case, the default setting is not acceptable, so we need to change it
            $scope.uiState.topLevelMode = "RECURSIVE_DOWNSTREAM";
        }
        
        if ($scope.partitioning && selectPartitions === undefined && objectFullInfo.creatingRecipe) { 
            // if displaying partition selector is not explicitly required, try to fetch parent recipe settings to determine if
            // dataset creating recipe recompute all partitions each run and thus modal should not display partition selector.
            DataikuAPI.flow.recipes.get($stateParams.projectKey, objectFullInfo.creatingRecipe.name).success((recipe) => {
                $scope.selectPartitions = !recipe.redispatchPartitioning;
            }).error(setErrorInScope.bind($scope));
        } else {
            $scope.selectPartitions = selectPartitions ?? true;
        }
        $scope.redirectToJobPage = redirectToJobPage;

        Logger.info("Build modal initialized, type=" + computableType +  " loc=" + computableLoc.fullId, "upstreamBuildable=" +
                     upstreamBuildable + " downstreamBuildable=" + downstreamBuildable);
        
        if (downstreamBuildable) {
            $scope.downstreamBuildPossible = true;
            // delayed extra check that downstream build is possible. Done after init because it could 
            // be(come) a bit costly
            DataikuAPI.flow.checkDownstreamBuildable($stateParams.projectKey, {computable: $scope.computableLoc.fullId}).success(function(data) {
                $scope.downstreamBuildPossible = data.ok;
                $scope.downstreamBuildReason = data.reason;
                if (!data.ok) {
                    $scope.uiState.recursiveDownstreamBuildMode = 'FIND_OUTPUTS'; // because only the "backwards recursive downstream" is possible
                    $scope.uiState.advancedMode = true
                }
            }).error(setErrorInScope.bind($scope));
        }
    }

    function onJobStarted(jobId) {
        if ($scope.dismiss) {
            $scope.dismiss();
        }
        if ($scope.redirectToJobPage) {
            $state.go('projects.project.jobs.job', {projectKey : $stateParams.projectKey, jobId: jobId})
        } else {
            $scope.initiatingScope.$emit("datasetBuildStarted");
        }
    }

    function createBuildWT1Event(computable) {
        const topLevelMode = $scope.uiState.topLevelMode;
        let ret = {
            buildType: topLevelMode,
            computableType: computable.type,
        };

        if (!($scope.uiState.recursiveDownstreamBuildMode === "FIND_OUTPUTS" && topLevelMode === "RECURSIVE_DOWNSTREAM")) {
            ret.updateOutputSchemas = $scope.jobOptions.autoUpdateSchemaBeforeEachRecipeRun;
        }

        if (topLevelMode != "BUILD_SINGLE_COMPUTABLE") {
            ret.dependenciesHandling = topLevelMode === "RECURSIVE_UPSTREAM" ? $scope.uiState.recursiveUpstreamBuildMode : $scope.uiState.recursiveDownstreamBuildMode
        };

        if (topLevelMode === 'RECURSIVE_UPSTREAM') {
            ret.stopAtFlowZoneBoundary = $scope.jobOptions.stopAtFlowZoneBoundary;
        };

        return ret
    }

    function startReverseJob(scope, loc, isDataset) {
        const updateSchemas = $scope.jobOptions.autoUpdateSchemaBeforeEachRecipeRun;
        let reloadSchemaPromise;

        if (isDataset && loc.projectKey == $stateParams.projectKey && updateSchemas) {
            /* This is a local dataset, reload its schema */
            const deferred = $q.defer();
            reloadSchemaPromise = deferred.promise;
            DataikuAPI.datasets.reloadSchema(loc.projectKey, loc.localId).success(function(){
                deferred.resolve();
            }).error(setErrorInScope.bind(scope));
        } else {
            reloadSchemaPromise = $q.resolve();
        }

        let jd = {
            projectKey: $stateParams.projectKey,
            refreshHiveMetastore: true,
            type: "REVERSE_FORCED_BUILD",
            autoUpdateSchemaBeforeEachRecipeRun: updateSchemas,
            reverseStartingPoints: [{
                graphNodeSupertype: "COMPUTABLE",
                projectKey: loc.projectKey,
                id: loc.localId
            }]
        }

        reloadSchemaPromise.then(function() {
            WT1.tryEvent(WT1BuildEvents.Start, () =>  createBuildWT1Event($scope.computable));
            DataikuAPI.flow.jobs.start(jd).success(function(data) {
                scope.startedJob = data;
                onJobStarted(data.id);
            }).error((data, status, headers) => {
                setErrorInScope.bind(scope)(data, status, headers);
                WT1.tryEvent(WT1BuildEvents.StartFailed, () => createBuildWT1Event($scope.computable));
            });
        });
    }

    function startNormalJob(scope, computableLoc, computable, buildMode, isPreview) {
        let jd = JobDefinitionComputer.computeJobDefForComputable($stateParams.projectKey, buildMode, computable, $scope.buildPartitions, null, null, $scope.jobOptions);

        Logger.info("Starting regular job, with definition: " + JSON.stringify(jd));

        $scope.isBuildingDataset = true;
        let apiCall = isPreview ? DataikuAPI.flow.jobs.startPreview : DataikuAPI.flow.jobs.start;
        let wt1Label = isPreview ? WT1BuildEvents.Preview : WT1BuildEvents.Start;

        WT1.tryEvent(wt1Label, () =>  createBuildWT1Event(computable));
        $scope.redirectToJobPage = $scope.redirectToJobPage || isPreview;
        apiCall(jd).success(function(data) {
            $scope.startedJob = data;
            onJobStarted(data.id);
        }).error((data, status, headers) => {
            setErrorInScope.bind($scope)(data, status, headers)
            $scope.isBuildingDataset = false;
            WT1.tryEvent(`${wt1Label}-failed`, () => createBuildWT1Event(computable));
        });
    }

    function switchToFindDownstreamComputablesMode() {
        // Ugly, we must reattach to the grand-parent scope since the current modal dies
        const grandParentScope = $scope.$parent.$parent;
        $scope.dismiss();
        DataikuAPI.flow.listDownstreamComputables($stateParams.projectKey, {computable: $scope.computableLoc.fullId})
            .success(function(data) {
            if (!data.length) {
                // Should not happen ...
                Dialogs.error(grandParentScope, translate("FLOW.COMMON_BUILD_ERRORS.NOTHING_TO_BUILD", "Nothing to build"));
            } else {
                const originalBuildContext = {
                    origin: "COMPUTABLE",
                    computable: $scope.computable
                }
                CreateModalFromTemplate(
                    "/templates/flow-editor/tools/build-multiple-flow-computables-modal.html", grandParentScope,
                    "BuildMultipleComputablesController", function(modalScope) {
                        modalScope.initModal(data, undefined, originalBuildContext);
                });
            }
        }).error(setErrorInScope.bind($scope));
    }

    $scope.startJob = function(isPreview) {
        if ($scope.uiState.topLevelMode == "RECURSIVE_DOWNSTREAM") {
            if ($scope.uiState.recursiveDownstreamBuildMode == 'FIND_OUTPUTS') {
                WT1.tryEvent(WT1BuildEvents.Next, () =>  createBuildWT1Event($scope.computable));
                switchToFindDownstreamComputablesMode();
            } else {
                startReverseJob($scope, $scope.computableLoc,
                                $scope.computable.type == "DATASET");
            }
        } else {
            let mode = null;
            if ($scope.uiState.topLevelMode == "BUILD_SINGLE_COMPUTABLE") {
                mode = "NON_RECURSIVE_FORCED_BUILD";
            } else {
                mode = $scope.uiState.recursiveUpstreamBuildMode;
            }

            startNormalJob($scope, $scope.computableLoc, $scope.computable, mode, isPreview);
        }
    };

    $scope.getBuildComputableButtonLabel = function() {
        if ($scope.uiState.topLevelMode === 'RECURSIVE_UPSTREAM' || $scope.uiState.topLevelMode === 'BUILD_SINGLE_COMPUTABLE') {
            return translate('FLOW.BUILD_SINGLE.BUILD.' + $scope.computable.type, 'Build ' + $scope.targetType);
        } else if ($scope.uiState.topLevelMode === 'RECURSIVE_DOWNSTREAM') {
            if ($scope.uiState.recursiveDownstreamBuildMode === 'FIND_OUTPUTS') {
                return translate('FLOW.BUILD_SINGLE.NEXT', 'Next');
            } else {
                return translate('FLOW.BUILD_SINGLE.BUILD_DOWNSTREAM.' + $scope.computable.type, 'Build')
            }
        }
    }

    $scope.isBuildingDataset = false;
});


app.service("FlowBuildService", function($stateParams, DataikuAPI, Dialogs, CreateModalFromTemplate, RecipeComputablesService, RecipeRunJobService, JobDefinitionComputer, WT1, AnyLoc, $q, Logger, $state, translate){
    var svc = {
        /**
         * open single flow computable build modal and loc this object.
         *
         * @param {string} objectType - the computable type ("DATASET", "MANAGED_FOLDER", ...). (Required)
         * @param {Object} objectLoc - computable location. (Required)
         * @param {Object} options - options used to initialize the modal.
         * @param {boolean} [options.upstreamBuildable=['true']] - modal option to display upstream build options. Defaults to true. (Optional)
         * @param {boolean} [options.downstreamBuildable=['true']] - modal option to display downstream build options. Defaults to true. (Optional)
         * @param {boolean} [options.redirectToJobPage=['false']] - modal option to redirect to job page. Defaults to false. (Optional)
         * @param {boolean} [options.selectPartitions=['true']] - modal option to display partition selection options. Defaults to true. (Optional)
        */
        openSingleComputableBuildModalFromObjectTypeAndLoc: function (scope, objectType, objectLoc, options) {
            switch (objectType) {
            case "DATASET": {
                DataikuAPI.datasets.getFullInfo($stateParams.projectKey, objectLoc.projectKey, objectLoc.localId).success(function (datasetInfo) {
                    CreateModalFromTemplate("/templates/flow-editor/tools/build-flow-computable.html", scope, "BuildSingleFlowComputableController", function (modalScope) {
                        modalScope.initModal("DATASET", objectLoc, datasetInfo, options);
                        modalScope.initiatingScope = scope;
                    }, "build-dataset-modal");
                }).error(setErrorInScope.bind(scope));
                break;
            }
            case "MANAGED_FOLDER": {
                DataikuAPI.managedfolder.getFullInfo($stateParams.projectKey, objectLoc.projectKey, objectLoc.localId).success(function(folderInfo) {
                        CreateModalFromTemplate("/templates/flow-editor/tools/build-flow-computable.html", scope, "BuildSingleFlowComputableController", function(modalScope) {
                            modalScope.initModal("MANAGED_FOLDER", objectLoc, folderInfo, options);
                            modalScope.initiatingScope = scope;
                        }, "build-dataset-modal");
                }).error(setErrorInScope.bind(scope));
                break;
            }
            case "SAVED_MODEL": {
                DataikuAPI.savedmodels.getFullInfo(objectLoc.projectKey, objectLoc.localId).success(function(modelInfo) {
                    CreateModalFromTemplate("/templates/flow-editor/tools/build-flow-computable.html", scope, "BuildSingleFlowComputableController", function(modalScope) {
                        modalScope.initModal("SAVED_MODEL", objectLoc, modelInfo, options);
                        modalScope.initiatingScope = scope;
                    }, "build-dataset-modal");
                }).error(setErrorInScope.bind(scope));
                break;
            }
            case "MODEL_EVALUATION_STORE": {
                DataikuAPI.modelevaluationstores.getFullInfo(objectLoc.projectKey, objectLoc.localId).success(function(mesInfo) {
                    CreateModalFromTemplate("/templates/flow-editor/tools/build-flow-computable.html", scope, "BuildSingleFlowComputableController", function(modalScope) {
                        modalScope.initModal("MODEL_EVALUATION_STORE", objectLoc, mesInfo, options);
                        modalScope.initiatingScope = scope;
                    }, "build-dataset-modal");
                }).error(setErrorInScope.bind(scope));
                break;
            }
            case "RETRIEVABLE_KNOWLEDGE": {
                DataikuAPI.retrievableknowledge.getFullInfo(objectLoc.projectKey, objectLoc.localId).success(function(retrievableKnowledgeInfo) {
                    CreateModalFromTemplate("/templates/flow-editor/tools/build-flow-computable.html", scope, "BuildSingleFlowComputableController", function(modalScope) {
                        modalScope.initModal("RETRIEVABLE_KNOWLEDGE", objectLoc, retrievableKnowledgeInfo, options);
                        modalScope.initiatingScope = scope;
                    }, "build-dataset-modal");
                }).error(setErrorInScope.bind(scope));
                break;
            }

            }
        },

        openRecipeRunModal : function(scope, recipeProjectKey, recipe, redirectToJobPage) {
            RecipeComputablesService.getComputablesMap(recipe, scope).then(function(computablesMap){
                const outputAndPartitioning = RecipeRunJobService.getOutputAndPartitioning(recipe, computablesMap);
                Logger.info("oanda from ", recipe, computablesMap, "-->", outputAndPartitioning)
                const outputRef = outputAndPartitioning.output.ref;

                scope.redirectToJobPage = redirectToJobPage;

                /* Output is partitioned, so we can't do the smart thing */
                if ("partitioning" in outputAndPartitioning) {
                    
                    if (computablesMap && computablesMap[outputRef]) {
                        svc.openSingleComputableBuildModalFromObjectTypeAndLoc(scope, computablesMap[outputRef].type, AnyLoc.getLocFromSmart(recipeProjectKey, outputRef), { selectPartitions: !recipe.redispatchPartitioning });
                    }

                } else {
                    /* Output is not partitioned, propose more choices */
                    let recipeScope = scope.$new();
                    recipeScope.recipe = recipe

                    const WT1RecipeRunEvents = {
                        Start: 'flow-run-recipe-start',
                        StartFailed: 'flow-run-recipe-start-failed',
                        Next: 'flow-run-recipe-find-outputs-next',
                    }
                    CreateModalFromTemplate("/templates/recipes/run-recipe-modal.html", recipeScope, "RunRecipeBuildOptionsController", function(modalScope) {
                        function createBuildWT1Event(recipe) {
                            const topLevelMode = modalScope.selectedBuildMode;
                            let ret = {
                                buildType: topLevelMode,
                                recipeType: recipe.type,
                            };

                            if (modalScope.jobOptions.recursiveDownstreamBuildMode != "FIND_OUTPUTS") {
                                ret.updateOutputSchemas = modalScope.jobOptions.autoUpdateSchemaBeforeEachRecipeRun;
                            }

                            if (topLevelMode != "RUN_RECIPE_ONLY") {
                                ret.dependenciesHandling = topLevelMode === "RECURSIVE_UPSTREAM" ? modalScope.jobOptions.recursiveUpstreamBuildMode : modalScope.jobOptions.recursiveDownstreamBuildMode
                            };

                            return ret;
                        }

                        modalScope.confirm = function() {
                            modalScope.jobStarted = true;
                            // Ugly, we must reattach to the parent scope to call the nested CreateModalFromTemplate since the current scope dies at modal dismissal.
                            const parentScope = modalScope.$parent;
                            if (modalScope.selectedBuildMode === 'RUN_RECIPE_ONLY') {
                                if (computablesMap && computablesMap[outputRef]) {
                                    let jd = JobDefinitionComputer.computeJobDefForComputable($stateParams.projectKey, "NON_RECURSIVE_FORCED_BUILD",
                                                            computablesMap[outputRef], null, null, null, {
                                                                autoUpdateSchemaBeforeEachRecipeRun: modalScope.jobOptions.autoUpdateSchemaBeforeEachRecipeRun
                                                            });
                                    WT1.tryEvent(WT1RecipeRunEvents.Start, () => createBuildWT1Event(recipe));
                                    DataikuAPI.flow.jobs.start(jd).success(function(data) {
                                        modalScope.startedJob = data;
                                        modalScope.$emit("datasetBuildStarted");
                                        modalScope.dismiss();
                                    }).error((data, status, headers) => {
                                        setErrorInScope.bind(modalScope)(data, status, headers)
                                        modalScope.jobStarted = false;
                                        WT1.tryEvent(WT1RecipeRunEvents.StartFailed, () => createBuildWT1Event(recipe));
                                    });
                                }
                            } else if (modalScope.selectedBuildMode === 'RECURSIVE_DOWNSTREAM') {
                                if (modalScope.jobOptions.recursiveDownstreamBuildMode === 'FIND_OUTPUTS') {
                                    WT1.tryEvent(WT1RecipeRunEvents.Next, () => createBuildWT1Event(recipe));

                                    DataikuAPI.flow.listDownstreamComputables($stateParams.projectKey, {runnable: recipe.name})
                                        .success(function(data) {
                                            if (!data.length) {
                                                // Should not happen ...
                                                Dialogs.error(parentScope, translate("FLOW.COMMON_BUILD_ERRORS.NOTHING_TO_BUILD", "Nothing to build"));
                                                modalScope.jobStarted = false;
                                            } else {
                                                const originalBuildContext = {
                                                    origin: "RECIPE",
                                                    recipe: modalScope.recipe
                                                }
                                                CreateModalFromTemplate(
                                                    "/templates/flow-editor/tools/build-multiple-flow-computables-modal.html", parentScope,
                                                    "BuildMultipleComputablesController", function(nextModalScope) {
                                                        nextModalScope.initModal(data, undefined, originalBuildContext);
                                                });
                                            }
                                            modalScope.dismiss();
                                        }).error((data, status, headers) => {
                                              setErrorInScope.bind(modalScope)(data, status, headers)
                                              modalScope.jobStarted = false;
                                          });
                                } else if (modalScope.jobOptions.recursiveDownstreamBuildMode === 'REVERSE_BUILD') {
  
                                    let jd = {
                                        projectKey: $stateParams.projectKey,
                                        refreshHiveMetastore: true,
                                        type: "REVERSE_FORCED_BUILD",
                                        autoUpdateSchemaBeforeEachRecipeRun: modalScope.jobOptions.autoUpdateSchemaBeforeEachRecipeRun,
                                        reverseStartingPoints: [{
                                            graphNodeSupertype: "RUNNABLE",
                                            projectKey: $stateParams.projectKey,
                                            id: recipe.name
                                        }]
                                    }

                                    WT1.tryEvent(WT1RecipeRunEvents.Start, () => createBuildWT1Event(recipe));
                                    DataikuAPI.flow.jobs.start(jd).success(function(data) {
                                        modalScope.startedJob = data;
                                        modalScope.$emit("datasetBuildStarted");
                                        modalScope.dismiss();
                                    }).error((data, status, headers) => {
                                          setErrorInScope.bind(modalScope)(data, status, headers)
                                          modalScope.jobStarted = false;
                                          WT1.tryEvent(WT1RecipeRunEvents.StartFailed, () => createBuildWT1Event(recipe));
                                      });
                                }
                            }
                        };
                    });

                }
            });
        }
    }
    return svc;
});


app.directive('partitionSelector', function(PartitionSelection) {
    return {
        restrict: 'AE',
        scope: {
            selectPartitions: '<?',
            partitioning: '=',
            buildPartitions: '=',
            bemModifier: '=?'
        },
        templateUrl: '/templates/datasets/partition-selector.html',
        link: function ($scope, element, attrs) {
            if ($scope.selectPartitions === undefined) {
                $scope.selectPartitions = true;
            }

            $scope.$watch('partitioning.dimensions', function(nv, ov){
                $scope.buildPartitions = PartitionSelection.getBuildPartitions($scope.partitioning);
            },true);

            $scope.$watch("buildPartitions", function(nv, ov) {
                if (nv !== ov) {
                    PartitionSelection.saveBuildPartitions($scope.partitioning, $scope.buildPartitions);
                }
            }, true);
        }
    }
});


//Expects in scope: streamingEndpointId
app.controller('BuildStreamingEndpointController', function($scope, DataikuAPI, $state, $stateParams, JobDefinitionComputer, Assert) {
    $scope.compute = { mode : "NON_RECURSIVE_FORCED_BUILD"};
    $scope.build_partitions = {};
    $scope.jobOptions = {
        autoUpdateSchemaBeforeEachRecipeRun: false
    }

    // $scope.$watch('streamingEndpointId', function() {
    //     if (!$scope.odbId) return;
    //     DataikuAPI.streamingEndpoints.get($stateParams.projectKey, $scope.streamingEndpointId).success(function(data) {
    //         $scope.streamingEndpoint = data;
    //     }).error(setErrorInScope.bind($scope));
    // });

    $scope.buildModes = [
        ["NON_RECURSIVE_FORCED_BUILD", "Build only this streaming endpoint"],
        ["RECURSIVE_BUILD", "Build required datasets and this streaming endpoint"],
        ["RECURSIVE_FORCED_BUILD", "Force-rebuild all dependencies and build the streaming endpoint"],
        ["RECURSIVE_MISSING_ONLY_BUILD", "Build missing dependencies and build the streaming endpoint"]
    ];

    $scope.isBuildingDataset = false;
    $scope.startJob = function() {
        Assert.inScope($scope, 'streamingEndpointId');
        $scope.isBuildingDataset = true;
        DataikuAPI.flow.jobs.start(JobDefinitionComputer.computeJobDefForStreamingEndpoint($stateParams.projectKey, $scope.compute.mode, {id:$scope.streamingEndpointId})).success(function(data) {
            $scope.startedJob = data;
            // This is really a ugly hack ... It's used to dismiss the modal
            // when this controller is called from the
            // build-dataset-box.html template
            if ($scope.dismiss) $scope.dismiss();
        }).error(setErrorInScope.bind($scope)).then(function(){$scope.isBuildingDataset = false;});
    };

     $scope.startJobPreview = function() {
        $scope.isBuildingDataset = true;
        DataikuAPI.flow.jobs.startPreview(JobDefinitionComputer.computeJobDefForStreamingEndpoint($stateParams.projectKey, $scope.compute.mode, {id:$scope.streamingEndpointId})).success(function(data) {
            $scope.startedJob = data;
            // This is really a ugly hack ... It's used to dismiss the modal
            // when this controller is called from the
            // build-dataset-box.html template
            if ($scope.dismiss) $scope.dismiss();
            $state.go('projects.project.jobs.job', {projectKey : $stateParams.projectKey, jobId: data.id});
        }).error(setErrorInScope.bind($scope)).then(function(){$scope.isBuildingDataset = false;});
    };
})

app.controller('RunRecipeBuildOptionsController', function($scope, DataikuAPI, $stateParams, translate) {
    $scope.jobStarted = false;
    $scope.modalTitle = translate("FLOW.RUN_RECIPE.TITLE", "Run recipe");
    $scope.isRunningFromRecipe = false;
    if ($scope.recipe && $scope.recipe.name) {
        $scope.modalTitle += " \"" + $scope.recipe.name + "\"";
    }
    $scope.withJobOptions = true;
    $scope.selectedBuildMode = "RUN_RECIPE_ONLY";
    $scope.advancedMode = false;

    function computeBuildModeAnimationData() {
        if ($scope.selectedBuildMode === 'RUN_RECIPE_ONLY') {
            return "/static/dataiku/images/flow/run-recipe-run-recipe-only.json";
        } else if ($scope.selectedBuildMode === 'RECURSIVE_DOWNSTREAM') {
            if ($scope.jobOptions.recursiveDownstreamBuildMode === 'FIND_OUTPUTS') {
                return "/static/dataiku/images/flow/run-recipe-recursive-downstream-find-outputs.json"
            } else if ($scope.jobOptions.recursiveDownstreamBuildMode === 'REVERSE_BUILD') {
                return "/static/dataiku/images/flow/run-recipe-recursive-downstream-reverse-build.json";
            }
        }
    }

    $scope.selectBuildMode = function(item) {
        $scope.selectedBuildMode = item;
    };

    $scope.toggleAdvancedMode = function() {
        $scope.advancedMode = !$scope.advancedMode;
    };

    $scope.uiState = {
        updateOutputSchemasPerMode: {
            "RUN_RECIPE_ONLY": true,
            "RECURSIVE_DOWNSTREAM": true,
        },
    };

    // TODO add all form controls in uiState (could be done by factorising code with the build flow computable modale)
    // Watch settings changes to update UI and default parameters, based on target type and both levels of build mode
    $scope.$watch("uiState", function (nv) {
        if (!nv) return;
        $scope.jobOptions.autoUpdateSchemaBeforeEachRecipeRun = $scope.uiState.updateOutputSchemasPerMode[$scope.selectedBuildMode]
        $scope.buildModelAnimationData = computeBuildModeAnimationData();
    }, true)

    $scope.jobOptions = {
        autoUpdateSchemaBeforeEachRecipeRun: $scope.uiState.updateOutputSchemasPerMode[$scope.selectedBuildMode],
        recursiveDownstreamBuildMode: "REVERSE_BUILD"
    }

    $scope.$watch("jobOptions", function (nv) {
        if (!nv) return;
        $scope.buildModelAnimationData = computeBuildModeAnimationData();
    }, true);

    $scope.$watch("selectedBuildMode", function (nv) {
        if (!nv) return;
        $scope.buildModelAnimationData = computeBuildModeAnimationData();
    });

    $scope.downstreamBuildPossible = true;
    // delayed extra check that downstream build is possible. Done after init because it could 
    // be(come) a bit costly
    if ($scope.recipe) {
        DataikuAPI.flow.checkDownstreamBuildable($stateParams.projectKey, {runnable: $scope.recipe.name}).success(function(data) {
            $scope.downstreamBuildPossible = data.ok;
            $scope.downstreamBuildReason = data.reason;
            if (!data.ok) {
                $scope.jobOptions.recursiveDownstreamBuildMode = 'FIND_OUTPUTS'; // because only the "backwards recursive downstream" is possible
                $scope.advancedMode = true
            }
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller('BuildMultipleComputablesController', function ($scope, JobDefinitionComputer, DataikuAPI, $stateParams, $state, $filter, translate, WT1){
    
    $scope.initModal = function (computables, startingPoint, originalBuildContext) {
        $scope.startingPoint = startingPoint;
        $scope.computables = computables;
        $scope.originalBuildContext = originalBuildContext;
        $scope.computeAllPartitions = new Set();
        $scope.needsToSelectPartitions = computable => {
            return !$scope.computeAllPartitions.has(getComputableKey(computable.type, computable.projectKey, computable.id));
        };
        // for each partitioned computable try to fetch its creating recipe 
        // to decide if we need to show partition selector in the build modal
        const requestParams = computables
            .filter(computable => {
                const partitioning = $scope.getPartitioning(computable);
                return (partitioning !== undefined && partitioning.dimensions.length)
            })
            .map(computable => ({ projectKey: $stateParams.projectKey, type: computable.type, id: computable.id }));
        if (requestParams.length > 0) {
            DataikuAPI.flow.recipes.getCreatingRecipes(JSON.stringify(requestParams)).success(recipeInfos => {
                if (recipeInfos) {
                    recipeInfos.forEach(recipeInfo => {
                        if (recipeInfo?.creatingRecipe && recipeInfo.creatingRecipe.redispatchPartitioning) {
                            $scope.computeAllPartitions.add(getComputableKey(recipeInfo.computableType, recipeInfo.computableProjectKey, recipeInfo.computableId));
                        }
                    })
                }
            }).error(setErrorInScope.bind($scope));
        }
    };
    $scope.buildModes = [
            ["RECURSIVE_BUILD", translate("FLOW.BUILD_MULTIPLE.DEPENDENCY_HANDLING.RECURSIVE_BUILD", "Build required dependencies")],
            ["RECURSIVE_FORCED_BUILD", translate("FLOW.BUILD_MULTIPLE.DEPENDENCY_HANDLING.RECURSIVE_FORCED_BUILD", "Force-rebuild all dependencies")]
    ];
    $scope.buildMode = "RECURSIVE_BUILD";

    $scope.jobOptions = {
        autoUpdateSchemaBeforeEachRecipeRun: false,
        stopAtFlowZoneBoundary: false
    }

    $scope.removeRestore = function(index, isRemove) {
        $scope.computables[index].removed = isRemove;
        $scope.validateForm();
    };

    $scope.isAllDataset = function() {
        return $scope.computables.filter(function(c) {return c.type == 'DATASET';}).length == $scope.computables.length;
    };

    $scope.validateForm = function () {
        $scope.theform.$invalid = $scope.computables.find(i => !i.removed) == undefined;
    };

    const WT1BuildMultipleComputablesEvents = {
        BuildComputableFindOutputsStart : 'flow-build-computable-find-outputs-start',
        RunRecipeFindOutputsStart : 'flow-run-recipe-find-outputs-start',
        FindOuputsPreview:'flow-build-computable-find-outputs-preview',
        FindOutputsPreviewFailed:'flow-build-computable-find-outputs-preview-failed',
    }

    const getComputableKey = (type, projectKey, id) => (type + '.' + projectKey + '.' + id);

    function getJobDef() {
        var outputs = $scope.computables.filter(d => !d.removed).map(function(d) {
            if (d.type === 'DATASET') {
                return JobDefinitionComputer.computeOutputForDataset(d.serializedDataset, d.buildPartitions);
            } else if (d.type === 'MANAGED_FOLDER') {
                return JobDefinitionComputer.computeOutputForBox(d.box, d.buildPartitions);
            } else if (d.type === 'SAVED_MODEL') {
                return JobDefinitionComputer.computeOutputForSavedModel(d.model, d.buildPartitions);
            } else {
                return { "targetDataset": d.id, "targetDatasetProjectKey": d.projectKey, "type": d.type };
            }
        });

        return {
            "type": $scope.buildMode,
            "refreshHiveMetastore":true,
            autoUpdateSchemaBeforeEachRecipeRun: $scope.jobOptions.autoUpdateSchemaBeforeEachRecipeRun,
            stopAtFlowZoneBoundary: $scope.jobOptions.stopAtFlowZoneBoundary,
            "projectKey": $stateParams.projectKey,
            "outputs": outputs
        };
    }

    function createBuildWT1Event(originalBuildContext) {
        let ret = {
            buildType: "RECURSIVE_DOWNSTREAM",
            dependenciesHandling: $scope.buildMode,
            updateOutputSchemas: $scope.jobOptions.autoUpdateSchemaBeforeEachRecipeRun,
            stopAtFlowZoneBoundary: $scope.jobOptions.stopAtFlowZoneBoundary
        };

        if (originalBuildContext.origin === "COMPUTABLE") {
            ret.computableType = originalBuildContext.computable.type;
        } else {
            ret.recipeType = originalBuildContext.recipe.type;
        }
        return ret
    }

    $scope.isBuildingDataset = false;
    $scope.startJob = function() {
        $scope.isBuildingDataset = true;

        let wt1Label;
        if ($scope.originalBuildContext) {
            wt1Label = $scope.originalBuildContext.origin === "COMPUTABLE" ? WT1BuildMultipleComputablesEvents.BuildComputableFindOutputsStart : WT1BuildMultipleComputablesEvents.RunRecipeFindOutputsStart;
            WT1.tryEvent(wt1Label, () => createBuildWT1Event($scope.originalBuildContext));
        }

        DataikuAPI.flow.jobs.start(getJobDef()).success(function(startedJob) {
            $scope.$emit("datasetBuildStarted");
            $scope.dismiss();
            $scope.isBuildingDataset = false;
        }).error((data, status, headers) => {
            setErrorInScope.bind($scope)(data, status, headers)
            $scope.isBuildingDataset = false;

            if ($scope.originalBuildContext) {
                wt1Label = `${wt1Label}-failed`;
                WT1.tryEvent(wt1Label, () => createBuildWT1Event($scope.originalBuildContext));
            }
        })
    };

    $scope.startJobPreview = function() {
        $scope.isBuildingDataset = true;
        if ($scope.originalBuildContext) {
            WT1.tryEvent(WT1BuildMultipleComputablesEvents.FindOuputsPreview, () => createBuildWT1Event($scope.originalBuildContext));
        }

        DataikuAPI.flow.jobs.startPreview(getJobDef()).success(function(startedJob) {
            $state.go('projects.project.jobs.job', {projectKey : $stateParams.projectKey, jobId: startedJob.id});
            $scope.dismiss();
        }).error((data, status, headers) => {
            if ($scope.originalBuildContext) {
                WT1.tryEvent(WT1BuildMultipleComputablesEvents.FindOutputsPreviewFailed, () => createBuildWT1Event($scope.originalBuildContext));
            }
            setErrorInScope.bind($scope)(data, status, headers);
        }).finally(function(){$scope.isBuildingDataset = false;});
    };

    $scope.getIcon = function(computable) {
        switch(computable.type) {
            case 'DATASET':            return 'dataset ' + $filter('toModernIcon')($filter('datasetTypeToIcon')(computable.serializedDataset.type, 16), 16);
            case 'MANAGED_FOLDER':     return 'dku-icon-dataset-files-in-folder-16';
            case 'SAVED_MODEL':        return 'dku-icon-machine-learning-regression-16';
        }
    };

    $scope.getPartitioning = function(computable) {
        if (computable.type === 'DATASET') {
            return computable.serializedDataset.partitioning;
        }
        if (computable.type === 'MANAGED_FOLDER') {
            return computable.box.partitioning;
        }
        if (computable.type === 'SAVED_MODEL') {
            return computable.model.partitioning;
        }
    };
});

app.controller('XmlFormatController', function ($scope) {
	// record the last field visited that is used to input XPath. The fields
	// have a directive to send a setter on their model to xpathFieldGotFocus().
	// by default, the focus is on the root path element field
	$scope.lastFocusedXpathFieldSetter = function (value) {$scope.dataset.formatParams.rootPath = value;};

	$scope.xpathFieldGotFocus = function (fieldSetter) {
		$scope.lastFocusedXpathFieldSetter = fieldSetter;
	};
});

})();
