(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');
    // const LEFT_MENU_OPENED_KEY = 'dashboardLeftMenuOpened';

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/edit.js
     */
    app.controller('DashboardEditController', function($scope, $state, $timeout, DashboardFilters,
        DataikuAPI, CreateModalFromTemplate, Dialogs, TopNav, INSIGHT_TYPES, DSSVisualizationThemeUtils,
        TileUtils, ActivityIndicator, DashboardUtils, $rootScope, ATSurveyService, WT1, FiltersPanelPosition, StateUtils,
        DashboardPageUtils) {
        TopNav.setLocation(TopNav.TOP_DASHBOARD, 'dashboards', null, 'edit');

        if ($scope.dashboard) {
            TopNav.setPageTitle($scope.dashboard.name + ' - Dashboard');
        }

        // Call ATSurveyService to trigger survey if needed
        ATSurveyService.updateCounter('DashboardEdit');

        $scope.insightTypes = INSIGHT_TYPES;
        $scope.hasFilterTile = false;
        $scope.FiltersPanelPosition = FiltersPanelPosition;
        $scope.selectedTiles = [];
        $scope.selectedInsights = [];

        Dialogs.saveChangesBeforeLeaving($scope, $scope.isDirty, $scope.saveDashboard, $scope.revertChanges, 'This dashboard has unsaved changes.');
        Dialogs.checkChangesBeforeLeaving($scope, $scope.isDirty);

        DashboardUtils.toggleAllFilterPanels($scope.dashboard, true);

        $scope.$on('dashboardSyncModelsDone', function() {
            const insightIdSet = retrieveInsightIdSetFromDashboard($scope.dashboard);
            $scope.hasFilterTile = alreadyHaveGlobalFilters();
            Object.keys($scope.insightsMap).forEach(function(insightId) {
                if (!insightIdSet.has(insightId)) {
                    delete $scope.insightsMap[insightId];
                }
            });
        });

        const unregisterDashboardSelectTileEvent = $rootScope.$on('dashboardSelectTiles', (_, { tiles }) => {
            $scope.selectedTiles = tiles;
            $scope.selectedInsights = tiles.map(tile => $scope.insightsMap[tile.insightId]);
        });

        const unregisterDashboardUnselectTilesEvent = $rootScope.$on('dashboardUnselectTiles', () => {
            $scope.selectedTiles = [];
            $scope.selectedInsights = [];
        });

        $scope.$watch('uiState.selectedTile', (nv) => {
            if (nv) {
                $scope.selectedTiles = [nv];
                $scope.selectedInsights = [$scope.insightsMap[nv.insightId]];
            }
        });

        $scope.createNewTile = function() {
            CreateModalFromTemplate('/templates/dashboards/insights/new-insight-modal.html', $scope, 'NewInsightModalController', function(modalScope) {
                modalScope.withSimpleTiles = true;
                modalScope.inDashboard = true;
                modalScope.pointerMode = { isPointerMode: false };
                modalScope.dashboardTheme = $scope.dashboard.theme;
                $scope.filtersCompatibleTiles = DashboardFilters.getSlideFilterableTiles($state.params.pageId, $scope.dashboard.pages);
                modalScope.setDashboardCreationId($scope.dashboard.id);
                modalScope.addToDashboard = function(insight) {
                    if (!modalScope.pointerMode.isPointerMode) {
                        DataikuAPI.dashboards.insights.copy($scope.dashboard.projectKey, [insight.id], [insight.name], $scope.dashboard.id)
                            .error(setErrorInScope.bind($scope))
                            .success(function(data) {
                                const insightCopy = data[0];
                                $scope.insightsMap[insightCopy.id] = insightCopy;
                                $scope.addInsightToPage(insightCopy);
                                DashboardUtils.sendWT1InsightCreation(insightCopy.type, {
                                    triggeredFrom: 'dashboard-new-tile-modal-pick-existing'
                                });
                                ActivityIndicator.success('Added');
                            });
                    } else {
                        $scope.addInsightToPage(insight);
                        ActivityIndicator.success('Added');
                    }
                    DashboardUtils.sendWT1TileCreation('INSIGHT', {
                        insightType: insight.type,
                        reuseExistingInsight: modalScope.pointerMode.isPointerMode,
                        triggeredFrom: 'dashboard-new-tile-modal-pick-existing'
                    });
                };
                modalScope.isInsightInDashboard = function(insight) {
                    return typeof($scope.insightsMap[insight.id]) !== 'undefined';
                };
            }, undefined, 'static', false).then(function(ret) {
                if (ret.tileType) { // simple tile
                    $scope.addSimpleTileToPage(ret.tileType);
                    DashboardUtils.sendWT1TileCreation(ret.tileType, {
                        triggeredFrom: 'dashboard-new-tile-modal'
                    });

                } else {
                    if (DashboardUtils.getInsightHandler(ret.insight.type).goToEditAfterCreation) {
                        $scope.$on('dashboardSyncModelsDone', function() {
                            $scope.saveDashboard().success(function() {
                                $timeout(function() {
                                    const options = {
                                        name: ret.insight.name,
                                        tab: 'edit',
                                        originDashboard: {
                                            id: $scope.dashboard.id,
                                            name: $scope.dashboard.name,
                                            pageId: $scope.dashboard.pages[$scope.uiState.currentPageIdx].id,
                                            edit: true
                                        }
                                    };
                                    StateUtils.go.insight(ret.insight.id, $scope.dashboard.projectKey, options);
                                });
                            });
                        });
                    }

                    $scope.insightsMap[ret.insight.id] = ret.insight;
                    $scope.insightAccessMap[ret.insight.id] = ret.insight.isReaderAccessible ? 'READER' : 'ANALYST';
                    $scope.addInsightToPage(ret.insight);
                    DashboardUtils.sendWT1TileCreation('INSIGHT', {
                        insightType: ret.insight.type,
                        triggeredFrom: 'dashboard-new-tile-modal'
                    });
                }
            });
        };

        $scope.$on('dashboardOpenAddInsightModal', $scope.createNewTile);

        $scope.addFilterTileToPage = function() {
            const filterInsight = {
                type: 'filters',
                name:'Filters',
                params: {},
                dashboardCreationId: $scope.dashboard.id,
                projectKey: $scope.dashboard.projectKey,
                titleOptions: { showTitle: 'NO' }
            };
            DataikuAPI.dashboards.insights.save(filterInsight)
                .error(setErrorInScope.bind($scope))
                .success((insightId) => {
                    filterInsight.id = insightId;
                    $scope.addInsightToPage(filterInsight);
                });
        };

        $scope.$on('dashboardAddFilterTile', $scope.addFilterTileToPage);

        $scope.addInsightToPage = function(insight) {
            if (!$scope.insightsMap[insight.id]) {
                $scope.insightsMap[insight.id] = insight;
            }

            $scope.addTileToPage(TileUtils.newInsightTile(insight));
        };

        $scope.addTextTileToPage = function() {
            $scope.addTileToPage({
                tileType: 'TEXT',
                tileParams: TileUtils.getDefaultTileParams({ type: 'text' }),
                box: TileUtils.getDefaultTileBox({ type: 'text' }),
                titleOptions: { showTitle: 'NO' }
            });
        };

        $scope.addSimpleTileToPage = function(type) {
            let tile = {
                tileType: type.toUpperCase(),
                tileParams: TileUtils.getDefaultTileParams({ type: type.toLowerCase() }),
                box: TileUtils.getDefaultTileBox({ type: type.toLowerCase() }),
                titleOptions: {}
            };

            if (type.toUpperCase() == 'IMAGE') {
                tile.displayMode = 'IMAGE';
                tile.titleOptions.showTitle = 'MOUSEOVER';
                tile.resizeImageMode = 'FIT_SIZE';
            } else if (type.toUpperCase() == 'GROUP') {
                const page = $scope.dashboard.pages[$scope.uiState.currentPageIdx];
                tile = TileUtils.createGroupTile([], { useDashboardSpacing: true, tileSpacing: $scope.dashboard.tileSpacing, backgroundColor: page.backgroundColor });
            } else {
                tile.titleOptions.showTitle = 'NO';
            }

            $scope.addTileToPage(tile);
        };

        $scope.switchPage = (index) => {
            $scope.uiState.currentPageIdx = index;
        };


        $scope.createPage = function() {
            const page = {
                /*
                 * client id or cid is a unique identifier automatically assigned
                 * when they're first created.
                 * Client ids are handy when the model has not yet been saved to the server,
                 * and does not yet have its eventual true id, but already needs to be visible in the UI.
                 */
                cid: _.uniqueId(),
                titleAlign: 'CENTER',
                grid: {
                    tiles: []
                },
                enableCrossFilters: true,
                showFilterPanel: true,
                filtersParams: {
                    panelPosition: FiltersPanelPosition.TOP
                },
                show: true
            };
            $timeout(() => {
                DSSVisualizationThemeUtils.applyToPage(page, $scope.dashboard.theme);
                $scope.dashboard.pages.push(page);
                $scope.uiState.currentPageIdx = $scope.dashboard.pages.length - 1;
                $scope.$apply();
            });

            $scope.hasFilterTile = alreadyHaveGlobalFilters();
            WT1.event('dashboard-page-create', { newPageCount: $scope.dashboard.pages.length });
        };

        $scope.removePage = function(pageIdx) {
            $scope.dashboard.pages.splice(pageIdx, 1);
            if ($scope.dashboard.pages.length === 0) {
                $scope.createPage();
            }
            $scope.uiState.currentPageIdx = $scope.uiState.currentPageIdx ? Math.min($scope.uiState.currentPageIdx, $scope.dashboard.pages.length - 1) : $scope.dashboard.pages.length - 1;
            $scope.hasFilterTile = alreadyHaveGlobalFilters();
        };

        $scope.openCopySlideModal = function(pageIdx) {
            CreateModalFromTemplate('/templates/dashboards/copy-page-modal.html', $scope, 'CopyPageController', function(newScope) {
                newScope.pages = $scope.dashboard.pages;
                newScope.page = $scope.dashboard.pages[pageIdx];
                newScope.pageIdx = pageIdx;
                newScope.uiState = $scope.uiState;
                newScope.insightsMap = $scope.insightsMap;
                newScope.init();
            });
        };

        $scope.handlePageIndexChange = (previousIndex, currentIndex) => {
            if (previousIndex === $scope.uiState.currentPageIdx) {
                $scope.uiState.currentPageIdx = currentIndex;
            } else if (previousIndex < $scope.uiState.currentPageIdx && currentIndex >= $scope.uiState.currentPageIdx) {
                $scope.uiState.currentPageIdx--;
            } else if (previousIndex > $scope.uiState.currentPageIdx && currentIndex <= $scope.uiState.currentPageIdx) {
                $scope.uiState.currentPageIdx++;
            }
        };

        $scope.goToPage = (index) => {
            $scope.uiState.currentPageIdx = index;
            $scope.uiState.selectedTile = null;
        };

        $scope.goToNextPage = function() {
            this.goToPage(DashboardPageUtils.getNextPageIndex($scope.dashboard, $scope.uiState.currentPageIdx));
        };

        $scope.goToPreviousPage = function() {
            this.goToPage(DashboardPageUtils.getPreviousPageIndex($scope.dashboard, $scope.uiState.currentPageIdx));
        };

        $scope.addTileToPage = function(tile) {
            tile.backgroundOpacity = 1;
            tile.locked = false;
            tile.borderOptions = angular.copy(TileUtils.DEFAULT_BORDER_OPTIONS);
            tile.titleOptions = { ...tile.titleOptions, ...angular.copy(TileUtils.DEFAULT_TITLE_OPTIONS) };

            $scope.dashboard.pages[$scope.uiState.currentPageIdx].grid.tiles.push(tile);
            DSSVisualizationThemeUtils.applyToTile(tile, $scope.dashboard.theme);
            $scope.uiState.selectedTile = tile;
        };

        // TODO: factorize with tile-params directive
        $scope.getDefaultTileTitle = (tile, insight) => {
            if (!tile) {
                return '';
            }
            switch (tile.tileType) {
                case 'INSIGHT':
                    return (insight || {}).name;
                case 'TEXT':
                    return 'Text tile';
                case 'IMAGE':
                    return 'Image tile';
            }
        };

        $scope.copyURL = (urlToCopy) => {
            DashboardUtils.copyToClipboard(urlToCopy);
        };

        $scope.$on('$viewContentLoaded', () => {
            // Action requested from view mode
            if ($scope.editActionRequested) {
                switch($scope.editActionRequested.type) {
                    case 'create':
                        $scope.createPage();
                        break;
                    case 'remove':
                        if ($scope.editActionRequested.pageIdx !== undefined) {
                            $scope.removePage($scope.editActionRequested.pageIdx);
                        }
                        break;
                    case 'duplicate':
                        if ($scope.editActionRequested.pageIdx !== undefined) {
                            $scope.duplicatePage($scope.editActionRequested.pageIdx);
                        }
                        break;
                    case 'edit':
                        if ($scope.editActionRequested.pageIdx !== undefined) {
                            $scope.goToPage($scope.editActionRequested.pageIdx);
                        }
                        break;
                }
                $scope.editActionRequested = null;
            }
        });

        $scope.$on('$destroy', () => {
            if (unregisterDashboardSelectTileEvent != null) {
                unregisterDashboardSelectTileEvent();
            }
            if (unregisterDashboardUnselectTilesEvent != null) {
                unregisterDashboardUnselectTilesEvent();
            }
        });

        function alreadyHaveGlobalFilters() {
            return !!DashboardFilters.findFiltersTileInPage($scope.dashboard.pages[$scope.uiState.currentPageIdx]);
        }

        function retrieveInsightIdSetFromDashboard(dashboard) {
            const insightIdsSet = new Set();
            const stack = [];
            dashboard.pages.forEach(page => stack.push(...page.grid.tiles));
            while (stack.length) {
                const currentTile = stack.pop();
                if (currentTile.insightId != null) {
                    insightIdsSet.add(currentTile.insightId);
                }
                if (TileUtils.isGroupTile(currentTile) &&
                        currentTile.grid != null &&
                        currentTile.grid.tiles != null &&
                        currentTile.grid.tiles.length > 0) {
                    stack.push(...currentTile.grid.tiles);
                }
            }
            return insightIdsSet;
        }
    });
})();
