(function() {
    'use strict';

    angular.module('dataiku.dashboards', ['dataiku.constants']);
    angular.module('dataiku.dashboards.insights', []);
})();

;
(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('dashboardFilterDatasetColumnPicker', {
        bindings: {
            // inputs
            datasetSmartName: '<', // string
            columns: '<', // UsableColumn[]
            enableDatasetSelection: '<', // boolean
            // outputs
            selectColumns: '&', // ($columns) => void
            selectDataset: '&', // ($sourceObject, $sourceType, $selectedTile) => void
            cancelSelection: '&' // () => void
        },
        templateUrl:
      '/static/dataiku/js/dashboards/components/dashboard-filter-dataset-column-picker/dashboard-filter-dataset-column-picker.html',
        controller: function($stateParams, $scope, $rootScope, DashboardFilters, WT1) {
            const ctrl = this;
            ctrl.selectFromExistingTile = true;
            ctrl.filtersCompatibleTiles = DashboardFilters.getSlideFilterableTiles(
                $stateParams.pageId,
                $rootScope.topNav.item.data.pages
            );
            ctrl.selectedTile = undefined;
            ctrl.selectedTileId = undefined;
            ctrl.sourceObject = undefined;
            ctrl.isSourcePreselected = false;
            ctrl.sourceTypes = DashboardFilters.filtersSourceTypes;
            ctrl.source = ctrl.filtersCompatibleTiles.length ? ctrl.sourceTypes.TILE : ctrl.sourceTypes.DATASET;
            ctrl.selectedColumns = [];

            ctrl.$onChanges = (changes) => {
                if (changes.columns) {
                    ctrl.selectedColumns = [];
                }
            };

            $scope.$watch('$ctrl.sourceObject', function(nv) {
                if (!nv || (ctrl.source === ctrl.sourceTypes.TILE && ctrl.selectedTile == null)) {
                    return;
                }
                ctrl.selectDataset({ $sourceObject: nv, $sourceType: ctrl.source, $selectedTile: ctrl.selectedTile });
            });

            ctrl.setFilters = () => {
                ctrl.selectColumns({
                    $columns: ctrl.selectedColumns
                });
            };

            ctrl.cancel = () => {
                ctrl.cancelSelection();
            };
        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('dashboardFilterPanelCacheLoader', {
        bindings: {
            filterFacetsLoadingState: '<',
            direction: '<', // FiltersPanelDirection,
            raiseError: '&' // ({ $errorData }) => void,
        },
        templateUrl:
      '/static/dataiku/js/dashboards/components/dashboard-filter-panel-cache-loader/dashboard-filter-panel-cache-loader.component.html',
        controller: function(DataikuAPI, ProgressStackMessageBuilder, FiltersPanelDirection) {
            const ctrl = this;
            ctrl.FiltersPanelDirection = FiltersPanelDirection;
            ctrl.percentage = 0;
            ctrl.started = false;
            ctrl.$onChanges = function(changes) {
                if(changes.filterFacetsLoadingState && ctrl.filterFacetsLoadingState != null && !ctrl.filterFacetsLoadingState.hasResult) {
                    ctrl.percentage = ProgressStackMessageBuilder.getPercentage(ctrl.filterFacetsLoadingState.progress);
                    ctrl.started = ctrl.filterFacetsLoadingState.progress && ctrl.filterFacetsLoadingState.progress.states && ctrl.filterFacetsLoadingState.progress.states.length;
                    ctrl.stackMessage = ProgressStackMessageBuilder.build(ctrl.filterFacetsLoadingState.progress, false, ctrl.direction === FiltersPanelDirection.Vertical);
                }
            };

            ctrl.abort = function() {
                if (ctrl.filterFacetsLoadingState && ctrl.filterFacetsLoadingState.jobId) {
                    DataikuAPI.futures.abort(ctrl.filterFacetsLoadingState.jobId).catch(({ data, status, headers, config, statusText }) => {
                        ctrl.raiseError({ $errorData: { data, status, headers, config, statusText } });
                    });
                }
            };
        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('dashboardFilterPanelDatasetIcon', {
        bindings: {
            dataset: '<'
        },
        templateUrl: '/static/dataiku/js/dashboards/components/dashboard-filter-panel-dataset-icon/dashboard-filter-panel-dataset-icon.component.html',
        controller: function() {
        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('dashboardFilterPanel', {
        bindings: {
            page: '<', // DashboardPage
            filters: '<', // FrontendChartFilter[]
            filtersParams: '<',
            editable: '<', // boolean
            enableCrossFilters: '<', // boolean
            isDeactivated: '<', // boolean
            direction: '<', // FiltersPanelDirection
            isTileLocked: '<', // boolean
            isTileSelected: '<', // boolean
            theme: '<', // DSSVisualizationTheme
            filtersChange: '&', // ({ $filters }) => void
            filtersParamsChange: '&', // ({ $filtersParams }) => void
            raiseError: '&', // ({ $errorData }) => void,
            isDeactivatedChange: '&', // ({ $isDeactivated }) => void
            isTileLockedChange: '&' // () => void
        },
        templateUrl: '/static/dataiku/js/dashboards/components/dashboard-filter-panel/dashboard-filter-panel.component.html',
        controller: function($stateParams, $timeout, $scope, $rootScope, MonoFuture, CrossFiltersService, DashboardFilters, $location, ActiveProjectKey, WT1, DatasetUtils, ChartFilters, DashboardFiltersUrlParams, ClipboardUtils, ActivityIndicator, DataikuAPI, FiltersPanelPosition, FiltersPanelDirection, ChartsStaticData, ChartDataUtils, DashboardFilterPanelService, ChartFilterUtils) {
            const ctrl = this;
            const activeProjectKey = ActiveProjectKey.get();
            const DEFAULT_FILTERS_OPTIONS = { hasSamplingParamsChanged:  false, disableLoadingIndicator: false, canPropagageBeforeFacetUpdate: false, needAggregationRequest: true };

            let summary;
            let shaker;
            let dataSpec;
            let isDestroyed = false;
            const filtersPivotRequestMonoFuture = MonoFuture();


            ctrl.filtersApi;
            ctrl.canAllFiltersBeCleared;
            ctrl.usableColumns = [];
            ctrl.error = null;
            ctrl.validity = { valid: true };
            ctrl.filterFacets = [];
            ctrl.datasetLoadingError = false;
            ctrl.loading;
            ctrl.globalSummary = '';
            ctrl.hasAllColumnsSelected = false;
            ctrl.canModerateDashboards = $rootScope.projectSummary && $rootScope.projectSummary.canModerateDashboards;
            ctrl.hasInitializedFilters = false;
            ctrl.dataset = null;
            ctrl.schema = null;
            ctrl.areFiltersBeingCreated = false;
            ctrl.areFiltersBeingReordered = false;
            ctrl.sampleIsWholeDataset = null;
            ctrl.filterFacetsLoadingState = null;
            ctrl.filterFacetsLoadingError = null;
            ctrl.filterFacetsLoadingErrorMessage = '';
            ctrl.displayRefreshCacheButton = false;

            ctrl.FiltersPanelPosition = FiltersPanelPosition;
            ctrl.FiltersPanelDirection = FiltersPanelDirection;
            ctrl.warningBadge = ChartsStaticData.WARNING_BADGE;

            ctrl.$onChanges = function(changes) {
                if (changes.filters) {
                    if (ctrl.filters == null) {
                        ctrl.filters = [];
                    }
                    if (ctrl.filters.length === 0) {
                        ctrl.sampleIsWholeDataset = null;
                    }
                    synchronizeSelectedColumns();
                }

                if (changes.filtersParams) {
                    if (ctrl.filtersParams == null) {
                        ctrl.filtersParams = {};
                    }
                    const previousFiltersParams = changes.filtersParams.previousValue;
                    const currentFiltersParams = changes.filtersParams.currentValue;

                    if (!ctrl.filtersParams || !ctrl.filtersParams.datasetSmartName) {
                        ctrl.usableColumns = [];
                        ctrl.availableColumns = [];
                        ctrl.hasInitializedFilters = true;
                        return;
                    }

                    const datasetChanged = hasDatasetChanged(previousFiltersParams, currentFiltersParams);

                    if (changes.filtersParams.isFirstChange() || datasetChanged) {
                        ctrl.loading = true;
                        handleDatasetChange(ctrl.filtersParams.datasetSmartName).finally(() => {
                            try {
                                // we execute handleFiltersChange even if handleDatasetChange fails to display the error message in the filter panel.
                                ctrl.handleFiltersChange(ctrl.filters, { hasSamplingParamsChanged: !changes.filtersParams.isFirstChange(), disableLoadingIndicator: false, canPropagageBeforeFacetUpdate: false });
                            } catch {
                                // we ignore
                            }

                            $timeout(() => {
                                ctrl.samplingParams = DashboardFilters.getSamplingProps(ctrl.filtersParams);
                                ctrl.hasInitializedFilters = true;
                                ctrl.loading = false;
                            });
                        });
                    } else if (!changes.filtersParams.isFirstChange() && hasSamplingParamsChanged(previousFiltersParams, currentFiltersParams)) {
                        handleSamplingParamsChange(previousFiltersParams, currentFiltersParams);
                    }
                }
            };

            ctrl.$onDestroy = function() {
                unregisterCrossFilterEvent();
                filtersPivotRequestMonoFuture.destroy();
                isDestroyed = true;
            };

            ctrl.handleFiltersChange = (newFilters, requestOptions = DEFAULT_FILTERS_OPTIONS) => {
                // Check if component has been destroyed before executing the promise has method is included in promise chains
                if (isDestroyed) {
                    return;
                }
                const options = { ...DEFAULT_FILTERS_OPTIONS, ...requestOptions };
                if (requestOptions.needAggregationRequest == null) {
                    options.needAggregationRequest = true;
                }
                if (options.canPropagageBeforeFacetUpdate && options.needAggregationRequest) {
                    saveFiltersAndPropagateToDashboard(newFilters);
                }
                if (options.needAggregationRequest === false) {
                    saveFiltersInternally(newFilters);
                    return;
                }
                return getFilterFacets(newFilters, options.disableLoadingIndicator)
                    .then((filterFacets) => {
                        ctrl.filterFacets = filterFacets;

                        if (filterFacets == null) {
                            return;
                        }

                        const updatedFilters = newFilters.map((filter, index) => {
                            /*
                             * Cross filtering is only available in view mode and reordering is only available in edit mode, so the facet indexes always match the filter indexes.
                             */
                            if (filter.useMinimalUi) {
                                return filter;
                            }
                            return ChartFilters.updateFilterWithFacet(filter, filterFacets[index], options.hasSamplingParamsChanged);
                        });

                        if (ChartFilters.areFilterListsEqual(updatedFilters, ctrl.filters)) {
                            return;
                        }

                        if (!options.canPropagageBeforeFacetUpdate) {
                            saveFiltersAndPropagateToDashboard(updatedFilters);
                        } else {
                            saveFiltersInternally(updatedFilters);
                        }
                    });
            };


            ctrl.getPivotResponseOptions = function() {
                return {
                    projectKey: activeProjectKey,
                    dataSpec,
                    requestedSampleId: summary.requiredSampleId
                };
            };

            ctrl.handleCanAllFilteredBeClearedChange = function(canAllFiltersBeCleared) {
                ctrl.canAllFiltersBeCleared = canAllFiltersBeCleared;
            };

            ctrl.toggleFilterPanelActivation = function(isDeactivated) {
                ctrl.isDeactivated = isDeactivated;
                ctrl.globalSummary = ctrl.isDeactivated ? 'Filters tile disabled' : ChartFilters.getAllFiltersSummary(ctrl.filters);
                ctrl.isDeactivatedChange({ $isDeactivated: ctrl.isDeactivated });
            };

            ctrl.switchView = () => {
                const areFiltersBeingCreated = !ctrl.areFiltersBeingCreated;
                if (areFiltersBeingCreated) {
                    WT1.tryEvent('dashboard-filters-create-open', () => ({}));
                }
                $timeout(() => ctrl.areFiltersBeingCreated = areFiltersBeingCreated);
            };

            ctrl.handleCancelSelection = () => {
                WT1.tryEvent('dashboard-filters-create-cancel', () => ({}));
                ctrl.switchView();
            };

            ctrl.handleSelectColumns = (columns) => {
                WT1.tryEvent('dashboard-filters-create-success', () => ({
                    previouslyExistingFiltersCount: (ctrl.filters || []).length,
                    filtersCreatedCount: (ctrl.filters || []).length + (columns || []).length
                }));

                addSelectedColumns(columns);
                ctrl.switchView();
            };

            ctrl.handleSelectDataset = ({ sourceObject, sourceType, selectedTile }) => {
                let engineType;
                if (sourceType === DashboardFilters.filtersSourceTypes.DATASET) {
                    engineType = getSourceEngineType(sourceObject.object);
                } else {
                    engineType = selectedTile.engineType || getSourceEngineType(sourceObject.object);
                }

                const datasetSmartName = sourceObject.smartId;

                ctrl.filtersParamsChange({
                    $filtersParams: {
                        ...(ctrl.filtersParams || {}),
                        datasetSmartName,
                        engineType,
                        refreshableSelection: getDefaultRefreshableSelection()
                    }
                });
            };

            ctrl.copyUrl = function() {
                try {
                    const baseUrl = $location.absUrl().split('?')[0];
                    const filtersQueryStringValue = DashboardFiltersUrlParams.getFiltersQueryStringValue(ctrl.filters, ctrl.isDeactivated, ctrl.filterFacets);
                    const queryParams = { ...$location.search(), pageFilters: filtersQueryStringValue };
                    const url = `${baseUrl}?${Object.entries(queryParams).map(([key, value]) => `${key}=${encodeURIComponent(value)}`).join('&')}`;
                    /*
                     * Max URL size (defined in the nginx configuration) is 256K.
                     * A character is encoded on 1B, so the URL can be at most 256,000 characters long.
                     */
                    if (url.length > 256000) {
                        WT1.tryEvent('dashboard-filters-url-copy-error', () => ({
                            reason: 'EXCEEDS_MAX_URL_LENGTH',
                            urlLength: url.length
                        }));
                        throw 'query is too long';
                    }
                    ClipboardUtils.copyToClipboard(url);
                    WT1.tryEvent('dashboard-filters-url-copy', () => ({}));
                } catch (error) {
                    ActivityIndicator.error(`Filters cannot be copied to URL (${error})`, 5000);
                }
            };

            ctrl.emitPanelPositionChange = function(panelPosition) {
                ctrl.filtersParamsChange({ $filtersParams: { ...ctrl.filtersParams, panelPosition } });
            };

            ctrl.expandAll = function() {
                $timeout(() => ctrl.filters = ctrl.filters.map((filter) => ({ ...filter, folded: false })));
            };

            ctrl.collapseAll = function() {
                $timeout(() => ctrl.filters = ctrl.filters.map((filter) => ({ ...filter, folded: true })));
            };

            ctrl.clearAll = function() {
                if (!ctrl.filtersApi) {
                    return;
                }
                ctrl.filtersApi.clearAllFilters();
            };

            ctrl.removeAll = function() {
                ctrl.handleFiltersChange([]);
            };

            ctrl.handleAreFiltersBeingReorderedChange = function(areFiltersBeingReordered) {
                $timeout(() => ctrl.areFiltersBeingReordered = areFiltersBeingReordered);
            };

            ctrl.getEmptyStateText = () => {
                if (ctrl.enableCrossFilters) {
                    if (ctrl.canModerateDashboards) {
                        return 'Add filters from charts and datasets, or edit the dashboard to add default filters.';
                    } else {
                        return 'Add filters from charts and datasets.';
                    }
                } else {
                    if (ctrl.canModerateDashboards) {
                        return 'Cross-filtering is disabled. Enable it or add default filters.';
                    } else {
                        return 'Cross-filtering is disabled. Ask a dashboard owner to enable it or add default filters.';
                    }
                }
            };

            ctrl.shouldDisplayEmptyState = () => {
                return ctrl.filters.length === 0 && !ctrl.editable;
            };

            ctrl.openSamplingModal = function() {
                DashboardFilterPanelService.openSamplingModal(ctrl.dataset, ctrl.schema, ctrl.samplingParams)
                    .then(samplingParams => ctrl.filtersParamsChange({ $filtersParams: { ...ctrl.filtersParams, ...samplingParams } }));
            };

            ctrl.refreshSelection = function() {
                ctrl.filtersParams.refreshableSelection._refreshTrigger = Date.now();
                handleSamplingParamsChange(ctrl.filtersParams, ctrl.filtersParams, ctrl.canModerateDashboards);
            };

            function saveFiltersAndPropagateToDashboard(newFilters) {
                DashboardFilters.setGlobalFilters(newFilters, $stateParams.pageId);
                ctrl.filtersChange({ $filters: newFilters });
                if (newFilters.length === 0) {
                    ctrl.filtersParamsChange({ $filtersParams: { panelPosition: ctrl.filtersParams.panelPosition } });
                }
            }

            function saveFiltersInternally(newFilters) {
                // Useful for dashboard export to retrieve last filter version
                DashboardFilters.setGlobalFilters(newFilters, $stateParams.pageId);
                ctrl.filters = newFilters;
            }

            function handleSamplingParamsChange(previousSamplingParams, currentSamplingParams, saveDashboard = false) {
                WT1.tryEvent('dashboard-filters-sample-setting-update', () => ({
                    engineType: currentSamplingParams.engineType,
                    reuseFromExplore: false,
                    samplingMethod: currentSamplingParams.refreshableSelection.selection.samplingMethod,
                    recordsNumber: currentSamplingParams.refreshableSelection.selection.maxRecords,
                    targetRatio: currentSamplingParams.refreshableSelection.selection.targetRatio,
                    filtersNumber:
                    currentSamplingParams.refreshableSelection.selection.filter &&
                    currentSamplingParams.refreshableSelection.selection.filter.enabled &&
                    currentSamplingParams.refreshableSelection.selection.filter.uiData &&
                    currentSamplingParams.refreshableSelection.selection.filter.uiData.conditions
                        ? currentSamplingParams.refreshableSelection.selection.filter.uiData.conditions.length
                        : 0
                }));

                const engineHasChanged = currentSamplingParams.engineType !== (previousSamplingParams || {}).engineType;
                updateDatasetAndColumnsInfo(currentSamplingParams.datasetSmartName, currentSamplingParams.engineType, currentSamplingParams.refreshableSelection)
                    .then(() =>{
                        ctrl.samplingParams = DashboardFilters.getSamplingProps(ctrl.filtersParams);
                        if (engineHasChanged) {
                            migrateFiltersOnNewEngine(ctrl.filters).then(migratedFilters => ctrl.handleFiltersChange(migratedFilters, { hasSamplingParamsChanged: true, disableLoadingIndicator: false, canPropagageBeforeFacetUpdate: false }));
                        } else {
                            ctrl.handleFiltersChange(ctrl.filters, { hasSamplingParamsChanged: true, disableLoadingIndicator: false, canPropagageBeforeFacetUpdate: false });
                        }
                    })
                    .finally(()=> saveDashboard && $rootScope.$broadcast('saveDashboard'));
            };

            const unregisterCrossFilterEvent = $rootScope.$on('crossFiltersAdded', function(_, { filters: crossFilters, wt1Args }) {
                const usableColumns = summary ? summary.usableColumns : [];
                const { filters, hasChanged } = CrossFiltersService.applyCrossFiltersToFilters(crossFilters, ctrl.filters, ctrl.filterFacets, usableColumns);
                if (hasChanged) {
                    ctrl.handleFiltersChange(filters, { hasSamplingParamsChanged: false, disableLoadingIndicator: false, canPropagageBeforeFacetUpdate: true });
                    WT1.tryEvent('dashboards-create-cross-filter', () => ({
                        ...(wt1Args || {}),
                        columnTypes: (crossFilters || []).map(filter => filter.columnType)
                    }));
                }
            });

            function migrateFiltersOnNewEngine(filters) {
                const resolvedDataset = resolveDatasetFullName(ctrl.filtersParams.datasetSmartName, activeProjectKey);
                return DashboardFilters.hasCaseInsensitiveColumnNames(resolvedDataset.projectKey, resolvedDataset.datasetName, activeProjectKey).then(hasCaseInsensitiveColumnNames => {
                    return filters.map(filter => {
                        const { column, columnType } = ctrl.usableColumns.find(col => {
                            const filterColumnName = hasCaseInsensitiveColumnNames ? filter.column.toLowerCase() : filter.column;
                            const currentColumnName = hasCaseInsensitiveColumnNames ? col.column.toLowerCase() : col.column;
                            return filterColumnName === currentColumnName;
                        });
                        if (filter.columnType === columnType) {
                            return { ...filter, column, label : column };
                        }
                        return ChartFilters.createFilter({
                            column,
                            columnType,
                            isAGlobalFilter: true,
                            id: filter.id
                        });
                    });
                });
            }

            function getFilterFacets(filters, disableLoadingIndicator = false) {
                const alreadyUsedColumns = new Set(filters.map(({ column }) => column));
                ctrl.availableColumns = ctrl.usableColumns.filter(({ column }) => !alreadyUsedColumns.has(column));
                ctrl.filterFacetsLoadingError = null;
                ctrl.filterFacetsLoadingErrorMessage = '';

                ctrl.globalSummary = ChartFilters.getAllFiltersSummary(filters || []);
                if (!filters || !filters.length) {
                    // no filters, prevent calling executeFiltersPivotRequest and initialize empty data directly
                    return Promise.resolve([]);
                }

                // If there are only cross filters or custom filters, there is no facet to update
                const numberOfCrossFilters = filters.filter(filter => ChartFilterUtils.isCrossFilter(filter)).length;
                const numberOfCustomFilters = filters.filter(filter => ChartFilterUtils.isCustomFilter(filter)).length;
                if ((numberOfCrossFilters + numberOfCustomFilters) === filters.length) {
                    // Create fake facets for customFilters so we remove the loading indicator
                    return Promise.resolve(numberOfCustomFilters > 0 ? Array.from({ length: numberOfCustomFilters }, (_, index) => ({ filterIdx: index, values: [] })) : []);
                }

                if (!disableLoadingIndicator) {
                    ctrl.loading = true;
                }

                return DashboardFilters.executeFiltersPivotRequest(
                    activeProjectKey, summary || {}, filters, dataSpec, filtersPivotRequestMonoFuture
                )
                    .update(data => {
                        ctrl.filterFacetsLoadingState = data;
                    })
                    .then((data) => {
                        ctrl.filterFacetsLoadingState = data;
                        const pivotResponse = data.data.result.pivotResponse;
                        ctrl.sampleIsWholeDataset = pivotResponse.sampleMetadata ? pivotResponse.sampleMetadata.sampleIsWholeDataset : true;

                        const sampleUIData = ChartDataUtils.getSampleMetadataAndSummaryMessage(pivotResponse, 'Click to open sampling settings', 'Click to open sampling settings');
                        ctrl.samplingSummaryMessage = ctrl.editable ? sampleUIData.clickableSummaryMessage : sampleUIData.summaryMessage;

                        return pivotResponse.filterFacets;
                    })
                    .catch(({ data, status, headers, config, statusText }) => {
                        ctrl.filterFacetsLoadingError = { data, status, headers, config, statusText };
                        if (ChartDataUtils.isPivotRequestAborted(data)) {
                            // Manually aborted => do not report as error)
                            ctrl.filterFacetsLoadingErrorMessage = 'Filter panel construction was aborted, open the Sampling & Engine setting to update its configuration.';

                        } else {
                            ctrl.apiErrorAlert = ctrl.filterFacetsLoadingError.data;
                            ctrl.isCredentialError = ctrl.apiErrorAlert && ctrl.apiErrorAlert.code && (ctrl.apiErrorAlert.code==='ERR_CONNECTION_OAUTH2_REFRESH_TOKEN_FLOW_FAIL' || ctrl.apiErrorAlert.code==='ERR_CONNECTION_NO_CREDENTIALS');
                            ctrl.filterFacetsLoadingErrorMessage = ctrl.apiErrorAlert.message;
                            ctrl.displayRefreshCacheButton = 'com.dataiku.dip.pivot.backend.model.CorruptedDataException' === ctrl.apiErrorAlert.errorType;
                        }
                    }).finally(() => {
                        ctrl.filterFacetsLoadingState = null;
                        ctrl.loading = false;
                    });
            }

            // Clear column selection and apply tileParams one
            function synchronizeSelectedColumns() {
                if (ctrl.selectedColumns == null) {
                    ctrl.selectedColumns = new Set();
                }
                ctrl.selectedColumns.clear();
                ctrl.filters.forEach((filter) => {
                    ctrl.selectedColumns.add(filter.column);
                });
                ctrl.hasAllColumnsSelected = ctrl.usableColumns.length && ctrl.selectedColumns.size === ctrl.usableColumns.length;
            }

            function getSourceEngineType(filtersSource) {
                return filtersSource && DatasetUtils.isSQL(filtersSource) ? 'SQL' : 'LINO';
            }

            function addSelectedColumns(columns) {
                const newFilters = [...ctrl.filters];
                for (const column of columns) {
                    const columnType = ctrl.usableColumns.find(current => current.column === column).type;
                    const filter = ChartFilters.createFilter({
                        column,
                        columnType,
                        isAGlobalFilter: true
                    });
                    newFilters.push(filter);
                }
                // Save filters immediately so that they appear in the panel and then get their facets.
                saveFiltersInternally(newFilters);
                ctrl.handleFiltersChange(newFilters, { hasSamplingParamsChanged:  false, disableLoadingIndicator: true, canPropagageBeforeFacetUpdate: false });
            }

            function hasDatasetChanged(previousFiltersParams, currentFiltersParams) {
                const previousDatasetSmartName = previousFiltersParams && previousFiltersParams.datasetSmartName;
                const currentDatasetSmartName = currentFiltersParams && currentFiltersParams.datasetSmartName;
                return previousDatasetSmartName !== currentDatasetSmartName;
            }

            function hasSamplingParamsChanged(previousFiltersParams, currentFiltersParams) {
                // As we might abort the cache building process, we might need need to trigger the sampling change even if the sampling is the same when clicking the "Apply" button
                if (DashboardFilters.getSamplingRefreshTrigger(previousFiltersParams) !== DashboardFilters.getSamplingRefreshTrigger(currentFiltersParams)) {
                    return true;
                }
                const previousSamplingParams = previousFiltersParams && DashboardFilters.getSamplingProps(previousFiltersParams);
                const currentSamplingParams = currentFiltersParams && DashboardFilters.getSamplingProps(currentFiltersParams);
                return !angular.equals(previousSamplingParams, currentSamplingParams);
            }

            function handleDatasetChange(datasetSmartName) {
                // The active project key and the dataset project key can be different if the dataset has been shared from another project.
                const resolvedDataset = resolveDatasetFullName(datasetSmartName, activeProjectKey);

                const promises = [
                    updateDatasetAndColumnsInfo(datasetSmartName, ctrl.filtersParams.engineType, ctrl.filtersParams.refreshableSelection),
                    DataikuAPI.datasets.get(resolvedDataset.projectKey, resolvedDataset.datasetName, activeProjectKey)
                        .error(() => {
                            ctrl.datasetLoadingError = true;
                        })
                        .success(function(data) {
                            ctrl.dataset = data;
                            ctrl.schema = data.schema;
                        })
                ];
                return Promise.all(promises);
            }

            function updateDatasetAndColumnsInfo(datasetSmartName, engineType, refreshableSelection) {
                if (!datasetSmartName) {
                    shaker = null;
                    dataSpec = null;
                    summary = null;
                    ctrl.usableColumns = [];
                    return Promise.resolve();
                }
                return DashboardFilters.getShaker(activeProjectKey, datasetSmartName)
                    .then(_shaker => {
                        shaker = _shaker;
                        dataSpec = DashboardFilters.getDataSpec(activeProjectKey, datasetSmartName, engineType, refreshableSelection, shaker);
                        return DashboardFilters.fetchColumnsSummary(activeProjectKey, dataSpec, ctrl.filtersParams);
                    })
                    .then(({ summary: _summary, usableColumns: _usableColumns }) => {
                        summary = _summary;
                        ctrl.usableColumns = _usableColumns;
                    });
            }

            function getDefaultRefreshableSelection() {
                return {
                    _refreshTrigger: Date.now(),
                    selection: {
                        samplingMethod: 'FULL'
                    }
                };
            }
        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('dashboardGridBackground', {
        bindings: {
            columnNumber: '<', // number
            backgroundColor: '<', // string
            gridColor: '<', // string
            tileSpacing: '<', // number
            cellWidth: '<', // number
            cellHeight: '<', // number
            padding: '<' // number
        },
        templateUrl: '/static/dataiku/js/dashboards/components/dashboard-grid-background/dashboard-grid-background.component.html',
        controller: function() {
            const ctrl = this;
            ctrl.patternId = _.uniqueId('grid-pattern-');
            ctrl.verticalLines = [];

            ctrl.$onChanges = function() {
                updateGrid();
            };

            function updateGrid() {
                const newVerticalLines = [];
                if (ctrl.padding != null && ctrl.padding > 0) {
                    const x = ctrl.padding / 2;
                    newVerticalLines.push({
                        x1: x,
                        y1: 0,
                        x2: x,
                        y2: '100%',
                        strokeWidth: ctrl.padding
                    });
                }
                for (let i = 1; i < ctrl.columnNumber; i++) {
                    const x = i * ctrl.cellWidth + ctrl.padding - ctrl.tileSpacing / 2;
                    newVerticalLines.push({
                        x1: x,
                        y1: 0,
                        x2: x,
                        y2: '100%',
                        strokeWidth: ctrl.tileSpacing || 0.5
                    });
                }

                if (ctrl.padding != null && ctrl.padding > 0) {
                    const x = ctrl.columnNumber * ctrl.cellWidth + ctrl.padding / 2;
                    newVerticalLines.push({
                        x1: x,
                        y1: 0,
                        x2: x,
                        y2: '100%',
                        strokeWidth: ctrl.padding
                    });
                }

                ctrl.verticalLines = newVerticalLines;
                ctrl.gridWidth = ctrl.columnNumber * ctrl.cellWidth + ctrl.padding * 2 - ctrl.tileSpacing;
            }
        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('dashboardGrid', {
        bindings: {
            editable: '<', // boolean
            minHeight: '<', // number
            minWidth: '<', // number
            minNumberOfRows: '<', // number
            padding: '<', // number
            tileSpacing: '<', // number
            gridColor: '<', // string
            addExtraGridSpaceBelow: '<', // boolean
            backgroundColor: '<', // string
            columnNumber: '<', // number
            showGrid: '<', // boolean
            isInGroupTile: '<', // boolean
            tiles: '<', // DashboardTile[]
            page: '<', // DashboardPage
            preselectedTile: '<', // DashboardTile
            autoStackUp: '<', // boolean
            canExportDatasets: '<', // boolean
            insightsMap: '<', // { [insightId: string]: Insight }
            accessMap: '<', // { [insightId: string]: string }
            activeFilters: '<',
            filters: '<',
            dashboardTheme: '<',
            loadingStateChange: '&', // ({ $isLoading }) => void
            tilesChange: '&', // ({ $tiles }) => void
            filtersChange: '&', // ({ $filters }) => void
            filtersParamsChange: '&', // ({ $filtersParams }) => void,
            raiseError: '&', // ({ $errorData }) => void,
            isDeactivatedChange: '&', // ({ $isDeactivated }) => void
            preselectedTileChange: '&', // ({ $preselectedTile }) => void
            gridInit: '&', // ({$api: { reload: () => void }}),
            toggleGroupTileScrollBarDisplay: '&' // ({ $forceShow: boolean, $scrollTopOnHide: boolean }) => void
        },
        templateUrl: '/static/dataiku/js/dashboards/components/dashboard-grid/dashboard-grid.component.html',
        controller: function(DashboardFilters, $timeout, $rootScope, DashboardUtils, DashboardPageUtils, TileLoadingState, TileLoadingBehavior, $element, $scope, $q, ContextualMenu, TileUtils, LinkedList, DetectUtils, Debounce, WT1, Logger) {
            const ctrl = this;

            /*
             * Tile IDs are used to track the tiles, so they need to be defined before they are used by the ng-repeat.
             * To be sure it's the case, we keep a copy of the tile array (still referencing the original tiles) with their IDs and update it when the tiles change.
             */
            ctrl.tileWithIds = [];
            ctrl.gridContainer;
            ctrl.gridListRendered = false;
            ctrl.appConfig = $rootScope.appConfig;
            ctrl.isErrorMap = {};
            ctrl.hook = null;
            ctrl.isLoading = false;
            ctrl.cellWidth = null;
            ctrl.cellHeight = null;
            ctrl.currentGroupTileIdInEditMode = null;
            ctrl.gridMinHeight = null;
            ctrl.TileUtils = TileUtils;
            ctrl.tileMinSizeByTileId = {};

            let isTryingToDragLockedTile = false;
            let blockTileDeselection;
            let tileVisibilityObserver = null;
            const isInExport = getCookie('dku_graphics_export') === 'true';
            const nbSimultaneousLoading = $rootScope.appConfig.nbSimultaneousInsightLoading;
            let deferredLoadTileInProgress = false;
            let lockGridResize = false;
            let pendingResizeRequest = false;
            const selectedTileIds = new Set();
            const initializedTileIds = new Set();
            let scrollingContainer = null;
            let resizeObserver = null;
            let grid = null;
            // Flags used to track the selection mode of the tiles for WT1 events
            let hasUsedRectangleSelection = false;
            let hasUsedMultiSelectKey = false;

            /*
             * In export mode we wait for all the tiles to be loaded independently of their visibility
             * In normal mode we only load the visible tiles
             */
            if (isInExport) {
                $scope.$watch('$ctrl.hook.loadStates', triggerIncommingTileLoading, true);
            } else {
                tileVisibilityObserver = getTileVisibilityObserver();
            }

            ctrl.$onInit = function() {
                ctrl.hook = {
                    loadPromises : {},
                    reloadPromises : {},
                    loadStates: {},
                    visibleStates: {},
                    adjacentStates: {},
                    isErrorMap : ctrl.isErrorMap,
                    setErrorInDashboardPageScope : function(data, status, headers, config, statusText) {
                        ctrl.raiseError({ $errorData : { data, status, headers, config, statusText } });
                    },
                    editable: ctrl.editable
                };
                ctrl.gridInit({ $api: { reload: ctrl.reloadTiles, resizeGrid: resizeGridTree } });

                // The parent grid is in charge of resizing itself and then the child grid in group tiles
                if (!ctrl.isInGroupTile) {
                    $(window).on('resize.dashboard', handleResizeEvent);
                }
                initGrid();
            };

            ctrl.$onChanges = function(changes) {
                if (changes.tiles && ctrl.tiles != null) {
                    for (const tile of ctrl.tiles) {
                        tile.$tileId = tile.$tileId != null ? tile.$tileId : TileUtils.getUniqueTileId();
                    }
                    ctrl.tileWithIds = [...ctrl.tiles];
                    if (ctrl.gridListRendered) {
                        syncGridFromTiles();
                    }
                    ctrl.tileMinSizeByTileId = ctrl.tiles.reduce((acc, tile) => {
                        acc[tile.$tileId] = TileUtils.getTileMinSizeThreshold(tile);
                        return acc;
                    }, {});
                }
                if (changes.tileSpacing && ctrl.tileSpacing != null && selectedTileIds.size) {
                    updateSelectedTilesResizeHandles();
                }
                if (changes.preselectedTile &&
                    ctrl.preselectedTile != null &&
                    ctrl.tiles != null &&
                    ctrl.gridListRendered) {
                    if (ctrl.tiles.includes(ctrl.preselectedTile)) {
                        selectedTileIds.clear();
                        if (ctrl.isGroupTile && !ctrl.editable) {
                            $scope.$emit('dashboardEnterGroupTileEditMode', { grid });
                        }
                        $timeout(() => {
                            selectedTileIds.add(ctrl.preselectedTile.$tileId);
                            updateSelectedTilesResizeHandles();
                            ctrl.preselectedTileChange({ $preselectedTile: null });
                        });
                    } else if (ctrl.isInGroupTile && ctrl.editable) {
                        $scope.$emit('dashboardExitGroupTileEditMode', { grid });
                    }
                }
                if (changes.editable && !changes.editable.isFirstChange()) {
                    if (ctrl.gridListRendered) {
                        grid.setIsReadOnly(!ctrl.editable);
                    }
                    if (ctrl.hook != null) {
                        ctrl.hook.editable = ctrl.editable;
                    }
                    if (!ctrl.editable) {
                        selectedTileIds.clear();
                    }
                }
                if (ctrl.gridListRendered &&
                        ((changes.minHeight && !changes.minHeight.isFirstChange()) ||
                        (changes.tileSpacing && !changes.tileSpacing.isFirstChange()) ||
                        (changes.padding && !changes.padding.isFirstChange()) ||
                        (changes.minNumberOfRows && !changes.minNumberOfRows.isFirstChange()))) {
                    grid.options.cellSizeMatchCustomHeight = ctrl.minHeight + ctrl.tileSpacing - ctrl.padding * 2;
                    grid.options.numberOfCellsMatchingCustomHeight = ctrl.minNumberOfRows;
                    grid.options.offsetHeight = ctrl.isInGroupTile ? -ctrl.tileSpacing : 0;
                    grid.options.itemSpacing = ctrl.tileSpacing;
                    resizeGridTree();
                }

                if (changes.columnNumber && !changes.columnNumber.isFirstChange() && ctrl.gridListRendered) {
                    grid.resize(ctrl.columnNumber);
                    $timeout(function() {
                        angular.element(ctrl.gridContainer).scope().$broadcast('resize');
                        syncTilesFromGridDeletionsAndMoves();
                    }, 200);
                }

                if (changes.autoStackUp && !changes.autoStackUp.isFirstChange() && ctrl.gridListRendered) {
                    grid.toggleAutoStackUp(ctrl.autoStackUp);
                    syncTilesFromGridDeletionsAndMoves();
                }
            };

            $scope.$on('dashboardToggleTileGrouping', () => {
                if (ctrl.isInGroupTile || !ctrl.editable) {
                    return;
                }
                if (canGroupSelectedTiles()) {
                    groupSelectedTiles();
                } else if (hasSingleGroupTileSelected()) {
                    ungroupTile(getSelectedTiles()[0]);
                }
            });

            $scope.$on('dashboardUngroupTile', (_, { tile }) => {
                if (ctrl.isInGroupTile || !ctrl.editable) {
                    return;
                }
                ungroupTile(tile);
            });

            $scope.$on('dashboardStackUpGrid', (_, params) => {
                if (params.grid == null || params.grid !== grid) {
                    return;
                }
                stackTilesUp(params.isFromGroupTile || false);
            });

            $scope.$on('dashboardEnterGroupTileEditMode', (_, { grid }) => {
                if (ctrl.isInGroupTile || !ctrl.editable) {
                    return;
                }
                const groupTile = getParentGroupTileFromElement(grid.$element);
                if (groupTile != null) {
                    const element = DashboardPageUtils.getTileElementFromTileId($element, groupTile.$tileId);
                    selectTiles([groupTile], [element], false);
                    enterGroupTileEditModeIfInactive(groupTile.$tileId);
                }
            });

            $scope.$on('dashboardExitGroupTileEditMode', (_, { grid }) => {
                if (ctrl.isInGroupTile) {
                    return;
                }
                const groupTile = getParentGroupTileFromElement(grid.$element);
                if (groupTile != null) {
                    exitGroupTileEditModeIfActive(groupTile.$tileId);
                }
            });

            $scope.$on('dashboardSelectParentGroupTile', (_, { grid }) => {
                const groupTile = getParentGroupTileFromElement(grid.$element);
                if (groupTile != null) {
                    const element = DashboardPageUtils.getTileElementFromTileId($element, groupTile.$tileId);
                    selectTiles([groupTile], [element], false);
                    exitGroupTileEditModeIfActive(groupTile.$tileId);
                }
            });

            $scope.$on('dashboardLockResizeGridTree', () => {
                lockGridResize = true;
            });

            $scope.$on('dashboardUnlockResizeGridTree', () => {
                lockGridResize = false;
            });

            const unregisterDashboardSelectTilesEvent = $rootScope.$on('dashboardSelectTiles', (_, params) => {
                if (params.grid !== grid) {
                    // Unselect tiles from other grids
                    selectedTileIds.clear();
                }
            });

            const unregisterDashboardUnselectTilesEvent = $rootScope.$on('dashboardUnselectTiles', () => {
                exitGroupTileEditModeIfActive();
                selectedTileIds.clear();
                hasUsedRectangleSelection = false;
                hasUsedMultiSelectKey = false;
            });

            $scope.$on('dashboardRemoveFilterTile', () => {
                const filterTile = DashboardFilters.findFiltersTileList(ctrl.tiles);
                if (filterTile) {
                    ctrl.deleteTile(filterTile);
                }
            });

            $scope.$on('dashboardSaved', () => {
                if (ctrl.gridListRendered) {
                    grid.captureItemSizeAndPositionSnapshot();
                }
            });

            ctrl.isTileSelected = (tile) => selectedTileIds.has(tile.$tileId);

            Mousetrap.bind('backspace', function() {
                deleteSelectedTiles();
            });

            ctrl.handleIsTileLockedChange = function(tile) {
                toggleTilesLockedState([tile]);
            };

            ctrl.deleteTile = function(tile) {
                deleteTiles([tile]);
            };

            ctrl.reloadTiles = function() {
                Object.keys(ctrl.hook.loadStates).forEach(key => {
                    ctrl.hook.loadStates[key] = TileLoadingState.WAITING_FOR_RELOAD;
                    ctrl.isErrorMap[key] = false;
                });
                ctrl.loadTiles(DashboardPageUtils.getVisibleTilesToLoadOrReloadPromiseByTileId(ctrl.hook));
            };

            ctrl.loadTiles = function(tileLoadingPromiseByTileId) {
                // Compute how many tiles we can load given the number of already loading tiles
                const nbLoading = DashboardPageUtils.getNumberOfLoadingTiles(ctrl.hook);
                if (nbLoading > nbSimultaneousLoading) {
                    Logger.warn('More tiles than allowed are loading at the same time. There might be an issue.');
                }
                const remainingLoadingSlots = Math.max(0, nbSimultaneousLoading - nbLoading);

                // Build the list of tiles that we should load now.
                const tilesToLoad = DashboardPageUtils.getTilesToLoadOrderedList(tileLoadingPromiseByTileId, ctrl.hook, getTilesById());

                // If there are too many loading requests. Only keeping the first ones
                if (tilesToLoad.length > remainingLoadingSlots) {
                    tilesToLoad.length = remainingLoadingSlots;
                }

                if (tilesToLoad.length > 0) {
                    updateLoadingState(true);
                    for (const i in tilesToLoad) {
                        const tileId = tilesToLoad[i];
                        const d = $q.defer();
                        ctrl.hook.loadStates[tileId] = TileLoadingState.LOADING;
                        const markComplete = tileLoadingPromiseByTileId[tileId](d.resolve, d.reject) !== TileLoadingBehavior.DELAYED_COMPLETE;
                        const tileLoadingCallback = getTileLoadingCallback(tileLoadingPromiseByTileId, tileId, markComplete);
                        d.promise.then(tileLoadingCallback, tileLoadingCallback); // Don't try to reload a tile if an error appear the first time for now.
                    }
                } else {
                    completeTileLoading(tileLoadingPromiseByTileId, 1000);
                }
            };

            ctrl.openGridMenu = function($event) {
                if (!canStackGridTilesUp()) {
                    return;
                }

                const menu = new ContextualMenu({
                    template: '/templates/dashboards/grid-contextual-menu.html'
                });
                const newScope = $rootScope.$new();
                newScope.stackTilesUp = () => stackTilesUp(ctrl.isInGroupTile);
                menu.scope = newScope;
                menu.openAtXY($event.pageX, $event.pageY);
            };

            ctrl.openTileMenu = function($event, tile) {
                $event.stopPropagation();
                const menu = new ContextualMenu({
                    template: '/templates/dashboards/tile-contextual-menu.html'
                });
                // Don't display group tile context menu when the group tile is in edit mode
                if (TileUtils.isGroupTile(tile) && ctrl.currentGroupTileIdInEditMode && tile.$tileId === ctrl.currentGroupTileIdInEditMode) {
                    return;
                }
                const newScope = $rootScope.$new();
                newScope.tileCopy = angular.copy(tile);
                newScope.tile = tile;
                if (ctrl.isTileSelected(tile) && hasMultipleTilesSelected()) {
                    newScope.isMultiSelectionMenu = true;
                    newScope.openMoveCopyTileModal = openMoveCopyTileModal;
                    newScope.deleteSelectedTiles = deleteSelectedTiles;
                    newScope.canLockSelectedTiles = canLockSelectedTiles();
                    newScope.lockSelectedTiles = lockSelectedTiles;
                    newScope.canUnlockSelectedTiles = canUnlockSelectedTiles();
                    newScope.unlockSelectedTiles = unlockSelectedTiles;
                    newScope.canGroupSelectedTiles = canGroupSelectedTiles();
                    newScope.groupSelectedTiles = groupSelectedTiles;
                } else {
                    newScope.isMultiSelectionMenu = false;
                    newScope.openMoveCopyTileModal = openMoveCopyTileModal;
                    newScope.deleteTile = ctrl.deleteTile;
                    newScope.handleIsTileLockedChange = ctrl.handleIsTileLockedChange;
                    newScope.ungroupTile = ungroupTile;
                    newScope.canStackGroupTileChildrenUp = canStackGroupTileChildrenUp(tile);
                    newScope.stackGroupTileChildrenUp = stackGroupTileChildrenUp;
                }

                menu.scope = newScope;
                menu.openAtXY($event.pageX, $event.pageY);
            };

            ctrl.handleMainGridClick = function(event) {
                if (!ignoreMainGridClick(event)) {
                    deselectTiles();
                }
            };

            ctrl.handleTileClick = function(tile, event) {
                if (!ignoreTileSelectionClick(event)) {
                    const multiSelectKeyPressed = isMultiSelectKeyPressed(event);
                    if (isGroupTileEditionClick(tile, multiSelectKeyPressed)) {
                        handleGroupTileEditionClick(tile, event);
                    } else {
                        handleTileSelectionClick(tile, event, multiSelectKeyPressed);
                    }
                    event.stopPropagation();
                }
            };

            ctrl.handleTileMouseDown = function(tile, event) {
                if (!tile.locked || DashboardPageUtils.isPerformingDradAngDropWithinTile(event.target)) {
                    return;
                }
                event.preventDefault();
                const overlayTimeout = $timeout(() => {
                    isTryingToDragLockedTile = true;
                }, 50);

                const onMouseMove = () => {
                    if (isTryingToDragLockedTile) {
                        toggleLockedOverlayOnTile(tile, true);
                    }
                };

                const onMouseUp = () => {
                    toggleLockedOverlayOnTile(tile, false);
                    $timeout.cancel(overlayTimeout);
                    isTryingToDragLockedTile = false;
                    document.removeEventListener('mousemove', onMouseMove);
                    document.removeEventListener('mouseup', onMouseUp);
                };
                document.addEventListener('mousemove', onMouseMove);
                document.addEventListener('mouseup', onMouseUp);
            };

            ctrl.startRectangleSelection = (e) => {
                if (e == null) {
                    return;
                }

                // Allow lasso only in edit mode, on the main grid, when shift key is pressed
                if (!ctrl.editable
                    || ctrl.isInGroupTile
                    || !e.shiftKey
                    || e.target == null
                ) {
                    return;
                }
                // Only respond to left mouse button
                if (e.button !== 0) {
                    return;
                }

                // block text selection
                e.preventDefault();

                const gridElement = ctrl.gridContainer[0];
                const bounds = gridElement.getBoundingClientRect();
                const startX = e.clientX - bounds.left;
                const startY = e.clientY - bounds.top;
                const startScrollY = scrollingContainer.scrollTop;

                let selectionBox = document.createElement('div');
                selectionBox.className = 'selection-box';
                selectionBox.style.left = `${startX}px`;
                selectionBox.style.top = `${startY}px`;
                selectionBox.style.position = 'absolute';
                selectionBox.style.border = '2px dashed #3b99fc';
                selectionBox.style.backgroundColor = 'rgba(59, 153, 252, 0.2)';
                selectionBox.style.pointerEvents = 'none';
                selectionBox.style.zIndex = '100000';

                gridElement.appendChild(selectionBox);
                blockTileDeselection = true;

                const onMouseMove = (evt) => {
                    evt.preventDefault();
                    const x = evt.clientX - bounds.left;
                    const scrollY = scrollingContainer.scrollTop;
                    const y = evt.clientY - bounds.top + scrollY - startScrollY;

                    const width = Math.abs(x - startX);
                    const height = Math.abs(y - startY);

                    selectionBox.style.width = `${width}px`;
                    selectionBox.style.height = `${height}px`;
                    selectionBox.style.left = `${Math.min(x, startX)}px`;
                    selectionBox.style.top = `${Math.min(y, startY)}px`;
                };

                const onMouseUp = (evt) => {
                    document.removeEventListener('mousemove', onMouseMove);
                    document.removeEventListener('mouseup', onMouseUp);
                    evt.preventDefault();
                    if (!selectionBox) {
                        return;
                    }

                    // Determine selection area
                    const rect = selectionBox.getBoundingClientRect();
                    const tileWrappers = gridElement.querySelectorAll(':scope > .tile-wrapper');
                    if (tileWrappers && tileWrappers.length) {
                        const selectedTileWrappers = [...tileWrappers].filter(tileWrapper => {
                            const itemRect = tileWrapper.getBoundingClientRect();
                            // To select what is touching the selection area
                            return (
                                itemRect.right > rect.left &&
                                itemRect.left < rect.right &&
                                itemRect.bottom > rect.top &&
                                itemRect.top < rect.bottom
                            );
                            // To select only what is inside the selection area
                            /*
                             * return (
                             *     itemRect.right < rect.right &&
                             *     itemRect.left > rect.left &&
                             *     itemRect.top > rect.top &&
                             *     itemRect.bottom < rect.bottom
                             * );
                             */
                        });
                        const selectedTiles = selectedTileWrappers.map(selectedTileWrapper => DashboardPageUtils.getTileByElement(ctrl.tiles, selectedTileWrapper));

                        // Remove selection box
                        gridElement.removeChild(selectionBox);
                        selectionBox = null;

                        hasUsedRectangleSelection = true;
                        selectTiles(selectedTiles, selectedTileWrappers, false).then(() => {
                            blockTileDeselection = false;
                        });
                    }
                };

                // Add event listeners
                document.addEventListener('mousemove', onMouseMove);
                document.addEventListener('mouseup', onMouseUp);
            };

            ctrl.scrollToTile = function(tile) {
                return ctrl.editable && tile.$added && ctrl.isTileSelected(tile) && ctrl.currentGroupTileIdInEditMode == null ;
            };

            function stackGroupTileChildrenUp(groupTile) {
                const tileElement = DashboardPageUtils.getTileElementFromTileId($element, groupTile.$tileId);
                const groupTileGridElement = tileElement.find('.dashboard-grid');
                $scope.$broadcast('dashboardStackUpGrid', { grid: groupTileGridElement.getGridListInstance(), isFromGroupTile: true });
            }

            function canStackGroupTileChildrenUp(groupTile) {
                if (ctrl.autoStackUp || !TileUtils.isGroupTile(groupTile)) {
                    return false;
                }
                const tileElement = DashboardPageUtils.getTileElementFromTileId($element, groupTile.$tileId);
                const groupTileGridElement = tileElement.find('.dashboard-grid');
                const groupTileGrid = groupTileGridElement.getGridListInstance();
                return groupTileGrid != null && groupTileGrid.canStackItemsUp();
            }

            function canStackGridTilesUp() {
                if (ctrl.autoStackUp) {
                    return false;
                }
                return grid.canStackItemsUp();
            }

            function stackTilesUp(isFromGroupTile) {
                if (isFromGroupTile) {
                    WT1.event('dashboard-stack-up-group-tile');
                } else {
                    WT1.event('dashboard-stack-up-dashboard-page');
                }
                grid.stackItemsUp();
                syncTilesFromGridDeletionsAndMoves();
            }

            function isGroupTileEditionClick(tile, isMultiSelectKeyPressed) {
                return !isMultiSelectKeyPressed &&
                    ctrl.isTileSelected(tile) &&
                    TileUtils.isGroupTile(tile) &&
                    selectedTileIds.size === 1 &&
                    !blockTileDeselection; // To avoid entering group tile edition mode after resizing it
            }

            function handleTileSelectionClick(tile, event, isMultiSelectKeyPressed) {
                exitGroupTileEditModeIfActive();
                const element = DashboardPageUtils.getTileElementFromTileId($element, tile.$tileId);
                selectTiles([tile], [element], ctrl.isInGroupTile ? false : isMultiSelectKeyPressed);
            }

            function handleGroupTileEditionClick(tile, event) {
                if (isGroupTileInEditMode(tile.$tileId)) {
                    /**
                     * Here the click is on an empty space of a group tile grid while the group tile is in edit mode
                     * - unselect its child tiles
                     * - exit its edit mode
                     * - select the group tile itself
                     */
                    exitGroupTileEditModeIfActive();
                    const element = DashboardPageUtils.getTileElementFromTileId($element, tile.$tileId);
                    selectTiles([tile], [element], false);
                } else {
                    /*
                     * Here the group tile is already selected but not in edit mode
                     * 1. If the click is on one of its child tiles:
                     *   - Enter group tile edit mode
                     *   - Select the child tile
                     * 2. If the click is on an empty space of a group tile grid
                     *   - Only enter group tile edit mode
                     */
                    enterGroupTileEditModeIfInactive(tile.$tileId);
                    const childTileWrapper = DashboardPageUtils.getTileWrapperFromElement(event.target);
                    const childTile = DashboardPageUtils.getTileByElement(tile.grid.tiles, childTileWrapper);
                    if (childTile != null) {
                        ctrl.preselectedTileChange({ $preselectedTile: childTile });
                    }
                }
            }

            function syncGridFromTiles() {
                return $timeout(() => {
                    grid.synchronizeWithDOM();
                    resetTileVisibilityObserver();
                    if (ctrl.tiles == null || !ctrl.tiles.length) {
                        updateLoadingState(false);
                        return;
                    }
                    const hasUninitializedTiles = initializeTiles();
                    // Grid might have attributed a position to newly added tiles
                    if (hasUninitializedTiles) {
                        syncTilesFromGridDeletionsAndMoves();
                    }
                    updateSelectedTilesResizeHandles();
                });
            }

            function initializeTiles() {
                let hasUninitializedTiles = false;

                ctrl.tiles.forEach(tile => {
                    if (!initializedTileIds.has(tile.$tileId)) {
                        tile.$added = true;
                        hasUninitializedTiles = true;
                        initializedTileIds.add(tile.$tileId);
                    }

                    if (tileVisibilityObserver) {
                        const tileElement = DashboardPageUtils.getTileElementFromTileId($element, tile.$tileId);
                        observeTileVisibilityOnceItsLoadingDataIsReady(tileVisibilityObserver, tile, tileElement);
                    }
                });

                return hasUninitializedTiles;
            }

            function resetTileVisibilityObserver() {
                if (tileVisibilityObserver) {
                    tileVisibilityObserver.disconnect();
                }
            }

            function syncTilesFromGridDeletionsAndMoves() {
                return $timeout(() => {
                    if (grid != null) {
                        const items = grid.gridList.items;
                        const itemById = new Map(items.map(item => [item.id, item]));

                        updateTilesPosition(itemById);
                        const tileIdsToRemoveSet = getTileIdsToRemoveSet(itemById);
                        cleanIndexesFromRemovedTiles(tileIdsToRemoveSet);
                        const remainingTiles = ctrl.tiles.filter(tile => !tileIdsToRemoveSet.has(tile.$tileId));
                        ctrl.tilesChange({ $tiles: remainingTiles });
                        $scope.$emit('dashboardSyncModelsDone');
                    }
                });
            };

            function updateTilesPosition(itemById) {
                ctrl.tiles.forEach((tile) => {
                    if (itemById.has(tile.$tileId)) {
                        const item = itemById.get(tile.$tileId);
                        updateTilePositionFromGridItem(tile, item);
                    }
                });
            }

            function updateTilePositionFromGridItem(tile, item) {
                tile.box.left = item.x;
                tile.box.top = item.y;
                tile.box.width = item.w;
                tile.box.height = item.h;
            }

            function getTileIdsToRemoveSet(itemById) {
                const tileIdsToRemove = new Set();
                ctrl.tiles.forEach((tile) => {
                    if (!tile.$new && !itemById.has(tile.$tileId)) {
                        tileIdsToRemove.add(tile.$tileId);
                    }
                    delete tile.$new;
                });
                return tileIdsToRemove;
            }

            function cleanIndexesFromRemovedTiles(tileIdsToRemoveSet) {
                tileIdsToRemoveSet.forEach((tileId) => {
                    initializedTileIds.delete(tileId);
                    delete ctrl.hook.loadPromises[tileId];
                    delete ctrl.hook.reloadPromises[tileId];
                    delete ctrl.hook.loadStates[tileId];
                    delete ctrl.hook.visibleStates[tileId];
                    delete ctrl.hook.adjacentStates[tileId];
                });
            }

            function updateSelectedTilesResizeHandles() {
                selectedTileIds.forEach(tileId => {
                    const tileElement = DashboardPageUtils.getTileElementFromTileId($element, tileId);
                    const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(tileElement);
                    DashboardPageUtils.positionResizeHandles(tileWrapper, ctrl.tileSpacing);
                });
            }

            function isGroupTileInEditMode(tileId) {
                return ctrl.currentGroupTileIdInEditMode === tileId;
            }

            function hasGroupTileInEditMode() {
                return ctrl.currentGroupTileIdInEditMode != null;
            }

            function enterGroupTileEditModeIfInactive(tileId) {
                if (isGroupTileInEditMode(tileId)) {
                    return;
                }
                if (hasGroupTileInEditMode()) {
                    exitGroupTileEditModeIfActive();
                }
                const tileElement = DashboardPageUtils.getTileElementFromTileId($element, tileId);
                const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(tileElement);
                tileWrapper.addClass('tile-wrapper--no-resize-handle');
                ctrl.currentGroupTileIdInEditMode = tileId;
                grid.blockDragging(tileElement);
            }

            function exitGroupTileEditModeIfActive() {
                if (ctrl.currentGroupTileIdInEditMode == null) {
                    return;
                }
                const groupTile = getTilesById()[ctrl.currentGroupTileIdInEditMode];
                if (groupTile != null && !groupTile.locked) {
                    const tileElement = DashboardPageUtils.getTileElementFromTileId($element, ctrl.currentGroupTileIdInEditMode);
                    const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(tileElement);
                    tileWrapper.removeClass('tile-wrapper--no-resize-handle');
                    grid.unblockDragging(DashboardPageUtils.getTileElementFromTileId($element, ctrl.currentGroupTileIdInEditMode));
                }
                ctrl.currentGroupTileIdInEditMode = null;
            }

            function canLockSelectedTiles() {
                return getSelectedTiles().some(tile => !tile.locked);

            }
            function lockSelectedTiles() {
                toggleTilesLockedState(getSelectedTiles(), true);
            }
            function canUnlockSelectedTiles() {
                return getSelectedTiles().some(tile => tile.locked);

            }
            function unlockSelectedTiles() {
                toggleTilesLockedState(getSelectedTiles(), false);
            }

            function toggleTilesLockedState(tiles, forcedLockedState = null) {
                tiles.forEach(tile => {
                    const el = DashboardPageUtils.getTileElementFromTileId($element, tile.$tileId);
                    const locked = forcedLockedState != null ? forcedLockedState : !tile.locked;
                    tile.locked = locked;
                    if (locked) {
                        grid.lockItem(el);
                    } else {
                        grid.unlockItem(el);
                        // Need to reposition resize handles on unlock
                        const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(el);
                        DashboardPageUtils.positionResizeHandles(tileWrapper, ctrl.tileSpacing);
                    }
                });
                syncTilesFromGridDeletionsAndMoves();
            }

            function hasSingleGroupTileSelected() {
                return selectedTileIds.size === 1 && TileUtils.isGroupTile(getSelectedTiles()[0]);
            }

            function canGroupSelectedTiles() {
                return selectedTileIds.size > 0 && TileUtils.canGroupTiles(getSelectedTiles());
            }

            function groupSelectedTiles() {

                const tilesToGroup = getSelectedTiles();
                const groupTile = TileUtils.createGroupTile(tilesToGroup, { useDashboardSpacing: true, tileSpacing: ctrl.tileSpacing, backgroundColor: ctrl.backgroundColor });

                DashboardUtils.sendWT1TileCreation(groupTile.tileType, {
                    triggeredFrom: 'dashboard-multi-selection',
                    hasUsedRectangleSelection: hasUsedRectangleSelection,
                    hasUsedMultiSelectKey: hasUsedMultiSelectKey,
                    childTileCount: selectedTileIds.size
                });

                deselectTiles();
                tilesToGroup.forEach(tile => {
                    const el = $('[data-id=' + tile.$tileId + ']');
                    grid.deleteItem($(el));
                });
                ctrl.tiles.push(groupTile);
                syncTilesFromGridDeletionsAndMoves();
                if (ctrl.preselectedTileChange) {
                    ctrl.preselectedTileChange({ $preselectedTile: groupTile });
                }
            }

            function ungroupTile(groupTile) {
                if (!TileUtils.isGroupTile(groupTile)) {
                    return;
                }

                if (groupTile.locked) {
                    const el = DashboardPageUtils.getTileElementFromTileId($element, groupTile.$tileId);
                    groupTile.locked = false;
                    grid.unlockItem(el);
                }

                // remove group tile
                const el = $('[data-id=' + groupTile.$tileId + ']');
                grid.deleteItem($(el));

                // move up child tiles to parent
                const newTiles = TileUtils.adaptChildTilesToMainGrid(groupTile);
                ctrl.tiles.push(...newTiles);
                syncTilesFromGridDeletionsAndMoves();
            };

            function openMoveCopyTileModal(tile) {
                $scope.$emit('dashboardOpenMoveCopyTileModal', { tile });
            }

            function hasMultipleTilesSelected(){
                return selectedTileIds.size > 1;
            }

            function getSelectedTiles() {
                return ctrl.tiles.filter(tile => selectedTileIds.has(tile.$tileId));
            }

            function deleteTiles(tiles) {
                tiles.forEach(tile => {
                    const el = $('[data-id=' + tile.$tileId + ']');
                    grid.deleteItem($(el));
                });
                syncTilesFromGridDeletionsAndMoves();
                deselectTiles();
            };

            function deleteSelectedTiles() {
                deleteTiles(getSelectedTiles());
            }

            ctrl.getBackgroundColor = (opacity, backgroundColor) => {
                if (backgroundColor == null) {
                    backgroundColor = '#ffffff';
                }
                const opacityHex = Math.round((opacity != null ? opacity : 1) * 255).toString(16).padStart(2, '0');
                return `${backgroundColor}${opacityHex}`;
            };

            $scope.$on('$destroy', function() {
                $(window).off('resize.dashboard', handleResizeEvent);
                Mousetrap.unbind('backspace');

                if (tileVisibilityObserver != null) {
                    tileVisibilityObserver.disconnect();
                }
                if (resizeObserver != null) {
                    resizeObserver.disconnect();
                }
                if (unregisterDashboardUnselectTilesEvent != null) {
                    unregisterDashboardUnselectTilesEvent();
                }
                if (unregisterDashboardSelectTilesEvent != null) {
                    unregisterDashboardSelectTilesEvent();
                }
                if (scrollingContainer != null) {
                    scrollingContainer.removeEventListener('wheel', handleShiftWheelScroll, { passive: false });
                }
                if (ctrl.gridListRendered) {
                    grid.destroy();
                }
            });

            function handleResizeEvent(event) {
                if (event.detail && event.detail.skipInDashboards || ctrl.gridContainer == null) {
                    return;
                }
                if (!event.originalEvent || !event.originalEvent.data || !event.originalEvent.data.propagate) {
                    event.stopPropagation();
                    event.stopImmediatePropagation();
                    resizeGridTree();
                }
            }

            function resizeGridTree() {
                if (lockGridResize) {
                    pendingResizeRequest = true;
                    return;
                }
                lockGridResize = true;

                // block css transition on tile size change
                DashboardPageUtils.getTileWrappers($element).addClass('tile-wrapper--no-animation');
                return reflowGridTreeBFSAsync().then(() => {
                    const customEvent = new Event('resize');
                    customEvent.data = { propagate: true };

                    window.dispatchEvent(customEvent);

                    // set back css transition on tile size change
                    DashboardPageUtils.getTileWrappers($element).removeClass('tile-wrapper--no-animation');
                    lockGridResize = false;
                    if (pendingResizeRequest) {
                        pendingResizeRequest = false;
                        resizeGridTree();
                    }
                });
            }

            function reflowGridTreeBFSAsync() {
                if (ctrl.gridContainer == null) {
                    return Promise.resolve();
                }

                const queue = new LinkedList([ctrl.gridContainer]);

                function processNext() {
                    if (!queue.length) {
                        return Promise.resolve();
                    }

                    const currentGridElement = queue.shift();

                    return $timeout(() => {
                        currentGridElement.getGridListInstance().reflow();
                    }).then(() => {
                        currentGridElement.find('.dashboard-grid:not(:has(.dashboard-grid))').each((_, element) => {
                            queue.push($(element));
                        });

                        return processNext();
                    });
                }

                return processNext();
            }

            function selectTiles(tiles, tileElements, isMultiSelectKeyPressed = false) {
                if (tiles == null || tiles.length === 0 || tileElements == null || tileElements.length !== tiles.length) {
                    return Promise.resolve();
                }
                tiles.forEach((tile, i) => {
                    const tileElement = tileElements[i];
                    const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(tileElement);
                    if (ctrl.isTileSelected(tile)) {
                        if (tileWrapper != null && !tileWrapper.hasClass('ui-draggable-dragging') && !tileWrapper.hasClass('ui-resizable-resizing')) {
                            if (!tile.$showLockedOverlay) {
                                tile.$showEditOverlay = true;
                            }
                        }
                    }
                    DashboardPageUtils.positionResizeHandles(tileWrapper, ctrl.tileSpacing);
                });
                return $timeout(() => {
                    hasUsedMultiSelectKey = isMultiSelectKeyPressed;
                    const tilesById = getTilesById();
                    if (tiles.length === 1) {
                        if (isMultiSelectKeyPressed) {
                            // it might unselect tile if already selected
                            if (selectedTileIds.has(tiles[0].$tileId)) {
                                selectedTileIds.delete(tiles[0].$tileId);
                            } else {
                                selectedTileIds.add(tiles[0].$tileId);
                            }
                        } else {
                            // Here one single tile is selected by a single click
                            hasUsedRectangleSelection = false;
                            selectedTileIds.clear();
                            selectedTileIds.add(tiles[0].$tileId);
                        }

                        if (selectedTileIds.size === 0) {
                            $rootScope.$emit('dashboardUnselectTiles');
                        } else {
                            $rootScope.$emit('dashboardSelectTiles', { tiles: Array.from(selectedTileIds).map(id => tilesById[id]), grid });
                        }
                    } else {
                        selectedTileIds.clear();
                        tiles.forEach(tile => selectedTileIds.add(tile.$tileId));
                        $rootScope.$emit('dashboardSelectTiles', { tiles, grid });
                    }
                });
            };

            function deselectTiles() {
                if (blockTileDeselection) {
                    return;
                }
                $rootScope.$emit('dashboardUnselectTiles');
            };

            function ignoreTileSelectionClick(event){
                return (!ctrl.editable && ctrl.isInGroupTile) ||
                    event.target.hasAttribute('dashboard-no-select') ||
                    event.target.closest('[dashboard-no-select]') !== null;
            }

            function isMultiSelectKeyPressed(event) {
                return DetectUtils.getOS() === 'macos' ? !!event.metaKey : !!event.ctrlKey;
            }

            function ignoreMainGridClick(event){
                return ctrl.isInGroupTile || isMultiSelectKeyPressed(event) ||
                    (!event.target.classList.contains('grid-display') && !event.target.classList.contains('dashboard-grid'));
            }

            function toggleLockedOverlayOnAllLockedTiles(showLockedOverlay, ignoreGroupTiles = false) {
                $timeout(() => {
                    ctrl.tiles.forEach(tile => {
                        if (!tile.locked) {
                            return;
                        }
                        if (TileUtils.isGroupTile(tile) && ignoreGroupTiles) {
                            return;
                        }
                        if (showLockedOverlay) {
                            tile.$showEditOverlay = false;
                        }
                        tile.$showLockedOverlay = showLockedOverlay;
                    });
                });
            }

            function toggleLockedOverlayOnTile(tile, showLockedOverlay) {
                if (showLockedOverlay === tile.$showLockedOverlay) {
                    return;
                }
                $timeout(() => {
                    if (showLockedOverlay) {
                        tile.$showEditOverlay = false;
                    }
                    tile.$showLockedOverlay = showLockedOverlay;
                });
            }

            function observeTileVisibilityOnceItsLoadingDataIsReady(tileVisibilityObserver, tile, el) {
                if (ctrl.hook.loadPromises && ctrl.hook.loadStates[tile.$tileId] != null) {
                    tileVisibilityObserver.observe(el.get(0));
                } else {
                    const unregisterLoadPromiseWatcher = $scope.$watch('$ctrl.hook.loadStates', function(newValue) {
                        if (newValue && newValue[tile.$tileId] != null) {
                            tileVisibilityObserver.observe(el.get(0));
                            unregisterLoadPromiseWatcher();
                        }
                    }, true);
                }
            }

            function getScrollingContainer() {
                if (ctrl.isInGroupTile) {
                    return $element.closest('.dashboard-tile__group-tile-container')[0];
                } else {
                    return document.querySelector('.dashboard-export-page-wrapper');
                }
            }

            function handleShiftWheelScroll(event) {
                if (event.shiftKey && scrollingContainer != null) {
                    event.preventDefault(); // Prevent horizontal scrolling
                    scrollingContainer.scrollBy(0, event.deltaY); // Scroll vertically
                }
            }

            const dispatchDebouncedResize = Debounce().withDelay(50, 50).wrap(() => {
                $timeout(() => {
                    window.dispatchEvent(new Event('resize'));
                });
            });

            function initGrid() {
                ctrl.gridContainer = $($element.find('.dashboard-grid').addBack('.dashboard-grid')); //addBack to add self
                /*
                 * Instantiating gridList
                 */
                const flashItems = function(items) {
                    // Hack to flash changed items visually
                    for (let i = 0; i < items.length; i++) {
                        (function(el) {
                            el.addClass('changed');
                            setTimeout(function() {
                                el.removeClass('changed');
                            }, 0);
                        })(items[i].$element);
                    }
                };
                $timeout(() => {
                    scrollingContainer = getScrollingContainer();
                    grid = ctrl.gridContainer.gridList(
                        {
                            lanes: ctrl.columnNumber,
                            cellSizeMatchCustomHeight: ctrl.minHeight != null ? ctrl.minHeight + ctrl.tileSpacing - ctrl.padding * 2: false,
                            numberOfCellsMatchingCustomHeight: ctrl.minNumberOfRows != null ? ctrl.minNumberOfRows : false,
                            addExtraGridSpaceBelow: ctrl.addExtraGridSpaceBelow,
                            autoStackUp: ctrl.autoStackUp,
                            offsetHeight: ctrl.isInGroupTile ? - ctrl.tileSpacing : 0,
                            // In group tiles the drop zone is the group tile container and not the group tile grid as its scrolling zone might overflow on the main grid
                            dropZone: ctrl.isInGroupTile ? DashboardPageUtils.getTileWrapperFromElement($element) : null,
                            onCellSizeChange: (cellWidth, cellHeight) => {
                                ctrl.cellWidth = cellWidth;
                                ctrl.cellHeight = cellHeight;
                            },
                            onChange: function(changedItems) {
                                flashItems(changedItems);
                            },
                            canDragItemOut: (draggedItem) => {
                                const tile = DashboardPageUtils.getTileByElement(ctrl.tiles, draggedItem.$element);
                                return !TileUtils.isGroupTile(tile) && !TileUtils.isFilterTile(tile);
                            },
                            onBeforeDragItemOut: (_, targetGrid) => {
                                targetGrid.setIsReadOnly(false);
                            },
                            onAfterDragItemOut: () => {
                                if (ctrl.isInGroupTile) {
                                    grid.setIsReadOnly(true);
                                    ctrl.toggleGroupTileScrollBarDisplay({ $forceShow: false, $scrollTopOnHide: true });
                                }
                                disableTileDropIntoGroupTiles();
                            },
                            onBeforeDragItemIn: (draggedItem, sourceGrid) => {
                                // The tile spacing of target grid can be different from the source grid
                                DashboardPageUtils.positionResizeHandles(draggedItem.$element, ctrl.tileSpacing);
                                draggedItem.$element.css('padding', `${ctrl.tileSpacing / 2}px`);

                                if (ctrl.isInGroupTile) {
                                    ctrl.toggleGroupTileScrollBarDisplay({ $forceShow: true });
                                }

                                const sourceGroupTile = getParentGroupTileFromElement(sourceGrid.$element);
                                if (sourceGroupTile == null) {
                                    return;
                                }
                                const groupTileElement = DashboardPageUtils.getTileElementFromTileId($element, sourceGroupTile.$tileId);

                                // Need to lock source group tile to prevent it from moving when positionning element dragged from it
                                if (!sourceGroupTile.locked) {
                                    grid.lockItem(groupTileElement);
                                }
                            },
                            onAfterDragItemIn: (_, sourceGrid) => {
                                const sourceGroupTile = getParentGroupTileFromElement(sourceGrid.$element);
                                if (sourceGroupTile == null) {
                                    return;
                                }
                                const groupTileElement = DashboardPageUtils.getTileElementFromTileId($element, sourceGroupTile.$tileId);

                                // Can unlock back the source group tile
                                if (!sourceGroupTile.locked) {
                                    grid.unlockItem(groupTileElement);
                                }
                                enableTileDropIntoGroupTiles();
                            },
                            onDropItemOut: (draggedItem) => {
                                const tile = DashboardPageUtils.getTileByElement(ctrl.tiles, draggedItem.$element);
                                selectedTileIds.clear();
                                // Cleaning work and tile removal done in drag stop callback method below
                                return { tile };
                            },
                            onDropItemIn: (draggedItem, _, dropOutData) => {
                                disableTileDropIntoGroupTiles();

                                if (ctrl.isInGroupTile) {
                                    ctrl.toggleGroupTileScrollBarDisplay({ $forceShow: false });
                                }

                                // Need the tile to have a new object reference so it gets selected after Angular rendering by calling preselectedTileChange
                                const tile = { ...dropOutData.tile };
                                tile.box.left = draggedItem.x;
                                tile.box.top = draggedItem.y;
                                tile.box.width = draggedItem.w;
                                tile.box.height = draggedItem.h;

                                // AngularJS will recreate the tile element and add it to the DOM itself so it can be tracked by its digest cycle
                                draggedItem.$element.remove();
                                ctrl.tilesChange({ $tiles: [...ctrl.tiles, tile] });
                                $timeout(() => {
                                    if (ctrl.isInGroupTile) {
                                        $scope.$emit('dashboardEnterGroupTileEditMode', { grid });
                                    } else {
                                        exitGroupTileEditModeIfActive(ctrl.isInGroupTile);
                                    }
                                    ctrl.preselectedTileChange({ $preselectedTile: tile });
                                });

                            },
                            direction: 'vertical',
                            readOnly: !ctrl.editable,
                            isChildGrid: ctrl.isInGroupTile,
                            itemSpacing: ctrl.tileSpacing,
                            scrollingContainer: $(scrollingContainer)
                        },
                        {
                            start: function(event) {
                                $rootScope.$broadcast('dashboardLockResizeGridTree');
                                if (event.shiftKey || DashboardPageUtils.isPerformingDradAngDropWithinTile(event.target)) {
                                    return false;
                                }
                                $(event.target).addClass('tile-wrapper--no-animation');
                                const draggedTile = DashboardPageUtils.getTileByElement(ctrl.tiles, event.target);

                                exitGroupTileEditModeIfActive();
                                selectTiles([draggedTile], [event.target], false);

                                // Avoid group tile to enter in edit mode after performing the D&D
                                event.target.setAttribute('dashboard-no-select','');

                                // Avoid deselecting child group tile when cursor is moving out of the group tile
                                blockTileDeselection = true;


                                let ingoreLockedOverlayOnLockedGroupTiles = false;

                                if (!TileUtils.isGroupTile(draggedTile) && !TileUtils.isFilterTile(draggedTile)) {
                                    enableTileDropIntoGroupTiles();
                                    ingoreLockedOverlayOnLockedGroupTiles = true;
                                }

                                if (ctrl.isInGroupTile) {
                                    ctrl.toggleGroupTileScrollBarDisplay({ $forceShow: true });
                                }

                                toggleLockedOverlayOnAllLockedTiles(true, ingoreLockedOverlayOnLockedGroupTiles);

                                $timeout(function() {
                                    draggedTile.$showEditOverlay = false;
                                });

                            },
                            stop: function(event) {
                                if (ctrl.isInGroupTile) {
                                    ctrl.toggleGroupTileScrollBarDisplay({ $forceShow: false });
                                }

                                $rootScope.$broadcast('dashboardUnlockResizeGridTree');
                                if (pendingResizeRequest) {
                                    pendingResizeRequest = false;
                                    resizeGridTree();
                                }
                                $(event.target).removeClass('tile-wrapper--no-animation');
                                toggleLockedOverlayOnAllLockedTiles(false);
                                const draggedTile = DashboardPageUtils.getTileByElement(ctrl.tiles, event.target);

                                if (!TileUtils.isGroupTile(draggedTile)) {
                                    disableTileDropIntoGroupTiles();
                                }
                                syncTilesFromGridDeletionsAndMoves();
                                $timeout(() => {
                                    event.target.removeAttribute('dashboard-no-select');
                                    blockTileDeselection = false;
                                });
                            },
                            distance: 5,
                            scrollSensitivity: ctrl.isGroupTile ? 60 : 20,
                            scroll: true,
                            containment: $('[data-dashboard-grid="main"]')
                        },
                        {
                            resize: function(event) {
                                $rootScope.$broadcast('dashboardLockResizeGridTree');
                                event.stopPropagation();
                                toggleLockedOverlayOnAllLockedTiles(true);
                            },
                            stop: function(event) {
                                $rootScope.$broadcast('dashboardUnlockResizeGridTree');
                                if (pendingResizeRequest) {
                                    pendingResizeRequest = false;
                                    resizeGridTree();
                                }
                                toggleLockedOverlayOnAllLockedTiles(false);
                                syncTilesFromGridDeletionsAndMoves();
                                // For group tiles, its inner-grid will handle itself the resize of its child tiles
                                const resizedTile = DashboardPageUtils.getTileByElement(ctrl.tiles, event.target);
                                if (TileUtils.isGroupTile(resizedTile)) {
                                    return;
                                }
                                // For other tile types, trigger the resize of the tile content
                                blockTileDeselection = true;
                                $timeout(function() {
                                    if (TileUtils.isGroupTile) {
                                        angular.element(event.target).scope().$broadcast('resize');
                                    }
                                    blockTileDeselection = false;
                                }, 200);
                            }
                        }
                    );
                }).then(() => {
                    $rootScope.spinnerPosition = undefined;
                    ctrl.gridListRendered = true;
                    $rootScope.$broadcast('gridListRendered');
                }).then(() => {
                    syncGridFromTiles();
                }).then(() => {
                    /*
                     * Main dashboard grid resizes itself of scroll bar display change
                     * For inner grid it is the group tile component in charge of that
                     */
                    if (!ctrl.isInGroupTile) {
                        scrollingContainer.addEventListener('wheel', handleShiftWheelScroll, { passive: false });
                        resizeObserver = new ResizeObserver(() => dispatchDebouncedResize(), { box: 'border-box' });
                        resizeObserver.observe(scrollingContainer);
                    }
                    return syncTilesFromGridDeletionsAndMoves();
                });
            };

            function enableTileDropIntoGroupTiles() {
                ctrl.tiles.forEach(currentTile => {
                    if (TileUtils.isGroupTile(currentTile) && !currentTile.locked) {
                        grid.blockItemWhenHoveredByAnotherItem(DashboardPageUtils.getTileElementFromTileId($element, currentTile.$tileId));
                    }
                });
            }

            function disableTileDropIntoGroupTiles() {
                ctrl.tiles.forEach(currentTile => {
                    if (TileUtils.isGroupTile(currentTile) && !currentTile.locked) {
                        grid.unblockItemWhenHoveredByAnotherItem(DashboardPageUtils.getTileElementFromTileId($element, currentTile.$tileId));
                    }
                });
            }

            function getParentGroupTileFromElement(element) {
                const groupTileWrapper = DashboardPageUtils.getTileWrapperFromElement(element);
                return DashboardPageUtils.getTileByElement(ctrl.tiles, groupTileWrapper);
            }

            function updateLoadingState(newVal) {
                ctrl.isLoading = newVal;
                ctrl.loadingStateChange({ $isLoading: newVal });
            }

            /**
             * Return tile loading callback
             * @param {Record<string, Promise<void>>} tileLoadingPromiseByTileId
             * @param {string} tileId
             * @param {boolean} markComplete
             * @returns
             */
            function getTileLoadingCallback(tileLoadingPromiseByTileId, tileId, markComplete = true) {
                return function() {
                    /*
                     * For simple tiles, mark them as fully loaded (i.e. complete). For tiles with delayed
                     * complete events, just mark them as loaded. This way we are still waiting for them but
                     * in the meantime we can load more tiles.
                     */
                    if (ctrl.hook.loadStates[tileId] !== TileLoadingState.COMPLETE) {
                        ctrl.hook.loadStates[tileId] = markComplete ? TileLoadingState.COMPLETE : TileLoadingState.LOADED;
                    }
                    completeTileLoading(tileLoadingPromiseByTileId);
                };
            };

            /**
             * Complete tile loading
             * @param {Record<string, Promise<void>>} tileLoadingPromiseByTileId
             * @param {number} timeout
             */
            function completeTileLoading(tileLoadingPromiseByTileId, timeout) {
                updateLoadingState(DashboardPageUtils.hasLoadingTiles(tileLoadingPromiseByTileId, ctrl.hook));
                if (ctrl.isLoading && !deferredLoadTileInProgress) {
                    // If there are pending loading insights, call us back later
                    deferredLoadTileInProgress = true;
                    const loadTiles = () => {
                        deferredLoadTileInProgress = false;
                        ctrl.loadTiles(tileLoadingPromiseByTileId);
                    };
                    timeout ? $timeout(loadTiles, timeout) : loadTiles();
                }
            }

            /**
             * Returns tile visibility observer
             * @return {IntersectionObserver} tile visibility observer
             */
            function getTileVisibilityObserver() {
                return new IntersectionObserver((entries) => {
                    $timeout(() => {
                        const newVisibleStates = {};
                        entries.forEach((entry) => {
                            // In export mode we set all the tiles as visible
                            if (entry.isIntersecting || isInExport) {
                                newVisibleStates[entry.target.dataset.id] = true;
                            } else {
                                newVisibleStates[entry.target.dataset.id] = false;
                            }
                        });

                        ctrl.hook.visibleStates = {
                            ...ctrl.hook.visibleStates,
                            ...newVisibleStates
                        };

                        ctrl.hook.adjacentStates = DashboardPageUtils.getIsTileAdjacentToVisibleTilesByTileId(ctrl.hook.visibleStates, getTilesById());
                        ctrl.loadTiles(DashboardPageUtils.getVisibleTilesToLoadOrReloadPromiseByTileId(ctrl.hook));
                    });

                }, {
                    root: document.querySelector('.dashboard-export-page-wrapper')
                });
            }

            /**
             * Returns tile by id index
             * @return {Record<string, Tile>} tile by id index
             */
            function getTilesById(){
                return Object.fromEntries(ctrl.tiles.map(tile => [tile.$tileId, tile]));
            }

            /**
             * Watcher callback to trigger tile loading when scope.hook.loadStates is updated.
             * @param {Record<string, TileLoadingState>} nv
             * @param {Record<string, TileLoadingState>} ov
             */
            function triggerIncommingTileLoading(nv, ov) {
                // Waiting for newcomers
                if (!angular.equals(nv, ov)) {
                    $timeout(function() {
                        if (!ctrl.isLoading) {
                            ctrl.loadTiles(ctrl.hook.loadPromises);
                        }
                    });
                }
            }
        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('dashboardNavigationDrawer', {
        templateUrl: '/static/dataiku/js/dashboards/components/dashboard-navigation-drawer/dashboard-navigation-drawer.component.html',
        bindings: {
            // Inputs
            dashboard: '<', // Dashboard
            canEdit: '<', // boolean
            selectedTiles: '<', // Tile[]
            selectedInsights: '<', // Insight[],
            currentPageIndex: '<', // number
            canModerateDashboards: '<', //boolean
            // Outputs
            createPage: '&', // () => void
            navigateToPage: '&', // ({ $pageIndex: number }) => void
            openCopySlideModal: '&', // ({ $pageIndex: number }) => void
            pageIndexChange: '&', // ({ $previousIndex: number, $currentIndex }) => void
            removePage: '&' //  ({ $pageIndex: number }) => void
        },
        controller: function($timeout, DASHBOARD_SETTINGS_PANEL_VIEWS, translate) {
            const ctrl = this;
            const TRANSITION_DURATION_MS = 100;
            const DEFAULT_DRAWER_WIDTH = 299;
            const MIN_DRAWER_WIDTH = 149;
            const MAX_DRAWER_WIDTH = MIN_DRAWER_WIDTH*4;
            const BASE_PAGE_SORT_OPTIONS = {
                helper: 'clone',
                axis: 'y',
                cursor: 'move',
                update: handlePageSortChange,
                handle: '.dashboard-navigation-drawer-left-bar__page-miniature-content',
                items: '> .dashboard-navigation-drawer-left-bar__page-miniature'
            };

            ctrl.CLOSED_DRAWER_WIDTH = 49;
            ctrl.isSettingsPanelExpanded = false;
            ctrl.isDrawerExpanded = ctrl.canEdit ? false : (localStorage.getItem('dku.dashboards.drawer.expand') === 'true' || false);
            ctrl.expandedDrawerWidth = getStoredExpandedDrawerWidth();
            ctrl.activeView = DASHBOARD_SETTINGS_PANEL_VIEWS.PAGE_SETTINGS;
            ctrl.pageSortOptions = { ...BASE_PAGE_SORT_OPTIONS };
            ctrl.resizing = false;
            ctrl.translate = translate;
            ctrl.DASHBOARD_SETTINGS_PANEL_VIEWS = DASHBOARD_SETTINGS_PANEL_VIEWS;

            ctrl.$onChanges = (changes) => {
                if (changes.canEdit) {
                    handleCanEditChange(ctrl.canEdit);
                }
                if (changes.selectedTiles) {
                    handleSelectedTilesChange(ctrl.selectedTiles);
                }
                if (changes.dashboard) {
                    ctrl.visiblePages = ctrl.dashboard.pages.filter(page => page.show);
                }

                if (changes.currentPageIndex) {
                    scrollToSelectedPage();
                }
            };

            ctrl.$onInit = () => {
                scrollToSelectedPage();
            };

            function scrollToSelectedPage() {
                $timeout(function() {
                    const selectedPage = document.querySelector('.dashboard-navigation-drawer-left-bar__page-miniature--current');
                    if (selectedPage) {
                        selectedPage.scrollIntoView({ block: 'nearest' });
                    }
                });
            }

            /**
             * @param {boolean} [toggle]
             */
            ctrl.toggleDrawer = (toggle = !ctrl.isDrawerExpanded) => {
                localStorage.setItem('dku.dashboards.drawer.expand', toggle);
                ctrl.isDrawerExpanded = toggle;
            };


            ctrl.getVisiblePages = () => {
                return ctrl.dashboard.pages.filter(page => page.show);
            };

            /**
             * @param {boolean} [toggle]
             * @param {DASHBOARD_SETTINGS_PANEL_VIEWS} [view]
             */
            ctrl.toggleEditionMenu = (toggle = !ctrl.isSettingsPanelExpanded, view = ctrl.activeView) => {
                ctrl.isSettingsPanelExpanded = toggle;
                if (toggle) {
                    ctrl.activeView = view;
                }
                $timeout(() => {
                    if (!toggle) {
                        ctrl.activeView = view;
                    }
                }, TRANSITION_DURATION_MS);
            };

            /**
             * @param {number} pageIndex
             */
            ctrl.handleMiniatureClick = (pageIndex) => {
                document.activeElement.blur();
                if (pageIndex !== ctrl.currentPageIndex) {
                    ctrl.navigateToPage({ $pageIndex: pageIndex });
                }
                if (ctrl.canEdit === true) {
                    ctrl.toggleEditionMenu(ctrl.isSettingsPanelExpanded, DASHBOARD_SETTINGS_PANEL_VIEWS.PAGE_SETTINGS);
                }
            };

            /**
             * @param {MouseEvent} event
             */
            ctrl.handleResizeStart = (event) => {
                event.preventDefault();
                event.stopPropagation();
                ctrl.resizing = true;
                document.addEventListener('mouseup', handleResizeStop, true);
                document.addEventListener('mousemove', handleResizeChange, true);
            };

            /**
             * @param {MouseEvent} event
             */
            function handleResizeStop(event) {
                event.preventDefault();
                event.stopPropagation();
                ctrl.resizing = false;
                localStorage.setItem('dku.dashboards.drawer.expanded-width', ctrl.expandedDrawerWidth);
                document.removeEventListener('mouseup', handleResizeStop, true);
                document.removeEventListener('mousemove', handleResizeChange, true);
            };

            /**
             * @param {MouseEvent} event
             */
            function handleResizeChange(event) {
                event.preventDefault();
                event.stopPropagation();
                const newWidth = event.clientX - document.body.offsetLeft;
                if (newWidth >= MIN_DRAWER_WIDTH && newWidth < MAX_DRAWER_WIDTH) {
                    $timeout(() => ctrl.expandedDrawerWidth = newWidth);
                }
            }

            function getStoredExpandedDrawerWidth() {
                const storedWidth = localStorage.getItem('dku.dashboards.drawer.expanded-width');
                if (!storedWidth) {
                    return DEFAULT_DRAWER_WIDTH;
                }
                const parsedWidth = parseInt(storedWidth);
                if (isNaN(parsedWidth)) {
                    return DEFAULT_DRAWER_WIDTH;
                }
                return parsedWidth;
            }

            function handlePageSortChange(_, ui) {
                ctrl.pageIndexChange({ $previousIndex: ui.item.sortable.index, $currentIndex: ui.item.sortable.dropindex });
            };

            function handleSelectedTilesChange(selectedTiles) {
                if (selectedTiles == null) {
                    return;
                }
                if (selectedTiles.length === 1 && selectedTiles[0].insightType !== 'filters') {
                    ctrl.activeView = DASHBOARD_SETTINGS_PANEL_VIEWS.TILE_SETTINGS;
                } else if (selectedTiles.length > 1) {
                    ctrl.activeView = DASHBOARD_SETTINGS_PANEL_VIEWS.TILE_SETTINGS;
                    ctrl.toggleEditionMenu(true);
                } else {
                    ctrl.activeView = DASHBOARD_SETTINGS_PANEL_VIEWS.PAGE_SETTINGS;
                }
            }

            /**
             * @param {boolean} canEdit
             */
            function handleCanEditChange(canEdit) {
                ctrl.isSettingsPanelExpanded = canEdit;
                if (canEdit) {
                    ctrl.isDrawerExpanded = false;
                }
                ctrl.pageSortOptions = {
                    ...BASE_PAGE_SORT_OPTIONS,
                    disabled: !canEdit
                };
            }
        }
    });
})();

;
(function() {
    'use strict';

    const DEFAULT_BACKGROUND_COLOR = '#ffffff';

    angular.module('dataiku.dashboards').component('dashboardSettingsPanel', {
        templateUrl: '/static/dataiku/js/dashboards/components/dashboard-settings-panel/dashboard-settings-panel.component.html',
        bindings: {
            // Inputs
            activeView: '<', // DASHBOARD_SETTINGS_PANEL_VIEWS
            dashboard: '<', // Dashboard
            page: '<', // DashboardPage
            currentPageIndex: '<', // number
            selectedTiles: '<', // Tile
            selectedInsights: '<', // Insight,
            canModerateDashboards: '<', // boolean
            theme: '<', //DSSVisualizationTheme
            // Outputs
            openCopySlideModal: '&', // ({ $pageIndex: number }) => void
            removePage: '&', // ({ $pageIndex: number }) => void
            toggleDashboardSettings: '&' // () => void
        },
        controller: function($scope, $timeout, WT1, DashboardFilters, DashboardUtils, DASHBOARD_SETTINGS_PANEL_VIEWS, translate, TileUtils, ColorUtils, DSSVisualizationThemeUtils, DefaultDSSVisualizationTheme) {
            const ctrl = this;

            ctrl.themeColors = ColorUtils.getThemeColorsWithBlackWhite(ctrl.theme);
            ctrl.colors = ColorUtils.generateThemePaletteColors(DefaultDSSVisualizationTheme.colors, ctrl.themeColors.length > 0).foregroundColors;
            ctrl.TileUtils = TileUtils;
            ctrl.translate = translate;
            ctrl.resizeDashboard = DashboardUtils.resizeDashboard;

            ctrl.DASHBOARD_SETTINGS_PANEL_VIEWS = DASHBOARD_SETTINGS_PANEL_VIEWS;
            ctrl.page = { 'backgroundColor': DEFAULT_BACKGROUND_COLOR };
            ctrl.canCrossFilter = false;
            ctrl.isSingleTileSelected = false;
            ctrl.isSingleGroupTileSelected = false;
            ctrl.areMultipleTilesSelected = false;
            ctrl.shouldShowResetTileSpacing = shouldShowResetTileSpacing();

            let needsResize = false;

            ctrl.$onChanges = (changes) => {
                if ((changes.dashboard || changes.currentPageIndex) && ctrl.dashboard && ctrl.currentPageIndex != null) {
                    ctrl.page = ctrl.dashboard.pages[ctrl.currentPageIndex];
                    if (ctrl.page) {
                        ctrl.page.backgroundColor = ctrl.page.backgroundColor || DEFAULT_BACKGROUND_COLOR;
                        ctrl.canCrossFilter = !!DashboardFilters.findFiltersTileInPage(ctrl.page);
                    }
                }
                if (changes.theme && changes.theme.currentValue && (!ctrl.themeColors || !_.isEqual(ctrl.themeColors, ctrl.dashboard.theme.colors))) {
                    ctrl.themeColors = ColorUtils.getThemeColorsWithBlackWhite(changes.theme.currentValue);
                    ctrl.colors = ColorUtils.generateThemePaletteColors(changes.theme.currentValue.colors, ctrl.themeColors.length > 0).foregroundColors;
                }
                if (changes.selectedTiles) {
                    updateTileSelectionFlags(ctrl.selectedTiles);
                }

                // The template might have changed before the blur event is triggered, which would prevent the resize from happening if the flag is not checked.
                if (needsResize) {
                    needsResize = false;
                    DashboardUtils.resizeDashboard();
                }
            };

            ctrl.enableCrossFiltersChanged = function(dashboardId, pageId, isCrossFilteringEnabled) {
                WT1.event('dashboard-cross-filtering-enabled-change', {
                    dashboardId,
                    pageId,
                    isCrossFilteringEnabled
                });
            };

            ctrl.setTitleFontColor = (color) => {
                $timeout(() => ctrl.page.titleFontColor = color);
            };

            ctrl.setTitleAlignment = (alignment) => {
                $timeout(() => ctrl.page.titleAlign = alignment);
            };

            ctrl.shouldShowResetTitle = () => {
                const theme = (ctrl.theme ? ctrl.theme: DefaultDSSVisualizationTheme);
                return ctrl.page && (
                    (ctrl.page.titleFontColor !== theme.pageTitleFormatting.fontColor)
                    || (ctrl.page.titleFontSize !== theme.pageTitleFormatting.fontSize)
                    || (ctrl.page.titleAlign !== theme.pageTitleAlignment)
                );
            };

            ctrl.resetTitle = () => {
                const theme = (ctrl.theme ? ctrl.theme: DefaultDSSVisualizationTheme);
                ctrl.page.titleFontColor = theme.pageTitleFormatting.fontColor;
                ctrl.page.titleFontSize = theme.pageTitleFormatting.fontSize;
                ctrl.page.titleAlign = theme.pageTitleAlignment;

                $timeout(() => $scope.$apply());
            };

            ctrl.setBackgroundColor = (color) => {
                $timeout(() => ctrl.page.backgroundColor = color);
            };

            ctrl.resetBackgroundColor = () => {
                $timeout(() => ctrl.page.backgroundColor = ctrl.theme ? ctrl.theme.backgroundColor : DEFAULT_BACKGROUND_COLOR);
            };

            ctrl.shouldShowResetBackgroundColorButton = () => {
                return ctrl.page && ctrl.page.backgroundColor !== (ctrl.theme ? ctrl.theme.backgroundColor : DEFAULT_BACKGROUND_COLOR);
            };

            ctrl.getPageTitle = function() {
                return ctrl.page && ctrl.page.title ? ctrl.page.title : 'Page ' + (ctrl.currentPageIndex + 1);
            };

            ctrl.changePageVisibility = function() {
                if (ctrl.page) {
                    ctrl.page.show = !ctrl.page.show;
                }
            };

            ctrl.handleTileSpacingChange = () => {
                ctrl.shouldShowResetTileSpacing = shouldShowResetTileSpacing();
                needsResize = true;
            };

            ctrl.overrideFormattingWithTheme = function(theme) {
                $timeout(() => {
                    const currentDashboardCopy = angular.copy(ctrl.dashboard);
                    const newTheme = angular.copy(theme);
                    ctrl.dashboard.theme = newTheme;
                    DSSVisualizationThemeUtils.applyToDashboard(ctrl.dashboard, newTheme);
                    DashboardUtils.resizeDashboard();

                    DSSVisualizationThemeUtils.showThemeAppliedSnackbar(
                        ctrl.dashboard,
                        currentDashboardCopy,
                        () => DashboardUtils.resizeDashboard()
                    );
                });
            };

            ctrl.handleTileSpacingBlur = () => {
                needsResize = false;
                DashboardUtils.resizeDashboard();
            };

            ctrl.toggleAutoStackUp = () => {
                WT1.event('dashboard-auto-stack-up-toggled-from-page-settings', {
                    autoStackUp: ctrl.dashboard.autoStackUp
                });
                if (!ctrl.dashboard.autoStackUp) {
                    return;
                }
                const gridsToUpdate = [];

                /*
                 * When disabling vertival gaps we need to update the tile positions
                 * of pages that are not the current ones
                 */
                ctrl.dashboard.pages.forEach((page, index) => {
                    if (ctrl.currentPageIndex === index) {
                        return;
                    }
                    gridsToUpdate.push({ grid: page.grid, columnNumber: DashboardUtils.getColumnNumber() });
                    page.grid.tiles.forEach(tile => {
                        if (TileUtils.isGroupTile(tile) && tile.grid != null && tile.grid.tiles != null) {
                            gridsToUpdate.push({ grid: tile.grid, columnNumber: tile.box.width });
                        }
                    });
                });

                gridsToUpdate.forEach(tiles => removeGridVerticalGaps(tiles));
            };

            function removeGridVerticalGaps({ grid, columnNumber }) {
                const items = grid.tiles.map(tile => ({
                    x: tile.box.left >= 0 ? tile.box.left : null,
                    y: tile.box.top >= 0 ? tile.box.top : null,
                    w: tile.box.width,
                    h: tile.box.height,
                    tile
                }));
                // eslint-disable-next-line
                new GridList(items, {
                    lanes: columnNumber,
                    direction: 'vertical',
                    allowDirectionalGaps: true
                }).toggleGapMode(false);

                grid.tiles = items.map(item => {
                    item.tile.box.left = item.x;
                    item.tile.box.top = item.y;
                    return item.tile;
                });
            }

            ctrl.resetTileSpacing = () => {
                ctrl.dashboard.tileSpacing = (ctrl.theme ? ctrl.theme.tileSpacing : DefaultDSSVisualizationTheme.tileSpacing);
                $timeout(() => $scope.$apply()).then(() => ctrl.handleTileSpacingChange());
            };

            function updateTileSelectionFlags(selectedTiles) {
                ctrl.isSingleTileSelected = selectedTiles && selectedTiles.length === 1;
                ctrl.isSingleGroupTileSelected = ctrl.isSingleTileSelected && ctrl.TileUtils.isGroupTile(selectedTiles[0]);
                ctrl.areMultipleTilesSelected = selectedTiles && selectedTiles.length > 1;
            }

            function shouldShowResetTileSpacing() {
                return ctrl.dashboard && ctrl.dashboard.tileSpacing !== (ctrl.theme ? ctrl.theme.tileSpacing : DefaultDSSVisualizationTheme.tileSpacing);
            };
        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('groupTile', {
        bindings: {
            tile: '<',
            selected: '<', // boolean
            insight: '<',
            editable: '<',
            minHeight: '<', // number
            backgroundColor: '<', // string
            gridColor: '<', // string
            currentGroupTileIdInEditMode: '<', // string | null
            hook: '<',
            activeFilters: '<',
            filters: '<',
            page: '<',
            showGrid: '<', // boolean
            tileSpacing: '<', // number
            preselectedTile: '<', // DashboardTile
            autoStackUp: '<', // boolean
            canExportDatasets: '<', // boolean
            insightsMap: '<', // { [insightId: string]: Insight }
            accessMap: '<', // { [insightId: string]: string }
            dashboardTheme: '<', // DSSVisualizationTheme
            filtersChange: '&', // ({ $filters }) => void
            filtersParamsChange: '&', // ({ $filtersParams }) => void,
            isDeactivatedChange: '&', // ({ $isDeactivated }) => void
            preselectedTileChange: '&', // ({ $preselectedTile }) => void,
            raiseError: '&', // ({ $errorData }) => void,
            isTileLockedChange: '&' // () => void,
        },
        templateUrl: '/static/dataiku/js/dashboards/components/group-tile/group-tile.component.html',
        controller: function(TileLoadingState, Logger, $timeout, TileLoadingBehavior, $scope, TileUtils, $element) {
            const ctrl = this;

            ctrl.display = false;

            const overflowCSSClass = 'dashboard-tile__group-tile-container--overflowing-inner-tiles';
            let successRoutine = null;
            let reloadGrid = null;
            let resizeGrid = null;
            let hasOverflowingInnerTiles = false;
            let scrollingContainer = null;

            ctrl.$onInit = function() {
                if (ctrl.tile.autoLoad) {
                    ctrl.hook.loadStates[ctrl.tile.$tileId] = TileLoadingState.WAITING;
                }
                ctrl.hook.loadPromises[ctrl.tile.$tileId] = ctrl.load;
                ctrl.hook.reloadPromises[ctrl.tile.$tileId] = ctrl.reload;
                scrollingContainer = getScrollingContainer();
                /*
                 * We eliminate the visible padding around the grid by applying negative top and left margins to the child tiles.
                 * To ensure the grid fully occupies the container's space, we extend its width (width: calc(var(--tile-spacing) + 100%)).
                 * This setup causes the content to overflow horizontally, but the horizontal overflow is hidden.
                 *
                 * We attach a scroll event handler to prevent any unexpected horizontal scrolling while the user drags and drops the tiles.
                 */
                scrollingContainer.on('scroll', removeLeftScroll);
            };

            ctrl.$onChanges = function(changes) {
                if (changes.currentGroupTileIdInEditMode || changes.editable) {
                    ctrl.isGroupTileInnerGridEditable = ctrl.editable && ctrl.tile.$tileId === ctrl.currentGroupTileIdInEditMode;
                }
            };

            ctrl.$onDestroy = function() {
                if (scrollingContainer != null) {
                    scrollingContainer.off('scroll', removeLeftScroll);
                    scrollingContainer = null;
                }
            };

            ctrl.handleTilesChange = function(tiles) {
                handleChildTileVerticallyOverflowingGroupTile(tiles);
                $timeout(() => {
                    ctrl.tile.grid.tiles = [...tiles];
                });
            };

            ctrl.handleGridLoadingStateChange = function(isLoading) {
                $timeout(() => {
                    ctrl.isLoading = isLoading;
                    if (successRoutine != null) {
                        ctrl.hook.loadStates[ctrl.tile.$tileId] = TileLoadingState.COMPLETE;
                        successRoutine();
                        successRoutine = null;
                    }
                });
            };

            ctrl.handleGridInit = function(api) {
                reloadGrid = api.reload;
                resizeGrid = api.resizeGrid;
                handleChildTileVerticallyOverflowingGroupTile(ctrl.tile.grid.tiles);
                $scope.$watch('$ctrl.tile.box.height', () => {
                    handleChildTileVerticallyOverflowingGroupTile(ctrl.tile.grid.tiles);
                });
            };

            ctrl.handleToggleGroupTileScrollBarDisplay = function(forceShow, scrollTopOnHide) {
                // If the group tile has overflowing inner tiles, the scroll bar is always shown
                if (hasOverflowingInnerTiles) {
                    return;
                }
                if (forceShow) {
                    scrollingContainer.addClass(overflowCSSClass);
                } else {
                    if (scrollTopOnHide) {
                        scrollingContainer.scrollTop(0);
                    }
                    scrollingContainer.removeClass(overflowCSSClass);
                }
            };

            ctrl.load = function(resolve) {
                Logger.info('Start loading group tile', ctrl.tile.$tileId);
                ctrl.hook.loadStates[ctrl.tile.$tileId] = TileLoadingState.WAITING;
                ctrl.isLoading = true;
                ctrl.display = true;
                successRoutine = resolve;
                return TileLoadingBehavior.DELAYED_COMPLETE;
            };

            ctrl.reload = function(resolve) {
                if (reloadGrid == null) {
                    return;
                }
                Logger.info('Start re-loading group tile', ctrl.tile.$tileId);
                ctrl.isLoading = true;
                reloadGrid();
                successRoutine = resolve;
                return TileLoadingBehavior.DELAYED_COMPLETE;
            };

            $scope.$on('dashboardResizeGroupTile', (_, { tile }) => {
                if (tile != null && tile.$tileId === ctrl.tile.$tileId && resizeGrid != null) {
                    resizeGrid();
                }
            });

            function getScrollingContainer() {
                return $element.closest('.dashboard-tile__group-tile-container');
            }

            function removeLeftScroll() {
                if (scrollingContainer != null && scrollingContainer.scrollLeft() > 0) {
                    scrollingContainer.scrollLeft(0);
                }
            }

            function handleChildTileVerticallyOverflowingGroupTile(tiles) {
                let nextHasOverflowingInnerTiles = false;
                tiles.forEach(tile => {
                    if (TileUtils.isChildTileVerticallyOverflowingGroupTile(tile, ctrl.tile.box.height)) {
                        nextHasOverflowingInnerTiles = true;
                    }
                });
                if (hasOverflowingInnerTiles !== nextHasOverflowingInnerTiles) {
                    hasOverflowingInnerTiles = nextHasOverflowingInnerTiles;
                    if (hasOverflowingInnerTiles) {
                        scrollingContainer.addClass(overflowCSSClass);
                    } else {
                        scrollingContainer.scrollTop(0).removeClass(overflowCSSClass);
                    }
                    if (resizeGrid != null) {
                        resizeGrid();
                    }
                }
            }
        }
    });
})();

;
(function() {
    'use strict';

    angular.module('dataiku.dashboards').constant('DASHBOARD_SETTINGS_PANEL_VIEWS', {
        PAGE_SETTINGS: 'PAGE_SETTINGS',
        DASHBOARD_SETTINGS: 'DASHBOARD_SETTINGS',
        TILE_SETTINGS: 'TILE_SETTINGS',
        THEME_SETTINGS: 'THEME_SETTINGS'
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/view.js
     */
    app.constant('TileLoadingBehavior', {
        // Once the promise registered in loadPromises has be resolved, the insight is fully loaded.
        NORMAL: undefined,

        /*
         * Additional work is needed to fully load the insight once the promise registered in loadPromises has be resolved.
         * The insight will itself change the loading state to COMPLETE asynchronously in `loadStates` map.
         */
        DELAYED_COMPLETE: 'DELAYED_COMPLETE'
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/view.js
     */
    app.constant('TileLoadingState', {
        // Insight has registered itself and is waiting for the dashboard/insight-preview component to load it.
        WAITING: 0,

        // Dashboard/insight-preview component has called the promise in loadPromises map and is waiting for the promise to be resolved.
        LOADING: 1,

        // Loading promise has been resolved
        LOADED: 2,

        // Loading promise has been resolved and any additional asynchronous work has also completed.
        COMPLETE: 3,

        // Insight is waiting for the dashboard/insight-preview component to reload it.
        WAITING_FOR_RELOAD: 4
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/edit.js
     */
    app.controller('CopyDashboardModalController', function($scope, DataikuAPI, ActivityIndicator, StateUtils) {
        $scope.init = function(dashboard) {
            $scope.dashboard = dashboard;
            $scope.dashboard.newName = 'Copy of ' + dashboard.name;
            $scope.pointerMode = false;
        };

        $scope.copy = function() {
            DataikuAPI.dashboards.copy($scope.dashboard.projectKey, $scope.dashboard.id, $scope.dashboard.newName, !$scope.pointerMode)
                .error(setErrorInScope.bind($scope))
                .success(function(data) {
                    ActivityIndicator.success( $scope.dashboard.name + ' copied into ' + data.name + ', <a href=\'' + StateUtils.href.dashboard(data.id, data.projectKey, { name: data.name }) + '\' >view dashboard</a>.', 5000);
                    $scope.resolveModal();
                });
        }
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/edit.js
     */
    app.controller('CopyPageController', function($scope, DataikuAPI, $timeout, StateUtils, DashboardUtils, TileUtils) {

        $scope.pointerMode = false;

        // b/c when this function is called, CreateModalFromTemplate callback is not executed yet, therefore $scope.page does not exist yet.
        $scope.init = function() {
            $scope.copyPageName = $scope.page.title || 'Page ' + ($scope.pageIdx + 1);
            $scope.copyPageName += ' - COPY';

            DataikuAPI.dashboards.listEditable($scope.dashboard.projectKey, $scope.dashboard.id)
                .error(setErrorInScope.bind($scope))
                .success(function(data) {
                    $scope.dashboards = data.dashboards;
                    for (let i = 0; i < $scope.dashboards.length; i++) {
                        const d = $scope.dashboards[i];
                        if (d.id == $scope.dashboard.id) {
                            $scope.targetedDashboard = d;
                            break;
                        }
                    }
                    $scope.$apply();
                });
        };

        $scope.copyPage = function() {
            const copyPageFront = function(insights) {
                const newPage = angular.copy($scope.dashboard.pages[$scope.pageIdx]);
                newPage.title = $scope.copyPageName;
                newPage.id = null;
                if (insights) {
                    TileUtils.getFlattenTileList(newPage.grid.tiles).forEach(function(tile) {
                        if (tile.tileType == 'INSIGHT') {
                            const insight = insights.shift();
                            tile.insightId = insight.id;
                        }
                    });
                }
                $scope.pages.splice($scope.pageIdx + 1, 0, newPage);
                $scope.uiState.currentPageIdx = $scope.pageIdx+1;
            };

            const tgt = $scope.targetedDashboard;

            if (tgt.id == $scope.dashboard.id) {
                if (!$scope.pointerMode) {
                    const insightIds = [];
                    const insightNames = [];

                    TileUtils.getFlattenTileList($scope.page.grid.tiles).forEach(function(tile) {
                        if (tile.tileType == 'INSIGHT') {
                            insightIds.push(tile.insightId);
                            insightNames.push($scope.insightsMap[tile.insightId].name);
                        }
                    });
                    DataikuAPI.dashboards.insights.copy($scope.dashboard.projectKey, insightIds, insightNames, $scope.dashboard.id)
                        .error(setErrorInScope.bind($scope))
                        .success(function(newInsights) {
                            newInsights.forEach(function(insight) {
                                $scope.insightsMap[insight.id] = insight;
                                DashboardUtils.sendWT1InsightCreation(insight.type, {
                                    triggeredFrom: 'copy-page-modal'
                                });
                            });
                            copyPageFront(newInsights);
                            $scope.dismiss();
                        });
                } else {
                    copyPageFront();
                    $scope.dismiss();
                }
            } else {
                DataikuAPI.dashboards.copyPage($scope.dashboard.projectKey, $scope.dashboard.id, $scope.page, tgt.id, $scope.copyPageName, $scope.pointerMode)
                    .error(setErrorInScope.bind($scope))
                    .success(function(data) {
                        $timeout(function() {
                            StateUtils.go.dashboard(tgt.id, tgt.projectKey, { name: tgt.name, tab: 'edit', pageId: data });
                        });
                        $scope.dismiss();
                    });
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.controller('DashboardCoreController', function($scope, $element, $rootScope, $timeout, $stateParams, DataikuAPI, TopNav, $filter, $state, $controller, CreateModalFromTemplate, WT1, DashboardUtils, DashboardFilters, DSSVisualizationThemeUtils) {
        $controller('DashboardsCommonController', { $scope: $scope });
        $scope.uiState = { tooltipReload: 'Reload tiles', displayNotification: false };

        DataikuAPI.dashboards.getSummary($stateParams.projectKey, $stateParams.dashboardId).success(function(data) {
            $scope.ownerDisplayName = data.ownerDisplayName;

            $scope.dashboard = data.object;
            $scope.dashboard.pages.forEach(page => {
                /*
                 * client id or cid is a unique identifier automatically assigned
                 * when they're first created.
                 * Client ids are handy when the model has not yet been saved to the server,
                 * and does not yet have its eventual true id, but already needs to be visible in the UI.
                 */
                page.cid = _.uniqueId();
            });

            $scope.interest = data.interest;
            $scope.timeline = data.timeline;
            $scope.canEdit = $scope.canEditDashboard($scope.dashboard);
            if ($scope.canEdit) {
                $scope.origDashboard = angular.copy(data.object);
            }

            TopNav.setItem(TopNav.ITEM_DASHBOARD, $stateParams.dashboardId, $scope.dashboard);

            $scope.$watch('dashboard.theme', function(theme) {
                if (theme && theme.generalFormatting && theme.generalFormatting.fontFamily) {
                    $element.css('--visualization-font-family', theme.generalFormatting.fontFamily);
                    $element.css('--visualization-font-color', theme.generalFormatting.fontColor);
                }
            });

            $scope.$watch('dashboard.name', function(nv) {
                if (!nv) {
                    return;
                }
                TopNav.setPageTitle(nv + ' - Dashboard');
                $state.go($state.current, { dashboardName: $filter('slugify')(nv), separator: '_' }, { location: 'replace', inherit: true, notify: false, reload: false });
            });
        }).error(setErrorInScope.bind($scope));

        $scope.isDirty = function() {
            return !angular.equals($scope.dashboard, $scope.origDashboard);
        };

        $scope.revertChanges = function() {
            $scope.dashboard = angular.copy($scope.origDashboard);
        };

        $scope.toggleDashboardListed = function() {
            if (!$scope.dashboard.listed) {
                DataikuAPI.dashboards.getMissingReaderAuthorizations($scope.dashboard.projectKey, [$scope.dashboard.id]).success(function(data) {
                    if (data.length) {
                        CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                            newScope.initForDashboardsWithAuths([$scope.dashboard], true, data);
                        }).then(function() {
                            $scope.makeDashboardListed($scope.dashboard);
                        });
                    } else {
                        $scope.makeDashboardListed($scope.dashboard);
                    }
                });
            } else {
                $scope.makeDashboardListed($scope.dashboard);
            }
        };

        $scope.requestFullScreen = function() {
            $scope.$broadcast('toggleFullScreen');
        };

        $scope.reloadAllTiles = function() {
            $scope.uiState.displayNotification = false;
            $scope.uiState.tooltipReload = '';
            $scope.$broadcast('dashboard-reload-tiles');
        };

        $scope.displayNotification = function() {
            $scope.uiState.displayNotification = true;
            $scope.uiState.tooltipReload = 'Update available, click to reload tiles';
        };

        $scope.$on('dashboards-reload-needed', () => {
            if (!$scope.dashboard) {
                return;
            }
            if ($scope.dashboard.reloadWhenEventReceived) {
                $scope.reloadAllTiles();
            } else {
                $scope.displayNotification();
            }
        });

        $scope.toggleFilterPanel = function() {
            $timeout(() => DashboardUtils.toggleFilterPanel($scope.dashboard, getCurrentPage()));
        };

        $scope.toggleAllFilterPanels = (toggle) => {
            $scope.dashboard.pages.forEach(page => $scope.toggleFilterPanel(page, toggle));
        };

        $scope.isFilterPanelOpened = function() {
            const currentPage = getCurrentPage();
            if (currentPage) {
                return currentPage.showFilterPanel;
            }
        };

        let crossFilterNotificationTimeoutId;
        const unregisterDashboardFiltersChangedListener = $rootScope.$on('dashboardFiltersChanged', (_, { currentFilters, previousFilters }) => {

            $scope.filtersCount = currentFilters ? currentFilters.length : 0;

            const isInitialisation = !currentFilters.length && !previousFilters.length;

            if (isInitialisation || $scope.isFilterPanelOpened()) {
                return;
            }

            $timeout.cancel(crossFilterNotificationTimeoutId);

            if (currentFilters && previousFilters) {
                if (currentFilters.length === previousFilters.length) {
                    $scope.filterChangedNotificationLabel = 'Filter updated';
                } else if (currentFilters.length > previousFilters.length) {
                    $scope.filterChangedNotificationLabel = 'Filter added';
                }

                crossFilterNotificationTimeoutId = $timeout(() => {
                    $scope.filterChangedNotificationLabel = null;
                }, 5000);
            }
        });

        $scope.shouldDisplayShowHideFiltersButton = (editable) => {
            const currentPage = getCurrentPage();
            if (currentPage) {
                const canFilter = DashboardFilters.canFilter(editable, currentPage.filters, currentPage.enableCrossFilters);
                const isFiltersPanelDocked = DashboardFilters.isFiltersPanelDocked(currentPage.filtersParams.panelPosition);
                return canFilter && isFiltersPanelDocked;
            }
        };

        $scope.$on('$destroy', () => {
            unregisterDashboardFiltersChangedListener();
            DSSVisualizationThemeUtils.hideThemeAppliedSnackbar();
        });

        function getCurrentPage() {
            return DashboardUtils.getCurrentPage($scope.dashboard, $state.params.pageId);
        }

        function getVisiblePageId(viewMode) {
            let pageId = '';
            const currentPage = $scope.dashboard.pages.find(page => page.id === $state.params.pageId);
            const firstVisiblePage = $scope.dashboard.pages.find(page => page.show);
            if (viewMode === 'view' && currentPage && !currentPage.show) {
                if (firstVisiblePage) {
                    pageId = firstVisiblePage.id;
                }
            } else if ($state.params.pageId) {
                pageId = $state.params.pageId;
            } else if ($scope.dashboard.pages && $scope.dashboard.pages.length) {
                pageId = $scope.dashboard.pages[0].id;
            }
            return pageId;
        }

        function getVisiblePageIndex(pageId) {
            if (pageId) {
                return $scope.dashboard.pages.findIndex(page => page.id === pageId);
            }
            return '';
        }

        function switchViewMode(viewMode) {
            if (viewMode !== 'view' && viewMode !== 'edit') {
                return;
            }
            WT1.event(`dashboard-to-${viewMode}-mode-switch`);
            DSSVisualizationThemeUtils.hideThemeAppliedSnackbar();
            DashboardFilters.resetAllDashboardPagesFilterPanelStates();
            const pageId = getVisiblePageId(viewMode);
            $scope.uiState.currentPageIdx = getVisiblePageIndex(pageId);
            $state.go(`projects.project.dashboards.dashboard.${ viewMode }`, { pageId });
        }

        $scope.switchToView = () => switchViewMode('view');
        $scope.switchToEdit = () => switchViewMode('edit');
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.controller('DashboardDetailsController', function($scope, $stateParams, $state, FutureProgressModal, DatasetUtils, StateUtils, Dialogs, ActiveProjectKey) {
        $scope.isOnDashboardObjectPage = function() {
            return $state.includes('projects.project.dashboards.dashboard');
        }
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');
    // const LEFT_MENU_OPENED_KEY = 'dashboardLeftMenuOpened';

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/edit.js
     */
    app.controller('DashboardEditController', function($scope, $state, $timeout, DashboardFilters,
        DataikuAPI, CreateModalFromTemplate, Dialogs, TopNav, INSIGHT_TYPES, DSSVisualizationThemeUtils,
        TileUtils, ActivityIndicator, DashboardUtils, $rootScope, ATSurveyService, WT1, FiltersPanelPosition, StateUtils,
        DashboardPageUtils) {
        TopNav.setLocation(TopNav.TOP_DASHBOARD, 'dashboards', null, 'edit');

        if ($scope.dashboard) {
            TopNav.setPageTitle($scope.dashboard.name + ' - Dashboard');
        }

        // Call ATSurveyService to trigger survey if needed
        ATSurveyService.updateCounter('DashboardEdit');

        $scope.insightTypes = INSIGHT_TYPES;
        $scope.hasFilterTile = false;
        $scope.FiltersPanelPosition = FiltersPanelPosition;
        $scope.selectedTiles = [];
        $scope.selectedInsights = [];

        Dialogs.saveChangesBeforeLeaving($scope, $scope.isDirty, $scope.saveDashboard, $scope.revertChanges, 'This dashboard has unsaved changes.');
        Dialogs.checkChangesBeforeLeaving($scope, $scope.isDirty);

        DashboardUtils.toggleAllFilterPanels($scope.dashboard, true);

        $scope.$on('dashboardSyncModelsDone', function() {
            const insightIdSet = retrieveInsightIdSetFromDashboard($scope.dashboard);
            $scope.hasFilterTile = alreadyHaveGlobalFilters();
            Object.keys($scope.insightsMap).forEach(function(insightId) {
                if (!insightIdSet.has(insightId)) {
                    delete $scope.insightsMap[insightId];
                }
            });
        });

        const unregisterDashboardSelectTileEvent = $rootScope.$on('dashboardSelectTiles', (_, { tiles }) => {
            $scope.selectedTiles = tiles;
            $scope.selectedInsights = tiles.map(tile => $scope.insightsMap[tile.insightId]);
        });

        const unregisterDashboardUnselectTilesEvent = $rootScope.$on('dashboardUnselectTiles', () => {
            $scope.selectedTiles = [];
            $scope.selectedInsights = [];
        });

        $scope.$watch('uiState.selectedTile', (nv) => {
            if (nv) {
                $scope.selectedTiles = [nv];
                $scope.selectedInsights = [$scope.insightsMap[nv.insightId]];
            }
        });

        $scope.createNewTile = function() {
            CreateModalFromTemplate('/templates/dashboards/insights/new-insight-modal.html', $scope, 'NewInsightModalController', function(modalScope) {
                modalScope.withSimpleTiles = true;
                modalScope.inDashboard = true;
                modalScope.pointerMode = { isPointerMode: false };
                modalScope.dashboardTheme = $scope.dashboard.theme;
                $scope.filtersCompatibleTiles = DashboardFilters.getSlideFilterableTiles($state.params.pageId, $scope.dashboard.pages);
                modalScope.setDashboardCreationId($scope.dashboard.id);
                modalScope.addToDashboard = function(insight) {
                    if (!modalScope.pointerMode.isPointerMode) {
                        DataikuAPI.dashboards.insights.copy($scope.dashboard.projectKey, [insight.id], [insight.name], $scope.dashboard.id)
                            .error(setErrorInScope.bind($scope))
                            .success(function(data) {
                                const insightCopy = data[0];
                                $scope.insightsMap[insightCopy.id] = insightCopy;
                                $scope.addInsightToPage(insightCopy);
                                DashboardUtils.sendWT1InsightCreation(insightCopy.type, {
                                    triggeredFrom: 'dashboard-new-tile-modal-pick-existing'
                                });
                                ActivityIndicator.success('Added');
                            });
                    } else {
                        $scope.addInsightToPage(insight);
                        ActivityIndicator.success('Added');
                    }
                    DashboardUtils.sendWT1TileCreation('INSIGHT', {
                        insightType: insight.type,
                        reuseExistingInsight: modalScope.pointerMode.isPointerMode,
                        triggeredFrom: 'dashboard-new-tile-modal-pick-existing'
                    });
                };
                modalScope.isInsightInDashboard = function(insight) {
                    return typeof($scope.insightsMap[insight.id]) !== 'undefined';
                };
            }, undefined, 'static', false).then(function(ret) {
                if (ret.tileType) { // simple tile
                    $scope.addSimpleTileToPage(ret.tileType);
                    DashboardUtils.sendWT1TileCreation(ret.tileType, {
                        triggeredFrom: 'dashboard-new-tile-modal'
                    });

                } else {
                    if (DashboardUtils.getInsightHandler(ret.insight.type).goToEditAfterCreation) {
                        $scope.$on('dashboardSyncModelsDone', function() {
                            $scope.saveDashboard().success(function() {
                                $timeout(function() {
                                    const options = {
                                        name: ret.insight.name,
                                        tab: 'edit',
                                        originDashboard: {
                                            id: $scope.dashboard.id,
                                            name: $scope.dashboard.name,
                                            pageId: $scope.dashboard.pages[$scope.uiState.currentPageIdx].id,
                                            edit: true
                                        }
                                    };
                                    StateUtils.go.insight(ret.insight.id, $scope.dashboard.projectKey, options);
                                });
                            });
                        });
                    }

                    $scope.insightsMap[ret.insight.id] = ret.insight;
                    $scope.insightAccessMap[ret.insight.id] = ret.insight.isReaderAccessible ? 'READER' : 'ANALYST';
                    $scope.addInsightToPage(ret.insight);
                    DashboardUtils.sendWT1TileCreation('INSIGHT', {
                        insightType: ret.insight.type,
                        triggeredFrom: 'dashboard-new-tile-modal'
                    });
                }
            });
        };

        $scope.$on('dashboardOpenAddInsightModal', $scope.createNewTile);

        $scope.addFilterTileToPage = function() {
            const filterInsight = {
                type: 'filters',
                name:'Filters',
                params: {},
                dashboardCreationId: $scope.dashboard.id,
                projectKey: $scope.dashboard.projectKey,
                titleOptions: { showTitle: 'NO' }
            };
            DataikuAPI.dashboards.insights.save(filterInsight)
                .error(setErrorInScope.bind($scope))
                .success((insightId) => {
                    filterInsight.id = insightId;
                    $scope.addInsightToPage(filterInsight);
                });
        };

        $scope.$on('dashboardAddFilterTile', $scope.addFilterTileToPage);

        $scope.addInsightToPage = function(insight) {
            if (!$scope.insightsMap[insight.id]) {
                $scope.insightsMap[insight.id] = insight;
            }

            $scope.addTileToPage(TileUtils.newInsightTile(insight));
        };

        $scope.addTextTileToPage = function() {
            $scope.addTileToPage({
                tileType: 'TEXT',
                tileParams: TileUtils.getDefaultTileParams({ type: 'text' }),
                box: TileUtils.getDefaultTileBox({ type: 'text' }),
                titleOptions: { showTitle: 'NO' }
            });
        };

        $scope.addSimpleTileToPage = function(type) {
            let tile = {
                tileType: type.toUpperCase(),
                tileParams: TileUtils.getDefaultTileParams({ type: type.toLowerCase() }),
                box: TileUtils.getDefaultTileBox({ type: type.toLowerCase() }),
                titleOptions: {}
            };

            if (type.toUpperCase() == 'IMAGE') {
                tile.displayMode = 'IMAGE';
                tile.titleOptions.showTitle = 'MOUSEOVER';
                tile.resizeImageMode = 'FIT_SIZE';
            } else if (type.toUpperCase() == 'GROUP') {
                const page = $scope.dashboard.pages[$scope.uiState.currentPageIdx];
                tile = TileUtils.createGroupTile([], { useDashboardSpacing: true, tileSpacing: $scope.dashboard.tileSpacing, backgroundColor: page.backgroundColor });
            } else {
                tile.titleOptions.showTitle = 'NO';
            }

            $scope.addTileToPage(tile);
        };

        $scope.switchPage = (index) => {
            $scope.uiState.currentPageIdx = index;
        };


        $scope.createPage = function() {
            const page = {
                /*
                 * client id or cid is a unique identifier automatically assigned
                 * when they're first created.
                 * Client ids are handy when the model has not yet been saved to the server,
                 * and does not yet have its eventual true id, but already needs to be visible in the UI.
                 */
                cid: _.uniqueId(),
                titleAlign: 'CENTER',
                grid: {
                    tiles: []
                },
                enableCrossFilters: true,
                showFilterPanel: true,
                filtersParams: {
                    panelPosition: FiltersPanelPosition.TOP
                },
                show: true
            };
            $timeout(() => {
                DSSVisualizationThemeUtils.applyToPage(page, $scope.dashboard.theme);
                $scope.dashboard.pages.push(page);
                $scope.uiState.currentPageIdx = $scope.dashboard.pages.length - 1;
                $scope.$apply();
            });

            $scope.hasFilterTile = alreadyHaveGlobalFilters();
            WT1.event('dashboard-page-create', { newPageCount: $scope.dashboard.pages.length });
        };

        $scope.removePage = function(pageIdx) {
            $scope.dashboard.pages.splice(pageIdx, 1);
            if ($scope.dashboard.pages.length === 0) {
                $scope.createPage();
            }
            $scope.uiState.currentPageIdx = $scope.uiState.currentPageIdx ? Math.min($scope.uiState.currentPageIdx, $scope.dashboard.pages.length - 1) : $scope.dashboard.pages.length - 1;
            $scope.hasFilterTile = alreadyHaveGlobalFilters();
        };

        $scope.openCopySlideModal = function(pageIdx) {
            CreateModalFromTemplate('/templates/dashboards/copy-page-modal.html', $scope, 'CopyPageController', function(newScope) {
                newScope.pages = $scope.dashboard.pages;
                newScope.page = $scope.dashboard.pages[pageIdx];
                newScope.pageIdx = pageIdx;
                newScope.uiState = $scope.uiState;
                newScope.insightsMap = $scope.insightsMap;
                newScope.init();
            });
        };

        $scope.handlePageIndexChange = (previousIndex, currentIndex) => {
            if (previousIndex === $scope.uiState.currentPageIdx) {
                $scope.uiState.currentPageIdx = currentIndex;
            } else if (previousIndex < $scope.uiState.currentPageIdx && currentIndex >= $scope.uiState.currentPageIdx) {
                $scope.uiState.currentPageIdx--;
            } else if (previousIndex > $scope.uiState.currentPageIdx && currentIndex <= $scope.uiState.currentPageIdx) {
                $scope.uiState.currentPageIdx++;
            }
        };

        $scope.goToPage = (index) => {
            $scope.uiState.currentPageIdx = index;
            $scope.uiState.selectedTile = null;
        };

        $scope.goToNextPage = function() {
            this.goToPage(DashboardPageUtils.getNextPageIndex($scope.dashboard, $scope.uiState.currentPageIdx));
        };

        $scope.goToPreviousPage = function() {
            this.goToPage(DashboardPageUtils.getPreviousPageIndex($scope.dashboard, $scope.uiState.currentPageIdx));
        };

        $scope.addTileToPage = function(tile) {
            tile.backgroundOpacity = 1;
            tile.locked = false;
            tile.borderOptions = angular.copy(TileUtils.DEFAULT_BORDER_OPTIONS);
            tile.titleOptions = { ...tile.titleOptions, ...angular.copy(TileUtils.DEFAULT_TITLE_OPTIONS) };

            $scope.dashboard.pages[$scope.uiState.currentPageIdx].grid.tiles.push(tile);
            DSSVisualizationThemeUtils.applyToTile(tile, $scope.dashboard.theme);
            $scope.uiState.selectedTile = tile;
        };

        // TODO: factorize with tile-params directive
        $scope.getDefaultTileTitle = (tile, insight) => {
            if (!tile) {
                return '';
            }
            switch (tile.tileType) {
                case 'INSIGHT':
                    return (insight || {}).name;
                case 'TEXT':
                    return 'Text tile';
                case 'IMAGE':
                    return 'Image tile';
            }
        };

        $scope.copyURL = (urlToCopy) => {
            DashboardUtils.copyToClipboard(urlToCopy);
        };

        $scope.$on('$viewContentLoaded', () => {
            // Action requested from view mode
            if ($scope.editActionRequested) {
                switch($scope.editActionRequested.type) {
                    case 'create':
                        $scope.createPage();
                        break;
                    case 'remove':
                        if ($scope.editActionRequested.pageIdx !== undefined) {
                            $scope.removePage($scope.editActionRequested.pageIdx);
                        }
                        break;
                    case 'duplicate':
                        if ($scope.editActionRequested.pageIdx !== undefined) {
                            $scope.duplicatePage($scope.editActionRequested.pageIdx);
                        }
                        break;
                    case 'edit':
                        if ($scope.editActionRequested.pageIdx !== undefined) {
                            $scope.goToPage($scope.editActionRequested.pageIdx);
                        }
                        break;
                }
                $scope.editActionRequested = null;
            }
        });

        $scope.$on('$destroy', () => {
            if (unregisterDashboardSelectTileEvent != null) {
                unregisterDashboardSelectTileEvent();
            }
            if (unregisterDashboardUnselectTilesEvent != null) {
                unregisterDashboardUnselectTilesEvent();
            }
        });

        function alreadyHaveGlobalFilters() {
            return !!DashboardFilters.findFiltersTileInPage($scope.dashboard.pages[$scope.uiState.currentPageIdx]);
        }

        function retrieveInsightIdSetFromDashboard(dashboard) {
            const insightIdsSet = new Set();
            const stack = [];
            dashboard.pages.forEach(page => stack.push(...page.grid.tiles));
            while (stack.length) {
                const currentTile = stack.pop();
                if (currentTile.insightId != null) {
                    insightIdsSet.add(currentTile.insightId);
                }
                if (TileUtils.isGroupTile(currentTile) &&
                        currentTile.grid != null &&
                        currentTile.grid.tiles != null &&
                        currentTile.grid.tiles.length > 0) {
                    stack.push(...currentTile.grid.tiles);
                }
            }
            return insightIdsSet;
        }
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.controller('DashboardPageRightColumnActions', async function($controller, $scope, $rootScope, $stateParams, DataikuAPI, ActiveProjectKey) {

        $controller('_TaggableObjectPageRightColumnActions', { $scope: $scope });

        const dashboard = (await DataikuAPI.dashboards.get($stateParams.projectKey, $stateParams.dashboardId)).data;
        dashboard.nodeType = 'DASHBOARD';
        dashboard.interest = {};

        $scope.selection = {
            selectedObject: dashboard,
            confirmedItem: dashboard
        };

        $scope.renameObjectAndSave = function(newName) {
            $scope.dashboard.name = newName;
            return DataikuAPI.dashboards.save($scope.dashboard);
        };

        function updateListed() {
            DataikuAPI.dashboards.get($stateParams.projectKey, $stateParams.dashboardId).success(function(data) {

                $scope.selection.selectedObject.listed = data.listed;

            }).error(setErrorInScope.bind($scope));
        }

        function updateUserInterests() {
            DataikuAPI.interests.getForObject($rootScope.appConfig.login, 'DASHBOARD', ActiveProjectKey.get(), $stateParams.dashboardId).success(function(data) {

                $scope.selection.selectedObject.interest = data;
                $scope.dashboardFullInfo.interest = data;

            }).error(setErrorInScope.bind($scope));
        }

        updateUserInterests();
        const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
        const listedListener = $scope.$on('objectTimelineChanged', updateListed);
        $scope.$on('$destroy', interestsListener);
        $scope.$on('$destroy', listedListener);
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/view.js
     */
    app.controller('DashboardViewController', function($scope, $location, $stateParams, $state, Notification, DashboardUtils, DashboardPageUtils) {

        if (typeof($scope.uiState)==='undefined'){
            $scope.uiState = {};
        }

        $scope.isInExport = getCookie('dku_graphics_export') === 'true';

        if (DashboardUtils.isDashboardEmbeddedInAWorkspace()) {
            // Implementing show/hide filters button in workspace is too complex, so we always force the visibility of filter panels
            DashboardUtils.toggleAllFilterPanels($scope.dashboard, true);
        } else {
            DashboardUtils.showFiltersPanelForPagesWithFilters($scope.dashboard);
        }

        $scope.getNextVisiblePageIdx = () => {
            let nextVisiblePageIdx;
            if ($scope.uiState.currentPageIdx === $scope.dashboard.pages.length-1) {
                return;
            }
            for (let i = $scope.uiState.currentPageIdx + 1; i < $scope.dashboard.pages.length; i++) {
                if ($scope.dashboard.pages[i].show) {
                    nextVisiblePageIdx = i;
                    break;
                }
            }
            return nextVisiblePageIdx;
        };

        $scope.hasNextVisiblePage = () => {
            return !_.isNil($scope.getNextVisiblePageIdx());
        };

        $scope.getPreviousVisiblePageIdx = () => {
            let prevVisiblePageIdx;
            if (!$scope.uiState.currentPageIdx) {
                return;
            }
            for (let i = $scope.uiState.currentPageIdx - 1; i >= 0; i--) {
                if ($scope.dashboard.pages[i].show) {
                    prevVisiblePageIdx = i;
                    break;
                }
            }
            return prevVisiblePageIdx;
        };

        $scope.hasPreviousVisiblePage = () => {
            return !_.isNil($scope.getPreviousVisiblePageIdx());
        };

        $scope.$watch('fullScreen', function(nv) {
            const wrapper = $('body')[0];
            if (nv == null) {
                return;
            }

            if (nv) {
                requestFullscreen();
            } else {
                exitFullscreen();
            }
            $state.go($state.current, { fullScreen: (nv && nv != 'false') ? true : null }, { location: true, inherit:true, notify:false, reload:false });

            $scope.$on('$destroy', () => {
                exitFullscreen();
            });

            function handleFullscreenExit() {
                if (!document.fullscreenElement) {
                    $scope.fullScreen = false;
                    document.removeEventListener('fullscreenchange', handleFullscreenExit);
                }
            }

            function requestFullscreen() {
                const startFsNames = ['requestFullscreen', 'mozRequestFullScreen', 'webkitRequestFullScreen'];
                executeFirstAvailableFunction(startFsNames, wrapper);
                document.addEventListener('fullscreenchange', handleFullscreenExit);
            }

            function exitFullscreen() {
                if (document.fullscreenElement ||
                    document.webkitFullscreenElement ||
                    document.mozFullScreenElement) { // Can't exit fullscreen if there is no element in fullscreen
                    const stopFsNames = ['exitFullscreen', 'cancelFullScreen', 'webkitCancelFullScreen', 'mozCancelFullScreen'];
                    executeFirstAvailableFunction(stopFsNames, document);
                }
            }

            function executeFirstAvailableFunction(functionNames, elmt) {
                for (let i = 0; i < functionNames.length; i++) {
                    if (elmt[functionNames[i]]) {
                        elmt[functionNames[i]]();
                        break;
                    }
                }
            }
        });
        $scope.fullScreen = $stateParams.fullScreen && $stateParams.fullScreen !== 'false';
        $scope.embedded = $stateParams.embedded && $stateParams.embedded !== 'false';

        $scope.switchPage = function(index) {
            $scope.uiState.currentPageIdx = index;
            // clear query params filters on slide change
            $location.search('filters', undefined);
            $location.search('pageFilters', undefined);
        };

        $scope.goToNextPage = function() {
            $scope.switchPage(DashboardPageUtils.getNextPageIndex($scope.dashboard, $scope.uiState.currentPageIdx));
        };

        $scope.goToPreviousPage = function() {
            $scope.switchPage(DashboardPageUtils.getPreviousPageIndex($scope.dashboard, $scope.uiState.currentPageIdx));
        };

        $scope.$on('toggleFullScreen', () => {
            $scope.fullScreen = !$scope.fullScreen;
        });

        Notification.registerEvent('dashboards-reload-needed',function(evt, data) {
            if (data && data.params) {
                const dashboardConcerned = data.params.dashboards.some(d => d.projectKey === $stateParams.projectKey && $stateParams.dashboardId === d.id);

                if (dashboardConcerned) {
                    $scope.$emit('dashboards-reload-needed');
                }
            }
        });
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.controller('DashboardsCommonController', function($scope, TileUtils, $rootScope, TopNav, DataikuAPI, ActivityIndicator, CreateModalFromTemplate, $state, WT1, DashboardUtils) {
        function makeDashboardListed(dashboard, noNotification) {
            return DataikuAPI.dashboards.makeListed(dashboard.projectKey, [dashboard.id], !dashboard.listed)
                .success(function(data) {
                    if (!noNotification) {
                        ActivityIndicator.success('Saved!');
                    }
                    $scope.$broadcast('objectTimelineChanged');
                    dashboard.listed = !dashboard.listed;
                    if ($scope.origDashboard) {
                        $scope.origDashboard.listed = dashboard.listed;
                    }
                }).error(setErrorInScope.bind($scope));
        }

        function findFilterTiles(pages) {
            const filterTilesMap = new Object();
            pages && pages.forEach(page => {
                const filterTile = page.grid.tiles.find(tile => TileUtils.isFilterTile(tile));
                if (filterTile) {
                    filterTilesMap[page.id] = filterTile;
                }
            });
            return filterTilesMap;
        }

        $scope.DashboardUtils = DashboardUtils;

        $scope.canEditDashboard = function(dashboard) {
            return dashboard && $scope.canWriteDashboards() && ($scope.canModerateDashboards() || dashboard.owner == $scope.appConfig.login);
        };

        $scope.toggleDashboardListed = function(dashboard, closeToggle) {
            if (!dashboard.listed && dashboard.hasMissingReaderAuthorizations) {
                CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                    newScope.initForDashboards([dashboard], true);
                }).then(function() {
                    makeDashboardListed(dashboard, true).success($scope.list);
                });
            } else {
                if (closeToggle) {
                    $('.tooltip').remove();
                }
                makeDashboardListed(dashboard, true);
            }
        };

        $scope.makeDashboardListed = makeDashboardListed;

        $scope.openInsightAccessModal = function(dashboard) {
            CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                newScope.initForDashboards([dashboard], false);
            }).then($scope.list);
        };


        $scope.copy = function(dashboard, callBackFunc) {
            CreateModalFromTemplate('/templates/dashboards/copy-dashboard-modal.html', $scope, 'CopyDashboardModalController', function(newScope) {
                newScope.init(dashboard);
            })
                .then(function() {
                    if (typeof callBackFunc === 'function') {
                        callBackFunc();
                    }
                });
        };


        $scope.exportDashboard = function(massExport, fromDashboardViewOrEdit = false, dashboardNotSaved = false) {
            CreateModalFromTemplate('/templates/dashboards/export-dashboard-modal.html', $scope, 'ExportDashboardModalController', function(newScope) {
                if (fromDashboardViewOrEdit) {
                    if ($scope.dashboard.pages.length !== 1) {
                        newScope.showCheckbox = true;
                        newScope.pageIdx = newScope.uiState.currentPageIdx;
                    }
                    newScope.dashboardNotSaved = dashboardNotSaved;
                }

                const fromByStateName = {
                    'projects.project.dashboards.dashboard.edit': 'EDIT',
                    'projects.project.dashboards.dashboard.view': 'VIEW',
                    'projects.project.dashboards.list': 'LIST'
                };
                const options = {
                    from: fromByStateName[$state.current.name],
                    massExport
                };
                newScope.init($scope.dashboard, options);
            });
        };


        $scope.saveAndCopy = function(dashboard, callBackFund) {
            $scope.saveDashboard().then(function() {
                $scope.copy($scope.dashboard);
            });
        };

        $scope.saveCustomFields = function(newCustomFields) {
            WT1.event('custom-fields-save', { objectType: 'DASHBOARD' });
            const oldCustomFields = angular.copy($scope.dashboard.customFields);
            $scope.dashboard.customFields = newCustomFields;
            return $scope.saveDashboard().then(function() {
                $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), $scope.dashboard.customFields);
            }, function() {
                $scope.dashboard.customFields = oldCustomFields;
            });
        };

        $scope.editCustomFields = function() {
            if (!$scope.dashboard) {
                return;
            }
            const modalScope = angular.extend($scope, { objectType: 'DASHBOARD', objectName: $scope.dashboard.name, objectCustomFields: $scope.dashboard.customFields });
            CreateModalFromTemplate('/templates/taggable-objects/custom-fields-edit-modal.html', modalScope).then(function(customFields) {
                $scope.saveCustomFields(customFields);
            });
        };

        $scope.deleteUnusedFiltersInsights = function() {
            /*
             * Delete previous dashboard filters when needed :
             * -> if a new filter insight has replaced a previous one from slide; or if it was on a slide that has been deleted.
             */
            const oldFilterTiles = findFilterTiles($scope.origDashboard.pages);
            const newFilterTiles = $scope.currentFilterTiles || findFilterTiles($scope.dashboard.pages);
            for (const [key, oldFilterTile] of Object.entries(oldFilterTiles)) {
                const isDeleted = !newFilterTiles[key];
                const isReplaced = newFilterTiles[key] && newFilterTiles[key].insightId !== oldFilterTile.insightId;
                if (isDeleted || isReplaced) {
                    const deletionRequests = [{
                        type: 'INSIGHT',
                        projectKey: $scope.dashboard.projectKey,
                        id: oldFilterTile.insightId,
                        displayName: oldFilterTile.name
                    }];
                    DataikuAPI.taggableObjects.delete(deletionRequests, $scope.dashboard.projectKey).error(setErrorInScope.bind($scope));
                }
            };
        };

        $scope.saveDashboard = function(commitMessage) {
            $scope.currentFilterTiles = findFilterTiles($scope.dashboard.pages);
            if (Object.keys($scope.currentFilterTiles).length) {
                WT1.event('dashboard-filters-save');
            }

            $scope.$broadcast('dashboardSaved');

            return DataikuAPI.dashboards.save($scope.dashboard, commitMessage)
                .success(function(data) {
                    data.pages.forEach(function(page, i) {
                        $scope.dashboard.pages[i].id = page.id;
                    });
                    $scope.dashboard.versionTag = data.versionTag;

                    if ($scope.origDashboard) {
                        $scope.deleteUnusedFiltersInsights();
                    }

                    $scope.origDashboard = angular.copy($scope.dashboard);
                }).error(setErrorInScope.bind($scope));
        };

        $scope.$on('saveDashboard', () => $scope.saveDashboard());

        $rootScope.$on('dashboard-edit-action-request', (_, options) => {
            $scope.editActionRequested = options;
        });
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Everything to store and manipulate dashboard filters.
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.controller('DashboardsListController', function($scope, $controller, $stateParams, DataikuAPI, CreateModalFromTemplate, Dialogs, $state, $q, TopNav, Fn, $filter, ActivityIndicator, DashboardUtils, StateUtils) {

        $controller('_TaggableObjectsListPageCommon', { $scope: $scope });
        $controller('DashboardsCommonController', { $scope: $scope });

        $scope.DashboardUtils = DashboardUtils;

        $scope.listHeads = DataikuAPI.dashboards.listHeads;
        $scope.getDashboardPromotionTitle = DashboardUtils.getTooltipPromotionTitleFunction('dashboard');

        $scope.sortBy = [
            { value: 'name', label: 'Name' },
            { value: '-lastModifiedOn', label: 'Last modified' }
        ];
        $scope.selection = $.extend({
            filterQuery: {
                userQuery: '',
                tags: [],
                listed: '',
                interest: {
                    starred: ''
                }
            },
            filterParams: {
                userQueryTargets: ['name', 'tags'],
                propertyRules: { tag: 'tags' }
            },
            inclusiveFilter: {
                owner: []
            },
            customFilterWatch: 'selection.inclusiveFilter',
            customFilter: function(objList) {
                return objList.filter(function(obj) {
                    if ($scope.selection.inclusiveFilter.owner.length > 0) {
                        if ($scope.selection.inclusiveFilter.owner.indexOf(obj.owner) > -1) {
                            return true;
                        }
                        return false;
                    } else {
                        return true;
                    }
                });
            },
            orderQuery: '-lastModifiedOn',
            orderReversed: false
        }, $scope.selection || {});
        $scope.sortCookieKey = 'dashboards';
        $scope.maxItems = 20;

        $scope.setOwnerFilter = function(owner) {
            if (!owner) {
                $scope.selection.inclusiveFilter.owner = [];
                return;
            }

            const arr = $scope.selection.inclusiveFilter.owner;
            const index = arr.indexOf(owner);

            if (index > -1) {
                arr.splice(index, 1);
            } else {
                arr.push(owner);
            }
        };

        $scope.setListedFilterQuery = function(value) {
            $scope.selection.filterQuery.listed = value ? 'true' : '';
        };

        if ($state.current.name.indexOf('dashboards') != -1) {
            TopNav.setLocation(TopNav.TOP_DASHBOARD, 'dashboards', TopNav.TABS_NONE, null);
            TopNav.setNoItem();
        }

        $scope.list();

        $scope.$watch('selection.selectedObject', function(nv) {
            if (!nv) {
                return;
            }

            DataikuAPI.dashboards.getSummary($stateParams.projectKey, nv.id).success(function(data) {
                $scope.dashboard = data.object;
                $scope.dashboard.pages.forEach(page => {
                    /*
                     * client id or cid is a unique identifier automatically assigned
                     * when they're first created.
                     * Client ids are handy when the model has not yet been saved to the server,
                     * and does not yet have its eventual true id, but already needs to be visible in the UI.
                     */
                    page.cid = _.uniqueId();
                });
            }).error(setErrorInScope.bind($scope));
        });

        /* Specific actions */
        $scope.goToItem = function(data) {
            $state.go('projects.project.analyses.analysis.script', { projectKey: $stateParams.projectKey, analysisId: data.id });
        };

        $scope.newDashboard = function(language) {
            CreateModalFromTemplate('/templates/dashboards/new-dashboard-modal.html', $scope)
                .then(function(dashboard) {
                    StateUtils.go.dashboard(dashboard.id, dashboard.projectKey, { name: dashboard.name, tab: 'edit' });
                });
        };

        $scope.isAllListed = function(items) {
            if (!items) {
                return true;
            }
            return items.map(Fn.prop('listed')).reduce(function(a, b) {
                return a && b;
            }, true);
        };

        $scope.canMassMakeListed = true;
        $scope.massMakeListed = function(items, listed) {
            const apiCall = function() {
                DataikuAPI.dashboards.makeListed(items[0].projectKey, ids, listed)
                    .success(function(data) {
                        ActivityIndicator.success('Saved!');
                        $scope.list();
                    }).error(setErrorInScope.bind($scope));
            };

            if (!(items && items.length > 0)) {
                return;
            }
            var ids = [];
            const hasMissingReaderAuthorizationsItems = [];
            items.forEach(function(item) {
                if (item.listed != listed) {
                    ids.push(item.id);
                }
                if (item.hasMissingReaderAuthorizations) {
                    hasMissingReaderAuthorizationsItems.push(item);
                }
            });

            if (listed && hasMissingReaderAuthorizationsItems.length > 0) {
                CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                    newScope.initForDashboards(hasMissingReaderAuthorizationsItems, true);
                }).then(apiCall);
            } else {
                apiCall();
            }
        };

        // Used in the list
        $scope.canMassExportDashboard = true;

        $scope.owners = [];
        $scope.list = function() {
            $scope.listHeads($stateParams.projectKey, $scope.tagFilter).success(function(data) {
                $scope.filteredOut = data.filteredOut;
                $scope.listItems = data.items;
                $scope.restoreOriginalSelection();

                const ownersMap = {};
                data.items.forEach(function(dashboard) {
                    ownersMap[dashboard.owner] = {
                        login: dashboard.owner,
                        displayName: dashboard.ownerDisplayName
                    };
                });
                $scope.owners.length = 0;
                for (const login in ownersMap) {
                    $scope.owners.push(ownersMap[login]);
                }
                $scope.owners.sort(function(a, b) {
                    if (a.displayName < b.displayName) {
                        return -1;
                    }
                    if (a.displayName > b.displayName) {
                        return 1;
                    }
                    return 0;
                });

            }).error(setErrorInScope.bind($scope));
        };
        $scope.list();

        $scope.getNbListedDashboards = function(dashboards) {
            if (dashboards && dashboards.length > 0) {
                return dashboards.filter(function(dashboard) {
                    return dashboard.listed;
                }).length;
            }
            return 0;
        };
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/edit.js
     */
    app.controller('ExportDashboardModalController', function($scope, DataikuAPI, ActivityIndicator, FutureProgressModal, WT1, DashboardFilters, DashboardFiltersUrlParams) {
        /**
         * @param {Dashboard | undefined} selection when doing mass export, the selection is undefined and
         * $scope.selection.selectedObjects contains the dashboards to export
         * @param {{ massExport: boolean, from: 'EDIT' | 'VIEW' | 'LIST' }} options
         */
        $scope.init = function(selection, options) {
            // Selection can correspond to multiple dashboards or a single dashboard
            $scope.params = {};

            let filtersBySlide = [];
            // It's only possible to take into account filters when from a dashboard in view mode.
            if (options.from === 'VIEW') {
                filtersBySlide = selection.pages.map(page => DashboardFiltersUrlParams.getFiltersQueryStringValue(DashboardFilters.getActiveGlobalFilters(page.id) || []));
            }

            if (options.massExport == true) {
                if ($scope.selection.selectedObjects.length > 0) {
                    $scope.projectKey = $scope.selection.selectedObjects[0].projectKey;
                } else {
                    $scope.projectKey = null;
                }
                $scope.params.dashboards = $scope.selection.selectedObjects.map(function(dashboard) {
                    return { dashboardId: dashboard.id, filtersBySlide, slideIndex: undefined };
                });
                if ($scope.selection.selectedObjects.length == 1) {
                    $scope.projectKey = $scope.dashboard.projectKey;
                    $scope.modalTitle = 'Export dashboard : ' + $scope.dashboard.name;
                } else {
                    $scope.projectKey = $scope.selection.selectedObjects.length == 0 ? null : $scope.selection.selectedObjects[0].projectKey;
                    $scope.modalTitle = 'Export a list of ' + $scope.selection.selectedObjects.length + ' dashboards';
                }
            } else {
                $scope.projectKey = $scope.dashboard.projectKey;
                $scope.params.dashboards = [ { dashboardId: $scope.dashboard.id, filtersBySlide, slideIndex: undefined } ];
                $scope.modalTitle = 'Export dashboard : ' + $scope.dashboard.name;
            }
        };

        $scope.exportDashboard = function() {
            WT1.event('dashboard-exported-from-dashboard-universe', {
                format: $scope.params.exportFormat
            });

            if ($scope.params.exportOnlyCurrentSlide && $scope.params.dashboards.length == 1) {
                $scope.params.dashboards[0].slideIndex = $scope.pageIdx;
            }
            DataikuAPI.dashboards.export($scope.projectKey, $scope.params.exportFormat, $scope.params.dashboards)
                .error(setErrorInScope.bind($scope))
                .success(function(resp) {
                    FutureProgressModal.show($scope, resp, 'Export dashboard').then(function(result) {
                        if (result) { // undefined in case of abort
                            downloadURL(DataikuAPI.dashboards.getExportURL(result.projectKey, result.exportId));
                            ActivityIndicator.success('Dashboard export(s) downloaded!', 5000);
                        } else {
                            ActivityIndicator.error('Export dashboard failed', 5000);
                        }
                        $scope.resolveModal();
                    });
                });
        };
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/view.js
     */
    app.controller('InsightTileTemplateController', function($scope, TileLoadingState) {
        $scope.getInsightTypeClass = function() {
            const tile = $scope.tile;
            return (tile.tileParams && tile.tileParams.viewKind === 'SEARCH') ? 'dataset_table_search' : tile.insightType;
        };

        $scope.isCredentialError = function() {
            return $scope.error && $scope.error.code && ($scope.error.code==='ERR_CONNECTION_OAUTH2_REFRESH_TOKEN_FLOW_FAIL' || $scope.error.code==='ERR_CONNECTION_NO_CREDENTIALS');
        };

        $scope.manualLoad = function() {
            $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
            $scope.load();
        };
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/edit.js
     */
    app.controller('NewDashboardModalController', function($scope, $stateParams, $rootScope, DataikuAPI, DashboardUtils, DSSVisualizationThemeUtils) {
        $scope.dashboard = {
            projectKey: $stateParams.projectKey,
            owner: $scope.appConfig.user.login,
            pages: [{ grid: { tiles: [] } }]
        };

        $scope.defaultDashboardName = DashboardUtils.getDashboardDefaultName();

        $scope.create = function() {
            $scope.dashboard.name = $scope.dashboard.name || $scope.defaultDashboardName;
            $scope.dashboard.autoStackUp = $rootScope.appConfig.userSettings.dashboards.autoStackUp;
            $scope.dashboard.theme = DSSVisualizationThemeUtils.getThemeOrDefault($rootScope.appConfig.selectedDSSVisualizationTheme);
            DSSVisualizationThemeUtils.applyToDashboard($scope.dashboard, $scope.dashboard.theme);

            DataikuAPI.dashboards.save($scope.dashboard).success(function(dashboard) {
                $scope.dashboard = dashboard;
                $scope.resolveModal($scope.dashboard);
            }).error(setErrorInScope.bind($scope));
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.controller('ProjectHomeDashboardListController', function($scope, $stateParams, DataikuAPI) {
        $scope.listHeads = DataikuAPI.dashboards.listSummaries($stateParams.projectKey).success(function(data) {
            $scope.dashboards = data.map(function(summary) {
                const dashboard = summary.object;
                dashboard.interest = summary.interest;
                dashboard.numTiles = 0;
                dashboard.pages.forEach(function(page) {
                    dashboard.numTiles += page.grid.tiles.length;
                });
                dashboard.numPages = dashboard.pages.length;
                return dashboard;
            });
        }).error(setErrorInScope.bind($scope));
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/edit.js
     */
    app.directive('dashboardExportForm', function(GRAPHIC_EXPORT_OPTIONS, WT1, GraphicImportService) {
        return {
            replace: false,
            require: '^form',
            restrict: 'EA',
            scope: {
                params: '=',
                origin: '@origin',
                pageIdx: '=?'
            },
            templateUrl: '/templates/dashboards/export-form.html',
            link: function($scope, element, attrs, formCtrl) {
                WT1.event('dashboard-export-form-displayed', {});

                const gridOrWindowWidth = function() {
                    if (document.querySelector('.dashboard-export-grid') != null) {
                        // Add css margin of 5 and 5 for each side
                        return Math.round(Math.max(960, document.querySelector('.dashboard-export-grid').getBoundingClientRect().width + 10));
                    } else {
                        return Math.round(Math.max(960, window.outerWidth));
                    }
                };

                $scope.exportFormController = formCtrl;
                // Utilities that give us all the choices possible
                $scope.paperSizeMap = GRAPHIC_EXPORT_OPTIONS.paperSizeMap;
                $scope.orientationMap = GRAPHIC_EXPORT_OPTIONS.orientationMap;
                $scope.fileTypes = GRAPHIC_EXPORT_OPTIONS.fileTypes;

                $scope.minResW = 960;
                $scope.minResH = 540;
                $scope.maxResW = 7200;
                $scope.maxResH = 10000;
                $scope.selectedDashboard = {};

                // Parameters of the export
                if (angular.isUndefined($scope.params.exportFormat)) {
                    $scope.params.exportFormat = {
                        paperSize: 'A4',
                        orientation: 'LANDSCAPE',
                        fileType: 'PDF',
                        width: ($scope.origin == 'modal') ? gridOrWindowWidth('LANDSCAPE') : 1920
                    };
                    $scope.params.exportFormat.height = GraphicImportService.computeHeight($scope.params.exportFormat.width, $scope.params.exportFormat.paperSize);
                }

                $scope.$watch('params.exportFormat.paperSize', function(newVal, oldVal) {
                    if (newVal !== oldVal) {
                        if (newVal != 'CUSTOM') {
                            $scope.params.exportFormat.width = ($scope.origin == 'modal') ? gridOrWindowWidth() :
                                $scope.params.exportFormat.orientation == 'PORTRAIT' ? 1080 : 1920;
                            $scope.params.exportFormat.height = GraphicImportService.computeHeight($scope.params.exportFormat.width, $scope.params.exportFormat.paperSize, $scope.params.exportFormat.orientation);
                        }
                    }
                });
            }
        }
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/view.js
     */
    app.directive('dashboardPage', function($timeout, CreateModalFromTemplate, AnyLoc, $rootScope, ChartFormattingPaneSections, DetectUtils,
        $state, $stateParams, executeWithInstantDigest,WT1, DashboardFilters, DashboardUtils, TileUtils, FiltersPanelPosition, FiltersPanelDirection, $location, DashboardPageUtils) {
        return {
            restrict: 'EA',
            replace: true,
            templateUrl: '/templates/dashboards/page.html',
            scope: {
                page: '=',
                insightsMap: '=',
                accessMap: '=',
                editable: '=',
                selectedTile: '=',
                showGrid: '=',
                pageIdx: '=',
                dashboard: '=',
                uiState: '=dkuUiState',
                canEditDashboard: '=',
                canModerateDashboards: '=',
                exportDatasetFn: '=',
                canExportDatasets: '='
            },
            link: function(scope) {

                function goToPage(pageIdx) {
                    if (pageIdx < 0) {
                        return Promise.reject('Invalid page index');
                    }
                    if (scope.uiState.currentPageIdx === pageIdx) {
                        return Promise.resolve();
                    }
                    executeWithInstantDigest(() => {
                        scope.isLoading = true;
                        scope.uiState.currentPageIdx = pageIdx;
                    }, scope);
                    return new Promise((resolve) => {
                        const unlisten = $rootScope.$on('dashboardUrlUpdateDone', () => {
                            unlisten();
                            resolve();
                        });
                    });
                }

                /*
                 * Dashboard Export Toolbox
                 */
                scope.dashboardExportToolbox = {

                    replaceWebGLCanvas: function() {
                        // Iterate over all 2D canvas and remove the display so we can see them
                        document.querySelectorAll('d3fc-group > canvas').forEach(canvasNode => {
                            canvasNode.style.display = 'block';
                        });
                    },

                    checkLoading: function() {
                        return scope.isLoading || scope.isLoadingFilters;
                    },

                    getPagesToExport: function() {
                        const indexes = [];
                        scope.dashboard.pages.forEach((page, index) => {
                            if (page.show) {
                                indexes.push(index);
                            }
                        });
                        return indexes;
                    },

                    getFirstVisiblePageIdx: function() {
                        return scope.dashboard.pages.findIndex(page => page.show);
                    },

                    getVerticalBoxesCount: function(pageIndex) {
                        let total = 0;
                        for (const tile of scope.dashboard.pages[pageIndex].grid.tiles) {
                            if (tile.box.top + tile.box.height > total) {
                                total = tile.box.top + tile.box.height;
                            }
                        }
                        return total;
                    },

                    getTilesCount: function(pageIndex) {
                        const pages = scope.dashboard.pages;
                        return (pageIndex >= 0 && pageIndex < pages.length) ? pages[pageIndex].grid.tiles.length : 0;
                    },

                    scroll: function(scrolledHeight) {
                        return document.querySelector('.dashboard-export-page-wrapper').scrollTop = scrolledHeight;
                    },

                    getTitleBoundaries: function() {
                        return document.querySelector('.dashboard-export-title').getBoundingClientRect();
                    },

                    getGridBoundaries: function() {
                        return document.querySelector('.dashboard-export-grid').getBoundingClientRect();
                    },

                    clearDashboard: function() {
                        executeWithInstantDigest(() => scope.uiState.hideForExport = true, scope);
                    },

                    goToFirstPage: function() {
                        goToPage(0);
                    },

                    goToPage,

                    applyFilters(pageFilters) {
                        executeWithInstantDigest(() => {
                            scope.isLoadingFilters = true;
                        });
                        DashboardFilters.resetGlobalFilters(scope.page.id);
                        return DashboardFilters.initPageFilters(scope.page, { pageFilters: pageFilters }, Object.values(scope.insightsMap))
                            .then(({ filters }) => {
                                scope.handleFiltersChange(filters);
                                scope.isLoadingFilters = false;
                            });
                    }
                };

                /*
                 * If all pages are hidden, the page is undefined, we display a placeholder and this directive is hidden.
                 * It needs to stay in the DOM though cause we need it for export.
                 */
                if (!scope.page) {
                    return;
                }

                if ($stateParams.pageId == null) {
                    $stateParams.pageId = scope.page.id;
                }

                scope.columnNumber = DashboardUtils.getColumnNumber();

                scope.$watchGroup(['page.backgroundColor', 'showGrid', 'editable'], function() {
                    scope.gridColor = DashboardPageUtils.getGridColor(scope.page.backgroundColor);
                    scope.gridContainerBackgroundColor = DashboardPageUtils.computeGridContainerBackgroundColor(scope.showGrid, scope.editable, scope.page.backgroundColor, scope.gridColor);
                });

                scope.FiltersPanelPosition = FiltersPanelPosition;
                scope.FiltersPanelDirection = FiltersPanelDirection;
                scope.isInExport = getCookie('dku_graphics_export') === 'true';
                scope.isFilterPanelDeactivated = !!DashboardFilters.getIsFilterPanelDeactivated(scope.page.id);

                DashboardPageUtils.flushCachedColumnSummaries();
                DashboardFilters.flushDatasetLowerCaseStatus();

                DashboardFilters.initPageFilters(scope.page, $location.search(), Object.values(scope.insightsMap))
                    .then(({ filters, activeFilters }) => {
                        scope.filters = filters;
                        scope.activeFilters = activeFilters;
                        $rootScope.$broadcast('dashboardFiltersChanged', { currentFilters: scope.activeFilters, previousFilters: scope.activeFilters });
                    });

                DashboardFilters.setCrossFilterStatusToPageId(scope.page.id, scope.page.enableCrossFilters);

                scope.tiles = scope.page.grid.tiles ? [...scope.page.grid.tiles] : [];
                scope.$watchCollection('page.grid.tiles', function(newTiles) {
                    scope.tiles = [...newTiles];
                });

                scope.redirectToEdit = () => {
                    WT1.event('dashboard-to-edit-mode-switch');
                    $state.go('projects.project.dashboards.dashboard.edit', {
                        projectKey: $stateParams.projectKey,
                        dashboardId: DashboardUtils.isDashboardEmbeddedInAWorkspace() ? $stateParams.objectId : $stateParams.dashboardId,
                        pageId: $stateParams.pageId
                    });
                };

                scope.$on('dashboardOpenInsightAccessModal', (_, { insight }) => {
                    openInsightAccessModal(insight);
                });

                scope.$on('dashboardOpenMoveCopyTileModal', (_, { tile }) => {
                    openMoveCopyTileModal(tile);
                });

                scope.$on('dashboardExportDatasetTile', (_, { tile }) => {
                    exportDatasetTile(tile);
                });

                scope.$on('dashboardOpenSamplingTab', (_, { tile, section }) => {
                    openSamplingTab(tile, section);
                });

                scope.$on('dashboardOpenMisc', (_, { tile }) => {
                    openMisc(tile);
                });

                scope.$on('dashboardOpenTileInsight', (_, { tile }) => {
                    openTileInsight(tile);
                });

                scope.shouldDisplayDockedFiltersPanel = (page, filters, editable) => {
                    if (scope.isInExport) {
                        return false;
                    }
                    const canFilter = DashboardFilters.canFilter(editable, filters, page.enableCrossFilters);
                    const isDocked = DashboardFilters.isFiltersPanelDocked(page.filtersParams.panelPosition);
                    const isShown = page.showFilterPanel;
                    return canFilter && isDocked && isShown;
                };

                scope.handlePreselectedTileChange = function(preselectedTile) {
                    $timeout(() => {
                        scope.selectedTile = preselectedTile;
                    });
                };

                scope.handleFiltersChange = function(filters) {
                    $timeout(() => {
                        scope.filters = filters;
                        if (scope.editable) {
                            scope.page.filters = filters;
                        }
                        $rootScope.$broadcast('dashboardFiltersChanged', { currentFilters: filters, previousFilters: scope.activeFilters });
                        const activeFilters = DashboardFilters.getActiveGlobalFilters(scope.page.id);
                        if (!(activeFilters || []).length && !(scope.activeFilters || []).length) {
                            return;
                        } else {
                            scope.activeFilters = activeFilters;
                        }
                    });
                };

                scope.removeTileFromDashboardPage = (tileId) => {
                    return _removeTileFromGrid(scope.page.grid, tileId);
                };

                function _removeTileFromGrid(grid, tileId) {
                    const newTiles = grid.tiles.filter(tile => tile.$tileId !== tileId);

                    if (newTiles.length < grid.tiles.length) {
                        grid.tiles = newTiles;
                        return true;
                    }

                    for (const tile of grid.tiles) {
                        if (tile.tileType === 'GROUP' && tile.grid) {
                            if (_removeTileFromGrid(tile.grid, tileId)) {
                                return true;
                            }
                        }
                    }

                    return false;
                }

                scope.handleTilesChange = function(tiles) {
                    $timeout(() => {
                        scope.page.grid.tiles = tiles;
                    });
                };

                scope.handleGridLoadingStateChange = function(isLoading) {
                    $timeout(() => {
                        scope.isLoading = isLoading;
                    });
                };

                scope.handleIsDeactivatedChange = function(isDeactivated) {
                    $timeout(() => {
                        DashboardFilters.setIsFilterPanelDeactivated(scope.page.id, isDeactivated);
                        scope.activeFilters = isDeactivated ? [] : scope.filters;
                    });
                };

                scope.handleFiltersParamsChange = function(filtersParams) {
                    $timeout(() => {
                        if (filtersParams.panelPosition !== scope.page.filtersParams.panelPosition) {
                            handlePanelPositionChange(filtersParams.panelPosition);
                        }
                        scope.page.filtersParams = filtersParams;
                    });
                };

                scope.handleRaisedError = function(errorData) {
                    const { data, status, headers, config, statusText } = errorData;
                    setErrorInScope.bind(scope)(data, status, headers, config, statusText);
                };

                scope.handleGridInit = function(api) {
                    scope.reloadGrid = api.reload;
                };

                scope.addElements = function() {
                    scope.$emit('dashboardOpenAddInsightModal');
                };

                scope.$on('dashboard-reload-tiles', () => {
                    if (scope.reloadGrid) {
                        scope.reloadGrid();
                    }
                });

                $rootScope.spinnerPosition = 'dashboard-page';

                function openSamplingTab(tile, section) {
                    if (tile.insightType === 'chart' || tile.insightType === 'dataset_table') {
                        const tabName = tile.insightType === 'chart' ? 'sampling-engine' : 'sample-settings';
                        redirectToInsight({ tile, tabName, section });
                    }
                };

                function openMisc(tile) {
                    redirectToInsight({ tile, tabName: null, section: ChartFormattingPaneSections.MISC });
                };

                function openTileInsight(tile) {
                    if (!scope.editable && tile.clickAction !== 'DO_NOTHING' && TileUtils.canLink(tile)) {
                        redirectToInsight({ tile });
                    }
                };

                function exportDatasetTile(tile) {
                    if (tile.insightType === 'dataset_table' && scope.exportDatasetFn) {
                        const insight = scope.insightsMap[tile.insightId];
                        if (insight && insight.params) {
                            const loc = AnyLoc.getLocFromSmart(insight.projectKey, insight.params.datasetSmartName);
                            const hasFilters = insight.$shakerAndDashboardExplorationFilters != null &&
                                        insight.$shakerAndDashboardExplorationFilters.length > 0;
                            scope.exportDatasetFn(loc.projectKey, loc.localId, {
                                exportFiltersText: 'Apply filters from the dashboard',
                                downloadMethod: false,
                                downloadOnly: true,
                                advancedSampling: false,
                                pluginsExport: true,
                                allowFiltering: hasFilters,
                                applyExplorationFilters: hasFilters,
                                explorationFiltersAndSearchQuery: {
                                    explorationFilters: insight.$shakerAndDashboardExplorationFilters
                                }
                            });
                            WT1.event('dashboard-export-dataset-click');
                        }
                    }
                };

                function redirectToInsight({ tile, tabName, section }) {
                    const insight = scope.insightsMap[tile.insightId];
                    const targetInsight = scope.insightsMap[tile.targetInsightId] || { id: tile.targetInsightId };
                    const sref = TileUtils.getTargetUiSref(tile, insight, scope.editable, targetInsight, tabName, section);
                    const [, route, params] = /((?:[a-z]|\.)+)\((.+)\)/i.exec(sref);

                    if (sref) {
                        $state.go(route, JSON.parse(params));
                    }
                };

                function openInsightAccessModal(insight) {
                    CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', scope, null, function(newScope) {
                        newScope.initForInsights([insight], false);
                    }).then(function(readerAccessChanged) {
                        if (readerAccessChanged) {
                            scope.$emit('dashboardReaderAccessChanged');
                        }
                    });
                };

                function openMoveCopyTileModal(tile) {
                    CreateModalFromTemplate('/templates/dashboards/insights/move-copy-tile-modal.html', scope, 'MoveCopyTileModalController', function(newScope) {
                        newScope.dashboard = scope.dashboard;
                        newScope.page = scope.page;
                        newScope.tile = tile;
                        newScope.uiState = scope.uiState;
                        newScope.insight = tile.tileType == 'INSIGHT' ? scope.insightsMap[tile.insightId] : null;
                        newScope.insightName = tile.tileType == 'INSIGHT' ? newScope.insight.name : tile.titleOptions.title;
                        newScope.insightsMap = scope.insightsMap;
                        newScope.init(newScope.insight, tile.tileParams);
                    });
                };

                function handlePanelPositionChange(panelPosition) {
                    if (panelPosition === FiltersPanelPosition.TILE) {
                        const filterTile = DashboardFilters.findFiltersTileInPage(scope.page);
                        if (filterTile == null) {
                            scope.$emit('dashboardAddFilterTile');
                        }
                    } else {
                        scope.$broadcast('dashboardRemoveFilterTile');
                    }
                };

                /** DASHBOARDS SHORTCUTS */
                const groupingShortcut = DetectUtils.getOS() === 'macos' ? 'command+option+g' : 'ctrl+alt+g';

                Mousetrap.bind(groupingShortcut, function() {
                    $rootScope.$broadcast('dashboardToggleTileGrouping');
                });

                const ctrlOrOption = DetectUtils.getOS() === 'macos' ? 'option' : 'ctrl';
                const nextPageShortcuts = [ctrlOrOption + '+down', ctrlOrOption + '+right'];
                const previousPageShortcuts = [ctrlOrOption + '+up', ctrlOrOption + '+left'];

                Mousetrap.bind(nextPageShortcuts, function() {
                    goToPage(DashboardPageUtils.getNextPageIndex(scope.dashboard, scope.uiState.currentPageIdx));
                });

                Mousetrap.bind(previousPageShortcuts, function() {
                    goToPage(DashboardPageUtils.getPreviousPageIndex(scope.dashboard, scope.uiState.currentPageIdx));
                });

                scope.$on('$destroy', function() {
                    Mousetrap.unbind(groupingShortcut);
                    Mousetrap.unbind(nextPageShortcuts);
                    Mousetrap.unbind(previousPageShortcuts);
                });
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.directive('dashboardRightColumnSummary', function(DataikuAPI, $stateParams, $controller, QuickView, ActivityIndicator) {
        return {
            templateUrl: '/templates/dashboards/right-column-summary.html',
            link: function($scope, element, attrs) {

                $controller('_TaggableObjectsMassActions', { $scope: $scope });

                $scope.QuickView = QuickView;

                /* Auto save when summary is modified */
                $scope.$on('objectSummaryEdited', function() {
                    DataikuAPI.dashboards.save($scope.dashboard).success(function(data) {
                        ActivityIndicator.success('Saved');
                    }).error(setErrorInScope.bind($scope));
                });

                $scope.refreshData = function() {
                    DataikuAPI.dashboards.getFullInfo($scope.selection.selectedObject.projectKey, $scope.selection.selectedObject.id).success(function(data) {
                        if (!$scope.selection.selectedObject
                            || $scope.selection.selectedObject.id != data.dashboard.id
                            || $scope.selection.selectedObject.projectKey != data.dashboard.projectKey) {
                            return; //too late!
                        }
                        $scope.dashboardFullInfo = data;
                        $scope.dashboard = data.dashboard;
                    }).error(setErrorInScope.bind($scope));
                };

                $scope.$on('customFieldsSaved', $scope.refreshData);

                $scope.refreshTimeline = function() {
                    DataikuAPI.timelines.getForObject($stateParams.projectKey || $scope.selection.selectedObject.projectKey, 'DASHBOARD', $scope.selection.selectedObject.id)
                        .success(function(data) {
                            $scope.timeline = data;
                        })
                        .error(setErrorInScope.bind($scope));
                };

                $scope.$watch('selection.selectedObject', function(nv) {
                    if (!$scope.selection) {
                        $scope.selection = {};
                    }
                    $scope.dashboardFullInfo = { dashboard: $scope.selection.selectedObject, timeline: {} }; // display temporary (incomplete) data
                });

                $scope.$watch('selection.confirmedItem', function(nv, ov) {
                    if (!nv) {
                        return;
                    }
                    $scope.refreshTimeline();
                    $scope.refreshData();
                });
            }
        }
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/edit.js
     */
    app.directive('dashboardTileParams', function($stateParams, DashboardPageUtils, DashboardUtils, $timeout, TileUtils, ColorUtils, DSSVisualizationThemeUtils, DefaultDSSVisualizationTheme) {
        return {
            restrict: 'EA',
            templateUrl: '/templates/dashboards/tile-params.html',
            scope: {
                tile: '<', // Tile
                insight: '<', // Insight
                dashboard: '<', // Dashboard
                canModerateDashboards: '<', // boolean
                toggleDashboardSettings: '&', // () => void
                theme: '<'
            },
            link: function($scope, $element) {
                $scope.$stateParams = $stateParams;
                $scope.getDefaultTileTitle = getDefaultTileTitle;
                $scope.canEditInsight = function(insight) {
                    return DashboardUtils.canEditInsight(insight, $scope.canModerateDashboards);
                };
                $scope.DashboardUtils = DashboardUtils;
                $scope.TileUtils = TileUtils;
                $scope.themeColors = ColorUtils.getThemeColorsWithBlackWhite($scope.theme);
                $scope.colors = ColorUtils.generateThemePaletteColors(DSSVisualizationThemeUtils.getThemeOrDefault($scope.theme).colors, $scope.themeColors.length > 0);
                $scope.titleDefaultFormatting = getTitleDefaultFormatting();
                $scope.originDashboard = {
                    id: $stateParams.dashboardId,
                    name: $stateParams.dashboardName,
                    pageId: $stateParams.pageId,
                    edit: true
                };

                const DEFAULT_BACKGROUND_COLOR = '#ffffff';

                $scope.handleTileTitleChange = function() {
                    if ($scope.tile.titleOptions.title === '' || $scope.tile.titleOptions.title === getDefaultTileTitle($scope.tile, $scope.insight)) {
                        $scope.updateTileTitleOptions({
                            title: undefined
                        });
                    } else {
                        $scope.updateTileTitleOptions({
                            title: $scope.tile.titleOptions.title
                        });
                    }
                };

                $scope.fillIfEmpty = function() {
                    if (!$scope.tile.titleOptions.title) {
                        $scope.updateTileTitleOptions({
                            title: getDefaultTileTitle($scope.tile, $scope.insight)
                        });
                    }
                };

                $scope.$watch('tile.insightType', function() {
                    $timeout(function() {
                        $element.find('select.display-mode-select').selectpicker('refresh');
                    });
                });

                $scope.$watch('insight', function(nv) {
                    if (!nv || nv.type !== 'chart') {
                        $scope.chartHasCustomColors = null;
                        return;
                    }
                    if (!nv.params.def.colorOptions || !nv.params.def.colorOptions.customColors) {
                        $scope.chartHasCustomColors = false;
                        return;
                    }
                    const customColors = nv.params.def.colorOptions.customColors;
                    $scope.chartHasCustomColors = Object.keys(customColors).length >= 1;
                });

                $scope.$watch('theme', function(nv, ov) {
                    if (nv && (!ov || !_.isEqual(nv.colors, ov.colors))) {
                        $scope.themeColors = ColorUtils.getThemeColorsWithBlackWhite($scope.theme);
                        $scope.colors = ColorUtils.generateThemePaletteColors(nv.colors, $scope.themeColors.length > 0);
                    }
                    if (nv && (!ov || !_.isEqual(nv.tileTitleFormatting, ov.tileTitleFormatting))) {
                        this.titleDefaultFormatting = getTitleDefaultFormatting();
                    }
                });

                $scope.openUploadPictureDialog = TileUtils.openUploadPictureDialog;

                $scope.setTileBorderOptions = function(borderOptions) {
                    $scope.assignToTile('borderOptions', { ...$scope.tile.borderOptions, ...borderOptions })
                        .then(() => resizeTile($scope.tile.$tileId));
                };

                $scope.setTileTextAlignment = function(newAlignment) {
                    $scope.assignToTile('tileParams.textAlign', newAlignment);
                };

                $scope.setTileVerticalTextAlignment = function(newAlignment) {
                    $scope.assignToTile('tileParams.verticalAlign', newAlignment);
                };

                $scope.updateTileTitleDisplayMode = function(showTitle) {
                    $scope.updateTileTitleOptions({ showTitle })
                        .then(() => resizeTile($scope.tile.$tileId));
                };

                $scope.resetBorder = () => {
                    const baseTheme = getBaseTheme();
                    $scope.tile.borderOptions = baseTheme.tileFormatting.borderOptions;
                    $timeout(() => $scope.$apply());
                };

                $scope.shouldShowResetBorder = function() {
                    const baseTheme = getBaseTheme();
                    return (
                        $scope.tile.borderOptions.color !== baseTheme.tileFormatting.borderOptions.color
                        || $scope.tile.borderOptions.radius !== baseTheme.tileFormatting.borderOptions.radius
                        || $scope.tile.borderOptions.size !== baseTheme.tileFormatting.borderOptions.size
                    );
                };

                $scope.resetBackgroundColor = () => {
                    $scope.assignToTile('backgroundColor', $scope.theme ? $scope.theme.backgroundColor : DEFAULT_BACKGROUND_COLOR);
                };

                $scope.shouldShowResetBackgroundColorButton = () => {
                    return $scope.tile && $scope.tile.backgroundColor !== ($scope.theme ? $scope.theme.backgroundColor : DEFAULT_BACKGROUND_COLOR);
                };

                $scope.resetBackgroundOpacity = () => {
                    $scope.assignToTile('backgroundOpacity', getBaseTheme().tileFormatting.backgroundOpacity);
                };

                $scope.shouldShowResetBackgroundOpacity = () => {
                    return $scope.tile && $scope.tile.backgroundOpacity !== getBaseTheme().tileFormatting.backgroundOpacity;
                };

                $scope.updateTileTitleOptions = function(partialTitleOptions) {
                    let partialUpdate = partialTitleOptions;
                    if (!$scope.tile.titleOptions.hasBackground && !partialTitleOptions.hasBackground) {
                        /*
                         * In case hasBackground is not defined or false, we remove from the update
                         * to avoid setting the tile in dirty as titleOptions for tile doesn't have
                         * a hasBackground field.
                         */
                        partialUpdate = _.omit(partialTitleOptions, 'hasBackground');
                    }
                    return $scope.assignToTile('titleOptions', { ...$scope.tile.titleOptions, ...partialUpdate });
                };

                // Easiest way to apply change from Angular component. Inspired by assignToChartDef
                $scope.assignToTile = function(propertyPath, value) {
                    _.set($scope.tile, propertyPath, value);
                    return $timeout(() => $scope.$apply());
                };

                function getDefaultTileTitle(tile, insight) {
                    switch (tile.tileType) {
                        case 'INSIGHT':
                            return (insight || {}).name;
                        case 'TEXT':
                            return 'Text tile';
                        case 'IMAGE':
                            return 'Image tile';
                    }
                }

                function resizeTile(tileId) {
                    const tileElement = DashboardPageUtils.getTileElementFromTileId($(document.querySelector('.dashboard-export-page-wrapper')), tileId);
                    if (tileElement == null || tileElement.length === 0) {
                        return;
                    }
                    const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(tileElement);
                    if (tileWrapper == null || tileWrapper.length === 0) {
                        return;
                    }
                    angular.element(tileWrapper).scope().$broadcast('resize');
                }

                function getBaseTheme() {
                    return ($scope.theme || DefaultDSSVisualizationTheme);
                }

                function getTitleDefaultFormatting() {
                    const baseTheme = getBaseTheme();
                    return { fontColor: baseTheme.tileTitleFormatting.fontColor, fontSize: baseTheme.tileTitleFormatting.fontSize };
                }
            }
        };
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/view.js
     */
    app.directive('dashboardTile', function($stateParams, TileUtils, FiltersPanelPosition, FiltersPanelDirection, $timeout, ColorUtils, DashboardPageUtils, Debounce) {
        return {
            restrict: 'EA',
            templateUrl: '/templates/dashboards/tile.html',
            scope: {
                tile: '<',
                selected: '<', // boolean
                insight: '<',
                editable: '<',
                hook: '<',
                activeFilters: '<',
                filters: '<',
                page: '<',
                dashboardTheme: '<',
                currentGroupTileIdInEditMode: '<', // string | null
                showGrid: '<', // boolean
                tileSpacing: '<', // number
                preselectedTile: '<', // DashboardTile
                autoStackUp: '<', // boolean
                canExportDatasets: '<', // boolean
                insightsMap: '<', // { [insightId: string]: Insight }
                accessMap: '<', // { [insightId: string]: string }

                filtersChange: '&', // ({ $filters }) => void
                filtersParamsChange: '&', // ({ $filtersParams }) => void,
                isDeactivatedChange: '&', // ({ $isDeactivated }) => void
                preselectedTileChange: '&', // ({ $preselectedTile }) => void
                raiseError: '&', // ({ $errorData }) => void,
                isTileLockedChange: '&' // () => void,
            },
            link: function($scope, $element) {
                $scope.$stateParams = $stateParams;
                $scope.FiltersPanelPosition = FiltersPanelPosition;
                $scope.FiltersPanelDirection = FiltersPanelDirection;
                $scope.openUploadPictureDialog = TileUtils.openUploadPictureDialog;

                $scope.getResizeImageClass = function(resizeImageMode) { //convert mode to css class name
                    switch (resizeImageMode) {
                        case 'STRETCH_SIZE': return 'stretch-size';
                        case 'ORIG_SIZE': return 'orig-size';
                        case 'FIT_CROP_SIZE': return 'fit-crop-size';
                        default: return 'fit-size';
                    }

                };

                if (TileUtils.isGroupTile($scope.tile)) {
                    $scope.tileHeight = 0;
                    const { backgroundColor, gridColor } = computeColors();
                    $scope.backgroundColor = backgroundColor;
                    $scope.gridColor = gridColor;
                    $scope.groupTileContainerBackgroundColor = DashboardPageUtils.computeGridContainerBackgroundColor($scope.showGrid, $scope.editable, backgroundColor, gridColor);

                    const updateTileHeight = Debounce().withDelay(50, 50).wrap(() => {
                        $timeout(() => {
                            const boundingClientRect = $element[0].getBoundingClientRect();
                            $scope.tileHeight = boundingClientRect.height;
                            $scope.tileWidth = boundingClientRect.width;
                        });
                    });

                    updateTileHeight();

                    $scope.$watchGroup(
                        ['tile.backgroundColor', 'tile.backgroundOpacity', 'showGrid', 'editable'], () => {
                            const { backgroundColor, gridColor } = computeColors();
                            $scope.backgroundColor = backgroundColor;
                            $scope.gridColor = gridColor;
                            $scope.groupTileContainerBackgroundColor = DashboardPageUtils.computeGridContainerBackgroundColor($scope.showGrid, $scope.editable, backgroundColor, gridColor);
                        }
                    );

                    $scope.$watchCollection('tile.box', () => {
                        updateTileHeight();
                    });

                    $(window).on('resize', updateTileHeight);

                    $scope.$on('$destroy', function() {
                        $(window).off('resize', updateTileHeight);
                    });
                }

                function computeColors() {
                    const backgroundColor = ColorUtils.getHexWithAlpha($scope.tile.backgroundColor, $scope.tile.backgroundOpacity);
                    const gridColor = DashboardPageUtils.getGridColor($scope.tile.backgroundColor, $scope.tile.backgroundOpacity);
                    return { backgroundColor, gridColor };
                }
            }
        };
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/view.js
     */
    app.directive('dashboardZone', function($rootScope, DataikuAPI, $stateParams, $state, $location, DashboardFilters, $timeout) {
        return {
            restrict: 'EA',
            templateUrl: '/templates/dashboards/zone.html',
            replace: true,
            link: function($scope, elements, attrs) {
                $scope.editable = $scope.$eval(attrs.editable);
                $scope.insightsMap = {};
                $scope.insightAccessMap = {};
                DashboardFilters.setInsightsMap($scope.insightsMap);

                if ($scope.dashboard && $scope.dashboard.pages) {
                    $scope.dashboard.pages.forEach(function(page) {
                        delete page.$enriched;
                    });
                }

                $scope.loadPage = function(pageIdx, forceRefresh, onlyAccessMap) {
                    if (!($scope.dashboard.pages[pageIdx] || {}).id) {
                        return;
                    } // Page has not been saved yet
                    if (!$scope.dashboard.pages[pageIdx].$enriched || forceRefresh) {
                        DataikuAPI.dashboards.getEnrichedPage($scope.dashboard.projectKey, $scope.dashboard.id, pageIdx)
                            .success(function(data) {
                                $scope.currentPageIdx = pageIdx;
                                if (!onlyAccessMap) {
                                    angular.extend($scope.insightsMap, data.insightsData);
                                }
                                angular.extend($scope.insightAccessMap, data.insightAccessData);
                                $scope.dashboard.pages[pageIdx].$enriched = true;
                            })
                            .error(setErrorInScope.bind($scope));
                    } else {
                        $scope.currentPageIdx = pageIdx;
                    }
                };

                $scope.hasPagesToDisplay = () => {
                    return $scope.uiState.currentPageIdx != null && !!$scope.dashboard.pages[$scope.uiState.currentPageIdx];
                };

                $scope.$on('dashboardReaderAccessChanged', function() {
                    $scope.loadPage($scope.uiState.currentPageIdx, true, true);
                });

                $scope.$watch('dashboard', function(nv) {
                    if (!nv) {
                        return;
                    }

                    $scope.canEdit = $scope.canEditDashboard(nv);

                    if (!$scope.dashboard.pages.length) {
                        $scope.dashboard.pages.push({});
                    }

                    const firstVisiblePageIdx = $scope.editable ? 0 : nv.pages.findIndex(page => page.show);
                    if ($scope.uiState.currentPageIdx == null && firstVisiblePageIdx >= 0) {
                        if ($stateParams.pageId) {
                            $scope.uiState.currentPageIdx = Math.max(getPageIdxById($stateParams.pageId), firstVisiblePageIdx);
                        } else {
                            $scope.uiState.currentPageIdx = firstVisiblePageIdx;
                        }
                    } else {
                        updateUrl();
                    }
                });

                $scope.$on('dashboardSelectLastTile', function() {
                    if ($scope.uiState.currentPageIdx != null && $scope.dashboard.pages[$scope.uiState.currentPageIdx] != null) {
                        const lastTileIndex = $scope.dashboard.pages[$scope.uiState.currentPageIdx].grid.tiles.length - 1;
                        $scope.uiState.selectedTile = $scope.dashboard.pages[$scope.uiState.currentPageIdx].grid.tiles[lastTileIndex];
                    }
                });

                $scope.$watch('uiState.currentPageIdx', function(nv) {
                    if (nv != null) {
                        $scope.loadPage($scope.uiState.currentPageIdx);
                        updateUrl();
                    }
                });

                $scope.$watch('uiState.pageFilters', function(nv, ov) {
                    if (nv != null && !_.isEqual(nv, ov)) {
                        updateUrl();
                    }
                });

                // On dashboard save
                $scope.$watch('origDashboard', updateUrl);

                function getPageIdxById(pageId) {
                    for (let i = 0; i < $scope.dashboard.pages.length; i++) {
                        if ($scope.dashboard.pages[i].id == pageId) {
                            return i;
                        }
                    }

                    return -1;
                }

                function updateUrl() {

                    const handleUiRouterStateTransition = (statePromise) => statePromise.catch((err) => {
                        /*
                         * A 'transition superseded' error occurs because updateUrl is called twice
                         * before the first ui-router state transition has completed.
                         * In this case, only the last transition is applied.
                         * This error can be safely ignored.
                         */
                        if (err.message !== 'transition superseded') {
                            throw err;
                        }
                    }).finally(() => {
                        $rootScope.$broadcast('dashboardUrlUpdateDone');
                    });

                    const stateOptions = {
                        location: 'replace',
                        inherit: true,
                        notify: false,
                        reload: false
                    };

                    if ($scope.uiState.currentPageIdx == null || $scope.uiState.currentPageIdx === '') {
                        handleUiRouterStateTransition($state.go($state.current, { pageId: '' }, stateOptions));
                        return;
                    }
                    const pageId = ($scope.dashboard.pages[$scope.uiState.currentPageIdx] || {}).id || $scope.uiState.currentPageIdx;
                    const hasUiStateFilters = $scope.uiState.pageFilters || $scope.uiState.pageFilters === '' ;
                    const pageFilters = hasUiStateFilters ? $scope.uiState.pageFilters : $state.params.pageFilters;
                    const filters = pageFilters == null ? $state.params.filters : null;
                    const dashboardId = $scope.dashboard.id;
                    const dashboardName = $scope.dashboard.name;
                    // ensure filters are set in the location as we might load them from it before the async $state.go
                    if (pageFilters != null) {
                        $location.search('pageFilters', pageFilters);
                    } else if (filters != null) {
                        $location.search('filters', filters);
                    } else {
                        $location.search('filters', null);
                        $location.search('pageFilters', null);
                    }
                    handleUiRouterStateTransition(
                        $state.go($state.current,
                            {
                                dashboardId,
                                dashboardName,
                                pageId,
                                pageFilters,
                                filters
                            }, stateOptions)
                    );
                }
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('ArticleInsightHandler', {
        name: 'Article',
        desc: 'Wiki article',
        icon: 'icon-file-text',
        color: 'article',

        getSourceId: function(insight) {
            return insight.params.articleId;
        },
        getSourceType: function(insight) {
            return 'ARTICLE';
        },

        hasEditTab: false,
        defaultTileParams: {},
        defaultTileDimensions: [15, 15]
    });

    app.controller('_articleInsightViewCommon', function($scope, $stateParams, DataikuAPI) {
        $scope.fetchArticle = function(resolve, reject, noSpinner) {
            const p = DataikuAPI.wikis.getArticlePayload($stateParams.projectKey, $scope.insight.params.articleId);
            if (noSpinner) {
                p.noSpinner();
            }
            p.noSpinner()
                .success(function(data) {
                    $scope.article = data;
                    if (typeof(resolve)==='function') {
                        resolve();
                    }
                }).error(function(data, status, headers, config, statusText) {
                    setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                    if (typeof(reject)==='function') {
                        reject(data, status, headers, config, statusText);
                    }
                });
        };
    });

    app.directive('articleInsightTile', function($controller, TileLoadingState) {
        return {
            templateUrl: '/templates/dashboards/insights/article/article_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs) {
                $controller('_articleInsightViewCommon', { $scope: $scope });

                $scope.loading = false;
                $scope.loaded = false;
                $scope.error = null;

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    $scope.fetchArticle(
                        function() {
                            $scope.loading = false;
                            $scope.loaded = true;
                            $scope.error = null;
                            if ($scope.hook && $scope.hook.isErrorMap) {
                                $scope.hook.isErrorMap[$scope.tile.$tileId] = false;
                            }
                            if (typeof(resolve)==='function') {
                                resolve();
                            }
                        }, function(data, status, headers, config, statusText) {
                            $scope.loading = false;
                            $scope.loaded = false;
                            $scope.error = data;
                            if ($scope.hook && $scope.hook.isErrorMap) {
                                $scope.hook.isErrorMap[$scope.tile.$tileId] = true;
                            }
                            $scope.hook.setErrorInDashboardPageScope(data, status, headers, config, statusText);
                            if (typeof(reject)==='function') {
                                reject();
                            }
                        }
                    );
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.directive('articleInsightTileParams', function() {
        return {
            templateUrl: '/templates/dashboards/insights/article/article_tile_params.html',
            scope: {
                tileParams: '='
            }
        };
    });

    app.directive('articleInsightCreateForm', function() {
        return {
            templateUrl: '/templates/dashboards/insights/article/article_create_form.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.hook.defaultName = 'article';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = 'article ' + nv.label;
                });
            }
        };
    });

    app.directive('articleInsightView', function($controller, TaggableObjectsService) {
        return {
            templateUrl: '/templates/dashboards/insights/article/article_view.html',
            scope: {
                insight: '=',
                onError: '&'
            },
            link: function($scope) {
                $controller('_articleInsightViewCommon', { $scope: $scope });
                $scope.fetchArticle(() => {
                    const article = {
                        projectKey: $scope.insight.projectKey,
                        id: $scope.insight.params.articleId,
                        type: 'ARTICLE',
                        versionTag: $scope.insight.params.versionTag
                    };
                    TaggableObjectsService.checkAndUpdateThumbnailData(article, '.wiki-article-content > .markdown-std');
                }, (error) => {
                    if ($scope.onError) {
                        $scope.onError({ error });
                    }
                });
            }
        };
    });

    app.directive('articleInsightEdit', function() {
        return {
            templateUrl: '/templates/dashboards/insights/article/article_edit.html',
            scope: {
                insight: '='
            }
        };
    });

})();

;
(function(){
    'use strict';
    /** @typedef {import('../types').GeneratedSources.DSSVisualizationTheme} DSSVisualizationTheme */

    const app = angular.module('dataiku.dashboards.insights');

    const hasFacetDimension = function(insight) {
        return insight.params && insight.params.def && insight.params.def.facetDimension && insight.params.def.facetDimension.length > 0;
    };

    app.constant('ChartInsightHandler', {
        name: 'Chart',
        desc: 'Visualize data from your source',
        icon: 'icon-dku-nav_dashboard',
        color: 'chart',

        sourceType: 'DATASET',
        getSourceId: function(insight) {
            return insight.params.datasetSmartName;
        },
        hasOptions: (insight) => {
            return insight.params.def; // KPI charts have no axes, legends or tooltips.
        },
        hasEditTab: true,
        goToEditAfterCreation: true,
        getDefaultTileParams: function(insight) {
            return {
                showXAxis: true,
                showYAxis: true,
                showLegend: false,
                showBrush: false,
                inheritLegendPlacement: true,
                showTooltips: true,
                autoPlayAnimation: true,
                legendPlacement: insight.params && insight.params.def.legendPlacement,
                useInsightTheme: false
            };
        },
        getDefaultTileDimensions(insight) {
            if (insight && hasFacetDimension(insight)) {
                return [15, 12];
            }
            return [8, 8];
        }
    });

    app.directive('chartInsightTile', function($controller, ChartRequestComputer, MonoFuture, ChartTypeChangeHandler, ChartFeatures, ChartDataUtils, DashboardUtils, DashboardPageUtils,
        TileLoadingState, TileLoadingBehavior, DashboardFilters, Logger, $stateParams, $q, ChartStoreFactory, DSSVisualizationThemeUtils, translate) {
        return {
            templateUrl: '/templates/dashboards/insights/chart/chart_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '=',
                filters: '=',
                filtersParams: '<',
                dashboardTheme: '<'
            },
            link: function($scope, element) {
                let originChartTheme;
                let originChartDef;

                $scope.DashboardUtils = DashboardUtils;
                $scope.insight.$filteringStatus = {};
                $scope.originDashboard = {
                    id: $stateParams.dashboardId,
                    name: $stateParams.dashboardName,
                    pageId: $stateParams.pageId,
                    edit: true
                };

                $controller('ChartInsightViewCommon', { $scope });

                $scope.initChartCommonScopeConfig($scope);

                $scope.timing = {
                    directiveLink: new Date().getTime()
                };

                $scope.loading = false;
                $scope.loaded = false;
                $scope.error = null;

                function broadcastResize(event) {
                    if (event.detail && (event.detail.skipInDashboards || event.detail.skipInCharts)) {
                        return;
                    }

                    $scope.$broadcast('resize');
                }
                $(window).on('resize', broadcastResize);

                $scope.loadedCallback = function() {
                    if ($scope.hook.loadStates[$scope.tile.$tileId] != TileLoadingState.COMPLETE) {
                        $scope.timing.loadDone = new Date().getTime();
                        Logger.info('Done loading chart tile', $scope.tile.$tileId,
                            'link->load', $scope.timing.loadStart - $scope.timing.directiveLink,
                            'load->req', $scope.timing.requestReady - $scope.timing.loadStart,
                            'req->resp', $scope.timing.responseReceived - $scope.timing.requestReady,
                            'resp->draw', $scope.timing.drawStart - $scope.timing.responseReceived,
                            'draw->done', $scope.timing.loadDone - $scope.timing.responseReceived,
                            'load->done', $scope.timing.loadDone-$scope.timing.loadStart);
                    }
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.COMPLETE;
                };
                const executePivotRequest = MonoFuture($scope).wrap($scope.getExecutePromise);
                let errorRoutine, loadReject; // Stored on load() in order to be used by applySlideFilters afterwards
                let unregisterLoadingStateWatcher;
                let unregisterVisibilityWatcher;
                let unregisterFiltersWatcher;
                let hasCaseInsensitiveColumnNames = false;
                let isChartFixedUp = false;


                $scope.$on('$destroy', function() {
                    $(window).off('resize', broadcastResize);

                    if (unregisterLoadingStateWatcher != null) {
                        unregisterLoadingStateWatcher();
                    }
                    if (unregisterVisibilityWatcher != null) {
                        unregisterVisibilityWatcher();
                    }
                    if (unregisterFiltersWatcher != null) {
                        unregisterFiltersWatcher();
                    }
                });

                $scope.load = function(resolve, reject) {
                    Logger.info('Start loading chart tile', $scope.tile.$tileId);

                    $scope.timing.loadStart = new Date().getTime();
                    loadReject = reject;
                    errorRoutine = DashboardUtils.setError.bind([$scope, reject]);

                    const loadingData = {
                        successRoutine: DashboardUtils.setLoaded.bind([$scope, resolve]),
                        errorRoutine: errorRoutine,
                        unconfiguredRoutine: DashboardUtils.setUnconfigured.bind([$scope, reject]),
                        resolve: resolve,
                        reject: reject
                    };

                    $scope.loading = true;
                    DashboardFilters.hasCaseInsensitiveColumnNames($scope.resolvedDataset.projectKey, $scope.resolvedDataset.datasetName, $stateParams.projectKey)
                        .then(hasCaseInsensitiveColumnNamesStatus => {
                            hasCaseInsensitiveColumnNames = hasCaseInsensitiveColumnNamesStatus;

                            if ($scope.globalFiltersData.usableColumns == null) {
                                return $scope.fetchColumnsSummary($scope.insight.projectKey)
                                    .then(() => {
                                        $scope.globalFiltersData.usableColumns = $scope.usableColumns;
                                    });
                            }
                        }).then(() => {
                            if (!isChartFixedUp) {
                                fixUpChartAndSetSummary($scope.insight, $scope.chart.summary);
                                if ($scope.origInsight) {
                                    fixUpChartAndSetSummary($scope.origInsight, $scope.chart.summary);
                                }
                                isChartFixedUp = true;
                            }
                            return loadChart(loadingData);
                        }).then(() => {
                            if (unregisterFiltersWatcher) {
                                unregisterFiltersWatcher();
                            }
                            unregisterFiltersWatcher = $scope.$watch('filters', (nv, ov) => {
                                // Avoid initialization call as it is already called above
                                if (nv === ov) {
                                    return;
                                }
                                loadChart(loadingData);
                            });
                        }).catch((error) => {
                            handleLoadingFailure( loadingData, error );
                        });

                    return TileLoadingBehavior.DELAYED_COMPLETE;
                };

                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                const additionalRequestCallbacks = {
                    successRoutine: () => {
                        DashboardUtils.setLoaded.call([$scope]);
                        $scope.saveChart();
                    },
                    errorRoutine: DashboardUtils.setError.bind([$scope]),
                    unconfiguredRoutine: DashboardUtils.setLoaded.bind([$scope])
                };

                $scope.rebuildSampling = () => {
                    DashboardUtils.setLoading.call([$scope]);
                    $scope.chart.refreshableSelection._refreshTrigger = new Date().getTime();
                    $scope.fetchColumnsSummary().then(() => {
                        executeFiltersRequest($scope.filters, additionalRequestCallbacks);
                    });
                };

                $scope.forceExecuteChart = (requestOptions) => {
                    const { store, id } = ChartStoreFactory.getOrCreate($scope.chart.def.$chartStoreId);
                    $scope.chart.def.$chartStoreId = id;
                    store.setRequestOptions(requestOptions);
                    DashboardUtils.setLoading.call([$scope]);
                    executeFiltersRequest($scope.filters, additionalRequestCallbacks);
                };

                function fixUpChartAndSetSummary(chartInsight, summary) {
                    chartInsight.params.summary = summary;
                    const theme = getTheme(chartInsight);
                    ChartTypeChangeHandler.fixupChart(chartInsight.params.def, theme);
                    ChartTypeChangeHandler.fixupSpec(chartInsight.params, theme);
                }

                /**
                 * To handle the failure of the loading routine
                 *
                 * @param {Object}          [loadingData]     - Insight loading instructions. Use when loading the insight.
                 * @param {Function}        loadingData.successRoutine
                 * @param {Function}        loadingData.errorRoutine
                 * @param {Function}        loadingData.unconfiguredRoutine
                 * @param {Function}        loadingData.resolve
                 * @param {Function}        loadingData.reject
                 * @param {Object}          [error]           - Caught error
                 * @param {Function}        error.data
                 * @param {Function}        error.status
                 * @param {Function}        error.headers
                 * @param {Function}        error.config
                 * @param {Function}        error.statusText
                 */
                const handleLoadingFailure = ( loadingData, error ) => {
                    // Handle possible client side error to avoid tile being stuck in loading state
                    if (!_.isObject(error) || !error.data || !error.data.detailedMessage) {
                        Logger.error(`Failed to load insight ${$scope.insight.id}`, error);
                        error = {
                            data:{
                                detailedMessage: translate('DASHBOARD.FILTERABLE_INSIGHTS.ERRORS.LOADING_FAILED', 'error occurred while loading insight')
                            }
                        };
                    }

                    const { data, status, headers, config, statusText } = error;
                    if (loadingData) {
                        loadingData.errorRoutine && loadingData.errorRoutine(data, status, headers, config, statusText);
                        if (loadingData.reject && typeof(loadingData.reject) === 'function') {
                            loadingData.reject();
                        }
                    } else {
                        errorRoutine && errorRoutine(data, status, headers, config, statusText);
                        if (loadReject && typeof(loadReject) === 'function') {
                            loadReject();
                        }
                    }
                };

                /**
                 * To handle the success of the loading routine
                 *
                 * @param {Object}          [loadingData]     - Insight loading instructions. Use when loading the insight.
                 * @param {Function}        loadingData.successRoutine
                 * @param {Function}        loadingData.errorRoutine
                 * @param {Function}        loadingData.unconfiguredRoutine
                 * @param {Function}        loadingData.resolve
                 * @param {Function}        loadingData.reject
                 * @param {Function}        [loadedCallback]  - Optional callback to call after loading
                 */
                const handleLoadingSuccess = ( loadingData, loadedCallback ) => {
                    if (loadingData) {
                        loadingData.successRoutine && loadingData.successRoutine();
                        if (loadingData.resolve && typeof(loadingData.resolve) === 'function') {
                            loadingData.resolve();
                        }
                        if(loadedCallback){
                            loadedCallback();
                        }
                    }
                };

                /**
                 * Compute and execute chart's pivot request...
                 *  - With or without additional global filters.
                 *  - With or without insight loading instructions.
                 *
                 * @param {ChartDef.java}   chartDef                - Classical ChartDef
                 * @param {Object}          [loadingData]           - Insight loading instructions. Use when loading the insight.
                 * @param {Function}        loadingData.successRoutine
                 * @param {Function}        loadingData.errorRoutine
                 * @param {Function}        loadingData.unconfiguredRoutine
                 * @param {Function}        loadingData.resolve
                 * @param {Function}        loadingData.reject
                 * @param {Boolean}         hasGlobalFilters                 - Are global filters applied on top of chart's ones
                 */
                const computeAndExecutePivotRequest = (chartDef, loadingData, hasGlobalFilters) => {
                    let request;

                    try {
                        $scope.chartSpecific = {
                            ...$scope.chartSpecific,
                            datasetProjectKey: $scope.getDataSpec().datasetProjectKey,
                            datasetName: $scope.getDataSpec().datasetName,
                            context: $scope.getCurrentChartsContext()
                        };

                        request = ChartRequestComputer.compute(chartDef, element.width(), element.height(), $scope.chartSpecific);
                        const validity = ChartTypeChangeHandler.getValidity({ def: chartDef });
                        if (loadingData && validity.valid === false) {
                            loadingData.errorRoutine({ detailedMessage: 'Invalid configuration, ' + validity.message });
                        }
                    } catch (e) {
                        if (hasGlobalFilters) {
                            const deferred = $q.defer();
                            deferred.reject({ data: e });
                            return deferred.promise;
                        }
                    }

                    if (loadingData && !request) {
                        loadingData.unconfiguredRoutine();
                    } else {
                        $scope.timing.requestReady = new Date().getTime();
                        return executePivotRequest(request).update(data => {
                            $scope.request = request;
                            $scope.response = data;
                        }).success(data => {
                            Logger.info('Got pivot response for chart tile', $scope.tile.$tileId);
                            $scope.timing.responseReceived = new Date().getTime();
                            $scope.request = request;
                            $scope.response = data;
                            $scope.validity = { valid: true };

                            handleLoadingSuccess(loadingData);

                            if (hasGlobalFilters) {
                                $scope.tile.$allFilteredOut = false;
                            }
                            const sampleUIData = ChartDataUtils.getSampleMetadataAndSummaryMessage(data.result.pivotResponse, 'Click to edit the insight');
                            $scope.tile.$sampleMetadata = sampleUIData.sampleMetadata;
                            $scope.tile.$samplingSummaryMessage = sampleUIData.summaryMessage;
                            $scope.tile.$clickableSamplingSummaryMessage = sampleUIData.clickableSummaryMessage;
                            $scope.tile.$recordsMetadata = ChartFeatures.getRecordsMetadata(chartDef, sampleUIData.sampleMetadata, data.result.pivotResponse.beforeFilterRecords, data.result.pivotResponse.afterFilterRecords);
                            $scope.tile.$display0Warning = ChartFeatures.shouldDisplay0Warning($scope.insight.params.def.type, data.result.pivotResponse);
                        }).error(function(data, status, headers, config, statusText) {
                            $scope.tile.$allFilteredOut = false;
                            $scope.tile.$display0Warning = false;
                            if (data.code === 'FILTERED_OUT') {
                                // We still need to display the insight to say it's been filtered out with a proper empty state.
                                handleLoadingSuccess(loadingData, $scope.loadedCallback());
                                $scope.tile.$allFilteredOut = true;
                            } else if (data.code === 'CANNOT_DISPLAY_WITH_EMPTY_AXES') {
                                handleLoadingSuccess(loadingData, $scope.loadedCallback());
                                $scope.tile.$display0Warning = true;
                            } else if (ChartTypeChangeHandler.hasRequestResponseWarning($scope.chart.def, data)) {
                                $scope.validity = ChartTypeChangeHandler.getRequestResponseWarning($scope.chart.def, data);
                                handleLoadingSuccess(loadingData, $scope.loadedCallback());
                            } else {
                                handleLoadingFailure( loadingData, { data, status, headers, config, statusText } );
                            }
                        });
                    }
                };

                function watchLoadingStateChange() {
                    if (unregisterLoadingStateWatcher) {
                        return;
                    }
                    unregisterLoadingStateWatcher = $scope.$watch('loaded', (nv) => {
                        if (nv === true) {
                            unregisterLoadingStateWatcher(); // remove watch
                            loadChart();
                        }
                    });
                }

                function watchVisibilityChange() {
                    if (unregisterVisibilityWatcher) {
                        return;
                    }
                    unregisterVisibilityWatcher = $scope.$watch('hook.visibleStates[tile.$tileId]', function(nv) {
                        if (nv === true) {
                            unregisterVisibilityWatcher(); // remove watch
                            unregisterVisibilityWatcher = null;
                            loadChart();
                        }
                    });
                }

                /**
                 *
                 * @param {Object}      [loadingData]           - Insight loading instructions. Use when loading the insight.
                 * @param {Function}    loadingData.successRoutine
                 * @param {Function}    loadingData.errorRoutine
                 * @param {Function}    loadingData.unconfiguredRoutine
                 * @param {Function}    loadingData.resolve
                 * @param {Function}    loadingData.reject
                 * @param {boolean}     [forceFilterableTilesRefresh=false]
                 */
                function loadChart(loadingData) {
                    /*
                     * If the chart is not yet loaded and no loading instructions are provided,
                     * wait for it to complete before applying latest filter changes (which happened in the meantime)
                     */
                    if (!loadingData && !$scope.loaded && !unregisterLoadingStateWatcher) {
                        watchLoadingStateChange();
                        return Promise.resolve();
                    }

                    /* If the tile is not visible nor adjacent, wait for it to become visible before loading the chart */
                    if (!DashboardPageUtils.isTileVisibleOrAdjacent($scope.tile.$tileId, $scope.hook)) {
                        watchVisibilityChange();
                        return Promise.resolve();
                    }
                    // It actually trigger the chart loading even if there is no filter to apply
                    return DashboardFilters.applyFilters($scope.filters, $scope.filtersParams, $scope.globalFiltersData, loadingData, hasCaseInsensitiveColumnNames)
                        .catch((error) => {
                            handleLoadingFailure( loadingData, error );
                        });
                }

                function executeFiltersRequest(filters, loadingData) {
                    const chartDef = { ...$scope.insight.params.def };
                    if (filters && filters.length) {
                        chartDef.filters = [
                            ...chartDef.filters,
                            ...angular.copy(filters)
                        ];
                    }
                    const { store, id } = ChartStoreFactory.getOrCreate($scope.chart.def.$chartStoreId);
                    $scope.chart.def.$chartStoreId = id;
                    store.setAppliedDashboardFilters(angular.copy(filters));
                    return computeAndExecutePivotRequest(chartDef, loadingData, true);
                }

                $scope.globalFiltersData = {
                    insightId: $scope.insight.id,
                    usableColumns: $scope.usableColumns,
                    applyFilters: executeFiltersRequest,
                    parseResponse: (data) => data.data.result.pivotResponse,
                    filteringStatus: angular.copy($scope.insight.$filteringStatus),
                    sourceDataset: $scope.insight.params.datasetSmartName,
                    sampleSettings: $scope.insight.params.refreshableSelection,
                    pageId: $stateParams.pageId
                };

                // globalFiltersData.filteringStatus is mutated by applyFilters, and we need to update insight.$filteringStatus reference to trigger Angular's digest cycle.
                $scope.$watch('globalFiltersData.filteringStatus', filteringStatus => {
                    $scope.insight.$filteringStatus = angular.copy(filteringStatus);
                }, true);

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }

                // As soon as the chart is available, set its originLegendPlacement property (done only once, needed in chartInsightTileParams).
                const deregisterLegendPlacementWatcher = $scope.$watch('insight.params.def.legendPlacement', () => {
                    $scope.chart.def.originLegendPlacement = $scope.insight.params.def.legendPlacement;
                    deregisterLegendPlacementWatcher();
                });

                const unregisterChartThemeWatcher = $scope.$watch('insight.params.theme', () => {
                    originChartTheme = angular.copy($scope.insight.params.theme);
                    unregisterChartThemeWatcher();
                });

                /*
                 * Before calling the fixUp function we need to make sure summary is defined to have access to the
                 * usableColumns.
                 */
                const unregisterChartSummaryWatcher = $scope.$watch('chart.summary', (nv) => {
                    if (!nv) {
                        return;
                    }
                    unregisterChartSummaryWatcher();

                    /*
                     * We need to wait for the chart summary to be available before fixing up the chart def,
                     * because some fixup operations need the summary (e.g. setting a default color palette
                     * based on the columns types).
                     */
                    const unregisterChartDefWatcher = $scope.$watch('insight.params.def', () => {
                        /*
                         * The goal here is to cache a version of the chart def as it is displayed in the
                         * chart view page. To do so, we need to fix it up.
                         */
                        const originChart = angular.copy($scope.insight.params);
                        ChartTypeChangeHandler.fixupChart(originChart.def, originChart.theme);
                        ChartTypeChangeHandler.fixupSpec(originChart, originChart.theme);
                        originChartDef = originChart.def;

                        unregisterChartDefWatcher();
                    });
                });



                $scope.$watch('dashboardTheme', function(newDashboardTheme) {
                    if ($scope.tile.tileParams.useInsightTheme || !newDashboardTheme) {
                        return;
                    }
                    /*
                     * We create a deep copy to make sure that if the dashboard theme is mutated
                     * (which ideally should not happen), then the chart theme won't be affected.
                     */
                    applyTheme(angular.copy(newDashboardTheme));
                });


                $scope.$watch('tile.tileParams', function(nv, ov) {
                    if (!nv) {
                        return;
                    }
                    const isFirstChange = nv === ov;

                    $scope.noXAxis = !nv.showXAxis;
                    $scope.noYAxis = !nv.showYAxis;
                    $scope.noTooltips = !nv.showTooltips;
                    $scope.autoPlayAnimation = nv.autoPlayAnimation;

                    if ($scope.chart && (isFirstChange || nv.useInsightTheme !== ov.useInsightTheme)) {

                        if (!nv.useInsightTheme) {
                            const newTheme = getTheme();
                            applyTheme(newTheme, { redraw: false });
                        } else {
                            resetChartDef();
                            resetChartTheme();
                            element[0].classList.toggle('chart-insight-no-theme', !!$scope.chart.theme);
                        }
                    }

                    if ($scope.chart) {
                        applyTileChartOptions();
                    }

                    $scope.$broadcast('redraw');
                }, true);

                /**
                 * Apply the given theme to the chart insight and optionally redraw it.
                 * @param {DSSVisualizationTheme | undefined} theme
                 * @param {{ redraw: boolean }} options
                 * @returns {void}
                 */
                function applyTheme(theme, options = { redraw: true }) {
                    if (!theme) {
                        /*
                         * theme can be undefined if:
                         * - both chart and dashboard theme is undefined
                         * - the tile ignores the dashboard theme and the insight theme is undefined
                         * - the tile uses the dashboard theme and the dashboard theme is undefined
                         *
                         * In these cases we need to make sure the chart insight is just displayed
                         * as it is outside of a dashboard.
                         */
                        resetChartDef();
                        // Re-apply the tile chart options on top of the resetted chart def.
                        applyTileChartOptions();
                        element.addClass('chart-insight-no-theme');
                    } else {
                        element.removeClass('chart-insight-no-theme');
                        DSSVisualizationThemeUtils.applyToChart({ chart:$scope.chart.def, theme, formerTheme: $scope.chart.theme });
                    }
                    $scope.chart.theme = theme;
                    if (options.redraw) {
                        $scope.$broadcast('redraw');
                    }
                }

                function applyTileChartOptions() {
                    if (!$scope.tile.tileParams.showLegend) {
                        $scope.chart.def.legendPlacement = 'SIDEBAR';
                    } else {
                        if ($scope.tile.tileParams.inheritLegendPlacement) {
                            $scope.chart.def.legendPlacement = $scope.chart.def.originLegendPlacement;
                        } else {
                            $scope.chart.def.legendPlacement = $scope.tile.tileParams.legendPlacement ? $scope.tile.tileParams.legendPlacement : $scope.chart.def.originLegendPlacement;
                        }
                    }
                    $scope.chart.def.showXAxis = $scope.tile.tileParams.showXAxis;
                    $scope.chart.def.linesZoomOptions.displayBrush = $scope.tile.tileParams.showBrush;
                }

                function resetChartDef() {
                    replaceDeep($scope.chart.def, angular.copy(originChartDef)); // we need to copy originChartDef because some operations mutate $scope.chart.def
                }

                function resetChartTheme() {
                    if (originChartTheme) {
                        replaceDeep($scope.chart.theme, angular.copy(originChartTheme));
                    } else {
                        $scope.chart.theme = undefined;
                    }
                }

                /**
                 * Returns the theme that needs to be used by the insight.
                 * @returns {DSSVisualizationTheme | undefined}
                 */
                function getTheme() {
                    if (!$scope.tile.tileParams.useInsightTheme && $scope.dashboardTheme) {
                        return angular.copy($scope.dashboardTheme);
                    }

                    return originChartTheme;
                }
            }
        };
    });

    app.directive('chartInsightTileParams', function(ChartFeatures){
        return {
            templateUrl: '/templates/dashboards/insights/chart/chart_tile_params.html',
            scope: {
                tileParams: '=',
                insight: '='
            },
            link: function($scope){
                $scope.$watch('insight.params.def.type', (type) => {
                    $scope.noAxis = !ChartFeatures.canHideAxes(type);
                    $scope.noTooltips = !ChartFeatures.canShowTooltips(type);
                });

                $scope.$watch('insight.params.def', (def) =>
                    $scope.canDisplayBrush = ChartFeatures.canDisplayBrush(def));

                $scope.$watch('insight.params.def.animationDimension', (animationDimension) =>
                    $scope.noAnimation = (animationDimension || []).length === 0);

                $scope.$watchGroup(['insight.params.def.type', 'insight.params.def.originLegendPlacement'], ([type, legendPlacement]) =>
                    $scope.noLegend = !ChartFeatures.canDisplayLegend(type));
            }
        };
    });

    app.directive('chartInsightCreateForm', function(DataikuAPI, ChartTypeChangeHandler, DatasetChartsUtils, DSSVisualizationThemeUtils) {
        return {
            templateUrl: '/templates/dashboards/insights/chart/chart_create_form.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.hook.sourceTypes = ['DATASET'];
                $scope.hook.beforeSave = function(resolve, reject) {
                    DataikuAPI.explores.get($scope.insight.projectKey, $scope.insight.params.datasetSmartName)
                        .success(function(data) {
                            $scope.insight.params.refreshableSelection = DatasetChartsUtils.makeSelectionFromScript(data.script);
                            $scope.insight.params.def = ChartTypeChangeHandler.defaultNewChart(DSSVisualizationThemeUtils.getThemeOrDefault($scope.insight.params.theme));
                            $scope.insight.params.theme = DSSVisualizationThemeUtils.getThemeOrDefault($scope.insight.params.theme);
                            $scope.insight.params.engineType = 'LINO';

                            resolve();
                        })
                        .error(function(data, status, headers, config, statusText){
                            reject(arguments);
                        });
                };

                $scope.hook.defaultName = 'Chart';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = 'Chart on ' + nv.label;
                });
            }
        };
    });

    app.controller('ChartInsightViewCommon', function($scope, $rootScope, DataikuAPI, $stateParams, $controller, ActiveProjectKey, DashboardPageUtils) {
        $controller('ShakerChartsCommonController', { $scope: $scope });

        $scope.isProjectAnalystRW = function() {
            return true;
        };

        $scope.resolvedDataset = resolveDatasetFullName($scope.insight.params.datasetSmartName, ActiveProjectKey.get());

        // Needed by chart directives
        $scope.chart = {
            def: $scope.insight.params.def,
            theme: $scope.insight.params.theme,
            refreshableSelection: $scope.insight.params.refreshableSelection,
            engineType: $scope.insight.params.engineType
        };

        $scope.validity = {}; // Needed by chart redraw

        if ($scope.tile) {
            $scope.chart.def.showLegend = $scope.tile.tileParams.showLegend;
            $scope.chart.def.showXAxis = $scope.tile.tileParams.showXAxis;
        }

        $scope.saveChart = function() {
            DataikuAPI.dashboards.insights.save($scope.insight)
                .error(setErrorInScope.bind($scope))
                .success(function() {});
        };

        $scope.getDataSpec = function() {
            return {
                datasetProjectKey: $scope.resolvedDataset.projectKey,
                datasetName: $scope.resolvedDataset.datasetName,
                copyScriptFromExplore: true,
                copySelectionFromScript: false,
                sampleSettings : $scope.insight.params.refreshableSelection,
                engineType : $scope.insight.params.engineType
            };
        };

        $scope.getExecutePromise = function(request, saveShaker=false, noSpinner=true, requiredSampleId=undefined, dataSpec=$scope.getDataSpec()) {
            if (request) {
                if (requiredSampleId === undefined) {
                    requiredSampleId = $scope.chart.summary == null ? null : $scope.chart.summary.requiredSampleId;
                }
                const projectKey = $scope.insight.projectKey || ActiveProjectKey.get();
                let promise = DataikuAPI.shakers.charts.getPivotResponse(
                    projectKey,
                    dataSpec,
                    request,
                    requiredSampleId
                );
                if (noSpinner === true) {
                    promise = promise.noSpinner();
                }
                return promise;
            }
        };

        $scope.fetchColumnsSummary = function(projectKey){
            // get columns summary
            if (!projectKey) {
                projectKey = ActiveProjectKey.get();
            }
            return DashboardPageUtils.getColumnSummary(projectKey, $scope.getDataSpec())
                .then(data => {
                    $scope.chart.summary = data;
                    $scope.insight.params.summary = data;
                    if ($scope.origInsight) {
                        $scope.origInsight.params.summary = data;
                    }
                    $scope.makeUsableColumns(data);
                });
        };

        function fetchSummaryAndExecute(saveChart = false) {
            $scope.fetchColumnsSummary().then(function() {
                saveChart && $scope.saveChart();
                $scope.forceExecuteChartOrWait();
            }).catch(({ data, status, headers, config, statusText, xhrStatus }) => {
                setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
            });
        }

        $rootScope.$on('chartSamplingChanged', function(event, opts) {
            if (angular.equals($scope.chart, opts.chart)) {
                $scope.chart.summary = null;
                fetchSummaryAndExecute(true);
            }
        });

        // chartHandler options
        $scope.noThumbnail = true;
        $scope.addCustomMeasuresToScopeAndCache($scope.insight.params.customMeasures);
        $scope.addBinnedDimensionToScopeAndCache($scope.insight.params.reusableDimensions);
    });

    app.directive('chartInsightView', function($controller, translate) {
        return {
            templateUrl: '/templates/dashboards/insights/chart/chart_view.html',
            scope: {
                insight: '='
            },
            link: function($scope) {
                $controller('ChartInsightViewCommon', { $scope: $scope });
                $controller('ChartsCommonController', { $scope: $scope });

                $scope.translate = translate;
                $scope.noClickableAxisLabels = true;
                $scope.readOnly = true;
                $scope.bigChart = true;
                $scope.bigChartDisabled = true;
                $scope.showLegends = true;
                $scope.saveChart = function() {};

                $scope.fetchColumnsSummary().then(function(){
                    $scope.forceExecuteChartOrWait();
                }).catch(({ data, status, headers, config, statusText, xhrStatus }) => {
                    setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
                });
            }
        };
    });

    app.directive('chartInsightEdit', function($controller, $stateParams, DataikuAPI, $rootScope, ChartTypeChangeHandler, DSSVisualizationThemeUtils, StateUtils, ActivityIndicator) {
        return {
            templateUrl: '/templates/dashboards/insights/chart/chart_edit.html',
            scope: {
                insight: '='
            },
            link: function($scope, element, attrs){
                $controller('ChartInsightViewCommon', { $scope: $scope });
                $controller('ChartsCommonController', { $scope: $scope });

                $scope.currentInsight = $scope.insight;
                $scope.appConfig = $rootScope.appConfig;
                $scope.uiDisplayState = {};

                $scope.bigChart = false;

                $scope.getDefaultNewChart = function() {
                    const defaultTheme = DSSVisualizationThemeUtils.getThemeOrDefault($rootScope.appConfig.selectedDSSVisualizationTheme);
                    return {
                        theme: defaultTheme,
                        def: ChartTypeChangeHandler.defaultNewChart(defaultTheme)
                    };
                };

                function fetchSummaryAndExecute(){
                    $scope.fetchColumnsSummary().then(function(){
                        $scope.forceExecuteChartOrWait();
                    }).catch(({ data, status, headers, config, statusText, xhrStatus }) => {
                        setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
                    });
                }

                $scope.$watch('chart.engineType', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.insight.params.engineType = nv;
                });

                DataikuAPI.datasets.get($scope.resolvedDataset.projectKey, $scope.resolvedDataset.datasetName, $stateParams.projectKey)
                    .success(function(data) {
                        $scope.dataset = data;
                        fetchSummaryAndExecute();
                    }).error(setErrorInScope.bind($scope));

                $scope.$watch('chart.def.name', function(nv, ov) {
                    if ($scope.insight.name == 'Chart on ' + $scope.insight.params.datasetSmartName
                        || $scope.insight.name == ov + ' on ' + $scope.insight.params.datasetSmartName) {
                        $scope.insight.name = nv + ' on ' + $scope.insight.params.datasetSmartName;
                    }
                });

                $scope.overrideFormattingWithTheme = function(theme) {
                    const currentChartCopy = angular.copy($scope.chart);

                    DSSVisualizationThemeUtils.applyToChart({ chart: $scope.chart.def, theme, formerTheme: $scope.chart.theme });
                    $scope.chart.theme = theme;
                    $scope.insight.params.theme = theme;

                    DSSVisualizationThemeUtils.showThemeAppliedSnackbar(
                        $scope.chart,
                        currentChartCopy,
                        () => {
                            $scope.insight.params.theme = currentChartCopy.theme;
                        }
                    );
                };

                $scope.$on('$destroy', () => DSSVisualizationThemeUtils.hideThemeAppliedSnackbar());
            }
        };
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('INSIGHT_TYPES', [
        'dataset_table',
        'chart',
        'discussions',
        'jupyter',
        'metrics',
        'saved-model_report',
        'data-quality',
        'model-evaluation_report',
        'managed-folder_content',
        'web_app',
        'report',
        'project_activity',
        'scenario',
        'runnable-button',
        'static_file',
        'article',
        'eda'
    ]);

    app.constant('FILTERABLE_INSIGHT_TYPES', [
        'dataset_table',
        'chart'
    ]);

    const reportInsightsDownloader = $('<iframe>').attr('id', 'reports-downloader');

    app.controller('_InsightsCommonController', function($scope, $controller, $rootScope, TopNav, DataikuAPI, ActivityIndicator, CreateModalFromTemplate, Dialogs, $stateParams, $state, RMARKDOWN_ALL_OUTPUT_FORMATS, WT1) {

        function makeInsightListed(insight, noNotification) {
            return DataikuAPI.dashboards.insights.makeListed(insight.projectKey, [insight.id], !insight.listed)
                .success(function(data) {
                    if (!noNotification) {
                        ActivityIndicator.success('Saved!');
                    }
                    $scope.$broadcast('objectTimelineChanged');
                    insight.listed = !insight.listed;
                    if ($scope.origInsight) {
                        $scope.origInsight.listed = insight.listed;
                    }
                }).error(setErrorInScope.bind($scope));
        }

        $scope.makeInsightListed = makeInsightListed;

        $scope.toggleInsightListed = function(insight, closeToggle) {
            if (!insight.listed && (insight.accessState && insight.accessState !== 'READER')) {
                CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                    newScope.initForInsights([insight], true);
                }).then(function() {
                    makeInsightListed(insight, true).success($scope.list);
                });
            } else {
                if (closeToggle) {
                    $('.tooltip').remove();
                }
                makeInsightListed(insight, true);
            }
        };

        $scope.openInsightAccessModal = function(insight) {
            CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                newScope.initForInsights([insight], false);
            }).then($scope.list);
        };

        $scope.canEditInsight = function(insight) {
            return insight && $scope.canWriteDashboards() && ($scope.canModerateDashboards() || insight.owner == $scope.appConfig.login);
        };

        $scope.saveCustomFields = function(newCustomFields) {
            WT1.event('custom-fields-save', { objectType: 'INSIGHT' });
            const oldCustomFields = angular.copy($scope.insight.customFields);
            $scope.insight.customFields = newCustomFields;
            return DataikuAPI.dashboards.insights.save($scope.insight)
                .success(function() {
                    $scope.origInsight = angular.copy($scope.insight);
                    $scope.$broadcast('objectTimelineChanged');
                    $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), $scope.insight.customFields);
                })
                .error(function(a, b, c) {
                    $scope.insight.customFields = oldCustomFields;
                    setErrorInScope.bind($scope)(a, b, c);
                });
        };

        $scope.editCustomFields = function() {
            if (!$scope.insight) {
                return;
            }
            const modalScope = angular.extend($scope, { objectType: 'INSIGHT', objectName: $scope.insight.name, objectCustomFields: $scope.insight.customFields });
            CreateModalFromTemplate('/templates/taggable-objects/custom-fields-edit-modal.html', modalScope).then(function(customFields) {
                $scope.saveCustomFields(customFields);
            });
        };

        $scope.hasEditTab = function(insight) {
            /*
             * TODO @insights use handler!
             *  Exclusion list
             */
            return insight && [
                'metrics',
                'data-quality',
                'discussions',
                'web_app',
                'managed-folder_content',
                'saved-model_report',
                'model-evaluation_report',
                'scenario_run_button',
                'scenario_last_runs',
                'project_activity',
                'static_file',
                'article'
            ].indexOf(insight.type) == -1;
        };

        $scope.downloadRMarkdownReportInsight = function(insight) {
            const newScope = $scope.$new();
            newScope.insight = insight;
            const t = insight.params.loadLast ? 0 : insight.params.exportTimestamp;
            DataikuAPI.reports.snapshots.get(insight.projectKey, insight.params.reportSmartId, t).success(function(snapshot) {
                CreateModalFromTemplate('/templates/code-reports/download-report-modal.html', newScope, null, function(modalScope) {
                    modalScope.formats = RMARKDOWN_ALL_OUTPUT_FORMATS.filter(f => (snapshot.availableFormats||[]).includes(f.name));
                    modalScope.allFormats = RMARKDOWN_ALL_OUTPUT_FORMATS.length == modalScope.formats.length;

                    modalScope.options = {};
                    if (modalScope.formats.find(f => f.name == 'PDF_DOCUMENT')) {
                        modalScope.options.format = 'PDF_DOCUMENT';
                    } else if (modalScope.formats.length) {
                        modalScope.options.format = modalScope.formats[0].name;
                    }

                    modalScope.downloadReport = function() {
                        modalScope.dismiss(); // dismiss modal 1
                        const url = '/dip/api/reports/snapshots/download?' + $.param({
                            projectKey: insight.projectKey,
                            id: insight.id,
                            format: modalScope.options.format
                        });

                        reportInsightsDownloader.attr('src', url);
                        $('body').append(reportInsightsDownloader);
                    };
                });
            }).error(setErrorInScope.bind($scope));
        };

        $scope.copy = function(insight, callBackFunc) {
            CreateModalFromTemplate('/templates/dashboards/insights/copy-insight-modal.html', $scope, 'CopyInsightModalController', function(newScope) {
                newScope.init(insight);
            })
                .then(function() {
                    if (typeof(callBackFunc) === 'function') {
                        callBackFunc();
                    }
                });
        };

        $scope.mutiPin = function(insight, callbackFunc) {
            CreateModalFromTemplate('/templates/dashboards/insights/multi-pin-insight-modal.html', $scope, 'MultiPinInsightModalController', function(newScope) {
                newScope.multiPinCallback = callbackFunc;
                newScope.init(insight);
            });
        };

        function switchViewMode(viewMode) {
            if (viewMode !== 'view' && viewMode !== 'edit') {
                return;
            }
            $state.go(`^.${ viewMode }`);
        };

        $scope.switchToView = () => switchViewMode('view');
        $scope.switchToEdit = () => switchViewMode('edit');
    });

    // Resolves with true if any reader authorization was modified, false otherwise
    app.controller('InsightAccessWarningModalController', function($scope, $controller, DataikuAPI, $rootScope, SmartId) {
        $scope.SmartId = SmartId;

        $scope.initForInsights = function(insights, listing) {
            $scope.insightOrDashboard = 'insight';
            $scope.listing = listing;
            $scope.projectKey = insights[0].projectKey;
            DataikuAPI.dashboards.insights.getMissingReaderAuthorizations($scope.projectKey, insights.map(function(insight){
                return insight.id;
            })).success(function(data) {
                $scope.readerAuthorizations = data;
                $scope.selectedReaderAuthorizations = angular.copy($scope.readerAuthorizations);
            }).error(setErrorInScope.bind($scope));
        };

        $scope.initForDashboards = function(dashboards, listing) {
            $scope.insightOrDashboard = 'dashboard';
            $scope.listing = listing;
            $scope.projectKey = dashboards[0].projectKey;
            DataikuAPI.dashboards.getMissingReaderAuthorizations($scope.projectKey, dashboards.map(function(dashboard){
                return dashboard.id;
            })).success(function(data) {
                $scope.readerAuthorizations = data;
                $scope.selectedReaderAuthorizations = angular.copy($scope.readerAuthorizations);
            }).error(setErrorInScope.bind($scope));
        };

        $scope.initForDashboardsWithAuths = function(dashboards, listing, requiredAuths) {
            $scope.insightOrDashboard = 'dashboard';
            $scope.listing = listing;
            $scope.projectKey = dashboards[0].projectKey;
            $scope.readerAuthorizations = requiredAuths;
            $scope.selectedReaderAuthorizations = angular.copy(requiredAuths);
        };

        $scope.initForInsightsWithAuths = function(insights, listing, requiredAuths) {
            $scope.insightOrDashboard = 'dashboard';
            $scope.listing = listing;
            $scope.projectKey = insights[0].projectKey;
            $scope.readerAuthorizations = requiredAuths;
            $scope.selectedReaderAuthorizations = angular.copy(requiredAuths);
        };

        $scope.add = function() {
            if ($rootScope.projectSummary.canManageDashboardAuthorizations && $scope.selectedReaderAuthorizations.length) {
                DataikuAPI.projects.addReaderAuthorizations($scope.projectKey, $scope.selectedReaderAuthorizations).success(function(data) {
                    $scope.resolveModal(true);
                }).error(setErrorInScope.bind($scope));
            } else {
                $scope.resolveModal(false);
            }
        };
    });

    /** List of dashboards */
    app.controller('InsightsListController', function($scope, $controller, $stateParams, DataikuAPI, CreateModalFromTemplate, Dialogs,$state,$q, TopNav, Fn, $filter, ActivityIndicator, DashboardUtils, StateUtils) {
        $controller('_TaggableObjectsListPageCommon', { $scope: $scope });
        $controller('_InsightsCommonController', { $scope:$scope });

        $scope.listHeads = DataikuAPI.dashboards.insights.listHeads;
        $scope.getInsightPromotionTitle = DashboardUtils.getTooltipPromotionTitleFunction('insight');

        $scope.sortBy = [
            { value: 'name', label: 'Name' },
            { value: '-lastModifiedOn', label: 'Last modified' },
            { value: 'type', label: 'Insight type' }
        ];
        $scope.selection = $.extend({
            filterQuery: {
                userQuery: '',
                tags: [],
                type: [],
                listed : '',
                interest: {
                    starred: ''
                }
            },
            filterParams: {
                userQueryTargets: ['name','tags','type'],
                propertyRules: { tag: 'tags' }
            },
            inclusiveFilter: {

                owner: []
            },
            customFilterWatch: 'selection.inclusiveFilter',
            customFilter: function(objList) {
                return objList.filter(function(obj) {
                    if ($scope.selection.inclusiveFilter.owner.length > 0) {
                        if ($scope.selection.inclusiveFilter.owner.indexOf(obj.owner) > -1) {
                            return true;
                        }
                        return false;
                    } else {
                        return true;
                    }
                });
            },
            orderQuery: '-lastModifiedOn',
            orderReversed: false
        }, $scope.selection || {});


        $scope.setOwnerFilter = function(owner) {
            if (!owner) {
                $scope.selection.inclusiveFilter.owner = [];
                return;
            }

            const arr = $scope.selection.inclusiveFilter.owner;
            const index = arr.indexOf(owner);

            if (index > -1) {
                arr.splice(index, 1);
            } else {
                arr.push(owner);
            }
        };

        $scope.setTypeFilter = function(type) {
            if (!type) {
                $scope.selection.filterQuery.type = [];
                return;
            }
            const arr = $scope.selection.filterQuery.type;
            const index = arr.indexOf(type);

            if (index > -1) {
                arr.splice(index, 1);
            } else {
                arr.push(type);
            }
        };

        $scope.setListedFilterQuery = function(value) {
            $scope.selection.filterQuery.listed = value ? 'true' : '';
        };

        $scope.sortCookieKey = 'insights';
        $scope.maxItems = 20;

        TopNav.setLocation(TopNav.TOP_DASHBOARD, 'insights', TopNav.TABS_NONE, null);
        TopNav.setNoItem();
        $scope.list() ;

        $scope.$watch('selection.selectedObject',function(nv) {
            if (!nv) {
                return;
            }

            DataikuAPI.dashboards.insights.getFullInfo($stateParams.projectKey, nv.id).success(function(data) {
                $scope.insight = data.insight;
            }).error(setErrorInScope.bind($scope));
        });

        $scope.newInsight = function() {
            CreateModalFromTemplate('/templates/dashboards/insights/new-insight-modal.html', $scope, 'NewInsightModalController')
                .then(function(ret) {
                    if(ret.callback) {
                        ret.callback();
                    }
                    StateUtils.go.insight(ret.insight.id, $stateParams.projectKey, { name: ret.insight.name, tab: 'edit' });
                });
        };

        $scope.isAllListed = function(items) {
            if (!items) {
                return true;
            }
            return items.map(Fn.prop('listed')).reduce(function(a,b){
                return a&&b;
            },true);
        };

        $scope.canMassMakeListed = true;
        $scope.massMakeListed = function(items, listed) {
            const apiCall = function() {
                DataikuAPI.dashboards.insights.makeListed(items[0].projectKey, ids, listed)
                    .success(function(data) {
                        ActivityIndicator.success('Saved!');
                        $scope.list();
                    }).error(setErrorInScope.bind($scope));
            };

            if (!(items && items.length > 0)) {
                return;
            }
            var ids = [];
            const unreadableItems = [];
            items.forEach(function(item) {
                if (item.listed != listed) {
                    ids.push(item.id);
                }
                if (item.accessState !== 'READER') {
                    unreadableItems.push(item);
                }
            });
            if (listed && unreadableItems.length > 0) {
                CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                    newScope.initForInsights(unreadableItems, true);
                }).then(apiCall);
            } else {
                apiCall();
            }
        };

        $scope.owners = [];
        $scope.types = [];
        $scope.ownersMap = {};
        $scope.list = function() {
            $scope.listHeads($stateParams.projectKey, $scope.tagFilter).success(function(data) {
                $scope.filteredOut = data.filteredOut;
                $scope.listItems = data.items;
                $scope.restoreOriginalSelection();

                const ownersMap = {};
                data.items.forEach(function(insight) {
                    ownersMap[insight.owner] = {
                        login : insight.owner,
                        displayName : insight.ownerDisplayName
                    };
                    if ($scope.types.indexOf(DashboardUtils.getInsightTypeGroup(insight.type)) == -1) {
                        $scope.types.push(DashboardUtils.getInsightTypeGroup(insight.type));
                    }
                });
                $scope.owners.length = 0;
                for (const login in ownersMap) {
                    $scope.owners.push(ownersMap[login]);
                }
                $scope.owners.sort(function(a, b){
                    if(a.displayName < b.displayName) {
                        return -1;
                    }
                    if(a.displayName > b.displayName) {
                        return 1;
                    }
                    return 0;
                });
                $scope.types.sort();
            }).error(setErrorInScope.bind($scope));

        };
        $scope.list();
    });

    app.directive('insightTypeSelector', function(DashboardUtils){
        return {
            template:
            '<ul>' +
            '    <li ng-repeat="insightType in insightTypesList" ng-click="toggleType(insightType)" ng-class="{\'selected\' : isTypeSelected(insightType)}">' +
            '       <i class="insight-icon {{insightType | insightTypeToIcon}} universe-background {{insightType | insightTypeToColor}}"></i> <span class="type-name">{{insightType | insightTypeToDisplayableName}}</span>' +
            '   </li>' +
            '</ul>',
            scope: {
                insightTypesList: '=insightTypeSelector',
                selectedTypesList: '='
            },
            link: function(scope, element){
                scope.DashboardUtils = DashboardUtils;
                scope.toggleType = function(type) {
                    if (type) {
                        const index = scope.selectedTypesList.indexOf(type);
                        index > -1 ? scope.selectedTypesList.splice(index, 1) : scope.selectedTypesList.push(type);
                    }
                };

                scope.isTypeSelected = function(type) {
                    return scope.selectedTypesList.indexOf(type) > -1;
                };
            }
        };
    });


    app.controller('InsightCoreController', function($scope, $rootScope, $stateParams, DataikuAPI, TopNav, $filter, $state, $controller, StateUtils, DashboardUtils, CreateModalFromTemplate, DSSVisualizationThemeUtils, ChartTypeChangeHandler) {

        $controller('_InsightsCommonController', { $scope:$scope });

        $scope.DashboardUtils = DashboardUtils;

        DataikuAPI.dashboards.insights.getFullInfo($stateParams.projectKey, $stateParams.insightId).success(function(data) {
            $scope.pristineInsight = angular.copy(data.insight);
            $scope.dashboardsPinnedOn = data.dashboardsPinnedOn;
            $scope.dashboardsLinkedFrom = data.dashboardsLinkedFrom;
            $scope.timeline = data.timeline;
            $scope.interest = data.interest;

            computeNbDashboardsPinnedOn();
            if (data.insight.type == 'chart') {
                const insightTheme = data.insight.params.theme;
                /*
                 * If the summary is not defined the fixupChart will not have access to the usableColumns and may end
                 * up doing bad stuff like replacing a meaning palette with the default one.
                 */
                const unwatch = $scope.$watch('chart.summary', nv => {
                    if (!nv) {
                        return;
                    }
                    ChartTypeChangeHandler.fixupChart(data.insight.params.def, insightTheme);
                    ChartTypeChangeHandler.fixupSpec(data.insight.params, insightTheme);
                    unwatch();
                });

            }
            $scope.insight = data.insight;
            $scope.origInsight = angular.copy($scope.insight);

            TopNav.setItem(TopNav.ITEM_INSIGHT, $stateParams.insightId, data.insight);
            TopNav.setPageTitle(data.insight.name + ' - Insight');

            $scope.$watch('insight.name', function(nv) {
                if (!nv) {
                    return;
                }
                $state.go($state.current, { insightName: $filter('slugify')(nv) }, { location: 'replace', inherit:true, notify:false, reload:false });
            });
        }).error(setErrorInScope.bind($scope));


        $scope.isDirty = function() {
            if (!$scope.canEditInsight($scope.insight) || !$scope.origInsight) {
                return false;
            }

            if ($scope.insight.type == 'chart') {
                delete $scope.insight.params.def.thumbnailData;
                delete $scope.origInsight.params.def.thumbnailData;
            }
            return !angular.equals($scope.insight, $scope.origInsight);
        };

        $scope.revertChanges = function() {
            $scope.insight = angular.copy($scope.origInsight);
        };

        $scope.saveInsight = function(commitMessage) {
            return DataikuAPI.dashboards.insights.save($scope.insight, commitMessage)
                .success(function(data) {
                    $scope.origInsight = angular.copy($scope.insight);
                }).error(setErrorInScope.bind($scope));
        };

        const computeNbDashboardsPinnedOn = function() {
            $scope.nbDashboardsPinnedOn = $scope.dashboardsPinnedOn.length;
            $scope.nbListedDashboardsPinnedOn = $scope.dashboardsPinnedOn.filter(function(d){
                return d.listed;
            }).length;
            $scope.nbDashboardsLinkedFrom = $scope.dashboardsLinkedFrom.length;
            $scope.nbListedDashboardsLinkedFrom = $scope.dashboardsLinkedFrom.filter(function(d){
                return d.listed;
            }).length;
        };

        $scope.originDashboardHref = $stateParams.originDashboard ? StateUtils.href.dashboard($stateParams.originDashboard.id, $stateParams.projectKey, { pageId: $stateParams.originDashboard.pageId, name: $stateParams.originDashboard.name, tab: $stateParams.originDashboard.edit ? 'edit' : 'view' }) : null;

        $scope.getSourceObjectHref = function(insight) {
            return StateUtils.href.dssObject(DashboardUtils.getInsightSourceType(insight), DashboardUtils.getInsightHandler(insight.type).getSourceId(insight), insight.projectKey);
        };

        $scope.toggleInsightListed = function() {
            if (!$scope.insight.listed) {
                DataikuAPI.dashboards.insights.getMissingReaderAuthorizations($scope.insight.projectKey, [$scope.insight.id]).success(function(data) {
                    if (data.length) {
                        CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                            newScope.initForInsightsWithAuths([$scope.insight], true, data);
                        }).then(function() {
                            $scope.makeInsightListed($scope.insight);
                        });
                    } else {
                        $scope.makeInsightListed($scope.insight);
                    }
                });
            } else {
                $scope.makeInsightListed($scope.insight);
            }
        };
    });

    app.controller('InsightDetailsController', function($scope, $stateParams, $state, FutureProgressModal, DatasetUtils, StateUtils, Dialogs, ActiveProjectKey) {
        $scope.isOnInsightObjectPage = function() {
            return $state.includes('projects.project.dashboards.insights.insight');
        };
    });

    app.directive('insightRightColumnSummary', function($controller, $rootScope, $state, DataikuAPI, QuickView, ActiveProjectKey, ActivityIndicator){
        return {
            templateUrl :'/templates/dashboards/insights/right-column-summary.html',
            link : function($scope, element, attrs) {

                $controller('_TaggableObjectsMassActions', { $scope: $scope });

                $scope.QuickView = QuickView;

                /* Auto save when summary is modified */
                $scope.$on('objectSummaryEdited', function(){
                    DataikuAPI.dashboards.insights.save($scope.insight).success(function(data) {
                        ActivityIndicator.success('Saved');
                    }).error(setErrorInScope.bind($scope));
                });

                $scope.refreshData = function() {
                    DataikuAPI.dashboards.insights.getFullInfo($scope.selection.selectedObject.projectKey, $scope.selection.selectedObject.id).success(function(data) {
                        if (!$scope.selection.selectedObject
                            || $scope.selection.selectedObject.id != data.insight.id
                            || $scope.selection.selectedObject.projectKey != data.insight.projectKey) {
                            return; //too late!
                        }
                        $scope.insightFullInfo = data;
                        $scope.insightFullInfo.isProjectAnalystRO = $scope.isProjectAnalystRO ? $scope.isProjectAnalystRO() : false;
                        $scope.insight = data.insight;
                        computeNbDashboardsPinnedOn();

                        if (data.insight.type === 'dataset_table') {
                            $scope.isLocalDataset = !data.insight.params.datasetSmartName.includes('.');
                            $scope.datasetProjectKey = $scope.isLocalDataset ? data.insight.projectKey : data.insight.params.datasetSmartName.split('.')[0];
                            $scope.datasetName = $scope.isLocalDataset ? data.insight.params.datasetSmartName : data.insight.params.datasetSmartName.split('.')[1];
                            $scope.referencedDatasetAuthorizations = data.referencedObjectAuthorizations;
                            $scope.canExportDatasetsData = $scope.projectSummary != null && $scope.projectSummary.canExportDatasetsData;
                        }

                    }).error(setErrorInScope.bind($scope));
                };

                $scope.$on('customFieldsSaved', $scope.refreshData);

                $scope.exportDatasetOptions = function() {
                    let filters = [];
                    let globalSearchQuery = null;
                    /*
                     * We don't allowFiltering if none has been specified at the front end,
                     * or we would get the dataset filtering at the backend based on the explore settings, which would be strange here
                     */
                    if ($state.includes('projects.project.dashboards.insights.insight') && $scope.explorationFiltersAndSearchQuery) {
                        filters = $scope.explorationFiltersAndSearchQuery.explorationFilters;
                        globalSearchQuery = $scope.explorationFiltersAndSearchQuery.globalSearchQuery;
                    }
                    return {
                        downloadOnly: true,
                        pluginsExport: true,
                        allowFiltering: (filters && filters.length > 0) || globalSearchQuery,
                        explorationFiltersAndSearchQuery: $scope.explorationFiltersAndSearchQuery
                    };
                };

                $scope.explorationFiltersAndSearchQuery = null; // Null means retrieve the saved ones
                $rootScope.$on('$destroy', $rootScope.$on('shakerTableChangedGlobal', (evt, shaker) => {
                    $scope.explorationFiltersAndSearchQuery = {
                        explorationFilters: shaker.explorationFilters,
                        globalSearchQuery: shaker.globalSearchQuery
                    };
                }));

                $scope.refreshTimeline = function(){
                    DataikuAPI.timelines.getForObject(ActiveProjectKey.get(), 'INSIGHT', $scope.selection.selectedObject.id)
                        .success(function(data){
                            $scope.timeline = data;
                        })
                        .error(setErrorInScope.bind($scope));
                };


                $scope.$watch('selection.selectedObject',function(nv) {
                    if (!$scope.selection) {
                        $scope.selection = {};
                    }
                    $scope.insightFullInfo = { insight: $scope.selection.selectedObject, timeline: {} }; // display temporary (incomplete) data
                });

                $scope.$watch('selection.confirmedItem', function(nv, ov) {
                    if (!nv) {
                        return;
                    }
                    $scope.refreshTimeline();
                    $scope.refreshData();
                });

                var computeNbDashboardsPinnedOn = function() {
                    const ii = $scope.insightFullInfo;
                    $scope.nbDashboardsPinnedOn = ii.dashboardsPinnedOn.length;
                    $scope.nbListedDashboardsPinnedOn = ii.dashboardsPinnedOn.filter(function(d){
                        return d.listed;
                    }).length;
                    $scope.nbDashboardsLinkedFrom = ii.dashboardsLinkedFrom.length;
                    $scope.nbListedDashboardsLinkedFrom = ii.dashboardsLinkedFrom.filter(function(d){
                        return d.listed;
                    }).length;
                };
            }
        };
    });

    app.controller('InsightPageRightColumnActions', async function($controller, $scope, $rootScope, $stateParams, GlobalProjectActions, DataikuAPI, ActiveProjectKey) {

        $controller('_TaggableObjectPageRightColumnActions', { $scope: $scope });

        const insight = (await DataikuAPI.dashboards.insights.get(ActiveProjectKey.get(), $stateParams.insightId)).data;

        insight.nodeType = 'INSIGHT';
        insight.interest = {};

        $scope.selection = {
            selectedObject : insight,
            confirmedItem : insight
        };

        $scope.renameObjectAndSave = function(newName) {
            $scope.insight.name = newName;
            return DataikuAPI.dashboards.insights.save($scope.insight);
        };

        function updateListed() {
            DataikuAPI.dashboards.insights.get(ActiveProjectKey.get(), $stateParams.insightId).success(function(data) {
                $scope.selection.selectedObject.listed = data.listed;
            }).error(setErrorInScope.bind($scope));
        }

        function updateUserInterests() {
            DataikuAPI.interests.getForObject($rootScope.appConfig.login, 'INSIGHT', ActiveProjectKey.get(), $stateParams.insightId).success(function(data) {

                $scope.selection.selectedObject.interest = data;
                $scope.insightFullInfo.interest = data;

            }).error(setErrorInScope.bind($scope));
        }

        updateUserInterests();
        const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
        const listedListener = $scope.$on('objectTimelineChanged', updateListed);
        $scope.$on('$destroy', interestsListener);
        $scope.$on('$destroy', listedListener);
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('DataQualityInsightHandler', {
        name: 'Data Quality',
        desc: 'Dataset or project Data Quality status information',
        icon: 'dku-icon-shield-check',
        color: 'data-quality-insight',

        getSourceId: function(insight) {
            return insight.params.objectSmartId;
        },
        getSourceType: function(insight) {
            return insight.params.objectType;
        },
        hasEditTab: false,
        defaultTileParams: {
        },
        defaultTileDimensions: [9, 6]
    });

    app.service('DataQualityInsightUtils', function() {
        return {
            computeName(objectType, objectSmartId, statusType) {
                const prefix = {
                    'CURRENT_STATUS': 'Data quality status on',
                    'BREAKDOWN': 'Data quality breakdown on'
                }[statusType];
                const suffix = objectType === 'PROJECT'
                    ? 'project'
                    : (objectSmartId || 'object');

                return `${prefix} ${suffix}`;
            }
        };
    });

    app.directive('dqInsightCreateForm', function($stateParams, DataQualityInsightUtils) { // dq-insight-create-form
        return {
            templateUrl: '/templates/dashboards/insights/data-quality/data-quality_create_form.html',
            scope: true,
            link: function($scope) {
                $scope.hook.sourceTypes = ['DATASET', 'PROJECT'];
                // disabled & set the forced value for the project in the 'pick existing' tab.
                $scope.$watchGroup(['filter.sourceType', 'filter.sourceId'], ([selectedSourceType]) => {
                    // we need to also watch sourceId because the default value cannot be applied before the object-picker has selected the list of projects.
                    if(selectedSourceType === 'PROJECT') {
                        $scope.hook.disableObjectSelector = true;
                        $scope.filter.sourceId = $stateParams.projectKey;
                    } else {
                        $scope.hook.disableObjectSelector = false;
                    }
                });

                // default initial values
                $scope.insight.params.objectType = 'DATASET';
                $scope.insight.params.statusType = 'CURRENT_STATUS';

                // clear selection on object type change, or re-select last selected of this type if any
                const lastSelectedByType = {};
                $scope.$watch('insight.params.objectType', (nv, ov) => {
                    if (ov) {
                        lastSelectedByType[ov] = $scope.insight.params.objectSmartId;
                    }
                    if (nv) {
                        const defaultValue = nv === 'PROJECT' ? $stateParams.projectKey : null;
                        $scope.insight.params.objectSmartId = lastSelectedByType[nv] || defaultValue;
                    }
                });

                // auto-update default name
                $scope.$watchGroup([
                    'insight.params.objectType',
                    'insight.params.objectSmartId',
                    'insight.params.statusType'
                ], ([objectType, objectSmartId, statusType]) => {
                    $scope.hook.defaultName = DataQualityInsightUtils.computeName(objectType, objectSmartId, statusType);
                });
            }
        };
    });


    function loadDataQualityInsightData(DataikuAPI, contextProjectKey, insightParams) {
        switch(insightParams.objectType) {
            case 'PROJECT': {
                switch(insightParams.statusType) {
                    case 'BREAKDOWN': return DataikuAPI.dataQuality.getProjectCurrentStatusBreakdown(contextProjectKey);
                    case 'CURRENT_STATUS': return DataikuAPI.dataQuality.getProjectCurrentDailyStatus(contextProjectKey);
                }
                break;
            }
            case 'DATASET': {
                const { projectKey, datasetName } = resolveDatasetFullName(insightParams.objectSmartId, contextProjectKey);
                switch(insightParams.statusType) {
                    case 'BREAKDOWN': return DataikuAPI.dataQuality.getDatasetCurrentStatusBreakdown(contextProjectKey, projectKey, datasetName);
                    case 'CURRENT_STATUS': return DataikuAPI.dataQuality.getDatasetCurrentDailyStatus(contextProjectKey, projectKey, datasetName);
                }
                break;
            }
        }
    }

    // converts the backend object repr to the frontend monitoring scope one.
    function fixupBreakdownScope(data) {
        if(data.monitoringScope) {
            data.fixedMonitoringScope = {
                'OBJECT': 'dataset',
                'PARTITION': 'partition',
                'RULE': 'rule'
            }[data.monitoringScope];
        }
    }

    // dq-insight-tile
    app.directive('dqInsightTile', function(DataikuAPI, DashboardUtils, TileLoadingState, $stateParams) {
        return {
            templateUrl: '/templates/dashboards/insights/data-quality/data-quality_tile.html',
            scope: {
                insight: '<',
                tile: '<',
                hook: '='
            },
            link: function($scope){
                $scope.loading = false;
                $scope.loaded = false;

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    loadDataQualityInsightData(DataikuAPI, $stateParams.projectKey, $scope.insight.params)
                        .then(({ data }) => {
                            fixupBreakdownScope(data);
                            $scope.insightData = data;
                        }).then(
                            DashboardUtils.setLoaded.bind([$scope, resolve]),
                            DashboardUtils.setError.bind([$scope, reject])
                        );
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    // dq-insight-view
    app.directive('dqInsightView', function(DataikuAPI, $stateParams) {
        return {
            templateUrl: '/templates/dashboards/insights/data-quality/data-quality_view.html',
            scope: {
                insight: '<',
                preLoadedInsightData: '<?'
            },
            link: function($scope){
                if($scope.preLoadedInsightData) {
                    $scope.insightData = $scope.preLoadedInsightData;
                } else {
                    loadDataQualityInsightData(DataikuAPI, $stateParams.projectKey, $scope.insight.params)
                        .success((data) => {
                            fixupBreakdownScope(data);
                            $scope.insightData = data;
                        })
                        .error(setErrorInScope.bind($scope));
                }
            }
        };
    });

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('DatasetTableInsightHandler', {
        name: 'Dataset',
        nameForTileParams: 'Dataset table',
        desc: 'Partial or whole datatable',
        icon: 'icon-table',
        color: 'dataset',

        sourceType: 'DATASET',
        getSourceId: function(insight) {
            return insight.params.datasetSmartName;
        },
        hasEditTab: true,
        defaultTileParams: {
            viewKind: 'EXPLORE',
            showName: true,
            showDescription: true,
            showCustomFields: true,
            showMeaning: false,
            showProgressBar: false
        },
        defaultTileSize: [],
        defaultTileDimensions: [18, 9]
    });

    app.controller('DatasetTableViewCommon', function($scope, $stateParams, DashboardUtils) {
        $scope.resolvedDataset = resolveDatasetFullName($scope.insight.params.datasetSmartName, $stateParams.projectKey);
        $scope.loading = false;

        $scope.setLoading = (loading) => {
            $scope.loading = loading;
        };
    });

    app.directive('datasetTableInsightTile', function($controller, DashboardUtils, TileLoadingState) {
        return {
            templateUrl: '/templates/dashboards/insights/dataset_table/dataset_table_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '=',
                filters: '=',
                filtersParams: '<'
            },
            link: function($scope) {
                $controller('DatasetTableViewCommon', { $scope: $scope });

                $scope.ngShowLoaded = true;
                $scope.loaded = false;
                $scope.error = null;

                $scope.load = function(resolve, reject) {
                    $scope.refreshNoSpinner = true;
                    $scope.setDashboardTileLoaded = DashboardUtils.setLoaded.bind([$scope, resolve]);
                    $scope.setDashboardTileError = DashboardUtils.setError.bind([$scope, reject]);
                    $scope.loading = true;
                };

                $scope.reload = function(resolve, reject) {
                    $scope.load(resolve, reject);
                    if ($scope.applySlideFilters) {
                        $scope.applySlideFilters();
                    }
                };

                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.reload;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.directive('datasetTableInsightView', function($controller) {
        return {
            templateUrl: '/templates/dashboards/insights/dataset_table/dataset_table_view.html',
            scope: {
                insight: '=',
                tileParams: '='
            },
            link: function($scope) {
                $controller('DatasetTableViewCommon', { $scope: $scope });
            }
        };
    });

    app.directive('datasetTableInsightEdit', function($controller) {
        return {
            templateUrl: '/templates/dashboards/insights/dataset_table/dataset_table_edit.html',
            scope: true,
            link: function($scope) {
                $controller('DatasetTableViewCommon', { $scope: $scope });
            },
            controller: function($scope, translate) {
                $scope.translate = translate; // Add 'translate' to the scope as this directive might be called from Angular as a downgraded component
                $scope.sanitize = sanitize;
            }
        };
    });

    app.directive('shakerExploreInsight', function($filter, $timeout, $q, Assert, DataikuAPI, WT1, SmartId, DashboardFilters, $controller, DashboardPageUtils, Logger, translate) {
        return {
            scope: true,
            controller: function($scope, $stateParams, DatasetChartsUtils) {
                $controller('ShakerChartsCommonController', { $scope: $scope });

                const resolvedDataset = SmartId.resolve($scope.insight.params.datasetSmartName);
                $scope.inInsight = true;

                /* ********************* Callbacks for shakerExploreBase ******************* */

                $scope.shakerHooks.saveForAuto = function() {
                    return $q((resolve, reject) => {
                        try {
                            $scope.insight.params.shakerScript = $scope.getShakerData();
                            if ($scope.hook && $scope.hook.onSave) {
                                $scope.hook.onSave($scope.insight.params);
                            }

                            resolve();
                        } catch (error) {
                            reject(error);
                        }
                    });
                };

                $scope.shakerHooks.setColumnMeaning = function(column, newMeaning) {
                };

                $scope.shakerHooks.getSetColumnStorageTypeImpact = function(column, newType) {
                    return null;
                };
                $scope.shakerHooks.setColumnStorageType = function(column, newType, actionId) {
                };

                $scope.shakerHooks.updateColumnDetails = function(column) {
                };

                $scope.setSpinnerPosition = function() { };

                /*
                 * Keep the existing signature but ignore filterRequest because
                 * we override it with the filters from the filter tile
                 */
                // eslint-disable-next-line  no-unused-vars
                $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                    $scope.setLoading(true);
                    // shakerAndDashboardExplorationFilters is not null in case we have applied a globalFilter (Dashboard filters) on it
                    const tableChunkFilterRequest = $scope.insight.$shakerAndDashboardExplorationFilters == null ? filterRequest : { 'elements': $scope.buildFilterRequest($scope.insight.$shakerAndDashboardExplorationFilters) };
                    return DataikuAPI.shakers.getTableChunk(
                        $stateParams.projectKey,
                        $scope.inputDatasetProjectKey,
                        $scope.inputDatasetName,
                        $scope.shakerHooks.shakerForQuery(),
                        $scope.requestedSampleId,
                        firstRow,
                        nbRows,
                        firstCol,
                        nbCols,
                        tableChunkFilterRequest
                    ).noSpinner()
                        .error((data, status, headers, config, statusText, xhrStatus) => {
                            const error = { data, status, headers, config, statusText, xhrStatus };
                            $scope.setLoading(false);
                            handleLoadingFailure(error);
                        }).then((data) => {
                            $scope.setLoading(false);
                            return data;
                        });
                };

                const baseExploreGetRefreshTablePromise = $scope.shakerHooks.getRefreshTablePromise;
                $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                    const tableChunkFilterRequest = $scope.insight.$shakerAndDashboardExplorationFilters == null ? filterRequest : { 'elements': $scope.buildFilterRequest($scope.insight.$shakerAndDashboardExplorationFilters) };
                    return baseExploreGetRefreshTablePromise(filtersOnly, tableChunkFilterRequest);
                };

                /* ********************* Main ******************* */

                // Set base context and call baseInit
                Assert.inScope($scope, 'shakerHooks');

                let unregisterLoadingStateWatcher;
                let unregisterVisibilityWatcher;
                let unregisterFiltersWatcher;

                $scope.$on('$destroy', function() {
                    if (unregisterLoadingStateWatcher != null) {
                        unregisterLoadingStateWatcher();
                    }
                    if (unregisterVisibilityWatcher != null) {
                        unregisterVisibilityWatcher();
                    }
                    if (unregisterFiltersWatcher != null) {
                        unregisterFiltersWatcher();
                    }
                });

                $scope.table = null;
                $scope.scriptId = '__pristine__';
                $scope.shakerWithSteps = false;
                $scope.shakerWritable = false;
                $scope.shakerReadOnlyActions = true;
                $scope.inputDatasetProjectKey = resolvedDataset.projectKey;
                $scope.inputDatasetName = resolvedDataset.id;
                $scope.inputDatasetSmartName = $scope.insight.params.datasetSmartName;

                WT1.event('shaker-explore-open');

                $scope.shaker = $scope.insight.params.shakerScript;
                $scope.shaker.origin = 'DATASET_EXPLORE';
                if ($scope.origInsight) {
                    $scope.origInsight.params.shakerScript.origin = 'DATASET_EXPLORE';
                }

                $scope.shakerState.writeAccess = true;
                if ($scope.insight.params.shakerState) {
                    Object.entries($scope.insight.params.shakerState).forEach(([key, value]) => $scope.shakerState[key] = value);
                }

                if ($scope.tile) {
                    $scope.shaker.$headerOptions = $scope.tile.tileParams;
                } else {
                    $scope.shaker.$headerOptions = {
                        showName: true,
                        showMeaning: true,
                        showDescription: true,
                        showCustomFields: true,
                        showProgressBar: true
                    };
                }

                $scope.fixupShaker();
                if ($scope.origInsight) {
                    $scope.fixupShaker($scope.origInsight.params.shakerScript);
                }
                $scope.requestedSampleId = null;

                $scope.insight.$filteringStatus = {};
                const initialFilters = angular.copy($scope.shaker.explorationFilters);

                $scope.getDataSpec = () => {
                    return {
                        datasetProjectKey: $scope.inputDatasetProjectKey,
                        datasetName: $scope.inputDatasetName,
                        copyScriptFromExplore: true,
                        copySelectionFromScript: true,
                        sampleSettings: $scope.shaker.explorationSampling,
                        engineType: 'LINO'
                    };
                };

                $scope.fetchColumnsSummary = function(projectKey) {
                    if (!projectKey) {
                        projectKey = $scope.inputDatasetProjectKey;
                    }

                    return DashboardPageUtils.getColumnSummary(projectKey, $scope.getDataSpec())
                        .then(data => $scope.makeUsableColumns(data));
                };

                function watchLoadingState() {
                    if (unregisterLoadingStateWatcher) {
                        return;
                    }
                    unregisterLoadingStateWatcher = $scope.$watch('loaded', (nv) => {
                        if (nv === true) {
                            unregisterLoadingStateWatcher(); // remove watch
                            unregisterLoadingStateWatcher = null;
                            $scope.applySlideFilters();
                        }
                    });
                }

                function watchVisibilityChange() {
                    if (unregisterVisibilityWatcher) {
                        return;
                    }
                    unregisterVisibilityWatcher = $scope.$watch('hook.visibleStates[tile.$tileId]', function(nv) {
                        if (nv === true) {
                            unregisterVisibilityWatcher(); // remove watch
                            unregisterVisibilityWatcher = null;
                            $scope.applySlideFilters();
                        }
                    });
                }

                const handleLoadingFailure = (error) => {
                    if (!_.isObject(error) || !error.data || !error.data.detailedMessage) {
                        Logger.error(`Failed to load dataset  insight ${$scope.insight.id ? $scope.insight.id : ''}`, error);
                        error = {
                            data:{
                                detailedMessage: translate('DASHBOARD.FILTERABLE_INSIGHTS.ERRORS.LOADING_FAILED', 'error occurred while loading insight')
                            }
                        };
                        // In dataset tile context, handle possible client side error to avoid being stuck in loading state
                        if ($scope.setDashboardTileError) {
                            $scope.setDashboardTileError(error.data);
                        }
                        return;
                    }

                    const { data, status, headers, config, statusText, xhrStatus } = error;
                    setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
                    if ($scope.setDashboardTileError) {
                        $scope.setDashboardTileError(data, status, headers, config, statusText);
                    }
                    if ($scope.onRefreshFutureFailed) {
                        $scope.onRefreshFutureFailed(data, status, headers);
                    }
                };


                $scope.applySlideFilters = () => {
                    if (!$scope.loading && !$scope.loaded) {
                        watchLoadingState();
                        return;
                    }
                    if (!DashboardPageUtils.isTileVisibleOrAdjacent($scope.tile.$tileId, $scope.hook)) {
                        watchVisibilityChange();
                        return;
                    }
                    if ($scope.usableColumns == null) {
                        $scope.fetchColumnsSummary($stateParams.projectKey).then(() => {
                            return $scope.applySlideFilters();
                        }).catch(error => {
                            handleLoadingFailure(error);
                        });
                        return;
                    }
                    if (!$scope.deRegisterTableWatch) {
                        registerTableWatch();
                    }
                    $scope.globalFiltersData.usableColumns = $scope.usableColumns;

                    DashboardFilters.hasCaseInsensitiveColumnNames($scope.resolvedDataset.projectKey, $scope.resolvedDataset.datasetName, $stateParams.projectKey).then(hasCaseInsensitiveColumnNames => {
                        return DashboardFilters.applyFilters($scope.filters, $scope.filtersParams, $scope.globalFiltersData, null, hasCaseInsensitiveColumnNames);
                    }).catch((error) => {
                        handleLoadingFailure(error);
                    });
                };

                unregisterFiltersWatcher = $scope.$watch('filters', () => {
                    $scope.applySlideFilters();
                });

                function executeFiltersRequest(filters) {
                    const dashboardExplorationFilters = DashboardFilters.getDashboardExplorationFilters(filters);
                    $scope.insight.$shakerAndDashboardExplorationFilters = [...initialFilters, ...dashboardExplorationFilters];
                    return $scope.refreshTable(false);
                }

                $scope.globalFiltersData = {
                    insightId: $scope.insight.id,
                    applyFilters: executeFiltersRequest,
                    filteringStatus: angular.copy($scope.insight.$filteringStatus),
                    sourceDataset: $scope.insight.params.datasetSmartName,
                    sampleSettings: DatasetChartsUtils.makeSelectionFromScript($scope.insight.params.shakerScript),
                    pageId: $stateParams.pageId
                };

                // globalFiltersData.filteringStatus is mutated by applyFilters, and we need to update insight.$filteringStatus reference to trigger Angular's digest cycle.
                $scope.$watch('globalFiltersData.filteringStatus', filteringStatus => {
                    $scope.insight.$filteringStatus = angular.copy(filteringStatus);
                }, true);

                const registerTableWatch = () => {
                    $scope.deRegisterTableWatch = $scope.$watch('table', (table) => {
                        if (table) {
                            DashboardFilters.onApplyFiltersSuccess($scope.globalFiltersData, $scope.filtersParams, table);
                            if ($scope.tile) {
                                $scope.tile.$allFilteredOut = table.totalKeptRows === 0;
                            }
                        }
                    }, true);
                };

                const globalFilters = DashboardFilters.getActiveGlobalFilters($stateParams.pageId);
                if (globalFilters) {
                    $scope.applySlideFilters();
                } else {
                    $scope.refreshTable(false);
                }

                $scope.$watch('shakerState.activeView', (nv, ov) => {
                    if (nv === 'table' && ov && nv !== ov) {
                        $scope.refreshTable(false);
                    }
                });

                $scope.$watch('table', (newValue) => {
                    if (!newValue) {
                        return;
                    }
                    setSamplingDetails(newValue);
                });

                $scope.getSamplingSummaryMessage = (clickActionMessage) => {
                    let samplingMessage = '<i class="dku-icon-warning-fill-16 tooltip-icon"></i> Data sampling can be misrepresentative</br>';
                    if (clickActionMessage) {
                        samplingMessage += `${clickActionMessage}</br>`;
                    }
                    samplingMessage += `${$scope.getSampleDesc()}`;
                    return samplingMessage;
                };

                const setSamplingDetails = (table) => {
                    if (!$scope.tile) {
                        return;
                    }
                    const sampleMetadata = table.sampleMetadata;
                    if (sampleMetadata && !sampleMetadata.sampleIsWholeDataset) {
                        $scope.tile.$sampleMetadata = sampleMetadata;
                        $scope.tile.$samplingSummaryMessage = $scope.getSamplingSummaryMessage();
                        $scope.tile.$clickableSamplingSummaryMessage = $scope.getSamplingSummaryMessage('Click to edit insight');
                    }
                };
            }
        };
    });

    app.directive('datasetTableInsightTileParams', function(SmartId, DataikuAPI) {
        return {
            templateUrl: '/templates/dashboards/insights/dataset_table/dataset_table_tile_params.html',
            scope: {
                tile: '=',
                insight: '='
            },
            link: function($scope, element, attrs) {
                const resolvedDataset = SmartId.resolve($scope.insight.params.datasetSmartName);
                DataikuAPI.datasets.getFullInfo($scope.insight.projectKey, resolvedDataset.projectKey, resolvedDataset.id).noSpinner().success(function(data) {
                    $scope.datasetFullInfo = data;
                }).error(setErrorInScope.bind($scope));

                if (!['SEARCH', 'EXPLORE'].includes($scope.tile.tileParams.viewKind)) {
                    $scope.tile.tileParams.viewKind = 'EXPLORE';
                }

                $scope.onViewKindChange = function() {
                    if ($scope.tile.tileParams.viewKind === 'SEARCH') {
                        $scope.tile.clickAction = 'DO_NOTHING';
                    }
                };
            }
        };
    });

    app.directive('datasetTableInsightCreateForm', function(DataikuAPI) {
        return {
            templateUrl: '/templates/dashboards/insights/dataset_table/dataset_table_create_form.html',
            scope: true,
            link: function($scope, element, attrs) {

                $scope.hook.beforeSave = function(resolve, reject) {
                    DataikuAPI.explores.get($scope.insight.projectKey, $scope.insight.params.datasetSmartName)
                        .success(function(data) {
                            $scope.insight.params.shakerScript = data.script;
                            resolve();
                        })
                        .error(function(data, status, headers, config, statusText) {
                            reject(arguments);
                        });
                };

                $scope.hook.defaultName = 'Dataset table';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = nv.label + ' table';
                });
            }
        };
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.directive('descriptionInsight', function(TileLoadingState){
        return {
            restrict: 'A',
            scope: {
                hook: '=',
                tile: '='
            },
            link: function($scope){

                if ($scope.tile.tileType !== 'INSIGHT' || $scope.tile.displayMode !== 'INSIGHT_DESC') {
                    return;
                }

                $scope.load = function(resolve) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.COMPLETE;
                    resolve();
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('DiscussionsInsightHandler', {
        name: 'Discussions',
        desc: 'Discussions feed on an object',
        icon: 'icon-comments-alt',
        color: 'discussions',

        getSourceId: function(insight) {
            return insight.params.objectId;
        },
        getSourceType: function(insight) {
            return insight.params.objectType;
        },

        hasEditTab: false,
        defaultTileParams: {
        },
        defaultTileDimensions: [15, 9]
    });

    app.controller('_discussionsInsightViewCommon', function($scope, $controller, DataikuAPI, $stateParams) {
        $scope.resolvedObject = { projectKey: $stateParams.projectKey, type: $scope.insight.params.objectType, id: $scope.insight.params.objectId };

        $scope.fetchdiscussions = function(resolve, reject, noSpinner) {
            const p = DataikuAPI.discussions.getForObject($stateParams.projectKey, $scope.insight.params.objectType, $scope.insight.params.objectId);
            if (noSpinner) {
                p.noSpinner();
            }
            p.noSpinner()
                .success(function(data) {
                    $scope.discussions = data.discussions;
                    if (typeof(resolve)==='function') {
                        resolve();
                    }
                }).error(function(data, status, headers, config, statusText) {
            	setErrorInScope.bind($scope)(data, status, headers, config, statusText);
            	if (typeof(reject)==='function') {
                        reject(data, status, headers, config, statusText);
                    }
        	});
        };
    });

    app.directive('discussionsInsightTile', function($controller, TileLoadingState) {
        return {
            templateUrl: '/templates/dashboards/insights/discussions/discussions_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs) {
                $controller('_discussionsInsightViewCommon', { $scope: $scope });

                $scope.loading = false;
                $scope.loaded = false;
                $scope.error = null;

                $scope.load = function(resolve, reject) {
            	    $scope.loading = true;
                    $scope.fetchdiscussions(
            		function() {
            			 $scope.loading = false;
                            $scope.loaded = true;
                            $scope.error = null;
                            if ($scope.hook && $scope.hook.isErrorMap) {
                         	$scope.hook.isErrorMap[$scope.tile.$tileId] = false;
                            }
                            if (typeof(resolve)==='function') {
                                resolve();
                            }
            		}, function(data, status, headers, config, statusText) {
            			$scope.loading = false;
                            $scope.loaded = false;
                            $scope.error = data;
                            if ($scope.hook && $scope.hook.isErrorMap) {
                        	$scope.hook.isErrorMap[$scope.tile.$tileId] = true;
                            }
                		$scope.hook.setErrorInDashboardPageScope(data, status, headers, config, statusText);
                            if (typeof(reject)==='function') {
                                reject();
                            }
            		}
        		);
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.directive('discussionsInsightTileParams', function() {
        return {
            templateUrl: '/templates/dashboards/insights/discussions/discussions_tile_params.html',
            scope: {
                tileParams: '='
            }
        };
    });

    app.directive('discussionsInsightCreateForm', function() {
        return {
            templateUrl: '/templates/dashboards/insights/discussions/discussions_create_form.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.insight.params.objectType = 'DATASET';
                $scope.hook.defaultName = 'discussions on object';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = 'discussions on ' + nv.label;
                });
            }
        };
    });

    app.directive('discussionsInsightView', function($controller) {
        return {
            templateUrl: '/templates/dashboards/insights/discussions/discussions_view.html',
            scope: true,
            link: function($scope, element, attrs) {
                $controller('_discussionsInsightViewCommon', { $scope: $scope });
                $scope.fetchdiscussions();
            }
        };
    });

    app.directive('discussionsInsightEdit', function($controller, DataikuAPI) {
        return {
            templateUrl: '/templates/dashboards/insights/discussions/discussions_edit.html',
            scope: {
                insight: '='
            }
        };
    });

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('EdaInsightHandler', {
        name: 'Eda',
        nameForTileParams: 'Eda',
        desc: 'Eda',
        icon: 'icon-dku-statistics',
        color: '',

        getSourceId: function(insight) {
            return insight.params.dataSpec.inputDatasetSmartName;
        },
        getSourceType: function() {
            return 'DATASET';
        },
        hasOptions: false,
        hasEditTab: true,
        defaultTileSize: [],
        defaultTileDimensions: [30, 15]
    });

    app.directive('edaInsightTile', function(TileLoadingState) {
        return {
            templateUrl: '/templates/dashboards/insights/eda/eda_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope) {
                $scope.loading = false;
                $scope.loaded = false;
                $scope.error = null;

                $scope.load = function(resolve) {
                    $scope.loading = false;
                    $scope.loaded = true;
                    $scope.error = null;
                    if ($scope.hook && $scope.hook.isErrorMap) {
                        $scope.hook.isErrorMap[$scope.tile.$tileId] = false;
                    }
                    if (typeof (resolve) === 'function') {
                        resolve();
                    }
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }

                $scope.tile.$showEdaSamplingBadge = false;
                $scope.displaySamplingBadge = (show) => {
                    $scope.tile.$showEdaSamplingBadge = show;
                };

                $scope.tile.$samplingSummaryMessage =
                    'Data sampling can be misrepresentative.<br>' +
                    'You can change the sampling settings in the edit view.';
            }
        };
    });

    app.controller('EdaInsightViewCommon', function($scope, DataikuAPI) {
        $scope.saveInsight = function({ card, result, dataSpec, isSampleWholeDataset }) {
            const newInsight = _.cloneDeep($scope.insight);
            newInsight.params.card = card;
            newInsight.params.dataSpec = dataSpec;
            newInsight.params.isSampleWholeDataset = isSampleWholeDataset;

            DataikuAPI.dashboards.insights.save(newInsight, undefined, JSON.stringify(result))
                .error(setErrorInScope.bind($scope))
                .success(() => $scope.insight = newInsight);
        };
    });

    app.directive('edaInsightView', function($controller) {
        return {
            templateUrl: '/templates/dashboards/insights/eda/eda_view.html',
            scope: true,
            link: function($scope) {
                $controller('EdaInsightViewCommon', { $scope });
            }
        };
    });

    app.directive('edaInsightEdit', function($controller) {
        return {
            templateUrl: '/templates/dashboards/insights/eda/eda_edit.html',
            scope: {
                insight: '='
            },
            link: function($scope) {
                $controller('EdaInsightViewCommon', { $scope });
            }
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.controller('InsightEditController', function($scope, Dialogs, TopNav) {
        TopNav.setLocation(TopNav.TOP_DASHBOARD, 'insights', null, 'edit');
        if ($scope.insight) {
            TopNav.setPageTitle($scope.insight.name + ' - Insight');
        }

        Dialogs.saveChangesBeforeLeaving($scope, $scope.isDirty, $scope.saveInsight, $scope.revertChanges, 'This insight has unsaved changes.');
        Dialogs.checkChangesBeforeLeaving($scope, $scope.isDirty);
    });

    app.directive('insightEditGoToView', function($state) {
        return {
            link: function() {
                $state.go('projects.project.dashboards.insights.insight.view', { location: 'replace', inherit: true });
            }
        };
    });

    app.controller('NewInsightModalController', function($scope, $controller, $stateParams, $q, $filter,
				   DataikuAPI, INSIGHT_TYPES, DashboardUtils, DSSVisualizationThemeUtils, $rootScope) {

        $scope.DashboardUtils = DashboardUtils;
        $scope.insightTypes = INSIGHT_TYPES;
        $scope.displayableInsightType = null;

        $scope.uiState = $scope.uiState || {};
        $scope.uiState.modalTab = 'new';
        $scope.filter = {};

        // Load insight list
        DataikuAPI.dashboards.insights.listWithAccessState($stateParams.projectKey)
            .success(function(data) {
                $scope.insights = data.insights;
                filterInsights();
                $scope.insightAccessMap = angular.extend($scope.insightAccessMap || {}, data.insightAccessData);
            }).error(setErrorInScope.bind($scope));

        $scope.$watch('filter', function(nv, ov) {
            filterInsights();
        }, true);

        $scope.$watch('insight.type', function(nv, ov) {
            filterInsights();
        });

        function filterInsights() {
            $scope.filteredInsights = $filter('filter')($scope.insights, { type: $scope.insight.type });
            $scope.noInsightOfSelectedType = !$scope.filteredInsights || $scope.filteredInsights.length == 0;
            $scope.filteredInsights = $filter('filter')($scope.filteredInsights, { name: $scope.filter.q });
            if ($scope.filter.sourceId) {
                $scope.filteredInsights = $scope.filteredInsights.filter(function(insight) {
                    return DashboardUtils.getInsightSourceId(insight) == $scope.filter.sourceId;
                });
            }
        }

        function isFiltersColumnsSelectionStep() {
            return $scope.displayableInsightType && $scope.displayableInsightType.toLowerCase() === 'filters' && $scope.uiState.source;
        };

        // Insights types that share the same create-form directive
        const INSIGHT_TYPE_GROUPS = {
            'scenario_last_runs': 'scenario',
            'scenario_run_button': 'scenario'
        };

        $scope.getInsightTypeGroup = function(insightType) {
            return INSIGHT_TYPE_GROUPS[insightType] || insightType;
        };

        $scope.simpleTileTypes = {
            text: { name: 'Text', desc: 'Zone of text' },
            image: { name: 'Image', desc: 'Upload an image' },
            iframe: { name: 'Web Content', desc: 'Embedded web page' },
            group: { name: 'Group', desc: 'Tile that can contain others' }
        };

        $scope.setDashboardCreationId = function(dashboardId) {
            $scope.insight.dashboardCreationId = dashboardId;
        };

        $scope.getModalTitle = function() {
            if ($scope.inDashboard) {
                const insightType = $scope.displayableInsightType ? $scope.displayableInsightType.toLowerCase() : 'tile';
                return insightType === 'filters' ? 'Add filters to slide' : 'Add ' + insightType + ' to dashboard';
            }
            return $scope.displayableInsightType ? 'New ' + $scope.displayableInsightType + ' insight' : 'New insight';
        };

        $scope.getModalStepHeaderTitle = function() {
            return $scope.displayableInsightType.toLowerCase() === 'filters' ?
                'Select origin of the source dataset' : 'Type selection';
        };

        $scope.isFiltersSourceSelectionStep = function() {
            return $scope.displayableInsightType && $scope.displayableInsightType.toLowerCase() === 'filters' && !$scope.uiState.source;
        };

        $scope.resetModal = function() {
            if (isFiltersColumnsSelectionStep() && $scope.filtersCompatibleTiles.length) {
                $scope.uiState.source = null;
                return; // return to filters source selection
            }

            const insightTheme = $scope.dashboard ? $scope.dashboard.theme : $rootScope.appConfig.selectedDSSVisualizationTheme;
            $scope.insight = {
                projectKey: $stateParams.projectKey,
                params: {
                    theme: insightTheme
                }
            };

            $scope.displayableInsightType = null;

            $scope.filter = {};

            $scope.hook = {
                //Can be overwritten by {{insightType}}InsightCreateForm directives
                beforeSave: function(resolve, reject) {
                    resolve();
                },

                //Can be overwritten by {{insightType}}InsightCreateForm directives
                afterSave: function(resolve, reject) {
                    resolve();
                },

                defaultName: null,
                sourceObject: {},
                setErrorInModaleScope : function(data, status, headers, config, statusText) {
                    setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                }
            };
        };

        $scope.resetModal();

        $scope.returnSimpleTile = function(tileType) {
            $scope.resolveModal({
                tileType: tileType
            });
        };

        $scope.selectType = function(type) {
            $scope.insight.type = type;
            $scope.canPickExistingInsight = DashboardUtils.canListInsight($scope.insight);
            $scope.displayableInsightType = DashboardUtils.getInsightHandler(DashboardUtils.getInsightTypeGroup(type)).name;
            if (type === 'static_file') {
                $scope.pointerMode.isPointerMode = true;
            }
        };

        function beforeSavePromise() {
            const deferred = $q.defer();
            $scope.hook.beforeSave(deferred.resolve, deferred.reject);
            return deferred.promise;
        }

        function afterSavePromise() {
            const deferred = $q.defer();
            $scope.hook.afterSave(deferred.resolve, deferred.reject);
            return deferred.promise;
        }

        $scope.create = function() {
            if (!$scope.insight.name) {
                $scope.insight.name = $scope.hook.defaultName;
            }

            beforeSavePromise().then(
                function() {
                    function save() {
                        DataikuAPI.dashboards.insights.save($scope.insight)
                            .error(setErrorInScope.bind($scope))
                            .success(function(insightId) {
                                $scope.insight.id = insightId;
                                if ($scope.hook.sourceObject && !$scope.hook.noReaderAuth) {
                                    $scope.insight.isReaderAccessible = $scope.hook.sourceObject.isReaderAccessible;
                                } else {
                                    $scope.insight.isReaderAccessible = true;
                                }
                                $scope.resolveModal({ insight: $scope.insight, redirect: $scope.hook.redirect });

                                const insightIsAPointer = $scope.$modalScope.pointerMode && $scope.$modalScope.pointerMode.isPointerMode || false;
                                const createdFromDashboard = !!$scope.$modalScope.inDashboard;
                                if (!insightIsAPointer) {
                                    DashboardUtils.sendWT1InsightCreation($scope.insight.type, {
                                        triggeredFrom: createdFromDashboard ? 'dashboard-new-tile-modal' : 'insight-list-new-insight-modal'
                                    });
                                }

                                afterSavePromise().then(
                                    function() {
                                        //nothing specific to do in case of success
                                    },
                                    function(data, status, headers, config, statusText) {
                                        setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                                    }
                                );
                            });
                    }

                    if ($scope.hook.addSourceToReaderAuthorizations) {
                        const neededReaderAuthorization = DashboardUtils.getNeededReaderAuthorization($scope.insight);

                        DataikuAPI.projects.addReaderAuthorizations($stateParams.projectKey, [neededReaderAuthorization])
                            .success(function() {
                                $scope.hook.sourceObject.isReaderAccessible = true;
                                save();

                                // For model evaluation insights, the referenced SM must be authorized also
                                if ($scope.insight.type === 'model-evaluation_report') {
                                    DataikuAPI.modelevaluationstores.getLatestEvaluationDetails($scope.insight.projectKey, $scope.insight.params.mesSmartId)
                                        .success(function(data) {
                                            if (data.evaluation && data.evaluation.hasModel) {
                                                const smResolved = resolveObjectSmartId(data.evaluation.modelRef.smId);
                                                const ref = {
                                                    objectType: 'SAVED_MODEL',
                                                    objectId: smResolved.id
                                                };
                                                if (smResolved.projectKey) {
                                                    ref.projectKey = smResolved.projectKey;
                                                }
                                                DataikuAPI.projects.addReaderAuthorizations($stateParams.projectKey, [{ objectRef: ref, modes: ['READ'] }])
                                                    .success(function() {
                                                        save();
                                                    })
                                                    .error(setErrorInScope.bind($scope));
                                            }
                                        })
                                        .error(setErrorInScope.bind($scope));
                                }
                            })
                            .error(setErrorInScope.bind($scope));
                    } else {
                        save();
                    }
                },
                function(argArray) {
                    if (argArray) {
                        setErrorInScope.bind($scope).apply(null, argArray);
                    }
                }
            );
        };
    });

    app.directive('insightSourceInfo', function(DashboardUtils) {
        return {
            templateUrl: 'templates/dashboards/insights/insight-source-info.html',
            link: function($scope, element, attrs) {

                if (!$scope.inDashboard) {
                    return;
                }

                function updateMatches() {
                    if (!$scope.insights) {
                        return;
                    }
                    const handler = DashboardUtils.getInsightHandler($scope.insight.type);
                    $scope.matching = $scope.insights.filter(function(insight) {
                        return $scope.getInsightTypeGroup(insight.type) == $scope.getInsightTypeGroup($scope.insight.type)
							&& handler.getSourceId(insight) == handler.getSourceId($scope.insight)
							&& (handler.sourceType || (handler.getSourceType(insight) == handler.getSourceType($scope.insight)));
                    });
                }

                $scope.go = function() {
                    const handler = DashboardUtils.getInsightHandler($scope.insight.type);
                    $scope.filter.sourceId = handler.getSourceId($scope.insight);
                    $scope.filter.sourceType = handler.sourceType || handler.getSourceType($scope.insight);
                    $scope.uiState.modalTab = 'existing';
                };

                $scope.$watch('insight', updateMatches, true);
            }
        };
    });


    app.controller('CopyInsightModalController', function($scope, DataikuAPI, ActivityIndicator, StateUtils, $stateParams, DashboardUtils) {
        $scope.init = function(insight) {
            $scope.insight = insight;
            $scope.insight.newName = 'Copy of ' + insight.name;
        };

        $scope.copy = function() {
            DataikuAPI.dashboards.insights.copy($stateParams.projectKey, [$scope.insight.id], [$scope.insight.newName])
                .error(setErrorInScope.bind($scope))
                .success(function(data) {
                    const insightCopy = data[0];
                    const href = StateUtils.href.insight(insightCopy.id, insightCopy.projectKey, { name: insightCopy.name });
                    ActivityIndicator.success($scope.insight.name + ' copied into ' + insightCopy.name + ', <a href=\'' + href + '\' >view insight</a>.', 5000);
                    $scope.resolveModal();
                    DashboardUtils.sendWT1InsightCreation($scope.insight.type, {
                        triggeredFrom: 'insight-list-copy-tile-modal'
                    });
                });
        };
    });


    app.controller('MoveCopyTileModalController', function($scope, DataikuAPI, $controller, StateUtils, $rootScope, $timeout, TileUtils, $stateParams, DashboardUtils) {
        $controller('MultiPinInsightModalController', { $scope:$scope });

        $scope.keepOriginal = true;
        $scope.pointerMode = {
            mode: false
        };

        const initCopy = $scope.init;
        $scope.init = function(insight, tileParams) {
            if ($scope.insight) {
                initCopy(insight, tileParams);
            } else {
                //creating new tile
                $scope.newTile = TileUtils.copyNonInsightTile($scope.tile);
                //listing dashboards
                $scope.listDashboards($scope.insight);
            }
        };

        /*
         * Methods used if copying/moving tile to other dashboard
         */
        $scope.addPinningOrder = function() {
            for (let i = 0; i < $scope.dashboards.length; i++) {
                if ($scope.dashboards[i].id == $scope.dashboard.id) {
                    $scope.pinningOrder = {
                        dashboard: $scope.dashboard,
                        page: $scope.dashboard.pages[$scope.uiState.currentPageIdx],
                        useInsightTheme: false
                    };
                    $scope.pinningOrders = [$scope.pinningOrder];
                    break;
                }
            }
        };

        $scope.multiPinCallback = function() {
            removeOriginalIfNeeded();
            $timeout(function() {
                const pin = $scope.pinningOrders[0];
                const options = {
                    name: pin.dashboard.name,
                    tab: 'edit',
                    pageId: pin.page.id
                };
                StateUtils.go.dashboard(pin.dashboard.id, $stateParams.projectKey, options).then(function() {
                    const unregister = $rootScope.$on('dashboardSyncModelsDone', () => {
                        unregister();
                        $rootScope.$broadcast('dashboardSelectLastTile');
                    });
                });
            });
        };

        /*
         * Methods used if copying/moving the tile in the current dashboard
         */

        function moveCopyTile(destinationPageCid) {
            let destinationPage = null;
            for (let i=0; i<$scope.dashboard.pages.length; i++) {
                if ($scope.dashboard.pages[i].cid == destinationPageCid || $scope.dashboard.pages[i].$$hashKey == destinationPageCid) {
                    destinationPage = $scope.dashboard.pages[i];
                    break;
                }
            }

            function moveCopyTileFront(insightId) {
                const copyTile = angular.copy($scope.tile);
                delete copyTile.$added;
                delete copyTile.$tileId;
                delete copyTile.box.top;
                delete copyTile.box.left;

                if (insightId) {
                    copyTile.insightId = insightId;
                }


                DashboardUtils.sendWT1TileCreation(copyTile.tileType, {
                    insightType: copyTile.insightType,
                    reuseExistingInsight: $scope.pointerMode.mode,
                    triggeredFrom: 'move-or-copy-tile-modal'
                });

                destinationPage.grid.tiles.push(copyTile);
                removeOriginalIfNeeded();
                $scope.uiState.currentPageIdx = $scope.dashboard.pages.indexOf(destinationPage);
                $scope.uiState.selectedTile = copyTile;
            }

            if ($scope.tile.tileType == 'INSIGHT' && $scope.keepOriginal && !$scope.pointerMode.mode) {
                DataikuAPI.dashboards.insights.copy($stateParams.projectKey, [$scope.tile.insightId], [$scope.insightName], $scope.dashboard.id)
                    .error(setErrorInScope.bind($scope))
                    .success(function(data) {
                        const insightCopy = data[0];
                        $scope.insightsMap[insightCopy.id] = insightCopy;
                        DashboardUtils.sendWT1InsightCreation(insightCopy.type, {
                            triggeredFrom: 'move-or-copy-tile-modal'
                        });
                        moveCopyTileFront(insightCopy.id);
                        $scope.dismiss();
                    });
            } else {
                moveCopyTileFront();
                $scope.dismiss();
            }
        }

        function removeOriginalIfNeeded() {
            if (!$scope.keepOriginal) {
                $scope.removeTileFromDashboardPage($scope.tile.$tileId);
            }
        }

        /*
         * Forms methods
         */

        $scope.validate = function() {
            if ($scope.pinningOrder.dashboard.id == $scope.dashboard.id) {
                moveCopyTile($scope.pinningOrder.page.cid || $scope.pinningOrder.page.$$hashKey);
            } else {
                $scope.sendPinningOrders();
            }
        };

        $scope.checkConsistency = function() {
            if ($scope.keepOriginal) {
                $scope.pointerMode.mode = false;
            }
        };

        $scope.getPagesList = function() {
            if (!$scope.pinningOrder) {
                return [];
            }

            if (!$scope.keepOriginal) {
                return $scope.pinningOrder.dashboard.pages.filter(function(page) {
                    return $scope.page.cid != page.cid;
                });
            } else {
                return $scope.pinningOrder.dashboard.pages;
            }
        };
    });


    app.controller('_MultiPinInsightModalCommonController', function($scope, $timeout, DataikuAPI, TileUtils, DatasetUtils, $stateParams, DashboardUtils, CreateAndPinInsight, translate) {
        $scope.existingPinningList = [];
        $scope.pinningOrders = [];
        $scope.initiated = false;
        $scope.hasSampledChartInsightSelected = false;
        $scope.dashboardNamePlaceholder = DashboardUtils.getDashboardDefaultName();
        $scope.dummyPageModel = { ...CreateAndPinInsight.getPageTemplate, title: translate('INSIGHTS.MODAL.DUMMY_PAGE_PLACEHOLDER', 'Page 1') };
        // Warning: Sets are not supported by AngularJS! Do not use it in AngularJS scope, templates, or inside methods like `angular.equals`, JSON.stringify...
        const sampledChartInsightSelected = new Set();

        /*
         * Form initialization
         */

        $scope.init = function(insight, tileParams, payload) {
            //create new tile
            $scope.insight = insight;
            $scope.newTile = $scope.initNewTile(insight, tileParams);
            $scope.payload = payload;

            // list dashboards where we could copy it
            $scope.listDashboards(insight);
        };

        $scope.initNewTile = function(insight, tileParams) {
            const newTile = TileUtils.newInsightTile(insight);
            if (tileParams) {
                angular.extend(newTile.tileParams, tileParams);
            }

            if ($scope.tile) {
                newTile.box = angular.copy($scope.tile.box);
            }

            newTile.box.left = -1;
            newTile.box.top = -1;

            return newTile;
        };

        function getChartSampleMetadataRequest(chartInsight){
            const inputDatasetLoc = DatasetUtils.getLocFromSmart($stateParams.projectKey, chartInsight.params.datasetSmartName);
            const dataSpec = {
                datasetProjectKey : inputDatasetLoc.projectKey,
                datasetName : inputDatasetLoc.name,
                script: angular.copy($scope.shaker),
                copySelectionFromScript: chartInsight.params.copySelectionFromScript,
                sampleSettings : chartInsight.params.refreshableSelection,
                engineType : chartInsight.params.engineType
            };
            dataSpec.script.origin = 'DATASET_EXPLORE';
            const sampleId = chartInsight.params.summary && chartInsight.params.summary.requiredSampleId || null;
            return { dataSpec, sampleId };
        }

        $scope.insightSelectionChange = function(insightDataItem){
            if (insightDataItem.selected && insightDataItem.sampleMetadata && insightDataItem.sampleMetadata.sampleIsWholeDataset === false){
                sampledChartInsightSelected.add(insightDataItem);
            } else {
                sampledChartInsightSelected.delete(insightDataItem);
            }
            $scope.hasSampledChartInsightSelected = sampledChartInsightSelected.size > 0;
        };

        $scope.fetchChartInsightsMetaData = function(chartInsights){
            const chartsData = chartInsights.map(chartInsight => getChartSampleMetadataRequest(chartInsight));
            DataikuAPI.shakers.charts.getSamplesMetaData($stateParams.projectKey, chartsData)
                .error(setErrorInScope.bind($scope))
                .success(function(samplesMetadata) {
                    samplesMetadata = samplesMetadata || [];
                    $scope.insightData.items.forEach((item, i) => {
                        item.sampleMetadata = samplesMetadata.length > i ? samplesMetadata[i] : {};
                        $scope.insightSelectionChange(item);
                    });
                });
        };

        $scope.listDashboards = function(insight){
            DataikuAPI.dashboards.listEditable($stateParams.projectKey)
                .error(setErrorInScope.bind($scope))
                .success(function(data) {
                    $scope.dashboards = data.dashboards;
                    $scope.allDashboardsCount = data.allDashboardsCount;

                    // Use local state of current dashboard to take into account slides being added / removed
                    $scope.dashboards.forEach(function(dashboard, index) {
                        if ($scope.dashboard && $scope.dashboard.id === dashboard.id) {
                            $scope.dashboards[index] = $scope.dashboard;
                        }

                        //listing where insight got already pinned
                        if (insight && insight.id) {
                            dashboard.pages.forEach(function(page, pageIndex) {
                                page.index = pageIndex;
                                TileUtils.getFlattenTileList(page.grid.tiles).forEach((tile) => {
                                    if (tile.insightId == insight.id) {
                                        $scope.existingPinningList.push({
                                            'dashboard': dashboard,
                                            'page': page
                                        });
                                    }
                                });
                            });
                        }
                    });

                    if ($scope.dashboards.length > 0) {
                        $scope.addPinningOrder();
                    }
                    $scope.dashboardOptions = [
                        ...$scope.dashboards,
                        CreateAndPinInsight.getCreateDashboardOption()
                    ];
                    $scope.pageOptionsByDashboardId = Object.fromEntries($scope.dashboards.map(({ id, pages }) => ([ id, [...pages, CreateAndPinInsight.getCreatePageOption()]])));
                    $scope.initiated = true;
                });
        };

        $scope.onDashboardSelected = function(pinningOrder, index) {
            // If the option that has been selected is the "Create new dashboard one".
            if (!pinningOrder.dashboard.id) {
                const dashboardToCreate = CreateAndPinInsight.getDashboardTemplate();
                pinningOrder.dashboard = dashboardToCreate;
                pinningOrder.page = dashboardToCreate.pages[0];
                //  Focus the last "Create new dashboard" input.
                $timeout(() => document.querySelector(`.multi-pin-insight-table .dashboard-and-slide-selection:nth-child(${index + 1}) .dashboard-and-slide-selection__dashboard .dashboard-and-slide-selection__create-new-item`).focus(), 100);
            } else {
                pinningOrder.page = pinningOrder.dashboard.pages[0];
            }
        };

        $scope.onPageSelected = function(pinningOrder, index) {
            if (pinningOrder.page.id) {
                return;
            }
            const newPage = CreateAndPinInsight.getPageTemplate();
            pinningOrder.page = newPage;
            $timeout(() => {
                const inputElement = document.querySelector(`.multi-pin-insight-table tr:nth-child(${index + 1}) .dashboard-and-slide-selection__page .dashboard-and-slide-selection__create-new-item`);
                if (!_.isNil(inputElement)) {
                    inputElement.focus();
                }
            });
        };

        /*
         * PinningOrder CRUD
         */

        $scope.addPinningOrder = function() {
            const projectHasDashboards = $scope.dashboards && $scope.dashboards.length > 0;
            $scope.pinningOrders.push({
                dashboard: projectHasDashboards ? $scope.dashboards[0] : CreateAndPinInsight.getDashboardTemplate(),
                page: projectHasDashboards ? $scope.dashboards[0].pages[0] : CreateAndPinInsight.getPageTemplate(),
                useInsightTheme: projectHasDashboards ? !$scope.dashboards[0].theme : true
            });
        };

        $scope.removePinningOrder = function(index) {
            $scope.pinningOrders.splice(index, 1);
        };

        $scope.getLightPinningOrders = function(pinningOrders = $scope.pinningOrders) {
            const lightPinningOrders = [];
            pinningOrders.forEach(function(pinningOrder) {
                lightPinningOrders.push({
                    dashboardId: pinningOrder.dashboard.id,
                    pageId: pinningOrder.page.id,
                    useInsightTheme: pinningOrder.useInsightTheme
                });
            });
            return lightPinningOrders;
        };

        /*
         * UI
         */

        const pagesLabels = {};
        $scope.getPageLabel = function(dashboard, page) {
            if (page.id && pagesLabels[page.id]) {
                return pagesLabels[page.id];
            }
            let pageLabel = '';
            if (page.title) {
                pageLabel = page.title;
            } else {
                pageLabel = 'Page ' + (dashboard.pages.indexOf(page) + 1);
            }
            if (page.id) {
                pagesLabels[page.id] = pageLabel;
            }
            return pageLabel;
        };
    });


    app.controller('MultiPinInsightModalController', function($scope, DataikuAPI, ActivityIndicator, CreateAndPinInsight, $controller, $stateParams, DSSVisualizationThemeUtils, DashboardUtils) {
        $controller('_MultiPinInsightModalCommonController', { $scope:$scope });

        const initCopy = $scope.init;
        $scope.init = function(insight, tileParams) {
            initCopy(insight, tileParams);
            $scope.pointerMode = {
                mode: insight.type == 'static_file'
            };
        };

        $scope.sendPinningOrders = function() {
            CreateAndPinInsight.createPinningOrdersDashboardAndPages($scope.pinningOrders, setErrorInScope.bind($scope)).subscribe((updatedPinningOrders) => {
                const lightPinningOrders = $scope.getLightPinningOrders(updatedPinningOrders);

                const newTiles = Array.from({ length: updatedPinningOrders.length }, () => _.cloneDeep($scope.newTile));

                for (let tileIndex = 0; tileIndex < newTiles.length; tileIndex++) {
                    DSSVisualizationThemeUtils.applyToTile(newTiles[tileIndex], updatedPinningOrders[tileIndex].dashboard.theme);
                }

                DataikuAPI.dashboards.multiPin($stateParams.projectKey, $scope.insight.id, newTiles, lightPinningOrders, $scope.pointerMode.mode)
                    .error(setErrorInScope.bind($scope))
                    .success(function() {
                        ActivityIndicator.success('Saved!');
                        lightPinningOrders.forEach(() => {
                            if (!$scope.pointerMode.mode) {
                                DashboardUtils.sendWT1InsightCreation($scope.insight.type, {
                                    triggeredFrom: 'insight-list-add-to-dashboard-modal'
                                });
                            }
                            DashboardUtils.sendWT1TileCreation('INSIGHT', {
                                insightType: $scope.insight.type,
                                reuseExistingInsight: $scope.pointerMode.mode,
                                triggeredFrom: 'insight-list-add-to-dashboard-modal'
                            });
                        });
                        if ($scope.multiPinCallback && typeof($scope.multiPinCallback)==='function') {
                            $scope.multiPinCallback();
                        }
                        $scope.resolveModal();
                    });
            });
        };
    });

    app.controller('_CreateAndPinInsightModalCommonController', function($scope, DataikuAPI, ActivityIndicator, TileUtils, $controller, $timeout, StateUtils, $rootScope, DashboardUtils, $stateParams, WT1, CreateAndPinInsight, DSSVisualizationThemeUtils) {
        $controller('_MultiPinInsightModalCommonController', { $scope:$scope });

        $scope.missingReaderAuthorizations = [];
        $scope.addReaderAuthorization = $scope.projectSummary.canManageDashboardAuthorizations;

        $scope.authorize = function(insights) {
            const neededReaderAuthorizations = insights.map(_ => DashboardUtils.getNeededReaderAuthorization(_));

            DataikuAPI.projects.checkReaderAuthorizations($stateParams.projectKey, neededReaderAuthorizations)
                .error(setErrorInScope.bind($scope))
                .success(function(data) {
                    $scope.missingReaderAuthorizations = data;
                });
        };

        $scope.sendCreateAndPinOrders = function(insights, newTiles, payloads) {
            function save() {
                CreateAndPinInsight.createPinningOrdersDashboardAndPages($scope.pinningOrders, setErrorInScope.bind($scope)).subscribe((updatedPinningOrders) => {
                    const lightPinningOrders = $scope.getLightPinningOrders(updatedPinningOrders);

                    const tileList = Array.from({ length: updatedPinningOrders.length }, () => _.cloneDeep(newTiles));

                    for (let pinningOrdersIndex = 0; pinningOrdersIndex < updatedPinningOrders.length; pinningOrdersIndex++) {
                        for (let insightIndex = 0; insightIndex < insights.length; insightIndex++) {
                            DashboardUtils.sendWT1TileCreation('INSIGHT', {
                                insightType: insights[insightIndex].type,
                                triggeredFrom: 'publish-to-dashboard-modal'
                            });
                            DSSVisualizationThemeUtils.applyToTile(tileList[pinningOrdersIndex][insightIndex], updatedPinningOrders[pinningOrdersIndex].dashboard.theme);
                        }
                    }

                    DataikuAPI.dashboards.insights.createAndPin($stateParams.projectKey, insights, tileList, lightPinningOrders, payloads)
                        .error(setErrorInScope.bind($scope))
                        .success(function(data) {
                            insights.forEach(insight => {
                                if (insight.type === 'chart') {
                                    WT1.event('chart-publish', {
                                        chartId: `${insight.projectKey.dkuHashCode()}.${insight.params.datasetSmartName.dkuHashCode()}.${insight.params.def.name.dkuHashCode()}`,
                                        chartType: insight.params.def.type,
                                        chartVariant: insight.params.def.variant
                                    });
                                }
                                DashboardUtils.sendWT1InsightCreation(insight.type, {
                                    triggeredFrom: 'publish-to-dashboard-modal'
                                });
                            });


                            ActivityIndicator.success('Saved!');
                            if (updatedPinningOrders.length === 0) {
                                StateUtils.go.insight(data[0], $stateParams.projectKey, { name: insights[0].name });
                            } else {
                                const pin = updatedPinningOrders[0];
                                const options = {
                                    name: pin.dashboard.name,
                                    tab: 'edit',
                                    pageId: pin.page.id
                                };
                                StateUtils.go.dashboard(pin.dashboard.id, $stateParams.projectKey, options).then(function() {
                                    // Save the new dashboard layout to avoid dirty state
                                    const unregister = $rootScope.$on('dashboardSyncModelsDone', () => {
                                        unregister();
                                        $rootScope.$broadcast('dashboardSelectLastTile');
                                        $timeout(() => $rootScope.$broadcast('saveDashboard'));
                                    });
                                });
                            }
                            $scope.resolveModal();
                        });
                });
            }

            if ($scope.addReaderAuthorization && $scope.missingReaderAuthorizations) {
                DataikuAPI.projects.addReaderAuthorizations($stateParams.projectKey, $scope.missingReaderAuthorizations)
                    .success(save)
                    .error(setErrorInScope.bind($scope));
            } else {
                save();
            }
        };
    });

    app.controller('CreateAndPinInsightModalController', function($scope, DataikuAPI, ActivityIndicator, TileUtils, $controller, $timeout, StateUtils, $rootScope, DashboardUtils, $stateParams) {
        $controller('_CreateAndPinInsightModalCommonController', { $scope:$scope });

        const initCopy = $scope.init;
        $scope.init = function(insight, tileParams, payload) {
            initCopy(insight, tileParams, payload);
            $scope.authorize([insight]);
            $scope.hasSampledInsight = false;
        };

        const sendOrders = $scope.sendCreateAndPinOrders;
        $scope.sendCreateAndPinOrders = function() {
            sendOrders([$scope.insight], [$scope.newTile], [$scope.payload]);
        };
    });

    app.controller('CreateAndPinInsightsModalController', function($scope, DataikuAPI, ActivityIndicator, TileUtils, $controller, $timeout, ChartsStaticData, StateUtils, $rootScope, DashboardUtils, $stateParams, CreateAndPinInsight) {
        $controller('_CreateAndPinInsightModalCommonController', { $scope });

        $scope.insights = [];
        $scope.newTiles = [];
        $scope.insightData = [];
        $scope.warningBadge = ChartsStaticData.WARNING_BADGE;

        $scope.init = function(insights, tileParams) {
            //create new tile
            insights.forEach(insight => {
                $scope.insights.push(insight);
                $scope.newTiles.push($scope.initNewTile(insight, tileParams));
            });

            const areChartInsights = insights.length && insights[0].type === 'chart';
            if (areChartInsights) {
                $scope.fetchChartInsightsMetaData(insights);
            }
            // list dashboards where we could copy it
            $scope.listDashboards($scope.insights[0]);
            $scope.authorize(insights);
        };

        const sendOrders = $scope.sendCreateAndPinOrders;
        $scope.sendCreateAndPinOrders = function() {
            const insights = $scope.insights.filter((_, i) => $scope.insightData.items[i].selected);
            const newTiles = $scope.newTiles.filter((_, i) => $scope.insightData.items[i].selected);

            sendOrders(insights, newTiles);
        };

        $scope.canCreate = function() {
            return $scope.insightData.items && $scope.insightData.items.some(_ => _.selected);
        };
    });
})();

;
(function() {
    'use strict';

    // Was previously located here: static/dataiku/js/dashboards/insights/filters.js
    angular.module('dataiku.dashboards.insights').constant('FiltersInsightHandler', {
        name: 'Filters',
        desc: 'Filter your slide\'s data',
        icon: 'dku-icon-recipe-filter-circle-fill-24',
        color: 'chart',
        sourceType: 'DATASET',
        getSourceId: (insight) => {
            return insight.params.datasetSmartName;
        },
        hasEditTab: true,
        getDefaultTileParams: () => {
            return {
                filters: []
            };
        }
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.directive('iframeTile', function($sce, $timeout, TileLoadingState, TileLoadingBehavior){
        return {
            templateUrl: '/templates/dashboards/insights/iframe/iframe_tile.html',
            scope: {
                tileParams: '=',
                editable: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs){

                $scope.$watch('tileParams.url', function(nv) {
                    if (!nv) {
                        return;
                    }
                    /* Copied from angular-sanitize LINKY_REGEX */
                    const URL_REGEX = /((ftp|https?):\/\/|(www\.)|(mailto:)?[A-Za-z0-9._%+-]+@)\S*[^\s.;,(){}<>"\u201d\u2019]/i;
                    if ($scope.tileParams.url.match(URL_REGEX)) {

                        if ($scope.tileParams.url.startsWith(window.location.origin)) {
                            $scope.sandboxedIframe = true;
                        } else {
                            $scope.sandboxedIframe = false;
                        }

                        $scope.trustedUrl = $sce.trustAsResourceUrl($scope.tileParams.url);
                    } else {
                        $scope.trustedUrl = $scope.tileParams.url; // Since it's not trusted it will fail
                    }
                });

                const timeoutInSeconds = Math.min($scope.tileParams.loadTimeoutInSeconds, 240);
                if (timeoutInSeconds > 0) {
                    $scope.load = function(resolve, reject) {
                        $timeout(function() {
                            $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.COMPLETE;
                        }, timeoutInSeconds * 1000);
                        if (typeof(resolve) === 'function') {
                            resolve();
                        }
                        return TileLoadingBehavior.DELAYED_COMPLETE;
                    };
                    $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                    $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.directive('iframeTileParams', function(){
        return {
            templateUrl: '/templates/dashboards/insights/iframe/iframe_tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs){
                // Used when creating a new tile to correctly initialize the timeout value in editor.
                $scope.$watch('tileParams', function(nv) {
                    if (nv && nv.loadTimeoutInSeconds === undefined) {
                        nv.loadTimeoutInSeconds = 0;
                    }
                });
                if ($scope.tileParams.loadTimeoutInSeconds === undefined) {
                    $scope.tileParams.loadTimeoutInSeconds = 0;
                }
            }
        };
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.directive('imageAndInsightDescription', function(TileLoadingState){
        return {
            restrict: 'A',
            scope: {
                hook: '=',
                tile: '='
            },
            link: function($scope){

                if ($scope.tile.tileType !== 'INSIGHT' || $scope.tile.displayMode !== 'IMAGE_AND_INSIGHT_DESC') {
                    return;
                }

                $scope.load = function(resolve) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.COMPLETE;
                    resolve();
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
            }
        };
    });

    app.directive('imageTile', function(TileLoadingState){
        return {
            restrict: 'A',
            scope: {
                hook: '=',
                tile: '='
            },
            link: function($scope){

                if ($scope.tile.tileType !== 'IMAGE' && $scope.tile.displayMode !== 'IMAGE') {
                    return;
                }

                $scope.load = function(resolve) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.COMPLETE;
                    resolve();
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
            }
        };
    });

    app.directive('imageTileParams', function(TileUtils){
        return {
            templateUrl: '/templates/dashboards/insights/image/image_tile_params.html',
            scope: {
                tile: '='
            },
            link: function($scope, element, attrs){
                $scope.openUploadPictureDialog = TileUtils.openUploadPictureDialog;
            }
        };
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('JupyterInsightHandler', {
        name: 'Notebook',
        desc: 'Code analysis',
        icon: 'icon-dku-nav_notebook',
        color: 'notebook',

        getSourceId: function(insight) {
            return insight.params.notebookSmartName;
        },
        sourceType: 'JUPYTER_NOTEBOOK',
        hasEditTab: false,
        defaultTileParams: {
            showCode: false
        },
        defaultTileDimensions: [12, 15]
    });

    app.controller('JupyterInsightCommonController', function($scope, $sce, DataikuAPI, $stateParams, $timeout) {

    	$scope.getLoadingPromise = function() {
        	if ($scope.insight.params.loadLast) {
        		return DataikuAPI.jupyterNotebooks.export.getLast($scope.insight.projectKey, $scope.insight.params.notebookSmartName);
        	} else {
        		return DataikuAPI.jupyterNotebooks.export.get($scope.insight.projectKey, $scope.insight.params.notebookSmartName, $scope.insight.params.exportTimestamp);
        	}
    	};

    	$scope.displayExport = function(html, showCode, pointer) {
    		if (html) {
    			$scope.exportNotFound = false;
            	$timeout(function() {
                    showHTML(html, showCode, pointer);
                }, 0);
            } else {
            	$scope.exportNotFound = true;
            }
    	};

    	function showHTML(html, showCode, pointer) {
            if (showCode != null) {
                // if one day this becomes too annoying one may consider using an html parser and pick up the element with id toggleCode
                html = html
                    // legacy nbconvert<7.7, see https://github.com/jupyter/nbconvert/pull/2021/files#diff-5d5d0e216c4c7e2bdbdce10cc1bac7804d432ee61e1643be8f880cd422d14cd0R260
                    .replace('<a id=\'toggleCode\' onclick=\'toggleCodeVisibility()\'>Show code</a>', '')
                    // new format
                    .replace('<a id="toggleCode" onclick="toggleCodeVisibility()">Show code</a>', '');
                if (showCode) {
                    html = html.replace('<body class="hideCode">', '<body>');
                } else {
                    html = html.replace('<body>', '<body class="hideCode">');
                }
                if (pointer) {
                    html = html.replace('<body', '<body style="cursor: pointer;" onload="_load()"');
                }
            }
            html = html.replace('</body>', '<style>div.output_subarea { max-width: none; } ::-webkit-scrollbar { -webkit-appearance: none; width: 5px; height: 7px; } ::-webkit-scrollbar-thumb { border-radius: 4px; background-color: rgba(0,0,0,.4); box-shadow: 0 0 1px rgba(255,255,255,.5); }</style></body>');
            $scope.jupyterInsightContent = $sce.trustAsHtml(html);
    	};
    });

    app.directive('jupyterInsightTile', function($stateParams, $timeout, DataikuAPI, $controller, DashboardUtils, TileLoadingState){
        return {
            templateUrl: '/templates/dashboards/insights/jupyter/jupyter_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs){
            	$controller('JupyterInsightCommonController', { $scope: $scope });

                let html;

                $scope.loaded = false;
            	$scope.loading = false;
                $scope.error = null;

                $scope.load = function(resolve, reject) {
                	$scope.loading = true;
                	const loadingPromise = $scope.getLoadingPromise().noSpinner();
                	//any case write in iframe to display html
                	loadingPromise
                        .success(DashboardUtils.setLoaded.bind([$scope, resolve]))
                        .success(function(data) {
                            html = data.html; $scope.displayExport(html, !!$scope.tile.tileParams.showCode, $scope.tile.clickAction != 'DO_NOTHING');

                            if ($scope.tile.clickAction != 'DO_NOTHING') {
                                /*
                                 * The following code aims to handle configurable click behavior on tile
                                 * It is broken for two reasons:
                                 * * the sandboxing prevents accessing the contentWindow: https://app.shortcut.com/dataiku/story/144229/resolve-xss-in-dashboards-related-to-jupyter-nb-r-markdown-reports-static-insight-file
                                 * * the main-click was removed: https://github.com/dataiku/dip/pull/17980/files#diff-b9d7e6d55478e7e6c5f1783bf22f061af476569967d0bde471ed49a7c99a88a8L59
                                 * On click on body, redirect event to main-click link
                                 */
                                $timeout(function() {
                                    element.find('iframe')[0].contentWindow._load = function() {
                                        element.find('iframe').contents().find('body').on('click', function(evt) {
                                            if(evt.originalEvent.preventRecursionMarker === 'norec') {
                                                // Prevent event recursion : do not handle this event if we generated it!
                                                return;
                                            }
                                            const cloneEvent = document.createEvent('MouseEvents');
                                            cloneEvent.preventRecursionMarker = 'norec';
                                            const e = evt.originalEvent;
                                            cloneEvent.initMouseEvent(e.type, e.bubbles, e.cancelable, window, e.detail,
                                                e.screenX, e.screenY, e.clientX, e.clientY, e.ctrlKey, e.altKey, e.shiftKey,
                                                e.metaKey, e.button, e.relatedTarget);
                                            element.closest('.tile-wrapper').find('[main-click]')[0].dispatchEvent(cloneEvent);
                                            e.stopPropagation();
                                        });
                                    };
                                });
                            }
                        })
                        .error(DashboardUtils.setError.bind([$scope, reject]));
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }

                $scope.$watch('tile.tileParams.showCode', function(nv) {
                    if (nv == null || !$scope.loaded) {
                        return;
                    }
                    $scope.displayExport(html, !!nv);
                });
            }
        };
    });

    app.directive('jupyterInsightView', function($controller){
        return {
            templateUrl: '/templates/dashboards/insights/jupyter/jupyter_view.html',
            scope: {
                insight: '=',
                tileParams: '='
            },
            link: function($scope, element, attrs) {
                $controller('JupyterInsightCommonController', { $scope: $scope });

                const loadingPromise = $scope.getLoadingPromise();
            	//any case write in iframe to display html
            	loadingPromise.success(function(data) {
            		$scope.displayExport(data.html);
            	}).error(function(data, status, headers, config, statusText) {
                	setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                });

            }
        };
    });

    app.directive('jupyterInsightTileParams', function(){
        return {
            templateUrl: '/templates/dashboards/insights/jupyter/jupyter_tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs){
            }
        };
    });


    app.directive('jupyterInsightCreateForm', function(DataikuAPI, $filter, $stateParams){
        return {
            templateUrl: '/templates/dashboards/insights/jupyter/jupyter_create_form.html',
            scope: true,
            link: function($scope, element, attrs){

                $scope.hook.defaultName = 'Jupyter notebook';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = nv.label;
                });

            	$scope.insight.params.loadLast = true;

                $scope.facade = {
            		notebookSmartName : null,
                    availableExports : [],
                    createExport : $scope.canWriteProject()
                };

                $scope.setNotebook = function() {
                    if (!$scope.facade.notebookSmartName) {
                        return;
                    }
                	$scope.insight.params.notebookSmartName = $scope.facade.notebookSmartName;
                	$scope.facade.availableExports = $scope.notebookToExportsMap[$scope.facade.notebookSmartName];
                };

                $scope.$watch('facade.notebookSmartName', $scope.setNotebook);

                $scope.hook.beforeSave = function(resolve, reject) {
                	if ($scope.facade.createExport) {
                		DataikuAPI.jupyterNotebooks.export.create($stateParams.projectKey, $scope.insight.params.notebookSmartName)
                		.success(function(data) {
                			if (!$scope.insight.params.loadLast) {
                				$scope.insight.params.exportTimestamp = data.timestamp;
                			}
                			resolve();
                		})
                		.error(function(data, status, headers, config, statusText){
                        	reject(arguments);
                            });
                	} else {
                		resolve();
                	}
                };

                $scope.checkLoadLastAndTimestampConsistency = function() {
                	if (!$scope.insight.params.loadLast && !$scope.insight.params.exportTimestamp && !$scope.facade.createExport) {
                		$scope.newInsightForm.$setValidity('loadLastAndTimestampConsistency',false);
                	} else {
                		$scope.newInsightForm.$setValidity('loadLastAndTimestampConsistency',true);
                	}
                	return true;
                };

                $scope.formatDate = function(timestamp) {
                	return $filter('date')(timestamp, 'short');
                };

                $scope.resetTimestamp = function() {
                	$scope.insight.params.exportTimestamp = null;
                };

                DataikuAPI.jupyterNotebooks.mapNotebooksToExports($scope.insight.projectKey).success(function(data) {
                    $scope.notebookMap = data.first;
                    $scope.notebookToExportsMap = data.second;
                }).error($scope.hook.setErrorInModaleScope);
            }
        };
    });

    app.directive('jupyterInsightEdit', function($controller, DataikuAPI, $rootScope, Dialogs){
        return {
            templateUrl: '/templates/dashboards/insights/jupyter/jupyter_edit.html',
            scope: {
                insight: '='
            },
            link: function($scope, element, attrs) {
                $controller('JupyterInsightCommonController', { $scope: $scope });

                $scope.canWriteProject = $rootScope.topNav.isProjectAnalystRW;

                DataikuAPI.jupyterNotebooks.export.list($scope.insight.projectKey, $scope.insight.params.notebookSmartName).success(function(data) {
                    $scope.exports = data;
                });

                function refresh() {
                    if (!$scope.insight.params) {
                        return;
                    }
                    if (!$scope.insight.params.loadLast && !$scope.insight.params.exportTimestamp) {
                        $scope.insight.params.exportTimestamp = $scope.exports[0].timestamp;
                    }

                    const loadingPromise = $scope.getLoadingPromise();
                    //any case write in iframe to display html
                    loadingPromise.success(function(data) {
                        $scope.displayExport(data.html);
                    }).error(function(data, status, headers, config, statusText) {
                        setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                    });
                }

                $scope.$watch('insight.params', refresh, true);

                $scope.createNewExport = function() {
                    Dialogs.confirmPositive($scope, 'Export Jupyter notebook',
                        'Create a new export of this Jupyter notebook? Note that it will not rerun the code of this notebook '+
                        'and will use the last saved state. To rerun the notebook, go to the notebook or use a DSS scenario').then(function(){
                        DataikuAPI.jupyterNotebooks.export.create($scope.insight.projectKey, $scope.insight.params.notebookSmartName)
                            .success(function(data) {
                                if (!$scope.insight.params.loadLast) {
                                    $scope.insight.params.exportTimestamp = data.timestamp;
                                }
                                refresh();
                                $scope.exports.unshift(data);
                            }).error(function(data, status, headers, config, statusText) {
                                setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                            });
                    });
                };
            }
        };
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('ManagedFolderContentInsightHandler', {
        name: 'Managed folder',
        desc: 'Display content of a folder',
        icon: 'icon-folder-close-alt',
        color: 'managed-folder',

        getSourceId: function(insight) {
            return insight.params.folderSmartId;
        },
        sourceType: 'MANAGED_FOLDER',
        hasEditTab: false,
        defaultTileParams: {

        },
        getDefaultTileDimensions: function(insight) {
            if (insight && insight.params && insight.params.filePath && !insight.params.isDirectory) {
                return [12, 15];
            } else {
                return [24, 15];
            }
        }
    });

    app.controller('ManagedFolderContentViewCommon', function($scope, DataikuAPI, $stateParams, DashboardUtils, ActiveProjectKey) {
        $scope.resolvedFolder = resolveDatasetFullName($scope.insight.params.folderSmartId,
            $stateParams.projectKey || $scope.insight.projectKey);
        $scope.previewedItem = null;
        $scope.getPreview = function(resolve, reject, noSpinner) {
            const p = DataikuAPI.managedfolder.getForInsight(ActiveProjectKey.get(), $scope.resolvedFolder.projectKey, $scope.resolvedFolder.datasetName)
                .success(function(data) {
                    $scope.folder = data;
                    $scope.odb = data;

                    if ($scope.insight.params.filePath != null && !$scope.insight.params.isDirectory) {
                        const p = DataikuAPI.managedfolder.previewItem($scope.insight.projectKey, $scope.odb.projectKey, $scope.insight.params.folderSmartId, $scope.insight.params.filePath)
                            .success(function(data){
                                $scope.previewedItem = data;
                            })
                            .success(DashboardUtils.setLoaded.bind([$scope, resolve]))
                            .error(DashboardUtils.setError.bind([$scope, reject]))
                            .error(setErrorInScope.bind($scope));

                        if (noSpinner) {
                            p.noSpinner();
                        }
                    } else {
                        DashboardUtils.setLoaded.bind([$scope, resolve])();
                    }
                })
                .error(setErrorInScope.bind($scope))
                .error(DashboardUtils.setError.bind([$scope, reject]));

            if (noSpinner) {
                p.noSpinner();
            }
        };

        $scope.skinState = { itemSkins:[] }; // to placate the js in the directives, not to offer webapp views in tiles (make a webapp tile for that)

    });

    app.directive('managedFolderContentInsightTile', function($controller, TileLoadingState){
        return {
            templateUrl: '/templates/dashboards/insights/managed-folder_content/managed-folder_content_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs){
                $controller('ManagedFolderContentViewCommon', { $scope: $scope });

                $scope.loaded = false;
                $scope.error = null;

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    $scope.getPreview(resolve, reject, true);
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                $scope.$on('load-tile', $scope.load);


                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.directive('managedFolderContentInsightTileParams', function(){
        return {
            templateUrl: '/templates/dashboards/insights/managed-folder_content/managed-folder_content_tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs){
            }
        };
    });


    app.directive('managedFolderContentInsightCreateForm', function(DataikuAPI, ChartTypeChangeHandler){
        return {
            templateUrl: '/templates/dashboards/insights/managed-folder_content/managed-folder_content_create_form.html',
            scope: true,
            link: function($scope, element, attrs){

                function refreshFiles() {
                    $scope.files = [];
                    if ($scope.insight.params.singleFile && $scope.insight.params.folderSmartId) {
                        DataikuAPI.managedfolder.listFS($scope.insight.projectKey, $scope.insight.params.folderSmartId)
                            .success(function(data){
                                $scope.files = data.items;
                            })
                            .error($scope.hook.setErrorInModaleScope);
                    }
                }

                $scope.$watch('insight.params.singleFile', refreshFiles);
                $scope.$watch('insight.params.folderSmartId', refreshFiles);

                function updateDefaultName() {
                    if (!$scope.hook.sourceObject || !$scope.hook.sourceObject.label) {
                        $scope.hook.defaultName = 'Content of folder';
                    } else if ($scope.insight.params.filePath) {
                        $scope.hook.defaultName = 'File ' + $scope.insight.params.filePath + ' of ' + $scope.hook.sourceObject.label;
                    } else {
                        $scope.hook.defaultName = 'Content of ' + $scope.hook.sourceObject.label;
                    }
                }

                $scope.$watch('hook.sourceObject', updateDefaultName);
                $scope.$watch('insight.params.filePath', updateDefaultName);
            }
        };
    });



    app.directive('managedFolderContentInsightView', function($controller, $timeout){
        return {
            templateUrl: '/templates/dashboards/insights/managed-folder_content/managed-folder_content_view.html',
            scope: {
                insight: '='
            },
            link: function($scope, element, attrs){
                $controller('ManagedFolderContentViewCommon', { $scope: $scope });
                $scope.getPreview();
            }
        };
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('MetricsInsightHandler', {
        name: 'Metrics',
        desc: 'Meta data about your source',
        icon: 'icon-external-link',
        color: 'metrics',

        getSourceId: function(insight) {
            return insight.params.objectSmartId;
        },
        getSourceType: function(insight) {
            return insight.params.objectType;
        },
        hasEditTab: false,
        defaultTileParams: {
            displayMode: 'LAST_VALUE'
        },
        defaultTileDimensions: [6, 6]
    });

    app.directive('metricsInsightTile', function($controller, DashboardUtils, TileLoadingState){
        return {
            templateUrl: '/templates/dashboards/insights/metrics/metrics_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs){
                $controller('MetricsInsightsViewCommon', { $scope: $scope });

                $scope.loading = false;
                $scope.loaded = false;

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    $scope.loadHistory(
                        DashboardUtils.setLoaded.bind([$scope, resolve]),
                        DashboardUtils.setError.bind([$scope, reject])
                    );
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.directive('metricsInsightTileParams', function(){
        return {
            templateUrl: '/templates/dashboards/insights/metrics/metrics_tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs){
            }
        };
    });


    app.directive('metricsInsightCreateForm', function(DataikuAPI, $stateParams, MetricsUtils, StateUtils){
        return {
            templateUrl: '/templates/dashboards/insights/metrics/metrics_create_form.html',
            scope: true,
            link: function($scope, element, attrs){
                $scope.MetricsUtils = MetricsUtils;
                $scope.insight.params.objectType = 'DATASET';


                const apis = {
                    'DATASET': 'datasets',
                    'SAVED_MODEL': 'savedmodels',
                    'MANAGED_FOLDER': 'managedfolder',
                    'PROJECT' : 'projects',
                    'MODEL_EVALUATION_STORE' : 'modelevaluationstores'
                };

                $scope.hook.sourceTypes = Object.keys(apis);

                function objectIsSeleted(){
                    $scope.computedMetrics = null;
                    $scope.selectedMetric = null;
                    $scope.insight.params.metricId = null;
                    DataikuAPI[apis[$scope.insight.params.objectType]].listComputedMetrics($stateParams.projectKey, $scope.insight.params.objectSmartId)
                        .success(function(data) {
                            $scope.computedMetrics = data.metrics.filter(function(m) {
                                return m.partitionsWithValue.length > 0;
                            });
                        })
                        .error($scope.hook.setErrorInModaleScope);
                    updateName();
                }

                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !$scope.insight.params.objectSmartId) {
                        return;
                    }
                    objectIsSeleted();
                });

                $scope.$watch('insight.params.objectType', function(nv, ov) {
                    if (nv === 'PROJECT') {
                        objectIsSeleted();
                    }
                });

                $scope.$watch('selectedMetric', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.insight.params.metricId = nv.metric.id;
                    updateName();
                });

                $scope.$watch('insight.params.metricId', updateName);

                function updateName() {
                    if ($scope.selectedMetric) {
                        $scope.hook.defaultName = MetricsUtils.getMetricDisplayName($scope.selectedMetric) + ' on ' + $scope.hook.sourceObject.label;
                    } else if ($scope.hook.sourceObject && $scope.hook.sourceObject.label) {
                        $scope.hook.defaultName = 'Metric of ' + $scope.hook.sourceObject.label;
                    } else {
                        $scope.hook.defaultName = 'Metric on object';
                    }
                }

                $scope.getMetricsSettingsUrl = function() {
                    if ($scope.insight.params.objectType && $scope.insight.params.objectSmartId) {
                        switch ($scope.insight.params.objectType) {
                            case 'DATASET' :
                                return StateUtils.href.dataset($scope.insight.params.objectSmartId, $stateParams.projectKey).replace('explore/', 'status/settings/');
                            case 'SAVED_MODEL':
                                return StateUtils.href.savedModel($scope.insight.params.objectSmartId, $stateParams.projectKey).replace('versions/', 'settings/#status-checks');
                            case 'MANAGED_FOLDER':
                                return StateUtils.href.managedFolder($scope.insight.params.objectSmartId, $stateParams.projectKey).replace('view/', 'status/settings');
                            case 'MODEL_EVALUATION_STORE':
                                return StateUtils.href.modelEvaluationStore($scope.insight.params.objectSmartId, $stateParams.projectKey).replace('evaluations/', 'settings/');
                            default:
                                break;
                        }
                    }
                };
            }
        };
    });

    app.controller('MetricsInsightsViewCommon', function($scope, DataikuAPI, $stateParams, MetricsUtils) {
        $scope.resolvedObject = resolveObjectSmartId($scope.insight.params.objectSmartId, $stateParams.projectKey);
        $scope.loadHistory = function(resolve, reject) {
            DataikuAPI.metrics.getComputedMetricWithHistory($scope.resolvedObject.projectKey, $scope.insight.params.objectType, $scope.resolvedObject.id, null, $scope.insight.params.metricId)
                .noSpinner()
                .success(function(data) {
                    $scope.metric = data.metric;
                    $scope.history = data.history;
                    $scope.fullRange = MetricsUtils.fixUpRange({ from: data.history.from, to: data.history.to });
                    $scope.selectedRange = angular.copy($scope.fullRange);
                    MetricsUtils.fixupDisplayType($scope.history);
                    if (typeof(resolve)==='function') {
                        resolve();
                    }
                })
                .error(function(data, status, headers, config, statusText) {
                    setErrorInScope.bind($scope);
                    if (typeof(reject)==='function') {
                        reject(data, status, headers, config, statusText);
                    }
                });
        };

        $scope.brushChanged = function() {
            $scope.$apply();
        };
    });

    app.directive('metricsInsightView', function($controller){
        return {
            templateUrl: '/templates/dashboards/insights/metrics/metrics_view.html',
            scope: {
                insight: '='
            },
            link: function($scope, element, attrs){
                $controller('MetricsInsightsViewCommon', { $scope: $scope });
                $scope.loadHistory();
            }
        };
    });

    app.directive('metricsInsightEdit', function($controller, DataikuAPI, $rootScope) {
        return {
            templateUrl: '/templates/dashboards/insights/metrics/metrics_edit.html',
            scope: {
                insight: '='
            },
            link: function($scope, element, attrs){
                $controller('ChartInsightViewCommon', { $scope: $scope });

                $scope.currentInsight = $scope.insight;

                $scope.bigChart = false;
                $scope.saveChart = function() {};

                $scope.saveChart = function(){
                    DataikuAPI.dashboards.insights.save($scope.insight)
                        .error(setErrorInScope.bind($scope))
                        .success(function() {});
                };

                $rootScope.$on('chartSamplingChanged', function() {
                    $scope.summary = null;
                    $scope.fetchColumnsSummary();
                    $scope.saveChart();
                });

                $scope.fetchColumnsSummary();
            }
        };
    });

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('ModelEvaluationReportInsightHandler', {
        name: 'Model evaluation report',
        desc: 'Full report of a model evaluation',
        icon: 'icon-model-evaluation-store',
        color: 'model-evaluation-store',
        getSourceId: function(insight) {
            return insight.params.mesSmartId;
        },
        sourceType: 'MODEL_EVALUATION_STORE',
        hasEditTab: false,
        defaultTileParams: {
            displayMode: 'summary'
        },
        defaultTileDimensions: [24, 12]
    });

    app.controller('ModelEvaluationReportViewCommon', function($scope, DataikuAPI, $controller, FullModelLikeIdUtils, WebAppsService, $state, $stateParams) {
        $scope.readOnly = true;
        $scope.noUrlChange = true;

        $scope.isClassicalPrediction = function() {
            if (!$scope.modelData) {
                return false;
            }
            return ['BINARY_CLASSIFICATION', 'REGRESSION', 'MULTICLASS'].includes($scope.modelData.coreParams.prediction_type);
        };

        $scope.getModelEvaluation = function(onLoadError) {
            const p = DataikuAPI.modelevaluationstores
                .get($scope.insight.projectKey, $scope.insight.params.mesSmartId)
                .success(function(data) {
                    $scope.modelEvaluationStore = data;
                    $scope.readOnly = true;
                    $scope.uiState = {};
                    $scope.noSetLoc = true;
                    $scope.mesContext = {};
                    $scope.versionsContext = {};
                    const getDetailsP = DataikuAPI.modelevaluationstores
                        .getLatestEvaluationDetails($scope.insight.projectKey, $scope.insight.params.mesSmartId)
                        .success(function(data) {
                            /*
                             * In the case of dashboards, the state does not contain the mesId nor
                             * the evaluationId (and you can see that it’s not in the URL), so we have
                             * to rely on the scope in those cases
                             */
                            $scope.$modelEvaluation = data;
                            $scope.mesId = $scope.insight.params.mesSmartId;
                            $scope.evaluationId = data.evaluation.ref.evaluationId;
                            $scope.modelData = data.details;
                            const isLLMEvaluation = $scope.$modelEvaluation.evaluation && $scope.$modelEvaluation.evaluation.type === 'llm';
                            if (isLLMEvaluation) {
                                $controller('_LLMPredictionModelReportController', { $scope });
                            } else {
                                $controller('_PredictionModelReportController', { $scope });
                            }
                            $controller('ModelEvaluationStoreBaseEvaluationController', { $scope });
                        })
                        .error(setErrorInScope.bind($scope));
                    if ($scope.noSpinner) {
                        getDetailsP.noSpinner();
                    }
                })
                .error(function(data, status, headers, config, statusText) {
                    if (typeof onLoadError === 'function') {
                        onLoadError(data, status, headers, config, statusText);
                    }
                });

            if ($scope.noSpinner) {
                p.noSpinner();
            }
        };
    });

    app.directive('modelEvaluationReportInsightTile', function($controller, DashboardUtils, TileLoadingState, SavedModelsService) {
        return {
            templateUrl: '/templates/dashboards/insights/model-evaluation_report/model-evaluation_report_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope) {
                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    $scope.onLoadSuccess = DashboardUtils.setLoaded.bind([$scope, resolve]);
                    $scope.onLoadError = DashboardUtils.setError.bind([$scope, reject]);
                    $scope.noSpinner = true;
                    $scope.getModelEvaluation($scope.onLoadError);
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                $controller('ModelEvaluationReportViewCommon', { $scope: $scope });

                $scope.$watch('modelData', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.insight.$modelData = nv;
                });

                $scope.$watch('evaluationDetails', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.insight.$evaluationDetails = nv;
                    $scope.insight.$isLLMEvaluation = nv.evaluation && nv.evaluation.type === 'llm';
                });

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.controller('ModelEvaluationReportInsightController', function($scope, $controller, $stateParams, DataikuAPI, $state, MLModelsUIRouterStates, GoToStateNameSuffixIfBase, ComputeTransitionStateName) {
        const baseStateName = 'insight.view.modelevaluations';

        $controller('ModelEvaluationReportViewCommon', { $scope: $scope });

        $scope.isLLMEvaluation = function() {
            return $scope.modelEvaluation && $scope.modelEvaluation.type === 'llm';
        };

        const getSummarySuffix = () => {
            return MLModelsUIRouterStates.getPredictionReportSummaryTab(false, $scope.isLLMEvaluation());
        };

        const updateUIRouterStateWithTarget = ({ stateName, stateParams }) => {
            const transitionStateName = ComputeTransitionStateName(baseStateName, stateName);

            if (transitionStateName || !angular.equals($state.current.params, stateParams)) {
                $state.go(transitionStateName, stateParams, { location: 'replace' });
            }
        };

        const getUIRouterStateNameTarget = () => {
            if (!$state.current.name.includes(baseStateName)) {
                return baseStateName;
            }

            // if stateName is 'insight.view.savedmodels' redirection on 'insight.view.savedmodels.summary'
            if ($state.current.name.endsWith(baseStateName)) {
                return `${baseStateName}.prediction.${getSummarySuffix()}`;
            }

            return $state.current.name.slice($state.current.name.lastIndexOf(baseStateName));
        };

        const getUIRouterStateNameAndParamsTargetFromTile = (tile) => {
            const pane = MLModelsUIRouterStates.dashboardTileToSavedModelPane(tile, false, $scope);
            return {
                stateName: `${baseStateName}.prediction.${pane.route}`,
                stateParams: pane.routeParams
            };
        };

        $scope.$on('$stateChangeStart', function(event, toState, toParams, fromState, fromParams) {
            if (!$scope.modelData) {
                return;
            }
            GoToStateNameSuffixIfBase(
                $state,
                toState,
                toParams,
                fromState,
                fromParams,
                '.insight.view',
                `modelevaluations.prediction.${MLModelsUIRouterStates.getPredictionReportSummaryTab(false, $scope.isLLMEvaluation())}`,
                event
            );

        });

        $scope.getModelEvaluation(setErrorInScope.bind($scope));
        $scope.$watch('evaluationDetails', function(nv, ov) {
            if (!nv) {
                return;
            }

            if ($state.current.name.endsWith(baseStateName)) {
                updateUIRouterStateWithTarget({ stateName: getUIRouterStateNameTarget() });
            }

            if ($scope.originDashboardStateParams) {
                DataikuAPI.dashboards.getFullInfo($stateParams.projectKey, $scope.originDashboardStateParams.dashboardId).success(function(data) {
                    const tile = data.dashboard.pages
                        .find(page => page.id === $scope.originDashboardStateParams.pageId)
                        .grid.tiles.find(tile => tile.insightId === $stateParams.insightId);
                    updateUIRouterStateWithTarget(getUIRouterStateNameAndParamsTargetFromTile(tile));
                });
            }
        });

        $scope.onLoadError = function(data, status, headers, config, statusText) {
            setErrorInScope.bind($scope)(data, status, headers, config, statusText);
        };
    });

    app.directive('modelEvaluationReportInsightView', function($controller, $stateParams, DataikuAPI, $state) {
        return {
            scope: true,
            link: function($scope, element, attrs) {
                if ($state.current.name.endsWith('.insight.view')) {
                    $state.go('.modelevaluations', null, { location: 'replace' });
                }
            }
        };
    });

    app.directive('llmModelEvaluationReportInsightTileParams', function($controller, $timeout, DataikuAPI, FullModelLikeIdUtils, $q, ModelDataUtils, $stateParams) {
        return {
            templateUrl: '/templates/dashboards/insights/model-evaluation_report/llm-model-evaluation_report_tile_params.html',
            scope: {
                tileParams: '=',
                insight: '='
            },
            link: function($scope, $element, attrs) {
                $scope.ModelDataUtils = ModelDataUtils;

                $scope.$watch('insight.$evaluationDetails', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.evaluationDetails = nv;

                    $controller('ModelEvaluationReportViewCommon', { $scope: $scope });

                    $controller('_LLMPredictionModelReportController', { $scope: $scope });

                    $timeout(function() {
                        $element.find('.view-select').selectpicker('refresh');
                    });

                    $scope.onDisplayModeExtendedChange = function() {
                        if (!$scope.tileParams || !$scope.tileParams.$displayModeExtended) {
                            return;
                        }
                        const displayModeExtended = $scope.tileParams.$displayModeExtended;
                        $scope.tileParams.displayMode = displayModeExtended;
                    };

                    const deregister = $scope.$watch('tileParams', function(nv) {
                        if (!nv) {
                            return;
                        }
                        $scope.tileParams.$displayModeExtended = $scope.tileParams.displayMode || 'llm-summary';
                        deregister();
                    });
                });
            }
        };
    });


    app.directive('modelEvaluationReportInsightTileParams', function($controller, $timeout, DataikuAPI, FullModelLikeIdUtils, $q, ModelDataUtils, $stateParams) {
        return {
            templateUrl: '/templates/dashboards/insights/model-evaluation_report/model-evaluation_report_tile_params.html',
            scope: {
                tileParams: '=',
                insight: '='
            },
            link: function($scope, $element, attrs) {
                $scope.ModelDataUtils = ModelDataUtils;

                const setDefaultOverridesMetricsDisplayMode = function() {
                    $scope.tileParams.advancedOptions = $scope.tileParams.advancedOptions || {};
                    $scope.tileParams.advancedOptions.overridesMetrics = $scope.tileParams.advancedOptions.overridesMetrics || {};
                    $scope.tileParams.advancedOptions.overridesMetrics.displayMode =
                        $scope.tileParams.advancedOptions.overridesMetrics.displayMode || 'sankey';
                };

                $scope.$watch('insight.$modelData', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.modelData = nv;
                    $scope.fullModelId = $scope.insight.$modelData.fullModelId;

                    $controller('ModelEvaluationReportViewCommon', { $scope: $scope });

                    if (!($scope.tileParams.advancedOptions && $scope.tileParams.advancedOptions.interactiveScoring) && $scope.fullModelId) {
                        Promise.all([
                            DataikuAPI.ml.prediction.getColumnImportance($scope.fullModelId),
                            DataikuAPI.ml.prediction.getSplitDesc($scope.fullModelId),
                            DataikuAPI.ml.prediction.getPreparationScript($scope.fullModelId),
                            DataikuAPI.ml.prediction.getInputDatasetSchema($scope.fullModelId).catch(e => e)
                        ])
                            .then(([columnImportanceResp, splitDescResp, preparationScriptResp, inputDatasetSchemaResp]) => {
                                let featuresOrder;
                                if (columnImportanceResp.data) {
                                    const importances = columnImportanceResp.data.importances;
                                    const columns = columnImportanceResp.data.columns;
                                    featuresOrder = columns.sort((c1, c2) => importances[columns.indexOf(c2)] - importances[columns.indexOf(c1)]);
                                } else {
                                    const perFeature = $scope.modelData.preprocessing.per_feature;
                                    const inputColumns = Object.keys(perFeature).filter(featureName => perFeature[featureName].role === 'INPUT');
                                    featuresOrder = splitDescResp.data.schema.columns.map(c => c.name).filter(c => inputColumns.includes(c));
                                }
                                const hasPreparationSteps = preparationScriptResp.data.steps.some(step => !step.disabled);
                                if (hasPreparationSteps) {
                                    if (inputDatasetSchemaResp.data.columns) {
                                        const preScriptFeatures = inputDatasetSchemaResp.data.columns.map(col => col.name);
                                        if (columnImportanceResp.data) {
                                            featuresOrder.push(...preScriptFeatures.filter(f => !featuresOrder.includes(f)));
                                        } else {
                                            featuresOrder = [...preScriptFeatures, ...featuresOrder.filter(f => !preScriptFeatures.includes(f))];
                                        }
                                    } else if (inputDatasetSchemaResp.status !== 404) {
                                        setErrorInScope.call($scope, inputDatasetSchemaResp);
                                    }
                                }
                                $scope.tileParams.advancedOptions = {
                                    ...$scope.tileParams.advancedOptions,
                                    interactiveScoring: {
                                        featuresOrder
                                    }
                                };
                            })
                            .catch(setErrorInScope.bind($scope));
                    }

                    $controller('_PredictionModelReportController', { $scope: $scope });

                    $timeout(function() {
                        $element.find('.view-select').selectpicker('refresh');
                    });


                    $scope.onDisplayModeExtendedChange = function() {
                        if (!$scope.tileParams || !$scope.tileParams.$displayModeExtended) {
                            return;
                        }
                        const displayModeExtended = $scope.tileParams.$displayModeExtended;
                        $scope.tileParams.displayMode = displayModeExtended;
                        if (displayModeExtended === 'tabular-overrides_metrics') {
                            setDefaultOverridesMetricsDisplayMode();
                        }
                    };
                    const deregister = $scope.$watch('tileParams', function(nv) {
                        if (!nv) {
                            return;
                        }
                        $scope.tileParams.$displayModeExtended = $scope.tileParams.displayMode || 'summary';
                        deregister();
                    });

                    if ($scope.tileParams.displayMode === 'tabular-overrides_metrics') {
                        setDefaultOverridesMetricsDisplayMode();
                    }
                    setDefaultImportanceDisplayMode($scope.hasVariableImportance(), $scope.hasGlobalExplanations());
                });

                if ($scope.tileParams.displayMode === 'tabular-overrides_metrics') {
                    setDefaultOverridesMetricsDisplayMode();
                }

                const setDefaultImportanceDisplayMode = function(hasVariableImportance, hasGlobalExplanations) {
                    let importanceDisplayMode = 'globalExplanations';
                    const tileParamsAdvancedOptions = $scope.tileParams.advancedOptions;
                    if (hasVariableImportance) {
                        if (!tileParamsAdvancedOptions || !tileParamsAdvancedOptions.featureImportance) {
                            if (!hasGlobalExplanations) {
                                importanceDisplayMode = 'variableImportance';
                            }
                        } else {
                            importanceDisplayMode = tileParamsAdvancedOptions.featureImportance.importanceDisplayMode;
                        }
                    }
                    const advancedOptions = $scope.tileParams.advancedOptions || { featureImportance: {} };
                    $scope.tileParams.advancedOptions = {
                        ...advancedOptions,
                        featureImportance: {
                            ...advancedOptions.featureImportance,
                            importanceDisplayMode
                        }
                    };
                    setDefaultGraphType();
                };

                const setDefaultGraphType = function() {
                    let graphType = 'absoluteFeatureImportance';
                    const tileParamsAdvancedOptions = $scope.tileParams.advancedOptions;
                    if (tileParamsAdvancedOptions && tileParamsAdvancedOptions.featureImportance.graphType) {
                        graphType = tileParamsAdvancedOptions.featureImportance.graphType;
                    }

                    $scope.tileParams.advancedOptions = {
                        ...tileParamsAdvancedOptions,
                        featureImportance: {
                            ...tileParamsAdvancedOptions.featureImportance,
                            graphType
                        }
                    };
                };
            }
        };
    });

    app.directive('modelEvaluationReportInsightCreateForm', function(DataikuAPI) {
        return {
            templateUrl: '/templates/dashboards/insights/model-evaluation_report/model-evaluation_report_create_form.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.hook.defaultName = 'Model Evaluation table';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = nv.label + ' table';
                });
            }
        };
    });

    app.filter('formatCustomTrainTestSplitDateByTimestep', function ($filter) {
        return function (input, timestep) {
            if (!input) return '';
            const date = new Date(input);

            switch (timestep) {
                case 'YEAR':
                    return $filter('date')(date, 'yyyy');
                case 'HALF_YEAR':
                case 'QUARTER':
                case 'MONTH':
                case 'WEEK':
                case 'BUSINESS_DAY':
                case 'DAY':
                    return $filter('date')(date, 'yyyy-MM-dd');
                case 'HOUR':
                case 'MINUTE':
                    return $filter('date')(date, 'yyyy-MM-dd HH:mm');
                case 'SECOND':
                    return $filter('date')(date, 'yyyy-MM-dd HH:mm:ss');
                case 'MILLISECOND':
                default:
                    return $filter('date')(date, 'yyyy-MM-dd HH:mm:ss.sss');
            }
        };
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('ProjectActivityInsightHandler', {
        name: 'Project activity',
        desc: 'Activity charts of a project',
        icon: 'dku-icon-arrow-trending-24',
        color: 'project',

        getSourceId: function(insight) {
            return insight.params.projectKey;
        },
        sourceType: 'PROJECT',
        hasEditTab: false,
        defaultTileParams: {
            displayMode: 'CONTRIBUTION_CHARTS',
            summaryChart: 'commits',
            contributorsChart: 'commits',
            timeSpan: 'year'
        }
    });

    app.controller('ProjectActivityInsightViewCommon', function($scope, DataikuAPI, $stateParams, MetricsUtils) {
        $scope.resolvedObject = resolveObjectSmartId($scope.insight.params.objectSmartId, $stateParams.projectKey);
        $scope.loadHistory = function(resolve, reject) {
            DataikuAPI.metrics.getComputedMetricWithHistory($scope.resolvedObject.projectKey, $scope.insight.params.objectType, $scope.resolvedObject.id, null, $scope.insight.params.metricId)
                .noSpinner()
                .success(function(data) {
                    $scope.metric = data.metric;
                    $scope.history = data.history;
                    $scope.fullRange = { from: data.history.from, to: data.history.to };
                    $scope.selectedRange = { from: data.history.from, to: data.history.to };
                    MetricsUtils.fixupDisplayType($scope.history);
                    if (typeof(resolve)==='function') {
                        resolve();
                    }
                })
                .error(function() {
                    setErrorInScope.bind($scope);
                    if (typeof(reject)==='function') {
                        reject();
                    }
                });
        };

        $scope.brushChanged = function() {
            $scope.$apply();
        };
    });


    app.directive('projectActivityInsightTile', function($controller, DataikuAPI, DashboardUtils, TileLoadingState){
        return {
            templateUrl: '/templates/dashboards/insights/project_activity/project_activity_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs){
                $controller('ProjectActivityViewCommonController', { $scope: $scope });

                $scope.loading = false;
                $scope.loaded = false;

                $scope.load = function(resolve, reject) {
                	$scope.loading = true;
                    DataikuAPI.projects.activity.getActivitySummary($scope.insight.params.projectKey, $scope.tile.tileParams.timeSpan || 'year')
                        .success($scope.prepareData)
                        .success(DashboardUtils.setLoaded.bind([$scope, resolve]))
                        .error(DashboardUtils.setError.bind([$scope, reject]))
                        .noSpinner();
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;


                $scope.$watch('tile.tileParams.timeSpan', function(nv, ov) {
                    if (nv && ov && nv != ov) {
                        DataikuAPI.projects.activity.getActivitySummary($scope.insight.params.projectKey, $scope.tile.tileParams.timeSpan || 'year')
                            .success($scope.prepareData)
                            .error(DashboardUtils.setError.bind([$scope]))
                            .noSpinner();
                    }
                });

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.directive('projectActivityInsightTileParams', function(){
        return {
            templateUrl: '/templates/dashboards/insights/project_activity/project_activity_tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs){
            }
        };
    });


    app.directive('projectActivityInsightCreateForm', function(DataikuAPI, $stateParams){
        return {
            templateUrl: '/templates/dashboards/insights/project_activity/project_activity_create_form.html',
            scope: true,
            link: function($scope, element, attrs){
                $scope.$watch('hook.sourceObject', updateName);
                $scope.hook.noReaderAuth = true;

                function updateName() {
                    if ($scope.hook.sourceObject && $scope.hook.sourceObject.label) {
                        $scope.hook.defaultName = 'Activity of ' + $scope.hook.sourceObject.label;
                    } else {
                        $scope.hook.defaultName = 'Activity of project';
                    }
                }
            }
        };
    });

    app.directive('projectActivityInsightView', function($controller, DataikuAPI){
        return {
            templateUrl: '/templates/dashboards/insights/project_activity/project_activity_view.html',
            scope: true,
            link: function($scope, element, attrs){
                $controller('ProjectActivityViewCommonController', { $scope: $scope });
                $scope.uiState = {
                    settingsPane : 'summary',
                    summaryChart: 'commits',
                    contributorsChart: 'commits',
                    timeSpan: 'year'
                };

                $scope.$watch('uiState.timeSpan', function(timeSpan) {
                    if (!timeSpan) {
                        return;
                    }
                    DataikuAPI.projects.activity.getActivitySummary($scope.insight.params.projectKey, timeSpan)
                        .success($scope.prepareData)
                        .error(setErrorInScope.bind($scope));
                });
            }
        };
    });

    app.directive('projectActivityInsightEdit', function($controller, DataikuAPI, $rootScope) {
        return {
            templateUrl: '/templates/dashboards/insights/project_activity/project_activity_edit.html',
            scope: {
                insight: '='
            },
            link: function($scope, element, attrs){
                $controller('ChartInsightViewCommon', { $scope: $scope });

                $scope.currentInsight = $scope.insight;

                $scope.bigChart = false;
                $scope.saveChart = function() {};

                $scope.saveChart = function(){
                    DataikuAPI.dashboards.insights.save($scope.insight)
                        .error(setErrorInScope.bind($scope))
                        .success(function() {});
                };

                $rootScope.$on('chartSamplingChanged', function() {
                    $scope.summary = null;
                    $scope.fetchColumnsSummary();
                    $scope.saveChart();
                });

                $scope.fetchColumnsSummary();
            }
        };
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    function getAvailablePreviewFormats(snapshot, formats) {
        if (!snapshot || !snapshot.availableFormats) {
            return [];
        }
        return formats.filter(f => snapshot.availableFormats.includes(f.name));
    }

    app.constant('ReportInsightHandler', {
        name: 'Report',
        desc: 'Display report',
        icon: 'icon-DKU_rmd',
        color: 'notebook',

        getSourceId: function(insight) {
            return insight.params.reportSmartId;
        },
        sourceType: 'REPORT',
        hasEditTab: true,
        defaultTileParams: {

        },
        defaultTileShowTitleMode: 'NO',
        defaultTileDimensions: [12, 15]
    });


    app.controller('ReportSnapshotCommonController', function($scope, $stateParams, $sce, $timeout, $q, Assert, DataikuAPI, RMARKDOWN_PREVIEW_OUTPUT_FORMATS) {
        $scope.resolvedReport = resolveObjectSmartId($scope.insight.params.reportSmartId, $stateParams.projectKey);


        $scope.getDisplaySnapshotPromise = function(snapshot) {
            if (!snapshot.timestamp) {
                return;
            }
            let format = $scope.insight.params.viewFormat;
            if (format == null && snapshot.availableFormats && snapshot.availableFormats.length) { // backwards compatibility
                const availablePreviewFormats = RMARKDOWN_PREVIEW_OUTPUT_FORMATS.filter(f => snapshot.availableFormats.includes(f.name));
                if (availablePreviewFormats.length) {
                    format = availablePreviewFormats[0].name;
                }
            }
            return DataikuAPI.reports.snapshots.view(snapshot.projectKey, snapshot.reportId, snapshot.timestamp, format);
        };

        $scope.setReportInsightContent = data => {
            $scope.reportInsightContent = data;
        };

        $scope.getLoadingPromise = function() {
            const params = $scope.insight.params;
            const t = params.loadLast ? 0 : params.exportTimestamp;
            return DataikuAPI.reports.snapshots.get($scope.insight.projectKey, params.reportSmartId, t);
        };

    });



    app.directive('reportInsightTile', function($controller, $timeout, DashboardUtils, TileLoadingState){
        return {
            templateUrl: '/templates/dashboards/insights/code-reports/report_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '=',
                editable: '='
            },
            link: function($scope, element, attrs){
                $scope.element = element;

                $controller('ReportSnapshotCommonController', { $scope: $scope });

                $scope.loaded = false;
                $scope.loading = false;
                $scope.error = null;

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    const loadingPromise = $scope.getLoadingPromise().noSpinner();
                    //any case write in iframe to display html
                    loadingPromise
                        .then(
                            resp => $scope.getDisplaySnapshotPromise(resp.data).noSpinner()
                        ).then(resp => {
                            DashboardUtils.setLoaded.bind([$scope, resolve])();
                            $timeout(() => {
                                $scope.setReportInsightContent(resp.data);
                            });
                            if ($scope.tile.clickAction != 'DO_NOTHING') {
                            /*
                             * The following code aims to handle configurable click behavior on tile
                             * It is broken for three reasons:
                             * * the sandboxing prevents accessing the contentWindow: https://github.com/dataiku/dip/pull/8767
                             * * the _load function is never used, it comes from a copy/paste from 1st commit from jupyter tile: https://github.com/dataiku/dip/pull/7378/files#diff-2c594ff2b3dba0908a9f37a66c88b9f57024a37ffc97e2311951d000e7502d9fR58
                             * * the main-click was removed: https://github.com/dataiku/dip/pull/17980/files#diff-b9d7e6d55478e7e6c5f1783bf22f061af476569967d0bde471ed49a7c99a88a8L59
                             * On click on body, redirect event to main-click link
                             */
                                $timeout(function() {
                                    element.find('iframe')[0].contentWindow._load = function() {
                                        element.find('iframe').contents().find('body').on('click', function(evt) {
                                            if(evt.originalEvent.preventRecursionMarker === 'norec') {
                                            // Prevent event recursion : do not handle this event if we generated it!
                                                return;
                                            }
                                            const cloneEvent = document.createEvent('MouseEvents');
                                            cloneEvent.preventRecursionMarker = 'norec';
                                            const e = evt.originalEvent;
                                            cloneEvent.initMouseEvent(e.type, e.bubbles, e.cancelable, window, e.detail,
                                                e.screenX, e.screenY, e.clientX, e.clientY, e.ctrlKey, e.altKey, e.shiftKey,
                                                e.metaKey, e.button, e.relatedTarget);
                                            element.closest('.tile-wrapper').find('[main-click]')[0].dispatchEvent(cloneEvent);
                                            e.stopPropagation();
                                        });
                                    };
                                });
                            }
                        }, DashboardUtils.setError.bind([$scope, reject]));
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.directive('reportInsightView', function($controller) {
        return {
            templateUrl: '/templates/dashboards/insights/code-reports/report_view.html',
            scope: {
                insight: '='
            },
            link: function($scope, element, attrs) {
                $scope.element = element;
                $controller('ReportSnapshotCommonController', { $scope: $scope });

                $scope.getLoadingPromise().then(
                    resp => $scope.getDisplaySnapshotPromise(resp.data)
                ).then(
                    resp => $scope.setReportInsightContent(resp.data)
                ).catch(
                    setErrorInScope.bind($scope)
                );
            }
        };
    });


    app.directive('reportInsightTileParams', function(){
        return {
            templateUrl: '/templates/dashboards/insights/code-reports/report_tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs){
                //No tile params
            }
        };
    });


    app.directive('reportInsightCreateForm', function($stateParams, $filter, DataikuAPI, RMARKDOWN_PREVIEW_OUTPUT_FORMATS) {
        return {
            templateUrl: '/templates/dashboards/insights/code-reports/report_create_form.html',
            scope: true,
            link: function($scope) {
                let snapshotsByReportSmartId = {};
                $scope.hook.defaultName = 'Rmarkdown report';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = nv.label;
                });

                $scope.insight.params.loadLast = true;

                $scope.facade = {
                    reportSmartId: null,
                    availableSnapshots: [],
                    createSnapshot: $scope.canWriteProject()
                };

                function setReport() {
                    if (!$scope.facade.reportSmartId) {
                        return;
                    }
                    $scope.insight.params.reportSmartId = $scope.facade.reportSmartId;
                    $scope.facade.availableSnapshots = snapshotsByReportSmartId[$scope.facade.reportSmartId];
                };

                $scope.$watch('facade.reportSmartId', setReport);

                $scope.hook.beforeSave = function(resolve, reject) {
                    const snapshot = $scope.facade.snapshot;
                    const params = $scope.insight.params;

                    if (snapshot) {
                        const formatNames = getAvailablePreviewFormats(snapshot, RMARKDOWN_PREVIEW_OUTPUT_FORMATS).map(_ => _.name);
                        if (formatNames.length && !formatNames.includes(params.viewFormat)) {
                            params.viewFormat = formatNames[0];
                        }
                        $scope.insight.params.exportTimestamp = snapshot.timestamp;
                    }
                    if ($scope.facade.createSnapshot) {
                        DataikuAPI.reports.snapshots.create($stateParams.projectKey, $scope.insight.params.reportSmartId)
                            .success(function(snapshot) {
                                if (!$scope.insight.params.loadLast) {
                                    const formatNames = getAvailablePreviewFormats(snapshot, RMARKDOWN_PREVIEW_OUTPUT_FORMATS).map(_ => _.name);
                                    if (formatNames.length && !formatNames.includes(params.viewFormat)) {
                                        params.viewFormat = formatNames[0];
                                    }
                                    $scope.insight.params.exportTimestamp = snapshot.timestamp;
                                }
                                resolve();
                            })
                            .error(function() {
                                reject(arguments);
                            });
                    } else {
                        resolve();
                    }
                };

                $scope.checkLoadLastAndTimestampConsistency = function() {
                    if (!$scope.insight.params.loadLast && !$scope.facade.snapshot && !$scope.facade.createSnapshot) {
                        $scope.newInsightForm.$setValidity('loadLastAndTimestampConsistency', false);
                    } else {
                        $scope.newInsightForm.$setValidity('loadLastAndTimestampConsistency', true);
                    }
                    return true;
                };

                $scope.formatDate = function(timestamp) {
                    return $filter('date')(timestamp, 'short');
                };

                $scope.resetTimestamp = function() {
                    $scope.insight.params.exportTimestamp = null;
                };

                DataikuAPI.reports.snapshots.listForAll($scope.insight.projectKey).success(function(data) {
                    snapshotsByReportSmartId = data;
                }).error($scope.hook.setErrorInModaleScope);
            }
        };
    });


    app.directive('reportInsightEdit', function($controller, DataikuAPI, FutureProgressModal, $rootScope, Dialogs, RMARKDOWN_PREVIEW_OUTPUT_FORMATS) {
        return {
            templateUrl: '/templates/dashboards/insights/code-reports/report_edit.html',
            scope: {
                insight: '='
            },
            link: function($scope, element) {
                $controller('ReportSnapshotCommonController', { $scope });

                $scope.canWriteProject = $rootScope.topNav.isProjectAnalystRW;

                DataikuAPI.reports.snapshots.list($scope.insight.projectKey, $scope.insight.params.reportSmartId)
                    .success(function(snapshots) {
                        $scope.snapshots = snapshots;
                        refresh();
                    })
                    .error(setErrorInScope.bind($scope));

                function refresh() {
                    const params = $scope.insight.params;
                    if (!params) {
                        return;
                    }
                    if (!params.loadLast && !params.exportTimestamp && $scope.snapshots && $scope.snapshots.length) {
                        params.exportTimestamp = $scope.snapshots[0].timestamp;
                    }


                    $scope.availablePreviewFormats = null;
                    if ($scope.snapshots) {
                        if (params.loadLast) {
                            $scope.snapshot = $scope.snapshots[0];
                        } else if (params.exportTimestamp) {
                            $scope.snapshot = $scope.snapshots.find(e => e.timestamp == params.exportTimestamp);
                        }
                        $scope.availablePreviewFormats = getAvailablePreviewFormats($scope.snapshot, RMARKDOWN_PREVIEW_OUTPUT_FORMATS);
                        const formatNames = $scope.availablePreviewFormats.map(_ => _.name);
                        if (formatNames.length && !formatNames.includes(params.viewFormat)) {
                            params.viewFormat = $scope.availablePreviewFormats[0].name;
                        }
                    }
                    // eslint-disable-next-line no-console
                    console.info('Using exportFormat', params.exportFormat);

                    $scope.getLoadingPromise().then(
                        resp => $scope.getDisplaySnapshotPromise(resp.data)
                    ).then(
                        resp => $scope.setReportInsightContent(resp.data)
                    ).catch(
                        setErrorInScope.bind($scope)
                    );
                }

                $scope.$watch('insight.params', refresh, true);

                $scope.createSnapshot = function() {
                    const params = $scope.insight.params;
                    Dialogs.confirmPositive($scope, 'Snapshot report', 'Run Rmarkdown?').then(function() {
                        DataikuAPI.reports.snapshots.create($scope.insight.projectKey, params.reportSmartId)
                            .success(function(data) {
                                FutureProgressModal.show($scope, data, 'Building report for snapshot...').then(function(result) {
                                    const snapshot = result.snapshot;
                                    if (!params.loadLast) {
                                        const formatNames = getAvailablePreviewFormats(snapshot, RMARKDOWN_PREVIEW_OUTPUT_FORMATS).map(_ => _.name);
                                        if (formatNames.length && !formatNames.includes(params.viewFormat)) {
                                            params.viewFormat = formatNames[0];
                                        }
                                        params.exportTimestamp = snapshot.timestamp;
                                    }
                                    refresh();
                                    $scope.snapshots.unshift(snapshot);
                                });
                            }).error(setErrorInScope.bind($scope));
                    });
                };
            }
        };
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('RunnableButtonInsightHandler', {
        name: 'Macro',
        desc: 'Run a DSS macro',
        icon: 'icon-table',
        color: 'project',

        getSourceType: function() {
            return null;
        },
        getSourceId: function() {
            return null;
        },

        hasEditTab: true,
        goToEditAfterCreation: true,
        defaultTileParams: {
            showName: true
        },
        defaultTileDimensions: [6,6]

    });


    app.controller('RunnableButtonViewCommon', function($scope, Assert, DataikuAPI, $stateParams, DashboardUtils, $rootScope, PluginConfigUtils) {
        $scope.runnable = null;
        $rootScope.appConfig.customRunnables.forEach(function(x) {
            if (x.runnableType == $scope.insight.params.runnableType) {
                $scope.runnable = x;
            }
        });

        Assert.inScope($scope, 'runnable');

        $scope.insight.params.config = $scope.insight.params.config || {};
        $scope.desc = $scope.runnable.desc;

        PluginConfigUtils.setDefaultValues($scope.desc.params, $scope.insight.params.config);

        $scope.pluginDesc = $rootScope.appConfig.loadedPlugins.filter(function(x){
            return x.id == $scope.runnable.ownerPluginId;
        })[0];

        $scope.hasSettings = $scope.pluginDesc.hasSettings || ($scope.desc.params && $scope.desc.params.length > 0);
        $scope.runOutput = {};

        $scope.resetSettings = function() {
            $scope.insight.params.config = {};
            PluginConfigUtils.setDefaultValues($scope.desc.params, $scope.insight.params.config);
        };
    });


    app.directive('runnableButtonInsightTile', function($controller, DashboardUtils, TileLoadingState){
        return {
            templateUrl: '/templates/dashboards/insights/runnable-button/runnable-button_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs){
                $controller('RunnableButtonViewCommon', { $scope: $scope });

                $scope.loaded = false;
                $scope.error = null;

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    DashboardUtils.setLoaded.bind([$scope, resolve])();
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.$on('load-tile', $scope.load);


                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.directive('runnableButtonInsightTileParams', function(){
        return {
            templateUrl: '/templates/dashboards/insights/runnable-button/runnable-button_tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs){
            }
        };
    });


    app.directive('runnableButtonInsightCreateForm', function(DataikuAPI, $stateParams){
        return {
            templateUrl: '/templates/dashboards/insights/runnable-button/runnable-button_create_form.html',
            scope: true,
            link: function($scope, element, attrs){

                const refreshList = function() {
                    DataikuAPI.runnables.listAccessible($stateParams.projectKey).success(function(data) {
                        $scope.runnables = data.runnables;
                        $scope.runnablesExist = data.runnablesExist;
                    }).error(setErrorInScope.bind($scope));
                };
                refreshList();

                $scope.hook.sourceObject = null;
                $scope.hook.defaultName = 'Execute macro';


                function updateName() {
                }

                $scope.onRunnableSelected = function(runnable) {
                    $scope.insight.params.runnableType = runnable.runnableType;
                    $scope.hook.defaultName = 'Execute ' + (((runnable.desc || {}).meta || {}).label || 'macro').toLowerCase();
                };
            }
        };
    });

    app.directive('runnableButtonInsightEdit', function($controller, DataikuAPI, SmartId, WT1, $stateParams, $timeout){
        return {
            templateUrl: '/templates/dashboards/insights/runnable-button/runnable-button_edit.html',
            scope: true,
            link: function($scope, element, attrs) {
                $controller('RunnableButtonViewCommon', { $scope: $scope });

                DataikuAPI.security.listUsers().success(function(data) {
                    $scope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
                    $scope.allUsersLogin = data.map(user => '@' + user.login);
                    // empty description for the `Current user` value
                    $scope.allUsersLogin.unshift('');
                }).error(setErrorInScope.bind($scope));

            }
        };
    });

    app.directive('runnableButtonInsightView', function($controller, $timeout){
        return {
            templateUrl: '/templates/dashboards/insights/runnable-button/runnable-button_view.html',
            scope: {
                insight: '='
            },
            link: function($scope, element, attrs){
                $controller('RunnableButtonViewCommon', { $scope: $scope });
            }
        };
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('SavedModelReportInsightHandler', {
        name: 'Saved model report',
        desc: 'Full report of a model',
        icon: 'icon-dku-modelize',
        color: 'saved-model',

        getSourceId: function(insight) {
            return insight.params.savedModelSmartId;
        },
        sourceType: 'SAVED_MODEL',
        hasEditTab: false,
        defaultTileParams: {
            displayMode: 'summary'
        },
        defaultTileDimensions: [24, 12]

    });

    app.controller('SavedModelReportViewCommon', function($scope, DataikuAPI, $controller, FullModelLikeIdUtils, WebAppsService,$state, $stateParams) {
        $scope.noMlReportTourHere = true; // the tabs needed for the tour are not present
        $scope.readOnly = true;
        $scope.noUrlChange = true;

        $scope._getSkins = function(versionId, contentType, algorithm, coreParams) {
            if (algorithm) {
                if (contentType && !contentType.endsWith('/')) {
                    contentType = contentType + '/';
                }
                contentType += algorithm.toLowerCase();
            }
            const skins = WebAppsService.getSkins(
                'SAVED_MODEL', versionId,
                { predictionType: coreParams.prediction_type, backendType: coreParams.backendType, contentType },
                $scope.staticModelSkins
            );
            return skins;
        };

        $scope.getModel = function(onLoadError) {
            const p = DataikuAPI.savedmodels.get($scope.insight.projectKey, $scope.insight.params.savedModelSmartId)
                .success(function(data) {
                    $scope.insight.$savedModel = data;
                    const version = $scope.insight.params.version || data.activeVersion;

                    $scope.insight.$fullModelId = FullModelLikeIdUtils.buildSavedModelFmi({
                        projectKey: data.projectKey,
                        savedModelId: data.id,
                        versionId: version
                    });
                    $scope.fullModelId = $scope.insight.$fullModelId;
                    const getDetailsP = DataikuAPI.ml[data.miniTask.taskType.toLowerCase()].getModelDetails($scope.fullModelId).success(function(modelData) {
                    /*
                     * For example, in the case of MLflow, coreParams may be undefined.
                     * The user will still be able to manually provide data.contentType through the model "settings" screen
                     */
                        const defaultValue = modelData.coreParams ? `${modelData.coreParams.taskType}/${modelData.coreParams.backendType}`.toLowerCase() : '';
                        const contentType = data.contentType ? data.contentType : defaultValue;
                        $scope.modelSkins = $scope._getSkins(
                            version, contentType, modelData.modeling.algorithm, modelData.coreParams
                        );
                    }).error(setErrorInScope.bind($scope));
                    if ($scope.noSpinner) {
                        getDetailsP.noSpinner();
                    }
                    /*
                     * TODO(jfy): is that good... ?
                     * set smId for saved models usage
                     * if using $state.go(".", ..., {reload: false}), there are exception due to backend calls because $stateParams.smId is not yet set
                     */
                    $stateParams.smId = data.id;
                    $scope.noSetLoc = true;
                    $scope.versionsContext = {};

                    switch (data.miniTask.taskType) {
                    // TODO: This could be better factorized with saved models to call the controller in the template or the route
                        case 'PREDICTION':
                            if ($scope.insight.$savedModel.miniTask.partitionedModel
                            && $scope.insight.$savedModel.miniTask.partitionedModel.enabled) {
                                $controller('PMLPartModelReportController', { $scope:$scope });
                            }
                            $controller('PredictionSavedModelReportController', { $scope:$scope });
                            break;
                        case 'CLUSTERING':
                            $controller('ClusteringSavedModelReportController', { $scope:$scope });
                            break;
                    }
                })
                .error(function(data, status, headers, config, statusText) {
                    if (typeof(onLoadError) === 'function') {
                        onLoadError(data, status, headers, config, statusText);
                    }
                });

            if ($scope.noSpinner) {
                p.noSpinner();
            }
        };

        $scope.isDeepHubPrediction = function(){
            return $scope.insight.$savedModel.miniTask.backendType === 'DEEP_HUB';
        };

        $scope.isCausalPrediction = function(){
            return ['CAUSAL_REGRESSION', 'CAUSAL_BINARY_CLASSIFICATION'].includes($scope.insight.$savedModel.miniTask.predictionType);
        };

    });

    app.directive('savedModelReportInsightTile', function($controller, DashboardUtils, TileLoadingState, SavedModelsService) {
        return {
            templateUrl: '/templates/dashboards/insights/saved-model_report/saved-model_report_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope){

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    $scope.onLoadSuccess = DashboardUtils.setLoaded.bind([$scope, resolve]);
                    $scope.onLoadError = DashboardUtils.setError.bind([$scope, reject]);
                    $scope.noSpinner = true;
                    $scope.getModel($scope.onLoadError);
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                $scope.isPartitionedModel = function() {
                    return $scope.insight.$modelData
                    && $scope.insight.$modelData.coreParams
                    && $scope.insight.$modelData.coreParams.partitionedModel
                    && $scope.insight.$modelData.coreParams.partitionedModel.enabled;
                };

                $scope.isExternalMLflowModel = function() {
                    return SavedModelsService.isExternalMLflowModel($scope.insight.$modelDat);
                };
                $scope.isMLflowModel = function() {
                    return SavedModelsService.isMLflowModel($scope.insight.$modelDat);
                };

                $controller('SavedModelReportViewCommon', { $scope:$scope });

                // Expose model data for savedModelReportInsightTileParams to display the appropriate tabs
                $scope.$watch('modelData', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.insight.$modelData = nv;
                });

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }

                $scope.noSkinControls = true; // no need to display controls widget in dashboards view
            }
        };
    });

    app.controller('SavedModelReportInsightController', function($scope, $controller, $stateParams, DataikuAPI, $state, MLModelsUIRouterStates, GoToStateNameSuffixIfBase, ComputeTransitionStateName) {
        const baseStateName = 'insight.view.savedmodels';

        $controller('SavedModelReportViewCommon', { $scope:$scope });


        const getSummarySuffix = () => {
            if ($scope.modelData.coreParams.taskType === 'PREDICTION') {
                return MLModelsUIRouterStates.getPredictionReportSummaryTab($scope.isDeepHubPrediction(), false);
            } else {
                return 'summary';
            }
        };

        const getRouteType = () => {
            return $scope.modelData.coreParams.taskType === 'PREDICTION' ? 'prediction' : 'clustering';
        };

        const updateUIRouterStateWithTarget = ({ stateName, stateParams }) => {
            const transitionStateName = ComputeTransitionStateName(baseStateName, stateName);

            if (transitionStateName || !angular.equals($state.current.params, stateParams)) {
                $state.go(transitionStateName, stateParams, { location: 'replace' });
            }
        };

        const getUIRouterStateNameTarget = () => {
            if (!$state.current.name.includes(baseStateName) || !$scope.modelData) {
                return baseStateName;
            }

            // if stateName is 'insight.view.savedmodels' redirection on 'insight.view.savedmodels.summary'
            if ($state.current.name.endsWith(baseStateName)) {
                return `${baseStateName}.${getRouteType()}.${getSummarySuffix()}`;
            }

            return $state.current.name.slice($state.current.name.lastIndexOf(baseStateName));
        };

        const getUIRouterStateNameAndParamsTargetFromTile = (tile) => {
            const stateNameTypePart = getRouteType();
            let stateParams = null;
            let stateNameTabPart;

            if (tile.tileParams.displayMode === 'skins') {

                if (tile.tileParams.advancedOptions && tile.tileParams.advancedOptions.customViews) {
                    stateNameTabPart = 'modelview';
                    stateParams = { skinId: tile.tileParams.advancedOptions.customViews.viewId };
                } else {
                    stateNameTabPart = getSummarySuffix();
                }

            } else {
                const pane = MLModelsUIRouterStates.dashboardTileToSavedModelPane(tile, $scope.modelData.coreParams.taskType === 'CLUSTERING', $scope);
                stateNameTabPart = pane.route;
                stateParams = pane.routeParams;
            }

            return {
                stateName: `${baseStateName}.${stateNameTypePart}.${stateNameTabPart}`,
                stateParams: stateParams
            };
        };


        $scope.$on('$stateChangeStart', function(event, toState, toParams, fromState, fromParams) {
            if (!$scope.modelData) {
                return;
            }
            GoToStateNameSuffixIfBase($state, toState, toParams, fromState, fromParams, '.insight.view', `savedmodels.${getRouteType()}.${getSummarySuffix()}`, event);
        });

        $scope.getModel(setErrorInScope.bind($scope));

        $scope.$watch('modelData', function(nv, ov) {
            if (!nv) {
                return;
            }
            if ($state.current.name.endsWith(baseStateName)) {
                updateUIRouterStateWithTarget({ stateName: getUIRouterStateNameTarget() });
            }

            if ($stateParams.originDashboard) {
                DataikuAPI.dashboards.getFullInfo($stateParams.projectKey, $stateParams.originDashboard.id).success(function(data) {
                    const tile = data.dashboard.pages.find(page => page.id === $stateParams.originDashboard.pageId).grid.tiles.find(tile => tile.insightId === $stateParams.insightId);
                    updateUIRouterStateWithTarget(getUIRouterStateNameAndParamsTargetFromTile(tile));
                });
            }
        });

        $scope.onLoadError = function(data, status, headers, config, statusText) {
            setErrorInScope.bind($scope)(data, status, headers, config, statusText);
        };

    });

    app.directive('savedModelReportInsightView', function($controller, $stateParams, DataikuAPI, $state) {
        return {
            scope: true,
            link: function($scope, element, attrs) {
                if ($state.current.name.endsWith('.insight.view')) {
                    $state.go('.savedmodels', null, { location: 'replace' });
                }
            }
        };
    });

    app.directive('savedModelReportInsightTileParams', function($controller, $timeout, DataikuAPI, FullModelLikeIdUtils, $q, ModelDataUtils){
        return {
            templateUrl: '/templates/dashboards/insights/saved-model_report/saved-model_report_tile_params.html',
            scope: {
                tileParams: '=',
                insight: '='
            },
            link: function($scope, $element, attrs) {
                $scope.ModelDataUtils = ModelDataUtils;

                function setDefaultUpliftCurveDisplayMode() {
                    $scope.tileParams.advancedOptions = $scope.tileParams.advancedOptions || {};
                    $scope.tileParams.advancedOptions.upliftCurve = $scope.tileParams.advancedOptions.upliftCurve || {};
                    $scope.tileParams.advancedOptions.upliftCurve.displayMode = $scope.tileParams.advancedOptions.upliftCurve.displayMode || 'cumulativeUplift';
                };

                const setDefaultOverridesMetricsDisplayMode = function() {
                    // Setting default value of `tileParams.advancedOptions.overridesMetrics.displayMode` to "sankey" if not already set
                    $scope.tileParams.advancedOptions = $scope.tileParams.advancedOptions || {};
                    $scope.tileParams.advancedOptions.overridesMetrics = $scope.tileParams.advancedOptions.overridesMetrics || {};
                    $scope.tileParams.advancedOptions.overridesMetrics.displayMode = $scope.tileParams.advancedOptions.overridesMetrics.displayMode || 'sankey';
                };

                $scope.$watch('insight.$modelData', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.modelData = nv;
                    $scope.fullModelId = $scope.insight.$fullModelId;

                    $controller('SavedModelReportViewCommon', { $scope:$scope });

                    function getVersionSkins(projectKey, smartId) {
                        const deferred = $q.defer();
                        DataikuAPI.savedmodels.get(projectKey, smartId)
                            .success(function(sm) {
                                const version = $scope.insight.params.version || sm.activeVersion;
                                if (!$scope.fullModelId) {
                                    $scope.fullModelId = FullModelLikeIdUtils.buildSavedModelFmi({
                                        projectKey: sm.projectKey,
                                        savedModelId: sm.id,
                                        versionId: version
                                    });
                                }
                                DataikuAPI.ml[sm.miniTask.taskType.toLowerCase()].getModelDetails($scope.fullModelId).success(function(modelDetails) {
                                    let contentType = sm.contentType;
                                    if (!contentType) {
                                        contentType = modelDetails.coreParams ? `${modelDetails.coreParams.taskType}/${modelDetails.coreParams.backendType}`.toLowerCase() : '';
                                    }
                                    // _getSkins() defined in SavedModelReportViewCommon
                                    deferred.resolve($scope._getSkins(
                                        version, contentType, modelDetails.modeling.algorithm, modelDetails.coreParams
                                    ));
                                }).error(setErrorInScope.bind($scope));
                            });
                        return deferred.promise;
                    };

                    getVersionSkins($scope.insight.projectKey, $scope.insight.params.savedModelSmartId).then(function(modelSkins) {
                        $scope.modelSkins = modelSkins;
                        $timeout(function() {
                            $scope.$broadcast('selectPickerRefresh');
                        });
                    });

                    // set default for tileParams.advancedOptions.interactiveScoring
                    if (!($scope.tileParams.advancedOptions && $scope.tileParams.advancedOptions.interactiveScoring) && $scope.insight.$savedModel.miniTask.taskType == 'PREDICTION') {
                        Promise.all([DataikuAPI.ml.prediction.getColumnImportance($scope.fullModelId),
                            DataikuAPI.ml.prediction.getSplitDesc($scope.fullModelId),
                            DataikuAPI.ml.prediction.getPreparationScript($scope.fullModelId),
                            DataikuAPI.ml.prediction.getInputDatasetSchema($scope.fullModelId).catch(e => e)]).then(
                            ([columnImportanceResp, splitDescResp, preparationScriptResp, inputDatasetSchemaResp]) => {
                                let featuresOrder;
                                if (columnImportanceResp.data) { // sort by importance
                                    const importances = columnImportanceResp.data.importances;
                                    const columns = columnImportanceResp.data.columns;
                                    featuresOrder = columns.sort((c1, c2) => importances[columns.indexOf(c2)] - importances[columns.indexOf(c1)]);
                                } else { // same order as in dataset
                                    const perFeature = $scope.modelData.preprocessing.per_feature;
                                    const inputColumns = Object.keys(perFeature).filter(featureName => perFeature[featureName].role === 'INPUT');
                                    featuresOrder = splitDescResp.data.schema.columns.map(c => c.name).filter(c => inputColumns.includes(c));
                                }
                                const hasPreparationSteps = preparationScriptResp.data.steps.some(step => !step.disabled);
                                if (hasPreparationSteps) {
                                    if (inputDatasetSchemaResp.data.columns) {
                                        const preScriptFeatures = inputDatasetSchemaResp.data.columns.map((col) => col.name);
                                        if (columnImportanceResp.data) {
                                            featuresOrder.push(...preScriptFeatures.filter(f => !featuresOrder.includes(f)));
                                        } else {
                                            featuresOrder = [...preScriptFeatures, ...featuresOrder.filter(f => !preScriptFeatures.includes(f))];
                                        }
                                    } else if (inputDatasetSchemaResp.status !== 404) {
                                    /*
                                     * 404 is expected when the model has no `input_dataset_schema.json` (old model)
                                     * and has no more origin analysis (deleted)
                                     */
                                        setErrorInScope.call($scope, inputDatasetSchemaResp);
                                    }
                                }

                                $scope.tileParams.advancedOptions = {
                                    ...$scope.tileParams.advancedOptions,
                                    interactiveScoring: {
                                        featuresOrder
                                    }
                                };
                            }
                        ).catch(setErrorInScope.bind($scope));
                    }

                    switch ($scope.insight.$savedModel.miniTask.taskType) {
                        case 'PREDICTION':
                            $controller('_PredictionModelReportController', { $scope:$scope });
                            break;
                        case 'CLUSTERING':
                            $controller('_ClusteringModelReportController', { $scope:$scope });
                            break;
                    }

                    $timeout(function() {
                        $element.find('.view-select').selectpicker('refresh');
                    });

                    /*
                     * tileParams.$displayModeExtended is used for models except deephub
                     * deephub uses regular displayMode
                     */
                    if ($scope.insight.$modelData.backendType !== 'DEEP_HUB') {
                        /**
                         * Returns the displayModeExtended to use for the given skin
                         */
                        $scope.displayModeExtendedForSkin = function(skinId) {
                            return 'skins_' + skinId;
                        };

                        /**
                         * Translates the displayModeExtended to the displayMode and the viewId
                         */
                        $scope.onDisplayModeExtendedChange = function() {
                            if (!$scope.tileParams || !$scope.tileParams.$displayModeExtended) {
                                return;
                            }
                            const displayModeExtended = $scope.tileParams.$displayModeExtended;
                            if (displayModeExtended.startsWith('skins_')) {
                                if (!$scope.tileParams.advancedOptions) {
                                    $scope.tileParams.advancedOptions = {};
                                }
                                if (!$scope.tileParams.advancedOptions.customViews) {
                                    $scope.tileParams.advancedOptions.customViews = {};
                                }
                                $scope.tileParams.advancedOptions.customViews.viewId = displayModeExtended.substring('skins_'.length);
                                $scope.tileParams.displayMode = 'skins';
                            } else {
                                $scope.tileParams.displayMode = displayModeExtended;
                            }
                            if (displayModeExtended === 'overrides_metrics') {
                                setDefaultOverridesMetricsDisplayMode();
                            }
                            if (displayModeExtended === 'uplift_curve') {
                                setDefaultUpliftCurveDisplayMode();
                            }
                        };
                        // Used for initializing the select
                        const deregister = $scope.$watch('tileParams', function(nv) {
                            if (!nv) {
                                return;
                            }
                            if ($scope.tileParams.displayMode == 'skins' && $scope.tileParams.advancedOptions && $scope.tileParams.advancedOptions.customViews) {
                                $scope.tileParams.$displayModeExtended = $scope.displayModeExtendedForSkin($scope.tileParams.advancedOptions.customViews.viewId);
                            } else {
                                $scope.tileParams.$displayModeExtended = $scope.tileParams.displayMode || 'summary';
                            }
                            deregister();
                        });
                    }

                    if ($scope.tileParams.displayMode === 'overrides_metrics') {
                        setDefaultOverridesMetricsDisplayMode();
                    }
                    setDefaultImportanceDisplayMode($scope.hasVariableImportance(), $scope.hasGlobalExplanations());
                });

                if ($scope.tileParams.displayMode === 'uplift_curve') {
                    setDefaultUpliftCurveDisplayMode();
                }

                if ($scope.tileParams.displayMode === 'overrides_metrics') {
                    setDefaultOverridesMetricsDisplayMode();
                }

                const setDefaultImportanceDisplayMode = function(hasVariableImportance, hasGlobalExplanations) {
                    let importanceDisplayMode = 'globalExplanations';
                    const tileParamsAdvancedOptions = $scope.tileParams.advancedOptions;
                    if (hasVariableImportance) {
                        if (!tileParamsAdvancedOptions || !tileParamsAdvancedOptions.featureImportance) {
                            if (!hasGlobalExplanations) {
                                importanceDisplayMode = 'variableImportance';
                            }
                        } else {
                            importanceDisplayMode = tileParamsAdvancedOptions.featureImportance.importanceDisplayMode;
                        }
                    }
                    const advancedOptions = $scope.tileParams.advancedOptions || { featureImportance: {} };
                    $scope.tileParams.advancedOptions = {
                        ...advancedOptions,
                        featureImportance: {
                            ...advancedOptions.featureImportance,
                            importanceDisplayMode
                        }
                    };
                    setDefaultGraphType();
                };

                const setDefaultGraphType = function() {
                    let graphType = 'absoluteFeatureImportance';
                    const tileParamsAdvancedOptions = $scope.tileParams.advancedOptions;
                    if (tileParamsAdvancedOptions && tileParamsAdvancedOptions.featureImportance.graphType) {
                        graphType = tileParamsAdvancedOptions.featureImportance.graphType;
                    }

                    $scope.tileParams.advancedOptions = {
                        ...tileParamsAdvancedOptions,
                        featureImportance: {
                            ...tileParamsAdvancedOptions.featureImportance,
                            graphType
                        }
                    };
                };
            }
        };
    });


    app.directive('savedModelReportInsightCreateForm', function(DataikuAPI){
        return {
            templateUrl: '/templates/dashboards/insights/saved-model_report/saved-model_report_create_form.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.hook.defaultName = 'Saved model';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = nv.label + ' table';
                });

                /*
                 *$scope.$watch("insight.params.savedModelSmartId", function(nv) {
                 *    $scope.versions = [];
                 *    if (!nv) return;
                 *    DataikuAPI.savedmodels.listVersionIds($scope.insight.projectKey, $scope.insight.params.savedModelSmartId).success(function() {
                 *        $scope.versions = data;
                 *    });
                 *})
                 */
            }
        };
    });


})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('ScenarioInsightHandler', {
        name: 'Scenario',
        desc: 'Run button or activity report of a scenario',
        icon: 'icon-list',
        color: 'scenario'
    });

    app.constant('ScenarioLastRunsInsightHandler', {

        icon: 'icon-list',
        color: 'scenario',
        name: 'Scenario last runs',

        getSourceId: function(insight) {
            return insight.params.scenarioSmartId;
        },
        sourceType: 'SCENARIO',
        hasEditTab: false,
        defaultTileParams: {
            displayMode: 'SIMPLE',
            range: 'CURRENT_MONTH'
        },
        defaultTileShowTitleMode: 'MOUSEOVER',
        defaultTileDimensions: [36, 3]
    });

    app.constant('ScenarioRunButtonInsightHandler', {

        icon: 'icon-list',
        color: 'scenario',
        name: 'Scenario run button',

        getSourceId: function(insight) {
            return insight.params.scenarioSmartId;
        },
        sourceType: 'SCENARIO',
        hasEditTab: false,
        defaultTileParams: {
        },
        defaultTileDimensions: [6, 6],
        accessMode: 'RUN'
    });

    app.controller('ScenarioLastRunsViewCommon', function($scope) {
        $scope.resolveRange = function(range) {
            let to, from;
            switch(range) {
                case 'CURRENT_DAY':
                    to = moment();
                    from = moment().startOf('day');
                    break;
                case 'PREVIOUS_DAY':
                    from = moment().subtract(1, 'day').startOf('day');
                    to = moment().subtract(1, 'day').endOf('day');
                    break;
                case 'LAST_NIGHT':
                    to = moment().set({ 'hours': 9, 'second': 0, 'millisecond': 0 });
                    from = moment().subtract(1, 'day').set({ 'hours': 17, 'second': 0, 'millisecond': 0 });
                    break;
                case 'CURRENT_WEEK':
                    to = moment();
                    from = moment().startOf('week');
                    break;
                case 'PREVIOUS_WEEK':
                    from = moment().subtract(1, 'week').startOf('week');
                    to = moment().subtract(1, 'week').endOf('week');
                    break;
                case 'CURRENT_MONTH':
                    to = moment();
                    from = moment().startOf('month');
                    break;
                case 'PREVIOUS_MONTH':
                    from = moment().subtract(1, 'month').startOf('month');
                    to = moment().subtract(1, 'month').endOf('month');
                    break;
                default:
                    throw 'Unexpected range: ' + range;
            }
            return { to: to.format(), from: from.format() };
        };


        $scope.ranges = [
            'CURRENT_DAY',
            'LAST_NIGHT',
            'PREVIOUS_DAY',
            'CURRENT_WEEK',
            'PREVIOUS_WEEK',
            'CURRENT_MONTH',
            'PREVIOUS_MONTH'
        ];
    });

    app.directive('scenarioLastRunsInsightTile', function($stateParams, $timeout, DataikuAPI, $controller, DashboardUtils, TileLoadingState){
        return {
            templateUrl: '/templates/dashboards/insights/scenario_last_runs/scenario_last_runs_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs){

            	$controller('ScenarioTimelineControllerCommon', { $scope: $scope });
                $controller('ScenarioLastRunsViewCommon', { $scope: $scope });

                $scope.scenarioId = $scope.insight.params.scenarioSmartId;
        		$scope.uiState.viewMode = $scope.insight.params.viewMode;

            	$scope.loading = false;
                $scope.loaded = false;
                $scope.error = null;

                $scope.load = function(resolve, reject) {
                	$scope.loading = true;

                    $scope.$watch('tile.tileParams.range', function(nv) {
                        if (!nv) {
                            return;
                        }

                        const resolvedRange = $scope.resolveRange($scope.tile.tileParams.range);
                        DataikuAPI.scenarios.getScenarioReport($scope.insight.projectKey, $scope.insight.params.scenarioSmartId, resolvedRange.from, resolvedRange.to)
                            .noSpinner()
                            .success($scope.setScenarioGantt)
                            .success(DashboardUtils.setLoaded.bind([$scope, resolve]))
                            .error(DashboardUtils.setError.bind([$scope, reject]));
                    });

                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }

            }
        };
    });

    app.directive('scenarioLastRunsInsightView', function($controller, DataikuAPI){
        return {
            templateUrl: '/templates/dashboards/insights/scenario_last_runs/scenario_last_runs_view.html',
            scope: {
                insight: '=',
                tileParams: '='
            },
            link: function($scope, element, attrs) {
            	$controller('ScenarioTimelineControllerCommon', { $scope: $scope });
                $controller('ScenarioLastRunsViewCommon', { $scope: $scope });

                $scope.editable = true;
                $scope.scenarioId = $scope.insight.params.scenarioSmartId;
                $scope.uiState.range = 'CURRENT_MONTH';

                $scope.$watch('uiState.range', function(nv) {
                    if (!nv) {
                        return;
                    }
                    const resolvedRange = $scope.resolveRange($scope.uiState.range);

                    DataikuAPI.scenarios.getScenarioReport($scope.insight.projectKey, $scope.insight.params.scenarioSmartId, resolvedRange.from, resolvedRange.to).success(function(data){
                        $scope.setScenarioGantt(data);
                    }).error(function(data, status, headers, config, statusText) {
                        setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                    });
                });
            }
        };
    });

    app.directive('scenarioLastRunsInsightTileParams', function($controller){
        return {
            templateUrl: '/templates/dashboards/insights/scenario_last_runs/scenario_last_runs_tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs){
                $controller('ScenarioLastRunsViewCommon', { $scope: $scope });
            }
        };
    });

    app.directive('scenarioInsightCreateForm', function(DataikuAPI, $filter, $controller){
        return {
            templateUrl: '/templates/dashboards/insights/scenario_last_runs/scenario_create_form.html',
            scope: true,
            link: function($scope, element, attrs){
                $controller('ScenarioLastRunsViewCommon', { $scope: $scope });

                $scope.insight.type = 'scenario_last_runs';
                $scope.insight.params.viewMode = 'TIMELINE';
                $scope.insight.params.range = 'CURRENT_DAY';
                $scope.hook.defaultName = 'Scenario';

                function updateDefaultName() {
                    if ($scope.insight.type == 'scenario_last_runs') {
                        $scope.hook.defaultName = $filter('niceConst')($scope.insight.params.viewMode) + ' view of scenario';
                    } else if ($scope.insight.type == 'scenario_run_button') {
                        $scope.hook.defaultName = 'Run scenario';
                    } else {
                        $scope.hook.defaultName = 'Scenario';
                    }
                    if ($scope.hook.sourceObject && $scope.hook.sourceObject.label) {
                        $scope.hook.defaultName += ' ' + $scope.hook.sourceObject.label;
                    }
                }
                $scope.$watch('hook.sourceObject', updateDefaultName);
                $scope.$watch('insight.params.viewMode', updateDefaultName);
                $scope.$watch('insight.type', updateDefaultName);
            }
        };

    });

    app.directive('scenarioRunButtonInsightTile', function(DataikuAPI, WT1, Notification, SmartId,
        ScenarioUtils, TileLoadingState, Dialogs) {
        return {
            templateUrl: '/templates/dashboards/insights/scenario_run_button/scenario_run_button_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs){

                $scope.getTriggerName = ScenarioUtils.getTriggerName;

                // Check if there is a loading scenario
                function refreshScenarioRunState(resolve, reject) {
                    return DataikuAPI.scenarios.getLastScenarioRuns($scope.insight.projectKey, $scope.insight.params.scenarioSmartId, true, 1)
                        .success(function(data) {
                            $scope.lastRun = data[0];
                            $scope.runStarting = false;
                            $scope.scenario = data.scenario;
                            $scope.loading = false;
                            $scope.loaded = true;
                            if (resolve) {
                                resolve();
                            }
                        })
                        .error($scope.hook.setErrorInDashboardPageScope.bind($scope))
                        .error(function() {
                            $scope.loading = false;
                            if (reject) {
                                reject();
                            }
                        }).noSpinner();
                }

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;

                    const resolvedScenario = SmartId.resolve($scope.insight.params.scenarioSmartId);

                    refreshScenarioRunState(resolve, reject)
                        .success(function() {
                            const unRegister = Notification.registerEvent('scenario-state-change', function(evt, message) {
                                if (message.scenarioId != resolvedScenario.id || message.projectKey != resolvedScenario.projectKey) {
                                    return;
                                }
                                refreshScenarioRunState();
                            });
                            $scope.$on('$destroy', unRegister);
                        });
                };

                $scope.abort = function() {
                    Dialogs.confirm($scope, 'Abort scenario run', 'Are you sure you want to abort this scenario?')
                        .then(() => {
                            DataikuAPI.futures.abort($scope.lastRun.futureId);
                        });
                };

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                    $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }

                $scope.runNow = function() {
                    $scope.runStarting = true;
                    WT1.event('scenario-manual-run-from-dashboard');
                    DataikuAPI.scenarios.manualRun($scope.insight.projectKey, $scope.insight.params.scenarioSmartId).noSpinner()
                        .success(function(data){})
                        .error(function(data, status, headers, config, statusText) {
                            $scope.runStarting = false;
                            $scope.hook.setErrorInDashboardPageScope.bind($scope)(data, status, headers, config, statusText);
                        });
                };
            }
        };
    });

    app.directive('scenarioRunButtonInsightTileParams', function(DataikuAPI){
        return {
            templateUrl: '/templates/dashboards/insights/scenario_run_button/scenario_run_button_tile_params.html',
            scope: {
                tileParams: '='
            }
        };
    });

})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('StaticFileInsightHandler', {
        name: 'Static insight',
        desc: 'Insight generated from code',
        icon: 'icon-file-alt',
        color: '',

        getSourceId: function(insight) {
            return insight.params.objectSmartId;
        },
        getSourceType: function(insight) {
            return insight.params.objectType;
        },
        hasEditTab: false,
        defaultTileParams: {
            numDisplayedComments: 5
        },
        defaultTileDimensions: [15, 9]
    });



    app.controller('StaticFileInsightViewCommon', function($scope, $controller, $sce, DataikuAPI, $stateParams, $timeout) {
        $scope.getLoadingPromise = function() {
            return DataikuAPI.dashboards.insights.viewStaticFile($scope.insight.projectKey, $scope.insight.id);
        };

        $scope.setStaticFileInsightContent = data => {
            $scope.staticFileInsightContent = data;
        };

        $scope.download = function() {
            const insightContentURL = '/dip/api/dashboards/insights/view-static-file?'
            + 'projectKey=' + $scope.insight.projectKey
            + '&insightId=' + $scope.insight.id
            +'&download=true';
            downloadURL(insightContentURL);
        };
    });

    app.directive('staticFileInsightTile', function($controller, $timeout, TileLoadingState, DashboardUtils) {
        return {
            templateUrl: '/templates/dashboards/insights/static_file/tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope, element, attrs) {
                $controller('StaticFileInsightViewCommon', { $scope: $scope });

                $scope.loading = false;
                $scope.loaded = false;
                $scope.error = null;
                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    const loadingPromise = $scope.getLoadingPromise().noSpinner();
                    loadingPromise
                        .then(function(resp) {
                            DashboardUtils.setLoaded.bind([$scope, resolve])();
                            $scope.setStaticFileInsightContent(resp.data);
                        }, DashboardUtils.setError.bind([$scope, reject]));
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });


    app.directive('staticFileInsightTileParams', function() {
        return {
            templateUrl: '/templates/dashboards/insights/static_file/tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs) {
            // No params
            }
        };
    });

    app.directive('staticFileInsightCreateForm', function() {
        return {
            templateUrl: '/templates/dashboards/insights/static_file/create_form.html',
            scope: true,
            link: function($scope, element, attrs) {
            // Can't create static file insight from new insight modal
            }
        };
    });

    app.directive('staticFileInsightView', function($controller) {
        return {
            templateUrl: '/templates/dashboards/insights/static_file/view.html',
            scope: true,
            link: function($scope, element, attrs) {
                $controller('StaticFileInsightViewCommon', { $scope: $scope });
                const loadingPromise = $scope.getLoadingPromise();
                loadingPromise.then(resp => {
                    $scope.setStaticFileInsightContent(resp.data);
                }, setErrorInScope.bind($scope));
            }
        };
    });

    app.directive('staticFileInsightEdit', function($controller, DataikuAPI, $rootScope) {
        return {
            templateUrl: '/templates/dashboards/insights/static_file/edit.html',
            scope: {
                insight: '='
            },
            link: function($scope, element, attrs) {
                $controller('ChartInsightViewCommon', { $scope: $scope });

                $scope.currentInsight = $scope.insight;

                $scope.bigChart = false;

                $scope.saveChart = function() {
                    DataikuAPI.dashboards.insights.save($scope.insight)
                        .error(setErrorInScope.bind($scope));
                };

                $rootScope.$on('chartSamplingChanged', function() {
                    $scope.summary = null;
                    $scope.fetchColumnsSummary();
                    $scope.saveChart();
                });

                $scope.fetchColumnsSummary();
            }
        };
    });

})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.directive('textTile', function(TileLoadingState){
        return {
            templateUrl: '/templates/dashboards/insights/text/text_tile.html',
            scope: {
                tileParams: '=',
                hook: '=',
                tile: '='
            },
            link: function($scope){

                $scope.load = function(resolve) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.COMPLETE;
                    resolve();
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
            }
        };
    });

    app.directive('textTileParams', function(CodeMirrorSettingService){
        return {
            templateUrl: '/templates/dashboards/insights/text/text_tile_params.html',
            scope: {
                tileParams: '=',
                setHorizontalAlignment: '&',
                setVerticalAlignment: '&'
            },
            link: function($scope){
                $scope.editorOptions = CodeMirrorSettingService.get('text/plain');
            }
        };
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.controller('InsightViewController', function($scope, $controller, $stateParams, DataikuAPI, CreateModalFromTemplate, Dialogs,$state,$q, TopNav) {
        TopNav.setLocation(TopNav.TOP_DASHBOARD, 'insights', null, 'view');
        if ($scope.insight) {
            TopNav.setPageTitle($scope.insight.name + ' - Insight');
        }

        $scope.uiState = $scope.uiState || {};
        $scope.uiState.fullScreen = $stateParams.fullScreen && $stateParams.fullScreen != 'false';

        $scope.$watch('uiState.fullScreen', function(nv) {
            if (nv == null) {
                return;
            }
            $state.go($state.current, { fullScreen: (nv && nv != 'false') ? true : null }, { location: true, inherit:true, notify:false, reload:false });
        });
    });

    app.directive('insightPreview', function(TileUtils) {
        return {
            template: '<dashboard-tile editable="false" insight="insight" dashboard-theme="dashboardTheme" tile="tile" hook="hook" />',
            scope: {
                insight: '=',
                dashboardTheme: '<',
                autoload: '=?'
            },
            link: function($scope, $el) {
                $scope.$watch('insight', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.tile = TileUtils.newInsightTile($scope.insight);
                    $scope.tile.$tileId = 'this';
                });

                $scope.hook = {
                    loadPromises: {},
                    reloadPromises: {},
                    loadStates: {}
                };

                function load() {
                    $scope.$watch('hook.loadPromises[\'this\']', function(nv) {
                        if (!nv) {
                            return;
                        }
                        nv();
                    });
                }

                if ($scope.autoload) {
                    load();
                }
                $el.on('loadInsightPreview', load);
            }
        };
    });

    app.directive('insightPreviewLoading', function() {
        return {
            scope: false,
            link: function($scope, $element) {
                $scope.loadInsightPreview = function() {
                    $element.find('.insight-details [insight-preview]').trigger('loadInsightPreview');
                };
            }
        }
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('WebAppInsightHandler', {
        name: 'Webapp',
        desc: 'Display webapp',
        icon: 'icon-code',
        color: 'notebook',

        getSourceId: function(insight) {
            return insight.params.webAppSmartId;
        },
        sourceType: 'WEB_APP',
        hasEditTab: false,
        defaultTileParams: {

        },
        defaultTileShowTitleMode: 'NO',
        defaultTileDimensions: [18, 12]
    });

    app.controller('WebAppViewCommon', function($scope, $stateParams, $controller, $q, DataikuAPI, Logger, WebAppsService) {
        $scope.resolvedWebApp = resolveObjectSmartId($scope.insight.params.webAppSmartId, $stateParams.projectKey);

        const baseType = WebAppsService.getBaseType($scope.insight.params.webAppType);
        if (baseType == 'STANDARD') {
            $controller('StandardWebAppController', { $scope: $scope });
        } else if (baseType == 'BOKEH') {
            $controller('BokehWebAppController', { $scope: $scope });
        } else if (baseType == 'DASH') {
            $controller('DashWebAppController', { $scope: $scope });
        } else if (baseType == 'SHINY') {
            $controller('ShinyWebAppController', { $scope: $scope });
        } else if (baseType == 'CODE_STUDIO_AS_WEBAPP') {
            $controller('CodeStudioWebAppController', { $scope: $scope });
        } else {
            Logger.error('Unknown app type: ', $scope.insight.params.webAppType);
        }

    });

    app.directive('webAppInsightTile', function($controller, $q, DataikuAPI, $timeout, DashboardUtils, StateUtils, TileLoadingState, TileLoadingBehavior, ChartFilters, DSSVisualizationThemeUtils){
        return {
            templateUrl: '/templates/dashboards/insights/web_app/web_app_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '=',
                editable: '=',
                filters: '<',
                filtersParams: '<',
                theme: '<'
            },
            link: function($scope, element, attrs){
                let iframe;
                $scope.element = element;
                $scope.StateUtils = StateUtils;

                $scope.iframe;

                $controller('WebAppViewCommon', { $scope: $scope });

                $scope.loading = false;
                $scope.loaded = false;
                $scope.error = null;

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    const app = {
                        projectKey: $scope.insight.projectKey,
                        id: $scope.insight.params.webAppSmartId
                    };
                    const promises = [];

                    promises.push($scope.getViewURL(app).then(function(url) {
                        $scope.iFrameUrl = url;
                    }));

                    promises.push(DataikuAPI.webapps.getTrustedCodeReport($scope.insight.projectKey, $scope.insight.params.webAppSmartId).spinner(false)
                        .success(trustedCodeReport => {
                            $scope.trustedCodeReport = trustedCodeReport;
                        }));

                    Promise.allSettled(promises).then(function() {
                        DashboardUtils.setLoaded.bind([$scope, resolve])();
                        const timeoutInSeconds = Math.min($scope.tile.tileParams.loadTimeoutInSeconds, 240);
                        if (timeoutInSeconds > 0) {
                            $timeout(function() {
                                $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.COMPLETE;
                            }, timeoutInSeconds * 1000);
                            return TileLoadingBehavior.DELAYED_COMPLETE;
                        }
                    });
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }

                function watchFilterChanges(iframe) {
                    if (!_.isNil($scope.unregisterFilterWatchGroup)) {
                        $scope.unregisterFilterWatchGroup();
                    }
                    $scope.unregisterFilterWatchGroup = $scope.$watchGroup(['filters', 'filtersParams'], function([newFilters, newFiltersParams], [oldFilters, oldFiltersParams]) {
                        if (newFilters !== oldFilters || newFiltersParams !== oldFiltersParams) {
                            sendFiltersToIframe(iframe);
                        }
                    });
                }

                function sendFiltersToIframe(iframe) {
                    if (!_.isNil($scope.filtersParams) && !_.isNil($scope.filters)) {
                        const message = {
                            type: 'filters',
                            filters: ChartFilters.getCleanFiltersForSharing($scope.filters),
                            filtersParams: ChartFilters.getCleanFiltersParamsForSharing($scope.filtersParams)
                        };
                        iframe.contentWindow.postMessage(message, '*');
                    }
                }

                $scope.onIframeLoaded = function(_iframe) {
                    if (!$scope.iFrameUrl) {
                        return;
                    }
                    iframe = _iframe;
                    sendFiltersToIframe(_iframe);
                    watchFilterChanges(_iframe);
                };

                $scope.$on('$destroy', function() {
                    if (!_.isNil($scope.unregisterFilterWatchGroup)) {
                        $scope.unregisterFilterWatchGroup();
                    }
                });
            }
        };
    });

    app.directive('webAppInsightView', function($controller, StateUtils, DataikuAPI, TaggableObjectsService){
        return {
            templateUrl: '/templates/dashboards/insights/web_app/web_app_view.html',
            scope: {
                insight: '=',
                onError: '&'
            },
            link: function($scope, element) {
                $scope.element = element;
                $scope.StateUtils = StateUtils;

                $controller('WebAppViewCommon', { $scope: $scope });
                const app = {
                    projectKey: $scope.insight.projectKey,
                    id: $scope.insight.params.webAppSmartId,
                    versionTag: $scope.insight.params.versionTag
                };

                $scope.onIframeLoaded = function(iframe) {
                    if(!$scope.iFrameUrl) {
                        return;
                    }
                    const content = iframe.contentDocument.body.innerText;
                    if (content.startsWith('{"errorType":') && $scope.onError) {
                        try {
                            $scope.onError({ error: JSON.parse(content) });
                        } catch (e) {
                            // Not a JSON error object
                        }
                    } else {
                        TaggableObjectsService.checkAndUpdateThumbnailData(
                            { ...app, type: 'WEB_APP' },
                            'body',
                            iframe.contentDocument
                        );
                    }
                };

                $scope.getViewURL(app).then(function(url) {
                    $scope.iFrameUrl = url;
                });

                DataikuAPI.webapps.getTrustedCodeReport($scope.insight.projectKey, $scope.insight.params.webAppSmartId)
                    .success(trustedCodeReport => $scope.trustedCodeReport = trustedCodeReport);
            }
        };
    });

    app.directive('webAppInsightTileParams', function(){
        return {
            templateUrl: '/templates/dashboards/insights/web_app/web_app_tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs){
                // Used when creating a new tile to correctly initialize the timeout value in editor.
                $scope.$watch('tileParams', function(nv) {
                    if (nv && nv.loadTimeoutInSeconds === undefined) {
                        nv.loadTimeoutInSeconds = 0;
                    }
                });
                if ($scope.tileParams.loadTimeoutInSeconds === undefined) {
                    $scope.tileParams.loadTimeoutInSeconds = 0;
                }
            }
        };
    });

    app.directive('webAppInsightCreateForm', function(DataikuAPI){
        return {
            templateUrl: '/templates/dashboards/insights/web_app/web_app_create_form.html',
            scope: true,
            link: function($scope, element, attrs){
                $scope.hook.defaultName = 'Webapp';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = nv.label;
                    $scope.insight.params.webAppType = nv.subtype;
                });
            }
        };
    });

})();

;
(function() {
    'use strict';

    angular.module('dataiku.dashboards').filter('mapFilterPanelPositionToDirection', function(DashboardFilterPanelService) {
        return function(position) {
            return DashboardFilterPanelService.mapPanelPositionToDirection(position);
        };
    });
})();

;
(function() {
    'use strict';
    /** @typedef {import('../../simple_report/types').GeneratedSources.ChartFilter} ChartFilter */
    /** @typedef {import('../../simple_report/types').GeneratedSources.DashboardPage} DashboardPage */

    const app = angular.module('dataiku.dashboards');

    /**
     * Everything to store and manipulate dashboard filters.
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.factory('DashboardFilters', function(CHART_FILTERS, FILTERABLE_INSIGHT_TYPES, DashboardFiltersUrlParams, ChartFilterUtils, FiltersPanelPosition, ActivityIndicator, ChartTypeChangeHandler, ChartFilters, FilterFacetsService, Logger, DataikuAPI, ChartUsableColumns, FilterPanelContext, $stateParams, CachedAPICalls, DashboardPageUtils, TileUtils) {
        const FILTERS_ON_MSG = 'Data is filtered';
        const SAMPLING_PROPS = ['refreshableSelection', 'engineType'];
        const MAX_DATA_BYTES = -1; // default to no memory limit (for sampling)
        const filtersSourceTypes = Object.freeze({ DATASET: 'dataset', TILE: 'tile' });

        /** @type {Map<string, ChartFilter[]>} */
        const globalFiltersByPageId = new Map();
        /** @type {Map<string, boolean>} */
        const enableCrossFiltersByPageId = new Map();
        /** @type {Map<string, boolean>} */
        const isFilterPanelDeactivatedByPageId = new Map();
        let insightsMap; // get all insights on dashboard mapped by id

        const lowerCaseStatusByDatasetKey = new Map(); // store the lower case status of each dataset
        const lowerCaseStatusRequestsQueues = new Map(); // used to handle simultaneous requests to fetch the lower case status of the same new dataset
        let datasetTypes;


        function getSamplingProps(src) {
            if (!src) {
                return {};
            }
            const samplingProps = {};
            for (const prop of SAMPLING_PROPS) {
                if (!src[prop]) {
                    continue;
                }
                if (prop === 'refreshableSelection') {
                    const newSelection = angular.copy(src[prop]);
                    delete newSelection._refreshTrigger; // ignore front internal prop
                    samplingProps[prop] = { ...samplingProps[prop], ...newSelection };
                } else {
                    samplingProps[prop] = angular.copy(src[prop]);
                }
            }
            return samplingProps;
        }

        function getSamplingRefreshTrigger(src) {
            if (!src || src.refreshableSelection == null) {
                return null;
            }
            return src.refreshableSelection._refreshTrigger;
        }

        function resetAllDashboardPagesFilterPanelStates() {
            globalFiltersByPageId.clear();
            isFilterPanelDeactivatedByPageId.clear();
            enableCrossFiltersByPageId.clear();
        }

        function resetGlobalFilters(pageId) {
            globalFiltersByPageId.delete(pageId);
        }

        function setGlobalFilters(filters, pageId) {
            if (filters !== globalFiltersByPageId.get(pageId) && pageId) {
                globalFiltersByPageId.set(pageId, filters);
            }
        }

        function getGlobalFilters(pageId) {
            return globalFiltersByPageId.get(pageId);
        }

        function getActiveGlobalFilters(pageId) {
            if (getIsFilterPanelDeactivated(pageId)) {
                return [];
            }
            return globalFiltersByPageId.get(pageId);
        }

        function setInsightsMap(newInsightsMap) {
            insightsMap = newInsightsMap;
        }

        function setCrossFilterStatusToPageId(pageId, enableCrossFilters) {
            enableCrossFiltersByPageId.set(pageId, enableCrossFilters);
        }

        function getSlideFilterableTiles(dashboardPageId, dashboardPages) {
            const filterableTiles = [];
            const dashboardPageIdx = findDashboardPageIdx(dashboardPageId, dashboardPages);
            if (dashboardPageIdx > -1 && insightsMap) {
                TileUtils.getFlattenTileList(dashboardPages[dashboardPageIdx].grid.tiles).forEach(tile => {
                    if (FILTERABLE_INSIGHT_TYPES.includes(tile.insightType)) {
                        const mappedInsight = insightsMap[tile.insightId];
                        if (!mappedInsight) {
                            return;
                        }
                        filterableTiles.push({
                            id: mappedInsight.id,
                            name: tile.titleOptions.title ? tile.titleOptions.title : mappedInsight.name,
                            type: mappedInsight.type,
                            datasetSmartName: mappedInsight.params.datasetSmartName,
                            refreshableSelection: mappedInsight.params.refreshableSelection,
                            script: mappedInsight.params.shakerScript,
                            engineType: mappedInsight.params.engineType
                        });
                    }
                });
            }

            return filterableTiles;
        }

        function findDashboardPageIdx(dashboardPageId, dashboardPages) {
            // new page does not have an id until it's saved, its temporary id is its index
            if (dashboardPages[dashboardPageId] && !dashboardPages[dashboardPageId].id) {
                return dashboardPageId;
            } else {
                return dashboardPages.findIndex(page => page.id === dashboardPageId);
            }
        }

        /**
         * Checks if the filter is actually filtering (for example a filter set to exclude nothing does nothing).
         *
         * @param {ChartFilter} filter
         * @param {DashboardPage.DashboardPageFiltersParams} filtersParams
         * @returns
         */
        function isFilterEffective(filter, filtersParams) {
            if (ChartFilterUtils.isAlphanumericalFilter(filter)) {
                // If excludeOtherValues is true, even if all values are selected the filter can be effective (for example in a dashboard if the filter insight sample contains less values than in the dataset insight sample).
                return filter.excludeOtherValues || (Object.keys(filter.excludedValues || {}).length > 0);
            }
            if (ChartFilterUtils.isNumericalFilter(filter)) {
                /*
                 * In dashboards, the samples of the filters tile and of the filtered insight can be different.
                 * As a result, if the min and max values haven't been changed by the user, a sampled filter may still be effective.
                 */
                return areFiltersSampled(filtersParams) || filter.minValue != null ||filter.maxValue != null || !filter.includeEmptyValues;
            }
            if (ChartFilterUtils.isRelativeDateFilter(filter)) {
                return ChartFilterUtils.isRelativeDateFilterEffective(filter.dateFilterPart, filter.dateFilterOption, filter.includeEmptyValues);
            }
            if (ChartFilterUtils.isExplicitFilter(filter)) {
                return true;
            }
            if (ChartFilterUtils.isCustomFilter(filter)) {
                return filter.selectedValues && Object.keys(filter.selectedValues).length || filter.excludedValues && Object.keys(filter.excludedValues).length;
            }
            return false;
        }

        function areFiltersSampled(filtersParams) {
            if (filtersParams.refreshableSelection == null) {
                return false;
            }
            if (filtersParams.refreshableSelection.selection != null && filtersParams.refreshableSelection.selection.filter != null && filtersParams.refreshableSelection.selection.filter.enabled) {
                return true;
            }
            return filtersParams.refreshableSelection.selection.samplingMethod !== 'FULL';
        }

        function hasGlobalFiltersApplied(pivotResponse, appliedFilters) {
            const hasDashboardFiltersApplied = appliedFilters.some(filter => filter.isAGlobalFilter);
            if (!hasDashboardFiltersApplied) {
                return false;
            }
            /*
             * We do not check what each filter has filtered out as it's the result of a global query
             * so we consider dashboard filters are applied when there are dashboard filters and filtered out rows.
             * => It means there can be false positive for chart/dataset insight with internal filters with inefective dashboard filters applied.
             */
            if (pivotResponse.totalKeptRows != null && pivotResponse.totalRows != null) {
                return pivotResponse.totalRows > pivotResponse.totalKeptRows;
            } else if (pivotResponse.beforeFilterRecords != null && pivotResponse.afterFilterRecords != null) {
                return pivotResponse.beforeFilterRecords > pivotResponse.afterFilterRecords;
            }
            return hasDashboardFiltersApplied;
        }

        function isSameSampling(tileSampling, filterSampling) {
            // Exclude check of backend internal properties
            const excludedProperties = ['useMemTable', 'maxRecordsForDisplay', 'maxStoredBytes', 'timeout'];

            for (const property in { ...tileSampling, ...filterSampling }) {
                if (excludedProperties.includes(property)) {
                    continue;
                }
                if (!angular.equals(filterSampling[property], tileSampling[property])) {
                    return false;
                }
            }
            return true;
        }

        function canCrossFilter(pageId) {
            return !!enableCrossFiltersByPageId.get(pageId);
        }

        function getInconsistentFilteringCause(tileData, filtersParams) {
            if (filtersParams == null){
                return '';
            }
            if (tileData.sourceDataset !== filtersParams.datasetSmartName) {
                return 'datasets are different';
            }
            // no sense on checking for samples if datasets are different
            if (!isSameSampling(tileData.sampleSettings.selection, filtersParams.refreshableSelection.selection)) {
                return 'samples are different';
            }
            return '';
        }

        function getUnusableFiltersMessage(unusableFilters, warningRootCause) {
            if (!unusableFilters || !unusableFilters.length) {
                return '';
            }
            const MAX_COLUMNS = 3; // display a maximum of 3 columns names in message
            const hasWarningRootCause = warningRootCause && warningRootCause.length > 0;
            const otherColumnsLength = unusableFilters.length - MAX_COLUMNS;
            const columnsNames = unusableFilters.slice(0, MAX_COLUMNS).map(uf => uf.columnName).join(', ');
            const isPluralColumns = unusableFilters.length > 1 ? 's' : '';
            const isPluralOthers = otherColumnsLength > 1 ? 's' : '';
            const hasDifferentReasons = unusableFilters.slice(1).some(uf => uf.reason !== unusableFilters[0].reason);
            return `${FILTERS_ON_MSG} - except for column${isPluralColumns} ${columnsNames}\
                    ${otherColumnsLength > 0 ? ` + ${otherColumnsLength} other${isPluralOthers}` : ''} (\
                        ${hasDifferentReasons ? 'various reasons' : unusableFilters[0].reason}\
                        ${hasWarningRootCause ? `as ${warningRootCause}` : ''}\
                    )`;
        }

        function setFilteringMessages(tileData, filtersParams) {
            if (!tileData.filteringStatus.isInsightFiltered) {
                tileData.filteringStatus.warningMessage = '';
                tileData.filteringStatus.infoMessage = '';
                return;
            }
            const inconsistentFilteringCause = getInconsistentFilteringCause(tileData, filtersParams);
            if (!inconsistentFilteringCause.length && !tileData.unusableColumnsFilters.length) {
                tileData.filteringStatus.warningMessage = ''; // no warning message if no inconsistent filtering detected cause detected
                tileData.filteringStatus.infoMessage = FILTERS_ON_MSG;
            } else {
                tileData.filteringStatus.warningMessage = getUnusableFiltersMessage(tileData.unusableColumnsFilters, inconsistentFilteringCause);
                tileData.filteringStatus.infoMessage = tileData.filteringStatus.warningMessage.length > 0 ? ''
                    : `${FILTERS_ON_MSG} - risk of inconsistency (${inconsistentFilteringCause})`;
            }
        }

        function hasUsableColumn(usableColumns, filter, hasCaseInsensitiveColumnNames) {
            if (!usableColumns || !filter) {
                return false;
            }
            const usableColumnMap = new Map(usableColumns.map(usableColumn => [hasCaseInsensitiveColumnNames? usableColumn.column.toLowerCase() : usableColumn.column, usableColumn.type]));
            if (ChartFilterUtils.isExplicitFilter(filter)) {
                return filter.explicitConditions.every(condition => {
                    const filterColumnName = hasCaseInsensitiveColumnNames? condition.column.toLowerCase() : condition.column;
                    return usableColumnMap.has(filterColumnName) && ChartFilters.isExplicitConditionCompatibleWithColumnType(usableColumnMap.get(filterColumnName), condition);
                });
            } else {
                const filterColumnName = hasCaseInsensitiveColumnNames? filter.column.toLowerCase() : filter.column;
                return usableColumnMap.has(filterColumnName) && usableColumnMap.get(filterColumnName) === filter.columnType;
            }
        }

        function adaptPristineRangeFilter(filter) {
            if (!filter || !ChartFilterUtils.isNumericalFilter(filter)) {
                return filter;
            }
            // minValue and maxValue can have a value of null if they haven't been changed by the user, but because of sampling they may still affect the dataset so we fallback to the filters tile global min and max values.
            const minValue = filter.minValue != null ? filter.minValue : filter.$globalMinValue;
            const maxValue = filter.maxValue != null ? filter.maxValue : filter.$globalMaxValue;
            return {
                ...filter,
                minValue,
                maxValue
            };
        }

        function getUsableFilters(filters, usableColumns, hasCaseInsensitiveColumnNames, filtersParams) {
            if (usableColumns && usableColumns.length && filters && filters.length) {
                return filters.filter(filter => isFilterEffective(filter, filtersParams) && hasUsableColumn(usableColumns, filter, hasCaseInsensitiveColumnNames)).map(filter => adaptPristineRangeFilter(filter));
            }
            return [];
        }

        function getReason(filter, usableColumns) {
            if (filter && usableColumns && usableColumns.length) {
                const columnFound = usableColumns.find(e => e.column === filter.column);
                if (columnFound && columnFound.type !== filter.type) {
                    return 'type mismatch';
                }
            }
            return 'not found';
        }

        function getUnusableColumnFilters(filters, usableColumns) {
            if (usableColumns && usableColumns.length && filters && filters.length) {
                return filters
                    .filter(filter => !hasUsableColumn(usableColumns, filter))
                    .map(filter => ({ columnName: filter.column, reason: getReason(filter, usableColumns) }));
            }
            return [];
        }

        function findFiltersTileInPage(page) {
            if (page && page.grid && page.grid.tiles) {
                return findFiltersTileList(page.grid.tiles);
            }
        }

        function findFiltersTileList(tiles) {
            return tiles.find(tile => TileUtils.isFilterTile(tile));
        }

        /**
         * @param {DashboardPage} page
         * @returns {ChartFilter[]}
         */
        function getPageFilters(page) {
            return (page.filters == null) ? [] : page.filters.filter(filter => !filter.useMinimalUi);
        }

        function onApplyFiltersSuccess(tileData, filtersParams, response) {
            tileData.filteringStatus.isApplyingFilters = false;
            tileData.filteringStatus.isInsightFiltered = hasGlobalFiltersApplied(response, tileData.appliedFilters);
            setFilteringMessages(tileData, filtersParams);
        }

        function onApplyFiltersError(error, filteringStatus, loadingData) {
            filteringStatus.isApplyingFilters = false;

            if (!error || !error.data) {
                return;
            }
            // If the error contains a filtered out message, the insight is considered as filtered (empty state)
            filteringStatus.isInsightFiltered = error.data.message.includes('filtered out');


            if (!filteringStatus.isInsightFiltered && error.data.message.includes('To finish your chart')) {
                loadingData.unconfiguredRoutine();
            }
        }

        function applyFilters(filters, filtersParams, tileData, loadingData, hasCaseInsensitiveColumnNames) {
            try {
                if (!tileData.usableColumns) {
                    return;
                }

                // Remove non usable columns
                let matchingFilters = getUsableFilters(filters, tileData.usableColumns, hasCaseInsensitiveColumnNames, filtersParams);
                if (hasCaseInsensitiveColumnNames && matchingFilters.length > 0) {
                    matchingFilters = getFiltersWithCorrectCaseColumnNames(matchingFilters, tileData.usableColumns);
                }

                tileData.appliedFilters = angular.copy(matchingFilters);
                tileData.unusableColumnsFilters = getUnusableColumnFilters(filters, tileData.usableColumns);
                tileData.filteringStatus.isApplyingFilters = !!matchingFilters && !!matchingFilters.length;
                return tileData.applyFilters(matchingFilters, loadingData)
                    .then(data => tileData.parseResponse ? onApplyFiltersSuccess(tileData, filtersParams, tileData.parseResponse(data)) : data)
                    .catch((error) => {
                        onApplyFiltersError(error, tileData.filteringStatus, loadingData);
                        // Let the insight handle the error
                        throw error;
                    });
            } catch (error) {
                return Promise.reject(error);
            }
        }

        function getFiltersWithCorrectCaseColumnNames(matchingFilters, usableColumns) {
            return matchingFilters.map(filter => {
                if (ChartFilterUtils.isExplicitFilter(filter)) {
                    return {
                        ...filter,
                        explicitConditions: filter.explicitConditions.map(condition => ({
                            ...condition,
                            column: usableColumns.find(column => column.column.toLowerCase() === condition.column.toLowerCase()).column
                        }))
                    };
                } else {
                    return {
                        ...filter,
                        column: usableColumns.find(column => column.column.toLowerCase() === filter.column.toLowerCase()).column
                    };
                }
            });
        }

        /**
         * Get explorations filters from charts filters
         * @param  {List<ChartFilter.java>} usableFilters - Filters to convert from
         * @return {List<FilterElement.java>}
         */
        function getDashboardExplorationFilters(usableFilters) {
            if (!usableFilters || !usableFilters.length) {
                return [];
            }
            const explorationFilters = [];
            usableFilters.forEach(filter => {
                const facet = {
                    type: 'facet',
                    column: filter.column,
                    columnType: filter.columnType,
                    currentMode: filter.filterType.split('_')[0], // remove "_FACET" part
                    sort: 'count',
                    minValue: filter.minValue,
                    maxValue: filter.maxValue,
                    isAGlobalFilter: true,
                    active: filter.active,
                    includeEmptyValues: filter.includeEmptyValues
                };

                if (filter.explicitConditions) {
                    facet.explicitConditions = filter.explicitConditions;
                }
                if (filter.explicitExclude) {
                    facet.explicitExclude = filter.explicitExclude;
                }

                if (filter.excludeOtherValues) {
                    facet.selectedValues = filter.selectedValues;
                } else {
                    facet.excludedValues = filter.excludedValues;
                }

                if (filter.dateFilterType === CHART_FILTERS.DATE_TYPES.RANGE) {
                    facet.dateFilterType = filter.dateFilterType;
                    facet.timezone = 'UTC';
                } else if (filter.dateFilterType === CHART_FILTERS.DATE_TYPES.RELATIVE) {
                    facet.dateFilterType = filter.dateFilterType;
                    facet.dateFilterPart = filter.dateFilterPart;
                    facet.dateFilterRelativeOption = filter.dateFilterOption;
                    facet.dateFilterRelativePreset = filter.dateFilterRelativePreset;
                } else if (filter.dateFilterType === CHART_FILTERS.DATE_TYPES.PART) {
                    facet.dateFilterType = filter.dateFilterType;
                    facet.dateFilterPart = filter.dateFilterPart;
                }
                const alphanumFilter = {
                    isAGlobalFilter: filter.isAGlobalFilter,
                    type: 'alphanum',
                    column: filter.column,
                    params: {
                        mode: (filter.customOptions && filter.customOptions.matchingMode) || 'FULL_STRING',
                        normalization: (filter.customOptions && filter.customOptions.normalizationMode) || 'EXACT',
                        ...(filter.customOptions && filter.customOptions.decimalPlaces ? { decimalPlaces: filter.customOptions.decimalPlaces } : {})
                    }
                };
                if (filter.explicitConditions) {
                    alphanumFilter.explicitConditions = filter.explicitConditions;
                }
                if (filter.explicitExclude) {
                    alphanumFilter.explicitExclude = filter.explicitExclude;
                }
                if (filter.selectedValues) {
                    alphanumFilter.selectedValues = Object.keys(filter.selectedValues);
                }
                if (filter.excludedValues) {
                    alphanumFilter.excludedValues = Object.keys(filter.excludedValues);
                }
                explorationFilters.push({
                    column: filter.column,
                    type: 'columnFilter',
                    currentMode: filter.filterSelectionType === 'CUSTOM' ? 'SIMPLE_ALPHANUM' : 'FACET',
                    active: filter.active,
                    facet,
                    alphanumFilter
                });
            });
            return explorationFilters;
        }

        /**
         * Computes the initial state of a dashboard slide filters, store them in the service and returns them.
         * @param {DashboardPage} page
         * @param {Record<string, string>} urlQueryParams
         * @param {any[]} insights
         * @returns {Promise<FrontendChartFilter[]>}
         */
        function initPageFilters(page, urlQueryParams, insights) {
            const filtersAreInSearchParams = DashboardFiltersUrlParams.areFiltersInSearchParams(urlQueryParams);
            const previousFilters = getGlobalFilters(page.id) || getPageFilters(page);

            if (filtersAreInSearchParams) {
                return getFilterPanelUsableColumns(page.filtersParams, urlQueryParams, insights)
                    .then(usableColumns => applyUrlFiltersOnPageFilters(urlQueryParams, previousFilters, usableColumns))
                    .then((filters) => {
                        if (filters) {
                            setGlobalFilters(filters, page.id);
                            return { activeFilters: filters, filters };
                        }
                        return [];
                    });
            }
            setGlobalFilters(previousFilters, page.id);
            return Promise.resolve({
                activeFilters: getActiveGlobalFilters(page.id) || previousFilters,
                filters: previousFilters
            });
        }

        /**
         * @param {any[]} insights
         * @returns {{ datasetSmartName: string, engineType: string }}
         */
        function findMostUsedDatasetWithEngine(insights) {
            const filterableInsights = (insights || []).filter(({ type }) => type === 'chart' || type === 'dataset_table');
            if (filterableInsights.length == 0) {
                return null;
            }
            const occurrencesByDatasetName = new Map();
            /** @type {{ datasetSmartName: string, engineType: string }} */
            let mostUsedDatasetWithEngine = { datasetSmartName: filterableInsights[0].params.datasetSmartName, engineType: filterableInsights[0].params.engineType || 'LINO' };
            let maxCount = 1;

            for (const insight of filterableInsights) {
                const currentDatasetOccurrences = occurrencesByDatasetName.get(insight.params.datasetSmartName);
                const newDatasetOccurrences = (currentDatasetOccurrences || 0) + 1;
                occurrencesByDatasetName.set(insight.params.datasetSmartName, newDatasetOccurrences);

                if (newDatasetOccurrences > maxCount) {
                    mostUsedDatasetWithEngine = { datasetSmartName: insight.params.datasetSmartName, engineType: insight.params.engineType || 'LINO' };
                    maxCount = newDatasetOccurrences;
                }
            }
            return mostUsedDatasetWithEngine;
        }

        /**
         * Retrieve the usable columns of the filter panel dataset
         * @param {Object} filtersParams
         * @param {Record<string, string>} urlQueryParams
         * @param {any[]} insights
         * @returns {Promise<UsableColumn[]>}
         */
        function getFilterPanelUsableColumns(filtersParams, urlQueryParams, insights) {
            if (DashboardFiltersUrlParams.shouldUseV1Strategy(urlQueryParams)) {
                // When filters from query parameters v1 are present, a dataset is required for column type resolution of the parsed filters. If the filter panel does not contain one, we utilize the most frequently used dataset from the dashboard.
                const datasetParams = filtersParams.datasetSmartName ?
                    { datasetSmartName: filtersParams.datasetSmartName, engineType: filtersParams.engineType }:
                    findMostUsedDatasetWithEngine(insights);
                if (datasetParams == null) {
                    // No dataset found neither in the filter panel nor in the dashbaord tiles
                    return Promise.resolve([]);
                }
                return getDatasetUsableColumns(datasetParams.datasetSmartName, datasetParams.engineType);
            } else {
                /*
                 * If there's no dataset in the filter panel for query parameters v2, there's no problem. We can still resolve the column types thanks to the v2 syntax.
                 * If there is a dataset in the filter panel, we check if the cross-filters from the URL belong to this dataset thanks to the usable columns. If they do, we consider them to compute the filter facets.
                 */
                if (filtersParams.datasetSmartName == null) {
                    // No dataset found in the filter panel
                    return Promise.resolve([]);
                } else {
                    return getDatasetUsableColumns(filtersParams.datasetSmartName, filtersParams.engineType);
                }
            }
        }

        /**
         * @param {string} datasetSmartName
         * @returns {Promise<UsableColumn[]>}
         */
        function getDatasetUsableColumns(datasetSmartName, engineType) {
            const dataSpec = getDataSpec($stateParams.projectKey, datasetSmartName, engineType);
            return fetchColumnsSummary(dataSpec.datasetProjectKey, dataSpec, { datasetSmartName })
                .then(({ usableColumns }) => usableColumns);
        }

        /**
         * @param {string} urlQueryParams
         * @param {FrontendChartFilter[]} filters
         * @param {UsableColumn[]} usableColumns
         * @returns {FrontendChartFilter[]}
         */
        function applyUrlFiltersOnPageFilters(urlQueryParams, filters, usableColumns) {
            const parsedFiltersData = DashboardFiltersUrlParams.getFiltersFromSearchParams(urlQueryParams, filters, usableColumns);
            const newFilters = parsedFiltersData.filters.length > 0 ? parsedFiltersData.filters : filters;
            if (parsedFiltersData.error) {
                ActivityIndicator.warning(`Some filters from URL are not applied (${parsedFiltersData.error})`, 5000);
            }
            return newFilters;
        }

        function setIsFilterPanelDeactivated(pageId, isDeactivated) {
            isFilterPanelDeactivatedByPageId.set(pageId, isDeactivated);
        }

        function getIsFilterPanelDeactivated(pageId) {
            return !!isFilterPanelDeactivatedByPageId.get(pageId);
        }

        function acceptColumnAsFilter(data) {
            const ret = ChartTypeChangeHandler.stdAggregatedAcceptDimension(data);
            return ret.accept && data.type !== 'GEOMETRY' && data.type !== 'GEOPOINT';
        }

        /**
         * Compute and return DataSpec (describes the data source for charts request)
         * @return {DataSpec.java}
         */
        function getDataSpec(projectKey, datasetSmartName, engineType, refreshableSelection, shaker) {
            const resolvedDataset = resolveDatasetFullName(datasetSmartName, projectKey);

            if (refreshableSelection == null) {
                refreshableSelection = {
                    _refreshTrigger: Date.now(),
                    selection: {
                        samplingMethod: 'FULL'
                    }
                };
            }

            return {
                datasetProjectKey: resolvedDataset.projectKey,
                datasetName:  resolvedDataset.datasetName,
                sampleSettings : refreshableSelection,
                engineType: engineType || 'LINO',
                ...(shaker == null ? {
                    copyScriptFromExplore: true,
                    copySelectionFromScript: false
                } : {
                    script: { ...shaker, origin: 'DATASET_EXPLORE' }
                })
            };
        };

        /**
         * Prepare request and call get-pivot-response
         * @param  {List<ChartFilters>} filters                         - filters to use for this request
         * @param  {Object}             samplingParams                  - sampling & engine configuration to use for this request
         * @param  {Object}             loadingData                     - callbacks to use for success & errors
         * @param  {Boolean}            samplingHasChanged              - flag to tell if the sampling has changed
         * @param  {MonoFuture}         filtersPivotRequestMonoFuture   - mono future to use for the request
         */
        function executeFiltersPivotRequest(projectKey, summary, filters, dataSpec, filtersPivotRequestMonoFuture) {
            try {
                const requestParams = {};
                /*
                 * if ($scope.insight.$chart) {
                 *     requestParams.maxDataBytes = $scope.insight.$chart.maxDataBytes;
                 * }
                 */
                const filtersWithMatchingColumn = filters.filter(filter => !filter.useMinimalUi || filter.hasMatchingColumn);
                if (!filtersWithMatchingColumn.length) {
                    return Promise.resolve([]);
                }
                const request = ChartFilters.buildFiltersRequest(filtersWithMatchingColumn, requestParams);

                const requestedSampleId = summary.requiredSampleId;
                return FilterFacetsService.getFilterFacets(projectKey, dataSpec, request, requestedSampleId, filtersPivotRequestMonoFuture);
            } catch (error) {
                Logger.info('Not executing filter request', error);
            }
        }

        function getShaker(projectKey, datasetSmartName) {
            return DataikuAPI.explores.get(projectKey, datasetSmartName).noSpinner()
                .then(data => {
                    return data.script || data.data.script;
                });
        }

        function fetchColumnsSummary(projectKey, dataSpec, filtersParams) {
            return DashboardPageUtils.getColumnSummary(projectKey, dataSpec)
                .then(data => {
                    let usableColumnsForEngine = ChartUsableColumns.makeUsableColumns(data, getCacheableContext(projectKey, filtersParams.datasetSmartName));
                    if (usableColumnsForEngine && usableColumnsForEngine.length) {
                        usableColumnsForEngine = usableColumnsForEngine.filter(acceptColumnAsFilter);
                        usableColumnsForEngine.forEach(filter => ChartFilters.autocompleteFilter(filter, usableColumnsForEngine, true));
                    }

                    return { summary: data, usableColumns: usableColumnsForEngine };
                });
        }

        function getCacheableContext(datasetProjectKey, datasetName) {
            return {
                datasetProjectKey,
                datasetName,
                context: FilterPanelContext.FILTER_PANEL
            };
        }

        function canFilter(editable, filters, crossFiltersEnabled) {
            return editable || (filters && filters.length || crossFiltersEnabled);
        }

        function isFiltersPanelDocked(panelPosition) {
            return panelPosition !== FiltersPanelPosition.TILE;
        }

        /**
         * Returns the dataset types
         * @returns {Promise<DatasetType[]>}
         */
        function getDatasetTypes() {
            if (!_.isNil(datasetTypes)) {
                return Promise.resolve(datasetTypes);
            }
            return CachedAPICalls.datasetTypes.then(data => {
                datasetTypes = data;
                return datasetTypes;
            });

        }

        function hasCaseInsensitiveColumnNames(projectKey, datasetName, contextProjectKey) {
            const datasetKey = `${projectKey}/${contextProjectKey}/${datasetName}`;
            if (lowerCaseStatusByDatasetKey.has(datasetKey)) {
                return Promise.resolve(lowerCaseStatusByDatasetKey.get(datasetKey));
            }
            return new Promise((resolve, reject) => {
                if (lowerCaseStatusRequestsQueues.has(datasetKey)) {
                    lowerCaseStatusRequestsQueues.get(datasetKey).push( { queuedResolve: resolve, queuedReject: reject } );
                } else {
                    lowerCaseStatusRequestsQueues.set(datasetKey, [{ queuedResolve: resolve, queuedReject: reject }]);
                    DataikuAPI.datasets.get(projectKey, datasetName, contextProjectKey).then(({ data }) => {
                        getDatasetTypes().then((datasetTypes) => {
                            const hasCaseInsensitiveColumnNamesStatus = datasetTypes[data.type] && datasetTypes[data.type]['hasCaseInsensitiveColumnNames'];
                            lowerCaseStatusByDatasetKey.set(datasetKey, hasCaseInsensitiveColumnNamesStatus);
                            lowerCaseStatusRequestsQueues.get(datasetKey).forEach(({ queuedResolve }) =>
                                queuedResolve(hasCaseInsensitiveColumnNamesStatus));
                            lowerCaseStatusRequestsQueues.delete(datasetKey);
                        });
                    }).catch(error => {
                        // reject all promises from the queue
                        lowerCaseStatusRequestsQueues.get(datasetKey).forEach(({ queuedReject }) =>
                            queuedReject(error));
                        lowerCaseStatusRequestsQueues.delete(datasetKey);
                    });
                }
            });
        }

        function flushDatasetLowerCaseStatus() {
            lowerCaseStatusByDatasetKey.clear();
        }

        return {
            SAMPLING_PROPS,
            MAX_DATA_BYTES,
            FILTERS_ON_MSG,
            filtersSourceTypes,
            getSamplingProps,
            getSamplingRefreshTrigger,
            resetGlobalFilters,
            setGlobalFilters,
            getGlobalFilters,
            getActiveGlobalFilters,
            hasGlobalFiltersApplied,
            getUsableFilters,
            getUnusableColumnFilters,
            findFiltersTileInPage,
            findFiltersTileList,
            getPageFilters,
            applyFilters,
            onApplyFiltersSuccess,
            setInsightsMap,
            getSlideFilterableTiles,
            getDashboardExplorationFilters,
            initPageFilters,
            isSameSampling,
            setCrossFilterStatusToPageId,
            canCrossFilter,
            setIsFilterPanelDeactivated,
            getIsFilterPanelDeactivated,
            acceptColumnAsFilter,
            getDataSpec,
            executeFiltersPivotRequest,
            fetchColumnsSummary,
            getShaker,
            canFilter,
            isFiltersPanelDocked,
            hasCaseInsensitiveColumnNames,
            flushDatasetLowerCaseStatus,
            resetAllDashboardPagesFilterPanelStates
        };
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Utility functions for dashboard pages.
     */
    app.factory('DashboardPageUtils', function(TileLoadingState, FILTERABLE_INSIGHT_TYPES, DataikuAPI, ColorUtils) {
        const isInExport = getCookie('dku_graphics_export') === 'true';

        const ADJACENT_SQUARES_TO_LOAD_AROUND_VIEWPORT = 3;

        const columnSummaryByProjectKeyAndDataSpec = new Map(); // store the column summary for each pair of {projectKey, dataSpec}
        const columnSummaryRequestsQueues = new Map(); // used to handle simultaneous requests to fetch the column summary of the same pair of {projectKey, dataSpec}

        function isTileAdjacent(tile, { top, bottom }) {
            if (_.isNil(top) || _.isNil(bottom) || _.isNil(tile) || _.isNil(tile.box) || _.isNil(tile.box.top) || _.isNil(tile.box.height)) {
                return false;
            }
            const tileBottom = tile.box.top + tile.box.height;
            const tileTop = tile.box.top;
            return (
                (tileBottom >= top - ADJACENT_SQUARES_TO_LOAD_AROUND_VIEWPORT && tileBottom <= top) ||
                (tileTop >= bottom && tileTop <= bottom + ADJACENT_SQUARES_TO_LOAD_AROUND_VIEWPORT)
            );

        }

        function getViewportTopAndBottomCoords(visibleTileIndex, tileById) {
            return Object.entries(visibleTileIndex).reduce((acc, [tileId, isVisible]) => {
                if (isVisible) {
                    const tile = tileById[tileId];
                    if (_.isNil(tile) || _.isNil(tile.box)) {
                        return;
                    }
                    let { top, bottom } = acc;
                    if (_.isNil(top) || _.isNil(bottom)) {
                        return { top: tile.box.top, bottom: tile.box.top + tile.box.height };
                    } else {
                        const tileBottom = tile.box.top + tile.box.height;
                        const tileTop = tile.box.top;

                        if (tileBottom > bottom) {
                            bottom = tileBottom;
                        }
                        if (tileTop < top) {
                            top = tileTop;
                        }
                        return { top, bottom };
                    }
                }
                return acc;
            }, { top: null, bottom: null });
        }

        function shadeColor(color, delta) {
            let R = parseInt(color.substring(1, 3), 16);
            let G = parseInt(color.substring(3, 5), 16);
            let B = parseInt(color.substring(5, 7), 16);

            R = parseInt(R + delta);
            G = parseInt(G + delta);
            B = parseInt(B + delta);

            R = R < 255 ? R : 255;
            G = G < 255 ? G : 255;
            B = B < 255 ? B : 255;

            const RR = R.toString(16).length == 1 ? '0' + R.toString(16) : R.toString(16);
            const GG = G.toString(16).length == 1 ? '0' + G.toString(16) : G.toString(16);
            const BB = B.toString(16).length == 1 ? '0' + B.toString(16) : B.toString(16);

            return '#' + RR + GG + BB;
        }

        const svc = {
            isTileVisible: (tileId, hook) => {
                if (isInExport || _.isNil(hook)) {
                    return true;
                } else {
                    return !!(hook.visibleStates[tileId]);
                }
            },
            isTileVisibleOrAdjacent: (tileId, hook) => {
                if (isInExport || _.isNil(hook) || (_.isNil(hook.visibleStates) && _.isNil(hook.adjacentStates))) {
                    return true;
                } else {
                    return !!(hook.visibleStates[tileId] || hook.adjacentStates[tileId]);
                }
            },
            getTilesToLoadOrderedList: (tileLoadingPromiseByTileId, hook, tileById) => {
                const tilesToLoad = svc.getTilesToLoad(tileLoadingPromiseByTileId, hook);
                /*
                 * Load tiles in the following order of priority:
                 * 1. Visible over adjacent
                 * 2. Unfilterable tiles first
                 * 3. Tiles at the top-most position (smallest top value)
                 * 4. Tiles at the left-most position (smallest left value)
                 */
                tilesToLoad.sort((tileId1, tileId2) => {
                    const isTile1Visible = svc.isTileVisible(tileId1, hook);
                    const isTile2Visible = svc.isTileVisible(tileId2, hook);
                    const isTile1Filterable = FILTERABLE_INSIGHT_TYPES.includes(tileById[tileId1].insightType);
                    const isTile2Filterable = FILTERABLE_INSIGHT_TYPES.includes(tileById[tileId2].insightType);
                    const tile1TopPosition = tileById[tileId1].box.top;
                    const tile2TopPosition = tileById[tileId2].box.top;
                    const tile1LeftPosition = tileById[tileId1].box.left;
                    const tile2LeftPosition = tileById[tileId2].box.left;

                    if (isTile1Visible && !isTile2Visible) {
                        return -1;
                    }
                    if (!isTile1Visible && isTile2Visible) {
                        return 1;
                    }

                    if (isTile1Filterable && !isTile2Filterable) {
                        return 1;
                    }
                    if (!isTile1Filterable && isTile2Filterable) {
                        return -1;
                    }

                    if (tile1TopPosition < tile2TopPosition) {
                        return -1;
                    }
                    if (tile1TopPosition > tile2TopPosition) {
                        return 1;
                    }

                    if (tile1LeftPosition < tile2LeftPosition) {
                        return -1;
                    }
                    if (tile1LeftPosition > tile2LeftPosition) {
                        return 1;
                    }

                    return 0;
                });
                return tilesToLoad;
            },
            getIsTileAdjacentToVisibleTilesByTileId: (visibleTileIndex, tileById = {}) => {
                const viewportTopAndBottomCoords = getViewportTopAndBottomCoords(visibleTileIndex, tileById);
                return Object.fromEntries(Object.entries(visibleTileIndex).map(([tileId, isVisible]) => {
                    const tile = tileById[tileId];
                    return [tileId, !isVisible && isTileAdjacent(tile, viewportTopAndBottomCoords)];
                }));
            },
            areAllVisibleAndAdjacentInsightsLoaded: (tiles, hook) => {
                return !tiles.some(tile =>
                    tile.tileType === 'INSIGHT' && svc.getLoadingState(tile.$tileId, hook) !== TileLoadingState.COMPLETE && svc.isTileVisibleOrAdjacent(tile.$tileId, hook));
            },
            getLoadingState: (tileId, hook) => {
                return (tileId in hook.loadStates) ? hook.loadStates[tileId] : TileLoadingState.WAITING;
            },
            hasLoadingTiles: (tileLoadingPromiseByTileId, hook) => {
                return Object.keys(tileLoadingPromiseByTileId).some(tileId =>
                    svc.getLoadingState(tileId, hook) !== TileLoadingState.COMPLETE && svc.isTileVisibleOrAdjacent(tileId, hook));
            },
            getTilesToLoad: (tileLoadingPromiseByTileId, hook) => {
                return Object.keys(tileLoadingPromiseByTileId).filter((tileId) => {
                    return svc.getLoadingState(tileId, hook) === TileLoadingState.WAITING || svc.getLoadingState(tileId, hook) === TileLoadingState.WAITING_FOR_RELOAD;
                });
            },
            getNumberOfLoadingTiles: (hook) => {
                return Object.keys(hook.loadStates).map(tileId => svc.getLoadingState(tileId, hook)).filter(state =>
                    state === TileLoadingState.LOADING).length;
            },
            getVisibleTilesToLoadOrReloadPromiseByTileId: (hook) => {
                return Object.fromEntries(Object.entries(hook.loadStates)
                    .filter(([key, loadState]) => svc.isTileVisibleOrAdjacent(key, hook) &&
                            (loadState === TileLoadingState.WAITING || loadState === TileLoadingState.WAITING_FOR_RELOAD))
                    .map(([key]) => [key, hook.loadStates[key] === TileLoadingState.WAITING ? hook.loadPromises[key] : hook.reloadPromises[key]]));
            },
            getColumnSummary: (projectKey, dataSpec) => {
                const key = `${projectKey}/${JSON.stringify(dataSpec)}`;
                if (columnSummaryByProjectKeyAndDataSpec.has(key)) {
                    return Promise.resolve(columnSummaryByProjectKeyAndDataSpec.get(key));
                }
                return new Promise((resolve, reject) => {
                    if (columnSummaryRequestsQueues.has(key)) {
                        columnSummaryRequestsQueues.get(key).push({ queuedResolve: resolve, queuedReject: reject });
                    } else {
                        columnSummaryRequestsQueues.set(key, [{ queuedResolve: resolve, queuedReject: reject }]);
                        DataikuAPI.shakers.charts.getColumnsSummary(projectKey, dataSpec)
                            .noSpinner()
                            .success(data => {
                                columnSummaryByProjectKeyAndDataSpec.set(key, data);
                                columnSummaryRequestsQueues.get(key).forEach(({ queuedResolve }) => queuedResolve(data));
                                columnSummaryRequestsQueues.delete(key);
                            }).error((data, status, headers, config, statusText, xhrStatus) => {
                                columnSummaryRequestsQueues.get(key).forEach(({ queuedReject }) => {
                                    queuedReject({ data, status, headers, config, statusText, xhrStatus });
                                });
                                columnSummaryRequestsQueues.delete(key);
                            });
                    }
                });
            },
            flushCachedColumnSummaries: () => {
                columnSummaryByProjectKeyAndDataSpec.clear();
            },
            getGridColor: (backgroundColor, backgroundOpacity) => {
                if (backgroundColor == null) {
                    backgroundColor = '#ffffff';
                }
                let contrastColor = shadeColor(backgroundColor, +20);
                if (contrastColor === '#ffffff') {
                    contrastColor = shadeColor(backgroundColor, -5);
                }
                return ColorUtils.getHexWithAlpha(contrastColor, backgroundOpacity);
            },
            positionResizeHandles: function(tileWrapper, tileSpacing) {
                if (tileWrapper == null) {
                    return;
                }
                const tileMargin = tileSpacing / 2;
                const handleMargin = `${tileMargin - 3}px`;
                tileWrapper.find('.ui-resizable-nw').css({
                    left: handleMargin,
                    top: handleMargin
                });

                tileWrapper.find('.ui-resizable-ne').css({
                    top: handleMargin,
                    right: handleMargin
                });

                tileWrapper.find('.ui-resizable-sw').css({
                    bottom: handleMargin,
                    left: handleMargin
                });

                tileWrapper.find('.ui-resizable-se').css({
                    bottom: handleMargin,
                    right: handleMargin
                });

                tileWrapper.find('.ui-resizable-n').css({
                    top: handleMargin,
                    left: '50%'
                });

                tileWrapper.find('.ui-resizable-s').css({
                    bottom: handleMargin,
                    left: '50%'
                });

                tileWrapper.find('.ui-resizable-w').css({
                    left: handleMargin,
                    top: '50%'
                });

                tileWrapper.find('.ui-resizable-e').css({
                    right: handleMargin,
                    top: '50%'
                });
            },
            isOverflowingVertically: function(element) {
                return element.scrollHeight > element.clientHeight;
            },
            getTileElementFromTileId: function(container, tileId) {
                return container.find('[data-id = ' + tileId+ ']');
            },
            getTileWrappers: function(container) {
                return container.find('.tile-wrapper');
            },
            getTileWrapperFromElement: function(element) {
                return $(element).closest('.tile-wrapper');
            },
            getTileByElement(tiles, el) {
                const tileId = $(el).data('id');
                for (let i = 0; i< tiles.length; i++) {
                    const tile = tiles[i];
                    if (tileId == tile.$tileId) {
                        return tile;
                    }
                }
            },
            isPerformingDradAngDropWithinTile(element) {
                return element.closest('[dashboard-no-drag]') != null;
            },
            computeGridContainerBackgroundColor(showGrid, editable, backgroundColor, gridColor) {
                return showGrid && editable ? gridColor : backgroundColor ? backgroundColor : '#ffffff';
            },
            getNextPageIndex: function(dashboard, currentPageIdx) {
                if (currentPageIdx < dashboard.pages.length - 1) {
                    return currentPageIdx + 1;
                } else if (dashboard.circularNavigation) {
                    return 0;
                }
                return currentPageIdx;
            },
            getPreviousPageIndex: function(dashboard, currentPageIdx) {
                if (currentPageIdx > 0) {
                    return currentPageIdx - 1;
                } else if (dashboard.circularNavigation) {
                    return dashboard.pages.length - 1;
                }
                return currentPageIdx;
            }
        };

        return svc;
    });
})();

;
(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Utility functions for dashboards.
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.factory('DashboardUtils', function($rootScope, $injector, $state, Logger, $stateParams, ClipboardUtils, ActivityIndicator, $timeout, Debounce, WT1, translate) {

        const COLUMN_NUMBER = 36;

        const svc = {
            canListInsight: (insight) => {
                return insight.type !== 'filters';
            },
            canEditInsight: function(insight, canModerateDashboards) {
                return insight && (canModerateDashboards || insight.owner == $rootScope.appConfig.login);
            },
            hasEditTab: function(insight) {
                return (insight && insight.type && svc.getInsightHandler(insight.type).hasEditTab);
            },

            setLoading() {
                const $scope = this[0];
                $scope.loaded = false;
                $scope.loading = true;
            },

            setError: function(data, status, headers, config, statusText) {
                const $scope = this[0];
                const reject = this[1];

                $scope.loaded = false;
                $scope.loading = false;
                $scope.error = data;
                $scope.unconfigured = false;
                if ($scope.hook && $scope.hook.isErrorMap) {
                    $scope.hook.isErrorMap[$scope.tile.$tileId] = true;
                } else if ($scope.hook && $scope.hook.setErrorInDashboardPageScope) {
                    $scope.hook.setErrorInDashboardPageScope(data, status, headers, config, statusText);
                }
                /*
                 * Mark chart loading process as COMPLETE as it fails initializing
                 * This allow the PDF export to know that the chart is ready to be snapshot
                 */
                if (typeof ($scope.loadedCallback) === 'function') {
                    $scope.loadedCallback();
                }
                if (typeof (reject) === 'function') {
                    reject();
                }
            },
            setLoaded: function(data, status, headers, config, statusText) {
                const $scope = this[0];
                const resolve = this[1];

                $scope.loading = false;
                $scope.loaded = true;
                $scope.error = null;
                $scope.unconfigured = false;
                if ($scope.hook && $scope.hook.isErrorMap) {
                    $scope.hook.isErrorMap[$scope.tile.$tileId] = false;
                }
                if (typeof (resolve) === 'function') {
                    resolve();
                }
            },
            setUnconfigured: function(data, status, headers, config, statusText) {
                const $scope = this[0];
                const reject = this[1];

                $scope.loading = false;
                $scope.loaded = false;
                $scope.error = null;
                $scope.unconfigured = true;
                if ($scope.hook && $scope.hook.isErrorMap) {
                    $scope.hook.isErrorMap[$scope.tile.$tileId] = false;
                }
                /*
                 * Mark chart loading process as COMPLETE as it's not configured
                 * This allow the PDF export to know that the chart is ready to be snapshot
                 */
                if (typeof ($scope.loadedCallback) === 'function') {
                    $scope.loadedCallback();
                }
                if (typeof (reject) === 'function') {
                    reject();
                }
            },
            canGoToInsight: (tile) => {
                return tile.insightType !== 'filters';
            },
            getInsightHandler: function(type) {
                if (!type || type == 'INSIGHT') {
                    return;
                }
                const camelCaseType = type.replace(/-|_/g, ' ').replace(/\b\w/g, function(l) {
                    return l.toUpperCase();
                }).replace(/ /g, '');
                try {
                    return $injector.get(camelCaseType + 'InsightHandler');
                } catch (err) {
                    Logger.error('Failed to inject insight handler for type: ' + type);
                };
            },
            getInsightSourceType: function(insight) {
                const handler = svc.getInsightHandler(insight.type);
                if (!handler) {
                    return null;
                }
                return handler.sourceType || handler.getSourceType(insight);
            },
            getInsightSourceId: function(insight) {
                const handler = svc.getInsightHandler(insight.type);
                return handler.getSourceId(insight);
            },
            getInsightTypeGroup: function(type) {
                const insightTypeGroups = {
                    'scenario_last_runs': 'scenario',
                    'scenario_run_button': 'scenario'
                };

                return insightTypeGroups[type] || type;
            },
            getInsightSourceAccessMode: function(insight) {
                const handler = svc.getInsightHandler(insight.type);
                return handler.accessMode || 'READ';
            },
            getNeededReaderAuthorization: function(insight) {
                const ref = {
                    objectType: svc.getInsightSourceType(insight)
                };

                const resolved = resolveObjectSmartId(svc.getInsightSourceId(insight));
                ref.objectId = resolved.id;
                if (resolved.projectKey) {
                    ref.projectKey = resolved.projectKey;
                }
                const mode = svc.getInsightSourceAccessMode(insight);
                return { objectRef: ref, modes: [mode] };
            },
            hasOptions: function(insight) {
                if (insight && insight.type) {
                    const handler = svc.getInsightHandler(insight.type);
                    if ('hasOptions' in handler) {
                        if (handler.hasOptions instanceof Function) {
                            return handler.hasOptions(insight);
                        }
                        return handler.hasOptions;
                    }
                }
                return true;
            },
            getTooltipPromotionTitleFunction: function(type) {
                return function(item) {
                    if (!item) {
                        return '';
                    }
                    if (item.listed && this.canModerateDashboards()) {
                        return this.isAV2ProjectPermissions() ? `Unpromote this ${type}` : `Make this ${type} private`;
                    } else if (item.listed) {
                        return this.isAV2ProjectPermissions() ? `This ${type} is promoted to all users` : `This ${type} is public to all users`;
                    } else if (this.canModerateDashboards()) {
                        return this.isAV2ProjectPermissions() ? `Promote this ${type} to all users` : `Make this ${type} public to all users`;
                    }
                    return `This ${type} is only yours to see`;
                };
            },
            getDashboardDefaultName: function() {
                return translate('INSIGHTS.MODAL.NEW_DASHBOARD_PLACEHOLDER', '{{name}}\'s dashboard', { name: $rootScope.appConfig.user.displayName });
            },
            isInDashboard: function() {
                return !!$stateParams.dashboardId || ($stateParams.workspaceKey != null && $stateParams.objectType === 'DASHBOARD' && $stateParams.insightId == null);
            },
            copyToClipboard: (urlToCopy) => {
                try {
                    ClipboardUtils.copyToClipboard(urlToCopy);
                } catch (error) {
                    ActivityIndicator.error('Error while copying URL');
                }
            },
            getCurrentPage: (dashboard, pageId) => {
                if (!dashboard || !dashboard.pages) {
                    return null;
                }
                let currentPage = dashboard.pages.find(({ id }) => id === pageId);
                if (currentPage != null) {
                    return currentPage;
                }

                // When we have just created a new page and before we save it, we look at the page index.
                const pageIndex = parseInt(pageId);
                if (!isNaN(pageIndex) && dashboard.pages.length > pageIndex) {
                    currentPage = dashboard.pages[pageIndex];
                }
                return currentPage;
            },
            toggleFilterPanel: (dashboard, page, toggle) => {
                if (!page) {
                    page = svc.getCurrentPage(dashboard, page.id);
                }
                if (page) {
                    if (_.isBoolean(toggle)) {
                        page.showFilterPanel = toggle;
                    } else {
                        page.showFilterPanel = !page.showFilterPanel;
                    }
                }
            },
            toggleAllFilterPanels: (dashboard, toggle) => {
                dashboard.pages.forEach(page => svc.toggleFilterPanel(dashboard, page, toggle));
            },
            showFiltersPanelForPagesWithFilters: (dashboard) => {
                dashboard.pages.forEach(page => svc.toggleFilterPanel(dashboard, page, Boolean(page.filters && page.filters.length)));
            },
            isDashboardEmbeddedInAWorkspace: () => {
                return $state.current.name === 'workspaces.object';
            },
            getColumnNumber: () => COLUMN_NUMBER,
            resizeDashboard: Debounce().withDelay(200, 200).wrap(() => {
                $timeout(() => {
                    window.dispatchEvent(new Event('resize'));
                });
            }),
            resizeGroupTile: Debounce().withDelay(200, 200).wrap(tile => {
                $timeout(() => {
                    $rootScope.$broadcast('dashboardResizeGroupTile', { tile });
                });
            }),
            sendWT1InsightCreation: function(insightType, options) {
                WT1.event('dashboard-insight-create', {
                    insightType,
                    triggeredFrom: undefined,
                    ...options
                });
            },
            sendWT1TileCreation: function(type, options) {
                WT1.event('dashboard-tile-create', {
                    type: type,
                    insightType: undefined,
                    triggeredFrom: undefined,
                    reuseExistingInsight: false,
                    hasUsedRectangleSelection: false,
                    hasUsedMultiSelectKey: false,
                    childTileCount: 0,
                    ...options
                });
            }
        };

        return svc;
    });
})();

;
(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/edit.js
     */
    app.factory('TileUtils', function(DashboardUtils, CreateModalFromTemplate, $rootScope, $stateParams, StateUtils) {

        const DEFAULT_BORDER_OPTIONS = {
            color: '#CCCCCC',
            radius: 0,
            size: 1
        };
        const DEFAULT_TITLE_OPTIONS = {
            fontColor: '#000000',
            fontSize: 13
        };

        const getDefaultShowTitleMode = function(insight) {
            return DashboardUtils.getInsightHandler(insight.type).defaultTileShowTitleMode || 'YES';
        };

        const getDefaultTileParams = function(insight) {
            switch(insight.type) {
                case 'text':
                    return {
                        textAlign: 'LEFT',
                        verticalAlign: 'TOP'
                    };
                case 'image':
                case 'iframe':
                case 'group':
                    return {};
                default:
                    var handler = DashboardUtils.getInsightHandler(insight.type);
                    if (handler.getDefaultTileParams) {
                        return handler.getDefaultTileParams(insight);
                    }
                    return angular.copy(handler.defaultTileParams || {});
            }
        };

        const getDefaultTileBox = function(insight) {
            let dimensions;
            switch (insight.type) {
                case 'text':
                    dimensions = [6, 6];
                    break;
                case 'image':
                    dimensions = [9, 9];
                    break;
                case 'iframe':
                    dimensions = [18, 12];
                    break;
                case 'group':
                    dimensions = [18, 12];
                    break;
                default:
                    var handler = DashboardUtils.getInsightHandler(insight.type);
                    if (handler != null && handler.getDefaultTileDimensions) {
                        dimensions = handler.getDefaultTileDimensions(insight);
                    } else {
                        dimensions = handler != null && handler.defaultTileDimensions || [9, 9];
                    }
            }

            return { width: dimensions[0], height: dimensions[1] };
        };

        const getTileMinSizeThreshold = function(tile) {
            const dimensions = getDefaultTileBox(
                tile.insight != null ?
                    tile.insight :
                    tile.tileType === 'INSIGHT' && tile.insightType != null ?
                        { type: tile.insightType } :
                        { type: tile.tileType.toLowerCase() }
            );
            return { minWidth: Math.ceil(dimensions.width / 1.5), minHeight: Math.ceil(dimensions.height / 1.5) };
        };

        const openUploadPictureDialog = function(tile) {
            CreateModalFromTemplate(
                '/templates/widgets/image-uploader-dialog.html',
                $rootScope,
                null,
                function(newScope) {
                    newScope.projectKey = $stateParams.projectKey;
                    newScope.objectType = 'DASHBOARD_TILE';
                    newScope.objectId = tile.imageId;
                },
                'image-uploader-dialog'
            )
                .then(function(id) {
                    tile.imageId = id;
                });
        };

        // TODO @dashboards move to insight handlers
        const getDefaultTileClickAction = function(insight) {
            switch (insight.type) {
                case 'metrics':
                case 'scenario_last_runs':
                case 'eda':
                    return 'OPEN_INSIGHT';
                case 'chart':
                case 'dataset_table':
                default:
                    return 'DO_NOTHING';
            }
        };

        const copyNonInsightTile = function(tile) {
            if (tile.tileType != 'INSIGHT') {
                const copy = angular.copy(tile);
                copy.box.left = -1;
                copy.box.top = -1;
                copy.clickAction = 'DO_NOTHING';
                return copy;
            }
        };

        const getIsTileInsightEditable = function(insight, editable) {
            if (!insight) {
                return false;
            }
            const canModerateDashboards = $rootScope.projectSummary && $rootScope.projectSummary.canModerateDashboards;
            return editable &&
                DashboardUtils.canEditInsight(insight, canModerateDashboards) &&
                DashboardUtils.hasEditTab(insight);
        };

        const getTargetUiSref = (tile, insight, editable, targetInsight, tabName, section) => {
            if (!insight) {
                return;
            }

            const options = {
                originDashboard: {
                    id: $stateParams.dashboardId,
                    name: $stateParams.dashboardName,
                    pageId: $stateParams.pageId,
                    edit: editable
                }
            };

            if (getIsTileInsightEditable(insight, !!editable)) {
                options.tab = 'edit';
            }
            if (tabName) {
                options.tabSelect = tabName;
            }
            if (section) {
                options.sections = JSON.stringify([section]);
            }

            const insightUiSref = (insightId) => DashboardUtils.isDashboardEmbeddedInAWorkspace()
                ? StateUtils.uiSref.workspaceDashboardInsight($stateParams.workspaceKey, $stateParams.projectKey, $stateParams.objectId, insightId)
                : StateUtils.uiSref.insight(insightId, $stateParams.projectKey, options);

            switch (tile.clickAction) {
                case 'DO_NOTHING':
                case 'OPEN_INSIGHT':
                    options.name = insight.name;
                    return insightUiSref(tile.insightId);
                case 'OPEN_OTHER_INSIGHT':
                    if (!targetInsight) {
                        return null;
                    }
                    options.name = targetInsight.name;
                    return insightUiSref(targetInsight.id || undefined);
            }
        };

        const isGroupTile = (tile) => tile != null && tile.tileType === 'GROUP';

        const isFilterTile = (tile) => tile != null && tile.insightType === 'filters';

        function canConfigureTileDisplay(tile) {
            return !isFilterTile(tile);
        }

        function canConfigureTileTitleAndBorder(tile) {
            return !isFilterTile(tile);
        }

        function canConfigureTileBackgroundOpacity(tile) {
            return tile.tileType === 'TEXT' || tile.insightType === 'chart' || tile.insightType === 'metrics' || isGroupTile(tile);
        }

        function canConfigureTileBackgroundColor(tile) {
            return isGroupTile(tile);
        }

        function canLink(tile) {
            if (isFilterTile(tile) || isGroupTile(tile)) {
                return false;
            } else if (tile.insightType != 'scenario_run_button') {
                return true;
            }
            return tile.displayMode != 'INSIGHT';
        }

        function canGroupTiles(tiles) {
            return tiles != null && tiles.every(tile => !isGroupTile(tile) && !isFilterTile(tile));
        }

        function createGroupTile(childTiles, { useDashboardSpacing, tileSpacing, backgroundColor }) {
            let groupBox = getDefaultTileBox({ type: 'group' });
            if (childTiles && childTiles.length) {
                groupBox = computeGroupTileBox(childTiles);
            }
            const groupTile = {
                tileType: 'GROUP',
                tileParams: getDefaultTileParams({ type: 'group' }),
                box: groupBox,
                backgroundOpacity: 1,
                backgroundColor: backgroundColor,
                useDashboardSpacing: useDashboardSpacing,
                tileSpacing: tileSpacing,
                padding: 0,
                locked: false,
                $new: true,
                autoLoad:true,
                borderOptions: angular.copy(DEFAULT_BORDER_OPTIONS),
                titleOptions: {
                    ...angular.copy(DEFAULT_TITLE_OPTIONS),
                    title: 'Group tile',
                    showTitle: 'NO'
                },
                grid: {
                    tiles: getChildTilesWithWithCoordinatesRelativeToGroupBox(childTiles, groupBox)
                }
            };
            return groupTile;
        }

        function getChildTilesWithWithCoordinatesRelativeToGroupBox(childTiles, groupBox) {
            if (childTiles == null) {
                return [];
            }
            return childTiles.map(childTile => {
                return {
                    ...childTile,
                    box: {
                        width: childTile.box.width,
                        height: childTile.box.height,
                        left: childTile.box.left - groupBox.left,
                        top: childTile.box.top - groupBox.top
                    }
                };
            });
        }

        function computeGroupTileBox(childTiles) {
            const initialBox = {
                left: Infinity,
                top: Infinity,
                right: -Infinity,
                bottom: -Infinity
            };

            // Reduce to find the boundaries
            const { left, top, right, bottom } = childTiles.reduce((acc, tile) => {
                const { left, top, width, height } = tile.box;
                const tileRight = left + width;
                const tileBottom = top + height;

                return {
                    left: Math.min(acc.left, left),
                    top: Math.min(acc.top, top),
                    right: Math.max(acc.right, tileRight),
                    bottom: Math.max(acc.bottom, tileBottom)
                };
            }, initialBox);

            // Calculate the width and height of the bounding box
            const width = right - left;
            const height = bottom - top;

            // Return the group tile box object
            return {
                left,
                top,
                width,
                height
            };
        }

        function adaptChildTilesToMainGrid(groupTile) {
            if (groupTile.grid.tiles == null || groupTile.grid.tiles.length === 0) {
                return [];
            }
            const newTiles = groupTile.grid.tiles.map(childTile => {
                return {
                    ...childTile,
                    $tileId: getUniqueTileId(),
                    $new: true,
                    box: computeUnGroupedChildTileBox(groupTile, childTile)
                };
            });

            // Sort tiles so those with defined `top` and `left` come first so that it get positionned first
            newTiles.sort((a, b) => {
                const aHasPosition = a.box.top !== undefined && a.box.left !== undefined;
                const bHasPosition = b.box.top !== undefined && b.box.left !== undefined;
                return bHasPosition - aHasPosition;
            });
            return newTiles;
        }

        function computeUnGroupedChildTileBox(groupTile, childTile) {
            const childBox = childTile.box;
            const groupBox = groupTile.box;

            // Determine if the child is outside the group's bounds due to scrolling
            const isOutOfGroupVertically =
                        childBox.top > groupBox.height ||
                        childBox.top + childBox.height > groupBox.height;

            // If it's outside the group, reset position properties
            if (isOutOfGroupVertically) {
                return {
                    width: childBox.width,
                    height: childBox.height
                };
            }

            // Otherwise, adjust position relative to the parent grid
            return {
                top: groupBox.top + childBox.top,
                left: groupBox.left + childBox.left,
                width: childBox.width,
                height: childBox.height
            };
        }

        function isChildTileVerticallyOverflowingGroupTile(tile, groupTileHeight) {
            return tile.box.top + tile.box.height > groupTileHeight;
        }

        function getUniqueTileId() {
            return _.uniqueId(`${Date.now().toString()}_`);
        }


        /**
         * Returns a flat list of tiles, including those nested within group tiles
         */
        function getFlattenTileList(tiles) {
            const flatList = [];
            const stack = [...tiles];

            while (stack.length) {
                const tile = stack.pop();
                flatList.push(tile);

                if (isGroupTile(tile) &&
                    tile.grid?.tiles?.length) {
                    stack.push(...tile.grid.tiles);
                }
            }

            return flatList;
        }

        return {
            DEFAULT_BORDER_OPTIONS,
            DEFAULT_TITLE_OPTIONS,
            newInsightTile : function(insight) {
                return {
                    tileType: 'INSIGHT',
                    insightId: insight.id,
                    insightType: insight.type,
                    tileParams: getDefaultTileParams(insight),
                    displayMode: 'INSIGHT',
                    box: getDefaultTileBox(insight),
                    clickAction: getDefaultTileClickAction(insight),
                    titleOptions: { showTitle: getDefaultShowTitleMode(insight) },
                    resizeImageMode: 'FIT_SIZE',
                    autoLoad: true
                };
            },
            getDefaultTileParams,
            getDefaultTileBox,
            getTileMinSizeThreshold,
            getDefaultTileClickAction,
            openUploadPictureDialog,
            copyNonInsightTile,
            getIsTileInsightEditable,
            getTargetUiSref,
            isGroupTile,
            isFilterTile,
            canConfigureTileDisplay,
            canConfigureTileTitleAndBorder,
            canConfigureTileBackgroundOpacity,
            canConfigureTileBackgroundColor,
            canLink,
            canGroupTiles,
            createGroupTile,
            getUniqueTileId,
            adaptChildTilesToMainGrid,
            isChildTileVerticallyOverflowingGroupTile,
            getFlattenTileList
        };
    });

})();
