(function() {
    'use strict';

    // (!) This service previously was in static/dataiku/js/simple_report/curves/stacked_area.js
    angular.module('dataiku.charts')
        .factory('StackedAreaChart', function(ChartManager, ChartDimension, ChartDataWrapperFactory, SVGUtils, StackedChartUtils, LinesUtils, ColorUtils, ChartAxesUtils, ChartYAxisPosition) {
            return function($container, chartDef, chartHandler, axesDef, data) {
                let currentAnimationFrame = 0;

                const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef),
                    xDimension = chartDef.genericDimension0[0],
                    xLabels = chartData.getAxisLabels('x'),
                    animationData = StackedChartUtils.prepareData(chartDef, chartData),
                    yExtent = [0, animationData.maxTotal];

                const drawFrame = function(frameIdx, chartBase) {
                    animationData.frames[frameIdx].facets.forEach(function(facetData, f) {
                        const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                        StackedAreaChartDrawer(g, facetData, chartBase, f);
                    });

                    currentAnimationFrame = frameIdx;
                };

                // Hack: by design stacked area start at 0, but if log scale is checked start at 1 (to make it possible)
                if (ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting)) {
                    yExtent[0] = 1;
                }

                const isPercentScale = ChartDimension.isPercentScale(chartDef.genericMeasures) || chartDef.variant == 'stacked_100';
                const yAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);

                ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                    {
                        x: { type: 'DIMENSION', mode: 'POINTS', dimension: chartDef.genericDimension0[0], name: 'x', customExtent: chartDef.xAxisFormatting.customExtent },
                        [yAxisID]: { id: yAxisID, type: 'MEASURE', extent: yExtent, isPercentScale: isPercentScale, customExtent: ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, yAxisID), position: ChartYAxisPosition.LEFT }
                    },
                    { type: 'DIMENSION', name: 'color', dimension: chartDef.genericDimension1[0] });

                function StackedAreaChartDrawer(g, stacksData, chartBase, f) {

                    const yAxis = chartBase.yAxes[0],
                        yScale = yAxis.scale(),
                        xCoord = LinesUtils.getXCoord(xDimension, chartBase.xAxis, chartBase.xAxis.ordinalScale, xLabels),
                        yCoord = function(d) {
                            return yScale(d.top);
                        };


                    const wrappers = g.selectAll('g.wrapper').data(stacksData.stacks[0].data, function(d, i) {
                        return d.color + '-' + d.measure;
                    });
                    wrappers.enter().append('g').attr('class', 'wrapper')
                        .attr('fill', function(d, i) {
                            return chartBase.colorScale(i);
                        })
                        .attr('opacity', chartDef.colorOptions.transparency)
                        .each(function(d) {
                            chartBase.tooltips.registerEl(this, { measure: d.measure, x: d.x, color: d.color, animation: currentAnimationFrame, facet: f }, 'fill', true);
                        });
                    wrappers.exit().remove();


                    const points = wrappers.selectAll('circle.point').data(function(d, i) {
                        return stacksData.stacks.map(function(stack) {
                            return stack.data[i];
                        });
                    }, function(d) {
                        return d.x + '-' + d.measure + '-' + d.color;
                    });
                    points.enter().append('circle')
                        .attr('class', 'point')
                        .attr('r', 5)
                        .attr('fill', function(d) {
                            return ColorUtils.darken(chartBase.colorScale(d.color + d.measure));
                        })
                        .attr('opacity', 0)
                        .each(function(d) {
                            chartBase.tooltips.registerEl(this, { measure: d.measure, x: d.x, color: d.color, animation: currentAnimationFrame, facet: f }, 'fill');
                            chartBase.contextualMenu.addContextualMenuHandler(this, { x: d.x, color: d.color, animation: currentAnimationFrame, facet: f });
                        });

                    points.exit().remove();
                    points.transition()
                        .attr('cx', xCoord)
                        .attr('cy', yCoord);

                    const area = d3.svg.area()
                        .x(xCoord)
                        .y0(function(d) {
                            return yScale(d.base);
                        })
                        .y1(function(d) {
                            return yScale(d.top);
                        });

                    if (chartDef.smoothing) {
                        area.interpolate('monotone');
                    }

                    const path = wrappers.selectAll('path.area').data(function(d, i) {
                        return [stacksData.stacks.map(function(stack) {
                            return stack.data[i];
                        })];
                    });
                    path.enter().insert('path', ':first-child').attr('class', 'area');
                    path.exit().remove();
                    path.transition().attr('d', area);


                    wrappers.on('mouseover.area', function(d) {
                        this.parentNode.insertBefore(this, $(this.parentNode).find('g.legend')[0]);
                        d3.select(this).selectAll('.wrapper circle').transition(500).attr('opacity', 1);
                    }).on('mouseout.area', function(d) {
                        d3.select(this).selectAll('.wrapper circle').transition(250).attr('opacity', 0);
                    });

                    // Clip paths to prevent area from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
                    ChartAxesUtils.isCroppedChart(chartDef) && SVGUtils.clipPaths(chartBase, g, wrappers);
                }

            };
        });
})();
