from wsgiref.headers import Headers


class DkuCustomHttpResponse:
    def __init__(self, code=200, body="", headers=None, is_base_64=False):
        """
        Can be instantiated directly, but you can in most cases use the static methods :meth:`~DkuCustomHttpResponse.create_text_response`,
        `~DkuCustomHttpResponse.create_json_response` or `~DkuCustomHttpResponse.create_binary_response`.

        :param code: HTTP status code
        :param body: Response body (may be None). Should be encoded in base64 if binary, and is_base_64 set to True
        :param headers: Dict, :class:`wsgiref.headers.Headers`, or a list of header name/value tuples
        :param is_base_64: Should be set to True if returning binary data. In that case, the body is base64-decoded before being returned to the API client.
        decode the body before returning it to the HTTP client.
        """

        self.code = code
        self.body = body
        self.is_base_64 = is_base_64
        self.headers = headers

    @property
    def headers(self):
        """Access the headers of the request."""
        return self._headers

    @headers.setter
    def headers(self, value):
        """Replace all headers of the request. Can be a Dict, an instance of :class:`wsgiref.headers.Headers`, or a list of header name/value tuples"""
        if value is None:
            self._headers = Headers()
        elif isinstance(value, Headers):
            self._headers = Headers(value.items())
        elif isinstance(value, dict):
            self._headers = Headers([(k, v) for k, v in value.items()])
        else:
            self._headers = Headers(value)

    def to_dict(self):
        return {
            "code": self.code,
            "body": self.body,
            "headers": self.headers.items(),
            "isBase64": self.is_base_64
        }

    def add_header(self, key, value, **kwargs):
        """Add a header, allowing duplicated keys. Support the same extended options as :meth:`wsgiref.headers.Headers.add_header`."""
        self.headers.add_header(key, value, **kwargs)

    @staticmethod
    def create_text_response(text_content, code=200, content_type="text/plain"):
        """
        Helper creating a text response, setting Content-Type header on the way.

        :param text_content: text to return. May be None.
        :param code: HTTP status code
        :param content_type: MIME Content Type
        :rtype: :class:`dataiku.DkuCustomHttpResponse`
        """
        headers = Headers()
        headers.add_header("Content-Type", content_type)
        response = DkuCustomHttpResponse(code, text_content, headers)
        return response

    @staticmethod
    def create_json_response(value, code=200, content_type="application/json"):
        """
        Helper creating a JSON response, setting Content-Type header on the way.
        Encoding is always utf-8.

        :param value: value to serialize as JSON
        :param code: HTTP status code
        :param content_type: MIME Content Type.
        :rtype: :class:`dataiku.DkuCustomHttpResponse`
        """
        import json
        headers = Headers()
        headers.add_header("Content-Type", content_type)
        response = DkuCustomHttpResponse(code, json.dumps(value), headers)
        return response

    @staticmethod
    def create_binary_response(binary_data, content_type="application/octet-stream", code=200):
        """
        Helper creating a binary response. Can be a JPG, PNG or GIF image or any other content.
        The binary data will be encoded in base64, sent to the java lambda server. The java lambda server will
        decode it and return it to the HTTP client/

        :param binary_data: data to return
        :param content_type: MIME Content Type.
        :param code: HTTP status code
        :rtype: :class:`dataiku.DkuCustomHttpResponse`
        """
        import base64
        headers = Headers()
        headers.add_header("Content-Type", content_type)
        response = DkuCustomHttpResponse(code, base64.b64encode(binary_data).decode("utf-8"), headers, is_base_64=True)
        return response
