from dataiku.core import base, flow, metrics, dkuio, default_project_key
from dataiku.base import remoterun
import os.path as osp, os
import json, logging, sys
from dataiku.core.intercom import jek_or_backend_json_call, jek_or_backend_void_call, jek_or_backend_stream_call, backend_json_call, backend_void_call, backend_stream_call


class ManagedFolderWriter:
    """
    Handle to write to a managed folder

    .. important::
        Do not create this class directly, use :meth:`dataiku.Folder.get_writer()`

    .. important::
        A ManagedFolderWriter MUST be closed after usage. Failure to close a ManagedFolderWriter will lead 
        to incomplete or no data being written to the output managed folder
    """
    active_writers = dict()

    @staticmethod
    def atexit_handler():
        """
        Internal use. Close active writers when process finishes.
        """
        tobeclosed = []
        if sys.version_info > (3,0):
            for k,v in ManagedFolderWriter.active_writers.items():
                print ('WARNING : A folder writer MUST be closed (%s)'%k)
                tobeclosed+=[v]
        else:
            for k in ManagedFolderWriter.active_writers:
                v = ManagedFolderWriter.active_writers[k]
                print ('WARNING : A folder writer MUST be closed (%s)'%k)
                tobeclosed+=[v]
        ManagedFolderWriter.active_writers = dict()
        for v in tobeclosed:
            v.close()

    def __init__(self, project_key, folder_id, path):
        """
        Create a writer to a managed folder.

        :param string project_key: project of the folder
        :param string folder_id: id of the folder
        :param string path: path within the folder to write to
        """
        self.project_key = project_key
        self.folder_id = folder_id
        self.path = path
        self.full_name = "%s.%s" % (project_key, folder_id)
        logging.info("Initializing folder writer for folder %s" % (self.full_name))
        if ManagedFolderWriter.active_writers.get(self.full_name):
            raise Exception('Unable to instanciate a new folder writer. There is already another active writer for this folder (%s).' % self.full_name)
        # Register itself as active writer
        ManagedFolderWriter.active_writers[self.full_name]= self
        
        def upload_call(g):
            jek_or_backend_void_call("managed-folders/upload-path", params={
                        "projectKey": self.project_key,
                        "lookup" : self.folder_id,
                        "path" : self.path
                    }, data=g)
        self.piping_thread = dkuio.PipeToGeneratorThread('%s.%s.%s' % (self.project_key, self.folder_id, self.path), upload_call)
    
    def write(self, b):
        """
        Write data to the file.

        Usage example:

        .. code-block:: python

            folder = dataiku.Folder("my_folder_id")
            with folder.get_writer('/path/inside/folder') as writer:
                writer.write("some text\\n".encode('utf8'))
                writer.write("more text\\n".encode('utf8'))

        :param bytes b: the data, as byte array
        """
        self.piping_thread.write(b)

    def close(self):
        """
        Close the writer.

        .. note:: 

            You can also use the :class:`ManagedFolderWriter` in a `with`
        """
        if ManagedFolderWriter.active_writers.get(self.full_name) == self:
            del ManagedFolderWriter.active_writers[self.full_name]
        self.piping_thread.close()
        self.piping_thread.wait_for_completion()

    def __enter__(self,):
        return self

    def __exit__(self, type, value, traceback):
        self.close()

def _folder_writer_atexit_handler():
    ManagedFolderWriter.atexit_handler()


class Folder(base.Computable):
    """
    Handle to interact with a folder.

    .. note:: 

        This class is also available as ``dataiku.Folder``
    """

    def __init__(self, lookup, project_key=None, ignore_flow=False):
        """
        Obtain a handle for a named folder.

        :param string lookup: name or identifier of the managed folder
        :param string project_key: project key of the managed folder, if it is not in the current project.
        :param boolean ignore_flow: if True, create the handle regardless of whether the managed folder is an input
                                    or output of the recipe (default: False)
        """
        self.lookup = lookup
        self.path = None
        self.info = None
        self.access_granted = None
        self.partition_infos = {}
        self.ignore_flow = ignore_flow
        self.read_partitions = None
        self.writePartition = None

        if flow.FLOW is not None and ignore_flow == False:
            self._init_data_from_flow(obj_type="Managed folder", project_key=project_key)

        else:
            if "." not in lookup:
                self.project_key = project_key or default_project_key()
                self.short_name = lookup
                self.name = self.project_key + "." + lookup
            else:
                self.project_key = lookup.split(".")[0]
                self.short_name = lookup.split(".")[1]
                self.name = lookup
                #except:
                #    raise Exception("Managed folder %s is specified with a relative name, "
                #                    "but no default project was found. Please use complete name" % id)

    def _repr_html_(self,):
        s = "Folder[   <b>%s</b>   ]" % (self.name)
        return s

    def get_info(self, sensitive_info=False):
        """
        Get information about the location and settings of this managed folder

        Usage sample:

        .. code-block:: python

            # construct the URL to a S3 object
            folder = dataiku.Folder("my_folder_name")
            folder_info = folder.get_info()
            access_info = folder_info["accessInfo"]
            folder_base_url = 's3://%s%s' % (access_info['bucket'], access_info['root'])
            target_url = '%s/some/path/to/a/file' % folder_base_url

        :param boolean sensitive_info: if True, the credentials of the connection of the managed folder are
                                       returned, if they're accessible to the user. (default: False)

        :returns: information about the folder. Fields are:

                    * **id** : identifier of the folder
                    * **projectKey** : project of the folder
                    * **name** : name of the folder
                    * **type** : type of the folder (S3 / HDFS / GCS / ...)
                    * **directoryBasedPartitioning** : whether the partitioning schema of the folder (if any) maps partitions to sub-folders
                    * **path** : path of the folder of the filesystem, for folder on the local filesystem
                    * **accessInfo** : extra information about the filesystem underlying the folder. The exact fields depend on the folder type. Typically contains the connection root and the parts needed to build a full URI, like bucket and storage account name. If the **sensitive_info** parameter is True, then credentials of the connection will be added (if accessible to the user)

        :rtype: dict
        """
        if self.info is None:
            self.info = jek_or_backend_json_call("managed-folders/get-info", {
                "projectKey": self.project_key,
                "lookup" : self.short_name,
                "sensitiveInfo" : sensitive_info
            })
        return self.info["info"]

    def get_partition_info(self, partition):
        """
        Get information about a partition of this managed folder

        :param string partition: partition identifier

        :returns: information about the partition. Fields are:

                    * **id** : identifier of the folder
                    * **projectKey** : project of the folder
                    * **name** : name of the folder
                    * **folder** : if the partitioning scheme maps partitions to a subfolder, the path of the subfolder within the managed folder
                    * **paths** : paths of the files in the partition, relative to the managed folder

        :rtype: dict
        """
        self.partition_infos[partition] = jek_or_backend_json_call("managed-folders/get-partition-paths", {
            "projectKey": self.project_key,
            "lookup" : self.short_name,
            "partition" : partition
        })
        return self.partition_infos[partition]["info"]

    def _ensure_and_check_direct_access(self):
        if remoterun.is_running_remotely():
            raise Exception('Python process is running remotely, direct access to folder is not possible')
        elif self.get_info().get("type", None) == 'Filesystem':
            if self.access_granted is None:
                self.access_granted = jek_or_backend_json_call("managed-folders/ensure-direct-access", {
                    "projectKey": self.project_key,
                    "lookup" : self.short_name,
                    "ignoreFlow": self.ignore_flow
                })
        else:
            raise Exception('Folder is not on the local filesystem (uses %s), cannot perform direct filesystem access. Use the read/write API instead. '%
                 self.get_info().get('type', 'unknown'))

    def get_path(self):
        """
        Get the filesystem path of this managed folder. 

        .. important::
            This method can only be called for managed folders that are stored on the local filesystem of the DSS server. For
            non-filesystem managed folders (HDFS, S3, ...), you need to use the various read/download and write/upload APIs.

        Usage example:

        .. code-block:: python

            # read a model off a local managed folder
            folder = dataiku.Folder("folder_where_models_are_stored")
            with open(os.path.join(f.get_path(), "path/to/model.pkl"), 'rb') as fd:
                model = pickle.load(fd)

        :returns: a path on the local filesystem that the python process can read from and write to
        :rtype: string
        """
        self._ensure_and_check_direct_access()
        if 'path' in self.get_info():
            return self.get_info()["path"]
        else:
            raise Exception("Path is not available for this folder storage backend : %s" % self.get_info().get('type', 'unknown'))

    def is_partitioning_directory_based(self):
        """
        Whether the partitioning of the folder maps partitions to sub-directories.

        :rtype: boolean
        """
        return self.get_info().get("directoryBasedPartitioning", False)

    def list_paths_in_partition(self, partition=''):
        """
        Gets the paths of the files for the given partition.

        :param string partition: identifier of the partition. Use `''` to get the paths of all the files in the folder, regardless
                                 of the partition

        :returns: a list of paths within the folder
        :rtype: list[string]
        """
        return self.get_partition_info(partition)["paths"]

    def list_partitions(self):
        """
        Get the partitions in the folder.
    
        :returns: a list of partition identifiers
        :rtype: list[string]
        """
        return jek_or_backend_json_call("managed-folders/list-partitions", {
            "projectKey": self.project_key,
            "lookup" : self.short_name
        })

    def get_partition_folder(self, partition):
        """
        Get the filesystem path of the directory corresponding to the partition (if the partitioning is directory-based).

        :param string partition: identifier of the partition

        :returns: sub-path inside the folder that corresponds to the partition. None if the partitioning scheme doesn't map
                  partitions to sub-folders
        :rtype: string
        """
        return self.get_partition_info(partition)["folder"]

    def get_id(self):
        """
        Get the identifier of the folder.

        :rtype: string
        """
        return self.get_info()["id"]

    def get_name(self):
        """
        Get the name of the folder.

        :rtype: string
        """
        return self.get_info()["name"]

    def file_path(self, filename):
        """
        Get the filesystem path for a given file within the folder. 

        .. important::
            This method can only be called for managed folders that are stored on the local filesystem of the DSS server. For
            non-filesystem managed folders (HDFS, S3, ...), you need to use the various read/download and write/upload APIs.

        :param string filename: path of the file within the folder

        :returns: the full path of the file on the local filesystem
        :rtype: string
        """
        self._ensure_and_check_direct_access()
        root_path = self.get_info().get("path", '')
        if filename is None or len(filename) == 0:
            return root_path
        else:
            if filename[0] == '/':
                clean_filename = filename[1:]
            else:
                clean_filename = filename
            return osp.join(root_path, clean_filename)

    def read_json(self, filename):
        """
        Read a JSON file within the folder and return its parsed content.

        Usage example:

        .. code-block:: python

            folder = dataiku.Folder("my_folder_id")
            # write a JSON-serializable object
            folder.write_json("/some/path/in/folder", my_object)

            # read back the object
            my_object_again = folder.read_json("/some/path/in/folder")            

        :param string filename: path of the file within the folder

        :returns: the content of the file
        :rtype: list or dict, depending on the content of the file
        """
        with self.get_download_stream(filename) as f:
            return json.load(f)

    def write_json(self, filename, obj):
        """
        Write a JSON-serializable object as JSON to a file within the folder.

        :param string filename: Path of the target file within the folder
        :param object obj: JSON-serializable object to write (generally dict or list)
        """
        self.upload_data(filename, json.dumps(obj).encode("utf-8"))

    def clear(self):
        """
        Remove all files from the folder.
        """
        return self.clear_partition('')
                    
    def clear_partition(self, partition):
        """
        Remove all files from a specific partition of the folder.

        :param string partition: identifier of the partition to clear
        """
        return jek_or_backend_void_call("managed-folders/clear-partition", {
                "projectKey": self.project_key,
                "lookup" : self.short_name,
                "partition" : partition
            })
                    
    def clear_path(self, path):
        """
        Remove a file or directory from the managed folder.

        .. caution::

            Deprecated. use :meth:`delete_path` instead

        :param string path: path inside the folder to the file or directory to delete
        """
        return self.delete_path(path)

    def delete_path(self, path):
        """
        Remove a file or directory from the managed folder.

        :param string path: path inside the folder to the file or directory to delete
        """
        return jek_or_backend_void_call("managed-folders/clear-path", {
            "projectKey": self.project_key,
            "lookup": self.short_name,
            "path": path
        })

    def get_path_details(self, path='/'):
        """
        Get details about a specific path (file or directory) in the folder.

        :param string path: path inside the folder to the file or directory
        
        :returns: information about the file or folder at `path`, as a dict. Fields are:

                    * **exists** : whether there is a file or folder at `path`
                    * **directory** : True if `path` denotes a directory in the managed folder, False if it's a file
                    * **fullPath** : path inside the folder
                    * **size** : if a file, the size in bytes of the file
                    * **lastModified** : last modification time of the file or directory at `path`, in milliseconds since epoch
                    * **mimeType** : for files, the detected MIME type
                    * **children** : for directories, a list of the contents of the directory, each element having the present structure (not recursive)

        :rtype: dict
        """
        return jek_or_backend_json_call("managed-folders/get-path-details", {
                "projectKey": self.project_key,
                "lookup" : self.short_name,
                "path" : path
            })

    def get_download_stream(self, path):
        """
        Get a file-like object that allows you to read a single file from this folder.

        Usage example:

        .. code-block:: python

            with folder.get_download_stream("myfile") as stream:
                data = stream.readline()
                print("First line of myfile is: {}".format(data))

        .. note::
            The file-like returned by this method is not seekable.

        :param string path: path inside the managed folder

        :returns: the data of the file at `path` inside the managed folder
        :rtype: file-like
        """
        return jek_or_backend_stream_call("managed-folders/download-path", params={
                "projectKey": self.project_key,
                "lookup" : self.short_name,
                "path" : path
            })

    def upload_stream(self, path, f):
        """
        Upload the content of a file-like object to a specific path in the managed folder.

        If the file already exists, it will be replaced.

        .. code-block:: python

            # This copies a local file to the managed folder
            with open("local_file_to_upload") as f:
                folder.upload_stream("name_of_file_in_folder", f)

        :param string path: Target path of the file to write in the managed folder
        :param stream f: file-like object open for reading
        """
        return jek_or_backend_void_call("managed-folders/upload-path", params={
                "projectKey": self.project_key,
                "lookup" : self.short_name,
                "path" : path
            }, data=f)

    def upload_file(self, path, file_path):
        """
        Upload a local file to a specific path in the managed folder.

        If the file already exists, it will be replaced.

        :param string path: Target path of the file to write in the managed folder
        :param string file_path: Absolute path to a local file
        """
        with open(file_path, 'rb') as f:
            self.upload_stream(path, f)

    def upload_data(self, path, data):
        """
        Upload binary data to a specific path in the managed folder.

        If the file already exists, it will be replaced.

        :param string path: Target path of the file to write in the managed folder
        :param bytes data: str or unicode data to upload
        """
        f = dkuio.new_bytesoriented_io(data)
        self.upload_stream(path, f)
        
    def get_writer(self, path):
        """
        Get a writer object to write incrementally to a specific path in the managed folder.

        If the file already exists, it will be replaced.

        :param string path: Target path of the file to write in the managed folder

        :rtype: :class:`dataiku.core.managed_folder.ManagedFolderWriter`
        """
        return ManagedFolderWriter(self.project_key, self.short_name, path)

    # ################################### Metrics #############################

    def get_last_metric_values(self, partition=''):
        """
        Get the set of last values of the metrics on this folder.

        :param string partition: (optional) a partition identifier to get metrics for. If not set, returns the metrics
                                 of a non-partitioned folder, and the metrics of the whole managed folder for a 
                                 partitioned managed folder

        :rtype: :class:`dataiku.core.metrics.ComputedMetrics`
        """
        return metrics.ComputedMetrics(backend_json_call("metrics/managed-folders/get-last-values", data = {
            "projectKey": self.project_key,
            "folderId" : self.get_id(),
            "partitionId" : partition
        }))

    def get_metric_history(self, metric_lookup, partition=''):
        """
        Get the set of all values a given metric took on this folder.

        :param string metric_lookup: metric name or unique identifier
        :param string partition: (optional) a partition identifier to get metrics for. If not set, returns the metrics
                                 of a non-partitioned folder, and the metrics of the whole managed folder for a 
                                 partitioned managed folder

        :returns: an object containing the values of the `metric_lookup` metric, cast to the appropriate type (double, 
                  boolean, ...). Top-level fields are:

                    * **metricId** : identifier of the metric
                    * **metric** : dict of the metric's definition
                    * **valueType** : type of the metric values in the values array
                    * **lastValue** : most recent value, as a dict of:

                      * **time** : timestamp of the value computation
                      * **value** : value of the metric at time
                    
                    * **values** : list of values, each one a dict of the same structure as **lastValue**

        :rtype: dict
        """
        return backend_json_call("metrics/managed-folders/get-metric-history", data = {
            "projectKey": self.project_key,
            "folderId" : self.get_id(),
            "partitionId": partition,
            "metricLookup" : metric_lookup if isinstance(metric_lookup, str) or isinstance(metric_lookup, unicode) else json.dumps(metric_lookup)
        })

    def save_external_metric_values(self, values_dict, partition=''):
        """
        Save metrics on this folder. The metrics are saved with the type "external".

        :param dict values_dict: the values to save, as a dict. The keys of the dict are used as metric names
        :param string partition: (optional), the partition for which to fetch the values. On partitioned folders,
                                 the partition value to use for accessing metrics on the whole dataset (ie. all
                                 partitions) is ALL
        """
        return backend_json_call("metrics/managed-folders/save-external-values", data = {
            "projectKey": self.project_key,
            "folderId" : self.get_id(),
            "partitionId": partition,
            "data" : json.dumps(values_dict)
        }, err_msg="Failed to save external metric values")

    def get_last_check_values(self, partition=''):
        """
        Get the set of last values of the checks on this folder, as a :class:`dataiku.core.metrics.ComputedChecks` object

        :param string partition: (optional), the partition for which to fetch the values. On partitioned folders,
                                 the partition value to use for accessing metrics on the whole dataset (ie. all
                                 partitions) is ALL

        :rtype: :class:`dataiku.core.metrics.ComputedChecks`
        """
        return metrics.ComputedChecks(backend_json_call("checks/managed-folders/get-last-values", data = {
            "projectKey": self.project_key,
            "folderId" : self.get_id(),
            "partitionId": partition
        }))

    def save_external_check_values(self, values_dict, partition=''):
        """
        Save checks on this folder. The checks are saved with the type "external".

        :param dict values_dict: the values to save, as a dict. The keys of the dict are used as check names
        :param string partition: (optional), the partition for which to fetch the values. On partitioned folders,
                                 the partition value to use for accessing metrics on the whole dataset (ie. all
                                 partitions) is ALL
        """
        return backend_json_call("checks/managed-folders/save-external-values", data = {
            "projectKey": self.project_key,
            "folderId" : self.get_id(),
            "partitionId": partition,
            "data" : json.dumps(values_dict)
        }, err_msg="Failed to save external check values")
        