#' Get the value of a Flow variable
#'
#' @param name the name of the variable
#' @export
dkuFlowVariable <- function(name) {

    envDef <- dku_remoterun__get_remote_run_env_def(noFail)
    if ("flowVariables" %in% names(envDef)) {
        parsedVariables <- envDef$flowVariables
    } else {
        rSpec <- dku_remoterun__get_env_var("DKUFLOW_VARIABLES")

        # the DKUFLOW_VARIABLES should not arrive here anymore, unless you start R from a shell recipe 
        # and even then the remote run env def will have the flowSpec field
        # Notebook mode
        if (is.null(rSpec) || rSpec == "") {
            return(NULL)
        } else {
            # Flow mode
            parsedVariables <- RJSONIO::fromJSON(rSpec)
        }
    }

    return(parsedVariables[[name]]);
}

#' Get the value of a user-defined variable
#'
#' @param name the name of the variable
#' @param typed true to try to cast the variable into its original type (eg. int rather than string)
#' @export
dkuCustomVariable <- function(name, typed=FALSE) {
    defaultProject <- dku_remoterun__get_env_var("DKU_CURRENT_PROJECT_KEY")
    getVariablesURL = dku_intercom__get_jek_or_backend_url("/variables/get-resolved-for-project")
    resp = POST(getVariablesURL,
            body = list(
              projectKey=defaultProject,
              typed=typed
            ),
            encode="form",
            dku__get_auth_headers(), dku_intercom__get_httr_config())

    dku__check_api_error(resp, "Getting variables failed");
    parsedVariables <- content(resp);
    return(parsedVariables[[name]]);
}


#' Get all variables of the current project, returns a named list with “standard” and “local”.
#' “standard” are regular variables, exported with bundles, “local” variables are not part of the bundles for this project"""
#'
#' @export
dkuGetProjectVariables <- function() {
    defaultProject <- dku_remoterun__get_env_var("DKU_CURRENT_PROJECT_KEY")
    getVariablesURL <- dku_intercom__get_jek_or_backend_url("/variables/get-for-project")
    resp <- POST(getVariablesURL,
                body = list(projectKey=defaultProject),
                encode="form",
                dku__get_auth_headers(), dku_intercom__get_httr_config())

    dku__check_api_error(resp, "Getting variables failed");
    return(content(resp));
}


#' Set all variables of the current project
#'
#' @param variables
#' @export
dkuSetProjectVariables <- function(variables) {
    if(!('standard' %in% names(variables))) {
        stop("Missing 'standard' key in argument")
    }
    if(!('local' %in% names(variables))) {
        stop("Missing 'local' key in argument")
    }

    defaultProject <- dku_remoterun__get_env_var("DKU_CURRENT_PROJECT_KEY")
    getVariablesURL <- dku_intercom__get_jek_or_backend_url("/variables/set-for-project")
    resp <- POST(getVariablesURL,
                query = list(projectKey=defaultProject),
                body = RJSONIO::toJSON(variables),
                encode="form",
                dku__get_auth_headers(), dku_intercom__get_httr_config())
    dku__check_api_error(resp, "Setting project variables failed");
}
