(function() {
    "use strict";

    const generateDocumentationModalBody = {
        templateUrl: "static/dataiku/ai-dataset-descriptions/generate-documentation-modal-body/generate-documentation-modal-body.component.html",
        bindings: {
            object: "<",
            canWriteProject: "<",
            explanationRequested: "=",
            lastJobId: "=",
            abortRequested: "=",
            resolveModal: "&",
            dismiss: "&",
        },
        controller: function(
            $scope,
            $rootScope,
            $timeout,
            ActivityIndicator,
            DataikuAPI,
            AIExplanationService,
            translate
        ) {
            const ctrl = this;
            $scope.translate = translate;

            ctrl.$onInit = function() {
                ctrl.appConfig = $rootScope.appConfig;
                ctrl.datasetDescriptions = {};
                ctrl.originalDescriptions = {};
                ctrl.toggleStates = {
                    shortDesc: !ctrl.object.shortDesc,
                    longDesc: !ctrl.object.description,
                    colDesc: {},
                };
                ctrl.object.schema.columns.forEach(function (column) {
                    ctrl.toggleStates.colDesc[column.name] = !column.comment;
                });
                ctrl.isApprovedAll = false;
                ctrl.toggleAll = function() {
                    ctrl.object.schema.columns.forEach(function(column) {
                        ctrl.toggleStates.colDesc[column.name] = ctrl.isApprovedAll;
                    });
                };
                ctrl.availableOptions = AIExplanationService.createAvailableOptions();
                ctrl.generateSettings = {
                    language: AIExplanationService.loadOptions().language
                };
                ctrl.emptyDescMessage = AIExplanationService.getDefaultEmptyMessage("EXISTING_SECTION");
                if (ctrl.appConfig.prepareAICompletionEnabled) {
                    ctrl.emptyMessage = AIExplanationService.getDefaultEmptyMessage("AI_SECTION");
                } else {
                    ctrl.emptyMessage = AIExplanationService.getDefaultEmptyMessage("AI_DISABLED_SECTION");
                }

                $timeout(() => {
                    $scope.$broadcast('tabSelect', "dataset-descriptions");
                });

                if (ctrl.appConfig.prepareAICompletionEnabled) {
                    ctrl.generateDocumentation();
                } else {
                    ctrl.hasGeneratedDescriptions = true;
                    ctrl.datasetDescriptions.shortDesc = ctrl.object.shortDesc || '';
                    ctrl.datasetDescriptions.longDesc = ctrl.object.description || '';
                    ctrl.datasetDescriptions.colDesc = {};
                    ctrl.object.schema.columns.forEach(function (column) {
                        ctrl.datasetDescriptions.colDesc[column.name] = column.comment || '';
                    });
                    ctrl.originalDescriptions.shortDesc = ctrl.object.shortDesc || '';
                    ctrl.originalDescriptions.longDesc = ctrl.object.description || '';
                    ctrl.originalDescriptions.colDesc = { ...ctrl.datasetDescriptions.colDesc };
                }
            };

            ctrl.generateDocumentation = function() {
                resetErrorInScope($scope);
                ctrl.isFutureRunning = true;
                ctrl.explanationRequested = true;

                AIExplanationService.explainObject(
                    "DATASET",
                    ctrl.object,
                    ctrl.generateSettings,
                    function(initialResponse) {
                        ctrl.lastJobId = initialResponse.jobId;
                    }
                )
                .then(function(data) {
                    if (data.aborted) {
                        return;
                    }
                    ctrl.isFutureRunning = false;
                    ctrl.datasetDescriptions = data.result;
                    ctrl.originalDescriptions.shortDesc = ctrl.datasetDescriptions.shortDesc;
                    ctrl.originalDescriptions.longDesc = ctrl.datasetDescriptions.longDesc;
                    ctrl.originalDescriptions.colDesc = { ...ctrl.datasetDescriptions.colDesc };
                    ctrl.explanationRequested = false;
                    ctrl.hasGeneratedDescriptions = true;
                })
                .catch(function(r) {
                    setErrorInScope.bind($scope)(r.data, r.status, r.headers);
                    ctrl.isFutureRunning = false;
                    ctrl.explanationRequested = false;
                });
                
                ctrl.lastJobId = null;
            };

            ctrl.abortGenerate = function() {
                AIExplanationService.abortExplain(ctrl.lastJobId)
                .then(function() {
                    ctrl.explanationRequested = false;
                    ctrl.abortRequested = false;
                })
                .catch(function(r) {
                    setErrorInScope.bind($scope)(r.data, r.status, r.headers);
                    ctrl.abortRequested = false;
                });
                ctrl.abortRequested = true;
                ctrl.isFutureRunning = false;
            };

            ctrl.noDatasetAndColDescApproved = function() {
                return !ctrl.toggleStates.shortDesc && !ctrl.toggleStates.longDesc && Object.values(ctrl.toggleStates.colDesc).every(value => value === false);
            }

            ctrl.gotResponse = function() {
                return Object.keys(ctrl.datasetDescriptions).length > 0 && !ctrl.explanationRequested;
            };

            ctrl.checkShortDescChange = function() {
                ctrl.toggleStates.shortDesc = !ctrl.object.shortDesc || (ctrl.datasetDescriptions.shortDesc !== ctrl.originalDescriptions.shortDesc);
            };

            ctrl.checkLongDescChange = function() {
                ctrl.toggleStates.longDesc = !ctrl.object.description || (ctrl.datasetDescriptions.longDesc !== ctrl.originalDescriptions.longDesc);
            };

            ctrl.checkColDescChange = function(colName) {
                const col = ctrl.object.schema.columns.find(col => col.name === colName);
                ctrl.toggleStates.colDesc[colName] = !col || !col.comment || (ctrl.datasetDescriptions.colDesc[colName] !== ctrl.originalDescriptions.colDesc[colName]);
            };

            // To track what the current tab is so we are able to switch tabs when clicking a button
            $scope.$on('paneSelected', (event, pane) => {
                ctrl.activeTab = pane.slug;
            });
            
            ctrl.goColumnDescTab = function(tabSlug) {
                $scope.$broadcast('tabSelect', tabSlug);
            };

            ctrl.approveDatasetAndColumnDesc = function() {
                const { toggleStates, datasetDescriptions, object } = ctrl;
            
                const shortDesc = toggleStates.shortDesc ? datasetDescriptions.shortDesc : object.shortDesc;
                const description = toggleStates.longDesc ? datasetDescriptions.longDesc : object.description;
                const toggledColDesc = {}

                for (const colName in datasetDescriptions.colDesc) {
                    if (toggleStates.colDesc[colName]) {
                        toggledColDesc[colName] = datasetDescriptions.colDesc[colName];
                    }
                }

                const apiPromise = DataikuAPI.datasets.setDatasetDescriptions(object.projectKey, object.name, shortDesc, description, toggledColDesc);
                apiPromise.success(function(data) {
                    ctrl.resolveModal();
                    if (toggleStates.shortDesc || toggleStates.longDesc) {
                        $rootScope.$broadcast("objectMetaDataChanged", { shortDesc: data.shortDesc, description: data.description });
                    }
                    $rootScope.$broadcast("datasetSchemaChanged", { schema: data.schema });
                    ActivityIndicator.success("Dataset and column descriptions saved");
                }).error(setErrorInScope.bind($scope));
            };
        }
    };

    angular.module("dataiku.aiDatasetDescriptions").component("generateDocumentationModalBody", generateDocumentationModalBody);
})();
