(function() {
    'use strict';

    angular.module('dataiku.flow.graph').factory('PageSpecificTourService', pageSpecificTourService);

    function pageSpecificTourService($rootScope, $timeout, translate, $filter, FlowGraphSelection, Dialogs, ContextualMenu, TopbarDrawersService, TOPBAR_DRAWER_IDS, WT1, QuestionnaireService, OpalsService, DataikuAPI, ProfileService) {

        const BUILT_DATASET_SELECTOR = 'g[data-type="LOCAL_DATASET"] > g:not(.never-built-computable)';

        /** Tours **/

        const TOUR_NAMES = {
            FLOW: 'flow',
            TUTORIAL: 'tutorial', // Flow Tour variation when coming from the questionnaire
            EXPLORE: 'explore',
            PREPARE: 'prepare'
        };

        function getTourCompletionSetting(tourName) {
            switch(tourName) {
                case TOUR_NAMES.FLOW:
                case TOUR_NAMES.TUTORIAL:
                    return $rootScope.appConfig?.pageSpecificTourSettings?.flowTourCompleted;
                case TOUR_NAMES.EXPLORE:
                    return $rootScope.appConfig?.pageSpecificTourSettings?.exploreTourCompleted;
                case TOUR_NAMES.PREPARE:
                    return $rootScope.appConfig?.pageSpecificTourSettings?.prepareTourCompleted;
            }
        }

        function setTourCompletionSetting(tourName, value) {
            switch(tourName) {
                case TOUR_NAMES.FLOW:
                case TOUR_NAMES.TUTORIAL:
                    $rootScope.appConfig.pageSpecificTourSettings.flowTourCompleted = value;
                    break;
                case TOUR_NAMES.EXPLORE:
                    $rootScope.appConfig.pageSpecificTourSettings.exploreTourCompleted = value;
                    break;
                case TOUR_NAMES.PREPARE:
                    $rootScope.appConfig.pageSpecificTourSettings.prepareTourCompleted = value;
                    break;
            }
        }

        /** Conditions to display the Flow Tour - exposed so we can check whether to open
         *  the Help Center on the Tutorial page when coming from the questionnaire
         */
        function checkFlowTourConditions(fromContext) {
            if (!canStartFlowTour()) {
                return Promise.resolve(false); 
            }

            return checkContextAllowTour(TOUR_NAMES.FLOW, fromContext);
        }

        function startFlowTour({
            scope,
            fromContext
        }) {
            return checkFlowTourConditions(fromContext)
            .then((tourNeedToStart) => {
                if (!tourNeedToStart) {
                    return;
                }

                TopbarDrawersService.getDrawer(TOPBAR_DRAWER_IDS.OPALS_HELP).hide(true);

                const isFromQuestionnaire = QuestionnaireService.isFromQuestionnaire();
                const tourName = isFromQuestionnaire ? TOUR_NAMES.TUTORIAL : TOUR_NAMES.FLOW;
                const introJSInstance = initialTourSetup(scope, tourName, fromContext);

                const builtDatasets = $(BUILT_DATASET_SELECTOR);
                const builtDataset = builtDatasets[0].parentNode;
                let builtDatasetNodeId = builtDataset.id;

                // Common steps for the tour when coming both from the Questionnaire or not
                const commonSteps = [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.DATASET.TITLE', 'Everything in Dataiku begins with data'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.DATASET.DESCRIPTION', 'Datasets are always represented as blue squares on the Flow.'),
                        beforeChange: function() {
                            hideHighlightDuringTransition(introJSInstance, 400);
                            this.element = document.querySelector(BUILT_DATASET_SELECTOR);
                            this.position = 'right';
                            scope.zoomGraph(builtDatasetNodeId, 5);
                            FlowGraphSelection.onItemClick(scope.nodesGraph.nodes[builtDatasetNodeId]); // select the item for the right panel and preview
                            if (scope.standardizedSidePanel.opened) {
                                // if the right panel is already opened, make sure the actions tab is opened
                                scope.standardizedSidePanel.toggleTab('actions');
                            }
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.RIGHT_PANEL.TITLE', 'Start building your data pipeline from the right panel'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.RIGHT_PANEL.DESCRIPTION', 'The right panel is the main command center of the Flow.<br/><br/>It enables you to view information about your data, use pre-built transformation tools (known as Recipes), create ML models and more.'),
                        element: '[data-page-tour="right-panel"]',
                        position: 'left',
                        beforeChange: function() {
                            if (!scope.standardizedSidePanel.opened) {
                                // open right panel on actions tab
                                scope.standardizedSidePanel.toggleTab('actions');
                                hideHighlightDuringTransition(introJSInstance, 300);
                            }
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.PREVIEW.TITLE', 'View a preview of your data'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.PREVIEW.DESCRIPTION', 'Quickly get a snapshot of your dataset in the preview panel.<br/><br/>You can minimize and reopen the preview tab at any time by clicking the "Hide Preview" button.'),
                        element: '[data-page-tour="preview-button"]',
                        position: 'top',
                        beforeChange: function() {
                            if (!scope.hasPreview) {
                                scope.togglePreview();
                                scope.$apply();
                                hideHighlightDuringTransition(introJSInstance, 400);
                            }
                                setTimeout(() => {
                                    // highlight the preview pane too
                                    $('[data-page-tour="preview-pane"]').addClass('introjs-showElement');
                                }, 0);
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.HOME_BUTTON.TITLE', 'Click the Home button to go to the home page'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.HOME_BUTTON.DESCRIPTION', 'View and create new projects, workspaces and more from the homepage.'),
                        element: '[data-page-tour="navbar-home-button"]',
                        position: 'bottom',
                        beforeChange: function() {
                            // close the preview panel
                            scope.togglePreview();
                            scope.$apply();
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.TOP_NAV.TITLE', 'All your work, accessible in one place'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.TOP_NAV.DESCRIPTION', 'The navigation bar is essential to moving around the different areas of your Dataiku project.<br/><br/>Simply hover your cursor on a menu item to see a full list of the available functionalities.'),
                        element: '[data-page-tour="project-menus"]',
                        position: 'bottom'
                    }
                ];

                // Extra steps the user will see when doing only the Flow Tour (not coming from the questionnaire)
                const tourOnlySteps = [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.INTRO.TITLE', 'Take a quick tour of the Flow?'),
                        intro: `<img src="static/dataiku/images/flow/flow-animation.gif"/>` + translate('PAGE_SPECIFIC_TOUR.FLOW.INTRO.DESCRIPTION', 'Learn how to build your data pipeline and navigate through the items in your project.'),
                        hideStepNumber: true,
                    },
                    ...commonSteps,
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.OUTRO.TITLE', '✅ You\'ve finished the Flow tour'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.OUTRO.DESCRIPTION', 'That\'s all for now. You can re-activate the Flow tour or browse resources at any time through the Help Center <i class="dku-icon-question-circle-fill-16 vab"></i>'),
                        hideStepNumber: true,
                        element: '[data-page-tour="help-center-trigger"]',
                        position: 'bottom'
                    },
                ];

                if (fromContext === 'opals') {
                    tourOnlySteps.shift(); // no need to display the first step to the user since they launched the tour
                }
                if (isFromQuestionnaire) {
                    introJSInstance.setOptions({
                        nextLabel: translate("PAGE_SPECIFIC_TOUR.BUTTONS.NEXT", "Next"),
                        doneLabel: translate("PAGE_SPECIFIC_TOUR.BUTTONS.NEXT", "Next")
                    });
                    introJSInstance.afterexit = () => {
                        // if the user finished the tour or exited before the end, show the user the Tutorial
                        displayTutorialPopup({ scope });
                    }
                }

                introJSInstance.setOptions({
                    steps: isFromQuestionnaire ? commonSteps : tourOnlySteps,
                });

                $timeout(() => {
                    introJSInstance.start();
                }, 100); // give some time for the DOM to fully load
            });
        }

        function displayTutorialPopup({
            scope,
        }) {
            const tutorialPage = QuestionnaireService.getOpalsPage();
            const tutorialStartingStep = QuestionnaireService.getTutorialStartingStep();
            if (!tutorialPage) {
                return; // don't know which tutorial to show, return
            }
            if (scope.standardizedSidePanel.opened) {
                scope.standardizedSidePanel.slidePanel(); // close right panel
            }
            OpalsService.navigateToAndShowDrawer(tutorialPage, { number : tutorialStartingStep, numCompletedTasks: tutorialStartingStep - 1 });

            const introJSInstance = initialPopupSetup();

            introJSInstance.setOptions({
                steps: [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.TUTORIAL.TITLE', 'Continue your tutorial here'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.TUTORIAL.DESCRIPTION', 'Follow the detailed step-by-step guideline to learn how to use Dataiku'),
                        hideStepNumber: true,
                        element: 'opals-help-center-container',
                        position: 'left'
                    }
                ]
            });

            $timeout(() => {
                introJSInstance.start();
            }, 0);
        }

        function displayReopenTutorialPopup() {
            if (!$rootScope.appConfig.opalsEnabled) {
                return; // the Help Center is deactivated, do not start the Help Center Tour
            }
            if (!$rootScope.appConfig.onboardingExperience) {
                return; // Onboarding Experience is disabled, do not start the Help Center Tour
            }
            if (ProfileService.isTechnicalAccount()) {
                return; // do not show the Tour to technical accounts
            }

            const introJSInstance = initialPopupSetup();

            introJSInstance.setOptions({
                steps: [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.TUTORIAL.INTRO.TITLE', 'You can re-open the tutorial from here'),
                        intro: translate('PAGE_SPECIFIC_TOUR.TUTORIAL.INTRO.DESCRIPTION', 'Open the Help Center with this button and go to \'Onboarding\' to find all the quick tutorials for your use case'),
                        hideStepNumber: true,
                        element: '[data-page-tour="help-center-trigger"]'
                    },
                ]
            });

            $timeout(() => {
                introJSInstance.start();
            }, 0); // give some time for the DOM to fully load
        }

        function startExploreTour({
            scope,
            fromContext
        }) {
            if (!canStartExploreTour(scope)) {
                return Promise.resolve();  
            }

            return checkContextAllowTour(TOUR_NAMES.EXPLORE, fromContext)
            .then((tourNeedToStart) => {
                if (!tourNeedToStart) {
                    return;
                }

                const introJSInstance = initialTourSetup(scope, TOUR_NAMES.EXPLORE, fromContext);

                const steps = [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.INTRO.TITLE', 'Take a quick tour of the Explore page ?'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.INTRO.DESCRIPTION', 'Learn how to analyze your data, adjust the sample, create charts and more..'),
                        hideStepNumber: true
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.SAMPLE.TITLE', 'Configure your dataset sample'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.SAMPLE.DESCRIPTION', 'For datasets larger than 10,000 rows, Dataiku shows only a sample of a dataset to enable efficient data exploration.<br/><br/>Use the sample settings panel to configure the sampling method, number of records and more.'),
                        element: '[data-page-tour="sampling-badge"]',
                        position: 'bottom'
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.COLUMN.TITLE', 'Instantly check your column characteristics'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.COLUMN.DESCRIPTION', 'Beneath each column name, you will see that Dataiku automatically detects the storage type (shown in grey), the meaning (shown in blue) and data validity (red/green bar).'),
                        beforeChange: function() {
                            // set the element now that it's rendered
                            this.element = document.querySelector('[data-page-tour="column-header"]');
                            this.position = "right";
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.ANALYZE.TITLE', 'Analyze your columns at a glance'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.ANALYZE.DESCRIPTION', 'Click on the <i>Analyze</i> option in the drop-down menu to get some quick descriptive statistics on the contents of your column.'),
                        beforeChange: function() {
                            document.querySelector('[data-page-tour="column-name"]').click(); // open the column dropdown
                            // set the element now that it's rendered
                            this.element = document.querySelector('[data-page-tour="analyze-column"]');
                            this.position = "right";
                            closeContextualMenuOnlyOnNextButtonClick(scope);
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.CHARTS.TITLE', 'Visualize your data in a couple of clicks'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.CHARTS.DESCRIPTION', 'Visit the Charts tab to start visualizing your data with our drag-and-drop chart building interface.'),
                        element: '[data-page-tour="tab-charts"]',
                        position: 'bottom'
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.OUTRO.TITLE', '✅ You\'ve finished the Explore tour'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.OUTRO.DESCRIPTION', 'That\'s all for now. You can re-activate the Explore tour or browse resources at any time through the Help Center <i class="dku-icon-question-circle-fill-16 vab"></i>'),
                        hideStepNumber: true,
                        element: '[data-page-tour="help-center-trigger"]',
                        position: 'bottom'
                    }
                ];

                if (fromContext === 'opals') {
                    steps.shift(); // no need to display the first step to the user since they launched the tour
                }

                introJSInstance.setOptions({
                    helperElementPadding: 5,
                    steps: steps
                });

                $timeout(() => {
                    introJSInstance.start();
                }, 0); // give some time for the DOM to fully load
            });
        }

        function startPrepareTour({
            scope,
            fromContext
        }) {
            if (!canStartPrepareTour(scope)) {
                return Promise.resolve();  
            }
            return checkContextAllowTour(TOUR_NAMES.PREPARE, fromContext)
            .then((tourNeedToStart) => {
               if (!tourNeedToStart) {
                return;
            }

                const introJSInstance = initialTourSetup(scope, TOUR_NAMES.PREPARE, fromContext);

                const steps = [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.INTRO.TITLE', 'Take a quick tour of the Prepare recipe?'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.INTRO.DESCRIPTION', 'Learn how to cleanse, normalize and enrich your data in visual and interactive way.'),
                        hideStepNumber: true,
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.NEW_STEP.TITLE', 'Add steps to your prepare script'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.NEW_STEP.DESCRIPTION', 'Access hundreds of pre-made processors such as filtering rows, rounding numbers, splitting columns and more.'),
                        element: '[data-page-tour="shaker-new-step-btn"]',
                        position: 'bottom'
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.PROCESSORS.TITLE', 'Choose from a range of pre-made processors'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.PROCESSORS.DESCRIPTION', 'Processors have been designed to handle one specific task, such as filtering rows, rounding numbers and more.'),
                        element: '[data-page-tour="processors-library"]',
                        position: 'top',
                        beforeChange: function() {
                            scope.toggleLibrary();
                            hideHighlightDuringTransition(introJSInstance, 400);
                        },
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.COLUMN_SUGGESTIONS.TITLE', 'Get suggested preparation steps for each column'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.COLUMN_SUGGESTIONS.DESCRIPTION', 'Get suggestions for processors that can be applied to each column of your dataset.'),
                        beforeChange: function() {
                            scope.toggleLibrary(false);
                            document.querySelector('[data-page-tour="column-name"]').click();
                            this.element = document.querySelector('[data-page-tour="suggested-actions"]');
                            this.position = "bottom";
                            closeContextualMenuOnlyOnNextButtonClick(scope);
                            hideHighlightDuringTransition(introJSInstance, 100);
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.PREVIEW.TITLE', 'Preview your Prepare recipe script'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.PREVIEW.DESCRIPTION', 'When new steps are added to the script, the step output is immediately visible thanks to the preview.<br/><br/>This preview is only computed on your sample dataset. The full script will be executed when you run the recipe.'),
                        element: '[data-page-tour="step-preview-button"]',
                        position: 'right'
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.OUTRO.TITLE', '✅ You\'ve finished the Prepare recipe tour'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.OUTRO.DESCRIPTION', 'That\'s all for now. You can re-activate the Prepare recipe tour or browse resources at any time through the Help Center <i class="dku-icon-question-circle-fill-16 vab"></i>'),
                        hideStepNumber: true,
                        element: '[data-page-tour="help-center-trigger"]',
                        position: 'bottom'
                    }
                ]
                const bypassPreviewStep =  document.querySelectorAll('[data-page-tour="step-preview-button"]').length === 0;
                if (bypassPreviewStep) {
                    steps.splice(4, 1);
                }
                if (fromContext === 'opals') {
                    steps.shift(); // no need to display the first step to the user since they launched the tour
                }

                introJSInstance.setOptions({
                    helperElementPadding: 5,
                    steps: steps
                });

                $timeout(() => {
                    introJSInstance.start();
                }, 0); // give some time for the DOM to fully load
            });
        }

        /****** WT1 Methods ******/

        function onTourStarted(introJSInstance) {
            if (introJSInstance.$tourName) {
                WT1.tryEvent('page-tour-started', () => ({ from: introJSInstance.$fromContext, scope: introJSInstance.$tourName }));
            }
        }

        function onTourDismissed(introJSInstance) {
            introJSInstance.$skipped = true;
            if (introJSInstance.$tourName) {
                WT1.tryEvent('page-tour-dismissed', () => ({ from:  introJSInstance.$fromContext, scope: introJSInstance.$tourName }));
            }
        }

        function onTourClosed(introJSInstance) {
            if (introJSInstance.$tourName && !introJSInstance.$skipped && introJSInstance.currentStep() >= 0) {
                WT1.tryEvent('page-tour-closed', () => ({
                    from: introJSInstance.$fromContext,
                    scope: introJSInstance.$tourName,
                    numberOfSteps: introJSInstance._options.steps.length,
                    currentStep: introJSInstance.currentStep() + 1
                }));
            }
        }


        /****** Utils methods *******/


        /** Conditions that prevent the Flow Tour from starting **/
        function canStartFlowTour() {
            const builtDatasets = $(BUILT_DATASET_SELECTOR);
            if (!builtDatasets.length) {
                return false; // don't start the tour if there are no built datasets
            }
            return checkGlobalSettingsAllowTour(TOUR_NAMES.FLOW);
        }

        /** Conditions that prevent the Explore Tour from starting **/
        function canStartExploreTour(scope) {
            if (!scope.table || scope.table.headers.length === 0) {
                return; // no table loaded -- dismiss temporarily
            }
            return checkGlobalSettingsAllowTour(TOUR_NAMES.EXPLORE);
        }

        /** Conditions that prevent the Prepare Tour from starting **/
        function canStartPrepareTour(scope) {
            if (!scope.projectSummary || !scope.projectSummary.canWriteProjectContent) {
                return; // user doesn't have write content on this project, do not show the Tour
            }
            if (!scope.table || scope.table.headers.length === 0) {
                return; // no table loaded -- dismiss temporarily
            }
            if (!scope.shaker || scope.shaker.origin !== 'PREPARE_RECIPE') {
                return; // not on a prepare recipe
            }
            return checkGlobalSettingsAllowTour(TOUR_NAMES.PREPARE);
        }


        /** Check if the tour can be started based on global settings and user profile - dismiss it permanently if appropriate  **/
        function checkGlobalSettingsAllowTour(tourName) {
            // the Help Center is deactivated, dismiss permanently
            if (!$rootScope.appConfig.opalsEnabled) {
                recordTourCompleted(tourName);
                return false;
            }

            // Onboarding Experience is disabled, dismiss permanently
            if (!$rootScope.appConfig.onboardingExperience) {
                recordTourCompleted(tourName);
                return false;
            }

            // do not show the Tour to technical accounts - dismiss permanently
            if (ProfileService.isTechnicalAccount()) {
                recordTourCompleted(tourName);
                return false;
            }

            // do not show the users which profiles don't allow write project content - dismiss permanently
            if (!$rootScope.appConfig.userProfile.mayWriteProjectContent) {
                recordTourCompleted(tourName);
                return false;
            }
            return true;
        }

        /** Check if the tour can be started based on contextual settings (already completed, Help Center opened, ...) - dismiss it temporarily if appropriate  **/
        function checkContextAllowTour(tourName, fromContext) {
            // if the tour is started from the Help Center, launch it even if the tour is completed
            if (fromContext === 'opals') {
                return Promise.resolve(true);
            }

            //  if was already dismissed or completed, do not show them again
            if (getTourCompletionSetting(tourName)) {
                return Promise.resolve(false);
            }

            // the Help Center is shown - dismiss temporarily
            if (TopbarDrawersService.getDrawer(TOPBAR_DRAWER_IDS.OPALS_HELP).isToggledOn() && tourName !== TOUR_NAMES.FLOW) {
                return Promise.resolve(false);
            }

            // check if the tour has been completed on another instance (using opals local storage)
            return OpalsService.getLocalStorage(tourName+"TourCompleted")
            .then(tourCompleted => {
                if (tourCompleted === "true") {
                    recordTourCompleted(tourName);
                    return false;
                }
                return  true;
            })
        }

        /** Common setup for all tours (callbacks, WT1, styling, state change handling, ...) **/
        function initialTourSetup(scope, tourName, fromContext) {
            const introJSInstance = introJs();
            introJSInstance.$tourName = tourName;
            introJSInstance.$fromContext = fromContext;

            // End the tour when leaving the page (e.g. with the back arrow)
            const unbindStateChangeListener = scope.$on('$stateChangeStart', () => {
                introJSInstance.$forceExit = true;
                introJSInstance.exit();
            });

            /** send the WT1 when leaving the page with the tour on **/
            const handlePageLeave = () => {
                onTourClosed(introJSInstance);
            }
            window.addEventListener('beforeunload', handlePageLeave)
            scope.$on("$destroy", () => {
                window.removeEventListener("beforeunload", handlePageLeave);
                unbindStateChangeListener();
            });

            /** Defaults common options, can be overridden in the specific tour if necessary **/
            introJSInstance.setOptions({
                disableInteraction: true,
                keyboardNavigation: false,
                buttonClass: 'btn btn--primary',
                doneLabel: translate("PAGE_SPECIFIC_TOUR.BUTTONS.FINISH_TOUR", "Finish Tour"),
                nextLabel: fromContext === 'opals' ? translate("PAGE_SPECIFIC_TOUR.BUTTONS.NEXT", "Next") : translate("PAGE_SPECIFIC_TOUR.BUTTONS.LETS_GO", "Let's go"),
                showBullets: false,
                helperElementPadding: 0
            });

            /** common callbacks **/
            introJSInstance.onstart(function() {
                onTourStarted(introJSInstance);
                Mousetrap.pause();
            });

            let skipped = false;
            introJSInstance.onskip(function() {
                skipped = true;
            });

            introJSInstance.onbeforechange(function() {
                const currentStep = this._introItems[this._currentStep];
                if (currentStep.beforeChange) {
                    currentStep.beforeChange(); // callback defined at step level
                }
            });
            introJSInstance.onafterchange(function() {
                restyleTour(introJSInstance);
                if (introJSInstance.currentStep() === introJSInstance._options.steps.length - 1) {
                    recordTourCompleted(introJSInstance.$tourName);
                    OpalsService.setLocalStorage(tourName+"TourCompleted","true");
                }
            });
            introJSInstance.onbeforeexit(function () {
                
                if (introJSInstance.currentStep() === introJSInstance._options.steps.length - 1) {
                    return true;  
                }

                if (skipped) {
                    // custom behavior when closing the tour by clicking the cross, directly go the bye step without confirmation modal
                    onTourClosed(introJSInstance);
                    goToByeStep(introJSInstance);
                    return false; 
                }

                if (!introJSInstance.$forceExit) {
                    pauseTour(introJSInstance);
                    const tourNameTranslated = translate('PAGE_SPECIFIC_TOUR.TOUR_NAME.' + $filter('uppercase')(tourName),$filter('capitalize')(tourName));
                    return Dialogs.confirm(scope,
                        translate('PAGE_SPECIFIC_TOUR.CONFIRM_MODAL.TITLE',"👋 Are you sure you want to leave the tour?"),
                        translate('PAGE_SPECIFIC_TOUR.REACTIVATE.DESCRIPTION', `You can re-activate the {{tourName}} tour or browse resources at any time through the Help Center <i class="dku-icon-question-circle-fill-16 vab"></i>`, {tourName: tourNameTranslated}),
                        { confirmOnExit : true }
                    ).then(() => {
                        recordTourCompleted(introJSInstance.$tourName);
                        OpalsService.setLocalStorage(tourName+"TourCompleted","true");
                    }, () => {
                        unpauseTour(introJSInstance);
                        return false;
                    });
                }
                return true;
            });
            introJSInstance.onexit(function() {
                if (!skipped) {
                    onTourClosed(introJSInstance);
                }
                window.removeEventListener("beforeunload", handlePageLeave);
                unbindStateChangeListener();
                ContextualMenu.prototype.closeAny();
                Mousetrap.unpause();
                if (introJSInstance.afterexit) {
                    introJSInstance.afterexit();
                }
            });

            return introJSInstance;
        }

        /** Minimal setup for single step tours (no WT1, no step change management, ...) **/
        function initialPopupSetup() {
            const introJSInstance = introJs();

            introJSInstance.setOptions({
                disableInteraction: true,
                keyboardNavigation: false,
                buttonClass: 'btn btn--primary',
                doneLabel: "OK",
                nextLabel: translate("PAGE_SPECIFIC_TOUR.BUTTONS.NEXT", "Next"),
                showBullets: false,
                helperElementPadding: 0
            });
            introJSInstance.onstart(function() {
                Mousetrap.pause();
            });
            introJSInstance.onafterchange(function() {
                restyleTour(introJSInstance);
            });
            introJSInstance.onexit(function() {
                Mousetrap.unpause();
            });
            return introJSInstance;
        }

        /**
            Save the page specific tour settings with the current tour marked as completed
        **/
        const recordTourCompleted = (tourName) => {
            if (tourName) {
                if (getTourCompletionSetting(tourName)) {
                    return;
                }
                setTourCompletionSetting(tourName, true);
                DataikuAPI.profile.updatePageSpecificTourSettings($rootScope.appConfig.pageSpecificTourSettings);
            }
        }

        /**
        * When opening a dropdown menu during a Tour, we most of the time don't want to close it on any click
        * This method helps with closing the dropdown only when going to the next step
        * If the tour is closed at that step, we also ensure that all dropdowns get closed
        **/
        const closeContextualMenuOnlyOnNextButtonClick = () => {
            window.setTimeout(() => {
                // override click handler to close contextual menu only when clicking next
                $(document).off('click.closeMenu');
                $(document).on('click.closeMenu', function(evt) {
                    const isClickOnNextButton = $(evt.target).hasClass('introjs-nextbutton');
                    if (isClickOnNextButton) {
                        ContextualMenu.prototype.closeAny();
                    }
                });
            }, 0);
        }

        /**
        *   introJS doesn't handle transitions very well so in that case we hide the highlight and tooltip
        *   during the transition and display them again when it's finished
        *   (looks like it will be fixed in v8.0.0 when it's released)
        **/
        const hideHighlightDuringTransition = (introJSInstance, delay) => {
            $timeout(() => $('.introjs-helperLayer, .introjs-tooltipReferenceLayer').addClass("display-none")); // hide the highlight and tooltip during the transition
            $timeout(() => {
                introJSInstance.refresh();  // refresh to position the highlight correctly
                $('.introjs-helperLayer, .introjs-tooltipReferenceLayer').removeClass("display-none"); // show the highlight and tooltip once the transition is complete
            }, delay);
        }

        let highlightedElements = [];
        /**
            Pause the tour to display the confirm modal. We hide all elements related to the Tour
        **/
        const pauseTour = (introJSInstance) => {
            $('.introjs-tooltipReferenceLayer, .introjs-helperLayer, .introjs-overlay, .introjs-disableInteraction').addClass("display-none");
            highlightedElements = $('.introjs-showElement');
            /** Elements that are highlighted before the pause */
            highlightedElements.removeClass('introjs-showElement');
            introJSInstance.refresh();
        }

        /**
            Resume the tour if user doesn't exit. We display again all elements of the Tour
        **/
        const unpauseTour = (introJSInstance) => {
            $('.introjs-tooltipReferenceLayer, .introjs-helperLayer, .introjs-overlay, .introjs-disableInteraction').removeClass('display-none');
            /** Elements that need to be highlighted again when un-pausing */
            highlightedElements.addClass('introjs-showElement');
            introJSInstance.refresh();
        }


        // ----- Tour styling ----- //

        const restyleTour = (introJSInstance) => {
            const currentStep = introJSInstance.currentStep();
            if (currentStep === 0) {
                // we only need to do the following on the first steps since it's persisted for later steps
                restyleDismissButton();
                introJSInstance.setOption('nextLabel', translate("PAGE_SPECIFIC_TOUR.BUTTONS.NEXT", "Next"));
                if (!QuestionnaireService.isFromQuestionnaire() && introJSInstance.$fromContext !== 'opals') {
                    // when coming from the questionnaire, do not show the 'No thanks' button
                    // as the user already agreed to follow the tutorial (they can still close)
                    addNoThanksButton(introJSInstance);
                }
            } else if (currentStep === 1) {
                removeNoThanksButton();
            }
            restyleStepNumber(introJSInstance);
        }

        const restyleDismissButton = () => {
            $('.introjs-skipbutton').addClass('dku-icon-dismiss-20');
            $('.introjs-skipbutton').text('');
        }

        /** Display the step numbers except for steps with the "hideStepNumber" property **/
        const restyleStepNumber = (introJSInstance) => {
            var stepNumberElements = document.getElementsByClassName("introjs-stepNumber");
            var helperNumberLayer;
            if (!stepNumberElements.length) {
                helperNumberLayer = document.createElement("div");
                helperNumberLayer.className = 'introjs-stepNumber';
                const tooltipButtons = $('.introjs-tooltipbuttons');
                const parentNode = tooltipButtons[0];
                parentNode.insertBefore(helperNumberLayer, parentNode.firstChild);
            } else {
                helperNumberLayer = stepNumberElements[0];
            }
            // give this step a number but only counting numbered steps
            const currentStep = introJSInstance._introItems[introJSInstance.currentStep()];
            const numberedSteps = introJSInstance._introItems.filter(item => !item.hideStepNumber);
            const stepNumber = numberedSteps.indexOf(currentStep);
            helperNumberLayer.innerText = !currentStep.hideStepNumber ? `${stepNumber + 1} ${translate("PAGE_SPECIFIC_TOUR.STEP_NUMBER_LABEL", "of")} ${numberedSteps.length}` : '';
        }

        const goToByeStep = (introJSInstance) => {
            removeNoThanksButton();
            const tourName = introJSInstance.$tourName;
            const tourNameTranslated = translate('PAGE_SPECIFIC_TOUR.TOUR_NAME.' + $filter('uppercase')(tourName),$filter('capitalize')(tourName));
            const steps = introJSInstance._introItems;
            // Update the last step and go to it to reflect that the user has skipped the tour
            steps[steps.length - 1] = {
                title: translate("PAGE_SPECIFIC_TOUR.SKIPPED_OUTRO.TITLE", "👋 Bye for now!"),
                intro: translate('PAGE_SPECIFIC_TOUR.REACTIVATE.DESCRIPTION', `You can re-activate the {{tourName}} tour or browse resources at any time through the Help Center <i class="dku-icon-question-circle-fill-16 vab"></i>`, {tourName: tourNameTranslated}),
                element: document.querySelector('[data-page-tour="help-center-trigger"]'),
                position: 'bottom',
                hideStepNumber: true
            };
            introJSInstance.goToStep(steps.length);
        }

        const addNoThanksButton = (introJSInstance) => {
            if ($('.introjs-nothanksbutton').length) {
                return;
            }
            const tooltipButtons = $('.introjs-tooltipbuttons');
            var btn = document.createElement("button");
            btn.className = 'introjs-nothanksbutton btn btn--text btn--secondary';
            btn.innerText = translate("PAGE_SPECIFIC_TOUR.BUTTONS.NO_THANKS", 'No Thanks');
            btn.onclick = () => {
                onTourDismissed(introJSInstance);
                goToByeStep(introJSInstance);
            }
            const parentNode = tooltipButtons[0];
            parentNode.insertBefore(btn, parentNode.firstChild);
        }

        const removeNoThanksButton = () => {
            $('.introjs-nothanksbutton').remove();
        }

        return {
            TOUR_NAMES,
            checkFlowTourConditions,
            canStartFlowTour,
            startFlowTour,
            canStartExploreTour,
            startExploreTour,
            canStartPrepareTour,
            startPrepareTour,
            displayReopenTutorialPopup
        };
    }
})();
