import dataclasses
import enum
import json
import logging
from typing import Optional

from langchain_core.documents import Document

from dataiku.langchain.metadata_generator import DKU_MULTIMODAL_CONTENT


@dataclasses.dataclass
class MultimodalContent:

    class Type(enum.Enum):
        TEXT = "text"
        IMAGES = "images"

    content: str
    type: Type

    @classmethod
    def from_doc(cls, doc: "Document", full_folder_id: Optional[str]) -> Optional['MultimodalContent']:
        multimodal_content = doc.metadata.get(DKU_MULTIMODAL_CONTENT)

        if multimodal_content is None:
            logging.info(f"{DKU_MULTIMODAL_CONTENT} Not found in metadata. Available fields are: {doc.metadata.keys()}. Assuming we aren't in the multimodal case.")
            return None

        try:
            multimodal_content_dict = json.loads(multimodal_content)
        except Exception as e:
            raise ValueError(f"Metadata {multimodal_content} is not a valid json", str(e))

        multimodal_content_type_str = multimodal_content_dict.get("type")
        if multimodal_content_type_str is None:
            raise ValueError(f"Missing multipart type in metadata: {multimodal_content}")

        try:
            multimodal_content_type = MultimodalContent.Type(multimodal_content_type_str)
        except ValueError:
            logging.warning(f"Unsupported document type {multimodal_content_type_str}.")
            return None

        content = multimodal_content_dict.get("content")
        if content is None:
            logging.warning("Content not available in document info.")
            return None

        if multimodal_content_type == MultimodalContent.Type.IMAGES and not full_folder_id:
            logging.warning("Folder id is not set.")
            return None

        return cls(content, multimodal_content_type)
