-- list DELETE history for artifacts history table 
-- => check the list of signoffs for artifactID at the same timestamp of the deletion 
-- => if there are any without a DELETE revision for all the signoffs tied to the same action all the revision, add them
INSERT INTO signoffs_history (_revision_json, signoff_json)
SELECT     
    jsonb_build_object(
        'id', revision_to_create,
        'user', user_to_attach_to,
        'actionId', action_to_attach_to,
        'timestamp', timestamp_to_attach_to,
        'operation', 'DELETE',
        'entityType', 'SIGNOFF'
    ),
    signoff_to_delete_in_history
FROM (
    SELECT 
        NEXTVAL('history_revisions_seq') as revision_to_create,
        artifacts_history._revision_action_id as action_to_attach_to,
        artifacts_history._revision_json -> 'timestamp' as timestamp_to_attach_to,
        artifacts_history._revision_json -> 'user' as user_to_attach_to,
        signoff_json as signoff_to_delete_in_history
    -- list operations on artifacts
    FROM artifacts_history
    -- retrieve all the signoffs present at this time (get the last known revision per sign-off before or equal this time, and then filter out the deletions)
    JOIN LATERAL (
        SELECT signoff_step_id, signoff_json
        FROM signoffs_history
        WHERE _revision_id IN (
            SELECT DISTINCT ON (h.signoff_artifact_id, h.signoff_step_id) h._revision_id
            FROM signoffs_history h
            WHERE h._revision_timestamp <= artifacts_history._revision_timestamp
            AND h.signoff_artifact_id = artifacts_history.artifact_id
            ORDER BY h.signoff_artifact_id, h.signoff_step_id, h._revision_timestamp DESC, h._revision_id DESC
        )
        AND signoffs_history._revision_json ->> 'operation' != 'DELETE'
    ) t ON true
    -- limit to DELETE operations on artifacts
    WHERE artifacts_history._revision_json ->> 'operation' = 'DELETE'
    -- check that a DELETE revision is not already registered for the signoff in the same action
    AND artifacts_history._revision_action_id NOT IN (
        SELECT signoffs_history._revision_action_id
        FROM signoffs_history 
        WHERE signoffs_history._revision_action_id = artifacts_history._revision_action_id
        AND signoffs_history.signoff_artifact_id = artifacts_history.artifact_id
        AND signoffs_history.signoff_step_id = t.signoff_step_id
        AND signoffs_history._revision_json ->> 'operation' = 'DELETE'
    )
) s
;

-- list DELETE history for BPV history table 
-- => check the list of all SignoffConfigs tied to the BPV at this precise moment,
-- and if there are any SignoffConfigs without DELETE revision for all the SignoffConfigs tied to the same action, add them
INSERT INTO signoff_configurations_history (_revision_json, signoff_configuration_json)
SELECT     
    jsonb_build_object(
        'id', revision_to_create,
        'user', user_to_attach_to,
        'actionId', action_to_attach_to,
        'timestamp', timestamp_to_attach_to,
        'operation', 'DELETE',
        'entityType', 'SIGNOFF_CONFIGURATION'
    ),
    signoff_configuration_to_delete_in_history
FROM (
    SELECT    
        NEXTVAL('history_revisions_seq') as revision_to_create,
        blueprint_versions_history_deleted_with_creation_date._revision_action_id as action_to_attach_to,
        blueprint_versions_history_deleted_with_creation_date._revision_json -> 'timestamp' as timestamp_to_attach_to,
        blueprint_versions_history_deleted_with_creation_date._revision_json -> 'user' as user_to_attach_to,
        t.signoff_configuration_json as signoff_configuration_to_delete_in_history
    -- list DELETE operations on BPV with associated creation dates
    FROM (
        -- make a counter incrementing on ADD operations so if a BPV has multiple lives, each of them will have a unique life-number across all operations (ADD, UPDATES, DELETE)
        WITH blueprint_versions_history_lives AS (
            SELECT 
                SUM(CASE WHEN _revision_json ->> 'operation' = 'ADD' THEN 1 ELSE 0 END) OVER (
                    ORDER BY blueprint_version_blueprint_id ASC, blueprint_version_version_id ASC, _revision_timestamp ASC, _revision_id ASC
                ) AS life, 
                blueprint_versions_history.* 
            FROM blueprint_versions_history
        ),
        -- partition by life number and retain the first timestamp (creation date)
        blueprint_versions_history_lives_w_creation_date AS (
            SELECT 
                first_value(_revision_timestamp) OVER (
                    PARTITION BY blueprint_version_blueprint_id, blueprint_version_version_id, life 
                    ORDER BY _revision_timestamp ASC, _revision_id ASC
                ) AS creation_date, -- the earliest revision we can find in the history (either ADD or UPDATE if no ADD) 
                blueprint_versions_history_lives.*
            FROM blueprint_versions_history_lives
        )
        -- then filter only on delete operations
        SELECT *
        FROM blueprint_versions_history_lives_w_creation_date
        WHERE blueprint_versions_history_lives_w_creation_date._revision_json ->> 'operation' = 'DELETE'
    ) blueprint_versions_history_deleted_with_creation_date
    -- retrieve signoff_configurations in the timeframe of the BPV (>=creation date && <deletion date)
    JOIN LATERAL (
        SELECT signoff_configuration_step_id, signoff_configuration_json
        FROM signoff_configurations_history
        WHERE _revision_id IN (
            SELECT DISTINCT ON (h.signoff_configuration_blueprint_id, h.signoff_configuration_version_id, h.signoff_configuration_step_id) h._revision_id
            FROM signoff_configurations_history h
            WHERE h.signoff_configuration_blueprint_id = blueprint_versions_history_deleted_with_creation_date.blueprint_version_blueprint_id
            AND h.signoff_configuration_version_id = blueprint_versions_history_deleted_with_creation_date.blueprint_version_version_id
            AND h._revision_timestamp >= blueprint_versions_history_deleted_with_creation_date.creation_date
            AND h._revision_timestamp < blueprint_versions_history_deleted_with_creation_date._revision_timestamp
            ORDER BY h.signoff_configuration_blueprint_id, h.signoff_configuration_version_id, h.signoff_configuration_step_id, h._revision_timestamp DESC, h._revision_id DESC
        )
        AND signoff_configurations_history._revision_json ->> 'operation' != 'DELETE'
    ) t ON true
    -- check that a DELETE revision is not already registered for the SignoffConfiguration in the same action
    WHERE blueprint_versions_history_deleted_with_creation_date._revision_action_id NOT IN (
        SELECT signoff_configurations_history._revision_action_id
        FROM signoff_configurations_history 
        WHERE signoff_configurations_history._revision_action_id = blueprint_versions_history_deleted_with_creation_date._revision_action_id
        AND signoff_configurations_history.signoff_configuration_blueprint_id = blueprint_versions_history_deleted_with_creation_date.blueprint_version_blueprint_id
        AND signoff_configurations_history.signoff_configuration_version_id = blueprint_versions_history_deleted_with_creation_date.blueprint_version_version_id
        AND signoff_configurations_history.signoff_configuration_step_id = t.signoff_configuration_step_id
        AND signoff_configurations_history._revision_json ->> 'operation' = 'DELETE'
    )
) s;

-- list UPDATE history for BPV history table, 
-- also add a filter to check retrieve all the list of deleted steps with diff between old and new BPV definition 
-- => check the list of signoffs for all artifact ids for the BPV at that time AND that the signoff stepID is one of the list of deleted steps, 
-- and if there were no DELETE revision for these, add them

-- first fix the signoff config history with a modifying CTE, and return the needed information to then fix the signoff history
WITH fixed_signoff_configurations AS (
    INSERT INTO signoff_configurations_history (_revision_json, signoff_configuration_json)
    SELECT     
        jsonb_build_object(
            'id', revision_to_create,
            'user', user_to_attach_to,
            'actionId', action_to_attach_to,
            'timestamp', timestamp_to_attach_to,
            'operation', 'DELETE',
            'entityType', 'SIGNOFF_CONFIGURATION'
        ),
        signoff_configuration_to_delete_in_history
    FROM (
        -- make a counter incrementing on ADD operations so if a BPV has multiple lives, each of them will have a unique life-number across all operations (ADD, UPDATES, DELETE)
        WITH blueprint_versions_history_lives AS (
            SELECT 
                SUM(CASE WHEN _revision_json ->> 'operation' = 'ADD' THEN 1 ELSE 0 END) OVER (
                    ORDER BY blueprint_version_blueprint_id ASC, blueprint_version_version_id ASC, _revision_timestamp ASC, _revision_id ASC
                ) AS life, 
                blueprint_versions_history.* 
            FROM blueprint_versions_history
        ),
        -- partition by life number and retain the first timestamp (creation date)
        blueprint_versions_with_creation_date AS (
            SELECT 
                first_value(_revision_timestamp) OVER (
                    PARTITION BY blueprint_version_blueprint_id, blueprint_version_version_id, life 
                    ORDER BY _revision_timestamp ASC, _revision_id ASC
                ) AS creation_date, -- the earliest revision we can find in the history (either ADD or UPDATE if no ADD)
                blueprint_versions_history_lives.*
            FROM blueprint_versions_history_lives
        ),
        -- blueprint_versions_history + a LAG on previous json (by id, ordered by revision timestamp and _revision_id)
        blueprint_versions_history_with_previous AS (
            SELECT
                blueprint_versions_with_creation_date.*,
                LAG(blueprint_versions_with_creation_date.blueprint_version_json, 1) OVER(
                    PARTITION BY blueprint_version_blueprint_id, blueprint_version_version_id, life
                    ORDER BY _revision_timestamp ASC, _revision_id ASC
                ) as previous_blueprint_version_json
            FROM blueprint_versions_with_creation_date
        ),
        -- limit on UPDATE operations and add a deleted_step column computed from the before/after json
        blueprint_versions_history_update_with_deleted_step AS (
            SELECT
                blueprint_versions_history_with_previous.*,
                -- keep only keys (=stepIds from the following diff)
                jsonb_object_keys(    
                    -- build an object containing the previous stepIds as keys, and any value (so let's say 1)
                    (
                        SELECT jsonb_object_agg(wf_step ->> 'id', 1) 
                        FROM jsonb_array_elements(previous_blueprint_version_json -> 'workflowDefinition' -> 'stepDefinitions') wf_step
                    )
                    -- uses minus operator to remove keys specified by the following array built from the stepsIds after the update
                    -
                    ARRAY(
                        SELECT wf_step ->> 'id'
                        FROM jsonb_array_elements(blueprint_version_json -> 'workflowDefinition' -> 'stepDefinitions') wf_step
                    )
                ) as deleted_step -- jsonb_object_keys will make one row by key, so one step by row
            FROM blueprint_versions_history_with_previous
            WHERE blueprint_versions_history_with_previous._revision_json ->> 'operation' = 'UPDATE'
            AND previous_blueprint_version_json IS NOT NULL -- discard if no previous version (can happen for example on BPV that existed before history (V11000) because history was initialized with an 'UPDATE' operation)
        )
        SELECT
            NEXTVAL('history_revisions_seq') as revision_to_create,
            blueprint_versions_history_update_with_deleted_step._revision_action_id as action_to_attach_to,
            blueprint_versions_history_update_with_deleted_step._revision_json -> 'timestamp' as timestamp_to_attach_to,
            blueprint_versions_history_update_with_deleted_step._revision_json -> 'user' as user_to_attach_to,
            t.signoff_configuration_json as signoff_configuration_to_delete_in_history
        FROM blueprint_versions_history_update_with_deleted_step
        -- retrieve signoff_configuration in the timeframe of the step (>=creation date && < deletion date)
        JOIN LATERAL (
            SELECT signoff_configuration_step_id, signoff_configuration_json
            FROM signoff_configurations_history
            WHERE _revision_id IN (
                SELECT DISTINCT ON (h.signoff_configuration_blueprint_id, h.signoff_configuration_version_id, h.signoff_configuration_step_id) h._revision_id
                FROM signoff_configurations_history h
                WHERE h.signoff_configuration_blueprint_id = blueprint_versions_history_update_with_deleted_step.blueprint_version_blueprint_id
                AND h.signoff_configuration_version_id = blueprint_versions_history_update_with_deleted_step.blueprint_version_version_id
                AND h.signoff_configuration_step_id = blueprint_versions_history_update_with_deleted_step.deleted_step
                AND h._revision_timestamp >= blueprint_versions_history_update_with_deleted_step.creation_date
                AND h._revision_timestamp < blueprint_versions_history_update_with_deleted_step._revision_timestamp
                ORDER BY h.signoff_configuration_blueprint_id, h.signoff_configuration_version_id, h.signoff_configuration_step_id, h._revision_timestamp DESC, h._revision_id DESC
            )
            AND signoff_configurations_history._revision_json ->> 'operation' != 'DELETE'
        ) t ON true
        -- check that a DELETE revision is not already registered for the SignoffConfiguration in the same action
        WHERE blueprint_versions_history_update_with_deleted_step._revision_action_id NOT IN (
            SELECT signoff_configurations_history._revision_action_id
            FROM signoff_configurations_history 
            WHERE signoff_configurations_history._revision_action_id = blueprint_versions_history_update_with_deleted_step._revision_action_id
            AND signoff_configurations_history.signoff_configuration_blueprint_id = blueprint_versions_history_update_with_deleted_step.blueprint_version_blueprint_id
            AND signoff_configurations_history.signoff_configuration_version_id = blueprint_versions_history_update_with_deleted_step.blueprint_version_version_id
            AND signoff_configurations_history.signoff_configuration_step_id = t.signoff_configuration_step_id
            AND signoff_configurations_history._revision_json ->> 'operation' = 'DELETE'
        )
    ) s
    -- return the fixed signoff configs so we can use that as a base to fix the associated signoffs
    RETURNING _revision_json, signoff_configuration_json
)
INSERT INTO signoffs_history (_revision_json, signoff_json)
SELECT     
    jsonb_build_object(
        'id', revision_to_create,
        'user', user_to_attach_to,
        'actionId', action_to_attach_to,
        'timestamp', timestamp_to_attach_to,
        'operation', 'DELETE',
        'entityType', 'SIGNOFF'
    ),
    signoff_to_delete_in_history
FROM (
    SELECT 
        NEXTVAL('history_revisions_seq') AS revision_to_create,
        _revision_json -> 'user' AS user_to_attach_to, 
        _revision_json -> 'actionId' AS action_to_attach_to,
        _revision_json -> 'timestamp' AS timestamp_to_attach_to,
        t.signoff_json AS signoff_to_delete_in_history
    FROM fixed_signoff_configurations
    -- retrieve the artifacts present at the time of the revision, based on the BPV ID of the fixed signoff configs
    -- we can't search directly the signoff history table using signoff.configuration.id.blueprintVersionId, because it could have been migrated from one BPV to another, keeping the old config
    JOIN LATERAL (
        WITH artifacts_at_time AS (
            SELECT *
            FROM artifacts_history
            WHERE _revision_id IN (
                SELECT DISTINCT ON (h.artifact_id) h._revision_id
                FROM artifacts_history h
                WHERE h._revision_timestamp < epoc_to_timestamp(fixed_signoff_configurations._revision_json ->> 'timestamp')
                ORDER BY h.artifact_id, h._revision_timestamp DESC, h._revision_id DESC
            )
            AND _revision_json ->> 'operation' != 'DELETE'
        ),
        signoffs_at_time AS (
            SELECT *
            FROM signoffs_history
            WHERE _revision_id IN (
                SELECT DISTINCT ON (h.signoff_artifact_id, h.signoff_step_id) h._revision_id
                FROM signoffs_history h
                WHERE h._revision_timestamp < epoc_to_timestamp(fixed_signoff_configurations._revision_json ->> 'timestamp')
                ORDER BY h.signoff_artifact_id, h.signoff_step_id, h._revision_timestamp DESC, h._revision_id DESC
            )
            AND _revision_json ->> 'operation' != 'DELETE'
        )
        SELECT signoffs_at_time.signoff_artifact_id, signoffs_at_time.signoff_step_id, signoffs_at_time.signoff_json
        FROM artifacts_at_time 
        JOIN signoffs_at_time ON signoff_artifact_id = artifact_id
        WHERE artifacts_at_time.artifact_blueprint_id = fixed_signoff_configurations.signoff_configuration_json -> 'id' -> 'blueprintVersionId' ->> 'blueprintId' 
        AND artifacts_at_time.artifact_version_id = fixed_signoff_configurations.signoff_configuration_json -> 'id' -> 'blueprintVersionId' ->> 'versionId' 
        AND signoffs_at_time.signoff_step_id = fixed_signoff_configurations.signoff_configuration_json -> 'id' ->> 'stepId'
    ) t ON true
    -- check that a DELETE revision is not already registered for the signoff in the same action
    AND (fixed_signoff_configurations._revision_json ->> 'actionId')::bigint NOT IN (
        SELECT signoffs_history._revision_action_id
        FROM signoffs_history 
        WHERE signoffs_history._revision_action_id = (fixed_signoff_configurations._revision_json ->> 'actionId')::bigint
        AND signoffs_history.signoff_artifact_id = t.signoff_artifact_id
        AND signoffs_history.signoff_step_id = t.signoff_step_id
        AND signoffs_history._revision_json ->> 'operation' = 'DELETE'
    )
) s;
