import os, json, re, logging
from dataiku.code_studio import CodeStudioBlock, get_dataiku_user_uid
from block_utils import LibLocationPathReplacer
from distutils.version import LooseVersion

class NGINXCodeStudioBlock(CodeStudioBlock):
    def __init__(self, config, plugin_config):
        self.config = config
        self.plugin_config = plugin_config

    _ENTRYPOINT_FILE = "nginx-block-entrypoint.sh"
    _CONFIG_FILE = "nginx-block.conf"
    _CONFIG_TPL_FILE = "nginx-block.conf.tpl"
    _NGINX_PREFIX = "/home/dataiku/nginx-block"
        
    def _get_entrypoint_path(self):
        entrypoint_path = self.config.get("startScript", "/opt/dataiku")
        if entrypoint_path.endswith("/") or not entrypoint_path.endswith(".sh"):
            entrypoint_path = os.path.join(entrypoint_path, self._ENTRYPOINT_FILE)
        return entrypoint_path

    def _get_port(self):
        return self.config.get("port", 8888)
        
    def build_spec(self, spec, env):
        port = self._get_port()
        entrypoint_path = self._get_entrypoint_path()
        index = self.config.get("indexFiles", "index.html index.htm")
        locations = self.config.get("locations", [])

        # checks settings
        if len(locations) == 0:
            raise ValueError("NGINX Block: you should at least have one location")

        # compute locations
        replacer = LibLocationPathReplacer(spec)
        def getZonePath(loc):
            zone = loc.get("zone")
            path = replacer.replace_variable_by_path("__" + zone.upper() + "__")
            subpath = loc.get("zonePath", "").strip()
            if len(subpath) > 0:
                if not subpath.startswith("/"):
                    path += "/"
                path += subpath
            return path

        def textIndent(lines, indentation):
            txt = ""
            for line in lines.split("\n"):
                txt += (" " * indentation) + line.strip() + "\n"
            return txt

        def LocationToNGINXFormat(loc):
            path = loc.get("location", "/")
            mode = loc.get("mode", "zone")
            cfg = ""
            
            if mode == "zone":
                
                if not path.startswith("/"):
                    path = "/" + path
                root = getZonePath(loc)
                cfg += "alias " + root + ";\n"

                if loc.get("zoneDisableCache", False):
                    cfg += "add_header Last-Modified $date_gmt;\n"
                    cfg += "add_header Cache-Control 'private no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0';\n"
                    cfg += "if_modified_since off;\n"
                    cfg += "expires off;\n"
                    cfg += "etag off;\n"
                
                if loc.get("zoneListFiles", False):
                    cfg += "autoindex on;\n"
                    cfg += "autoindex_exact_size off;\n"
                    cfg += "autoindex_format html;\n"
                    cfg += "autoindex_localtime on;\n"

            elif mode == "proxy":
                
                proxy_port = loc.get("proxyPort")
                proxy_path = loc.get("proxyPath")
                if len(proxy_path) > 0 and not proxy_path.startswith("/"):
                    proxy_path = "/" + proxy_path
                cfg += "proxy_pass http://localhost:%s%s;\n" % (proxy_port, proxy_path)
                
                if loc.get("proxyWebsocket", False):
                    cfg += "proxy_http_version 1.1;\n"
                    cfg += "proxy_set_header Upgrade $http_upgrade;\n"
                    cfg += "proxy_set_header Connection \"Upgrade\";\n"
                    cfg += "proxy_set_header Host $host;\n"

            elif mode == "custom":
                
                cfg += loc.get("customSettings", "") + "\n"

            if loc.get("subFilters", True):
                cfg += "sub_filter_types text/html text/css application/javascript;\n"
                cfg += "sub_filter 'src=\"/' 'src=\"$http_x_dku_basehref_def/';\n"
                cfg += "sub_filter 'src = \"/' 'src = \"$http_x_dku_basehref_def/';\n"
                cfg += "sub_filter 'href=\"/' 'href=\"$http_x_dku_basehref_def/';\n"
                cfg += "sub_filter 'href = \"/' 'href = \"$http_x_dku_basehref_def/';\n"
                cfg += "sub_filter 'action=\"/' 'action=\"$http_x_dku_basehref_def/';\n"
                cfg += "sub_filter 'action = \"/' 'action = \"$http_x_dku_basehref_def/';\n"
                cfg += "sub_filter '@import \"/' '@import \"$http_x_dku_basehref_def/';\n"
                cfg += "sub_filter 'url(\"/' 'url(\"$http_x_dku_basehref_def/';\n"
                cfg += "sub_filter '__DKU_RUNTIME_BASE_PATH__' '$http_x_dku_basehref_def';\n"
                cfg += "sub_filter_once off;\n"

            return "    location " + loc.get("pathMatchMode", "") + " " + path + " {\n" + textIndent(cfg, 6) + "    }"


        nginx_locations = [LocationToNGINXFormat(loc) for loc in locations]

        # generate index files list
        index_basename = re.split("\s+", index.strip())[0]
        index_files = ""
        location_paths = ""
        for loc in locations:
            if loc.get("mode", "zone") == "zone" and loc.get("zoneGenerateIndex", False):
                zone = loc.get("zone")
                index_files +=  "\"" + os.path.join(getZonePath(loc), index_basename) + "\" "
                location_paths += "\"" + loc.get("location", "/") + "\" "

        # add the nginx config in the buildir
        nginx_config = """
# This file is automatically generated.

daemon            off;
working_directory {nginx_prefix};
worker_processes  auto;
error_log         stderr;
pid               {nginx_prefix}/nginx-block.pid;
worker_rlimit_nofile 8192;

events {{
    worker_connections  1024;
}}

http {{
  include               /etc/nginx/mime.types;
  default_type          application/octet-stream;
  client_max_body_size  0;
  client_body_temp_path {nginx_prefix}/tmp;
  proxy_temp_path       {nginx_prefix}/tmp;
  fastcgi_temp_path     {nginx_prefix}/tmp;
  scgi_temp_path        {nginx_prefix}/tmp;
  uwsgi_temp_path       {nginx_prefix}/tmp;
  large_client_header_buffers 8 64k;

  index                 {index};
  log_format   anonymous '$remote_addr - - [$time_local]  $status '
                    '"$request" $body_bytes_sent "$http_referer" '
                    '"$http_user_agent" "$http_x_forwarded_for"';
  access_log   {nginx_prefix}/access.log anonymous;
  error_log    {nginx_prefix}/error.log;
  sendfile     on;

  keepalive_timeout  65;

  map $http_x_dku_basehref $http_x_dku_basehref_def {{
    default "__DKU_CODE_STUDIO_BROWSER_PATH__";
    ~.* "$http_x_dku_basehref";
  }}

  server {{
    listen {port};
{locations}
    {user_settings}
  }}
}}
""".format(nginx_prefix=self._NGINX_PREFIX, 
           index=index,
           port=port,
           locations="\n".join(nginx_locations),
           user_settings=self.config.get("serverUserSettings", ""))
        with open(os.path.join(env["buildDir"], self._CONFIG_TPL_FILE), "wb") as f:
            f.write(nginx_config.encode("utf8"))

        logging.info("NGINX BLOCK CONF:\n" + nginx_config)

        # add the entrypoint script in the buildir
        entrypoint_script = """#!/bin/bash

USER=dataiku
HOME=/home/dataiku

# generate index
files=({index_files})
locations=({location_paths})
for ((i=0; i<${{#files[@]}}; i++))
do
    file="${{files[$i]}}"
    if [ ! -f "$file" ]
    then
        mkdir -p $(dirname $file)
        location="${{locations[$i]}}"
        cat << 'EOF'| sed "s|__LOC__|$location|g" > $file
<!doctype html>
<html>
<!--
    NOTICE: 
        All your http paths, such as SRC or HREF attributes, 
        _should_ be relatives. But you can use __DKU_RUNTIME_BASE_PATH__ 
        as prefix of your absolute url.
-->
<head>
	<meta charset="UTF-8">
	<title>Index</title>
</head>
<html>
<head>
    <title>Sample index file</title>
    <meta name="description" content="auto generated sample index file">
    <style>
        header {{
            font-size: smaller;
        }}
        h1 {{
            text-align: center;
            margin-top: 10%;
        }}
    </style>
</head>
<body>
    <header>
        <a href="__DKU_RUNTIME_BASE_PATH__/">home</a>
    </header>
    <content>
        <h1>Welcome to __LOC__ !</h1>
    </content>
</body>
</html>
EOF
    fi
done

# generate nginx conf with the current proxified path for sub_filter rules
DKU_POD_BASE_URL=$(eval echo "$DKU_CODE_STUDIO_BROWSER_PATH_{port}")
sed "s|__DKU_CODE_STUDIO_BROWSER_PATH__|$DKU_POD_BASE_URL|g" {nginx_prefix}/{nginx_config_tpl_file} > {nginx_prefix}/{nginx_config_file}

LC_ALL=en_US.utf8 nginx -p {nginx_prefix} -c {nginx_config_file}
""".format(index_files=index_files,
           location_paths=location_paths,
           nginx_prefix=self._NGINX_PREFIX, 
           nginx_config_file=self._CONFIG_FILE,
           nginx_config_tpl_file=self._CONFIG_TPL_FILE,
           port=port)
        with open(os.path.join(env["buildDir"], self._ENTRYPOINT_FILE), "wb") as f:
            f.write(entrypoint_script.encode("utf8"))
            
        # the dockerfile addition
        spec["dockerfile"] = spec.get("dockerfile", "") + """

##### NGINX BLOCK #####

USER root
WORKDIR /opt/dataiku

# nginx config
RUN mkdir -p {nginx_prefix}/tmp
COPY {config_tpl_file} {nginx_config_tpl_path}
RUN chown -R dataiku:root {nginx_prefix}

# entrypoint.sh
COPY {entrypoint_file} {entrypoint_path}
RUN chown dataiku:root {entrypoint_path} && chmod +x {entrypoint_path}

# USER dataiku
USER {uid}
WORKDIR /home/dataiku
""".format(nginx_prefix=self._NGINX_PREFIX,
           config_tpl_file=self._CONFIG_TPL_FILE,
           nginx_config_tpl_path=self._NGINX_PREFIX + "/" + self._CONFIG_TPL_FILE,
           entrypoint_path=entrypoint_path, 
           entrypoint_file=self._ENTRYPOINT_FILE,
           uid=get_dataiku_user_uid())
        return spec

    def build_launch(self, spec, env):
        if env['launchedFrom'] == 'WEBAPP' and not self.config.get("useInWebapps", False):
            return spec
        port = self._get_port()
        spec['entrypoints'] = spec.get('entrypoints', []) + [self._get_entrypoint_path()]

        base_path = self.config.get("exposedPath", "/")

        readiness_probe_url = "http://localhost:" + str(port) + base_path
        if spec.get('readinessProbeUrl', "") == "":
            spec['readinessProbeUrl'] = readiness_probe_url

        exposed_port = {
            "label": "NGINX", 
            "proxiedUrlSuffix": base_path, 
            "exposeHtml": True, 
            "port": port,
            "readiness_probe_url": readiness_probe_url
        }
        spec['exposedPorts'] = spec.get('exposedPorts', []) + [exposed_port]
        return spec
        
    def build_creation(self, spec, env):
        return spec
        
