(function(){
'use strict';


const app = angular.module('dataiku.datasets.directives', ['dataiku.filters', 'colorpicker.module', 'colorContrast']);


    app.directive('simpleTableContent', () => {
        return {
            scope : {
                "table" : "="
            },
            replace : true,
            link : function(scope, element) {
                /* Refresh of the table content itself */
                scope.$watch("table", () => {
                    dispatchCustomTimelineEvent('StartSimpleTableContentUpdate');
                    if (scope.table == null) return;
                    var table = scope.table;
                    var tableData = "";//<tbody>";

                    for (var rowIdx in table.rows) {
                        tableData += "<tr>";
                        for (var cellIdx in table.rows[rowIdx].cells) {
                            var cell = table.rows[rowIdx].cells[cellIdx];
                            tableData += '<td class="cell"';
                            if (!angular.isUndefined(cell.value) && cell.value.length > 40) {
                                tableData +=' title="' + sanitize(cell.value) + '"';
                            }

                            if (cell.validSt == 'E') { // Empty
                                 tableData += '><div class="cell empty">';
                             } else {
                                 tableData += '><div class="cell filled">';
                            }
                            tableData +=  angular.isUndefined(cell.value) ? '&nbsp;' : sanitize(cell.value);
                            tableData += "</div></td>";
                        }
                        tableData += "</tr>";
                    }
                    tableData += "";//</tbody>";
                    $(element).html(tableData);
                    dispatchCustomTimelineEvent('EndSimpleTableContentUpdate');
                });
            }
        };
    });


    app.directive('simpleColumnHeader', function() {
        return {
            restrict: 'E',
            replace: true,
            templateUrl: '/templates/datasets/simple_column_header.html',
            scope: {
                columnIndex : '=',
                column : '=',
                dataset : '='
            },
            link: (scope) => {
                if (scope.columnIndex<0) return;

                scope.schema = {};
                scope.$watch("dataset.schema", (nv) => {
                    if (!nv) return;
                    scope.schema = scope.dataset.schema.columns[scope.columnIndex];
                });

                scope.column.selectedType.totalCount = (scope.column.selectedType.nbOK + scope.column.selectedType.nbNOK + scope.column.selectedType.nbEmpty);
                scope.column.okPercentage = scope.column.selectedType.nbOK * 100 / scope.column.selectedType.totalCount;
                scope.column.emptyPercentage = scope.column.selectedType.nbEmpty * 100 / scope.column.selectedType.totalCount;
                scope.column.nonemptyPercentage = (scope.column.selectedType.totalCount - scope.column.selectedType.nbEmpty) * 100 / scope.column.selectedType.totalCount;
                scope.column.nokPercentage = scope.column.selectedType.nbNOK * 100 / scope.column.selectedType.totalCount;
            }
        };
    });


    app.directive('simpleDetectionPreviewTable', () => {
        return {
            scope : {
                "headers" : "=",
                "table" : "=",
                "dataset" : "=",
                "setSchemaUserModified" : '=',
                "schemaIsUserEditable" : "="
            },
            replace : true,
            templateUrl : '/templates/datasets/fragments/simple-detection-preview-table.html',
            link : ($scope) => {
                $scope.$watch('table', () => {
                    if ($scope.table == null) {
                        return;
                    }
                    $scope.columnCount = $scope.headers.length;
                    if ($scope.columnCount > 320) {
                        $scope.tooManyColumns = true;
                        var truncateRowOrHeader = function(a, midElement) {
                            var ret = [];
                            var getWithRealIndex = function(a, idx) {
                                var o = a[idx];
                                o.realIndex = idx;
                                return o;
                            };
                            for (let i = 0; i < 300; i++) {ret.push(getWithRealIndex(a, i));}
                            midElement.realIndex = -1;
                            ret.push(midElement);
                            for (let i = 0; i < 20;i ++) {ret.push(getWithRealIndex(a, a.length - 20 + i));}
                            return ret;
                        };
                        // generate truncated table
                        $scope.displayedHeaders = truncateRowOrHeader($scope.headers, {name:'... ' + ($scope.headers.length - 320) + ' columns ...', selectedType:{}});
                        $scope.displayedTable = {
                                displayedRows : $scope.table.displayedRows, // keep stats
                                totalDeletedRows : $scope.table.totalDeletedRows,
                                totalEmptyCells : $scope.table.totalEmptyCells,
                                totalFullCells : $scope.table.totalFullCells,
                                totalKeptRows : $scope.table.totalKeptRows,
                                totalRows : $scope.table.totalRows,
                                headers : $scope.displayedHeaders,
                                rows : $scope.table.rows.map(function(row) {return {origRowIdx:row.origRowIdx, cells:truncateRowOrHeader(row.cells, {value:'...'})}})
                        };
                    } else {
                        $scope.tooManyColumns = false;
                        $scope.displayedTable = $scope.table;
                        $scope.displayedHeaders = $scope.headers;
                    }
                });
            }
        };
    });


    app.directive('schemaConsistencyStatus', function(translate) {
        return {
            templateUrl: '/templates/datasets/schema-consistency-status.html',
            scope: {
                consistency: '=',
                overwriteSchema: '=',
                clearManagedDataset:  '=',
                checkConsistency: '=',
                discardConsistencyError: '=',
                pushColumnDescriptionsToTable: '=',
                managed: '=',
                schemaJustModified: '=',
                currentSchema: '='
            },
            link: (scope) => {
                scope.translate = translate;
            }
        };
    });


    app.directive('simpleEditableColumnHeader', function() {
        return {
            restrict: 'E',
            replace: true,
            templateUrl: '/templates/datasets/simple_editable_column_header.html',
            scope: {
                column : '=',
                columnIndex : '=',
                setSchemaUserModified : '=',
                dataset : '='
            },
            link: (scope) => {
                if ( scope.columnIndex < 0) return;

                scope.schema = {};
                scope.$watch(
                    () => scope.dataset.schema.columns[scope.columnIndex],
                    () => {
                        scope.schema = scope.dataset.schema.columns[scope.columnIndex];
                    }
                );


                scope.column.selectedType.totalCount = (scope.column.selectedType.nbOK + scope.column.selectedType.nbNOK + scope.column.selectedType.nbEmpty);
                scope.column.okPercentage = scope.column.selectedType.nbOK * 100 / scope.column.selectedType.totalCount;
                scope.column.emptyPercentage = scope.column.selectedType.nbEmpty * 100 / scope.column.selectedType.totalCount;
                scope.column.nonemptyPercentage = (scope.column.selectedType.totalCount - scope.column.selectedType.nbEmpty) * 100 / scope.column.selectedType.totalCount;
                scope.column.nokPercentage = scope.column.selectedType.nbNOK * 100 / scope.column.selectedType.totalCount;
            }
        };
    });
    
    app.component('datasetPreviewTable', {
        bindings: {
            projectKey: '<',
            datasetName: '<',
            contextProjectKey: '<',
            embeddedLoadingDisplay: '<?',
            showName: '<?',
            showMeaning: '<?',
            showStorageType: '<?',
            showDescription: '<?',
            showCustomFields: '<?',
            showProgressBar: '<?',
            showHeaderSeparator: '<?',
            disableHeaderMenu: '<?',
            shakerOrigin: '<?',
            shakerColoringScheme: '<?'
        },
        templateUrl: '/templates/datasets/dataset-preview-table.html',
        controller: ['$scope', function ctrlDatasetPreviewTable($scope) {
            const ctrl = this;
            ctrl.$onInit = function () {
                // Populate $scope properties with those from the controller
                // Some directives need these, like for example `shakerForPreview`
                $scope.projectKey = ctrl.projectKey;
                $scope.datasetName = ctrl.datasetName;
                $scope.contextProjectKey = ctrl.contextProjectKey;
                $scope.showName = ctrl.showName;
                $scope.showMeaning = ctrl.showMeaning;
                $scope.showStorageType = ctrl.showStorageType;
                $scope.showDescription = ctrl.showDescription;
                $scope.showCustomFields = ctrl.showCustomFields;
                $scope.showProgressBar = ctrl.showProgressBar;
                $scope.showHeaderSeparator = ctrl.showHeaderSeparator;
                $scope.disableHeaderMenu = ctrl.disableHeaderMenu;
                $scope.shakerOrigin = ctrl.shakerOrigin;
                $scope.shakerColoringScheme = ctrl.shakerColoringScheme

                // Handle the case where the loading display (spinner + future waiting) must be embedded inside this preview component
                $scope.embeddedLoadingDisplay = ctrl.embeddedLoadingDisplay;
                if (ctrl.embeddedLoadingDisplay) {
                    $scope.refreshNoSpinner = true; // disable the global spinner for refresh request
                    $scope.$on('shakerTableRefresh', startLoading);
                    $scope.$on('shakerTableChangedGlobal', stopLoading);
                    $scope.$on('shakerTableChangeFailedGlobal', stopLoading);
                    $scope.$on('shakerTableChangeAbortedGlobal', stopLoading);
                }
            };

            ctrl.$onChanges = function () {
                $scope.projectKey = ctrl.projectKey;
                $scope.datasetName = ctrl.datasetName;
                $scope.contextProjectKey = ctrl.contextProjectKey;
                $scope.$broadcast('refresh-preview-table');
            }

            function startLoading() {
                $scope.isLoading = true;
            }

            function stopLoading() {
                $scope.isLoading = false;
            }
        }]
    });

    app.directive('excelSheetsSelection', () => {
        //To validate ranges - must be of a pattern like: `1` OR `2-3` OR `-4` OR `5-`
        const rangeRegExp = /^\s*(?<lower>\d*)\s*(-\s*(?<upper>\d*)\s*)?$/;
        return {
            restrict: 'E',
            templateUrl: '/templates/datasets/fragments/excel-sheets-selection.html',
            scope: {
                formatParams : '=',
                detectionResults : '<',
                onParamChange : '&'
            },
            link: function($scope) {
                $scope.sheetSelectionModes = [['NAMES', 'By names'],
                                                    ['ALL', 'All'],
                                                   ['PATTERN', 'By pattern'],
                                                   ['RANGES', 'By indices']];
                $scope.sheetSelectionModesDesc = ['Select only individual sheets from the file',
                                                        'Select all sheets from the file',
                                                       'Select sheets whose names match a regular expression',
                                                       'Select a range of sheets such as 5, 7-10, 15-, -3'];
                if (!$scope.formatParams.sheetSelectionMode) {
                    //Just to avoid "Nothing selected" ever appearing as it causes an off-by-one issue 
                    // though we expect a default value to be there for sheetSelectionMode
                    $scope.formatParams.sheetSelectionMode = "NAMES";
                }

                function isSheetsConfigValid() {
                    if ($scope.formatParams.sheetSelectionMode === "RANGES"
                            && (!$scope.formatParams.sheetRanges || !$scope.isRangesStringValid($scope.formatParams.sheetRanges))) {
                        return false;
                    }
                    if ($scope.formatParams.sheetSelectionMode  === "PATTERN" 
                            && (!$scope.formatParams.sheetPattern || !$scope.isPatternStringValid($scope.formatParams.sheetPattern))) {
                        return false;
                    }
                    return true;
                }

                $scope.$watchGroup(["formatParams.sheetRanges",  "formatParams.sheetPattern", "formatParams.sheetSelectionMode", "formatParams.sheets"], () => {
                    if (isSheetsConfigValid()) {
                        $scope.onParamChange();
                    }
                });

                $scope.isRangesStringValid = function(rangesString) {
                     //If there is no string this is not considered invalid here, but the field will be outlined in red due to being required  
                    if (rangesString) {
                        const rangeStrings = rangesString.split(',');
                        for (const range of rangeStrings) {
                            if (!range.trim().length) {
                                return false;
                            }
                            const match = range.match(rangeRegExp);
                            if (!match
                                || (match.groups.lower && parseInt(match.groups.lower) === 0)
                                || (match.groups.upper && parseInt(match.groups.upper) === 0)
                                || (!match.groups.lower && !match.groups.upper)
                                || (match.groups.lower && match.groups.upper
                                    && parseInt(match.groups.lower) > parseInt(match.groups.upper))) {
                                return false;
                            }
                        }
                    }
                    return true;
                };

                $scope.isPatternStringValid = function(patternString) {
                    //If there is no string this is not considered invalid here, but the field will be outlined in red due to being required 
                    if (patternString) {
                        try {
                            new RegExp(patternString);
                        } catch (e) {
                            return false;
                        }
                    }
                    return true;
                };
            }
        };
    });

    app.directive('excelSheets', function(Assert) {
        return {
            template: '<div>' +
                    '<ul class="excel-sheets__checkbox-list">' +
                        '<li ng-repeat="sheet in sheets">' +
                            '<input type="checkbox" ng-model="sheet.isSelected" /> {{sheet.name}}' +
                        '</li>' +
                    '</ul>' +
                    '<span class="dataset-format-params-section-description" ng-if="!detectionResults">Update dataset preview to list all sheets.</span>' +
                '</div>',
            restrict :'E',
            scope: {
                formatSheets : '=',
                detectionResults : '<'
            },
            link: (scope) => {
                function encodeSheets(sheetNames) {
                    return '*' + sheetNames.join('\n'); // Use '*' as first character to denote a list of sheet names.
                }
                function decodeSheets(str, meta) {
                    if (!angular.isDefined(str) || str == "") {
                        return [];
                    }
                    if (str.charAt(0) === '*') {
                        return str.substring(1).split("\n"); // List of sheet names
                    } else {
                        // List of sheet indexes (for compatibility)
                        return str.split(',').map(sheetIndex => meta["sheet." + sheetIndex + ".name"]);
                    }
                }

                scope.oldMeta = null;
                scope.$watch("detectionResults", () => {
                    if (scope.detectionResults == null) {
                        return;
                    }
                    
                    Assert.trueish(scope.detectionResults.format, 'no detected format');
                    if (scope.detectionResults.format == null) {
                        return;
                    }

                    const meta = scope.detectionResults.format.metadata;
                    if (angular.equals(scope.oldMeta, meta)) {
                        return;
                    }
                    if (scope.oldMeta == null) {
                        scope.oldMeta = angular.copy(meta);
                    }
                    const selectedSheetNames = decodeSheets(scope.formatSheets, meta);
                    if (meta && angular.isDefined(meta.nbSheets)) {
                        scope.sheets = [];
                        for (let i = 0; i < meta.nbSheets; i++) {
                            const sheetName = meta["sheet." + i + ".name"];
                            scope.sheets.push({"name": sheetName, "isSelected": selectedSheetNames.includes(sheetName)});
                        }
                    }
                }, true);

                scope.$watch("sheets", () => {
                    if (scope.sheets == null) {
                        return;
                    }
                    let selectedSheetNames = [];
                    for (let i in scope.sheets) {
                        const sheet = scope.sheets[i];
                        if (sheet.isSelected) {
                            selectedSheetNames.push(sheet.name);
                        }
                    }
                    scope.formatSheets = encodeSheets(selectedSheetNames);
                }, true);

                // Display the selected sheet names if we already know their names before detectionResults.
                if (scope.formatSheets && scope.formatSheets.startsWith("*")) {
                    scope.sheets = decodeSheets(scope.formatSheets).map(sheetName => ({ "name": sheetName, "isSelected": true }));
                }
            }
        };
    });


    app.directive('excelSheetsMatchingPattern', function() {
        return {
            template: '<div>' +
                    '<div ng-show="(!matchedSheets || matchedSheets.length === 0) && formatPattern " class="dataset-format-params-section-description">No matched sheets</div>' +
                    '<div ng-show="!formatPattern" class="dataset-format-params-section-description">Matched sheets will show here</div>' +
                    '<ul ng-show="matchedSheets" class="excel-sheets__plain-list">' +
                        '<li ng-repeat="sheet in matchedSheets">' +
                            '{{sheet}}' +
                        '</li>' +
                    '</ul>' +
                    '<span class="dataset-format-params-section-description" ng-show="!detectionResults">Update dataset preview to list sheets.</span>' +
                '</div>',
            restrict :'E',
            scope: {
                formatPattern : '=',
                detectionResults : '<'
            },
            link: (scope) => {
                scope.$watchGroup(["detectionResults", "formatPattern"], () => {
                    if (!scope.detectionResults || !scope.detectionResults.format) {
                        scope.matchedSheets = [];
                        return;
                    }

                    const meta = scope.detectionResults.format.metadata;
                    const metaChanged = !angular.equals(scope.oldMeta, meta);
                    const patternChanged = !angular.equals(scope.oldFormatPattern, scope.formatPattern);

                    if (patternChanged) {
                        scope.oldFormatPattern = scope.formatPattern;
                    }                        
                    if (metaChanged) {
                        scope.oldMeta = angular.copy(meta);
                    }
                    
                    if (metaChanged || patternChanged) {
                        scope.matchedSheets = [];
                        if (scope.formatPattern && meta && isFinite(meta.nbSheets)) {
                            scope.regexp = new RegExp(scope.formatPattern); 
                            for (let i = 0; i < meta.nbSheets; i++) {
                                const sheetName = meta["sheet." + i + ".name"];
                                if (scope.regexp.test(sheetName) ) {
                                    scope.matchedSheets.push(sheetName);
                                }
                            }
                        }
                    }
                }, true);
            }
        }
    });


    app.directive('possibleXpaths', function(Assert) {
        return {
            template: '<div class="xpath-tree">'
                     +'   <div ng-repeat="elem in xpaths" ng-include="\'/templates/datasets/format-xml-xpath-group.html\'" />'
                     +'</div>',
            restrict: 'E',
            scope: true,
            link: (scope) => {
                scope.$watch("detectionResults", () => {
                    if (scope.detectionResults == null) {
                        return;
                    }
                    Assert.trueish(scope.detectionResults.format, 'no detected format');
                    if (scope.detectionResults.format == null) {
                        return;
                    }
                    // a new format detection was done, so the metadata *could* have changed
                    const meta = scope.detectionResults.format.metadata;
                    if (meta && angular.isDefined(meta.possibleXPaths)) {
                        // backend sends a flat list of elements with depth and counts, frontend need to prepare for display
                        function groupRecursively(elems, treeDepth) {
                            if (!elems || !elems.length) {
                                return [];
                            }
                            elems.sort((a,b) => a.depth - b.depth);
                            // take the top level for the next grouping session
                            const minDepth = elems[0].depth;
                            const groups = elems.filter(elem => elem.depth == minDepth);
                            groups.forEach(function(group) {
                                group.treeDepth = treeDepth;
                            });
                            // split the rest in groups
                            elems.forEach(function(elem) {
                                const parent = groups.filter(function(group) {
                                    return elem.xpath.length > group.xpath.length && elem.xpath.slice(0, group.xpath.length) == group.xpath;
                                });
                                if (parent.length == 0 || parent.length > 1) {
                                    return; // not supposed to happen, by construction of the xpaths
                                }
                                elem.xpathSuffix = elem.xpath.slice(parent[0].xpath.length);
                                parent[0].xpaths = parent[0].xpaths || [];
                                parent[0].xpaths.push(elem);
                            });
                            // recurse in the groups, as needed
                            groups.forEach(function(group) {
                                if (group.xpaths !== undefined && group.xpaths.length > 0) {
                                    group.xpaths = groupRecursively(group.xpaths, treeDepth + 1);
                                }
                            });
                            groups.sort((a,b) => b.count - a.count); // sort by decreasing count
                            return groups;
                        }

                        scope.xpaths = groupRecursively(JSON.parse(meta.possibleXPaths), 0);
                    }
                }, true);

                scope.selectXpath = function(xpath) {
                    if (scope.lastFocusedXpathFieldSetter != null) {
                        scope.lastFocusedXpathFieldSetter(xpath, scope.dataset.formatParams.rootPath);
                        // force the preview to refresh
                        scope.onFormatParamsChanged();
                    }
                };
            }
        };
    });


    app.directive('xpathField', function() {
    	return {
        	restrict:'A',
        	require:'^ngModel',
            scope: true,
            link : function(scope, element, attrs, ngModel) {
            	var xpathType = attrs['xpathField'];
               	element.on('focus', function() {
                        scope.xpathFieldGotFocus(function(value, rootElementXpath) {
            			// be smarter and make the xpath appropriate to where it's set
            			if (rootElementXpath != null) {
                			if ( xpathType == 'parent') {
                				// for nodes that we know are parents of the root element, only take the attributes
                				if (rootElementXpath.startsWith(value))
                					value = value + '/@*';
                			}
                			if ( xpathType == 'child') {
                				// for nodes that we know are children of the root element, make them relative
                				if (value.startsWith(rootElementXpath))
                					value = '.' + value.substring(rootElementXpath.length);
                			}
            			}
            			// set the new value in the field and render
                		ngModel.$setViewValue(value);
                		ngModel.$render();
            		});
            	});
            }
        };
    });

    app.directive('datasetPathInput', function($timeout, PathUtils) {
        return {
            restrict: 'E',
            replace: true,
            templateUrl: '/templates/datasets/dataset-path-input.html',
            scope: {
                title: '@',
                path: '=',
                browseFn: '=',
                validateFn: '=',
                connection: '@',
                changeNeedsConfirm: '='
            },
            link: (scope) => {
                scope.changeConfirmed = false;

                // -- file selection

                scope.navObj = {selectedItems: []};
                scope.canBrowse = () => true;
                scope.canSelect = () => true;

                // -- browsing

                scope.onToggleBrowse = function() {
                    scope.browseActive = !scope.browseActive;
                    if (scope.browseActive) {
                        scope.path = scope.path || '/';
                        scope.navObj.browsePath = scope.path;
                    }
                };

                scope.onOKClick = function() {
                    if (scope.navObj.selectedItems.length == 0) {
                        scope.path = scope.navObj.browsePath;
                    } else {
                        const item = scope.navObj.selectedItems[0];
                        const path = item.fullPath;
                        scope.path = item.directory ? PathUtils.makeT(path) : path;
                    }
                    scope.browseActive = false;
                    /* Evaluate in timeout so that the nw path is correctly propagated before triggering change handlers
                     * (else, issues with digest cycles) */
                    $timeout(function() {
                        scope.validateFn();
                    });
                };

                scope.onCancelClick = function() {
                    scope.browseActive = false;
                };

                scope.$watch("connection", () => {
                    scope.browsePath = '/';
                    scope.onCancelClick();
                })
            }
        };
    });

    app.directive('folderPathInput', (openDkuPopin, PathUtils) => {
        return {
            restrict: 'E',
            replace: true,
            templateUrl: '/templates/projects-list/folder-path-input.html',
            scope: {
                title: '@',
                path: '=',
                browseFn: '=',
                validateFn: '=',
                connection: '@',
                changeNeedsConfirm: '=',
                displayItemFn: '=?',
                canSelectFn: '=?',
                folder: '=?',
                showRootFolderPath: '=?',
                cantWriteContentVerb: '@',
                searchable: '@',
                inputId: '@',
                inputEnabled: '&?',
            },
            link: (scope, element, attrs) => {
                scope.searchable = attrs.searchable ? scope.$eval(scope.searchable) : false;
                scope.changeConfirmed = false;
                const targetElement = element[0].querySelector('.browse-path-dropdown');
                const template = $(targetElement).detach();

                // -- file selection

                scope.navObj = { selectedItems: [] };
                scope.canBrowse = () => true;

                scope.canSelect = item => scope.canSelectFn ? scope.canSelectFn(item) : false;

                scope.displayItem = item => scope.displayItemFn ? scope.displayItemFn(item) : item;

                scope.inputEnabled = scope.inputEnabled ? scope.inputEnabled : () => true;

                scope.currentFolder = scope.folder;

                // -- browsing

                scope.onToggleBrowse = (event) => {
                    scope.browseActive = !scope.browseActive;

                    const isElsewhere = (tooltipElement, event) => $(event.target).parents('.browse-path-dropdown').length === 0 && $(event.target).parents('.browse-path-input').length === 0;
                    const onDismiss = () => scope.browseActive = false;

                    scope.dismissPopin = openDkuPopin(scope, event, { popinPosition: 'SMART', arrow: false, doNotCompile: true, onDismiss, isElsewhere, template });
                    if (scope.browseActive) {
                        if (scope.folder) {
                            scope.navObj.browsePath = scope.folder.id;
                        } else {
                            scope.navObj.browsePath = scope.path;
                        }
                    }
                };

                scope.onOKClick = () => {
                    if (!scope.browseActive ) return;
                    const selectedItems = scope.navObj.selectedItems;

                    if (scope.folder) {
                        scope.folder = scope.currentFolder;
                    }
                    scope.path = selectedItems.length === 1 ? selectedItems[0].fullPath : scope.navObj.browsePath;
                    if (scope.path === '/') {
                        scope.path = "";
                    }
                    scope.path = PathUtils.makeNLNT(scope.path);
                    scope.browseActive = false;
                    scope.dismissPopin();
                };

                scope.browseDoneFn = folder => {
                    // show / if folder is root
                    if (scope.showRootFolderPath && !folder.pathElts) {
                        folder.pathElts = '/';
                    }

                    scope.currentFolder = folder;
                };

                scope.onCancelClick = () => {
                    if (!scope.browseActive ) return;
                    scope.browseActive = false;
                    scope.dismissPopin();
                };
            }
        };
    });

    // This directive is much more complex than it should, because it has very specific properties
    // to not break the existing UI.
    // For instance:
    // - The model object is updated in-place (except when it's not impossible)
    // - The model is not updated if it has not been modified by the user
    //   (auto fixup is not propagated until necessary)
    // - It internally synchronizes 3 states : the model, the tree view model, and the JSON
    //
    // Notice: it's probably the worst code I ever wrote!
    app.service('ColumnTypeConstants', () => {
        var cst = {
    		types : [
                     {name:'tinyint',label:'tinyint (8 bit)'},
                     {name:'smallint',label:'smallint (16 bit)'},
                     {name:'int',label:'int'},
                     {name:'bigint',label:'bigint (64 bit)'},
                     {name:'float',label:'float'},
                     {name:'double',label:'double'},
                     {name:'boolean',label:'boolean'},
                     {name:'string',label:'string'},
                     {name:'date',label:'datetime with tz'},
                     {name:'dateonly',label:'date only'},
                     {name:'datetimenotz',label:'datetime no tz'},
                     {name:'geopoint',label:'geo point'},
                     {name:'geometry',label:'geometry'},
                     {name:'array',label:'array<...>'},
                     {name:'map',label:'map<...>'},
                     {name:'object',label:'object<...>'}
                 ],
             COMPLEX_TYPES : {
                     array:    {icon: 'icon-list'},
                     map:      {icon: 'icon-th'},
                     object:   {icon: 'icon-list-alt'},
                     geopoint: {icon: 'icon-map-marker', primitive: true}
    		}
        };
        return cst;
    });


    app.directive('complexTypeSelector', (ColumnTypeConstants) => {
        return {
            restrict:'E',
            scope: {
                'model':'=',
                'disabled': '<'
            },
            templateUrl : '/templates/datasets/type-editor/widget.html',
            link : (scope) => {
                scope.types = ColumnTypeConstants.types;
            }
        };
    });


    app.directive('complexTypeEditor', ($rootScope, ColumnTypeConstants) => {
        var COMPLEX_TYPES = ColumnTypeConstants.COMPLEX_TYPES, DEFAULT_ICON = 'icon-book';
        return {
            restrict:'E',
            scope: {
                'model':'=',
                'showCommentTab': '=',
                'hideCustomFields': '=',
                'disabled': '<'
            },
            templateUrl : '/templates/datasets/type-editor/editor.html',
            link : (scope) => {
                scope.appConfig = $rootScope.appConfig;

                // Foreign change
                scope.$watch('model',function(nv) {
                    if(!nv) return;
                    var currentCopy = angular.copy(scope.current);
                    scope.current = deepInplaceCopy(scope.model, scope.current);
                    fixup(scope.current, currentCopy);
                },true);

                scope.editMode = scope.showCommentTab ? 'comment' : 'view';

                function fixup(col,applyShow) {
                    if(!col) return;
                    delete col.$$hashKey;
                    if(applyShow) {
                        col.show = applyShow.show;
                        col.editing = applyShow.editing;
                        col.renaming = applyShow.renaming;
                        col.configuring = applyShow.configuring;
                        if(!col.show) {
                            delete col.show;
                        }
                        if(!col.editing) {
                            delete col.editing;
                        }
                        if(!col.renaming) {
                            delete col.renaming;
                        }
                        if(!col.configuring) {
                            delete col.configuring;
                        }
                    } else {
                        delete col.show;
                        delete col.editing;
                        delete col.renaming;
                        delete col.configuring;
                        applyShow = {};
                    }
                    if(col.type=='map') {
                        if(!(col.mapValues instanceof Object) || col.mapValues instanceof Array) {
                            // this automatic "fill" is used in the logic of AbstractDatasetSchemaRule.java, be careful in case of change
                            col.mapValues = {"name":"","type":"string"};
                        }
                        fixup(col.mapValues,applyShow.mapValues);
                        if(!(col.mapKeys instanceof Object) || col.mapKeys instanceof Array) {
                            col.mapKeys = {"name":"","type":"string"};
                        }
                        fixup(col.mapKeys,applyShow.mapKeys);
                        delete col.mapKeys.name;
                        delete col.mapValues.name;
                        delete col.arrayContent;
                        delete col.objectFields;
                        delete col.maxLength;
                    } else if(col.type=='array' || col.mapValues instanceof Array){
                        if(!(col.arrayContent instanceof Object) || col.arrayContent instanceof Array) {
                            // this automatic "fill" is used in the logic of AbstractDatasetSchemaRule.java, be careful in case of change
                            col.arrayContent = {"name":"","type":"string"};
                        }
                        fixup(col.arrayContent,applyShow.arrayContent);
                        delete col.arrayContent.name;
                        delete col.mapKeys;
                        delete col.mapValues ;
                        delete col.objectFields;
                        delete col.maxLength;
                    } else if(col.type == 'object') {
                        if(!(col.objectFields instanceof Array)) {
                            col.objectFields = [];
                        }
                        var flds = (applyShow.objectFields instanceof Array)?applyShow.objectFields:[];
                        var existingNames = {};
                        for(var i = 0 ; i < col.objectFields.length ; i++) {
                            var current = col.objectFields[i];
                            var currentName = current?current.name:undefined;
                            if(currentName) {
                                existingNames[currentName] = true;
                            }
                        }
                        for(let i = 0 ; i < col.objectFields.length ; i++) {
                            fixup(col.objectFields[i],flds[i]);
                            if(!col.objectFields[i].name) {
                                var newName = 'field_'+(i+1);
                                var cnt = 1;
                                while(existingNames[newName]) {
                                    newName ='field_'+(i+1)+'_'+cnt;
                                    cnt++;
                                }
                                col.objectFields[i].name= newName;
                            }
                        }
                        delete col.mapKeys;
                        delete col.mapValues;
                        delete col.arrayContent;
                        delete col.maxLength;
                    } else {
                        if(!col.type) {
                            // this automatic "fill" is used in the logic of AbstractDatasetSchemaRule.java, be careful in case of change
                            col.type = 'string';
                        }
                        if(col.type!='string') {
                            delete col.maxLength;
                        } else {
                            if(col.maxLength==undefined) {
                                col.maxLength = -1;
                            }
                        }
                        delete col.mapKeys;
                        delete col.mapValues;
                        delete col.arrayContent;
                        delete col.objectFields;
                        delete col.show;
                    }
                }

                // This code is NOT generic... and miss a lot of corner cases
                // It's okay for this usage only
                function deepInplaceCopy(from,to) {
                    if(!to || !(to instanceof Object)) {
                        to = {};
                    }
                    for(let k in to) {
                        if(!from[k]) {
                            delete to[k];
                        }
                    }
                    for(let k in from) {

                        var objFrom = from[k];
                        var objTo = to[k];
                        if(objFrom instanceof Array && objTo instanceof Array) {

                            for(var i = 0 ; i < objFrom.length ; i++) {
                                var itemFrom = objFrom[i];
                                var itemTo = objTo[i];
                                if(typeof itemFrom == 'string' || typeof itemFrom == 'number' || typeof itemFrom == 'boolean') {
                                    objTo[i] = itemFrom;
                                } else if(!(itemFrom instanceof Array) && itemFrom instanceof Object && itemFrom
                                    && !(itemTo instanceof Array) && itemTo instanceof Object && itemTo) {
                                    objTo[i] = deepInplaceCopy(itemFrom,itemTo);
                                } else {
                                    objTo[i] = angular.copy(itemFrom);
                                }
                            }
                            objTo.length = objFrom.length;

                        } else if(!(objFrom instanceof Array) && !(objTo instanceof Array) &&
                                objFrom instanceof Object && objTo instanceof Object && objTo!=null && objFrom!=null) {
                            to[k] = deepInplaceCopy(objFrom,objTo);
                        } else {
                            to[k] = angular.copy(objFrom);
                        }
                    }
                    return to;
                }


                function isDifferent(a,b,isRoot) {
                    if((!a && b) || (a &&!b)) return true;
                    if(a.name!=b.name) return true;
                    if(a.type!=b.type) return true;
                    if(a.timestampNoTzAsDate!=b.timestampNoTzAsDate) return true;
                    if (a.meaning != b.meaning) return true;
                    if(isRoot && a.comment && b.comment && b.comment!=a.comment) return true;
                    if (isRoot && a.isColumnEdited !== b.isColumnEdited) return true;
                    if(isRoot) {
                        if (!a.customFields && b.customFields) return true;
                        if (a.customFields && !b.customFields) return true;
                        if (a.customFields && b.customFields) {
                            const aKeys = Object.keys(a.customFields);
                            const bKeys = Object.keys(b.customFields);
                            if (aKeys.length!=bKeys.length) return true;
                            for (let key in a.customFields) {
                                if (!b.customFields.hasOwnProperty(key)) return true;
                                if (a.customFields[key]!=b.customFields[key]) return true;
                            }
                        }
                    }
                    if(a.type == 'string') {
                        if(a.maxLength != b.maxLength) {
                            return true;
                        }
                    }
                    if(a.type=='map') {
                        if(isDifferent(a.mapKeys,b.mapKeys)) {
                            return true;
                        }
                        if(isDifferent(a.mapValues,b.mapValues)) {
                            return true;
                        }
                    }
                    if(a.type=='array') {
                        if(isDifferent(a.arrayContent,b.arrayContent)) {
                            return true;
                        }
                    }
                    if(a.type=='object') {
                        if(a.objectFields && !b.objectFields) return true;
                        if(!a.objectFields && b.objectFields) return true;
                        if(!a.objectFields && !b.objectFields) return false;
                        if(a.objectFields.length != b.objectFields.length) {
                            return true;
                        }
                        for(var i = 0 ; i < a.objectFields.length; i++) {
                             if(isDifferent(a.objectFields[i],b.objectFields[i])) {
                                return true;
                            }
                        }
                    }
                    return false;
                }


                scope.toggleEdit = function(elm,event) {
                   if(event) {
                       event.stopPropagation();
                   }
                   scope.disableEditing();
                   elm.editing = !scope.disabled;
                };

                scope.toggleRenaming = function(elm,event) {
                   if(event) {
                       event.stopPropagation();
                   }
                   scope.disableEditing();
                   elm.renaming = !scope.disabled;
                };

                scope.toggleConfiguring = function(elm,event) {
                    if(event) {
                       event.stopPropagation();
                    }
                    scope.disableEditing();
                    elm.configuring = !scope.disabled;
                };

                scope.disableEditing = function() {
                    function recurse(sc) {
                        if(!sc) return;
                        delete sc.editing;
                        delete sc.configuring;
                        delete sc.renaming;
                        if(sc.mapKeys) {
                            recurse(sc.mapKeys);
                        }
                        if(sc.mapValues) {
                            recurse(sc.mapValues);
                        }
                        if(sc.objectFields) {
                            for(var k in sc.objectFields) {
                                recurse(sc.objectFields[k]);
                            }
                        }
                        if(sc.arrayContent) {
                            recurse(sc.arrayContent);
                        }
                    };
                    recurse(scope.current);
                };

                scope.ui = {};

                scope.customFieldsMap = $rootScope.appConfig.customFieldsMap['COLUMN'];

                scope.types = ColumnTypeConstants.types;

                scope.getTypeLabel = function(type) {
                    for(var k in scope.types) {
                        if(scope.types[k].name==type) {
                            return scope.types[k].label;
                        }
                    }
                    return 'Unknown type';
                };

                // UI current is the textarea (and yes, we should change that incredibly confusing naming)
                scope.$watch('ui.current',function() {
                    if(!scope.ui.current) return;
                    fixup(scope.ui.current,undefined);
                    var currentCopy = angular.copy(scope.current);
                    scope.current = deepInplaceCopy(scope.ui.current,scope.current)
                    fixup(scope.current,currentCopy);
                },true);

                // Current is the tree-view (and yes, we should change that incredibly confusing naming)
                scope.$watch('current',function(current) {
                    if(!current) return;

                    // Fix errors, keep states
                    fixup(current,current);

                    // Cleanup "show" status in textarea
                    scope.ui.current = angular.copy(current);
                    fixup(scope.ui.current,undefined);

                    // And in the model
                    var fixedModel = angular.copy(scope.model);
                    fixup(fixedModel,undefined);
                    if(isDifferent(current,fixedModel,true)) {
                        var noShowCopy = angular.copy(current);
                        fixup(noShowCopy,undefined);
                        var oldHash = scope.model.$$hashKey; // keep the hash (for angular ngRepeat tracking? otherwise https://github.com/dataiku/dip/issues/4206 )
                        scope.model = deepInplaceCopy(noShowCopy,scope.model);
                        scope.model.$$hashKey = oldHash;
                    }

                },true);

                scope.isEditable = function(elm) {
                    return elm && (elm.type == 'string' || !scope.isPrimitiveType(elm.type));
                };

                scope.getIconForType = function(t) {
                    return t in COMPLEX_TYPES ? COMPLEX_TYPES[t].icon : DEFAULT_ICON;
                };

                scope.setEditMode = function(m) {
                    scope.editMode = m;
                };

                scope.isPrimitiveType = function(t) {
                    return !(t in COMPLEX_TYPES) || COMPLEX_TYPES[t].primitive;
                }
            }
        };
    });


app.service('DatasetRenameService', function($stateParams, $state, $q, WT1, CreateModalFromTemplate, DataikuAPI, $rootScope, HistoryService, InfoMessagesModal) {
    // the $scope parameter is the current scope from where the rename is started.
    // It's used to find out if there is something to save before rename (if datasetHooks.askForSaveBeforeRenaming exists) & to display errors.
    // There are no specific prerequisites about stuff contained in that scope
    function startRenaming($scope, projectKey, datasetName, type="?") {
        // the hook is only defined if we are in the dataset settings. Otherwise, there is nothing to save.
        const saveBeforePromise = $scope.datasetHooks && $scope.datasetHooks.askForSaveBeforeRenaming
            ? $scope.datasetHooks.askForSaveBeforeRenaming()
            : $q.resolve();

        return saveBeforePromise
            .then(() => showRenameModal(projectKey, datasetName, type))
            .then((newName) => redirectAfterRename(datasetName, newName))
            // rejection here means the user canceled (either on save or on rename) or there was an error during saving / checking conflicts
            // but all errors have already been setErrorInScope-ed
            .catch(() => {})
    }

    function showRenameModal(projectKey, datasetName, type){
        WT1.event("dataset-rename-open-modal");
        // we use $rootScope as parent scope because the save conflict resolution may reload the state, killing the current scope.
        return CreateModalFromTemplate("/templates/datasets/rename-dataset-box.html", $rootScope, null, function(modalScope){
            modalScope.datasetName = datasetName;
            modalScope.currentProjectKey = $stateParams.projectKey;
            modalScope.uiState = {
                newName: datasetName,
                showImpacts: false,
                renameStarted: false,
            }
            
            DataikuAPI.datasets.computeRenamingImpact(projectKey, datasetName, true).success(function(data) {
                const size = (tab) => tab ? tab.length : 0;
                modalScope.computedImpact = data;
                modalScope.computedImpact.hasAnyBestEffortImpact = 0 < size(data.bestEffortAutomaticallyChangedScenarios)
                                                                     + size(data.bestEffortAutomaticallyChangedRecipes)
                                                                     + size(data.modifiedWebApps)
                                                                     + size(data.modifiedNotebooks)
                                                                     + size(data.modifiedReports);
                modalScope.computedImpact.hasManualChangeImpact = 0 < size(data.manuallyChangedRecipes)
                                                                    + size(data.manuallyChangedNotebooks);
            }).error(setErrorInScope.bind(modalScope));

            modalScope.doRename = () => {
                doRename(projectKey, datasetName, type, modalScope.uiState.newName, modalScope)
                .error((data, status, headers) => {
                        setErrorInScope.bind(modalScope)(data, status, headers);
                        modalScope.renameStarted = false;
                    });
                modalScope.renameStarted = true;
            }
        });
    }

    function doRename(projectKey, datasetName, type, newName, modalScope) {
        WT1.event("dataset-rename", {"datasetType": type});
        return DataikuAPI.datasets.rename(projectKey, datasetName, newName).success(function(renameResult) {
            HistoryService.notifyRenamed({
                type: "DATASET",
                id: datasetName,
                projectKey: projectKey
            }, newName);

            modalScope.resolveModal(newName);

            return InfoMessagesModal.showIfNeeded($rootScope, renameResult, "Dataset was renamed with warnings")
                .catch(() => {}) // we don't care how the modal was closed
        });
    }

    function redirectAfterRename(oldName, newName) {
        if($stateParams.datasetName === oldName) {
            // reload the page with the new value if we are in a route with datasetName param
            $state.go($state.current, {
                ...$stateParams,
                datasetName : newName
            }, {
                location: 'replace'
            });
        } else {
            // else still reload to refresh content that may reference the old name
            $state.reload();
        }
    }

    return {
        startRenaming
    };
})


app.controller("DatasetPageRightColumnActions", function($controller, $scope, $rootScope, $stateParams, ActiveProjectKey) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {
        selectedObject : {
            projectKey : ActiveProjectKey.get(),
            name : $stateParams.datasetName,
            id: $stateParams.datasetName,
            nodeType : 'LOCAL_DATASET',
            interest : {}, 
        },
        confirmedItem : {
            projectKey : ActiveProjectKey.get(),
            name : $stateParams.datasetName,
            nodeType : 'LOCAL_DATASET',
        }
    };
});


app.controller("ForeignDatasetPageRightColumnActions", ($controller, $scope, $stateParams, DatasetUtils) => {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    const loc = DatasetUtils.getLocFromFull($stateParams.datasetFullName);
    $scope.selection = {
        selectedObject : {
            projectKey : loc.projectKey,
            name : loc.name,
            id: loc.name,
            nodeType : 'FOREIGN_DATASET',
            interest : {}, 
        },
        confirmedItem : {
            projectKey : loc.projectKey,
            name : loc.name,
            nodeType : 'FOREIGN_DATASET',
        }
    }
});


app.controller("DatasetDetailsController", function($scope, DatasetUtils, StateUtils, DatasetDetailsUtils, translate) {
    $scope.StateUtils = StateUtils;
    $scope.translate = translate;
    $scope.DatasetUtils = DatasetUtils;

    $scope.isLocalDataset = function() {
        return DatasetDetailsUtils.isLocalDataset($scope.data);
    };

    $scope.isPartitioned = function() {
        return DatasetDetailsUtils.isPartitioned($scope.data);
    };

    $scope.refreshAndGetStatus = function(datasetData, computeRecords, forceRecompute) {
        DatasetDetailsUtils.refreshAndGetStatus($scope, datasetData, computeRecords, forceRecompute);
    };
});


app.component('inputOutputRecipes', {
    bindings: {
        data: '<',
    },
    controller: function ctrlComputableParentLink(StateUtils) {
        this.StateUtils = StateUtils;
    },
    templateUrl: 'templates/object-details/input-output-recipes.html'
});

app.directive('ellipsedList', [ '$window', '$timeout', function($window, $timeout){
    return {
        link: function($scope, element) {
            $scope.menuState = {
                useFullMoreActions: true,
                hideMoreActions: false,
            };
            $scope.hideMoreActions = function() {
                $scope.menuState.hideMoreActions = true;
                element.children('[label]').show();
            };
            var resizeMoreActions = function() {
                var elements = element.children('[label]');
                if (elements.length === 0 || elements.first().width() === 0) return;

                var MIN_VISIBLE_LINES = 2; // we want to always display at least two full lines of items before collapsing
                
                $scope.element = element;

                // calculate how many elements fit on a single line
                var lineElCount = Math.floor(element.width() / elements.width());
                $scope.menuState.useFullMoreActions = (MIN_VISIBLE_LINES * lineElCount < elements.length);
                var splitPosition = $scope.menuState.useFullMoreActions ? (MIN_VISIBLE_LINES * lineElCount - 1) : elements.length;
                elements.slice(splitPosition, elements.length).hide();
                elements.slice(0, splitPosition).show();
            };
            

            var handleElementResize = function() {
                const nv = element.width();
                if (ov != nv && !$scope.menuState.hideMoreActions && nv>0) {
                    ov = nv;
                    $timeout(resizeMoreActions);
                }
            }

            var ov = -1;

            angular.element($window).bind('resize', handleElementResize);

            $scope.$on("$destroy", function () {
                angular.element($window).unbind('resize', handleElementResize);
            });
    
    
            $scope.$watch('uiState.displayMoreActions', function(nv, ov){
                if($scope.menuState.hideMoreActions){
                    return;
                }
                if (!nv || (nv && ov)) {
                    element.addClass('ellipsed-list-loading');
                }
                if (nv) {
                    showMoreActions();
                }
            });

            var showMoreActions = function(){
                $scope.menuState.hideMoreActions = false;
                $timeout(function(){
                    resizeMoreActions();
                    element.removeClass('ellipsed-list-loading');
                },50);
            };
        }
    }
}]);


app.directive('datasetRightColumnSummary', function($controller, $stateParams, $state,
        DataikuAPI, CreateModalFromTemplate, GlobalProjectActions, QuickView, FlowGraphSelection, FlowBuildService,
        ActiveProjectKey, AnyLoc, ActivityIndicator, DatasetCustomFieldsService, $rootScope, SelectablePluginsService,
        DatasetRenameService, WT1, FlowGraph, translate, DataQualityComputationTrackingService, PluginCategoryService){
    return {
        templateUrl: '/templates/datasets/right-column-summary.html',
        link: function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;

            $scope.generateDatasetDescription = function() {
                DataikuAPI.datasets.get($scope.dataset.projectKey, $scope.dataset.name, ActiveProjectKey.get()).noSpinner()
                    .success(function(data) {
                        angular.extend($scope.dataset, data);
                        CreateModalFromTemplate(
                            "/static/dataiku/ai-dataset-descriptions/generate-documentation-modal/generate-documentation-modal.html",
                            $scope,
                            "AIDatasetDescriptionsModalController",
                            function(scope) {
                                scope.init($scope.dataset, $scope.canWriteProject)
                            }
                        )
                    })
                    .error(setErrorInScope.bind($scope));
            };

            $scope.$on('objectMetaDataChanged', (event, data) => {
                $scope.dataset.shortDesc = data.shortDesc;
                $scope.dataset.description = data.description;
            });

            $scope.$on('datasetSchemaChanged', (event, data) => {
                $scope.dataset.schema = data.schema;
            });

            $scope.$on('columnChanged', (event, data) => {
                const newSchema = angular.copy($scope.dataset.schema);
                const targetCol = newSchema.columns.find(col => col.name === data.column.name);

                if (targetCol) {
                    targetCol.comment = data.column.comment;
                    targetCol.type = data.column.type;
                }

                $scope.dataset.schema = newSchema;
            });

            /* Auto save when summary is modified */
            $scope.$on("objectSummaryEdited", function() {
                const editedDataset = $scope.dataset !== null ? $scope.dataset : $scope.datasetBeingDeselected;
                if (editedDataset) {
                    DataikuAPI.datasets.save(editedDataset.projectKey, editedDataset, { summaryOnly: true })
                        .then(() => ActivityIndicator.success("Saved"))
                        .catch(setErrorInScope.bind($scope));
                }
            });

            /* Save custom fields */
            $scope.$on('customFieldsSummaryEdited', function(event, customFields) {
                DatasetCustomFieldsService.saveCustomFields($scope.dataset, customFields);
            });


            $controller('_TaggableObjectsMassActions', { $scope });
            $controller('_TaggableObjectsCapabilities', { $scope });
            $controller('_FeatureGroupRightColumnActions', { $scope });

            $scope.QuickView = QuickView;

            $scope.$stateParams = $stateParams;

            var enrichSelectedObject = function (selObj, dataset) {
                selObj.tags = dataset.tags; // for apply-tagging modal
            }

            $scope.$on('taggableObjectTagsChanged', () => $scope.refreshData()); // update right panel when tags are changed (may have change this item too) - do not use this event just to reload right panel (it's far more costly than just a rightPanelSummary.triggerFullInfoUpdate)
            $scope.$on('rightPanelSummary.triggerFullInfoUpdate', () => $scope.refreshData()); // refresh on actions that are known to specifically require a right panel info update

            $scope.refreshData = function() {
                $scope.uiState.displayMoreActions = false;
                // From personal home page we want details of items in projects without having opened then

                const loc = AnyLoc.makeLoc($scope.selection.selectedObject.projectKey, $scope.selection.selectedObject.name);
                $scope.canAccessObject = false;
                DataikuAPI.datasets.getFullInfo(ActiveProjectKey.get(), loc.projectKey, loc.localId).then(function({data}) {
                    if (!$scope.selection.selectedObject
                        || loc.localId != data.dataset.name
                        || loc.projectKey != data.dataset.projectKey) {
                        return; //too late!
                    }

                    $scope.datasetFullInfo = data;
                    $scope.canInsertRecipeAfter = $scope.selection.selectedObject.successors && $scope.selection.selectedObject.successors.length && $scope.selection.selectedObject.successors.every(successor => FlowGraph.node(successor).nodeType === 'RECIPE');
                    $scope.dataset = data.dataset;
                    $scope.datasetBeingDeselected = null;
                    $scope.selection.selectedObject.interest = data.interest;
                    $scope.usability = GlobalProjectActions.getAllStatusForDataset(data.dataset);
                    $scope.selectablePlugins = SelectablePluginsService.listSelectablePlugins({'DATASET' : 1});
                    $scope.noRecipesCategoryPlugins = PluginCategoryService.standardCategoryPlugins($scope.selectablePlugins, ['visual', 'genai', 'other', 'code'])
                    $scope.canAccessObject = true;

                    enrichSelectedObject($scope.selection.selectedObject, $scope.dataset);

                    if (data.dataset.projectKey == ActiveProjectKey.get()) {
                        $scope.isLocalDataset = true;
                        $scope.dataset.smartName = data.dataset.name; // TODO: could be done in backend
                    } else {
                        $scope.isLocalDataset = false;
                        $scope.dataset.smartName = data.dataset.projectKey + "." + data.dataset.name;
                    }
                    $scope.objectAuthorizations = data.objectAuthorizations;
                    $scope.uiState.displayMoreActions = true;
                    $scope.dataset.zone = ($scope.selection.selectedObject.usedByZones || [])[0] || $scope.selection.selectedObject.ownerZone;
                }).catch(setErrorInScope.bind($scope))
                .finally(() => $scope.uiState.displayMoreActions = true);
            };

            $scope.getCommonZone = function () {
                return $scope.dataset.zone;
            };

            $scope.getSmartNames = function() {
                return [$scope.dataset.smartName];
            };

            $scope.isOnEditableTab = function() {
                return $state.current?.name == "projects.project.datasets.dataset.edit"
            }

            $scope.isOnSchemaTab = function() {
                return $state.current?.name === "projects.project.datasets.dataset.settings"
            }

            $scope.$watch("selection.selectedObject",function(nv) {
                $scope.datasetFullInfo = {dataset: $scope.selection.selectedObject, timeline: {}, interest: {}}; // display temporary (incomplete) data
                if($scope.selection.selectedObject != $scope.selection.confirmedItem) {
                    // We need to keep a reference on the dataset that was selected in case an event "objectSummaryEdited" arrives just after this event
                    // See https://app.shortcut.com/dataiku/story/152840 for more details.
                    $scope.datasetBeingDeselected = $scope.dataset;
                    $scope.dataset = null;
                }
                if (!nv) return;
                $scope.datasetType = nv.datasetType || nv.type;
                if (nv.nodeType === 'FOREIGN_DATASET') {
                    $scope.datasetSmartName = nv.projectKey + '.' + nv.name;
                    $scope.datasetHref = $state.href('projects.project.foreigndatasets.dataset.explore',
                        {datasetFullName: $scope.datasetSmartName, projectKey: nv.projectKey});
                } else {
                    $scope.datasetSmartName = nv.name;
                    $scope.datasetHref = $state.href('projects.project.datasets.dataset.explore',
                        {datasetName: nv.name, projectKey: nv.projectKey});
                }
            });

            $scope.$watch("selection.confirmedItem", (nv) => {
                if (!nv) return;
                if (!nv.projectKey) nv.projectKey = ActiveProjectKey.get();
                $scope.refreshData();
            });

            $scope.isAllDatasets = ( () => true ); // To reuse templates with multi-object right column

            $scope.buildDataset = function () {
                const loc = AnyLoc.makeLoc($scope.selection.selectedObject.projectKey, $scope.selection.selectedObject.name);
                const modalOptions = {
                    upstreamBuildable: $scope.datasetFullInfo.upstreamBuildable,
                    downstreamBuildable: $scope.datasetFullInfo.downstreamBuildable,
                    redirectToJobPage: attrs.jobStartRedirects !== undefined
                }
                FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "DATASET", loc, modalOptions);
            }

            // Export options to retrieve currently configured exploration filters & search query
            $scope.exportDatasetOptions = function() {
                if ($state.current.name.includes('explore')) {
                    return {
                        allowFiltering: true,
                        explorationFiltersAndSearchQuery: $scope.explorationFiltersAndSearchQuery
                    };
                } else {
                    return {};
                }
            }
            $scope.explorationFiltersAndSearchQuery = null; // Null means retrieve the saved ones in the backend
            $rootScope.$on("$destroy", $rootScope.$on("shakerTableChangedGlobal", (evt, shaker) => {
                $scope.explorationFiltersAndSearchQuery = {
                    explorationFilters: shaker.explorationFilters,
                    globalSearchQuery: shaker.globalSearchQuery
                }
            }));

            $scope.createWebAppForDataset = function(loadedWebapp, roleTarget, roleValue) {
                let defaultWebappName = loadedWebapp.desc.meta.label + ' on ' + $scope.datasetSmartName;
                $scope.showCreateVisualWebAppModal(loadedWebapp, roleTarget, roleValue, defaultWebappName);
            };

            $scope.editCustomFields = function(editingTabIndex = 0) {
                if (!$scope.selection.selectedObject) {
                    return;
                }
                DataikuAPI.datasets.getSummary($scope.selection.selectedObject.projectKey, $scope.selection.selectedObject.name).success(function(data) {
                    const dataset = data.object;
                    const modalScope = angular.extend($scope, {objectType: 'DATASET', objectName: dataset.name, objectCustomFields: dataset.customFields, editingTabIndex});
                    CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
                        DatasetCustomFieldsService.saveCustomFields(dataset, customFields);
                    });
                }).error(setErrorInScope.bind($scope));
            };

            $scope.anyPipelineTypeEnabled = function() {
                return $rootScope.projectSummary && ($rootScope.projectSummary.sparkPipelinesEnabled || $rootScope.projectSummary.sqlPipelinesEnabled);
            };

            function showVirtualizationAction(showDeactivate) {
                return function() {
                    const virtualized = !!$scope.selection.selectedObject.virtualizable;
                    return !$state.current.name.includes('explore')
                        && $scope.isProjectAnalystRW()
                        && $scope.isLocalDataset
                        && showDeactivate === virtualized;
                }
            }
            $scope.showAllowVirtualizationAction = showVirtualizationAction(false);
            $scope.showStopVirtualizationAction = showVirtualizationAction(true);

            $scope.zoomToOtherZoneNode = function(zoneId) {
                const otherNodeId = $scope.selection.selectedObject.id.replace(/zone__.+?__/, "zone__" + zoneId + "__");
                if ($stateParams.zoneId) {
                    $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId: zoneId, id: graphVizUnescape(otherNodeId) }));
                    return;
                }
                else {
                    $scope.zoomGraph(otherNodeId);
                    FlowGraphSelection.clearSelection();
                    FlowGraphSelection.onItemClick($scope.nodesGraph.nodes[otherNodeId]);
                }
            }

            function updateUserInterests() {
                DataikuAPI.interests.getForObject($rootScope.appConfig.login, "DATASET", ActiveProjectKey.get(), $scope.selection.selectedObject.name).success(function(data) {
                    $scope.selection.selectedObject.interest = data;
                    $scope.datasetFullInfo.interest = data;
                }).error(setErrorInScope.bind($scope));
            }

            const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
            $scope.$on("$destroy", interestsListener);

            $scope.renameDataset = (projectKey, datasetName, datasetType) => DatasetRenameService.startRenaming($scope, projectKey, datasetName, datasetType);

            $scope.insertRecipeAfter = function() {
                $scope.showInsertRecipeModal($scope.selection.selectedObject)
                    .then(selectedRecipe => WT1.event('rightpanelrecipe_actions_insertrecipe', {position: 'after', recipe: selectedRecipe && selectedRecipe.type || 'Unknown'}));
            }

            $scope.getInsertRecipeAfterTooltip = function() {
                if (!$scope.canWriteProject()) return translate('PROJECT.PERMISSIONS.WRITE_ERROR', 'You don\'t have the permission to write to this project');
                if (!$scope.canInsertRecipeAfter) return translate('PROJECT.PERMISSIONS.INSERT_RECIPE_ERROR', 'Select a dataset in the flow, or select both the dataset and its downstream recipes, to insert a recipe between them');
                return translate('PROJECT.DATASET.RIGHT_PANEL.ACTIONS.INSERT_RECIPE.TOOLTIP', 'Insert recipe (and corresponding output dataset) after selected dataset');
            }

            // watch the DQ computations to update the current status icon near the dataset name
            let dqComputationSubscription = null;
            $scope.$watch('selection.selectedObject', (nv) => {
                if(dqComputationSubscription) dqComputationSubscription.unsubscribe();
                if(!nv) return;

                const loc = AnyLoc.makeLoc(nv.projectKey, nv.name);
                dqComputationSubscription = DataQualityComputationTrackingService.observeObjectChanges(loc.projectKey, loc.localId, null).subscribe(() => {
                    DataikuAPI.dataQuality.getDatasetCurrentDailyStatus(
                        ActiveProjectKey.get(), loc.projectKey, loc.localId
                    ).success(
                        (res) => $scope.datasetFullInfo.dataQualityStatus = res
                    ).error(() => {}); // this is not directly related to a user action, there is no point showing them an error
                });
            })
            
            $scope.$on("$destroy", () => dqComputationSubscription && dqComputationSubscription.unsubscribe());

        }
    }
});

app.controller("ConnectionDetailsController", function ($scope, $stateParams, DataikuAPI, FutureProgressModal, ConnectionUtils, StateUtils,InfoMessagesModal) {
    $scope.StateUtils = StateUtils;
    $scope.noDescription = true;

    $scope.indexConnection = function () {
        $scope.$emit('indexConnectionEvent', $scope.data.name);
    };
    $scope.showMessages = function (messages) {
        InfoMessagesModal.showIfNeeded($scope, messages, "Indexing report");
    };
    $scope.isIndexable = function (connection) {
        return ConnectionUtils.isIndexable(connection);
    };
});

app.directive('connectionRightColumnSummary', ($state) => {
    return {
        templateUrl: '/templates/admin/connections-right-column-summary.html',
        link: (scope) => {
            scope.$watch("selection.selectedObject", function (nv) {
                if (!nv) return;
                scope.connectionHref = $state.href('admin.connections.edit', {connectionName: nv.name});
            });

        }
    };
});

// Cta stands for Call To Action
// Service to return a function that will decide whether we should display the error or a nicer CTA to the user when the dataset can't load itelf (used in explore and charts)
app.service('DatasetErrorCta', function() {

    function requiresDatasetBuild(error) {
        return ['USER_CONFIG_DATASET', 'USER_CONFIG_OR_BUILD'].includes(error.fixability);
    }

    return {
        getupdateUiStateFunc: function (scope) {
            return function(error) {
                scope.uiDisplayState = {
                    showError: false,
                    showBeingBuilt: false,
                    showAboutToBeBuilt: false,
                    showBuildEmptyCTA: false,
                    showBuildFailCTA: false,
                    showUI: false
                }

                const dfi = scope.datasetFullInfo;
                const managed = (dfi && dfi.dataset && dfi.dataset.managed);
                const beingBuilt = (dfi && dfi.currentBuildState && dfi.currentBuildState.beingBuilt && dfi.currentBuildState.beingBuilt.length);
                const aboutToBeBuilt = (dfi && dfi.currentBuildState && dfi.currentBuildState.aboutToBeBuilt && dfi.currentBuildState.aboutToBeBuilt.length);
                const neverBuiltBuildable = (dfi && dfi.buildable && dfi.lastBuild === undefined)
                const buildable = (dfi && dfi.buildable);

                if (beingBuilt) {
                    scope.uiDisplayState.showBeingBuilt = !error || error.errorType != 'com.dataiku.dip.exceptions.UnauthorizedException';
                }
                if (!beingBuilt && aboutToBeBuilt) {
                    scope.uiDisplayState.showAboutToBeBuilt = !error || error.errorType != 'com.dataiku.dip.exceptions.UnauthorizedException';
                }

                if (error && managed && neverBuiltBuildable) {
                    /* If there is an error, but it is a managed-buildable-never-built-dataset, then don't display the error,
                     and display the CTA instead */
                    scope.uiDisplayState.showError = false;
                    scope.uiDisplayState.showBuildEmptyCTA = !beingBuilt && !aboutToBeBuilt;
                } else if (error && managed && buildable) {
                    /* If there is an error, and it is managed-buildable, but has ever been built, then display error
                     and CTA if error requires to build dataset */
                    scope.uiDisplayState.showError = true;
                    scope.uiDisplayState.showBuildFailCTA = !beingBuilt && !aboutToBeBuilt && requiresDatasetBuild(error);
                } else {
                    /* Just an error on a non-managed-buildable dataset: just display it */
                    scope.uiDisplayState.showError = true;
                }

                /* Table is shown if: no error */
                if (!error) {
                    scope.uiDisplayState.showUI = true;
                    scope.error = null;
                } else {
                    scope.error = error;
                }
            }
        }
    };
});

})();


document.addEventListener('StartSimpleTableContentUpdate', function() {});
document.addEventListener('EndSimpleTableContentUpdate', function() {});

function dispatchCustomTimelineEvent(type, details) {
    var event = new CustomEvent(type, details);
    document.dispatchEvent(event);
}
