(function() {
'use strict';

const app = angular.module('dataiku.recipes', ['dataiku.common.lists', 'dataiku.aiSqlGeneration']);


app.directive('checkRecipeNameUnique', function(DataikuAPI, $stateParams) {
    return {
        require: 'ngModel',
        link: function(scope, elem, attrs, ngModel) {
            DataikuAPI.flow.recipes.list($stateParams.projectKey).success(function(data) {
                scope.unique_recipes_names = $.map(data, function(recipe) {
                    return recipe.name;
                });
                /* Re-apply validation as soon as we get the list */
                apply_validation(ngModel.$modelValue);
            });
            var initialValue = null, initialValueInitialized = false;
            function apply_validation(value) {
                // Implicitely trust the first value (== our own name)
                if (initialValueInitialized == false && value != undefined && value != null && value.length > 0) {
                    initialValue = value;
                    initialValueInitialized = true;
                }
                // It is fake, but other check will get it.
                if (value == null || value.length === 0) return true;
                // We are back to our name, accept.
                if (initialValueInitialized && value == initialValue) return value;
                var valid = scope.unique_recipes_names ? scope.unique_recipes_names.indexOf(value) === -1 : true;
                ngModel.$setValidity('recipeNameUnique', valid);
                return valid ? value : undefined;
            }
             //For DOM -> model validation
            ngModel.$parsers.unshift(apply_validation);

            //For model -> DOM validation
            ngModel.$formatters.unshift(function(value) {
                apply_validation(value);
                return value;
            });
        }
    };
});

app.filter("buildModeDescription", function(){
    var dict = {
        "NON_RECURSIVE_FORCED_BUILD": "Build only this dataset",
        "RECURSIVE_BUILD": "Build required datasets",
        "RECURSIVE_FORCED_BUILD": "Force-rebuild dataset and dependencies",
        "RECURSIVE_MISSING_ONLY_BUILD": "Build missing dependencies then this one"
    };
    return function(input) {
        return dict[input] || input;
    }
});

app.directive('recipePipelineConfig', function() {
    return {
        restrict: 'E',
        templateUrl: '/templates/recipes/fragments/recipe-pipeline-config.html',
        scope: {
          config: "=",
          anyPipelineTypeEnabled: "&"
        }
    };
});

app.directive('otherActionListItem', function() {
    return {
        restrict: 'E',
        templateUrl: '/templates/recipes/fragments/other-action-list-item.html',
        scope: {
            icon: "@",
            label: "@",
            onClick: '&',
            showCondition: "<?",
            enableCondition: "<?",
            disabledTooltip: "@?"
        },
        link: function(scope) {
            if (scope.showCondition === undefined) {
                scope.showCondition = true;
            }
            if (scope.enableCondition === undefined) {
                scope.enableCondition = true;
            }
        }
    };
});

app.directive("sparkDatasetsReadParamsBehavior", function(Assert, $stateParams, RecipesUtils, Logger, DatasetUtils) {
    return {
        scope: true,
        link: function($scope, element, attrs) {
            Logger.info("Loading spark behavior");
            Assert.inScope($scope, 'recipe');
            let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$scope.recipe.projectKey;

            $scope.readParams = $scope.$eval(attrs.readParams);
            Assert.inScope($scope, 'readParams');

            function autocomplete() {
                RecipesUtils.getFlatInputsList($scope.recipe).forEach(function(input) {
                    Assert.inScope($scope, 'computablesMap');
                    const computable = $scope.computablesMap[input.ref];
                    if (!computable) {
                        throw Error('dataset is not in computablesMap, try reloading the page');
                    }
                    const dataset = computable.dataset;
                    if (dataset && !$scope.readParams.map[input.ref]) {
                        $scope.readParams.map[input.ref] = {
                            repartition: ['HDFS', 'hiveserver2'].includes(dataset.type) ? 1 : 10,
                            cache: false
                        };
                    }
                });
                Logger.info("Updated map", $scope.readParams.map);
            }

            $scope.$watch("recipe.inputs", function(nv, ov) {
                if (nv && $scope.computablesMap) {
                    DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                        .then(_ => autocomplete());
                }
            }, true);
            $scope.$watch("computablesMap", function(nv, ov) {
                if (nv) {
                    DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                        .then(_ => autocomplete());
                }
            }, true);
        }
    }
});


app.directive("sparkDatasetsReadParams", function(Assert, RecipesUtils) {
    return {
        scope: true,
        templateUrl: "/templates/recipes/fragments/spark-datasets-read-params.html",
        link: function($scope, element, attrs) {
            Assert.inScope($scope, 'recipe');
            $scope.readParams = $scope.$eval(attrs.readParams);
            Assert.inScope($scope, 'readParams');
        }
    };
});


app.service('RecipesCapabilities', function(RecipeDescService, CodeEnvsService, AppConfig, $rootScope) {

    function getRecipeType(recipe) {
        if (recipe) {
            // A bit dirty, we don't know what recipe is (taggableObjectRef, graphNode, listItem...)
            if (recipe.recipeType) {
                return recipe.recipeType;
            } else if (recipe.subType) {
                return recipe.subType;
            } else if (recipe.type) {
                return recipe.type;
            }
        }
        return undefined;
    }

    this.isMultiEngine = function(recipe) {
        const desc = RecipeDescService.getDescriptor(getRecipeType(recipe));
        return !!desc && desc.isMultiEngine;
    };

    this.canEngine = function(recipe, engine) {
        if (!recipe) {
            return false;
        }
        const recipeType = getRecipeType(recipe);
        if (!recipeType) {
            return false;
        }
        if (recipeType.toLowerCase().includes(engine)) {
            return true;
        }
        const desc = RecipeDescService.getDescriptor(recipeType);
        // we can't be sure though...
        return !!(desc && desc.isMultiEngine);
    };

    this.isSparkEnabled = function() {
        return !AppConfig.get() || AppConfig.get().sparkEnabled;
    };

    this.canSpark = function(recipe) {
        return this.isSparkEnabled() && this.canEngine(recipe, 'spark');
    };

    this.canSparkPipeline = function (recipe) {
        return $rootScope.projectSummary.sparkPipelinesEnabled &&
            this.canSpark(recipe) &&
            !(['pyspark', 'sparkr'].includes(getRecipeType(recipe)));
    };

    this.canSqlPipeline = function(recipe) {
        const canEngine = this.canEngine(recipe, 'sql');
        const b = !(['spark_sql_query', 'sql_script'].includes(getRecipeType(recipe)));
        return $rootScope.projectSummary.sqlPipelinesEnabled &&
            canEngine &&
            b;
    };

    this.canChangeSparkPipelineability= function(recipe) {
        if (recipe) {
            // Prediction scoring is supported but there is a bug that prevent the backend to compute the pipelineabilty (Clubhouse #36393)
            if (getRecipeType(recipe) === 'prediction_scoring') {
                return false;
            }
            return this.canSpark(recipe);
        }
        return false;
    };

    this.canChangeSqlPipelineability= function(recipe) {
        const recipeType = getRecipeType(recipe);
        if (recipeType) {
            // The following recipes are the only ones that can run on SQL and be part of a SQL pipeline.
            if (['sync', 'shaker', 'sampling', 'grouping', 'upsert', 'distinct', 'window', 'join', 'split', 'topn', 'sort',
                    'pivot', 'vstack', 'sql_query', 'prediction_scoring'].includes(recipeType)) {
                return true;
            }
        }
        return false;
    };

    this.canImpala = function(recipe) {
        if (recipe) {
            const recipeType = getRecipeType(recipe);
            if (recipeType === 'impala') {
                return true;
            }
            if (AppConfig.get() && !AppConfig.get().sparkEnabled) {
                return false;
            }
            const desc = RecipeDescService.getDescriptor(recipeType);
            if (desc && desc.isMultiEngine) {
                return true; // we can't be sure...
            }
        }
        return false;
    };

    this.canHive = function(recipe) {
        if (recipe) {
            const recipeType = getRecipeType(recipe);
            if (recipeType === 'hive') {
                return true;
            }
            if (AppConfig.get() && !AppConfig.get().sparkEnabled) {
                return false;
            }
            const desc = RecipeDescService.getDescriptor(recipeType);
            if (desc && desc.isMultiEngine) {
                return true; // we can't be sure...
            }
        }
        return false;
    };

    this.canPythonCodeEnv = function(recipe) {
        return CodeEnvsService.canPythonCodeEnv(recipe);
    };

    this.canRCodeEnv = function(recipe) {
        return CodeEnvsService.canRCodeEnv(recipe);
    };
});

app.controller("RecipePageRightColumnActions", async function($controller, $scope, $stateParams, ActiveProjectKey, DataikuAPI) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});
    $controller('_RecipeWithEngineBehavior', {$scope: $scope});

    $scope.recipeData = (await DataikuAPI.flow.recipes.getFullInfo(ActiveProjectKey.get(), $stateParams.recipeName)).data;

    $scope.recipe = $scope.recipeData.recipe;
    $scope.recipe.recipeType = $scope.recipe.type;
    $scope.recipe.nodeType = 'RECIPE';
    $scope.recipe.id = $stateParams.recipeName;
    $scope.recipe.interest = $scope.recipeData.interest;

    $scope.selection = {
        selectedObject : $scope.recipe,
        confirmedItem : $scope.recipe
    };
});


app.directive('recipeRightColumnSummary', function($controller, $stateParams, $state, $rootScope, RecipeRenameService,
    DataikuAPI, TopNav, CreateModalFromTemplate, ActiveProjectKey, ActivityIndicator, WT1, FlowBuildService, LambdaServicesService,
    Logger, translate) {
    return {
        templateUrl: '/templates/recipes/right-column-summary.html',

        link: function(scope, element, attrs) {

            $controller('_TaggableObjectsMassActions', {$scope: scope});
            $controller('_TaggableObjectsCapabilities', {$scope: scope});

            var enrichSelectedObject = function (selObj, recipe) {
                selObj.tags = recipe.tags; // for apply-tagging modal
            }

            scope.isDeleteAndReconnectAllowed = false;

            scope.$watch("selection", () => {
                if (scope.selection && scope.selection.selectedObject) {
                    scope.canDeleteAndReconnectObject(scope.selection.selectedObject).then(function(result) {
                        scope.isDeleteAndReconnectAllowed = result;
                    }).catch(function(error) {
                        Logger.warn(`Problem determining whether to show delete & reconnect option for ${scope.selection.selectedObject.id}:`, error);
                    });
                }
            });

            scope.deleteAndReconnectRecipe = function(wt1_event_name) {
                scope.doDeleteAndReconnectRecipe(scope.selection.selectedObject, wt1_event_name);
            }

            scope.getDeleteAndReconnectTooltip = function() {
                if (!scope.canWriteProject()) return translate("FLOW.DELETE_AND_RECONNECT.NO_PERMISSIONS", "You don't have write permissions for this project");
                if (!scope.isDeleteAndReconnectAllowed) return translate("FLOW.DELETE_AND_RECONNECT.ONLY_AVAILABLE", "This feature is only available for single-input recipes before the flow's end where reconnection is possible");
                return translate("FLOW.DELETE_AND_RECONNECT.DELETE_THIS_RECIPE", "Delete this recipe and its output datasets, relinking the upstream dataset as an input to the downstream recipes");
            }

            scope.getRunTooltip = function() {
                if (scope.isOutputEmpty)
                    return scope.translate('PROJECT.RECIPE.RIGHT_PANEL.RUN.NO_OUTPUT_ERROR', 'Running a recipe that has no output is not possible');
                if (scope.payload?.backendType !== undefined && scope.payload.backendType === 'VERTICA')
                    return  scope.translate('PROJECT.PERMISSIONS.VERTICA_NOT_SUPPORTED', 'Vertica ML backend is no longer supported');
                if (!scope.canWriteProject())
                    return scope.translate('PROJECT.PERMISSIONS.WRITE_ERROR', 'You don\'t have write permissions for this project');
                return null;
            }

            scope.refreshData = function() {
                DataikuAPI.flow.recipes.getFullInfo(scope.selection.selectedObject.projectKey, scope.selection.selectedObject.name).success(function(data){
                    scope.recipeData = data;
                    scope.recipe = data.recipe;
                    scope.recipeBeingDeselected = null; // We no longer need to keep a reference to recipe that was selected for "event that will arrive late".
                    if (/^\s*\{/.test(data.script || '')) {
                        try { // payload may not be JSON; if it is we only need backendType
                            scope.payload = { backendType: JSON.parse(data.script).backendType };
                        } catch (ignored) { /* Nothing for now */ }
                    }

                    enrichSelectedObject(scope.selection.selectedObject, scope.recipe);

                    if (scope.selection.selectedObject.continuous) {
                        // update the build indicator on the flow
                        let selObj = scope.selection.selectedObject;
                        let ps = data.continuousState;
                        // the only change that could not be on the flow is when the activity fails
                        if (!selObj.continuousActivityDone) {
                            if (ps && ps.mainLoopState != null && ps.mainLoopState.futureInfo != null && ps.desiredState == "STARTED" && ps.mainLoopState.futureInfo.hasResult) {
                                selObj.continuousActivityDone = true;
                                $rootScope.$broadcast("graphRendered");
                            }
                        }
                    }
                    scope.recipe.zone = (scope.selection.selectedObject.usedByZones || [])[0] || scope.selection.selectedObject.ownerZone;

                    scope.outputs = data.outputs;
                    scope.isOutputEmpty = scope.outputs === undefined || _.isEmpty(scope.outputs);

                }).error(setErrorInScope.bind(scope));
            };

            scope.$on('taggableObjectTagsChanged', () => scope.refreshData());

            /* Auto save when summary is modified */
            scope.$on("objectSummaryEdited", () => {
                const editedRecipe = scope.recipe !== null ? scope.recipe : scope.recipeBeingDeselected;
                if (editedRecipe) {
                    DataikuAPI.flow.recipes.save(ActiveProjectKey.get(), editedRecipe, { summaryOnly: true })
                        .success(() => ActivityIndicator.success("Saved"))
                        .error(setErrorInScope.bind(scope));
                }
            });

            scope.$watch("selection.selectedObject", function(nv, ov) {
                if (!nv) return;
                scope.recipeData = {recipe: nv, timeline: {}}; // display temporary (incomplete) data
                if(scope.selection.confirmedItem != scope.selection.selectedObject) {
                    // We need to keep a reference on the recipe that was selected in case an event "objectSummaryEdited" arrives just after this event
                    // See https://app.shortcut.com/dataiku/story/152840 for more details.
                    scope.recipeBeingDeselected = scope.recipe;
                    scope.recipe = null;
                }
                scope.recipeType = nv.recipeType || nv.type;
            });

            scope.$watch("selection.confirmedItem", function(nv, ov) {
                if (!nv) {
                    return;
                }
                scope.refreshData();
            });

            scope.saveCustomFields = function(newCustomFields) {
                WT1.event('custom-fields-save', {objectType: 'RECIPE'});
                const oldCustomFields = angular.copy(scope.recipe.customFields);
                scope.recipe.customFields = newCustomFields;
                return DataikuAPI.flow.recipes.save(ActiveProjectKey.get(), scope.recipe, { summaryOnly: true })
                    .success(() => $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), scope.recipe.customFields))
                    .error((data, status, headers, config, statusText, xhrStatus) => {
                        scope.recipe.customFields = oldCustomFields;
                        setErrorInScope.bind(scope)(data, status, headers, config, statusText, xhrStatus);
                    });
            };

            scope.editCustomFields = function() {
                if (!scope.recipe) {
                    return;
                }
                let modalScope = angular.extend(scope, {objectType: 'RECIPE', objectName: scope.recipe.name, objectCustomFields: scope.recipe.customFields});
                CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
                    scope.saveCustomFields(customFields);
                });
            };

            scope.renameRecipe = function() {
                RecipeRenameService.startRenamingRecipe(scope, $stateParams.projectKey, scope.recipe.name);
            }

            scope.runRecipe = function(){
                FlowBuildService.openRecipeRunModal(scope, scope.recipe.projectKey, scope.recipe);
            }

            scope.startContinuous = function() {
                WT1.event("start-continuous", {from:'recipe'})
                CreateModalFromTemplate("/templates/continuous-activities/start-continuous-activity-modal.html", scope, "StartContinuousActivityController", function(newScope) {
                    newScope.recipeId = scope.recipe.name;
                }).then(function(loopParams) {
                    DataikuAPI.continuousActivities.start($stateParams.projectKey, scope.recipe.name, loopParams).success(function(data){
                        scope.refreshData();
                    }).error(setErrorInScope.bind(scope));
                });
            }
            scope.stopContinuous = function(){
                WT1.event("stop-continuous", {from:'recipe'})
                DataikuAPI.continuousActivities.stop($stateParams.projectKey, scope.recipe.name).success(function(data){
                    scope.refreshData();
                }).error(setErrorInScope.bind(scope));
            }

            scope.goToCurrentRun = function() {
                let recipeState = scope.recipeData.continuousState || {};
                let mainLoopState = recipeState.mainLoopState || {};
                $state.go("projects.project.continuous-activities.continuous-activity.runs", {continuousActivityId: recipeState.recipeId, runId: mainLoopState.runId, attemptId: mainLoopState.attemptId});
            };

            scope.updateUserInterests = function() {
                DataikuAPI.interests.getForObject($rootScope.appConfig.login, "RECIPE", ActiveProjectKey.get(), scope.selection.selectedObject.name)
                    .success(function(data){
                        scope.selection.selectedObject.interest = data;
                        scope.recipeData.interest = data;
                    })
                    .error(setErrorInScope.bind(scope));
            }
            const interestsListener = $rootScope.$on('userInterestsUpdated', scope.updateUserInterests);
            scope.$on("$destroy", interestsListener);
        }
    }
});


app.controller("RecipeDetailsController", function ($scope, $state, $filter, CachedAPICalls, ShakerProcessorsInfo, ShakerProcessorsUtils, Notification, RecipesUtils) {
    if (!$scope.processors) {
        // Get the processor, to display their description when they are failing
        CachedAPICalls.processorsLibrary.success(function(processors){
            $scope.processors = processors;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.getObjectType = function(object) {
        switch(object.type) {
            case 'SAVED_MODEL':     return 'SAVED_MODEL';
            case 'MANAGED_FOLDER':  return 'MANAGED_FOLDER';
            default:                return 'DATASET_CONTENT';
        }
    };

    $scope.isOnRecipeObjectPage = function() {
        return $state.includes('projects.project.recipes.recipe');
    }

    $scope.showCodeRecipeSummary = function() {
        if (!$scope.data) {
            return false;
        }

        const recipeHasLanguageAndScript = $filter('recipeTypeToLanguage')($scope.data.recipe.type) && $scope.data.script;
        if (!recipeHasLanguageAndScript) {
            return false
        }

        const recipeHasAssociatedNotebook =  $scope.data.notebook && $scope.data.notebook.projectKey && $scope.data.notebook.name;
        if ($scope.isOnRecipeObjectPage() && !recipeHasAssociatedNotebook) {
            return false;
        }

        return true;
    }

    $scope.getFlatAggregates = function(values) {
        if (!values) {
            return [];
        }
        var aggregates = [];
        values.forEach(function(value) {
            if (value.customExpr) {
                aggregates.push(value);
            } else {
                angular.forEach(value, function(x, agg) {
                    if (agg.startsWith("__")) return; // temp field
                    if (x === true) {
                        aggregates.push({agg:agg, column:value.column, type:value.type});
                    }
                });
            }
        });
        return aggregates;
    }

    $scope.validateStep = function(step) {
        return ShakerProcessorsUtils.validateStep(step, $scope.processors);
    };

    /*
     * Displaying info to user
     */
    var tmpFindGroupIndex = [];
    $scope.findGroupIndex = function(step) {
        const groupIndex = tmpFindGroupIndex(step);
        return groupIndex < 0 ? '' : groupIndex + 1;
    }

    $scope.$watch("data.script.steps", function(nv) {
        if (!nv) return;

        tmpFindGroupIndex =Array.prototype.indexOf.bind($scope.data.script.steps.filter(function(s) { return s.metaType === 'GROUP'; }));

        if ($scope.data.recipe.type === "shaker") {
            const { flattenedEnabledSteps } = ShakerProcessorsUtils.getStepsWithSubSteps($scope.data.script.steps, $scope.processors);
            $scope.data.stepsWithSubSteps = flattenedEnabledSteps;
        }
    });

    const recipeSaveListener = Notification.registerEvent("recipe-save", (type, data) => {
        if(!$scope.data) $scope.data = {};
        $scope.data.script = data.payloadData;
        $scope.data.recipe = data.recipe;
        RecipesUtils.parseScriptIfNeeded($scope.data);
    });

    $scope.$on('$destroy', recipeSaveListener);
});

app.controller("RecipeEditorController",
    function ($scope, $rootScope, $timeout, $stateParams, $filter, $location, $state, $q,
    Assert, BigDataService, DataikuAPI, Dialogs, WT1, FutureProgressModal,
    TopNav, PartitionDeps, DKUtils, Logger, HistoryService,
    CreateModalFromTemplate, AnyLoc, JobDefinitionComputer, RecipeComputablesService, RecipesUtils,
    RecipeRunJobService, PartitionSelection, RecipeDescService, InfoMessagesUtils, StateUtils, GraphZoomTrackerService,
    DatasetUtils, CodeStudiosService, Notification, RecipeContextService,
    ActivityIndicator, ManualLineageModalService, RatingFeedbackParams) {

    $scope.ratingFeedbackParams = RatingFeedbackParams;
    $scope.hasJobBanner = false;

    $scope.$on('$stateChangeSuccess', (event, toState, toParams) => {
        if (toState.name === "projects.project.recipes.recipe" && toParams.isAIGenerated === true && !$rootScope.appConfig.isUsingLocalAiAssitant?.prepareAICompletion) {
            // We are entering a freshly IA generated recipe, so we can show the rating feedback banner
            $scope.ratingFeedbackParams.showRatingFeedback = true;
        }
    });

    //This is necessary to make sure  the banner disappears when we route to another page (otherwise, it appears again after we open a recipe). In case we're routing to explore the dataset we instead want to display back the banner
    $scope.$on('$stateChangeStart', (event, toState, toParams, fromState, fromParams) => {
        const outputDatasets = $scope.recipe?.outputs?.['main']?.items.map(item => item.ref) ?? [];
        if (!(toState.name === 'projects.project.datasets.dataset.explore' && outputDatasets.includes(toParams.datasetName) && fromParams.isAIGenerated === true)) {
            // We are NOT leaving a freshly IA generated recipe to go to one of its output dataset, so we can drop the rating feedback banner
            $scope.ratingFeedbackParams.showRatingFeedback = false;
        }
    });

    $scope.onCloseRatingFeedbackBanner = function() {
        $scope.ratingFeedbackParams.showRatingFeedback = false;
    }

    $scope.InfoMessagesUtils = InfoMessagesUtils;

    let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;

    $scope.recipeUpdateData = function(initialCall = false) {
        return DataikuAPI.flow.recipes.getWithInlineScript($stateParams.projectKey, $scope.recipeName.name).success(function(data) {
            RecipeContextService.setCurrentRecipe(data);
            $scope.recipe = data.recipe;
            $scope.script.data = data.script;
            $scope.canEditInCodeStudio = data.canEditInCodeStudio;
            $scope.origRecipe = angular.copy($scope.recipe);
            $scope.origScript = angular.copy($scope.script);

            $scope.recipeDesc = RecipeDescService.getDescriptor($scope.recipe.type);

            TopNav.setItem(TopNav.ITEM_RECIPE, data.recipe.name, {
                recipeType :data.recipe.type,
                name : data.recipe.name,
                inputs: data.recipe.inputs,
                outputs: data.recipe.outputs
            });

            RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
                $scope.setComputablesMap(map);
                DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => {
                        if(initialCall){
                            $scope.onload();
                        }
                        $scope.$broadcast('computablesMapChanged'); // because the schema are there now (they weren't when setComputablesMap() was called)
                    });
            });
            DataikuAPI.flow.zones.getZoneId($stateParams.projectKey, {id: data.recipe.name, type: "RECIPE", projectKey: data.recipe.projectKey}).success(zone => {
                if (zone) {
                    // Put it in zone so the io:getDatasetCreationSettings can find it
                    // and we can target more recipes
                    $scope.zone = zone.id;
                }
            });

        }).error(function(){
            HistoryService.notifyRemoved({
                type: "RECIPE",
                id: $scope.recipeName.name,
                projectKey: $stateParams.projectKey
            });
            setErrorInScope.apply($scope, arguments);
        });
    }

	function main(){
        /* Init scope */
        $scope.uiState = {editSummary:false};
        $scope.startedJob = {};
        $scope.recipe = null;
        $scope.recipeStatus = null;
        $scope.payloadRequired = false; // override for recipe specific recipe types. Avoids to get-status before the payload is ready
        $scope.script = {};
        $scope.creation = false;
        $scope.recipeName = { "name" : $scope.$state.params.recipeName };
        $scope.projectKey = $stateParams.projectKey;
        $scope.hooks = $scope.hooks || {};
        GraphZoomTrackerService.setFocusItemByName("recipe", $scope.recipeName.name);
        $scope.RecipesUtils = RecipesUtils

        // Validation context
        $scope.valCtx = {};

        const tabToSelect = StateUtils.defaultTab("settings");
        TopNav.setLocation(TopNav.TOP_FLOW, "recipes", TopNav.TABS_RECIPE, tabToSelect);
        TopNav.setItem(TopNav.ITEM_RECIPE, $stateParams.recipeName);

        $scope.validations = [
            function(){
                return $scope.renaming.recipe_name.$valid && $scope.recipeName.name.length;
            }
        ];

        $scope.PartitionDeps = PartitionDeps;
        addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);

        // DO NOT INITIALIZE IT, IT HELPS CATCH ERRORS
        $scope.computablesMap = null;
        $scope.$broadcast('computablesMapChanged');

        Assert.trueish($scope.recipeName.name, 'no recipe name');

        $scope.recipeUpdateData(true);

        TopNav.setTab(tabToSelect);

        $scope.$watchGroup(['topNav.tab', 'valCtx.preRunValidationError', 'startedJob.jobId'], function([tab, preRunValidationError, jobId]) {
            // The job banner only shows up in the "settings" or "code" tab for recipes, except for the Sync recipe where it shows up in the "io" tab
            $scope.hasJobBanner = (['code', 'settings'].includes(tab) || ($scope.recipe?.type === 'sync' && tab === 'io')) && (preRunValidationError || jobId)
        })

    }
    main();

    function extractWT1EventParams(recipe, payload) {
        try {
            if (recipe.type === "prediction_scoring") {
                const recipeParams = JSON.parse(payload.data);
                let eventParams = {
                    filterInputColumns: recipeParams.filterInputColumns,
                    forceOriginalEngine: recipeParams.forceOriginalEngine,
                    outputExplanations: recipeParams.outputExplanations,
                    outputProbaPercentiles: recipeParams.outputProbaPercentiles,
                    outputProbabilities: recipeParams.outputProbabilities,
                    taskType: "PREDICTION",
                    predictionType: recipeParams.predictionType,
                    savedModelType: recipeParams.savedModelType,
                    proxyModelProtocol: recipeParams.proxyModelProtocol,
                };
                if (eventParams.outputExplanations) {
                    eventParams = {
                        individualExplMethod: recipeParams.individualExplanationParams.method,
                        individualExplCount: recipeParams.individualExplanationParams.nbExplanations,
                        ... eventParams
                    };
                }
                return eventParams;
            }
            if (["clustering_scoring", "clustering_training"].includes(recipe.type)) {
                return {
                    taskType: "CLUSTERING",
                    savedModelType: "DSS_MANAGED"
                    // No Prediction type, saved relates to clustering
                }
            }
            if (recipe.type === "eda_stats") {
                const recipeParams = JSON.parse(payload.data);
                return { recipeSubType: recipeParams.type };
            }
            if (recipe.type === "nlp_llm_finetuning") {
                const recipeParams = JSON.parse(payload.data);
                if (recipeParams.llmId !== undefined) {
                    const llmIdParts = recipeParams.llmId.split(":");
                    return { inputModelType: llmIdParts[0] === "savedmodel" ? llmIdParts[1] : llmIdParts[0] }
                }
            }
            if (recipe.type === "nlp_llm_evaluation") {
                const recipeParams = JSON.parse(payload.data);
                return { inputFormat: recipeParams.inputFormat, llmTaskType: recipeParams.llmTaskType}
            }
            if (recipe.type === "nlp_llm_rag_embedding") {
                const recipeParams = JSON.parse(payload.data);
                return { vectorStoreUpdateMethod: recipeParams.vectorStoreUpdateMethod };
            }
            if (recipe.type === "embed_documents") {
                const payloadData = JSON.parse(payload.data);
                const allOtherRuleAction = recipe.params.allOtherRule.actionToPerform;

                // parsing the vlm full ids to extract only the connectionType (the rest is user-private, shouldn't be sent to wt1)
                let vlms = recipe.params.rules.filter(r => r.actionToPerform === 'VLM').map(r => r.vlmSettings.llmId);
                if (allOtherRuleAction === 'VLM'){
                    vlms.push(recipe.params.allOtherRule.vlmSettings.llmId);
                }
                let vlmConnectionsType = vlms.filter(fullId => fullId !== undefined).map(fullId => fullId.split(':')[0]);

                return {
                    vectorStoreUpdateMethod: payloadData.vectorStoreUpdateMethod,
                    rulesCount: recipe.params.rules.length + 1, // includes allOtherRule
                    vlmRulesCount: recipe.params.rules.filter(r => r.actionToPerform === 'VLM').length + (allOtherRuleAction === 'VLM'? 1: 0),
                    structuredRulesCount: recipe.params.rules.filter(r => r.actionToPerform === 'STRUCTURED').length + (allOtherRuleAction === 'STRUCTURED'? 1: 0),
                    donotextractRulesCount: recipe.params.rules.filter(r => r.actionToPerform === 'DONOTEXTRACT').length + (allOtherRuleAction === 'DONOTEXTRACT'? 1: 0),
                    vlmConnectionTypes:  Array.from(new Set(vlmConnectionsType)).join(",") // only keep unique values
                };
            }
            if (recipe.type === "evaluation") {
                const recipeParams = JSON.parse(payload.data);
                return {
                    taskType: recipeParams.taskType,
                    predictionType: recipeParams.predictionType,
                    savedModelType: recipeParams.savedModelType,
                    proxyModelProtocol: recipeParams.proxyModelProtocol
                }
            }
            if (recipe.type === "prediction_training") {
                const recipeParams = JSON.parse(payload.data);
                return {
                    taskType: recipeParams.core.taskType,
                    predictionType: recipeParams.core.prediction_type,
                    savedModelType: "DSS_MANAGED"
                }
            }
            return {};
        } catch (e) {
            Logger.error("Failed to get wt1 loggable recipe event payload params", e);
            return {};
        }
    }

    const DEFAULT_WT1_EVENT_OPTS = {
        withRecipeEventParamsEnrichment: false,
        withRecipeOutputAppendModeTracking: false,
    };

    function enrichRecipeWT1Event(params, opts = {}) {
        if (params == null) params = {};
        params.recipeId = ($scope.recipeName && $scope.recipeName.name) ? $scope.recipeName.name.dkuHashCode() : "unknown";
        params.recipeType = ($scope.recipe ? $scope.recipe.type : "unknown");
        params.creation = $scope.creation;
        if ($scope.recipe && $scope.recipe.type) {
            const extractParams = extractWT1EventParams($scope.recipe, $scope.script);
            params = { ...params, ...extractParams };
        }
        const _opts = { ...DEFAULT_WT1_EVENT_OPTS, ...opts }
        if (_opts.withRecipeEventParamsEnrichment) {
            params = {
                ...params,
                ...RecipesUtils.getWT1LoggableRecipeEventParams($scope.recipe, $scope.recipeDesc),
            };
        }
        if (_opts.withRecipeOutputAppendModeTracking) {
            params = {
                ...params,
                ...RecipesUtils.getWT1OutputAppendModeTracking($scope.recipe),
            };
        }
        return params;
    }

    // prefer using tryRecipeWT1Event for new events
    $scope.recipeWT1Event = function(type, params, opts) {
        WT1.event(type, enrichRecipeWT1Event(params, opts));
    };

    $scope.tryRecipeWT1Event = function(type, paramsGetter, message) {
        WT1.tryEvent(type, () => enrichRecipeWT1Event(paramsGetter()), message);
    };

    $scope.canEditRecipeInNotebook = function() {
        return ['python', 'pyspark', 'r', 'julia', 'sparkr', 'spark_scala', 'sql_query'].includes($scope.recipe.type);
    };
    $scope.editThisRecipeInNotebook = function() {
        $scope.recipeWT1Event('recipe-edit-in-notebook');
        if($scope.recipe.type === 'sql_query') {
            $scope.saveRecipeIfPossible()
                .then(() => DataikuAPI.flow.recipes.editInSQLNotebook($stateParams.projectKey, $stateParams.recipeName))
                .then(({ data }) => StateUtils.go.sqlNotebook(data.notebookId, $stateParams.projectKey, {cellId: data.cellId}))
                .catch(setErrorInScope.bind($scope));
        } else {
            var editInNotebook = function() {
                DataikuAPI.flow.recipes.editInNotebook($stateParams.projectKey, $stateParams.recipeName, $scope.recipe.params.envSelection, $scope.recipe.params.containerSelection).success(function(data) {
                    StateUtils.go.jupyterNotebook(data.id, $stateParams.projectKey);
                }).error(setErrorInScope.bind($scope));
            };
            $scope.saveRecipeIfPossible().then(function() {
                DataikuAPI.flow.recipes.checkNotebookEdition($stateParams.projectKey, $stateParams.recipeName).success(function(data) {
                    if (!data || data.conflict !== true || !data.notebook) {
                        editInNotebook();
                    } else {
                        Dialogs.openEditInNotebookConflictDialog($scope).then(
                            function(resolutionMethod) {
                                if(resolutionMethod == 'erase') {
                                    editInNotebook();
                                } else if(resolutionMethod == 'ignore') {
                                    StateUtils.go.jupyterNotebook(data.notebook, $stateParams.projectKey);
                                }
                            }
                        );
                    }
                }).error(setErrorInScope.bind($scope));
            }).catch(setErrorInScope.bind($scope));
        }
    };

    $scope.canEditRecipeInCodeStudio = function() {
        // not all the python recipe or r recipes are listed here : it's because you won't realistically
        // be able to debug or test python code that relies on spark or kafka in a CodeStudio
        if (!$scope.canEditInCodeStudio) return false;
        return ['python', 'r', 'sql_query', 'sql_script', 'spark_sql_query'].includes($scope.recipe.type);
    };

    const getExtension = function() {
        switch($scope.recipe.type) {
            case 'sql_query':
            case 'sql_script':
            case 'spark_sql_query':
                return '.sql';
            case 'python':
                return '.py';
            case 'r':
                return '.r';
            default:
                throw Error('Edition of recipes of type ' + $scope.recipe.type + 'is not supported by Code Studio!');
        }
    }

    $scope.editThisRecipeInCodeStudio = function() {
        $scope.saveRecipeIfPossible().then(function() {
            CodeStudiosService.editFileInCodeStudio($scope, "recipes", $stateParams.recipeName + getExtension());
        }).catch(setErrorInScope.bind($scope));
    }

    $scope.saveCustomFields = function(newCustomFields) {
        WT1.event('custom-fields-save', {objectType: 'RECIPE'});
        let oldCustomFields = angular.copy($scope.recipe.customFields);
        $scope.recipe.customFields = newCustomFields;
        return $scope.hooks.save().then(function() {
                $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), $scope.recipe.customFields);
            }, function() {
                $scope.recipe.customFields = oldCustomFields;
            });
    };

    $scope.editCustomFields = function() {
        if (!$scope.recipe) {
            return;
        }
        let modalScope = angular.extend($scope, {objectType: 'RECIPE', objectName: $scope.recipe.name, objectCustomFields: $scope.recipe.customFields});
        CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
            $scope.saveCustomFields(customFields);
        });
    };

    $scope.gotoLine = function(cm, line) {
        if(cm && line>0) {
            var pos = {ch:0,line:line-1};
            cm.scrollIntoView(pos);
            cm.setCursor(pos);
            cm.focus();
        }
    };

    let lastVeLoopConfig = null;
    $scope.loadRecipeVariables = function() {
        const veLoopConfig = $scope.recipe.params && $scope.recipe.params.variablesExpansionLoopConfig;
        if (!angular.equals(lastVeLoopConfig, veLoopConfig) || lastVeLoopConfig === null) {
            lastVeLoopConfig = typeof veLoopConfig === "object"
                ? JSON.parse(JSON.stringify(veLoopConfig))
                : veLoopConfig;
            DataikuAPI.flow.recipes.generic.getVariables($scope.recipe).success(function(data) {
                $scope.recipeVariables = data;
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.specificControllerLoadedDeferred = $q.defer();

    /* Method called once recipe is loaded */
    var onloadcalled = false;
    $scope.onload = function() {
        Assert.inScope($scope, 'recipe');
        Assert.trueish(!onloadcalled, 'already loaded');
        onloadcalled = true;

        $scope.loadRecipeVariables();

        $scope.fixupPartitionDeps();

        $scope.recipeWT1Event("recipe-open");

        // TODO: Check if still needed
        $scope.ioFilter = {};

        $scope.testRun = {
            build_partitions: {},
            runMode: "NON_RECURSIVE_FORCED_BUILD"
        };

        /* Synchronize the definition of build_partitions for the test run
         * with the partitioning schema of the first partitioned output */
        $scope.$watch("recipe.outputs", function(nv, ov) {
            if (nv != null) {
                clear($scope.testRun.build_partitions);
                DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => {
                        const definingOutputPartitioning = RecipeRunJobService.getOutputAndPartitioning($scope.recipe, $scope.computablesMap).partitioning;
                        if (definingOutputPartitioning && definingOutputPartitioning.dimensions.length) {
                            $scope.outputPartitioning = definingOutputPartitioning;
                            $scope.testRun.build_partitions = PartitionSelection.getBuildPartitions($scope.outputPartitioning);
                        } else {
                            $scope.outputPartitioning = { dimensions: [] };
                        }
                    });
            } else {
                $scope.testRun.build_partitions = null;
            }
        }, true);

        $scope.fixupPartitionDeps();

        /* When the specific recipe controller has finished loading AND we have
         * the computables map, then we call its own onload hook */
        $scope.specificControllerLoadedDeferred.promise.then(function() {
            if ($scope.hooks && $scope.hooks.onRecipeLoaded){
                $scope.hooks.onRecipeLoaded();
            }
        });
    };

    $scope.hooks.getRecipeSerialized = function(){
        var recipeSerialized = angular.copy($scope.recipe);
        PartitionDeps.prepareRecipeForSerialize(recipeSerialized);
        return recipeSerialized;
    };

    $scope.hooks.resetScope = function() {
        clear($scope.startedJob);
        clear($scope.valCtx);
    };

    //Override it to return a string representing the payload
    $scope.hooks.getPayloadData = function() {};

    /* ***************************** Inputs/Outputs *************************** */

    $scope.hasAllRequiredOutputs = function() {
        if (!$scope.recipe || !$scope.recipe.outputs) {
            return false;
        }
        var out = $scope.recipe.outputs;
        //TODO implement for any role
        if(out.main) {
            return !!(out.main.items && out.main.items.length);
        }
        return true;//Other roles: don't know
    };

    $scope.hasPartitionedInput = function() {
        return $scope.getInputDimensions().length > 0;
    };

    $scope.hasPartitionedOutput = function() {
        return $scope.getOutputDimensions().length > 0;
    };

    $scope.hasDangerousAllAvailablePartitioning = function() {
        const hasAllAvailableDependency = () => {
            const recipeInputs = $scope.recipe.inputs || {};
            const inputValues = Object.values(recipeInputs);

            return inputValues.some(input =>
                input.items.some(item =>
                    item.deps.some(dep => dep.func === 'all_available')
                )
            );
        };
        return $scope.hasPartitionedInput() && $scope.hasPartitionedOutput() && hasAllAvailableDependency();
    }

    $scope.hasInvalidPartitionSelection = function() {
        if ($scope.recipe && $scope.recipe.redispatchPartitioning) {
            return false; // Redispatch partitioning does not require target partitions
        }
        return $scope.getOutputDimensions().some((dimension) => {
            return !$scope.testRun || $scope.testRun.build_partitions[dimension.name] === void 0 || $scope.testRun.build_partitions[dimension.name] === "";
        });
    };

    // This method should be called each time inputs or outputs are modified.
    $scope.fixupPartitionDeps = function(){
        if (!$scope.recipe || !$scope.computablesMap) return;
        var ret = PartitionDeps.fixup($scope.recipe, $scope.computablesMap);
        $scope.outputDimensions = ret[0];
        $scope.outputDimensionsWithNow = ret[1];
    };

    $scope.testPDep = function(inputRef, pdep) {
        PartitionDeps.test($scope.recipe, inputRef, pdep, $scope);
    };

    $scope.refreshDatasetInComputablesMap = function(dataset) {
        var found = null;
        $.each($scope.computablesMap, function(smartName, computable) {
            if (computable.projectKey == dataset.projectKey && computable.name == dataset.name)
                found = computable;
        });
        // the dataset has to be in the computablesMap, otherwise that means it's not even shown in the dataset left pane
        Assert.trueish(found);
        found.dataset = dataset;
    };

    // Keep veloop input in sync with VELoop config
    $scope.$watch("recipe.params.variablesExpansionLoopConfig", function() {
        if (!$scope.recipe) {
            return;
        }
        // Replicate what happens in the backend when a recipe is saved
        const role = "veloop";
        RecipesUtils.removeInputsForRole($scope.recipe, role);
        if (!$scope.recipe.params) {
            return;
        }
        const veLoopConfig = $scope.recipe.params.variablesExpansionLoopConfig;
        if (veLoopConfig && veLoopConfig.enabled && veLoopConfig.datasetRef) {
            RecipesUtils.addInput($scope.recipe, role, veLoopConfig.datasetRef);
        }
    }, true);

    /* Simple recipes that don't want to manage themselves inputs and outputs
     * should enable auto fixup */
    $scope.enableAutoFixup = function() {
        $scope.$watch("recipe.inputs", function() {
            DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => $scope.fixupPartitionDeps());
        }, true);
        $scope.$watch("recipe.outputs", function() {
            DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => $scope.fixupPartitionDeps());
        }, true);
    };

    $scope.$watch("recipe.inputs", function(nv, ov) {
        if (!nv) return;
        if (!$scope.outputDimensions) return;
        DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => {
            RecipesUtils.getFlatInputsList($scope.recipe).forEach(function(input) {
                if (!input.deps) return;
                input.deps.forEach(function(pdep){
                    PartitionDeps.autocomplete(pdep, $scope.outputDimensions, $scope.outputDimensionsWithNow);
                });
            });
        });
    }, true);

    $scope.$watch("recipe.outputs", function() {
        DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey);
    }, true);

    $scope.setComputablesMap = function(map) {
        $scope.computablesMap = map;
        $scope.$broadcast('computablesMapChanged');
    };

    $scope.getInputDimensions = function(){
        if (!$scope.recipe || !$scope.computablesMap) return [];
        return RecipeRunJobService.getInputDimensions($scope.recipe, $scope.computablesMap);
    };

    $scope.getOutputDimensions = function(){
        if (!$scope.recipe || !$scope.computablesMap) return [];
        return RecipeRunJobService.getOutputDimensions($scope.recipe, $scope.computablesMap);
    };

    $scope.hasAnyPartitioning = function(){
        return RecipesUtils.hasAnyPartitioning($scope.recipe, $scope.computablesMap);
    };

    /* ***************************** MANUAL LINEAGE  *************************** */

    $scope.showManualLineageModal = function() {
        const recipeInfo = {
            projectKey: $scope.recipe.projectKey,
            name: $scope.recipe.name,
            type: $scope.recipe.type,
        }
        const datasetInfos = (recipeIO) => {
             return Object.values(recipeIO).flatMap(inputs => {
                return inputs.items.map(input => {
                    const datasetLoc = AnyLoc.getLocFromSmart(recipeInfo.projectKey, input.ref);
                    const datasetInfo = {
                        projectKey: datasetLoc.projectKey,
                        name: datasetLoc.localId,
                    };
                    return datasetInfo;
                });
            });
        };
        const inputs = datasetInfos($scope.recipe.inputs);
        const outputs = datasetInfos($scope.recipe.outputs);
        const showInCurrentLineageOnlyFilter = false;
        const wt1EventFrom = "recipe-advanced-settings";
        ManualLineageModalService.openManualLineageModal(recipeInfo, inputs, outputs, showInCurrentLineageOnlyFilter, wt1EventFrom)
            .then(function(saveInformation) {
                if (saveInformation) {
                    DataikuAPI.dataLineage.saveManualLineage(saveInformation.projectKey, saveInformation.recipeName, saveInformation.manualDataLineages, saveInformation.ignoreAutoComputedLineage)
                        .then(({data}) => {
                            ActivityIndicator.success("Updated lineage");
                        })
                        .catch(setErrorInScope.bind($scope));
                }
            });
    }

    /* ***************************** Save *************************** */

    $scope.hooks.save = function() {
        return $scope.baseSave($scope.hooks.getRecipeSerialized(), $scope.script ? $scope.script.data : null);
    };
    $scope.hooks.origSaveHook = $scope.hooks.save;

    $scope.baseSave = function(recipeSerialized, payloadData){
        $scope.recipeWT1Event("recipe-save", null, {withRecipeEventParamsEnrichment: true, withRecipeOutputAppendModeTracking: true});
        Notification.publishToFrontend("recipe-save", {recipe: recipeSerialized, payloadData: payloadData});
        return DataikuAPI.flow.recipes.save($stateParams.projectKey, recipeSerialized,
            payloadData, $scope.currentSaveCommitMessage).success(function(savedRecipe){
            var newVersionTag = savedRecipe.versionTag;
            $scope.origRecipe = angular.copy($scope.recipe);
            $scope.origScript = angular.copy($scope.script);
            $scope.recipe.versionTag = newVersionTag;
            $scope.origRecipe.versionTag = newVersionTag;
            $scope.creation = false;
            $scope.currentSaveCommitMessage = null;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.canSave = function(){
        if (!$scope.creation) return true;
        return $scope.recipeName.name && $scope.recipeName.name.length;
    };

    $scope.hooks.recipeIsDirty = function() {
        if (!$scope.recipe) return false;
        if ($scope.creation) {
            return true;
        } else {
            // compare after fixing up the partition deps, otherwise their change is missed by the dirtyness tracking
            var recipeSerialized = angular.copy($scope.recipe);
            PartitionDeps.prepareRecipeForSerialize(recipeSerialized);
            var origRecipeSerialized = angular.copy($scope.origRecipe);
            PartitionDeps.prepareRecipeForSerialize(origRecipeSerialized, true);

            var dirty = !angular.equals(recipeSerialized, origRecipeSerialized);
            if ($scope.script) {
                dirty = dirty || !angular.equals($scope.origScript, $scope.script);
            }
            return dirty;
        }
    };

    $scope.hooks.recipeContainsUnsavedFormulaChanges = function() {
        return false;
    };

    //Don't link to the default recipeIsDirty is function, get the actual one that may be defined later
    checkChangesBeforeLeaving($scope, (function(_scope) { return function() {
        return _scope.hooks.recipeIsDirty() || _scope.hooks.recipeContainsUnsavedFormulaChanges();
     }
     })($scope));

    $scope.saveRecipe = function(commitMessage){
        var deferred = $q.defer();

        var saveAfterConflictCheck = function() {
            $scope.currentSaveCommitMessage = commitMessage;
            $scope.hooks.save().then(function() {
                $scope.$broadcast('recipeSaved');
                deferred.resolve('recipe saved');
            },function() {
                deferred.reject();
            });
        };

        DataikuAPI.flow.recipes.checkSaveConflict($stateParams.projectKey, $stateParams.recipeName,$scope.recipe).success(function(conflictResult) {
            if(!conflictResult.canBeSaved) {
                Dialogs.openConflictDialog($scope,conflictResult).then(
                        function(resolutionMethod) {
                            if(resolutionMethod == 'erase') {
                                saveAfterConflictCheck();
                            } else if(resolutionMethod == 'ignore') {
                                deferred.reject();
                                DKUtils.reloadState();
                            }
                        }
                );
            } else {
                saveAfterConflictCheck();
            }
        }).error(setErrorInScope.bind($scope));
        return deferred.promise;
    };

    $scope.saveRecipeIfPossible = function(){
        if ($scope.canSave()) {
            return $scope.saveRecipe();
        }
        return $q.defer().promise;
    };

    $scope.displayAllMessagesInModal = function(){
        Dialogs.infoMessagesDisplayOnly($scope, "Recipe validation",
            $scope.valCtx.validationResult.allMessagesForFrontend);
    };

    /* ***************************** Execution *************************** */

    $scope.buildModes = [
        ["NON_RECURSIVE_FORCED_BUILD", $scope.translate("RECIPE.RUN_OPTIONS.BUILD_MODE.NON_RECURSIVE_FORCED_BUILD", "Run only this recipe")],
        ["RECURSIVE_BUILD", $scope.translate("RECIPE.RUN_OPTIONS.BUILD_MODE.RECURSIVE_BUILD", "Build required dependent datasets")],
        ["RECURSIVE_FORCED_BUILD", $scope.translate("RECIPE.RUN_OPTIONS.BUILD_MODE.RECURSIVE_FORCED_BUILD", "Force-rebuild all dependent datasets")],
        ["RECURSIVE_MISSING_ONLY_BUILD", $scope.translate("RECIPE.RUN_OPTIONS.BUILD_MODE.RECURSIVE_MISSING_ONLY_BUILD", "Build missing dependencies and run this recipe")]
    ];

    $scope.jobCheckTimer = null;

    $scope.hooks.preRunValidate = function() {
        var deferred = $q.defer();
        DataikuAPI.flow.recipes.generic.validate($stateParams.projectKey,
            $scope.hooks.getRecipeSerialized()).success(function(data) {
            deferred.resolve(data);
        }).error(function(a,b,c) {
            setErrorInScope.bind($scope)(a,b,c);
            deferred.reject("Validation failed");
        });
        return deferred.promise;
    };

    $scope.editRunOptions = function(){
        CreateModalFromTemplate("/templates/recipes/recipe-run-options-modal.html", $scope);
    };

    $scope.multipleLogsToDisplay = function() {
        //We're restricting this down to specific engines for now
        let logsWantedForEngine = "python" == $scope.recipe.type || "r" == $scope.recipe.type || "shell" == $scope.recipe.type;
        return $scope.startedJob.jobId && $scope.startedJob.jobStatus && logsWantedForEngine && $scope.startedJob.jobStatus.manyLogs;
    }

    $scope.selectLogType = function(logTypeKey) {
        if (!$scope.startedJob || !$scope.startedJob.jobStatus || !$scope.startedJob.jobStatus.logsByType) {
            return;
        }

        let logType = $scope.startedJob.jobStatus.logsByType[logTypeKey];

        let updateChosenLogState = function() {
            $scope.startedJob.jobStatus.logTailHTML = logType.logHTML;
            $scope.startedJob.jobStatus.chosenLog = logTypeKey;
        }

        if (logType) {
            if (logType.logHTML) {
                updateChosenLogState();
            } else {
                 DataikuAPI.flow.jobs.smartTailActivityAdditionalLog($scope.projectKey, $scope.startedJob.jobId, logType.activityId, logType.logPath, 500)
                    .success(function (data) {
                        logType.logHTML = smartLogTailToHTML(data, false);
                        updateChosenLogState();
                    })
                    .error(setErrorInScope.bind($scope));
            }
        }
    }

    $scope.waitForEndOfStartedJob = function() {
        Logger.info("Wait for end of job:", $scope.startedJob.jobId);
        DataikuAPI.flow.jobs.getJobStatus($stateParams.projectKey, $scope.startedJob.jobId).success(function(data) {
            $scope.startedJob.jobStatus = data;
            data.totalWarningsCount = 0;
            if (data.logTail != null) {
                data.logTailHTML = smartLogTailToHTML(data.logTail, false);
            }
            for (var actId in data.baseStatus.activities) {
                var activity = data.baseStatus.activities[actId];
                if (activity.warnings) {
                    data.totalWarningsCount += activity.warnings.totalCount;
                }
            }
            if (data.baseStatus.state != "DONE" && data.baseStatus.state != "ABORTED" &&
                data.baseStatus.state != "FAILED") {
                $scope.jobCheckTimer = $timeout($scope.waitForEndOfStartedJob, 2000);
            } else {
                // The run has finished

                //We'll have a log if there was a failure
                if (data.logTailHTML) {
                    //If there are additional logs from the first failed activity (e.g. python ones), we offer these to the user
                    // but we don't download them yet, just collect their info
                    let logsByType = {};
                    $scope.startedJob.jobStatus.logsByType = logsByType;

                    // the main log is displayed by default but we add it to the logsByType map so the user can switch back to it
                    const MAIN_LOG_KEY_LABEL = "Main activity log";
                    logsByType[MAIN_LOG_KEY_LABEL] =  { logHTML : data.logTailHTML};
                    $scope.startedJob.jobStatus.chosenLog = MAIN_LOG_KEY_LABEL;
                    $scope.startedJob.jobStatus.manyLogs = false;

                    if (data.logTailActivityId) {
                        let failedActivity = data.baseStatus.activities[data.logTailActivityId];
                        if (failedActivity && failedActivity.statusOutputs && failedActivity.statusOutputs.length > 0) {
                            $scope.startedJob.jobStatus.manyLogs = true;
                            for (let statusOutput of failedActivity.statusOutputs) {
                                //Add an entry with empty logHTML - we will lazy load it
                                logsByType[statusOutput.label] = { activityId : failedActivity.activityId, logPath : statusOutput.path, logHTML : null};
                            }
                        }
                    }

                }

                $scope.recipeWT1Event("recipe-run-finished", {
                    state : data.baseStatus.state
                });
            }
            $timeout(function() {$rootScope.$broadcast("reflow");},50);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.waitForEndOfStartedContinuousActivity = function() {
        Logger.info("Wait for end of continuous activity:", $scope.startedJob.jobId);
        DataikuAPI.continuousActivities.getState($stateParams.projectKey, $stateParams.recipeName).success(function(data) {
            $scope.startedJob.persistent = data;
            $scope.startedJob.current = data.mainLoopState;
            if ($scope.startedJob.current && $scope.startedJob.current.futureInfo && ($scope.startedJob.current.futureInfo.alive || !$scope.startedJob.current.futureInfo.hasResult)) {
                $scope.jobCheckTimer = $timeout($scope.waitForEndOfStartedContinuousActivity, 2000);
            } else {
                // not running anymore
            }
            $timeout(function() {$rootScope.$broadcast("reflow");},50);
        }).error(setErrorInScope.bind($scope));

    };

    $scope.discardStartedJob = function(){
        clear($scope.startedJob);
        if($scope.jobCheckTimer) {
           $timeout.cancel($scope.jobCheckTimer);
           $scope.jobCheckTimer = null;
           $timeout(function() {
               $rootScope.$broadcast('redrawFatTable');
           });
        }
    };

    $scope.abortSingleRecipeExecution = function() {
        Dialogs.confirm($scope, 'Aborting a job','Are you sure you want to abort this job?').then(function() {
            DataikuAPI.flow.jobs.abort($stateParams.projectKey,$scope.startedJob.jobId).success(function(data) {
                $scope.discardStartedJob();
            }).error(function(e) {
                // swallow this error
                Logger.error(e);
            });
            $scope.recipeWT1Event("recipe-running-abort");
        });
    };

    $scope.isJobRunning = function() { return RecipeRunJobService.isRunning($scope.startedJob); };

    $scope.isJobPending = function() { return RecipeRunJobService.isPending($scope.startedJob); };

    $scope.isContinuousActivityRunning = function() { return $scope.startedJob && $scope.startedJob.jobId && $scope.startedJob.current && $scope.startedJob.current.futureInfo && ($scope.startedJob.current.futureInfo.alive || !$scope.startedJob.current.futureInfo.hasResult); };

    //TODO @recipes32 this is a little flawed, there is a short moment between starting and running...
    $scope.isJobRunningOrStarting = function() {
        return $scope.isJobRunning() || !!$scope.startedJob.starting || $scope.isJobPending();
    };
    $scope.isContinuousActivityRunningOrStarting = function() {
        return $scope.isContinuousActivityRunning() || !!$scope.startedJob.starting;
    };

    $scope.startSingleRecipeExecution = function(forced) {
        $scope.hooks.resetScope();
        $scope.startedJob.starting = true;

        $scope.currentSaveIsForAnImmediateRun = true;

        if ($scope.recipe.redispatchPartitioning) {
            $scope.testRun.build_partitions = {}; // Build partitions make no sense when in redispatch partitioning mode
        }

        function doIt() {
            const runRecipeOnlyRecipeTypes = ["nlp_llm_finetuning"]
            RecipeRunJobService.run($scope.recipe, $scope.computablesMap, $scope.testRun, $scope.startedJob, $scope.waitForEndOfStartedJob, runRecipeOnlyRecipeTypes.includes($scope.recipe.type), $scope)
        }

        $scope.saveRecipe().then(function() {
            $scope.currentSaveIsForAnImmediateRun = false;
            if (forced) {
                $scope.recipeWT1Event("recipe-run-start-forced");
                doIt();
            } else if ($scope.recipe.params && $scope.recipe.params.skipPrerunValidate) {
                $scope.recipeWT1Event("recipe-run-start-no-validation");
                doIt();
            } else {
                $scope.recipeWT1Event("recipe-run-start");
                $scope.hooks.preRunValidate().then(function(validationResult) {
                    if (validationResult.ok == true || validationResult.error == false || validationResult.allMessagesForFrontend && !validationResult.allMessagesForFrontend.error) {
                        $scope.recipeWT1Event("recipe-run-start-validated");
                        doIt();
                    } else {
                        $scope.startedJob.starting = false;
                        $scope.valCtx.preRunValidationError = validationResult;
                        $scope.recipeWT1Event("recipe-run-start-blocked", {
                            firstError : validationResult.allMessagesForFrontend && validationResult.allMessagesForFrontend.messages.length ? validationResult.allMessagesForFrontend.messages[0].message : "unknown"
                        });
                    }
                }, function(error) {
                    $scope.startedJob.starting = false;
                });
            }
        }, function(error) {
            $scope.currentSaveIsForAnImmediateRun = false;
            $scope.startedJob.starting = false;
        });
    };


    $scope.startContinuousActivity = function(forced) {
        $scope.hooks.resetScope();
        $scope.startedJob.starting = true;

        function doIt() {
            const onceLoopParams = { abortAfterCrashes: 0 };
            DataikuAPI.continuousActivities.start($stateParams.projectKey, $stateParams.recipeName, onceLoopParams).success(function(data){
                FutureProgressModal.show($scope, data, "Starting continuous recipe...").then(function(data) {
                    $scope.startedJob.jobId = data.futureId;
                    $scope.waitForEndOfStartedContinuousActivity();
                });
            }).error(setErrorInScope.bind($scope));
        }

        $scope.saveRecipe().then(function() {
            if (forced) {
                $scope.recipeWT1Event("recipe-run-start-forced");
                doIt();
            } else if ($scope.recipe.params && $scope.recipe.params.skipPrerunValidate) {
                $scope.recipeWT1Event("recipe-run-start-no-validation");
                doIt();
            } else {
                $scope.recipeWT1Event("recipe-run-start");
                $scope.hooks.preRunValidate().then(function(validationResult) {
                    if (validationResult.ok == true || validationResult.error == false || !validationResult.allMessagesForFrontend.error) {
                        $scope.recipeWT1Event("recipe-run-start-validated");
                        doIt();
                    } else {
                        $scope.startedJob.starting = false;
                        $scope.valCtx.preRunValidationError = validationResult;
                        $scope.recipeWT1Event("recipe-run-start-blocked", {
                            firstError : validationResult.allMessagesForFrontend && validationResult.allMessagesForFrontend.messages.length ? validationResult.allMessagesForFrontend.messages[0].message : "unknown"
                        });
                    }
                }, function(error) {
                    $scope.startedJob.starting = false;
                });
            }
        }, function(error) {
            $scope.startedJob.starting = false;
        });
    };

    $scope.stopContinuousActivity = function(){
        $scope.continuousActivityState = null;
        DataikuAPI.continuousActivities.stop($stateParams.projectKey, $stateParams.recipeName).success(function(data){
            // TODO - start displaying some useful stuff...
        }).error(setErrorInScope.bind($scope));
    }

    $scope.openContinuousActivity = function() {
        $state.go("projects.project.continuous-activities.continuous-activity.runs", {continuousActivityId: $scope.recipe.name});
    };

    // Stop the timer at exit
    $scope.$on("$destroy",function() {
       if($scope.jobCheckTimer) {
           $timeout.cancel($scope.jobCheckTimer);
           $scope.jobCheckTimer = null;
       }
       Mousetrap.unbind("@ r u n");
       Mousetrap.unbind("shift+enter");
       $scope.hooks = null;
    });

    Mousetrap.bind(['@ r u n','shift+enter'], function(){
        $scope.startSingleRecipeExecution();
    });

});


app.controller("_RecipeWithEngineBehavior", function($rootScope, $scope, $q, $stateParams, DataikuAPI, Dialogs, PartitionDeps, DKUtils, Logger, CreateModalFromTemplate, AnyLoc, ActivityIndicator, translate) {
    $scope.setRecipeStatus = function(data) {
        $scope.recipeStatus = data;

        const engineType = $scope.recipeStatus.selectedEngine && $scope.recipeStatus.selectedEngine.type;
        if (engineType === "SPARK") {
            $scope.anyPipelineTypeEnabled = function() {
                return $rootScope.projectSummary.sparkPipelinesEnabled;
            };
        } else if (engineType === "SQL") {
            $scope.anyPipelineTypeEnabled = function() {
                return $rootScope.projectSummary.sqlPipelinesEnabled;
            };
        }
    };

    $scope.hooks.updateRecipeStatus = function() {};

    var requestsInProgress = 0;
    var sendTime = 0;
    var lastSequenceId = 0;
    var lastPromise;
    // to avoid updating multiple times with same data:
    var lastPayload;
    var lastRequestData;
    var lastRecipeSerialized; //(json string)
    $scope.updateRecipeStatusBase = function(forceUpdate, payload, requestSettings) {
        var recipeCopy = angular.copy($scope.recipe);
        /* Complete the partition deps from the "fixedup" version */
        PartitionDeps.prepareRecipeForSerialize(recipeCopy);
        var recipeSerialized = angular.toJson(recipeCopy);
        var requestData = angular.toJson(requestSettings || {});

        if (!forceUpdate
            && lastPayload == payload
            && lastRequestData == requestData
            && lastRecipeSerialized == recipeSerialized) {
            Logger.info("Update recipe: cache hit, not requesting");
            // We already made this request
            return lastPromise;
        }

        lastPayload = payload;
        lastRequestData = requestData;
        lastRecipeSerialized = recipeSerialized;
        lastSequenceId++;

        requestsInProgress++;
        sendTime = new Date().getTime();
        $scope.recipeStateUpdateInProgress = true;
        lastPromise = DataikuAPI.flow.recipes.generic.getStatus(recipeCopy, payload, lastSequenceId, requestSettings)
            .catch(function(response) {
                setErrorInScope.bind($scope)(response.data, response.status, response.headers);
                //we can't get the sequenceId so wait for all answers to mark as idle
                if (requestsInProgress == 1) {
                    $scope.recipeStateUpdateInProgress = false;
                }
                return response;
            })
            .finally(function(){
                requestsInProgress--;
            })
            .then(function(response) {
                if (parseInt(response.data.sequenceId) < lastSequenceId) {
                    return; //Too late!
                }
                if (new Date().getTime() - sendTime > 1500) {
                    aPreviousCallWasLong = true;
                }
                $scope.recipeStateUpdateInProgress = false;
                $scope.setRecipeStatus(response.data);
                return response.data;
            });
        return lastPromise;
    };

    var timeout;
    $scope.updateRecipeStatusLater = function() {
        clearTimeout(timeout);
        timeout = setTimeout(function() {
            $('.CodeMirror').each(function(idx, el){Logger.debug(el.CodeMirror.refresh())});//Make sure codemirror is always refreshed (#6664 in particular)
            if (!$scope.hooks) return;
            $scope.hooks.updateRecipeStatus();
        }, 400);
    };

    /* this function helps the UI have a more appropriate look when status computation is long (small spinner, etc) */
    var aPreviousCallWasLong = false;
    $scope.expectLongRecipeStatusComputation = function() {
        return !$scope.recipeStatus || !$scope.recipeStatus.selectedEngine || aPreviousCallWasLong;
    };

    $scope.canChangeEngine = function() {
        if(!$scope.recipeStatus || !$scope.recipeStatus.engines) {
            return false;
        }
        if ($scope.isJobRunningOrStarting() || $scope.recipeStateUpdateInProgress) {
            return false;
        }
        return true;
    };

    $scope.convertToQueryRecipe = function(type, label) {
        Dialogs.confirm($scope, translate("RECIPES.CONVERT_TO", "Convert to " + label + " recipe", {recipeType: label}),
                        translate("RECIPES.CONVERTING_TO", "Converting the recipe to " + label + " will enable you to edit the query, but you will not be able to use the visual editor anymore.", {recipeType: label})
                        + "<br/><strong>" + translate("RECIPES.THIS_OPERATION_IS_IRREVERSIBLE", "This operation is irreversible.") + "</strong>")
        .then(function() {
            var payloadData = $scope.hooks.getPayloadData();
            var recipeSerialized = angular.copy($scope.recipe);
            PartitionDeps.prepareRecipeForSerialize(recipeSerialized);
            $scope.hooks.save().then(function() {
                DataikuAPI.flow.recipes.visual.convert($stateParams.projectKey, recipeSerialized, payloadData, type)
                .success(function(data) {
                    DKUtils.reloadState();
                }).error(setErrorInScope.bind($scope));
            });
        });
    };

    $scope.showSQLModal = function(){
        var newScope = $scope.$new();
        newScope.convert = $scope.convertToQueryRecipe;
        newScope.uiState = {currentTab: 'query'};
        $scope.hooks.updateRecipeStatus(false, true).then(function(){
            // get the latest values, not the ones of before the updatestatus call
        	newScope.query = $scope.recipeStatus.sql;
        	newScope.engine = $scope.recipeStatus.selectedEngine.type;
        	newScope.executionPlan = $scope.recipeStatus.executionPlan;
        	CreateModalFromTemplate("/templates/recipes/fragments/sql-modal.html", newScope);
        });
    };

    var save = $scope.baseSave;
    $scope.baseSave = function() {
        var p = save.apply(this, arguments);
        p.then($scope.updateRecipeStatusLater);
        return p;
    };

    $scope.$watchCollection("recipe.inputs.main.items", () => {
        //call updateRecipeStatus without args!
        Promise.resolve($scope.hooks.updateRecipeStatus()).catch(() => {
            Logger.info("Failed to updateRecipeStatus. Likely due to result of backend call discarded due to multiple parallel calls.");
        });
    });
    $scope.$watchCollection("recipe.outputs.main.items", () => {
        //call updateRecipeStatus without args!
        Promise.resolve($scope.hooks.updateRecipeStatus()).catch(() => {
            Logger.info("Failed to updateRecipeStatus. Likely due to result of backend call discarded due to multiple parallel calls.");
        });
    });

    $scope.$watch("params.engineParams", $scope.updateRecipeStatusLater, true);
});


app.controller("SqlModalController", function($scope, CodeMirrorSettingService) {

    $scope.editorOptions = CodeMirrorSettingService.get('text/x-sql2');

    // if ($scope.engine == 'HIVE' || $scope.engine == 'IMPALA' || $scope.engine == 'SPARK') {
    //     $scope.editorOptions.mode = 'text/x-hive';
    // }
});


app.directive("recipeEnginesPreferenceConfig", function(){
    return {
        restrict: 'A',
        templateUrl : '/templates/recipes/widgets/recipe-engines-preference-config.html',
        scope: {
            model: '='
        }
    }
});


app.service('RecipesEnginesService', function($rootScope, $q, Assert, CreateModalFromTemplate, DataikuAPI) {
    this.startChangeEngine = function(selectedItems) {
        return CreateModalFromTemplate("/templates/recipes/fragments/change-recipes-engines-modal.html", $rootScope, null, function(modalScope) {
            modalScope.selectedRecipes = selectedItems.filter(it => it.type == 'RECIPE');
            modalScope.options = {};
            modalScope.AUTO = '__AUTO__';

            modalScope.getEngineShortStatus = function(engine) {
                for(let msg of engine.messages.messages) {
                    if (msg.severity == "ERROR") {
                        return msg.details;
                    }
                }
                for(let msg of engine.messages.messages) {
                    if (msg.severity == "WARNING") {
                        return msg.details;
                    }
                }
            };

            DataikuAPI.flow.recipes.massActions.startChangeEngines(modalScope.selectedRecipes).success(function(data) {
                Assert.trueish(data.engines, 'no engines');
                modalScope.availableEngines = data.engines;
                modalScope.options.engine = data.currentEngine;
                modalScope.nUnselectableEngines = data.engines.filter(e => !e.isSelectable).length;
                modalScope.messages = data.messages;
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });


            modalScope.test = function() {
                const deferred = $q.defer();
                delete modalScope.messages;
                delete modalScope.maxSeverity;
                resetErrorInScope(modalScope);
                DataikuAPI.flow.recipes.massActions.testChangeEngines(modalScope.selectedRecipes, modalScope.options.engine).success(function(data) {
                    modalScope.messages = data.messages;
                    modalScope.maxSeverity = data.maxSeverity || 'OK';
                    if (modalScope.maxSeverity != 'OK') {
                        deferred.reject();
                    } else {
                        deferred.resolve(data)
                    }
                }).error(setErrorInScope.bind(modalScope));
                return deferred.promise;
            };

            modalScope.ok = function(force) {
                if (force || modalScope.options.engine == modalScope.AUTO) { //No need to test AUTO
                    performChange();
                } else {
                    modalScope.test().then(performChange);
                }
            };

            function performChange() {
                DataikuAPI.flow.recipes.massActions.changeEngines(modalScope.selectedRecipes, modalScope.options.engine).success(function(data) {
                    modalScope.resolveModal();
                }).error(setErrorInScope.bind(modalScope));
            }
        });
    };
});


    app.directive("codeEnvSelectionForm", function(DataikuAPI, $stateParams, translate) {
    return {
        restrict: 'A',
        templateUrl : '/templates/recipes/fragments/code-env-selection-form.html',
        scope: {
            envSelection: '=codeEnvSelectionForm',
            inPlugin: '=',
            isStep: '=',
            envLang: '=',
            selectionLabel: '=',
            callback: '=?',
            helpLabel: '=',
            checkCodeEnv: '=?',
            readOnly: '<',
        },
        link: function($scope, element, attrs) {
            $scope.customWarningMessage = null;
            if ($scope.inPlugin == true) {
                $scope.envModes = [
                    ['USE_BUILTIN_MODE', translate('SCENARIO.SETTINGS.CODE_ENV.IN_PLUGIN.USE_BUILTIN_MODE', 'Use plugin environment')],
                    ['EXPLICIT_ENV', translate('SCENARIO.SETTINGS.CODE_ENV.IN_PLUGIN.EXPLICIT_ENV', 'Select an environment')],
                ];
            } else {
                $scope.envModes = [
                    ['USE_BUILTIN_MODE', translate('SCENARIO.SETTINGS.CODE_ENV.USE_BUILTIN_MODE', 'Use DSS builtin env')],
                    ['INHERIT', translate('SCENARIO.SETTINGS.CODE_ENV.INHERIT', 'Inherit project default')],
                    ['EXPLICIT_ENV', translate('SCENARIO.SETTINGS.CODE_ENV.EXPLICIT_ENV', 'Select an environment')]
                ];
            }

            function setDefaultValue() {
                if (!$scope.envSelection) { // not ready
                    return;
                }
                if ($scope.envSelection.envMode == "EXPLICIT_ENV" && $scope.envSelection.envName == null && $scope.envNamesWithDescs && $scope.envNamesWithDescs.envs && $scope.envNamesWithDescs.envs.length > 0) {
                    $scope.envSelection.envName = $scope.envNamesWithDescs.envs[0].envName;
                }
            }
            $scope.$watch("envSelection.envMode", setDefaultValue);

            $scope.envNamesWithDescs = [];
            DataikuAPI.codeenvs.listNamesWithDefault($scope.envLang, $stateParams.projectKey).success(function(data) {
                $scope.envNamesWithDescs = data;
                data.envs.forEach(function(x) {
                    if (x.owner) {
                        x.envDesc = x.envName + " (" + x.owner + ")";
                    } else {
                        x.envDesc = x.envName;
                    }
                });
                if (!$scope.inPlugin) {
                    if (data.resolvedInheritDefault == null) {
                        $scope.envModes[1][1] = translate('SCENARIO.SETTINGS.CODE_ENV.INHERIT_DSS_BUILTIN', "Inherit project default (DSS builtin env)")
                    } else {
                        $scope.envModes[1][1] = translate('SCENARIO.SETTINGS.CODE_ENV.INHERIT_CUSTOM', "Inherit project default ({{resolvedInheritDefault}})", { resolvedInheritDefault: data.resolvedInheritDefault });
                        $scope.inheritedEnv = data.envs.filter(e=>e.envName === data.resolvedInheritDefault)[0];
                    }
                }
                setDefaultValue();
            }).error(setErrorInScope.bind($scope));
            function setSelectedEnv() {
                if ($scope.envNamesWithDescs.envs && $scope.envSelection) {
                    $scope.selectedEnv = $scope.envNamesWithDescs.envs.filter(e=>e.envName === $scope.envSelection.envName)[0];
                }
            }
            $scope.$watchGroup(["envSelection.envName", "envNamesWithDescs.envs"], setSelectedEnv);
            if (typeof $scope.callback === 'function') {
                $scope.callback($scope);
            }
            if (typeof $scope.checkCodeEnv === 'function') {
                $scope.$watch("envSelection", function(envSelection) {
                    $scope.customWarningMessage = $scope.checkCodeEnv(envSelection);
                }, true);
            }
        }
    }
});

app.directive("containerSelectionForm", function(DataikuAPI, $stateParams){
        return {
            restrict: 'A',
            templateUrl : '/templates/recipes/fragments/container-selection-form.html',
            scope: {
                containerSelection: '=containerSelectionForm',
                selectionLabel: '=',
                inPlugin: '=',
                workloadType: '=',
                hideInherit: '=',
                inheritedFrom: '<',
                readOnly: '<',
                inlineHelp: '<',
                containerSpecificContext: '=?',
            },
            link: {
                post: function($scope, element, attrs) {
                    if (!$scope.inheritedFrom) {
                        $scope.inheritedFrom = 'project'
                    }

                    if ($scope.hideInherit) {
                        $scope.containerModes = [
                            ['NONE', 'None - Use backend to execute'],
                            ['EXPLICIT_CONTAINER', 'Select a container configuration'],
                        ];
                    } else {
                        $scope.containerModes = [
                            ['NONE', 'None - Use backend to execute'],
                            ['INHERIT', 'Inherit ' + $scope.inheritedFrom +' default'],
                            ['EXPLICIT_CONTAINER', 'Select a container configuration'],
                        ];
                    }

                    $scope.containerNames = [];
                    let workloadType = $scope.workloadType || "USER_CODE";
                    if ($stateParams.projectKey) {
                        DataikuAPI.containers.listNamesWithDefault($stateParams.projectKey, null, workloadType, $scope.containerSpecificContext).success(function(data) {
                            $scope.containerNames = data.containerNames;
                            if (data.resolvedInheritValue) {
                                $scope.containerModes[1][1] += ' (' + data.resolvedInheritValue + ')';
                            } else {
                                $scope.containerModes[1][1] += ' (local execution)';
                            }
                        }).error(setErrorInScope.bind($scope));
                    } else {
                        DataikuAPI.containers.listNames(null, workloadType).success(function(data) {
                            $scope.containerNames = data;
                        }).error(setErrorInScope.bind($scope));
                    }
                }
            }



        }
    });

/**
 * Inputs for specifying the container configuration to apply in the context of the hyperparameter search.
 * @param {object} searchParams: the parameters of the hyperparameter search (either from an analysis or a recipe)
 * @param {function} hasSelectedK8sContainer: tells whether the user has selected a k8s container to run the search
 * @param {boolean} isTimeseriesForecasting: used to determine whether we show a warning for non reproductible search
 */
app.component('mlHpDistribution', {
     templateUrl : '/templates/recipes/fragments/ml-hp-distribution.html',
     bindings: {
         searchParams: '=',
         hasSelectedK8sContainer: '<',
         k8sRuntimeEnvTooltip: '@?',
         isTimeseriesForecasting: '<'
     },
     controller: function() {
         const $ctrl = this;

         $ctrl.getK8sRuntimeEnvTooltip = () => {
             if ($ctrl.k8sRuntimeEnvTooltip) {
                 return $ctrl.k8sRuntimeEnvTooltip;
             }

             return  "Distributed search requires a Kubernetes container configuration to be selected";
         };

         $ctrl.showHpSearchNotReproducibleWarning = () => {
            return $ctrl.isTimeseriesForecasting && $ctrl.searchParams.nJobs !== 1;
        };

     },
});

app.controller("_ContinuousRecipeInitStartedJobBehavior", function ($scope, $stateParams, DataikuAPI, Logger) {
    // get the current state
    DataikuAPI.continuousActivities.getState($stateParams.projectKey, $stateParams.recipeName).success(function(data) {
        $scope.startedJob = $scope.startedJob || {};
        $scope.startedJob.persistent = data;
        $scope.startedJob.current = data.mainLoopState;
        if (data.mainLoopState) {
            if (!data.mainLoopState.futureInfo || !data.mainLoopState.futureInfo.hasResult) {
                $scope.startedJob.jobId = data.mainLoopState.futureId;
                $scope.waitForEndOfStartedContinuousActivity();
            }
        }
    }).error(function() {
        Logger.warn("Recipe " + $stateParams.recipeName + " doesn't have a continuous activity yet")
    });

});

/**
 * Show the interface to chose options for some aggregations in Group, Pivot and Window visual recipes
 * Different options can be shown of not depending on whether they make sense for that recipe.
 * @param {boolean} showOrderBy controls whether the 'Order First/Last By' option is shown in the popover
 * @param {boolean} showFirstLastNotNull controls whether the 'First/Last not null' option is shown in the popover
 * @param {function} orderByColumnsGetter a function returning the list of valid columns that can be chosen in the order by option. Required if and only if showOrderBy is true
 * @param {OnBeforeUnloadEventHandlerNonNull} engineCanAggrFirstNotNull disables the 'First/Last not null' option with a popup explaining that the engine is limiting. Required if and only if showFirstLastNotNull is true
 * @param {OnBeforeUnloadEventHandlerNonNull} engineCanAggrConcatDistinct same with the 'Concat distinct' option
 * @param {object} column the colums on which the aggregations are being defined
 */
app.component('aggregationOptionsPopover', {
    templateUrl: '/templates/recipes/fragments/aggregation-options-popover.html',
    bindings: {
        showOrderBy: '<',
        orderByColumnsGetter: '<?', // required if and only if showOrderBy is true
        showFirstLastNotNull: '<',
        engineCanAggrFirstNotNull: '<?', // required if and only if showFirstLastNotNull is true
        engineCanAggrConcatDistinct: '<',
        column: '=',
    },
    controller: function(translate) {
        const $ctrl = this;

        $ctrl.translate = translate;
    },
});

})();
