(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * Initialize a chart implemented in d3.
     * (!) This service previously was in static/dataiku/js/simple_report/chart_view_commons.js
     */
    app.factory('ChartManager', function(ChartDimension, ChartTooltips, ChartContextualMenu, AnimatedChartsUtils, $timeout, ChartLegendUtils,
        ChartFormatting, ChartColorScales, D3ChartAxes, ChartStoreFactory, ChartAxesUtils, ChartColorUtils, CHART_AXIS_TYPES) {

        const svc = {};

        /**
         * Create svg(s) in $container for the given chart
         * @param $container
         * @param chartData
         * @param chartDef
         * @return {*|jQuery|HTMLElement}
         */
        const createSVGs = function($container, chartData, chartDef, isInteractiveChart, ignoreLabels = new Set()) {
            $container.find('.mainzone').remove();
            const mainzone = $('<div class="mainzone" data-qa-screenshot-scope__chart>').prependTo($container);

            if (isInteractiveChart && chartDef.linesZoomOptions.displayBrush && chartDef.linesZoomOptions.enabled) {
                if (chartDef.legendPlacement === 'OUTER_TOP' || chartDef.legendPlacement === 'OUTER_BOTTOM') {
                    mainzone.addClass('mainzone--with-brush-and-horizontal-legend');
                } else {
                    mainzone.addClass('mainzone--with-brush');
                }
            }

            const $chartsContainer = $('<div class="charts">').appendTo(mainzone),
                isFacetted = chartData.axesDef.facet != undefined,
                facetLabels = chartData.getAxisLabels('facet', ignoreLabels) || [null];


            if (isFacetted) {
                $chartsContainer.addClass('facetted');
                $chartsContainer.height(facetLabels.length * (1 + chartDef.chartHeight) - 1);
            }

            let $svgs = $();

            const $chartsTable = $('<div class="charts-table">').appendTo($chartsContainer);

            facetLabels.forEach(function(facetLabel) {
                const $div = $('<div class="chart">');

                $div.appendTo($chartsTable);
                if (facetLabel) {
                    const $facetInfo = $('<div class="facet-info">').appendTo($div);
                    $('<h2>').text(ChartFormatting.getForOrdinalAxis(facetLabel.label)).appendTo($facetInfo);
                }

                const $wrapper = $('<div class="chart-wrapper">').appendTo($div);
                if (isFacetted) {
                    $wrapper.css('height', chartDef.chartHeight);
                }

                $svgs = $svgs.add($('<svg style="width: 100%; height: 100%;" class="chart-svg">').appendTo($wrapper));
            });

            return $svgs;
        };

        /**
         * Do all the initial setup surrounding the actual drawing area of a chart, this includes:
         *     - Create scales (x, y, y2, color)
         *     - Draw color legend
         *     - Adjust margins based on axis sizes
         *     - Create svgs (only one if chart is not facetted)
         *     - Draw axes
         *     - Create measure formatters
         *     - Initialize tooltips
         *
         * @param {ChartDef.java}                               chartDef
         * @param {a $scope object}                             chartHandler
         * @param {ChartTensorDataWrapper}                      chartData
         * @param {jQuery}                                      $container
         * @param {function}                                    drawFrame           a function(frameIdx, chartBase) that will be called every time a new animation frame is requested (only once if the chart has no animation dimension) to draw the actual chart
         * @param {[axisId: string]: AxisSpec}                  axisSpecs           an object containing axisSpec for all the axes
         * @param {AxisSpec}                                    colorSpec           an AxisSpec object for the color axis (nullable)
         * @param {Function}                                    [handleZoom]        an optional function to setup zoom
         * @param {Object}                                      [zoomUtils]         an optional object containing zoom information
         * AxisSpec:
         *     - type ('DIMENSION', 'MEASURE' or 'UNAGGREGATED') : whether the axis represents a dimension, one (or several) measures, or an unaggregated column
         *
         *     DIMENSION only attributes:
         *     - name : the name of the corresponding dimension as set in chartData
         *     - mode (either 'POINTS' or 'COLUMNS') : whether we sould use rangeRoundPoints or rangeRoundBands for the d3 ordinal scale
         *
         *     MEASURE only attributes:
         *     - measureIdx : the index of the measure if it is the axis only shows one measure
         *     - extent : the extent of the axis, will default to the extent of measure measureIdx if not provided
         *     - values : the list of values for the measure. Used for color spec to compute quantile scales.
         *
         *     UNAGGREGATED only attributes:
         *     - data : the data for this column (ScatterAxis.java object)
         *
         *     COMMON
         *     - dimension : the DimensionDef.java/NADimensionDef.java object for this column
         *
         *
         *     - withRgba (for ColorSpec only) : weather the color scale should include the chart's transparency setting with rgba or not
         *
         * @returns {ChartBase} : a ChartBase object, with the following properties:
         *      - $svgs {jQuery} $svgs,
         *      - colorScale {*} : a scale instance as returned by ChartColorScales.createColorScale
         *      - margins {{top: number, bottom: number, left: number, right: number}} the final, adjusted margins of the chart
         *      - vizWidth {number} : the width of the drawing area (full container width - legend with - margin width)
         *      - vizHeight {number} : the height of the drawing area
         *      - xAxis {d3 axis}
         *      - yAxes {d3 axis[]} (nullable)
         *      - tooltips {*} : a tooltip instance as returned by ChartTooltips.create
         *      - measureFormatters {array} : a list of measure formatters for all chart measures
         *      - isPercentChart {boolean} : is the chart a '*_100' variant
         *      - chartData {ChartTensorDataWrapper} : the same chartData that was given as input (for convenience)
         *
         */
        svc.initChart = function(chartDef, chartHandler, chartData, $container, drawFrame, axisSpecs, colorSpec, zoomContext, initLinesZoom, ignoreLabels = new Set()) {
            // Store the axis specs so they can be reused (for example in the axis configuration menus).
            const chartStoreMeta = ChartStoreFactory.getOrCreate(chartDef.$chartStoreId);
            chartDef.$chartStoreId = chartStoreMeta.id;

            chartStoreMeta.store.setAxisSpecs(axisSpecs);
            //  Retrieve axis specs in chart def for future migration (used for formatting pane)
            chartDef.$axisSpecs = axisSpecs;
            let chartBase = {};

            // Create color scale
            /** type: ChartColorContext */
            const colorContext = {
                chartData,
                colorOptions: ChartColorUtils.getChartColorOptions(chartDef),
                defaultLegendDimension: ChartColorUtils.getDefaultLegendDimension(chartDef),
                colorSpec,
                chartHandler,
                ignoreLabels,
                theme: chartHandler.getChartTheme()
            };
            const colorScale = ChartColorScales.createColorScale(colorContext);

            // Create axis
            chartBase.isPercentChart = chartDef.variant && chartDef.variant.endsWith('_100');

            const ySpecs = ChartAxesUtils.getYAxisSpecs(axisSpecs);

            const xAxis = D3ChartAxes.createAxis(
                chartData,
                axisSpecs?.['x'],
                chartBase.isPercentChart,
                chartDef.xAxisFormatting.isLogScale,
                undefined,
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting,
                chartDef,
                ignoreLabels,
                (axisSpecs?.['x']?.type === CHART_AXIS_TYPES.MEASURE) && Object.keys(ySpecs || []).map((_, i) => i+1) // y axes indexes
            );

            const yAxes = [];
            ySpecs && Object.keys(ySpecs).forEach(key => {
                const spec = ySpecs[key];
                const includeZero = ChartAxesUtils.shouldIncludeZero(chartDef, key);
                const yAxis = D3ChartAxes.createAxis(
                    chartData,
                    spec,
                    chartBase.isPercentChart,
                    ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, spec.id),
                    includeZero,
                    ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, spec.id),
                    chartDef,
                    ignoreLabels,
                    (spec?.type === CHART_AXIS_TYPES.MEASURE) && [0]
                );
                if (yAxis) {
                    yAxes.push(yAxis);
                }
            });

            ChartLegendUtils.createLegend($container, chartDef, chartData, chartHandler, colorSpec, colorScale, undefined, ignoreLabels).then(function() {
                $timeout(() => {
                    const $svgs = createSVGs($container, chartData, chartDef, zoomContext && ChartDimension.isInteractiveChart(chartDef), ignoreLabels);

                    if ($container.data('previous-tooltip')) {
                        $container.data('previous-tooltip').destroy();
                    }

                    const yAxesOptions = [];
                    yAxes.forEach(axis => {
                        yAxesOptions.push({
                            id: axis.id,
                            tickValues: axis && axis.tickValues(),
                            tickFormat: axis && axis.tickFormat()
                        });
                    });

                    // Draw axes in every svg
                    const axisOptions = {
                        x: {
                            tickValues: xAxis && xAxis.tickValues(),
                            tickFormat: xAxis && xAxis.tickFormat()
                        },
                        y: yAxesOptions
                    };

                    const yAxesColors = chartData.getYAxesColors(ySpecs, chartDef, chartHandler.getChartTheme());
                    const d3DrawContext = {
                        $svgs,
                        chartDef,
                        chartHandler,
                        ySpecs,
                        xAxis,
                        yAxes,
                        axisOptions,
                        yAxesColors
                    };
                    const { margins, vizWidth, vizHeight } = D3ChartAxes.drawAxes(d3DrawContext);

                    /*
                     * If the legend placement was INNER_TOP_LEFT or INNER_BOTTOM_*, its position depends on the margins (to not overlap with the axes)
                     * Now that all axes have been positioned, we can adjust its placement
                     */
                    ChartLegendUtils.adjustLegendPlacement(chartDef, $container, margins);

                    const measureFormatters = ChartFormatting.createMeasureFormatters(chartDef, chartData, Math.max(vizHeight, vizWidth));

                    const tooltips = ChartTooltips.create($container, chartHandler, chartData, chartDef, measureFormatters, undefined, ignoreLabels);

                    const { store, id } = ChartStoreFactory.getOrCreate(chartDef.$chartStoreId);
                    chartDef.$chartStoreId = id;
                    const contextualMenu = ChartContextualMenu.create(chartData, chartDef, store, chartHandler.animation);


                    // Everything that the chart might need
                    chartBase = {
                        $svgs: $svgs,
                        colorScale: colorScale,
                        margins: margins,
                        vizWidth: vizWidth,
                        vizHeight: vizHeight,
                        xAxis: xAxis,
                        yAxes: yAxes,
                        axisOptions,
                        tooltips: tooltips,
                        contextualMenu,
                        measureFormatters: measureFormatters,
                        chartData: chartData,
                        xSpec: axisSpecs && axisSpecs['x'],
                        ySpecs: ySpecs
                    };

                    $container.data('previous-tooltip', tooltips);

                    angular.extend(chartHandler.tooltips, tooltips);

                    if (chartData.axesDef.animation != undefined) {
                        AnimatedChartsUtils.initAnimation(chartHandler, chartData, chartDef, function(frameIdx) {
                            chartBase.tooltips.setAnimationFrame(frameIdx);
                            return drawFrame(frameIdx, chartBase);
                        }, ignoreLabels);
                    } else {
                        AnimatedChartsUtils.unregisterAnimation(chartHandler);
                    }

                    if (zoomContext && initLinesZoom) {
                        zoomContext = { ...zoomContext, chartBase, svgs: chartBase.$svgs, xAxis };
                        chartDef.$zoomControlInstanceId = initLinesZoom(zoomContext);
                    }

                    $svgs.on('click', function(evt) {
                        if (evt.target.hasAttribute('data-legend') || evt.target.hasAttribute('tooltip-el')) {
                            return;
                        }
                        chartBase.tooltips.resetColors();
                        chartBase.tooltips.unfix();
                    });

                    if (chartData.axesDef.animation != undefined) {
                        // Draw first frame
                        chartHandler.animation.drawFrame(chartHandler.animation.currentFrame || 0, chartBase);
                        if (chartHandler.autoPlayAnimation) {
                            chartHandler.animation.play();
                        }
                    } else {
                        drawFrame(0, chartBase);
                    }
                    /*
                     * Signal to the callee handler that the chart has been loaded.
                     * Dashboards use it to determine when all insights are completely loaded.
                     */
                    if (typeof(chartHandler.loadedCallback) === 'function') {
                        chartHandler.loadedCallback();
                    }
                }, 100);
            });

            return chartBase;
        };

        return svc;
    });

})();
