(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('StackedColumnsChart', StackedColumnsChart);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/stacked-columns.js
     */
    function StackedColumnsChart(ChartManager, ChartDimension, ChartFeatures, ChartDataWrapperFactory, Fn, StackedChartUtils, BarChartUtils, SVGUtils, ChartAxesUtils, ReferenceLines, ColumnAvailability, ChartYAxisPosition, ChartCustomMeasures, ChartUsableColumns, ColorFocusHandler, ValuesInChartOverlappingStrategy, ChartDataUtils) {
        return function($container, chartDef, chartHandler, axesDef, data) {
            const backgroundXMargin = 4;

            const isPercentScale = ChartDimension.isPercentScale(chartDef.genericMeasures) || chartDef.variant == 'stacked_100';
            const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef);
            const yAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);

            const dataSpec = chartHandler.getDataSpec();
            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
            const xDimension = chartDef.genericDimension0[0];

            const xSpec = { type: 'DIMENSION', mode: 'COLUMNS', dimension: xDimension, name: 'x', customExtent: chartDef.xAxisFormatting.customExtent };
            ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);

            const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, xSpec, undefined),
                referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures),
                referenceLinesExtents = ReferenceLines.getReferenceLinesExtents(displayedReferenceLines, referenceLinesValues, { [yAxisID]: { isPercentScale } });

            ReferenceLines.mutateDimensionSpecForReferenceLine(ChartDataUtils.getAxisExtent(chartData, xSpec.name, xSpec.dimension), referenceLinesExtents.x, xSpec);

            const xLabels = chartData.getAxisLabels('x'),
                animationData = StackedChartUtils.prepareData(chartDef, chartData),
                yExtent = ReferenceLines.getExtentWithReferenceLines([animationData.minTotal, animationData.maxTotal], referenceLinesExtents[yAxisID]);

            const drawFrame = function(frameIdx, chartBase) {
                ReferenceLines.removeReferenceLines($container[0]);

                animationData.frames[frameIdx].facets.forEach(function(facetData, f) {
                    const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                    StackedColumnChartDrawer(g, facetData, chartBase);
                });
            };

            // Hack: by design stacked columns start at 0, but if log scale is checked start at 1 (to make it possible)
            if (ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting)) {
                yExtent[0] = 1;
            }

            const ySpecs = { [yAxisID]: { id: yAxisID, type: 'MEASURE', extent: yExtent, isPercentScale, customExtent: ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, yAxisID), position: ChartYAxisPosition.LEFT } };
            const axisSpecs = { x: xSpec, ...ySpecs };

            ChartAxesUtils.setNumberOfBinsToDimensions(chartData, chartDef, axisSpecs);
            ReferenceLines.updateAvailableAxisOptions([
                { axis: 'LEFT_Y_AXIS', isDisplayed: true, isNumerical: true, isPercentScale },
                { axis: 'X_AXIS', isDisplayed: true, isNumerical: ChartAxesUtils.isNumerical(xSpec) && !ChartDimension.hasOneTickPerBin(xSpec.dimension), isContinuousDate: ChartAxesUtils.isContinuousDate(xSpec) }
            ]);

            ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                axisSpecs,
                { type: 'DIMENSION', name: 'color', dimension: chartDef.genericDimension1[0] });

            function StackedColumnChartDrawer(g, stacksData, chartBase) {

                const xAxis = chartBase.xAxis,
                    yAxis = chartBase.yAxes[0],
                    yScale = yAxis.scale(),
                    percentFormatter = d3.format('.0%');

                let barWidth = ChartDimension.isUngroupedNumerical(xDimension) ? 10 : Math.max(1, xAxis.ordinalScale.rangeBand());
                // Adjust barWidth if there is a custom extent, (mainly to prevent bars from overlapping each others when extent is reduced)
                barWidth = Math.max(barWidth * ChartAxesUtils.getCustomExtentRatio(chartDef.xAxisFormatting.customExtent), 1);

                // Wrapper to contain all rectangles (used to apply clip-path)
                let wrapper = g.selectAll('g.stack-wrapper');
                if (wrapper.empty()) {
                    g.append('g').attr('class', 'stack-wrapper');
                    wrapper = g.selectAll('g.stack-wrapper');
                }

                const stacks = wrapper.selectAll('.stack-bars').data(stacksData.stacks);
                stacks.enter().append('g').attr('class', 'stack-bars');
                stacks.exit().remove();
                stacks.attr('transform', BarChartUtils.translate('x', xAxis, xLabels, barWidth));

                const rects = stacks.selectAll('rect').data(Fn.prop('data'));
                const getFinalColorScaleIndex = (d, i) => i;

                rects.enter().append('rect')
                    .attr('width', barWidth)
                    .attr('x', 0)
                    .attr('fill', function(d, i) {
                        return chartBase.colorScale(getFinalColorScaleIndex(d, i));
                    })
                    .attr('opacity', chartDef.colorOptions.transparency)
                    .each(function(d) {
                        chartBase.tooltips.registerEl(this, { measure: d.measure, x: d.x, color: d.color, facet: d.facet }, 'fill');
                        chartBase.contextualMenu.addContextualMenuHandler(this, { x: d.x, color: d.color, facet: d.facet });
                    });

                if (chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayValues) {

                    // init the collision detection
                    const labelCollisionDetectionHandler = SVGUtils.initLabelCollisionDetection(chartBase);

                    const getLabelXPosition = () => {
                        return barWidth / 2;
                    };

                    const commonDrawContext = {
                        node: wrapper,
                        axisName: 'x',
                        axis: xAxis,
                        labels: xLabels,
                        thickness: barWidth,
                        opacity: chartDef.colorOptions.transparency,
                        colorScale: chartBase.colorScale,
                        getLabelXPosition,
                        getBackgroundXPosition: (d, i) => {
                            return getLabelXPosition(d, i) - d.width / 2 - backgroundXMargin;
                        },
                        backgroundXMargin
                    };

                    const drawValuesLabels = () => {
                        const getLabelYPosition = function(d) {
                            return yScale((d.top + d.base) / 2) - (d.height / 2);
                        };

                        const getLabelText = (d) => {
                            if (BarChartUtils.shouldDisplayBarLabel(d.count, d.value)) {
                                if (chartDef.variant === 'stacked_100') {
                                    return percentFormatter(d.value);
                                } else {
                                    return chartBase.measureFormatters[d.aggregationIndex](chartData.aggr(d.aggregationIndex).get(d));
                                }
                            } else {
                                return '';
                            }
                        };

                        const overlappingStrategyForValues = [
                            ValuesInChartOverlappingStrategy.AUTO_WITH_VALUES,
                            ValuesInChartOverlappingStrategy.AUTO
                        ].includes(chartDef.valuesInChartDisplayOptions.overlappingStrategy)
                            ? ValuesInChartOverlappingStrategy.AUTO : ValuesInChartOverlappingStrategy.ALL;
                        const labelCollisionDetectionHandlersByColorScaleIndex = {};

                        const labelsDrawContext = {
                            ...commonDrawContext,
                            data: stacksData.stacks.map(stack => stack.data),
                            getLabelYPosition,
                            getExtraData: function(d, i, n, onHoverMode) {
                                const extraData = BarChartUtils.getExtraData(d, i, this, chartBase, labelCollisionDetectionHandler, labelCollisionDetectionHandlersByColorScaleIndex, getFinalColorScaleIndex, getLabelXPosition, getLabelYPosition, overlappingStrategyForValues, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize, onHoverMode);
                                return { labelPosition: extraData.labelPosition };
                            },
                            getLabelText,
                            getBackgroundYPosition: (d) => {
                                const y = getLabelYPosition(d);
                                return SVGUtils.getSubTextYPosition(d, y);
                            },
                            hasBackground: true,
                            overlappingStrategy: overlappingStrategyForValues,
                            theme: chartHandler.getChartTheme()
                        };

                        BarChartUtils.drawLabels(labelsDrawContext);
                    };

                    const drawTotalsLabels = () => {
                        if (ChartFeatures.shouldDisplayTotalValues(chartDef)) {

                            // specifc overrides for totals
                            const getLabelText = (d) => {
                                // aggregationIndex is not defined if no custom value
                                const text = d.aggregationIndex === undefined
                                    ? chartBase.measureFormatters[0](d.total)
                                    : chartBase.measureFormatters[d.aggregationIndex](d.aggregationTotal);
                                return BarChartUtils.shouldDisplayBarLabel(d.count, d.total) ? text : '';
                            };

                            const getLabelYPosition = (d) => {
                                const spacing = d.spacing || 5;
                                if (d.total < 0) {
                                    // for negative totals, the label is below the bar
                                    return yScale(d.negativeTotal) + spacing;
                                } else {
                                    return yScale(d.positiveTotal) - spacing - d.height;
                                }
                            };

                            const overlappingStrategyForTotals = [
                                ValuesInChartOverlappingStrategy.AUTO_WITH_TOTALS,
                                ValuesInChartOverlappingStrategy.AUTO
                            ].includes(chartDef.valuesInChartDisplayOptions.overlappingStrategy)
                                ? ValuesInChartOverlappingStrategy.AUTO : ValuesInChartOverlappingStrategy.ALL;
                            const labelCollisionDetectionHandlersByColorScaleIndex = {};
                            const totalLabelsDrawContext = {
                                ...commonDrawContext,
                                data: stacksData.stacks,
                                getLabelYPosition,
                                getExtraData: function(d, i, n, onHoverMode) {
                                    const extraData = BarChartUtils.getExtraData(d, i, this, chartBase, labelCollisionDetectionHandler, labelCollisionDetectionHandlersByColorScaleIndex, getFinalColorScaleIndex, getLabelXPosition, getLabelYPosition, overlappingStrategyForTotals, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize, onHoverMode);
                                    return { labelPosition: extraData.labelPosition };
                                },
                                getBackgroundYPosition: (d) => {
                                    const y = getLabelYPosition(d);
                                    return SVGUtils.getSubTextYPosition(d, y);
                                },
                                getLabelText,
                                hasBackground: false,
                                isTotals: true,
                                overlappingStrategy: overlappingStrategyForTotals,
                                theme: chartHandler.getChartTheme()
                            };
                            BarChartUtils.drawLabels(totalLabelsDrawContext);
                        }
                    };

                    if (chartDef.valuesInChartDisplayOptions.overlappingStrategy === ValuesInChartOverlappingStrategy.AUTO_WITH_VALUES) {
                        // in that case we want to display all totals no matter what, so we display them first, to make them priority over values
                        drawTotalsLabels();
                        drawValuesLabels();
                    } else {
                        drawValuesLabels();
                        drawTotalsLabels();
                    }
                }

                rects.transition()
                    .attr('height', function(d) {
                        return Math.max(0, Math.abs(yScale(d.base) - yScale(d.top)));
                    }) // height cannot be negative
                    .attr('y', function(d) {
                        return yScale(Math.max(d.base, d.top));
                    });

                // Clip paths to prevent bars from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
                ChartAxesUtils.isCroppedChart(chartDef) && SVGUtils.clipPaths(chartBase, g, wrapper);

                ReferenceLines.drawReferenceLines(
                    wrapper,
                    chartBase.vizWidth,
                    chartBase.vizHeight,
                    xAxis,
                    yAxis ? [{ ...yAxis, isPercentScale, formattingOptions: ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, yAxis.id) }] : [],
                    displayedReferenceLines,
                    referenceLinesValues
                );

                const colorFocusHandler = ColorFocusHandler.create(chartDef, chartHandler, wrapper, d3, g);
                ColorFocusHandler.appendFocusUnfocusMecanismToLegend(chartHandler.legendsWrapper.getLegend(0), colorFocusHandler);
            }
        };
    }

})();
