(function() {
'use strict';

const app = angular.module('dataiku.recipes');


app.service("LLMEvaluationMetrics", function() {
    const svc = {};

    svc.metrics = [
        {"label": "Answer relevancy", "name": "answerRelevancy", "shortDescription": "Focuses on assessing how pertinent the generated answer is to the given prompt. This metric is computed using the question, the context and the answer. It requires an embedding LLM and a completion LLM.", 'selectable': true},
        {"label": "Multimodal relevancy", "name": "multimodalRelevancy", "shortDescription": "Measures the relevance of the generated answer against both visual and textual context. The metric is computed using the question, the context and the answer. It requires an embedding LLM and a completion LLM.", 'promptRecipeFormatNeeded': true, 'selectable': true},
        {"label": "Answer correctness", "name": "answerCorrectness", "shortDescription": "Measures the accuracy of the answer compared to the ground truth. This metric is computed using the ground truth and the answer. It requires an embedding LLM and a completion LLM.", 'selectable': true},
        {"label": "Answer similarity", "name": "answerSimilarity", "shortDescription": "Assesses the semantic resemblance between the generated answer and the ground truth. The metric is computed using the ground truth and the answer. It requires an embedding LLM and a completion LLM.", 'selectable': true},
        {"label": "Context recall", "name": "contextRecall", "shortDescription": "Measures the extent to which the retrieved context aligns with the ground truth. The metric is computed using the ground truth and the context. It requires an embedding LLM and a completion LLM.", 'selectable': true},
        {"label": "Context precision", "name": "contextPrecision", "shortDescription": "Evaluates that the facts present in the context are ranked first by the ones most relevant to the ground-truth. This metric is computed using the question, ground truth and the context. It requires an embedding LLM and a completion LLM.", 'selectable': true},
        {"label": "Faithfulness", "name": "faithfulness", "shortDescription": "Measures the factual consistency of the generated answer against the given context. The metric is computed using the answer and retrieved context. It requires an embedding LLM and a completion LLM.", 'selectable': true},
        {"label": "Multimodal faithfulness", "name": "multimodalFaithfulness", "shortDescription": "Measures the factual consistency of the generated answer against both visual and textual context. The metric is computed using the answer and the retrieved context. It requires an embedding LLM and a completion LLM.", 'promptRecipeFormatNeeded': true, 'selectable': true},
        {"label": "BERT Score", "name": "bertScore", "shortDescription": "Computes the similarity of answer and ground truth(s) using their tokens' embeddings. The metric is computed using the answer and the ground truth.", 'selectable': true},
        {"label": "ROUGE", "name": "rouge", "shortDescription": "Measures the recall of word n-grams and longest common sequences for text summarization tasks. The metric is computed using the generated summary and a reference summary.", 'selectable': true},
        {"label": "BLEU", "name": "bleu", "shortDescription": "Measures the precision of word n-grams between generated and reference texts for machine translation tasks. The metric is computed using the generated translation and a reference translation.", 'selectable': true},
        {"label": "Avg Input tokens", "name": "inputTokensPerRow", "shortDescription": "The number of tokens that were ingested by the LLM. Average per row.", "labelRowByRow": "Input Tokens", "descriptionRowByRow": "The number of tokens that were ingested by the LLM.", 'selectable': false},
        {"label": "Avg Output tokens", "name": "outputTokensPerRow", "shortDescription": "The number of tokens that were generated by the LLM. Average per row.", "labelRowByRow": "Output Tokens", "descriptionRowByRow": "The number of tokens that were generated by the LLM.", 'selectable': false},
    ];

    svc.nameByMetricCode =
    {
        'ANSWER_RELEVANCY': "answerRelevancy",
        'ANSWER_CORRECTNESS': "answerCorrectness",
        'ANSWER_SIMILARITY': "answerSimilarity",
        'CONTEXT_RECALL': "contextRecall",
        'CONTEXT_PRECISION': "contextPrecision",
        'FAITHFULNESS': "faithfulness",
        'MULTIMODAL_FAITHFULNESS': "multimodalFaithfulness",
        'MULTIMODAL_RELEVANCY': "multimodalRelevancy",

        'BERT_SCORE_PRECISION': "bertScore",
        'BERT_SCORE_RECALL': "bertScore",
        'BERT_SCORE_F1': "bertScore",

        'BLEU': "bleu",

        'ROUGE_1_PRECISION': "rouge",
        'ROUGE_1_RECALL': "rouge",
        'ROUGE_1_F1': "rouge",
        'ROUGE_2_PRECISION': "rouge",
        'ROUGE_2_RECALL': "rouge",
        'ROUGE_2_F1': "rouge",
        'ROUGE_L_PRECISION': "rouge",
        'ROUGE_L_RECALL': "rouge",
        'ROUGE_L_F1': "rouge",

        'INPUT_TOKENS_PER_ROW': 'inputTokensPerRow',
        'OUTPUT_TOKENS_PER_ROW': 'outputTokensPerRow',
    }

    svc.llmTaskTypes = [
        {
            type: "QUESTION_ANSWERING",
            fullName: "Question Answering",
            shortName: "question answering"
        },
        {
            type: "SUMMARIZATION",
            fullName: "Summarization",
            shortName: "summarization"
        },
        {
            type: "TRANSLATION",
            fullName: "Translation",
            shortName: "translation"
        },
        {
            type: "OTHER_LLM_TASK",
            fullName: "Other LLM Evaluation Task",
            shortName: "other llm"
        },
    ];

    svc.inputFormats = [
        {
            type: "PROMPT_RECIPE",
            fullName: "Prompt Recipe",
            shortName: "prompt recipe"
        },
        {
            type: "DATAIKU_ANSWERS",
            fullName: "Dataiku Answers",
            shortName: "dataiku answers"
        },
        {
            type: "CUSTOM",
            fullName: "Custom",
            shortName: "custom"
        },
    ];

    svc.llmModelTaskTypes = [
        {
            type: "LLM",
            fullName: "Large Language Model",
            shortName: "LLM"
        }
    ];

    svc.metricsPerTaskType = {
        "QUESTION_ANSWERING": ["answerRelevancy", "answerCorrectness", "answerSimilarity", "contextRecall", "contextPrecision", "faithfulness", "bertScore"],
        "SUMMARIZATION": ["bertScore", "rouge", "answerSimilarity"],
        "TRANSLATION": ["bertScore", "bleu", "answerSimilarity"],
        "OTHER_LLM_TASK" : ["bertScore"]
    }

    svc.fieldsNeededPerMetric = {
        "answerRelevancy": ["input", "output", "context", "embeddingLLM", "completionLLM"],
        "answerCorrectness": ["input", "output", "groundTruth", "embeddingLLM", "completionLLM"],
        "answerSimilarity": ["input", "output", "groundTruth", "embeddingLLM", "completionLLM"],
        "contextRecall": ["input", "groundTruth", "context", "embeddingLLM", "completionLLM"],
        "contextPrecision": ["input", "groundTruth", "context", "embeddingLLM", "completionLLM"],
        "faithfulness": ["input", "output", "context", "embeddingLLM", "completionLLM"],
        "multimodalFaithfulness": ["input", "output", "context", "embeddingLLM", "completionLLM"],
        "multimodalRelevancy": ["input", "output", "context", "embeddingLLM", "completionLLM"],
        "bertScore": ["output", "groundTruth"],
        "bleu":["output", "groundTruth"],
        "rouge": ["output", "groundTruth"]
    }

    svc.isRecommendedMetric = function(metric, taskType) {
        if (!Object.keys(svc.metricsPerTaskType).includes(taskType)) return false;
        return svc.metricsPerTaskType[taskType].includes(metric);
    }

    svc.getPotentialWarningForMetricComputation = function(metric, packagesWarning, hasInput, hasOutput, hasGroundTruth, hasContext, hasEmbeddingLLM, hasCompletionLLM) {
        if (!Object.keys(svc.fieldsNeededPerMetric).includes(metric)) {
            return "";
        }

        if (svc.fieldsNeededPerMetric[metric].includes("input") && !hasInput) {
            return "Requires an Input column";
        }
        if (svc.fieldsNeededPerMetric[metric].includes("output") && !hasOutput) {
            return "Requires an Output column";
        }
        if (svc.fieldsNeededPerMetric[metric].includes("groundTruth") && !hasGroundTruth) {
            return "Requires a Ground truth column";
        }
        if (svc.fieldsNeededPerMetric[metric].includes("context") && !hasContext) {
            return "Requires a Context column";
        }
        if (svc.fieldsNeededPerMetric[metric].includes("embeddingLLM") && !hasEmbeddingLLM) {
            return "Requires an Embedding LLM";
        }
        if (svc.fieldsNeededPerMetric[metric].includes("completionLLM") && !hasCompletionLLM) {
            return "Requires a Completion LLM";
        }
        else {
            return "";
        }
    }

    svc.getMetricByCode = function(metricCode) {
        const name = svc.nameByMetricCode[metricCode];
        if (!name) {
            return undefined;
        }
        return svc.metrics.find(m => m.name === name);
    }

    svc.getCustomMetricNamesFromEvaluationDetails = function(evaluationDetails) {
        let names = [];

        if (evaluationDetails.metrics && evaluationDetails.metrics.customMetricsResults) {
            names = evaluationDetails.metrics.customMetricsResults.map(function(metric) { return metric.metric.name });
        }

        return names;
    }


    return svc;
});

app.controller("NLPLLMEvaluationRecipeCreationController", function($scope, DataikuAPI, ActiveProjectKey,  DatasetUtils, $controller, RecipeComputablesService) {
    $controller("_RecipeCreationControllerBase", {$scope:$scope});

    $scope.forceMainLabel = true;
    $scope.uiState = {inputDs: null} // needed to replace 'io' (we don't want to use _RecipeOutputNewManagedBehavior), so that binding inputDs to the dataset-selector works across scopes

    $scope.recipe = {
        projectKey : ActiveProjectKey.get(),
        type: "nlp_llm_evaluation",
        inputs : {},
        outputs : {},
        params: {}
    };


    addDatasetUniquenessCheck($scope, DataikuAPI, ActiveProjectKey.get());
    fetchManagedDatasetConnections($scope, DataikuAPI);

    $scope.$watch("uiState.inputDs", function(nv, ov) {
        if (nv) {
            $scope.recipe.name = "evaluate_" + nv.replace(/[A-Z]*\./, "");
        }
        if ($scope.uiState.inputDs) {
            $scope.recipe.inputs.main = {items:[{ref:$scope.uiState.inputDs}]}; // for the managed dataset creation options
        } else {
            $scope.recipe.inputs.main = {items:[]}; // for the managed dataset creation options
        }
    });


    DatasetUtils.listDatasetsUsabilityInAndOut(ActiveProjectKey.get(), $scope.recipe.type).then(function(data){
        $scope.availableInputDatasets = data[0];
    });

    RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
        $scope.setComputablesMap(map);
    });

    $scope.hasMain = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.main && outputs.main.items && outputs.main.items.length > 0 && outputs.main.items[0].ref
    }
    $scope.hasMetrics = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.metrics && outputs.metrics.items && outputs.metrics.items.length > 0 && outputs.metrics.items[0].ref
    }
    $scope.hasEvaluationStore = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.evaluationStore && outputs.evaluationStore.items && outputs.evaluationStore.items.length > 0 && outputs.evaluationStore.items[0].ref
    }

    $scope.canCreate = function(){
        return $scope.recipe.name
            && $scope.recipe.name.length > 0
            && $scope.recipe.outputs
            && !$scope.shouldDisplayOutputExplanation()
            && !($scope.newRecipeForm.$invalid);
    }

    $scope.shouldDisplayOutputExplanation = function () {
        return !$scope.hasMain() && !$scope.hasMetrics() && !$scope.hasEvaluationStore();
    };

    $scope.generateOutputExplanation = function () {
        const requiredOutputRoles = [];
        $scope.recipeDesc.outputRoles.forEach((role, outputRoleidx) => {
            requiredOutputRoles.push(role.name === "main" ? '"Output Dataset"' : '"' + (role.label || role.name) + '"');
        });
        const message = "This recipe requires at least one output in: "
            + requiredOutputRoles.slice(0, -1).join(', ')
            + (requiredOutputRoles.length === 2 ? ' or ' : ', or ')
            + requiredOutputRoles.slice(-1) + ".";
        return message;
    };

    $scope.doCreateRecipe = function() {
        $scope.creatingRecipe = true;
        const settings = {
            zone: $scope.zone
        }

        return DataikuAPI.flow.recipes.generic.create($scope.recipe, settings);
    };
});

app.controller("NLPLLMEvaluationRecipeEditorController", function($scope, $rootScope, $controller, $stateParams,
    ActiveProjectKey, DataikuAPI, FutureProgressModal, DOCUMENT_SPLITTING_METHOD_MAP, ModelLabelUtils, EmbeddingUtils, PromptUtils, LLMEvaluationMetrics, StringUtils, WT1) {
    $controller("_NLPLLMRecipeControllerBase", {$scope: $scope});

    $scope.NO_COLUMN_SELECTED = "None - no column selected";
    $scope.NO_LLM_SELECTED = {"id": "None", "friendlyName": "None - no model selected", type: "None"}
    $scope.DOCUMENT_SPLITTING_METHOD_MAP = DOCUMENT_SPLITTING_METHOD_MAP;
    $scope.COLUMNS_NEEDED_BY_PROMPT_RECIPE = ["llm_raw_query", "llm_raw_response"];
    $scope.COLUMNS_NEEDED_BY_DATAIKU_ANSWERS = ["question", "answer", "sources"];

    $scope.inputFormats = LLMEvaluationMetrics.inputFormats;
    $scope.llmTaskTypes = LLMEvaluationMetrics.llmTaskTypes;
    $scope.fieldsNeeded = LLMEvaluationMetrics.fieldsNeededPerMetric;

    const llmTaskTypeHelpers = {
        "QUESTION_ANSWERING": {
            "inputColumn": "Question asked to the model",
            "outputColumn": "Answer provided by the model (optional)",
            "groundTruthColumn": "Reference answer (optional)",
            "contextColumn": "Context(s) provided to the model (optional)"
        },
        "SUMMARIZATION": {
            "inputColumn": "Text to summarize",
            "outputColumn": "Summary provided by the model (optional)",
            "groundTruthColumn": "Reference summary (optional)",
            "contextColumn": "Context(s) provided to the model (optional)",
        },
        "TRANSLATION": {
            "inputColumn": "Text to translate",
            "outputColumn": "Translation provided by the model (optional)",
            "groundTruthColumn": "Reference translation (optional)",
            "contextColumn": "Context(s) provided to the model (optional)"
        },
        "OTHER_LLM_TASK": {
            "inputColumn": "",
            "outputColumn": "Optional",
            "groundTruthColumn": "Optional",
            "contextColumn": "Optional"
        }
    }

    $scope.getTaskTypeHelper = function(column) {
        if (!$scope.desc.llmTaskType) return "";
        return llmTaskTypeHelpers[$scope.desc.llmTaskType][column];
    }

    // Does context makes sense for the given task
    $scope.canUseContext = function() {
        return $scope.desc.llmTaskType !== 'TRANSLATION' && $scope.desc.llmTaskType !== 'SUMMARIZATION';
    }

    $scope.shouldDisplayContext = function() {
        return $scope.canUseContext() && ! ['PROMPT_RECIPE', 'DATAIKU_ANSWERS'].includes($scope.desc.inputFormat);
    }

    $scope.filteredMetrics = [];

    $scope.$watch("desc", (nv, ov) => {
        if (angular.equals(nv, ov)) { // only happens on init
            onInit();
        }
    });

    $scope.missingColumnsWarning = "";

    $scope.$watch("inputDatasetColumns", (nv, ov) => {
        if (nv) {
            $scope.canPromptRecipe = $scope.COLUMNS_NEEDED_BY_PROMPT_RECIPE.every(col => nv.includes(col));
            $scope.canDataikuAnswers = $scope.COLUMNS_NEEDED_BY_DATAIKU_ANSWERS.every(col => nv.includes(col));
            $scope.updateMissingColumnsWarning($scope.desc.inputFormat);
        }
    });

    $scope.saveColumnNames = function() {
        if ($scope.desc.inputColumnName !== $scope.NO_COLUMN_SELECTED) {
            $scope.savedInputColumnName = $scope.desc.inputColumnName;
        }
        if ($scope.desc.outputColumnName !== $scope.NO_COLUMN_SELECTED) {
            $scope.savedOutputColumnName = $scope.desc.outputColumnName;
        }
        if ($scope.desc.contextColumnName !== $scope.NO_COLUMN_SELECTED) {
            $scope.savedContextColumnName = $scope.desc.contextColumnName;
        }
    }
    $scope.restoreColumnNames = function() {
        if ($scope.savedInputColumnName !== undefined) {
            $scope.desc.inputColumnName = $scope.savedInputColumnName;
            $scope.savedInputColumnName = undefined;
        }
        if ($scope.savedOutputColumnName !== undefined) {
            $scope.desc.outputColumnName = $scope.savedOutputColumnName;
            $scope.savedOutputColumnName = undefined;
        }
        if ($scope.savedContextColumnName !== undefined) {
            $scope.desc.contextColumnName = $scope.savedContextColumnName;
            $scope.savedContextColumnName = undefined;
        }
    }

    $scope.$watch("desc.inputFormat", (nv, ov) => {
        if (nv === "PROMPT_RECIPE") {
            if (ov !== "DATAIKU_ANSWERS") {
                $scope.saveColumnNames();
            }
            $scope.desc.inputColumnName = "llm_raw_query";
            $scope.desc.outputColumnName = "llm_raw_response";
            $scope.desc.contextColumnName = "llm_raw_response";
        }
        else if (nv === "DATAIKU_ANSWERS") {
            if (ov !== "PROMPT_RECIPE") {
                $scope.saveColumnNames();
            }
            $scope.desc.inputColumnName = "question";
            $scope.desc.outputColumnName = "answer";
            $scope.desc.contextColumnName = "sources";
        } else {
            $scope.restoreColumnNames();
        }

        $scope.updateMissingColumnsWarning(nv);
        $scope.updateContextWarning();
    });

    $scope.$watch("desc.llmTaskType", (nv, ov) => {
        if (!$scope.canUseContext()) {
            if ($scope.desc.contextColumnName !== $scope.NO_COLUMN_SELECTED) {
                $scope.savedContextColumnName = $scope.desc.contextColumnName;
            }
            $scope.desc.contextColumnName = $scope.NO_COLUMN_SELECTED;
        } else if ($scope.savedContextColumnName !== undefined) {
            $scope.desc.contextColumnName = $scope.savedContextColumnName;
            $scope.savedContextColumnName = undefined;
        }
        $scope.filteredMetrics = getFilteredMetrics();
    });

    $scope.$watch("desc.inputFormat", () => {
        if ($scope.desc.inputFormat) {
            $scope.filteredMetrics = getFilteredMetrics();
        }
    })

    $scope.updateMissingColumnsWarning = function(inputFormat) {
        if ($scope.inputDatasetColumns) {
            if (inputFormat === "PROMPT_RECIPE" && !$scope.canPromptRecipe) {
                $scope.missingColumnsWarning = "Input dataset lacks some required columns : " + $scope.COLUMNS_NEEDED_BY_PROMPT_RECIPE.filter(col => !$scope.inputDatasetColumns.includes(col)) + ". Make sure \"Raw query output mode\" and \"Raw response output mode\" in your Prompt recipe are not set to \"None\".";
            } else if (inputFormat === "DATAIKU_ANSWERS" && !$scope.canDataikuAnswers) {
                $scope.missingColumnsWarning = "Input dataset lacks some required columns : " + $scope.COLUMNS_NEEDED_BY_DATAIKU_ANSWERS.filter(col => !$scope.inputDatasetColumns.includes(col)) + ". Make sure the \"Retrieval Method\" is set to 'Use knowledge bank retrieval'.";
            } else {
                $scope.missingColumnsWarning = "";
            }
        }
    }

    function getTaskTypeFullName() {
        if ($scope.desc.llmTaskType) {
            const llmTaskType = $scope.llmTaskTypes.find(t => t.type == $scope.desc.llmTaskType);
            if (llmTaskType) {
                return llmTaskType.fullName;
            }
        }
        return null;
    }

    $scope.getRecommendedForHelper = function() {
        const taskTypeFullName = getTaskTypeFullName();
        if (taskTypeFullName) {
            return `(recommended for ${taskTypeFullName})`;
        } else {
            return '(recommended)';
        }
    }

    function onInit() {
        // Initialize the metric listing
        $scope.filteredMetrics = getFilteredMetrics();
    }

    function getFilteredMetrics()  {
        return LLMEvaluationMetrics.metrics
            .filter(possibleMetric => possibleMetric.selectable)
            .filter(possibleMetric => possibleMetric.promptRecipeFormatNeeded ? $scope.desc.inputFormat === "PROMPT_RECIPE" : true)
            .map(possibleMetric => {
                if ($scope.desc.metrics.includes(possibleMetric.name)) {
                    return ({...possibleMetric, $selected: true})
                } else {
                    return ({...possibleMetric, $selected: false})
                }
            });
    }

    $scope.updateFilteredMetricsSelection = function() {
        $scope.desc.metrics = $scope.filteredMetrics.filter(m => m.$selected).map(m => m.name);
        $scope.updateContextWarning();
    };

    DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "TEXT_EMBEDDING_EXTRACTION").success(function(data) {
        $scope.availableEmbeddingLLMs = [$scope.NO_LLM_SELECTED].concat(data.identifiers);
        $scope.refreshEmbeddingLLM();
    }).error(setErrorInScope.bind($scope));

    $scope.refreshEmbeddingLLM = function() {
        if ($scope.desc && $scope.desc.embeddingLLMId && $scope.availableEmbeddingLLMs) {
            $scope.activeEmbeddingLLM = $scope.availableEmbeddingLLMs.find(l => l.id === $scope.desc.embeddingLLMId);
        }
    }

    DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").success(function(data) {
        $scope.availableCompletionLLMs = [$scope.NO_LLM_SELECTED].concat(data.identifiers);
        $scope.refreshCompletionLLM();
    }).error(setErrorInScope.bind($scope));

    $scope.refreshCompletionLLM = function() {
        if ($scope.desc && $scope.desc.completionLLMId && $scope.availableCompletionLLMs) {
            $scope.activeCompletionLLM = $scope.availableCompletionLLMs.find(l => l.id === $scope.desc.completionLLMId);
        }

        if ($scope.activeCompletionLLM && $scope.activeCompletionLLM.id !== $scope.NO_LLM_SELECTED.id) {
            $scope.temperatureRange = PromptUtils.getTemperatureRange($scope.activeCompletionLLM);
            $scope.topKRange = PromptUtils.getTopKRange($scope.activeCompletionLLM);
        }
    }

    $scope.getMaxTokenLimit = function() {
        return $scope.activeEmbeddingLLM ? $scope.activeEmbeddingLLM.maxTokensLimit : null;
    }

    $scope.shouldWarnAboutChunkSize = function() {
        return EmbeddingUtils.shouldWarnAboutChunkSize($scope.activeEmbeddingLLM, $scope.desc.embeddingSettings.documentSplittingMode, $scope.desc.embeddingSettings.chunkSizeCharacters);
    };

    $scope.hasAllRequiredOutputs = function() {
        if (!$scope.recipe || !$scope.recipe.outputs) {
            return false;
        }
        const out = $scope.recipe.outputs;
        // at least one of the outputs is needed
        if (out.main && out.main.items && out.main.items.length) {
            return true;
        } else if (out.evaluationStore && out.evaluationStore.items && out.evaluationStore.items.length) {
            return true;
        } else if (out.metrics && out.metrics.items && out.metrics.items.length) {
            return true;
        } else {
            return false;
        }
    };

    $scope.isRecommendedMetric = function(metric) { return LLMEvaluationMetrics.isRecommendedMetric(metric, $scope.desc.llmTaskType); }

    $scope.selectOnlyComputableMetrics = function() {
        $scope.filteredMetrics.forEach(metric=>
            metric.$selected = $scope.isRecommendedMetric(metric.name) && !$scope.getPotentialWarningForMetricComputation(true, metric.name)
        );
        $scope.updateFilteredMetricsSelection();
    }

    $scope.canRecommendMetrics = function() { return $scope.desc.llmTaskType !== null && $scope.desc.llmTaskType !== undefined }

    $scope.getPotentialWarningForMetricComputation = function(isSelected, metric) {
        if (!isSelected) {
            return "";
        }
        const hasInput = !!$scope.desc.inputColumnName && $scope.desc.inputColumnName !== $scope.NO_COLUMN_SELECTED;
        const hasOutput = !!$scope.desc.outputColumnName && $scope.desc.outputColumnName !== $scope.NO_COLUMN_SELECTED;
        const hasGroundTruth = !!$scope.desc.groundTruthColumnName && $scope.desc.groundTruthColumnName !== $scope.NO_COLUMN_SELECTED;
        const hasContext = !!$scope.desc.contextColumnName && $scope.desc.contextColumnName !== $scope.NO_COLUMN_SELECTED;
        const hasEmbeddingLLM = !!$scope.desc.embeddingLLMId && $scope.desc.embeddingLLMId !== $scope.NO_LLM_SELECTED.id;
        const hasCompletionLLM = !!$scope.desc.completionLLMId && $scope.desc.completionLLMId !== $scope.NO_LLM_SELECTED.id;
        return LLMEvaluationMetrics.getPotentialWarningForMetricComputation(metric, $scope.codeEnvWarning.reason, hasInput, hasOutput, hasGroundTruth, hasContext, hasEmbeddingLLM, hasCompletionLLM);
    }

    $scope.contextWarning = "";
    $scope.updateContextWarning = function() {
        const needsContext = $scope.desc.metrics.some(m => (LLMEvaluationMetrics.fieldsNeededPerMetric[m] || []).includes("context"));
        if (needsContext && $scope.desc.inputFormat === 'PROMPT_RECIPE') {
            $scope.contextWarning = "Context-based metrics require that the prompt recipe used a Retrieval-augmented LLM with \"Source output format\" set to \"Separated\"";
        } else if (needsContext && $scope.desc.inputFormat === 'DATAIKU_ANSWERS') {
            $scope.contextWarning = "Context-based metrics require that Dataiku Answers' \"Retrieval Method\" was set to 'Use knowledge bank retrieval'";
        } else {
            $scope.contextWarning = "";
        }
    }

    $scope.codeEnvWarning = {
        enabled: false,
        envName: "",
        reason: ""
    }

    DataikuAPI.codeenvs.listWithVisualMlPackages($stateParams.projectKey).success(function(data) {
        $scope.envsCompatibility = data;
        updateCodeEnvWarning();
    }).error(setErrorInScope.bind($scope));

    $scope.$watchGroup(['recipe.params.envSelection.envMode', 'recipe.params.envSelection.envName'], (nv, ov) => {
        updateCodeEnvWarning();
    });

    function updateCodeEnvWarning() {
        if ( $scope.envsCompatibility !== undefined ) {
            let compat = $scope.envsCompatibility.builtinEnvCompat; // Built-in, or inheriting with no default set at the instance-level
            if ($scope.recipe.params.envSelection.envMode == "EXPLICIT_ENV") {
                compat = $scope.envsCompatibility.envs.filter(env => env.envName == $scope.recipe.params.envSelection.envName)[0]
            } else if ($scope.recipe.params.envSelection.envMode == "INHERIT" && $scope.envsCompatibility.resolvedInheritDefault) {
                compat = $scope.envsCompatibility.envs.filter(env => env.envName == $scope.envsCompatibility.resolvedInheritDefault)[0]
            }

            $scope.codeEnvWarning.enabled = !compat.llmEvaluation.compatible;
            $scope.codeEnvWarning.envName = compat.envName ? compat.envName : "DSS builtin env"
            if (compat.llmEvaluation.reasons.length > 0) {
                $scope.codeEnvWarning.reason = compat.llmEvaluation.reasons[0];
            } else {
                $scope.codeEnvWarning.reason = "";
            }
        }
    }

    const baseCanSave = $scope.canSave;
    $scope.canSave = function() {
        return ModelLabelUtils.validateLabels($scope.desc) && baseCanSave();
    };

    const customMetricDefaultCode = `def evaluate(input_df, recipe_params, interpreted_columns, **kwargs):
    """
    Custom score function.
    Must return a float representing the metric value over the whole sample, and 
    optionally an array of shape (nb_records,) with the row-by-row values
    - input_df is the Input dataset of the LLM evaluation recipe, as a Pandas DataFrame
    - recipe_params is an object with the LLM evaluation recipe parameters:
        - input_column_name: Name of the Input column as a string
        - output_column_name (may be None): Name of the Output column as a string
        - ground_truth_column_name (may be None): Name of the Ground truth column as 
          a string
        - context_column_name (may be None): Name of the Context column as a string
        - embedding_llm (may be None): A DKUEmbeddings object, to be used to query 
          the LLM mesh to embed the inputs
        - completion_llm (may be None): A DKULLM object, to be used to query the 
          LLM mesh with an eventual prompt
    - interpreted_columns are the input columns as they were used by the recipe, 
      including some eventual pre-treatment, notably for the prompt recipe and dataiku
      answers cases. Formatted as Pandas Dataseries.
      If no pre-treatment happened, their content is the same as in the input_df
        - input : for Prompt recipe, this is a concatenation of all the prompt messages
        - output : for Prompt recipe, only the text, extracted from the json
        - ground_truth 
        - context : for Prompt recipe and Dataiku Answers, only the excerpt, extracted
          from the json
    """
    return 0.5, [optionally, one, value, per, row]`

    $scope.getNewMetricTemplate = function() {
        if (!$scope.desc.customMetrics) {$scope.desc.customMetrics = [];};
        const name = StringUtils.transmogrify("Custom Metric #" + ($scope.desc.customMetrics.length + 1).toString(),
            $scope.desc.customMetrics.map(a => a.name),
            function(i){return "Custom Metric #" + (i+1).toString() }
        );

        const template = {
            name,
            metricCode: customMetricDefaultCode,
            description: "",
            greaterIsBetter: true,
            minValue: 0,
            maxValue: 1,
            type: 'LLM',
            $foldableOpen: true
       };
       return template;
    }

    $scope.snippetCategory = 'py-llm-custom-metric';

    $scope.addNewCustomMetric = function() {
        if (!$scope.desc.customMetrics) {$scope.desc.customMetrics = [];};
        $scope.desc.customMetrics.push($scope.getNewMetricTemplate());
    }

    $scope.testCustomMetricResults = {}
    $scope.testCustomMetric = function(customMetricIndex) {
        const mainInput = $scope.recipe.inputs.main.items[0];
        const serialisedInputDataset = !!mainInput ? $scope.computablesMap[mainInput.ref] : undefined;
        return DataikuAPI.flow.recipes.testCustomMetric(ActiveProjectKey.get(), $scope.recipe, $scope.desc, customMetricIndex, serialisedInputDataset)
        .then(({data}) => {
            return FutureProgressModal.show($scope, data, "Testing");
        })
        .then((customMetricResult) => {
            $scope.testCustomMetricResults[customMetricIndex] = customMetricResult;
        })
        .catch(setErrorInScope.bind($scope));
    }

    $scope.fireCustomMetricAddedWT1Event = function() {
        WT1.event("clicked-item", {"item-id": 'llm-evaluation-add-custom-metric'});
    };

    $scope.fireCustomMetricRemovedWT1Event = function() {
        WT1.event("clicked-item", {"item-id": 'llm-evaluation-remove-custom-metric'});
    };

    $scope.toggleFoldable = function(index) {
        if($scope.desc.customMetrics[index]){
            $scope.desc.customMetrics[index].$foldableOpen = !$scope.desc.customMetrics[index].$foldableOpen
        }
    }

    $scope.defaultRagasMaxWorkers = $rootScope.appConfig.defaultRagasMaxWorkers;
    $scope.mayRunCustomMetrics = $scope.mayWriteSafeCode();

    $scope.setRagasMaxWorkers = function(value) {
        if (!value) {
            $scope.desc.ragasMaxWorkers = null;
        }
    }

    $scope.getContextColumnNeededIcon = function(metricName) {
        if (["multimodalFaithfulness", "multimodalRelevancy"].includes(metricName)) {
            return "dku-icon-image-16 dibvat";
        } else {
            return "dku-icon-data-text-16 dibvat";
        }
    }

    $scope.getContextColumnNeededTooltip = function(metricName) {
        if (["multimodalFaithfulness", "multimodalRelevancy"].includes(metricName)) {
            return "The context must be multimodal";
        } else {
            return "The context must be textual only";
        }
    }
});

app.filter('sortLLMEvalMetrics', function (LLMEvaluationMetrics) {

    return function (input, llmTaskType) {
        if (!angular.isArray(input)) return input;

        input.sort(function (a, b) {
            const aIsRecommended = LLMEvaluationMetrics.isRecommendedMetric(a.name, llmTaskType);
            const bIsRecommended = LLMEvaluationMetrics.isRecommendedMetric(b.name, llmTaskType);
            if (aIsRecommended != bIsRecommended) {
                return (bIsRecommended & 1) - (aIsRecommended & 1) // recommended first
            } else {
                return a.name.localeCompare(b.name); // then by alphabetical order
            }
        });

        return input;
    };
});

}());
