(function() {
'use strict';

const app = angular.module('dataiku.recipes');

app.controller("_NLPLLMRecipeControllerBase", function($scope, $stateParams, $q, DataikuAPI, PartitionDeps, ComputableSchemaRecipeSave, SmartId, PromptUtils) {

    $scope.onLLMChange = function() {
        if ($scope.desc && $scope.desc.llmId && $scope.availableLLMs) {
            $scope.activeLLM = $scope.availableLLMs.find(l => l.id === $scope.desc.llmId);
            $scope.temperatureRange = PromptUtils.getTemperatureRange($scope.activeLLM);
            $scope.topKRange = PromptUtils.getTopKRange($scope.activeLLM);
            if ($scope.activeLLM.type === "RETRIEVAL_AUGMENTED") {
                DataikuAPI.savedmodels.get($stateParams.projectKey, $scope.activeLLM.savedModelSmartId).then(function({data}) {
                    DataikuAPI.retrievableknowledge.get($stateParams.projectKey, data.inlineVersions[0].ragllmSettings.kbRef).then(function({data}) {
                        $scope.retrievableKnowledge = data;
                    })
                }).catch(setErrorInScope.bind($scope));
            } else {
                $scope.retrievableKnowledge = undefined;
            }
        }
    };

    $scope.$watch('recipe.inputs.main', function(nv, ov) {
        if (!nv) return;

        $scope.inputRef = ($scope.recipe.inputs.main.items || [{}])[0].ref;
        if (!$scope.inputRef) return;
                
        var mainRoleAcceptDataset = true;
        if($scope.recipeDesc && $scope.recipeDesc.inputRoles){
            mainRoleAcceptDataset = $scope.recipeDesc.inputRoles.find(role => role.name == 'main').acceptsDataset; // support for folder as main role (embed documents)
        } 

        if(mainRoleAcceptDataset){

            const resolvedRef = SmartId.resolve($scope.inputRef, $stateParams.projectKey);
            DataikuAPI.datasets.get(resolvedRef.projectKey, resolvedRef.id, $stateParams.projectKey).then(response => {
                const schema = response.data.schema;
                $scope.inputDatasetColumns = schema.columns.map(column => column.name);
                $scope.inputDatasetSchema = schema;
            });
        }
    });

    const deregister = $scope.$watch("script", function() {
        if (!$scope.script) return;

        $scope.desc = JSON.parse($scope.script.data);

        $scope.onLLMChange();
        deregister();
    });

    $scope.loadLLMs = function(purpose) {
        DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, purpose).success(function(data) {
            $scope.availableLLMs = data.identifiers;
            $scope.onLLMChange();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.hooks.recipeIsDirty = function() {
        if (!$scope.recipe) return false;
        if ($scope.creation)  return true;

        const dirtyRecipe = !angular.equals($scope.recipe, $scope.origRecipe);
        const origDesc = JSON.parse($scope.origScript.data);
        const dirtyDesc = !angular.equals(origDesc, $scope.desc);
        return dirtyRecipe || dirtyDesc;
    };

    $scope.hooks.save = function(){
        const recipeSerialized = angular.copy($scope.recipe);
        PartitionDeps.prepareRecipeForSerialize(recipeSerialized);

        const payload = angular.toJson($scope.desc);
        const deferred = $q.defer();
        ComputableSchemaRecipeSave.handleSave($scope, recipeSerialized, payload, deferred);
        $scope.script.data = payload;
        return deferred.promise;
    };
});


app.controller("NLPRecipeConvertToPromptRecipeModal", function($scope, $stateParams, $state, DataikuAPI) {
    DataikuAPI.flow.recipes.nlp.getPromptForNLPRecipe($stateParams.projectKey, $scope.recipe.name).success(function(data) {
        $scope.prompt = data;
    }).error(setErrorInScope.bind($scope));

    $scope.convert = function() {
        if (!$scope.recipe || !$scope.recipe.name) return;

        DataikuAPI.flow.recipes.nlp.convertToPromptRecipe($stateParams.projectKey, $scope.recipe.name).success(function() {
            $state.reload();
        }).error(setErrorInScope.bind($scope));
    };
});


app.component("llmSelector", {
  bindings: {
    availableLlms: "<",
    selectedLlmId: "=",
    onChange: "<",
    defaultOpened: '<',
    dropdownPosition: '@',
    disableAlertWarning: '<',
    placeholder: '<', // Optional, defaults to "Nothing selected"
    hideCopyButton: '<',
    hideGoToLlmButton: '<',
  },
  template: `
    <basic-select ng-if="$ctrl.availableLlms"
                  items="$ctrl.availableLlms"
                  ng-model="$ctrl.selectedLlmId"
                  bind-value="id"
                  bind-label="friendlyName"
                  group-by-fn="$ctrl.niceModelType"
                  searchable="true"
                  search-fn="$ctrl.searchFunction"
                  placeholder="{{ $ctrl.placeholder }}"
                  invalidate-on-ghosts="true"
                  default-opened="$ctrl.defaultOpened"
                  dropdown-position="{{ $ctrl.dropdownPosition }}"
                  ghost-items-tooltip="LLM not available. It has been disabled, retired, or is not supported for this feature. Choose another LLM or contact your administrator."
                  custom-ng-select-class="ng-dropdown-panel--500 
                                          ng-select--custom-dropdown-height 
                                          ng-select--custom-x-overflow
                                          ng-select--custom-reset-default-padding"></basic-select>
    <copy-to-clipboard-icon ng-if="!$ctrl.hideCopyButton" icon="'copy-step'" icon-class="$ctrl.selectedLlmId ? 'btn btn--text btn--icon btn--secondary' : 'btn btn--text btn--icon btn--secondary disabled'" disabled="!$ctrl.selectedLlmId" value="$ctrl.selectedLlmId" name="'LLM ID'"></copy-to-clipboard-icon>
    <div class="help-inline padleft0" ng-if="!$ctrl.hideGoToLlmButton && $ctrl.selectedLLMSmartID">
        <external-link ui-sref="projects.project.savedmodels.savedmodel.versions({projectKey: $ctrl.selectedLLMSmartID.projectKey, smId: $ctrl.selectedLLMSmartID.id, redirectToActiveVersion: true})" target="_blank">Go to {{$ctrl.selectedLlmId.startsWith("agent:") ? "agent" : "LLM" }}</external-link>
    </div>
    <div class="mbot0 mtop8 alert alert-warning" ng-if="!$ctrl.isAvailable && !$ctrl.disableAlertWarning">
        LLM not available. It has been disabled, retired, or is not supported for this feature. Choose another LLM or contact your administrator. <strong>{{$ctrl.selectedLlmId}}</strong>
    </div>`,
  controller: function ($scope, $filter, SmartId) {
    const $ctrl = this;

    $scope.$watch(
      function () {
        return $ctrl.selectedLlmId;
      },
      function (newVal, oldVal) {
        $ctrl.updateAvailability();
        $ctrl.updateLLMSourceLink();
        if (newVal !== oldVal && $ctrl.onChange) {
            $ctrl.onChange();
        }
      }
    );

    $ctrl.$onChanges = () => {
        if (typeof $ctrl.placeholder === 'undefined' || $ctrl.placeholder === null) {
            $ctrl.placeholder = "Nothing selected";
        }

        $ctrl.updateAvailability();
        $ctrl.updateLLMSourceLink();
    };

    $ctrl.isAvailable = true;
    $ctrl.updateAvailability = () => {
        if ($ctrl.availableLlms && $ctrl.selectedLlmId) {
            $ctrl.isAvailable = !!$ctrl.availableLlms.find((l) => l.id === $ctrl.selectedLlmId);
        }
    };

    $ctrl.selectedLLMSmartID = undefined;
    $ctrl.updateLLMSourceLink = () => {
        $ctrl.selectedLLMSmartID = undefined;
        if ($ctrl.availableLlms && $ctrl.selectedLlmId) {
            const selectedLLM = $ctrl.availableLlms.find((l) => l.id === $ctrl.selectedLlmId);
            if (selectedLLM && (["SAVED_MODEL_AGENT", "RETRIEVAL_AUGMENTED"].includes(selectedLLM.type) || selectedLLM.type.startsWith("SAVED_MODEL_FINETUNED"))) {
                $ctrl.selectedLLMSmartID = SmartId.resolve(selectedLLM.savedModelSmartId);
            }
        }
    };

    $ctrl.niceModelType = (modelOption) => {
        return $filter('niceLLMType')(modelOption.type);
    };

    $ctrl.searchFunction = (term, llm) => {
        if (llm.friendlyName.includes(term)) return true;
        if (llm.id.includes(term)) return true;
        return false;
    }
  },
});

app.component("perCharsSplittingSettings", {
    bindings: {
        chunkSizeCharacters: "=",
        chunkOverlapCharacters: "=",
        embeddingLlm: '<',
        documentSplittingMode: '<',
        defaultChunkSize: '<',
        defaultOverlapSize: '<',

    },
    templateUrl: "templates/recipes/nlp/per-chars-splitting-settings.component.html",
    controller: function ($scope, EmbeddingUtils) {
      const $ctrl = this;
  
        $ctrl.shouldWarnAboutChunkSize = () => {
            return EmbeddingUtils.shouldWarnAboutChunkSize($ctrl.embeddingLlm, $ctrl.documentSplittingMode, $ctrl.chunkSizeCharacters);
        };

        $ctrl.getModelMaxSizeInChars = () => {
            return EmbeddingUtils.tokensToCharsOrDefault($ctrl.embeddingLlm);
        };

        $ctrl.resetToDefaultSplittingSettings = function(){ 
            $ctrl.chunkSizeCharacters = $ctrl.defaultChunkSize;
            $ctrl.chunkOverlapCharacters = $ctrl.defaultOverlapSize;
        }
        
    },
  });

app.service('TraceExplorerService', function ($q, $rootScope, DataikuAPI, Dialogs, PluginsService, DataikuCloudService, $window, ActivityIndicator, SemanticVersionService) {
    const traceExplorerPluginId = 'traces-explorer';
    const requiredPluginVersion = '1.0.4';

    const buildTraceExplorerUrl = (traceExplorerWebApp, trace) => {
        return DataikuAPI.webapps
            .getPublicInfo(traceExplorerWebApp.projectKey, traceExplorerWebApp.webAppId)
            .then((resp) => {
                const publicInfo = resp.data;
                localStorage.setItem('ls.llm.traceExplorer.trace', JSON.stringify(trace));
                let baseUrl = '';
                if ($rootScope.appConfig.dssExternalURL) {
                    baseUrl += $rootScope.appConfig.dssExternalURL;
                }
                baseUrl += '/webapps/';

                const vanityOrId = publicInfo.securityInfo.vanityURL || `${publicInfo.webapp.projectKey}/${publicInfo.webapp.id}`;
                return `${baseUrl}${vanityOrId}/`;
            })
            .catch(() => {
                const message = `The "Trace Explorer" Web App instance ${traceExplorerWebApp.webAppId} in the project ${traceExplorerWebApp.projectKey} does not exist or you don't have read access to the project. Please contact your administrator.`;
                Dialogs.error($rootScope, 'Permission required', message);
                return $q.reject();
            });
    };

    const handlePluginInstalled = (trace) => {
        return PluginsService.getPluginVersion(traceExplorerPluginId).then((pluginVersion) => {
            if (SemanticVersionService.compareVersions(pluginVersion, requiredPluginVersion) < 0) {
                const message = `The currently installed "Trace Explorer" plugin (version ${pluginVersion}) does not support this feature. To proceed, the plugin must be updated to version ${requiredPluginVersion} or newer. Please contact your administrator.`;
                Dialogs.error($rootScope, 'Plugin update required', message);
                return $q.reject();
            }
            const traceExplorerDefaultWebApp = $rootScope.appConfig?.llmSettings?.traceExplorerDefaultWebApp;

            if (!traceExplorerDefaultWebApp?.projectKey || !traceExplorerDefaultWebApp?.webAppId) {
                const message = '"Trace Explorer" plugin is installed. To finalize the setup, <doclink page="/agents/tracing#accessing-via-explore-trace-shortcut" title="read the documentation"/> or contact your administrator.';
                Dialogs.errorUnsafeHTML($rootScope, 'Configuration required', message);
                return $q.reject();
            }

            return buildTraceExplorerUrl(traceExplorerDefaultWebApp, trace).then((traceExplorerUrl) => {
                $window.open(`${traceExplorerUrl}?readTraceFromLS=true`, '_blank');
            });
        });
    };

    const handlePluginNotInstalled = () => {
        let cloudInfo;

        return DataikuCloudService.getCloudInfo()
            .then((info) => {
                cloudInfo = info;
                let installMessage = 'Enhance your experience by installing the "Trace Explorer" plugin. You can ';
                if (cloudInfo.isDataikuCloud) {
                    installMessage += cloudInfo.isSpaceAdmin ? 'install it on the Launchpad.' : 'ask your administrator to install it.';
                } else {
                    installMessage += `${$rootScope.appConfig.admin ? 'install' : 'request'} it from the store.`;
                }
                return Dialogs.confirm($rootScope, 'Unlock Trace Exploration', installMessage, { positive: true, btnConfirm: 'OK' });
            })
            .then(() => {
                return PluginsService.getPluginStoreLink(cloudInfo, traceExplorerPluginId);
            })
            .then((pluginStoreUrl) => {
                $window.open(pluginStoreUrl, '_blank');
            });
    };

    this.openTraceExplorer = (trace) => {
        PluginsService.checkInstalledStatus('traces-explorer')
            .then((isPluginInstalled) => {
                if (isPluginInstalled) {
                    return handlePluginInstalled(trace);
                } else {
                    return handlePluginNotInstalled();
                }
            })
            .catch((error) => {
                if (error && error.message) {
                    ActivityIndicator.error(error.message, 8000);
                }
            });
    };
});

})();
