(function() {
    'use strict';

    angular.module('dataiku.savedmodels').factory('SavedModelHelperService', helperService);

    function helperService() {
        function suggestedVersionId(smStatus) {
            const versionCount = smStatus.versions.length;
            const biggerVersion = smStatus.versions.reduce((acc, el) => {
                const match = el.versionId.match(/\d+$/);
                if (match && parseInt(match[0]) > acc) {
                    acc = parseInt(match[0]);
                }
                return acc;
            }, 0);
    
            let result = 'v'
            if (biggerVersion > versionCount) {
                result += biggerVersion + 1;
            } else {
                result += versionCount + 1;
            }
    
            return result;
        }

        /**
         * Generates an ASCII tree representation of the trace data.
         * @param {object} data - The trace data object.
         * @returns {string} The ASCII string.
         */
        function generateAsciiTreeFromTrace(data) {
            if (data === undefined) {
                return "";
            }

            let result = "";

            function getNodeDisplay(node) {
                let emoji = "";
                let objectName = "";
                const nameLower = node.name.toLowerCase();
                if (nameLower === "agent") {
                    emoji = "🤖 ";
                } else if (nameLower === "tools") {
                    emoji = "🛠️ ";
                } else if (nameLower === "dku_llm_mesh_llm_call" || nameLower === "dku_llm_mesh_llm_call_streamed") {
                    emoji = "💬 ";
                    objectName = node.attributes["llmId"];
                } else if (nameLower === "dku_llm_mesh_embedding_query") {
                    objectName = node.attributes["llmId"];
                } else if (nameLower === "python_agent_tool_call") {
                    emoji = "🐍️ ";
                    objectName = node.attributes["class"];
                } else if (nameLower === "python_agent_mcp_subtool_call") {
                    emoji = "📡 ";
                    objectName = node.attributes["subtool_name"];
                }

                const durationInfo = node.duration !== undefined ? ` (${node.duration}ms)` : "";
                const objectInfo = objectName ? `: ${objectName}`: "";
                return `${emoji}${node.name}${objectInfo}${durationInfo}`;
            }

            // Recursive function to process each node in the tree
            function processNode(node, prefix, isLast) {
                // Determine the appropriate connector symbols based on position
                const connector = isLast ? "└─" : "├─";
                result += `${prefix}${connector} ${getNodeDisplay(node)}\n`;

                // Prepare the prefix for the next level of children
                const childPrefix = prefix + (isLast ? "    " : "│   ");
                const children = node.children || [];

                // Recursively call the function for each child
                children.forEach((child, index) => {
                    const isLastChild = index === children.length - 1;
                    processNode(child, childPrefix, isLastChild);
                });
            }

            // Start processing from the root node
            result += `${getNodeDisplay(data)}\n`;
            const children = data.children || [];
            children.forEach((child, index) => {
                const isLastChild = index === children.length - 1;
                processNode(child, "", isLastChild);
            });

            return result;
        }

        return { suggestedVersionId, generateAsciiTreeFromTrace };
    }
})();
