(function() {
    'use strict';

    const promptChat = {
        bindings: {
            initialMessages: '<',
            lastMessageId: '<',
            sessionId: '<',
            availableLlms: '<',
            hasActiveLlm: '<',
            warnAboutRaLlmSearchMode: '<',
            activeLlmIcon: '<',
            emptyStateMessage: '<',
            onBeforeSendMessage: '&?',
            onSendMessage: '&',
            onForkMessage: '&?',
            onStopStreaming: '&',
            onChatCompletion: '&',
            processResponse: '&?'
        },
        templateUrl: '/static/dataiku/shared/components/prompt-chat/prompt-chat.component.html',
        controller: function($rootScope, $scope, $timeout, PromptUtils, PromptChatService, TraceExplorerService, ClipboardUtils, Logger, $dkuSanitize) {
            const $ctrl = this;
            const entryElId = 'prompt-chat-entry';

            $ctrl.chatSessions = {};
            $ctrl.message = '';
            $ctrl.messageBranch = [];
            $ctrl.messages = {};
            $ctrl.iconSize = 16;
            $ctrl.editedMessage = '';
            $ctrl.editedMessageId = '';
            
            $ctrl.$onChanges = (changes) => {
                // don't reload the messages if the active session is still streaming
                if (changes.initialMessages) {
                    const activeSession = $ctrl.chatSessions[$ctrl.sessionId];
                    // if we come back to this chat session and it's still streaming, be sure the last user message is manually added back
                    if (activeSession?.currentlyStreaming) {
                        $ctrl.initialMessages[activeSession.lastUserMessage.id] = activeSession.lastUserMessage;
                        $ctrl.lastMessageId = activeSession.lastUserMessage.id;
                    }
                    if (changes.lastMessageId?.currentValue !== changes.lastMessageId?.previousValue) {
                        prepareMessages($ctrl.initialMessages, $ctrl.lastMessageId);
                    }
                }
                if (changes.sessionId) {
                    clearMessage();
                }
            };

            $scope.$on('prompt-chat--resetChat', function (event, data) {
                resetChat();
            });

            function getChatHandler(runData, isRerun) {
                const activeSessionId = $ctrl.sessionId;
                $ctrl.chatSessions[activeSessionId] = {
                    ...($ctrl.chatSessions[activeSessionId] || {}),
                    streamedMessage: '',
                    rawMessage: '',
                    currentlyStreaming: true,
                    useMarkdown: true,
                    streamingSupported: true,
                    streamingNotSupportedMessage: '',
                    abortController: new AbortController(),
                    runData,
                    isRerun,
                };
                const activeSession = $ctrl.chatSessions[activeSessionId];

                const CHAR_LIMIT_FOR_MARKDOWN = 500;
                const THROTTLE_MIN_LENGTH = 10000;
                let previousLength = 0;
                const callback = chunk => {
                    scrollChat(activeSessionId, false);

                    const event = chunk['event'];
                    const data = chunk['data'];
                    switch(event) {
                        case 'completion-chunk': {
                            if (data.text) {
                                const text = data.text;
                                activeSession.rawMessage += text;
                                let currentLength = activeSession.rawMessage.length;
                                const convertToMarkdown = currentLength < THROTTLE_MIN_LENGTH || (currentLength > THROTTLE_MIN_LENGTH && currentLength > previousLength + CHAR_LIMIT_FOR_MARKDOWN);
                                // throttle text streaming if the string is too long
                                if (activeSession.useMarkdown && convertToMarkdown) {
                                    // if there is an error parsing the HTML (e.g., LLM is spitting out garbage text), stop using markdown
                                    try {
                                        activeSession.streamedMessage = $dkuSanitize(marked(activeSession.rawMessage, {
                                            breaks: true
                                        }));
                                        $scope.$apply();
                                        previousLength = currentLength;
                                    } catch(e) {
                                        Logger.error('Error parsing markdown HTML, switching to plain text', e);
                                        activeSession.useMarkdown = false;
                                    }
                                }

                                if (!activeSession.useMarkdown) {
                                    activeSession.streamedMessage = activeSession.rawMessage;
                                    $scope.$apply();
                                }

                                scrollChat(activeSessionId, false);
                            }
                            break;
                        } case 'completion-response': {
                            activeSession.streamedMessage = '';
                            activeSession.currentlyStreaming = false;
                            activeSession.isRerun = false;

                            // only add the message if we are actually still on the same session
                            if ($ctrl.sessionId === activeSessionId) {
                                onChatCompletion($ctrl.processResponse ? $ctrl.processResponse({ response: data }) : data, activeSessionId);
                                $timeout(() => {
                                    scrollChat(activeSessionId, false);
                                });
                            }
                            break;
                        } case 'no-streaming': {
                            activeSession.streamingSupported = false;
                            activeSession.streamingNotSupportedMessage = data["text"];
                            $scope.$apply();
                            break;
                        } case 'completion-end': {
                            // ignore this chunk
                            break;
                        } default: {
                            Logger.error('Unknown chunk event: ' + event);
                        }
                    }
                };

                $timeout(() => {
                    scrollChat(activeSessionId, true);
                });

                return {
                    callback,
                    activeSession
                }
            };

            function onSendMessage(newMessage, chatHandlerParams, isRerun, isEdit) {
                const isEditOrRerun = isRerun || isEdit;
                if (isEditOrRerun && $ctrl.onBeforeSendMessage) {
                    $ctrl.onBeforeSendMessage({isEditOrRerun: isEditOrRerun});
                }
                // if chat is streaming and we switch to another chat then come back, store the user message so it's not lost
                $ctrl.chatSessions[$ctrl.sessionId] = {
                    ...($ctrl.chatSessions[$ctrl.sessionId] || {}),
                    lastUserMessage: newMessage,
                    defaultMessage: '' // reset default message if applicable
                };
                const { callback, activeSession } = getChatHandler(chatHandlerParams, isRerun);
                $ctrl.onSendMessage({ newMessage, abortController: activeSession.abortController, messages: $ctrl.messages, callback });
            }
            // used for auto textarea resizing
            function setMessageField(el) {
                el.parentNode.dataset.replicatedValue = el.value;
            };

            $ctrl.sendMessage = () => {
                if ($ctrl.onBeforeSendMessage) {
                    $ctrl.onBeforeSendMessage();
                }
                // create root message if first message
                if (!$ctrl.lastMessageId) {
                    const rootMessage = {
                        id: generateRandomId(7),
                        version: 0,
                        error: false,
                    };
                    $ctrl.lastMessageId = rootMessage.id;
                    $ctrl.messages[rootMessage.id] = rootMessage;
                    $ctrl.messageBranch.push(rootMessage);
                }
                const newMessage = {
                    parentId: $ctrl.lastMessageId,
                    id: generateRandomId(7),
                    runBy: $rootScope.appConfig.login,
                    message: {
                        role: 'user',
                        content : $ctrl.message
                    },
                    version: 0
                };
                $ctrl.messages[newMessage.id] = newMessage;
                $ctrl.messageBranch.push(newMessage);
                $ctrl.streamParentId = $ctrl.lastMessageId;
                onSendMessage(newMessage, { version: 0, parentId: newMessage.id })
                clearMessage();
            };

            $ctrl.sendEditedMessage = (parentId) => {
                const newMessage = {
                    parentId,
                    id: generateRandomId(7),
                    runBy: $rootScope.appConfig.login,
                    message: {
                        role: 'user',
                        content : $ctrl.editedMessage
                    }
                };
                $ctrl.messageBranch = PromptChatService.buildCurrentChatBranch($ctrl.messages, parentId);
                $ctrl.messageBranch.push(newMessage);
                $ctrl.streamParentId = parentId;
                onSendMessage(newMessage, null, false, true);
                $ctrl.clearEditedMessage();
            };

            $ctrl.forkMessage = (userMessage) => {
                $ctrl.onForkMessage({
                    sessionId: $ctrl.sessionId,
                    userMessage
                }).then(newSessionId => {
                    $ctrl.chatSessions[newSessionId] = {
                        ...($ctrl.chatSessions[newSessionId] || {}),
                        defaultMessage: userMessage.message.content
                    };
                    // for timing purposes, sometimes clearMessage in onChanges gets before fork is done
                    clearMessage();
                });
            };

            $ctrl.rerunResponse = (message) => {
                const parentId = message.parentId;
                const newMessage = $ctrl.messages[parentId];
                $ctrl.messageBranch = PromptChatService.buildCurrentChatBranch($ctrl.messages, parentId);
                onSendMessage(newMessage, { version: message.version + 1, parentId }, true);
            };

            $ctrl.clearEditedMessage = () => {
                $ctrl.editedMessage = '';
                $ctrl.editedMessageId = '';
            };

            $ctrl.cannotSendMessage = () => {
                return $ctrl.cannotRunChat() || $ctrl.message.trim() === '' || $ctrl.hasError() || $ctrl.editedMessageId;
            };

            $ctrl.hasError = () => {
                return $ctrl.messageBranch.some(x => x.error);
            };

            $ctrl.cannotSendEditedMessage = () => {
                return $ctrl.cannotRunChat() || $ctrl.editedMessage.trim() === '';
            };

            $ctrl.cannotRerunMessage = () => {
                return $ctrl.cannotRunChat();
            };

            $ctrl.isStreaming = () => {
                return $ctrl.chatSessions[$ctrl.sessionId] && $ctrl.chatSessions[$ctrl.sessionId].currentlyStreaming;
            };

            $ctrl.cannotRunChat = () => {
                return !$ctrl.hasActiveLlm || $ctrl.isStreaming();
            };

            $ctrl.switchBranch = (message, position) => {
                const parent = $ctrl.messages[message.parentId];
                let nextMessage = $ctrl.messages[parent.childrenIds[position]];
                while (nextMessage.childrenIds && nextMessage.childrenIds.length) {
                    nextMessage = $ctrl.messages[nextMessage.childrenIds[0]];
                }
                $ctrl.lastMessageId = nextMessage.id;
                $ctrl.messageBranch = PromptChatService.buildCurrentChatBranch($ctrl.messages, $ctrl.lastMessageId);
            };

            $ctrl.onKeydown = (event, sendEdited, parentId) => {
                if (event.key === 'Enter') {
                    if (!event.shiftKey) {
                        event.preventDefault();
                        if (sendEdited) {
                            !$ctrl.cannotSendEditedMessage() && $ctrl.sendEditedMessage(parentId);
                        } else {
                            !$ctrl.cannotSendMessage() && $ctrl.sendMessage();
                        }
                    }
                }
            };

            // note: this function assumes there is only one message part
            $ctrl.editMessage = (message) => {
                $ctrl.editedMessageId = message.id;
                $ctrl.editedMessage = message.message.content;
                $timeout(() => {
                    const editEl = document.getElementById('prompt-chat-edit-entry');
                    editEl.focus();
                    setMessageField(editEl);
                });
            };

            $ctrl.getLLMIconName = (llmStructuredRef) => {
                return PromptUtils.getLLMIcon(llmStructuredRef.type, $ctrl.iconSize);
            };

            $ctrl.getLLMIconStyle = (llmStructuredRef) => {
                return PromptUtils.getLLMColorStyle(llmStructuredRef.id, $ctrl.availableLlms);
            };

            $ctrl.getLLMFriendlyName = (llmStructuredRef) => {
                return llmStructuredRef.friendlyName || ($ctrl.availableLlms || []).find(llm => llm.id === llmStructuredRef.id)?.friendlyName;
            };

            $ctrl.copyTraceToClipboard = function(message) {
                ClipboardUtils.copyToClipboard(JSON.stringify(message.fullTrace, null, 2), "Trace copied to clipboard");
            };
            $ctrl.copyMessageToClipboard = function(message) {
                ClipboardUtils.copyToClipboard(message.error ? message.llmError : message.message.content, "Message copied to clipboard");
            };
            
            $ctrl.openTraceExplorer = function (message) {
                TraceExplorerService.openTraceExplorer(message.fullTrace);
            };

            $ctrl.stopStreaming = function() {
                const sessionId = $ctrl.sessionId;
                $ctrl.chatSessions[sessionId]?.abortController.abort();
                $timeout(function() {
                    const promise = $ctrl.onStopStreaming({ sessionId, activeSession: $ctrl.chatSessions[sessionId], messages: $ctrl.messages }) || Promise.resolve();
                    promise.then((response) => {
                        if (response) {
                            onChatCompletion(response);
                            $timeout(() => {
                                scrollChat(sessionId, false);
                            });
                            $ctrl.chatSessions[$ctrl.sessionId] = {
                                ...($ctrl.chatSessions[sessionId] || {}),
                                streamedMessage: '',
                                currentlyStreaming: false
                            };
                        }
                    }).catch(setErrorInScope.bind($scope));
                }, 600)
            };

            // Called if prompt chat kernel is manually stopped outside of prompt chat component
            $rootScope.$on('llmStopDevKernel', (_, sessionId) => {
                if ($ctrl.chatSessions[sessionId]?.currentlyStreaming) {
                    $ctrl.stopStreaming();
                }
            });

            $ctrl.$onDestroy = () => {
                PromptChatService.clearLogs();
            };

            function onChatCompletion(response) {
                $ctrl.lastMessageId = response.lastMessageId;
                prepareMessages(response.chatMessages, response.lastMessageId);
                if (response.log) {
                    PromptChatService.setLog($ctrl.sessionId, response.log);
                }

                $ctrl.onChatCompletion({ 
                    lastMessageId: response.lastMessageId,
                    messages: response.chatMessages 
                });
            }

            function clearMessage() {
                $ctrl.message = $ctrl.chatSessions[$ctrl.sessionId]?.defaultMessage || '';
                setMessageField(document.getElementById(entryElId));
            }

            function resetChat() {
                $ctrl.messages = {};
                $ctrl.messageBranch = [];
                $ctrl.lastMessageId = null;
                PromptChatService.setLog($ctrl.sessionId);
            }

            function scrollChat(activeSessionId, forceScroll) {
                // only scroll if the active session is streaming
                if ($ctrl.sessionId === activeSessionId) {
                    const containerEl = document.getElementById('prompt-chat');
                    if (containerEl) {
                        const SCROLL_BUFFER = 50; // don't autoscroll if more than SCROLL_BUFFER pixels from bottom
                        // only scroll to bottom if forceScroll is true or if we are already at the bottom
                        if (forceScroll || (containerEl.scrollTop >= containerEl.scrollHeight - containerEl.offsetHeight - SCROLL_BUFFER)) {
                            containerEl.scrollTop = containerEl.scrollHeight;
                        }
                    }
                }
            }

            function prepareMessages(messages, lastMessageId) {
                $ctrl.messages = PromptChatService.enrichChatMessages(messages);
                $ctrl.messageBranch = PromptChatService.buildCurrentChatBranch($ctrl.messages, lastMessageId);
                document.getElementById(entryElId).focus();
                $timeout(() => {
                    const containerEl = document.getElementById('prompt-chat');
                    containerEl.scrollTop = containerEl.scrollHeight;
                });
            }
        }
    };

    angular.module('dataiku.shared').component('promptChat', promptChat);
})();
