import logging

from dataiku.core import doctor_constants
from dataiku.core.base import PartitionEscaper
from dataiku.core.dkujson import loads


logging.basicConfig(level=logging.INFO, format='%(asctime)s %(levelname)s %(message)s')


def pred_to_dict_timeseries(pred_df, predictor_params):
    """
    Convert prediction dataframe to a list of com.dataiku.lambda.model.api.PredictionResponse.TimeseriesForecastingResponseItem

    :param predictor_params: Predictor parameters
    :param pred_df: dataframe returned by the predictor. Contains only future data (i.e. time series forecast)
    """
    from dataiku.doctor.timeseries.utils import build_quantile_column_name
    from dataiku.doctor.timeseries.utils import timeseries_iterator
    from dataiku.doctor.timeseries.utils import SINGLE_TIMESERIES_IDENTIFIER
    from dataiku.doctor.timeseries.utils import FORECAST_COLUMN

    logging.info('pre to_dict: %s', pred_df)

    partitioning_params = predictor_params.core_params.get("partitionedModel", {})
    if partitioning_params.get("enabled", False):
        dimension_names = partitioning_params.get("dimensionNames", [])
        dimension_types = partitioning_params.get("dimensionTypes", [])
    else:
        dimension_names = None
        dimension_types = None

    dicts = []

    # Convert date time column to ISO 8601 format
    time_variable = predictor_params.core_params[doctor_constants.TIME_VARIABLE]
    pred_df[time_variable] = pred_df[time_variable].dt.strftime('%Y-%m-%dT%H:%M:%S.%fZ')

    for timeseries_identifier, df_of_timeseries_identifier in timeseries_iterator(
            pred_df, predictor_params.core_params[doctor_constants.TIMESERIES_IDENTIFIER_COLUMNS]):
        for row in df_of_timeseries_identifier.to_dict(orient="records"):
            # TODO @timeseries add past data to forecast ?
            forecast_response_item = {
                "time": str(row[time_variable]),
                "partitionId": PartitionEscaper.build_partition_id(
                    [row[dimension] for dimension in dimension_names], dimension_names, dimension_types
                ) if dimension_names else None,
                "forecast": row[FORECAST_COLUMN],
                "quantiles": predictor_params.core_params[doctor_constants.QUANTILES],
                "quantilesValues": [
                    row[build_quantile_column_name(q)]
                    for q in predictor_params.core_params[doctor_constants.QUANTILES]
                ],
                "ignored": False,
            }
            if timeseries_identifier != SINGLE_TIMESERIES_IDENTIFIER:
                forecast_response_item["timeseriesIdentifier"] = loads(timeseries_identifier)
            dicts.append(forecast_response_item)

    return dicts
