import os
import datetime, time
import subprocess
from pathlib import Path
from itertools import chain

from cleaning import delete_and_report
from dataiku.runnables import Runnable


class MyRunnable(Runnable):
    def __init__(self, project_key, config, plugin_config):
        self.project_key = project_key
        self.config = config

    def get_progress_target(self):
        return (100, 'NONE')

    def run(self, progress_callback):
        # agent logs can be in
        # $DIP_HOME/saved_models/$PROJECTKEY/$MODELID/versions/$VERSIONID/logs/dev-runs/$DATE/llm.log
        # $DIP_HOME/saved_models/$PROJECTKEY/$MODELID/versions/$VERSIONID/logs/runs/python-llm-$PROJECTKEY-$RANDOMSTRING/llm.log[.1,.2]
        self.dip_home = os.environ['DIP_HOME']
        self.perform_deletion = self.config.get("performDeletion", False)
        self.maximum_age = int(self.config.get("age", 15))
        self.maximum_timestamp = int(time.mktime((datetime.datetime.now() - datetime.timedelta(days=self.maximum_age)).timetuple()))
        
        if self.config.get('allProjects', False):
            self.project_keys = [project_key for project_key in os.listdir(os.path.join(self.dip_home, 'saved_models'))]
        else:
            self.project_keys = [self.project_key]

        self.to_delete = []
        for project_key in self.project_keys:
            project_saved_model_folder = Path(self.dip_home) / 'saved_models' / project_key

            # Step 1: find agent logs directories
            runs_dirs = project_saved_model_folder.rglob('**/runs')
            dev_runs_dirs = project_saved_model_folder.rglob('**/dev-runs')
            parent_run_dirs = chain(runs_dirs, dev_runs_dirs)

            candidate_folders = []
            for parent_dir in parent_run_dirs:
                if parent_dir.is_dir():
                    candidate_folders.extend([
                        run_folder for run_folder in parent_dir.iterdir() if run_folder.is_dir()
                    ])

            if not candidate_folders:
                continue

            # Step 2: for each candidate folder, check if all its contents are old and not locked
            for folder in sorted(candidate_folders):
                is_folder_deletable = True

                for item in folder.rglob('*'):
                    if not item.is_file() or not str(item).endswith(".log"):
                        continue

                    # check that the file is old enough
                    if item.stat().st_mtime >= self.maximum_timestamp:
                        is_folder_deletable = False
                        break

                    lock_file_is_stale = False
                    lock_file_path = None

                    # check that the file is not being used by a kernel
                    try:
                        lock_file_path = str(item) + ".lock"
                        with open(lock_file_path, 'r') as f:
                            pid = int(f.read().strip())
                    except FileNotFoundError:
                        continue
                    except (IOError, ValueError):
                        lock_file_is_stale = True
                        print("failed to read pid from lock file")
                    else:
                        if os.name == 'posix':
                            try:
                                os.kill(pid, 0)  # sending signal 0 to a PID checks for its existence without harming it
                                print("the process referenced by a lock file is alive")
                            except OSError:
                                lock_file_is_stale = True
                        elif os.name == 'nt':
                            try:
                                command = ['tasklist', '/fi', f'PID eq {pid}']
                                output = subprocess.check_output(command, stderr=subprocess.DEVNULL).decode()
                                if str(pid) not in output:
                                    lock_file_is_stale = True
                            except (subprocess.CalledProcessError, FileNotFoundError):
                                lock_file_is_stale = True

                    # ignore lock files for dead processes
                    if lock_file_is_stale:
                        if self.perform_deletion and lock_file_path:
                            os.remove(lock_file_path)
                    else:
                        is_folder_deletable = False
                        break

                if is_folder_deletable:
                    self.to_delete.append([project_key, folder.relative_to(project_saved_model_folder)])

        html = delete_and_report(self.to_delete, os.path.join(self.dip_home, 'saved_models'), progress_callback, self.perform_deletion, 'agents', ['Project', 'Logs directory'])
        return html
