import dataiku
from dataiku.runnables import Runnable
from dataikuapi.utils import DataikuException
from datetime import datetime as dt
from datetime import timedelta
from dataiku.base.utils import safe_unicode_str
from functools import reduce


class MyRunnable(Runnable):
    """The base interface for a Python runnable"""

    def __init__(self, project_key, config, plugin_config):
        """
        :param project_key: the project in which the runnable executes
        :param config: the dict of the configuration of the object
        :param plugin_config: contains the plugin settings
        """
        super(MyRunnable, self).__init__(project_key, config, plugin_config)
        self.config = config
        self.perform_deletion = self.config.get("performDeletion", False)
        self.all_projects = self.config.get("allProjects", False)
        self.project = dataiku.api_client().get_project(project_key)
        if not config.get('age') >= 0:
            raise ValueError('Invalid number of days upon which older samples will be deleted, minimum 0.')

    def get_progress_target(self):
        return 100, 'NONE'

    def run(self, progress_callback):
        """
        This method first identifies which model comparison samples will be deleted and which will remain.
        It builds a summary of the actions for the user.
        If perform_deletion param is set to True, the samples in samples_to_delete will be deleted.
        """
        min_days = int(self.config.get('age')) - 1
        samples_raw = self.project.client._perform_json("GET", "/projects/%s/modelcomparisons/llm/list-samples" % self.project_key, params={"allProjects": self.all_projects})

        samples = sorted(samples_raw, key=lambda k: k['created'], reverse=True)

        samples_to_delete = list(filter(filter_old_samples(min_days), samples))
        samples_to_keep = [sample for sample in samples if sample not in samples_to_delete]

        if self.all_projects:
            html = "<h4>Summary for all projects</h4>"
        else:
            html = "<h4>Summary for project {}</h4>".format(self.project_key)
        html += "<span>{}</span><br><br>".format(summarize(samples_to_keep, 'to keep'))
        html += "<span>{}</span>".format(summarize(samples_to_delete, 'to delete'))

        if self.perform_deletion:
            try:
                nb_samples_deleted = self.project.client._perform_json("DELETE", "/projects/%s/modelcomparisons/llm/clean-sample-cache" % self.project_key, params={"allProjects": self.all_projects, "minDays": min_days})
                html += "<br/><br/>=> <span><strong>{} sample(s) deleted</strong></span>".format(nb_samples_deleted)
            except DataikuException as e:
                html += '<br/><span>An error occurred while trying to delete samples.</span><br>'
                html += u'<span>{}</span>'.format(safe_unicode_str(e))
        return html


def filter_old_samples(min_days):
    def ret(samples):
        today = dt.now().date()
        date = dt.utcfromtimestamp(samples['created']/1000).date()
        return today - date > timedelta(days=min_days)
    return ret


def summarize(samples, action):
    nb_samples, size_won = get_nb_samples_and_size(samples)

    if nb_samples == 0:
        text = "<strong>0 sample {}</strong>".format(action)
    else:
        text = "<strong>{} samples {}.</strong> Total size: <strong>{}Mo</strong>".format(nb_samples, action, size_won)
    return text


def get_nb_samples_and_size(samples):
    nb_samples = len(samples)
    size = reduce(lambda total, spl: total + (spl['size'] / 1e6), samples, 0)
    return nb_samples, round(size, 2)
