/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.shaker.processors.geo;

import com.dataiku.dip.datalayer.Column;
import com.dataiku.dip.datalayer.Processor;
import com.dataiku.dip.datalayer.Row;
import com.dataiku.dip.datalayer.SingleInputSingleOutputRowProcessor;
import com.dataiku.dip.datalineage.DatasetPairLineage;
import com.dataiku.dip.datalineage.RecipeLineage;
import com.dataiku.dip.exceptions.IllegalConfigurationException;
import com.dataiku.dip.pivot.backend.sql.utils.DoubleUtils;
import com.dataiku.dip.shaker.model.ProcessorScriptStep;
import com.dataiku.dip.shaker.model.StepParams;
import com.dataiku.dip.shaker.processors.Category;
import com.dataiku.dip.shaker.processors.ProcessorMeta;
import com.dataiku.dip.shaker.processors.ProcessorTag;
import com.dataiku.dip.shaker.server.ProcessorDesc;
import com.dataiku.dip.shaker.text.Labelled;
import com.dataiku.dip.shaker.types.GeoPoint;
import com.dataiku.dip.util.ParamDesc;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.Pair;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Sets;
import java.util.Objects;
import java.util.Set;
import org.apache.commons.lang.StringUtils;

public class GeoPointBufferProcessor
extends SingleInputSingleOutputRowProcessor
implements Processor {
    public static final double MILES_TO_KM = 1.6093444978925633;
    public static final ProcessorMeta<GeoPointBufferProcessor, Parameter> META = new ProcessorMeta<GeoPointBufferProcessor, Parameter>(){

        @Override
        public String getName() {
            return "GeoPointBufferProcessor";
        }

        @Override
        public String getDocPage() {
            return "geopoint-buffer";
        }

        @Override
        public Category getCategory() {
            return Category.TRANSFORMATION;
        }

        @Override
        public Set<ProcessorTag> getTags() {
            return Sets.newHashSet((Object[])new ProcessorTag[]{ProcessorTag.GEOGRAPHIC});
        }

        @Override
        public String getHelp(String language) {
            return this.translate(language, "SHAKER.PROCESSOR.GeoPointBufferProcessor.HELP", "This processor creates buffer polygons around geopoints. For each input geospatial point, a spatial polygon is created around it, delimiting the area of influence covered by the point (all the points that fall within a given distance from a geopoint). The shape area of the polygon can be either rectangular or circular (using an approximate) and the size will depend on the selected parameters.\n \n# Input column\nContains the geopoints on which the spatial polygon is centered\n \n# Output column\nContains created polygons in the WKT format\n\n# Polygon creation options\nSelect the shape of the polygon from:\n* Rectangular\n* Circular\n\nSelect the unit of distances from:\n* Kilometers\n* Miles\n\nIf Rectangle shape is selected:\nSelect the Width and Height of the Rectangle shape to compute.\n\nIf Circle shape is selected:\nSelect the Radius of the Circle shape to compute.\n\nEach distance is expressed according to the input unit.\n\n");
        }

        @Override
        public Class<Parameter> stepParamClass() {
            return Parameter.class;
        }

        @Override
        public ProcessorDesc describe(String language) {
            return ProcessorDesc.withCustomForm(this.getName(), this.translate(language, "SHAKER.PROCESSOR.GeoPointBufferProcessor.DESCRIPTION", 1.actionVerb("Create") + " area around geopoint")).withMNEColParam("inputColumn", this.translate(language, "SHAKER.PROCESSOR.GeoPointBufferProcessor.DESCRIPTION.INPUT_COLUMN", "Input column")).withColParam("outputColumn", this.translate(language, "SHAKER.PROCESSOR.GeoPointBufferProcessor.DESCRIPTION.OUTPUT_COLUMN", "Output column")).withParam(ParamDesc.advancedSelect("shapeMode", this.translate(language, "SHAKER.PROCESSOR.GeoPointBufferProcessor.DESCRIPTION.SHAPE_MODE", "Shape"), "", ShapeMode.class).withDefaultValue(ShapeMode.CIRCLE)).withParam(ParamDesc.advancedSelect("unitMode", this.translate(language, "SHAKER.PROCESSOR.GeoPointBufferProcessor.DESCRIPTION.UNIT_MODE", "Distance unit"), "", UnitMode.class).withDefaultValue(UnitMode.KILOMETERS)).withParam(ParamDesc.doubleP("radius", this.translate(language, "SHAKER.PROCESSOR.GeoPointBufferProcessor.DESCRIPTION.RADIUS", "Radius (For circle shape)"))).withParam(ParamDesc.doubleP("width", this.translate(language, "SHAKER.PROCESSOR.GeoPointBufferProcessor.DESCRIPTION.WIDTH", "Width  (For rectangle shape)"))).withParam(ParamDesc.doubleP("height", this.translate(language, "SHAKER.PROCESSOR.GeoPointBufferProcessor.DESCRIPTION.HEIGHT", "Height  (For rectangle shape)")));
        }

        @Override
        public GeoPointBufferProcessor build(Parameter parameter) throws Exception {
            return new GeoPointBufferProcessor(parameter);
        }

        @Override
        public RecipeLineage getUpdatedRecipeLineage(ProcessorScriptStep pss, RecipeLineage previousRecipeLineage) {
            if (!(pss.params instanceof Parameter)) {
                throw new IllegalArgumentException("Unsupported param type:" + pss.params.getClass().getSimpleName());
            }
            Parameter bufferParams = (Parameter)pss.params;
            if (StringUtils.isBlank((String)bufferParams.inputColumn)) {
                throw new IllegalConfigurationException("Missing input column information for lineage on the geopoint buffer processor.");
            }
            RecipeLineage updatedRecipeLineage = new RecipeLineage();
            previousRecipeLineage.getDatasetPairLineages().forEach((datasetPair, previousDatasetPairLineage) -> {
                DatasetPairLineage updatedDatasetPairLineage = new DatasetPairLineage((DatasetPairLineage)previousDatasetPairLineage);
                if (StringUtils.isNotBlank((String)bufferParams.outputColumn) && !Objects.equals(bufferParams.inputColumn, bufferParams.outputColumn)) {
                    updatedDatasetPairLineage.removeRelationsOnColumn(bufferParams.outputColumn);
                    updatedDatasetPairLineage.addFactorizedColumnRelations(bufferParams.inputColumn, bufferParams.outputColumn);
                }
                updatedRecipeLineage.setDatasetPairLineage((Pair<String, String>)datasetPair, updatedDatasetPairLineage);
            });
            return updatedRecipeLineage;
        }
    };
    private final Parameter params;
    private Column outputColumn;
    private Column cd;
    private AreaGenerator generator;
    static final double DEGREES_TO_RADIANS = Math.PI / 180;
    static final double RADIANS_TO_DEGREES = 57.29577951308232;
    static final double EARTH_RADIUS = 6378.137;
    private static DKULogger logger = DKULogger.getLogger((String)"dku.shaker.geobuffer");

    public GeoPointBufferProcessor(Parameter params) {
        this.params = params;
    }

    public void processRow(Row row) throws Exception {
        String str = row.get(this.cd);
        if (str == null || str.equals("")) {
            this.getProcessorOutput().emitRow(row);
            return;
        }
        String output = null;
        GeoPoint.Coords coords = GeoPoint.convert(str);
        if (coords != null) {
            output = this.generator.generateArea(coords);
        }
        if (output != null) {
            row.put(this.outputColumn, output);
        }
        this.getProcessorOutput().emitRow(row);
    }

    public void postProcess() throws Exception {
        this.getProcessorOutput().lastRowEmitted();
    }

    public void init() throws Exception {
        this.cd = this.getCf().column(this.params.inputColumn, Processor.ProcessorRole.INPUT_COLUMN);
        this.outputColumn = !StringUtils.isBlank((String)this.params.outputColumn) ? this.getCf().column(this.params.outputColumn, Processor.ProcessorRole.OUTPUT_COLUMN) : this.cd;
        this.generator = GeoPointBufferProcessor.newGenerator(this.params.unitMode, this.params.shapeMode, this.params.radius, this.params.height, this.params.width);
    }

    @VisibleForTesting
    static AreaGenerator newGenerator(UnitMode unitMode, ShapeMode shapeMode, double radius, double height, double width) {
        logger.info((Object)"Instantiated new generator.");
        double conversionFactor = 1.0;
        if (unitMode == UnitMode.MILES) {
            logger.info((Object)"Distances are expressed in miles.");
            conversionFactor = 1.6093444978925633;
        } else {
            logger.info((Object)"Distances are expressed in kilometers.");
        }
        if (shapeMode == ShapeMode.RECTANGLE) {
            logger.info((Object)"Instantiating a Rectangular Area Generator.");
            return new RectangleAreaGenerator(width * conversionFactor, height * conversionFactor);
        }
        logger.info((Object)"Instantiating a Circular Area Generator.");
        return new CircleAreaGenerator(radius * conversionFactor);
    }

    public static GeoPoint.Coords computeDestinationPoint(double latitudeInDegree, double longitudeInDegree, double bearing, double distance) {
        double latitudeInRadian = latitudeInDegree * (Math.PI / 180);
        double longitudeInRadian = longitudeInDegree * (Math.PI / 180);
        double endLat = Math.asin(Math.sin(latitudeInRadian) * Math.cos(distance / 6378.137) + Math.cos(latitudeInRadian) * Math.sin(distance / 6378.137) * Math.cos(bearing));
        double endLong = longitudeInRadian + Math.atan2(Math.sin(bearing) * Math.sin(distance / 6378.137) * Math.cos(latitudeInRadian), Math.cos(distance / 6378.137) - Math.sin(latitudeInRadian) * Math.sin(endLat));
        return new GeoPoint.Coords(57.29577951308232 * endLat, 57.29577951308232 * endLong);
    }

    public static class Parameter
    implements StepParams {
        private static final long serialVersionUID = -1L;
        public String inputColumn;
        public String outputColumn;
        public UnitMode unitMode;
        public ShapeMode shapeMode;
        public double radius;
        public double width;
        public double height;

        public void validate() throws IllegalArgumentException {
            AreaGenerator.checkInputParams(this.unitMode, this.shapeMode, this.radius, this.height, this.width);
        }
    }

    public static abstract class AreaGenerator {
        public abstract String generateArea(GeoPoint.Coords var1);

        public static void checkInputParams(UnitMode unitMode, ShapeMode shapeMode, double radius, double height, double width) {
            if (shapeMode == null) {
                throw new IllegalArgumentException("Parameter `shapeMode` must be RECTANGLE or CIRCLE.");
            }
            switch (shapeMode) {
                case RECTANGLE: {
                    if (!(width <= 0.0) && !(height <= 0.0)) break;
                    logger.info((Object)"Rectangle Area Generator: Received invalid parameters as input.");
                    logger.infoV("Got width= {}", new Object[]{width});
                    logger.infoV("Got height= {}", new Object[]{height});
                    break;
                }
                case CIRCLE: {
                    if (!(radius <= 0.0)) break;
                    logger.info((Object)"Circle Area Generator: Received invalid parameters as input.");
                    logger.infoV("Got radius= {}", new Object[]{radius});
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Invalid processing mode: " + String.valueOf(shapeMode));
                }
            }
            if (unitMode == null) {
                throw new IllegalArgumentException("Parameter `unitMode` must be MILES or KILOMETERS.");
            }
            switch (unitMode) {
                case MILES: {
                    logger.info((Object)"Detected unit distance = MILES");
                    break;
                }
                case KILOMETERS: {
                    logger.info((Object)"Detected unit distance = KILOMETERS");
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Invalid unit distance mode: " + String.valueOf(unitMode));
                }
            }
        }
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    public static enum UnitMode implements Labelled
    {
        MILES{

            @Override
            public String getLabel() {
                return "Miles";
            }
        }
        ,
        KILOMETERS{

            @Override
            public String getLabel() {
                return "Kilometers";
            }
        };

    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    public static enum ShapeMode implements Labelled
    {
        RECTANGLE{

            @Override
            public String getLabel() {
                return "Rectangle";
            }
        }
        ,
        CIRCLE{

            @Override
            public String getLabel() {
                return "Circle";
            }
        };

    }

    public static class RectangleAreaGenerator
    extends AreaGenerator {
        double width = 0.0;
        double height = 0.0;
        double radius = 0.0;
        double[] angles = null;

        public RectangleAreaGenerator(double width, double height) {
            if (width < 0.0 || height < 0.0) {
                throw new IllegalArgumentException("The height and width of the trade area must be greater than 0. (Received height value is equal to " + height + " and width value is equal to " + width + ")");
            }
            if (DoubleUtils.equals(width, 0.0) || DoubleUtils.equals(height, 0.0)) {
                logger.info((Object)"Rectangle Area Generator: Received invalid parameters as input.");
                logger.infoV("Got width = {}", new Object[]{width});
                logger.infoV("Got height = {}", new Object[]{height});
            } else {
                this.height = height;
                this.width = width;
                this.radius = Math.sqrt(Math.pow(width / 2.0, 2.0) + Math.pow(height / 2.0, 2.0));
                double diagonalAngle = Math.atan(this.height / this.width);
                this.angles = new double[]{1.5707963267948966 - diagonalAngle, 1.5707963267948966 + diagonalAngle, 4.71238898038469 - diagonalAngle, 4.71238898038469 + diagonalAngle};
            }
        }

        @Override
        public String generateArea(GeoPoint.Coords center) {
            if (DoubleUtils.equals(this.width, 0.0) || DoubleUtils.equals(this.height, 0.0) || center == null || this.angles == null) {
                return null;
            }
            GeoPoint.Coords initCoords = GeoPointBufferProcessor.computeDestinationPoint(center.latitude, center.longitude, this.angles[0], this.radius);
            PolygonBuilder polygonBuilder = new PolygonBuilder(initCoords);
            for (int i = 1; i < this.angles.length; ++i) {
                GeoPoint.Coords tmpCoords = GeoPointBufferProcessor.computeDestinationPoint(center.latitude, center.longitude, this.angles[i], this.radius);
                polygonBuilder.lineTo(tmpCoords);
            }
            return polygonBuilder.close();
        }
    }

    public static class CircleAreaGenerator
    extends AreaGenerator {
        static final int NB_OF_EDGES = 120;
        static final double ANGLE_STEP = 0.05235987755982988;
        double radius = 0.0;

        public CircleAreaGenerator(double radius) {
            if (radius < 0.0) {
                throw new IllegalArgumentException("The radius of the trade area must be greater than 0. (Received radius value is equal to " + radius + ")");
            }
            if (DoubleUtils.equals(radius, 0.0)) {
                logger.info((Object)"Circle Area Generator: Received invalid parameters as input.");
                logger.infoV("Got radius = {}", new Object[]{radius});
            } else {
                this.radius = radius;
            }
        }

        @Override
        public String generateArea(GeoPoint.Coords center) {
            if (DoubleUtils.equals(this.radius, 0.0) || center == null) {
                return null;
            }
            GeoPoint.Coords initCoords = GeoPointBufferProcessor.computeDestinationPoint(center.latitude, center.longitude, 0.0, this.radius);
            PolygonBuilder polygonBuilder = new PolygonBuilder(initCoords);
            for (int i = 1; i < 120; ++i) {
                GeoPoint.Coords tmpCoords = GeoPointBufferProcessor.computeDestinationPoint(center.latitude, center.longitude, 0.05235987755982988 * (double)i, this.radius);
                polygonBuilder.lineTo(tmpCoords);
            }
            return polygonBuilder.close();
        }
    }

    public static class PolygonBuilder {
        final StringBuilder wktPolygon = new StringBuilder("POLYGON((");
        final GeoPoint.Coords firstPoint;

        public PolygonBuilder(GeoPoint.Coords geopoint) {
            this.wktPolygon.append(geopoint.longitude).append(" ").append(geopoint.latitude);
            this.firstPoint = geopoint;
        }

        public void lineTo(GeoPoint.Coords geopoint) {
            this.wktPolygon.append(",");
            this.wktPolygon.append(geopoint.longitude).append(" ").append(geopoint.latitude);
        }

        public String close() {
            this.lineTo(this.firstPoint);
            this.wktPolygon.append("))");
            return this.wktPolygon.toString();
        }
    }
}

