(function() {
    "use strict";

    angular.module("dataiku.autoFeatureGeneration").factory("AutoFeatureGenerationRecipeService", AutoFeatureGenerationRecipeService);

    function AutoFeatureGenerationRecipeService(RecipesUtils, DatasetUtils, NumberFormatter) {
        const RELATIONSHIP_TYPES = {
            DEFAULT: null,
            ONE_TO_MANY: 'ONE_TO_MANY',
            MANY_TO_ONE: 'MANY_TO_ONE',
            ONE_TO_ONE: 'ONE_TO_ONE'
        };

        const relationshipNumDesc = {
            'ONE_TO_MANY': '1-to-many',
            'ONE_TO_ONE': '1-to-1',
            'MANY_TO_ONE': 'many-to-1'
        }

        //Defines available relationships and their order of appearance
        const SUPPORTED_RELATIONSHIP_TYPES = [ RELATIONSHIP_TYPES.ONE_TO_ONE, RELATIONSHIP_TYPES.MANY_TO_ONE, RELATIONSHIP_TYPES.ONE_TO_MANY];

        const relationshipDesc = {
            [RELATIONSHIP_TYPES.ONE_TO_MANY]: {
                type: "One-to-many",
                description: "One row in the left dataset corresponds to many rows in the right dataset.\n" +
                    "Row-by-row computations and aggregations will be performed."
            },
            [RELATIONSHIP_TYPES.MANY_TO_ONE]: {
                type: "Many-to-one",
                description: "Many rows in the left dataset correspond to one row in the right dataset.\n" +
                    "Only row-by-row computations will be performed."
            },
            [RELATIONSHIP_TYPES.ONE_TO_ONE]: {
                type: "One-to-one",
                description: "One row in the left dataset corresponds to one row in the right dataset.\n" +
                    "Only row-by-row computations will be performed."
            },

            [RELATIONSHIP_TYPES.DEFAULT]: {
                type: "Select the relationship between join keys",
                description: "Informs how new features are generated - from one or many rows"
            }
        };

        const primaryDatasetDesc = "This is the primary dataset; original values will also be retrieved for selected columns.";

        const DEFAULT_RELATIONSHIP = {
            type: RELATIONSHIP_TYPES.DEFAULT,
        }

        const RELATIONSHIP_SYMBOL = {
            MANY: "dku-icon-relationship-many-16",
            MANY_REVERSED: "dku-icon-relationship-many-reversed-16",
            ONE: "dku-icon-relationship-one-16"
        }

        const DEFAULT_COLUMN_DATA = { "isSectionOpen": true, "selectedColumns": [] };

        const VARIABLE_TYPES = {
            NUMERIC: { "name": "NUMERIC", "label": "Numerical", "icon": "icon-number" },
            TEXT: { "name": "TEXT", "label": "Text", "icon": "icon-italic" },
            DATE: { "name": "DATE", "label": "Date", "icon": "icon-calendar" },
            CATEGORY: { "name": "CATEGORY", "label": "Categorical", "icon": "icon-font" },
            GENERAL: { "name": "GENERAL", "label": "General", "icon": "" }
        }

        const SCHEMA_TYPES_BY_VARIABLE_TYPE = {
            [VARIABLE_TYPES.NUMERIC.name]: ["int", "double", "float", "bigint", "smallint", "tinyint"],
            [VARIABLE_TYPES.CATEGORY.name]: ["int", "double", "float", "bigint", "string", "date", "dateonly", "datetimenotz", "boolean", "geopoint", "geometry", "array", "complex object", "map", "object", "array"],
            [VARIABLE_TYPES.TEXT.name]: ["string"],
            [VARIABLE_TYPES.DATE.name]: ["date", "dateonly", "datetimenotz"]
        }

        const SUPPORTED_VARIABLE_TYPES = [VARIABLE_TYPES.NUMERIC, VARIABLE_TYPES.TEXT, VARIABLE_TYPES.CATEGORY, VARIABLE_TYPES.DATE];

        const GENERAL_FEATURES = [
            { "name": "COUNT", "label": "Count" }
        ];
        const CATEGORICAL_FEATURES = [
            { "name": "DISTINCT", "label": "Count distinct" }
        ];
        const DATE_FEATURES = [
            { "name": "DAY_OF_MONTH", "label": "Day of month" },
            { "name": "DAY_OF_WEEK", "label": "Day of week" },
            { "name": "MONTH_OF_YEAR", "label": "Month" },
            { "name": "HOUR_OF_DAY", "label": "Hour" },
            { "name": "WEEK_OF_YEAR", "label": "Week" },
            { "name": "YEAR", "label": "Year" }
        ];
        const NUMERICAL_FEATURES = [
            { "name": "AVG", "label": "Average" },
            { "name": "MAX", "label": "Maximum" },
            { "name": "MIN", "label": "Minimum" },
            { "name": "SUM", "label": "Sum" }
        ];
        const TEXT_FEATURES = [
            { "name": "CHARACTER_COUNT", "label": "Character count" },
            { "name": "WORD_COUNT", "label": "Word count" }
        ];
        const FEATURES_BY_CATEGORY = [
            { "categoryName": VARIABLE_TYPES.GENERAL.name, "features": GENERAL_FEATURES },
            { "categoryName": VARIABLE_TYPES.CATEGORY.name, "features": CATEGORICAL_FEATURES },
            { "categoryName": VARIABLE_TYPES.DATE.name, "features": DATE_FEATURES },
            { "categoryName": VARIABLE_TYPES.NUMERIC.name, "features": NUMERICAL_FEATURES },
            { "categoryName": VARIABLE_TYPES.TEXT.name, "features": TEXT_FEATURES }
        ];

        const TIME_UNITS = {
            SECOND: {"name": "SECOND", "label": "seconds"},
            MINUTE: {"name":"MINUTE", "label": "minutes"},
            HOUR: {"name":"HOUR", "label":"hours"},
            DAY: {"name":"DAY", "label": "days"},
            WEEK: {"name":"WEEK", "label": "weeks"},
            MONTH: {"name":"MONTH", "label": "months"},
            YEAR: {"name":"YEAR", "label": "years"}
        };

        const CUTOFF_TIME_MODE = {
            DEFAULT: {"name":"NONE", "label": "None"},
            EMPTY: {"name": null, "label": ""},
            DATE_COLUMN: {"name": "COLUMN","label": "Date column"}
        };

        const TIME_INDEX_MODE = {
            DEFAULT : {"name": "NONE", "label": "None"},
            EMPTY: {"name": null, "label": ""},
            DATE_COLUMN: {"name": "COLUMN", "label": "Date column"}
        }

        const DEFAULT_TIME_WINDOW = {
            "from": 30,
            "to":  0,
            "unit": TIME_UNITS.DAY.name
        };

        const EMPTY_TIME_WINDOW = null;

        const MAX_FEATURES_COUNT_TO_DISPLAY = 100000;

        const CUTOFF_TIME_TOOLTIP = "Cutoff time must be edited directly on the primary dataset by clicking on the pencil next to 'Cutoff time' in the main recipe window.";

        const PRIMARY_DATASET_TOOLTIP = "The dataset you want to enrich with new features. In most cases, this should contain the target variable that you want to predict.";

        const ENRICHMENT_DATASET_TOOLTIP = "Dataset to join with your primary dataset for feature generation.";

        const JOIN_KEY_TOOLTIP = "This column is used as a join key. You may want to deselect it to avoid unnecessary computations.";

        const countDatasetUses = function(datasetId, virtualInputs) {
            return virtualInputs.filter(function(inputTable) {
                return inputTable.index === datasetId;
            }).length;
        }

        const getDatasetColorClass = function(datasetVirtualIndex) {
            return 'dataset-color-' + (datasetVirtualIndex % 6);
        };

        const getDependantDatasets = function(virtualIndex, relationships) {
            let dependantDatasets = [];
            for (let i = 0; i < relationships.length; i++) {
                const relationship = relationships[i];
                if (relationship.table1 === virtualIndex) {
                    dependantDatasets.push(relationship.table2);
                    dependantDatasets = dependantDatasets.concat(getDependantDatasets(relationship.table2, relationships))
                }
            }
            return dependantDatasets;
        }

        const range = function(integer) {
            return [...Array(integer).keys()];
        }

        const isTimeIndexModeDefined = function(relationship) {
            return relationship.timeIndexMode && relationship.timeIndexMode!==TIME_INDEX_MODE.EMPTY.name;
        }

        const isCutoffTimeShown = function (numberOfInputs, cutoffTimeMode) {
            return (numberOfInputs > 1 || isCutoffTimeModeDefined(cutoffTimeMode));
        }

        const isCutoffTimeDate = function(cutoffTimeMode) {
            return cutoffTimeMode===CUTOFF_TIME_MODE.DATE_COLUMN.name;
        }

        const isTimeIndexDate = function(timeIndexMode) {
            return timeIndexMode===TIME_INDEX_MODE.DATE_COLUMN.name;
        }

        const isPrimaryTimeIndexColumnValid = function(relationship, cutoffTimeMode) {
            const isCutoffColumnMissing= isCutoffTimeDate(cutoffTimeMode) && !relationship.timeIndexColumn;
            return !isCutoffColumnMissing;
        }

        const isEnrichmentTimeIndexColumnValid = function(relationship) {
            const isTimeIndexColumnMissing = isTimeIndexDate(relationship.timeIndexMode) && !relationship.timeIndexColumn2;
            return !isTimeIndexColumnMissing;
        }

        const isTimeIndexSelected = function(relationship, cutoffTimeMode) {
            const isTimeIndexModeMissing = isCutoffTimeDate(cutoffTimeMode) && !isTimeIndexModeDefined(relationship);
            return !isTimeIndexModeMissing;
        }

        const isNewAfgRelationshipValid = function(relationship, isCreation, cutoffTimeMode) {
            if (isCreation) {
                return "dataset1" in relationship;
            }
            return isRelationshipValid(relationship, cutoffTimeMode);
        }

        const isRelationshipValid = function(relationship, cutoffTimeMode) {
            return !!(
                (relationship.dataset1 || relationship.table1Index !== null) &&
                relationship.dataset2 &&
                isPrimaryTimeIndexColumnValid(relationship, cutoffTimeMode) &&
                isEnrichmentTimeIndexColumnValid(relationship) &&
                isTimeIndexSelected(relationship, cutoffTimeMode)
            );
        }

        const getLeftSymbolFromRelationship = function(relationshipType) {
            switch (relationshipType) {
                case RELATIONSHIP_TYPES.MANY_TO_ONE:
                    return RELATIONSHIP_SYMBOL.MANY;
                case RELATIONSHIP_TYPES.ONE_TO_MANY:
                case RELATIONSHIP_TYPES.ONE_TO_ONE:
                    return RELATIONSHIP_SYMBOL.ONE;
                default:
                    throw new Error(`Unknown symbol for RELATIONSHIP_TYPE: ${relationshipType}`);
            }
        }

        const getRightSymbolFromRelationship = function(relationshipType) {
            switch (relationshipType) {
                case RELATIONSHIP_TYPES.MANY_TO_ONE:
                case RELATIONSHIP_TYPES.ONE_TO_ONE:
                    return RELATIONSHIP_SYMBOL.ONE;
                case RELATIONSHIP_TYPES.ONE_TO_MANY:
                    return RELATIONSHIP_SYMBOL.MANY_REVERSED;
                default:
                    throw new Error(`Unknown symbol for RELATIONSHIP_TYPE: ${relationshipType}`);
            }
        }

        const getRelationshipDescription = function(relationship) {
            if (!("type" in relationship) || relationship.type === RELATIONSHIP_TYPES.DEFAULT) {
                return relationshipDesc[RELATIONSHIP_TYPES.DEFAULT];
            } else {
                return relationshipDesc[relationship.type];
            }
        }

        const getRelationshipClass = function(relationshipType) {
            switch (relationshipType) {
                case RELATIONSHIP_TYPES.MANY_TO_ONE:
                    return "many-to-one";
                case RELATIONSHIP_TYPES.ONE_TO_ONE:
                    return "one-to-one";
                case RELATIONSHIP_TYPES.ONE_TO_MANY:
                    return "one-to-many";
                default:
                    throw new Error(`Unknown class for RELATIONSHIP_TYPE: ${relationshipType}`);
            }
        }

        const addNewDatasetToRecipe = function(newDatasetName, enrichmentDatasetIndex, recipe, params, isCreation, newColumnsForComputations, newRelationship) {
            if (isCreation) {
                addDatasetToRecipeInputs(newDatasetName, recipe, params.virtualInputs, newColumnsForComputations, newRelationship);
            } else {
                const relationship = createNewRelationship(enrichmentDatasetIndex, params.virtualInputs.length);
                addDatasetToRecipeInputs(newDatasetName, recipe, params.virtualInputs, newColumnsForComputations, newRelationship);
                params.relationships = params.relationships || [];
                params.relationships.push(relationship);
                //TODO : get suggestions for join columns from the back end (see getJoinSuggestions in join.js)
            }

        }

        const addDatasetToRecipeInputs = function(newDatasetName, recipe, virtualInputs, newColumnsForComputations, newRelationship) {
            const primaryTimeIndexColumn = newRelationship.timeIndexColumn;
            const enrichmentTimeIndexColumn = newRelationship.timeIndexColumn2;
            const timeWindow = newRelationship.timeWindow;
            if (RecipesUtils.getInput(recipe, "main", newDatasetName) === null) {
                RecipesUtils.addInput(recipe, "main", newDatasetName);
            }
            if (primaryTimeIndexColumn) {
                virtualInputs[0].timeIndexColumn = primaryTimeIndexColumn;
            }
            const inputNames = RecipesUtils.getInputsForRole(recipe, "main").map(function(input) {
                return input.ref
            });
            const enrichmentDataset = {
                index: inputNames.indexOf(newDatasetName),
                originLabel: newDatasetName,
                selectedColumns: newColumnsForComputations,
                timeWindows: []
            };
            if (enrichmentTimeIndexColumn) {
                enrichmentDataset.timeIndexColumn = enrichmentTimeIndexColumn;
            }
            if (timeWindow) {
                enrichmentDataset.timeWindows = [{
                    "from": timeWindow.from,
                    "to": timeWindow.to,
                    "windowUnit": timeWindow.unit
                }]
            }
            virtualInputs.push(enrichmentDataset);
        }

        const createNewRelationship = function(table1Index, table2Index) {
            return {
                table1: table1Index,
                table2: table2Index,
                type: DEFAULT_RELATIONSHIP.type,
                on: []
            };
        }

        const getMatchedVirtualInputs = function(datasetId, virtualInputs) {
            const virtualIndexes = [];
            for (let virtualIndex = 0; virtualIndex < virtualInputs.length; virtualIndex++) {
                if (virtualInputs[virtualIndex].index === datasetId) {
                    virtualIndexes.push(virtualIndex);
                }
            }
            return virtualIndexes;
        }

        const initFeaturesState = function(features) {
            const allFeatures = [];
            for (let featureType of FEATURES_BY_CATEGORY) {
                const featuresList = [];
                for (let feature of featureType["features"]) {
                    const newFeature = angular.copy(feature);
                    newFeature.selected = features.includes(feature.name);
                    featuresList.push(newFeature);
                }
                allFeatures.push({
                    "categoryLabel": VARIABLE_TYPES[featureType.categoryName].label,
                    "categoryName": featureType.categoryName,
                    "features": featuresList
                });
            }
            return allFeatures;
        }

        const computeListHeight = function(items, maxNumber, rowHeight) {
            return (items.length < maxNumber ? items.length * rowHeight : maxNumber * rowHeight) + 5 + 'px';
        }

        const initialiseColumn = function(columnToInitialise, selectedColumnsForComputation) {
            const columnIndex = selectedColumnsForComputation.findIndex(columnForComputation => columnForComputation.name === columnToInitialise.name);
            const isColumnSelected = columnIndex !== -1;
            if (isColumnSelected) {
                const columnForComputation = selectedColumnsForComputation[columnIndex];
                return {
                    name: columnToInitialise.name,
                    $selected: true,
                    schemaType: columnToInitialise.type,
                    variableType: columnForComputation.variableType
                };
            }
            return {
                name: columnToInitialise.name,
                $selected: false,
                schemaType: columnToInitialise.type,
                variableType: VARIABLE_TYPES.CATEGORY.name
            };
        }

        const getColumnIndex = function(selectedColumn, virtualInput) {
            return virtualInput.selectedColumns.findIndex(columnForComputation => columnForComputation.name === selectedColumn.name);
        }

        function retrieveSavedColumns(virtualInputsIndexes, virtualInputs) {
            if (virtualInputs.length > 0 && virtualInputsIndexes.length > 0) {
                const firstMatch = angular.copy(virtualInputs[virtualInputsIndexes[0]]);
                if ("selectedColumns" in firstMatch) {
                    return firstMatch.selectedColumns;
                }
            }
            return [];
        }


        const getDateColumns = function(columns) {
            return columns.filter(column => ["date", "dateonly", "datetimenotz"].indexOf(column.type) >= 0);
        }

        const getFeaturesCount = function(recipeStatus, virtualInputs) {
            if (isSchemaEmpty(recipeStatus) || virtualInputs.length === 0) {
                return 0
            }
            const primaryDatasetSelectedColumns = getSelectedColumnsFromIndex(0, virtualInputs);
            return countFeatures(primaryDatasetSelectedColumns, recipeStatus.outputSchema.columns)
        }

        function isSchemaEmpty(recipeStatus) {
            const isSchemaAvailable = (recipeStatus && recipeStatus.outputSchema && recipeStatus.outputSchema.columns && recipeStatus.outputSchema.columns.length > 0);
            return !isSchemaAvailable;
        }

        function countFeatures(primaryDatasetSelectedColumns, outputColumns) {
            const featuresCount = outputColumns.length - primaryDatasetSelectedColumns.length;
            const numberFormatter = NumberFormatter.longSmartNumberFilter();
            if (featuresCount > MAX_FEATURES_COUNT_TO_DISPLAY) {
                return numberFormatter(MAX_FEATURES_COUNT_TO_DISPLAY) + "+";
            }
            return numberFormatter(featuresCount);
        }

        const updateColumnSections = function(columnData, inputDatasetsCount) {
            if (columnData.length !== inputDatasetsCount) {
                columnData.push(angular.copy(DEFAULT_COLUMN_DATA));
            }
        }

        const showAddTimeWindowButton = function(timeWindow) {
            return !timeWindow;
        }

        const showTimeWindowSettings = function(newRelationship) {
            return newRelationship.timeWindow && isTimeIndexDate(newRelationship.timeIndexMode);
        }

        const disableTimeIndex = function(newRelationship, cutoffTimeMode, isDatasetValid) {
            return !isEnrichmentSelected(newRelationship, isDatasetValid) || !newRelationship.hasDateColumns[newRelationship.dataset2] || !isCutoffTimeDate(cutoffTimeMode) || !newRelationship.timeIndexColumn;
        }

        const isEnrichmentSelected =  function(newRelationship, isDatasetValid) {
            const enrichmentDatasetName = newRelationship.dataset2;
            return  enrichmentDatasetName && isDatasetValid;
        }

        const disableEnrichmentDataset = function(cutoffTimeMode, newRelationship) {
            return !isCutoffTimeValid(cutoffTimeMode, newRelationship.timeIndexColumn);
        }

        const isCutoffTimeValid = function(cutoffTimeMode, cutoffTimeColumn){
            return isCutoffTimeModeDefined(cutoffTimeMode) && !!(!isCutoffTimeDate(cutoffTimeMode) || cutoffTimeColumn);
        }

        const isCutoffTimeModeDefined = function(cutoffTimeMode)  {
            return !!(cutoffTimeMode && cutoffTimeMode!==CUTOFF_TIME_MODE.EMPTY.name);
        }

        const isTimeWindowInteger = function(timeWindow) {
            return  Number.isInteger(timeWindow.from) && Number.isInteger(timeWindow.to);
        }

        const isTimeWindowPositive = function(timeWindow) {
            return timeWindow.from>0 && timeWindow.to>=0;
        }

        const isFromBiggerThanTo = function (timeWindow) {
            return timeWindow.to<timeWindow.from;
        }

        const isTimeWindowValid = function(timeWindow) {
            return !timeWindow || (isTimeWindowInteger(timeWindow) && isTimeWindowPositive(timeWindow) && isFromBiggerThanTo(timeWindow));
        }

        const disableTimeIndexEdition = function(cutoffTimeMode, hasDateColumns, currentTimeIndexColumn) {
            return !isCutoffTimeDate(cutoffTimeMode) || (!hasDateColumns && !currentTimeIndexColumn);
        }

        const isOnDifferentConnection = function(dataset, primaryDatasetConnection) {
            return primaryDatasetConnection && dataset.dataset.params.connection !== primaryDatasetConnection;
        }

        const isOutputDataset = function(dataset, recipe, outputDatasetName) {
            return (dataset.dataset.name === outputDatasetName && recipe.projectKey && dataset.dataset.projectKey === recipe.projectKey);
        }

        const updateRelationship = function(relationship, conditions) {
            if (conditions.length > 0) {
                conditions.forEach(function(condition) {
                    relationship.on.push(condition);
                })
            }
        }

        const setCutoffTimeTooltip = function(primaryDateColumns) {
            if (!primaryDateColumns || primaryDateColumns.length === 0) {
                return "No date columns in primary dataset";
            }
            return null;
        }

        const setEnrichmentTooltip = function (cutoffTimeMode, cutoffTimeColumn) {
            if (!cutoffTimeMode || cutoffTimeMode === CUTOFF_TIME_MODE.EMPTY.name) {
                return "Cutoff time is required to add an enrichment dataset";
            }
            if (isCutoffTimeDate(cutoffTimeMode) && !cutoffTimeColumn) {
                return "A cutoff time date column is required to add an enrichment dataset";
            }
            return null;
        }

        const setTimeWindowTooltip = function(relationship, currentCutoffTimeMode, editTimeSettings, validDataset){
            if (!isCutoffTimeDate(currentCutoffTimeMode) || !relationship.timeIndexColumn) {
                return "Cutoff time is required to set time window";
            }
            if((!disableTimeIndex(relationship, currentCutoffTimeMode, editTimeSettings ? true : validDataset)) && (!isTimeIndexDate(relationship.timeIndexMode) || !relationship.timeIndexColumn2)){
                return "Time index is required to set time window";
            }
            if (!relationship.dataset2) {
                return "Enrichment dataset is required to set time window";
            }
            if (relationship.dataset2 && !relationship.hasDateColumns[relationship.dataset2]) {
                return "No date columns in enrichment dataset";
            }
            return null;
        }

        const setTimeIndexDateColumnTooltip = function (relationship, currentCutoffTimeMode) {
            if (!relationship.dataset2) {
                return "Enrichment dataset is required to set time index";
            }
            if (!isCutoffTimeDate(currentCutoffTimeMode) || !relationship.timeIndexColumn) {
                return "Cutoff time is required to set time index";
            }
            if (relationship.dataset2 && !relationship.hasDateColumns[relationship.dataset2]) {
                return "No date columns in enrichment dataset";
            }
            return null;
        }

        const setTimeIndexNoneTooltip = function(relationship, currentCutoffTimeMode, isDatasetValid) {
            if (!relationship.dataset2) {
                return "Enrichment dataset is required to set time index";
            }
            if ((!isCutoffTimeDate(currentCutoffTimeMode) || !relationship.timeIndexColumn) && !isDatasetValid) {
                return "Cutoff time is required to set time index";
            }
            return null;
        }

        const checkDatasetUsability = function(dataset, outputDatasetName, recipe, primaryDatasetConnection, allowAnyConnection) {
            const datasetWithUsability = angular.copy(dataset);
            datasetWithUsability.usable = dataset.usableAsInput.main.usable;
            if (!dataset.usableAsInput.main.usable) {
                datasetWithUsability.usableReason = dataset.usableAsInput.main.reason;
            } else if (!allowAnyConnection && DatasetUtils.isOnDifferentConnection(dataset, primaryDatasetConnection)) {
                datasetWithUsability.usable = false;
                datasetWithUsability.usableReason = DatasetUtils.INVALID_INPUT_DATASET_REASONS.DIFF_CONN;
            } else if (DatasetUtils.isOutputDataset(dataset, recipe, outputDatasetName)) {
                datasetWithUsability.usable = false;
                datasetWithUsability.usableReason = DatasetUtils.INVALID_INPUT_DATASET_REASONS.IS_RECIPE_OUTPUT;
            }

            return datasetWithUsability;
        }

        const checkDatasetsUsability = function(availableDatasets, primaryDataset, outputDatasetName, recipe, allowAnyConnection) {
            const availableDatasetsWithUsability = [];
            const primaryDatasetConnection = primaryDataset ? primaryDataset.dataset.params.connection : ""
            for (let dataset of availableDatasets) {
                const datasetWithUsability = checkDatasetUsability(dataset, outputDatasetName, recipe, primaryDatasetConnection, allowAnyConnection);
                availableDatasetsWithUsability.push(datasetWithUsability);
            }
            return availableDatasetsWithUsability;
        }

        const getDatasetNameFromId = function(datasetId, recipeInputs) {
            const inputs = recipeInputs.main.items;
            if (inputs && inputs[datasetId]) {
                return inputs[datasetId].ref;
            }
        }

        const getDatasetName = function(virtualInputIndex, virtualInputs, recipeInputs) {
            if (virtualInputs.length) {
                const dataset = virtualInputs[virtualInputIndex];
                const input = recipeInputs.main.items;
                if (input && dataset && input[dataset.index]) {
                    return input[dataset.index] ? input[dataset.index].ref : null;
                }
            }
            return null;
        }

        const getDateColumnsFromDataset = function(datasetName, computablesMap) {
            if (datasetName) {
                const datasetSchema = DatasetUtils.getSchemaFromComputablesMap(computablesMap, datasetName)
                if(datasetSchema && datasetSchema.columns) {
                    return getDateColumns(datasetSchema.columns);
                }
            }
            return [];
        }

        const getSelectedColumnsFromIndex = function(virtualInputIndex, virtualInputs) {
            let selectedColumns = virtualInputs[virtualInputIndex].selectedColumns;
            if (!selectedColumns) {
                return [];
            }
            return selectedColumns;
        }

        const getOutputDatasetName = function(recipe) {
            const outputs = RecipesUtils.getOutputsForRole(recipe, "main");
            return outputs[0].ref;
        }

        const getColumns = function(datasetName, computablesMap) {
            const datasetSchema = DatasetUtils.getSchemaFromComputablesMap(computablesMap, datasetName);
            return datasetSchema && datasetSchema.columns ? datasetSchema.columns : [];
        }

        const hasDateColumns = function(datasetName, computablesMap) {
            const allColumns = getColumns(datasetName, computablesMap);
            const dateColumns = getDateColumns(allColumns);
            return (dateColumns.length > 0);
        }

        const filterSelectedColumns = function(params, allColumns) {
            for (let virtualInputInd = 0; virtualInputInd < params.virtualInputs.length; virtualInputInd++) {
                let missingSelectedCols = [];
                const inputDataset = params.virtualInputs[virtualInputInd];
                for (let i = 0; i < inputDataset.selectedColumns.length; i++) {
                    if (!allColumns[virtualInputInd].map(elt => elt.name).includes(inputDataset.selectedColumns[i].name)) {
                        missingSelectedCols.push(i)
                    }
                }
                for (let i = missingSelectedCols.length -1; i >= 0; i--)
                   inputDataset.selectedColumns.splice(missingSelectedCols[i],1);
            }
        }

        /**
         * Auto select date colum if there is not one selected already and if there is only 1 available
         */
        const autoSelectDateColumn = function(selectedDateColumn, availableDateColumns) {
            if (!selectedDateColumn && availableDateColumns && availableDateColumns.length === 1) {
                return availableDateColumns[0].name;
            } else {
                return selectedDateColumn;
            }
        }

        const getUsedDatasets = function(virtualInputs, recipeInputs) {
            const usedDatasetsIds = [];
            const usedDatasets = []
            virtualInputs.forEach(function (vi) {
                if (usedDatasetsIds.indexOf(vi.index) < 0) {
                    usedDatasetsIds.push(vi.index);
                    usedDatasets.push({id: vi.index, name: getDatasetNameFromId(vi.index, recipeInputs)});
                }
            });
            return usedDatasets;
        }
        const removeUnusedInputs = function (virtualInputs, recipeInputs) {
            if (!virtualInputs) return;
            const usedDatasetsIds = [];
            virtualInputs.forEach(function (vi) {
                if (usedDatasetsIds.indexOf(vi.index) < 0) {
                    usedDatasetsIds.push(vi.index);
                }
            });
            const newDatasetIds = {};
            for (let oldDatasetId = 0; oldDatasetId < recipeInputs.main.items.length; oldDatasetId++) {
                newDatasetIds[oldDatasetId] = usedDatasetsIds.filter(function (usedDatasetId) {
                    return usedDatasetId < oldDatasetId;
                }).length
            }
            recipeInputs.main.items = recipeInputs.main.items.filter(function (input, idx) {
                return usedDatasetsIds.indexOf(idx) >= 0;
            });
            virtualInputs.forEach(function (vi) {
                vi.index = newDatasetIds[vi.index];
            });
        };

        const resetTimeIndexes = function (virtualInputs) {
            for (const input of virtualInputs) {
                input.timeIndexColumn = null;
                input.timeWindows = [];
            }
        }

        const replaceVirtualInput = function(params, virtualIndex, replacementName, computablesMap, recipeInputs) {
            const inputNames = recipeInputs.main.items.map(function (input) {
                return input.ref
            });
            params.virtualInputs[virtualIndex] = {
                index: inputNames.indexOf(replacementName),
                timeIndexColumn: null,
                timeWindows: [],
                originLabel: replacementName,
                selectedColumns: []
            };
            removeUnusedInputs(params.virtualInputs, recipeInputs);
        }

        const keepTimeSettings = function(replacementDatasetName, oldInput, computablesMap) {
            const replacementDateColumns = getDateColumnsFromDataset(replacementDatasetName, computablesMap);
            const dateColumnsNames = replacementDateColumns.map(column => column.name);
            return (oldInput.timeIndexColumn && dateColumnsNames.includes(oldInput.timeIndexColumn));
        }

        const removeMissingJoinKeys = function (params, computablesMap, recipeInputs) {
            params.relationships.forEach(function (relationship) {
                const dataset1 = getDatasetName(relationship.table1, params.virtualInputs, recipeInputs);
                const dataset2 = getDatasetName(relationship.table2, params.virtualInputs, recipeInputs);
                const columns1 = getColumns(dataset1, computablesMap).map(function (col) {
                    return col.name
                });
                const columns2 = getColumns(dataset2, computablesMap).map(function (col) {
                    return col.name
                });
                relationship.on = relationship.on.filter(function (cond) {
                    if (!columns1 || !columns2 || columns1.indexOf(cond.column1.name) < 0 || columns2.indexOf(cond.column2.name) < 0) {
                        return false;
                    }
                    return true;
                })
            })
        }

        const updateTimeSettings = function (params, virtualIndex, computablesMap, oldInput) {
            const replacementName = params.virtualInputs[virtualIndex].originLabel;
            if (params.cutoffTime && params.cutoffTime.mode === CUTOFF_TIME_MODE.DATE_COLUMN.name) {
                if (keepTimeSettings(replacementName, oldInput, computablesMap)) {
                    params.virtualInputs[virtualIndex].timeIndexColumn = oldInput.timeIndexColumn;
                    params.virtualInputs[virtualIndex].timeWindows = oldInput.timeWindows;
                } else if (virtualIndex === 0) {
                    params.cutoffTime = {mode: CUTOFF_TIME_MODE.DEFAULT.name};
                    resetTimeIndexes(params.virtualInputs);
                }
            }
        }

        const resyncSchemas = function(params, virtualIndex, computablesMap, recipeInputs, oldInput) {
            removeMissingJoinKeys(params, computablesMap, recipeInputs);
            updateTimeSettings(params, virtualIndex, computablesMap, oldInput)
        }

        const getDefaultColumnsTabData = function(recipe) {
            const hasInput = recipe && recipe.inputs && recipe.inputs.main && recipe.inputs.main.items;
            const inputDatasets = hasInput ? recipe.inputs.main.items : [];
            return Array.from({ length: inputDatasets.length }, () => (angular.copy(DEFAULT_COLUMN_DATA)));
        }

        const removeDatasets = function (virtualIndexes, deletedDatasetVirtualIndex, params, recipe, columnsTabData) {
            virtualIndexes.sort().reverse();

            const datasetNames = {};
            for (const virtualIndex of virtualIndexes) {
                datasetNames[virtualIndex] = getDatasetName(virtualIndex, params.virtualInputs, recipe.inputs);
            }

            virtualIndexes.forEach(function (virtualInputIndex) {
                params.relationships = params.relationships.filter(function (relationship) {
                    return relationship.table1 !== virtualInputIndex && relationship.table2 !== virtualInputIndex;
                });
                params.relationships.forEach(function (relationship) {
                    if (relationship.table1 > virtualInputIndex) {
                        relationship.table1--;
                        relationship.on.forEach(function (condition) {
                            condition.column1.table--;
                        })
                    }
                    if (relationship.table2 > virtualInputIndex) {
                        relationship.table2--;
                        relationship.on.forEach(function (condition) {
                            condition.column2.table--;
                        })
                    }
                });

                const datasetId = params.virtualInputs[virtualInputIndex].index;
                const numberOfUses = countDatasetUses(datasetId, params.virtualInputs);
                const datasetName = datasetNames[virtualInputIndex];
                if (numberOfUses === 1) {
                    RecipesUtils.removeInput(recipe, "main", datasetName);
                    columnsTabData.splice(datasetId, 1);
                    params.virtualInputs.forEach(function (vi) {
                        if (vi.index > datasetId) {
                            vi.index--;
                        }
                    });
                } else if (numberOfUses === 0) {
                    throw new Error(`The dataset ${datasetName} does not belong to the inputs of the recipe. Check your parameters.`);
                }
                params.virtualInputs.splice(virtualInputIndex, 1);
            });
            updateTimeSettingsAfterDeletion(deletedDatasetVirtualIndex, params);
        }

        const updateTimeSettingsAfterDeletion = function (virtualIndex, params) {
            if (virtualIndex === 0) {
                params.cutoffTime.mode = CUTOFF_TIME_MODE.EMPTY.name;
                if (params.virtualInputs.length) {
                    params.virtualInputs[0].timeIndexColumn = null;
                    params.virtualInputs[0].timeWindows = [];
                }
            }
        }

        const service = {
            addNewDatasetToRecipe: addNewDatasetToRecipe,
            addDatasetToRecipeInputs: addDatasetToRecipeInputs,
            autoSelectDateColumn: autoSelectDateColumn,
            checkDatasetsUsability: checkDatasetsUsability,
            computeListHeight: computeListHeight,
            countDatasetUses: countDatasetUses,
            disableEnrichmentDataset: disableEnrichmentDataset,
            disableTimeIndex: disableTimeIndex,
            disableTimeIndexEdition: disableTimeIndexEdition,
            isEnrichmentSelected: isEnrichmentSelected,
            filterSelectedColumns: filterSelectedColumns,
            getColumnIndex: getColumnIndex,
            getColumns: getColumns,
            getDateColumns: getDateColumns,
            getDateColumnsFromDataset: getDateColumnsFromDataset,
            getDatasetColorClass: getDatasetColorClass,
            getDatasetName: getDatasetName,
            getDatasetNameFromId: getDatasetNameFromId,
            getDefaultColumnsTabData: getDefaultColumnsTabData,
            getDependantDatasets: getDependantDatasets,
            getFeaturesCount: getFeaturesCount,
            getLeftSymbolFromRelationship: getLeftSymbolFromRelationship,
            getMatchedVirtualInputs: getMatchedVirtualInputs,
            getRelationshipClass: getRelationshipClass,
            getRelationshipDescription: getRelationshipDescription,
            getRightSymbolFromRelationship: getRightSymbolFromRelationship,
            getUsedDatasets: getUsedDatasets,
            getOutputDatasetName: getOutputDatasetName,
            hasDateColumns: hasDateColumns,
            initFeaturesState: initFeaturesState,
            initialiseColumn: initialiseColumn,
            isCutoffTimeDate: isCutoffTimeDate,
            isCutoffTimeShown: isCutoffTimeShown,
            isCutoffTimeModeDefined: isCutoffTimeModeDefined,
            isCutoffTimeValid: isCutoffTimeValid,
            isEnrichmentTimeIndexColumnValid: isEnrichmentTimeIndexColumnValid,
            isFromBiggerThanTo: isFromBiggerThanTo,
            isNewAfgRelationshipValid: isNewAfgRelationshipValid,
            isTimeIndexDate: isTimeIndexDate,
            isTimeWindowInteger: isTimeWindowInteger,
            isTimeWindowPositive: isTimeWindowPositive,
            isTimeWindowValid: isTimeWindowValid,
            keepTimeSettings: keepTimeSettings,
            primaryDatasetDesc: primaryDatasetDesc,
            range: range,
            relationshipDesc: relationshipDesc,
            relationshipNumDesc: relationshipNumDesc,
            removeDatasets: removeDatasets,
            removeUnusedInputs: removeUnusedInputs,
            replaceVirtualInput: replaceVirtualInput,
            resetTimeIndexes: resetTimeIndexes,
            resyncSchemas: resyncSchemas,
            retrieveSavedColumns: retrieveSavedColumns,
            setCutoffTimeTooltip: setCutoffTimeTooltip,
            setEnrichmentTooltip: setEnrichmentTooltip,
            setTimeIndexDateColumnTooltip: setTimeIndexDateColumnTooltip,
            setTimeIndexNoneTooltip: setTimeIndexNoneTooltip,
            setTimeWindowTooltip: setTimeWindowTooltip,
            showAddTimeWindowButton: showAddTimeWindowButton,
            showTimeWindowSettings: showTimeWindowSettings,
            updateColumnSections: updateColumnSections,
            updateRelationship: updateRelationship,
            CUTOFF_TIME_MODE: CUTOFF_TIME_MODE,
            CUTOFF_TIME_TOOLTIP: CUTOFF_TIME_TOOLTIP,
            DEFAULT_COLUMN_DATA: DEFAULT_COLUMN_DATA,
            DEFAULT_TIME_WINDOW: DEFAULT_TIME_WINDOW,
            EMPTY_TIME_WINDOW: EMPTY_TIME_WINDOW,
            ENRICHMENT_DATASET_TOOLTIP: ENRICHMENT_DATASET_TOOLTIP,
            JOIN_KEY_TOOLTIP: JOIN_KEY_TOOLTIP,
            PRIMARY_DATASET_TOOLTIP: PRIMARY_DATASET_TOOLTIP,
            RELATIONSHIP_TYPES: RELATIONSHIP_TYPES,
            SCHEMA_TYPES_BY_VARIABLE_TYPE: SCHEMA_TYPES_BY_VARIABLE_TYPE,
            SUPPORTED_RELATIONSHIP_TYPES: SUPPORTED_RELATIONSHIP_TYPES,
            SUPPORTED_VARIABLE_TYPES: SUPPORTED_VARIABLE_TYPES,
            TIME_UNITS: TIME_UNITS,
            TIME_INDEX_MODE: TIME_INDEX_MODE,
            VARIABLE_TYPES: VARIABLE_TYPES,
            FEATURES_BY_CATEGORY: FEATURES_BY_CATEGORY
        };
        return service;
    }
})();