(function(){
'use strict';

const app = angular.module('dataiku.analysis.mlcore');

app.controller("MLTaskFeaturesController", function($scope, $controller, $timeout, $stateParams, $rootScope, Assert, ColumnAnalysesService,
    DataikuAPI, Fn, Dialogs, PMLSettings, $q, VisualMlCodeEnvCompatibility, CreateModalFromTemplate, Logger, DatasetUtils) {
    Assert.inScope($scope, "analysisCoreParams");
    $scope.columnAnalysesService = ColumnAnalysesService.init($scope.analysisCoreParams, setErrorInScope.bind($scope));

    // List of roles that should not be impacted by mass actions
    const PROTECTED_ROLES = ["TARGET", "WEIGHT"];

    $scope.featureAutoHandlingReason = {
        "REJECT_ZERO_VARIANCE" : "DSS has rejected this feature because all values of this feature are equal.",
        "REJECT_MISSING" : "DSS has rejected this feature because too many values are missing in this feature.",
        "REJECT_IDENTIFIER" : "DSS has rejected this feature because this feature looks like a unique identifier.",
        "REJECT_DEFAULT_TEXT_HANDLING" : "DSS rejects text features by default.",
        "REJECT_CARDINALITY": "DSS has rejected this feature because it had too many categories for the task at hand."
    };

    $scope.featureAutoHandlingShortReason = {
        "REJECT_ZERO_VARIANCE" : "too many equal values",
        "REJECT_MISSING" : "too many missing values",
        "REJECT_IDENTIFIER" : "unique ID",
        "REJECT_DEFAULT_TEXT_HANDLING" : "text feature",
        "REJECT_CARDINALITY": "too many categories"
    };

    $scope.categoryHandlingModes = (function () {
        if ($scope.isMLBackendType('H2O')) {
            return [
                ["NONE", "H2O feature handling"],
                ["DUMMIFY", "Dummy encoding (vectorization)"]
            ];
        }

        const categoryHandlingModes = [
            ["DUMMIFY", "Dummy encoding (vectorization)"],
            ["FLAG_PRESENCE", "Replacing by 0/1 flag indicating presence"]
        ];

        if($scope.isMLBackendType('PY_MEMORY') || $scope.isMLBackendType('KERAS')) {
            if ($scope.mlTasksContext.activeMLTask.taskType === 'PREDICTION') {
                categoryHandlingModes.push(
                    ["IMPACT", "Target encoding"]
                );
            }
            categoryHandlingModes.push(
                ["ORDINAL", "Ordinal encoding"],
                ["FREQUENCY", "Frequency encoding"],
                ["HASHING", "Feature hashing (for high cardinality)"],
                ["CUSTOM", "Custom preprocessing"]
            );
        }
        return categoryHandlingModes;
    })();

    $scope.numericalHandlingModes = (function() {
        const  numericalHandlingModes = [
            ["REGULAR", "Keeping as a regular numerical feature"],
            ["FLAG_PRESENCE", "Replacing by 0/1 flag indicating presence"],
            ["BINARIZE", "Binarization based on a threshold"],
            ["QUANTILE_BIN", "Quantization"]
        ];
        if($scope.isMLBackendType('PY_MEMORY') || $scope.isMLBackendType('KERAS')) {
            numericalHandlingModes.push(
                ["DATETIME_CYCLICAL", "Cyclical datetime encoding"],
                ["CUSTOM", "Custom preprocessing"])
            ;
        }
        return numericalHandlingModes;
    })();

    $scope.textHandlingModes = (function() {
        const textHandlingModes = [
            ["TOKENIZE_HASHING", "Tokenization and hashing"],
            ["TOKENIZE_HASHING_SVD", "Tokenization, hashing and SVD"],
            ["TOKENIZE_COUNTS", "Count vectorization"],
            ["TOKENIZE_TFIDF", "TF/IDF vectorization"]
        ];
        if($scope.isMLBackendType('PY_MEMORY') || $scope.isMLBackendType('KERAS')) {
            textHandlingModes.push(["SENTENCE_EMBEDDING", "Text embedding"]);
            textHandlingModes.push(["CUSTOM", "Custom preprocessing"]);
        }
        return textHandlingModes;
    })();

    $scope.vectorHandlingModes = [
        ["UNFOLD", "Unfolding (creating one column per element)"],
    ];

    $scope.imageHandlingModes = (function() {
        const imageHandlingModes = [];
            
        if($scope.isMLBackendType('KERAS')) {
            imageHandlingModes.push(["CUSTOM", "Custom preprocessing"]);
        }
        if($scope.isMLBackendType('PY_MEMORY') && $scope.mlTasksContext.activeMLTask.taskType === 'PREDICTION') {
            imageHandlingModes.push(["EMBEDDING_EXTRACTION", "Image embedding"]);
        }
        return imageHandlingModes;
    })();

    $scope.dummyClippingModes = [
        ["MAX_NB_CATEGORIES", "Max nb. categories"],
        ["CUMULATIVE_PROPORTION", "Cumulative proportion"],
        ["MIN_SAMPLES", "Minimum samples"],
    ];

    $scope.dummyDrops = [
        ["AUTO", "Let DSS decide"],
        ["NONE", "Don't drop"],
        ["DROP", "Drop one dummy"]
    ];


    $scope.setMissingValuesModes = function(feature){
        switch(feature.type){
            case 'CATEGORY':
                $scope.missingValuesModes = [
                    ["NONE", "Treat as a regular value"],
                    ["IMPUTE", "Impute ..."],
                    ["DROP_ROW", "Drop rows (don't predict them either)"]
                ];
                break;
            case 'NUMERIC':
                if (feature.numerical_handling === 'BINARIZE' || feature.numerical_handling === 'QUANTILE_BIN' ) {
                    $scope.missingValuesModes = [["IMPUTE", "Impute ..."]];
                } else if (feature.numerical_handling === 'DATETIME_CYCLICAL') {
                    $scope.missingValuesModes = [["DROP_ROW", "Drop rows (don't predict them either)"]];
                } else {
                    $scope.missingValuesModes = [
                        ["IMPUTE", "Impute ..."],
                        ["DROP_ROW", "Drop rows (don't predict them either)"]];
                    let isCompatibleWithNans = $scope.mlTaskDesign.backendType=='PY_MEMORY' && $scope.mlTaskDesign.taskType=='PREDICTION' &&
                        !$scope.isTimeseriesPrediction() && !$scope.isCausalPrediction();
                    if (isCompatibleWithNans) {
                        $scope.missingValuesModes.push(["KEEP_NAN_OR_IMPUTE", "Keep empty | Impute for incompatible algorithms"]);
                        $scope.missingValuesModes.push(["KEEP_NAN_OR_DROP", "Keep empty | Drop rows for incompatible algorithms"]);
                    }
                };
                break;
            case 'VECTOR':
                $scope.missingValuesModes = [
                    ["DROP_ROW", "Drop rows (don't predict them either)"],
                    ["IMPUTE", "Impute ..."],
                    ["NONE", "Fail if missing values found"]
                ];
                break;
            case 'IMAGE':
                $scope.missingValuesModes = [
                    ["DROP_ROW", "Drop rows (don't predict them either)"],
                    ["NONE", "Fail if missing values found"]
                ];
                if (!$scope.isMLBackendType('KERAS')) {
                    $scope.missingValuesModes.push(["IMPUTE", "Impute missing values with empty embeddings (zeros)"]);
                }
                break;
            case 'TEXT':
                // missing values dropdown is not displayed for Text
                break;
            default: Logger.info("Invalid feature type: ", feature.type);
        }
    }
    $scope.categoryMissingHandlingImputeWithModes = [
        ["MODE", "Most frequent value"],
        ["CONSTANT", "A constant value"]
    ];
    $scope.vectorMissingHandlingImputeWithModes = [
        ["MODE", "Most frequent value"],
        ["CONSTANT", "A vector filled with a single value"]
    ];

    $scope.numericalMissingHandlingImputeWithModes = [
        ["MEAN", "Average of values"],
        ["MEDIAN", "Median of values"],
        ["CONSTANT", "A constant value"]
    ];

    $scope.rescalingModes = [
        ["NONE", "No rescaling"],
        ["MINMAX", "Min-max rescaling"],
        ["AVGSTD", "Standard rescaling"]
    ];
    $scope.monotonicModes = [
        ["NONE", "No Constraint"],
        ["INCREASE", "Increasing"],
        ["DECREASE", "Decreasing"]
    ];
    $scope.binarizeThresholdModes = [
        ["MEAN", "Average of values"],
        ["MEDIAN", "Median of values"],
        ["CONSTANT", "A constant value"]
    ];
    $scope.categoryOrdinalOrders = [
        ["COUNT", "Row count"],
        ["LEXICOGRAPHIC", "Lexicographic"]
    ];
    $scope.categoryOrdinalDefaultModes = [
        ["HIGHEST", "Highest value (maximum + 1)"],
        ["MEDIAN", "Median of values"],
        ["EXPLICIT", "A constant value"],
    ];
    $scope.categoryFrequencyDefaultModes = [
        ["MIN", "Minimum computed value"],
        ["MEDIAN", "Median of computed values"],
        ["MAX", "Maximum computed value"],
        ["EXPLICIT", "A constant value"],
    ];
    $scope.sendToInputModes = [
        ["MAIN", "Main input"],
        ["OTHER", "Other input"]
    ];

    function fillInitialCustomHandlingCode(nv, ov){
        if (nv == "CUSTOM" &&
                ($scope.selection.selectedObject.customHandlingCode == null ||
                    $scope.selection.selectedObject.customHandlingCode.length == 0) ) {
            if ($scope.isMLBackendType("KERAS") && $scope.selection.selectedObject.type === "TEXT") {
                $scope.selection.selectedObject.customHandlingCode =
                "from dataiku.doctor.deep_learning.preprocessing import TokenizerProcessor\n\n" +
                "# Defines a processor that tokenizes a text. It computes a vocabulary on all the corpus.\n" + 
                "# Then, each text is converted to a vector representing the sequence of words, where each \n" +
                "# element represents the index of the corresponding word in the vocabulary. The result is \n" + 
                "# padded with 0 up to the `max_len` in order for all the vectors to have the same length.\n\n" +
                "#   num_words  - maximum number of words in the vocabulary\n" +
                "#   max_len    - length of each sequence. If the text is longer,\n" +
                "#                it will be truncated, and if it is shorter, it will be padded\n" +
                "#                with 0.\n" +
                "processor = TokenizerProcessor(num_words=10000, max_len=32)";
            } else if ($scope.selection.selectedObject.type === "CATEGORY" ||  $scope.selection.selectedObject.type === "TEXT") {
                $scope.selection.selectedObject.customHandlingCode =
                    "from sklearn.feature_extraction import text\n\n"+
                    "# Applies count vectorization to the feature\n" +
                    "processor = text.CountVectorizer()\n";

            } else if ($scope.selection.selectedObject.type == "NUMERIC") {
                $scope.selection.selectedObject.customHandlingCode =
                    "from sklearn import preprocessing\nimport numpy as np\n\n"+
                    "# Applies log transformation to the feature\n" +
                    "processor = preprocessing.FunctionTransformer(np.log1p)\n";
                $scope.selection.selectedObject.customProcessorWantsMatrix = true;
            }
            $timeout(function() {
            // Force recomputation of all "remaining-height" directives
            $rootScope.$broadcast('reflow');
        });
        }
    }

    function onNumericalHandlingChange(oldVersion, newVersion) {
        fillInitialCustomHandlingCode(oldVersion, newVersion);

        //Missing values modes available depends on numerical handling mode chosen:
        const selectedFeature = $scope.selection.selectedObject;
        if (selectedFeature) {
            $scope.setMissingValuesModes(selectedFeature);
            fixupDatetimeParamsIfNeeded(selectedFeature);
        }
    }

    function fixupDatetimeParamsIfNeeded(selectedFeature) {
        if (selectedFeature && selectedFeature.numerical_handling == "DATETIME_CYCLICAL") {
            $scope.columnAnalysesService.fetchColumnAnalysisIfNeeded(selectedFeature._name).then((columnAnalysis) => {
                const numericalAnalysis = (columnAnalysis || {}).numericalAnalysis || {};
                if (!selectedFeature.datetime_cyclical_periods || selectedFeature.datetime_cyclical_periods.length == 0) {
                    selectedFeature.datetime_cyclical_periods = numericalAnalysis.relevantPeriods || [];
                }
                $scope.periodHistograms = numericalAnalysis.periodHistograms;
            });
        }
    }

    $scope.$watch("selection.selectedObject.category_handling", fillInitialCustomHandlingCode);
    $scope.$watch("selection.selectedObject.text_handling", fillInitialCustomHandlingCode);
    $scope.$watch("selection.selectedObject.numerical_handling", onNumericalHandlingChange);

    $scope.$watch('selection.selectedObject', function(nv, ov) {
        $timeout(function() {
            // Force recomputation of all "remaining-height" directives
            $rootScope.$broadcast('reflow');
        });
        // We wait in order to make sure the layout has been updated
        // (so that the computation of the real remaining height is correct)
        if (nv) {
            $scope.columnAnalysesService.fetchColumnAnalysisIfNeeded(nv._name);
            $scope.fixupFeatureConfiguration(nv, ov);
            if (ov === null || nv?.$idx !== ov?.$idx){
                // reinitialize possible modes, only when we change from one feature to another or select for the 1st time a feature
                $scope.setMissingValuesModes(nv);
            }
            // Only potentially override `missing_handling` if already set (to bad value), otherwise we leave it empty in order not to
            // potentially wrongly set the task as dirty
            if ($scope.missingValuesModes && nv.missing_handling && !$scope.missingValuesModes.some(mode => mode[0] === nv.missing_handling)) {
                nv.missing_handling = $scope.missingValuesModes[0][0];
            }
        }
    }, true);

    $scope.fixupFeatureConfiguration = function(feature, oldFeature) {
        if(feature.role == 'REJECT' || PROTECTED_ROLES.includes(feature.role)) {

            // For KERAS backend, need to check that feature is special in order to 
            // create/delete Special input accordingly when rejecting/accepting a
            // feature
            if ($scope.isMLBackendType('KERAS')) {
                handleSwitchingToSpecialFeature(feature, oldFeature);
            }

            return;
        }
        if (feature.type=='CATEGORY') {
            if (!feature.category_handling) {
                feature.category_handling = "DUMMIFY";
            }
            if (feature.category_handling == "DUMMIFY" && !feature.max_nb_categories) {
                feature.max_nb_categories = 100;
            }
            if (feature.category_handling == "DUMMIFY" && !feature.max_cat_safety) {
                feature.max_cat_safety = 200;
            }
            if (feature.category_handling == "DUMMIFY" && !feature.dummy_drop) {
                feature.dummy_drop = "AUTO";
            }
            if (feature.category_handling == "DUMMIFY" && !feature.cumulative_proportion) {
                feature.cumulative_proportion = 0.95;
            }
            if (feature.category_handling == "DUMMIFY" && !feature.min_samples) {
                feature.min_samples = 10;
            }
            if (feature.category_handling == "DUMMIFY" && !feature.dummy_clip) {
                feature.dummy_clip = "CUMULATIVE_PROPORTION";
            }

            if (feature.category_handling == "HASHING" && !feature.nb_bins_hashing) {
                feature.nb_bins_hashing = 1048576;
            }

            if (feature.category_handling == "IMPACT" && feature.impact_method === undefined) {
                feature.impact_method = "M_ESTIMATOR";
            }
            if (feature.category_handling == "IMPACT" && feature.impact_m === undefined) {
                feature.impact_m = 10;
            }
            if (feature.category_handling == "IMPACT" && feature.impact_kfold === undefined) {
                feature.impact_kfold = true;
            }
            if (feature.category_handling == "IMPACT" && feature.impact_kfold_k === undefined) {
                feature.impact_kfold_k = 5;
            }
            if (feature.category_handling == "IMPACT" && feature.impact_kfold_seed === undefined) {
                feature.impact_kfold_seed = 1337;
            }
            if (feature.category_handling == "IMPACT" && feature.categorical_rescaling === undefined) {
                feature.categorical_rescaling = "AVGSTD";
            }

            if (feature.category_handling == "ORDINAL" && feature.ordinal_order === undefined) {
                feature.ordinal_order = "COUNT";
            }
            if (feature.category_handling == "ORDINAL" && feature.ordinal_ascending === undefined) {
                feature.ordinal_ascending = false;
            }
            if (feature.category_handling == "ORDINAL" && feature.ordinal_default_mode === undefined) {
                feature.ordinal_default_mode = "HIGHEST";
            }
            if (feature.category_handling == "ORDINAL" && feature.ordinal_default_value === undefined) {
                feature.ordinal_default_value = -1;
            }

            if (feature.category_handling == "FREQUENCY" && feature.frequency_default_mode === undefined) {
                feature.frequency_default_mode = "EXPLICIT";
            }
            if (feature.category_handling == "FREQUENCY" && feature.frequency_default_value === undefined) {
                feature.frequency_default_value = 0.;
            }
            if (feature.category_handling == "FREQUENCY" && feature.frequency_normalized === undefined) {
                feature.frequency_normalized = true;
            }
            if (feature.category_handling == "FREQUENCY" && feature.categorical_rescaling === undefined) {
                feature.categorical_rescaling = "AVGSTD";
            }

            if (!feature.missing_handling) {
                feature.missing_handling = 'IMPUTE';
            }

            if(feature.missing_handling === 'IMPUTE' && !["MODE", "CONSTANT"].includes(feature.missing_impute_with)) {
                feature.missing_impute_with = "MODE";
            }

        } else if (feature.type=='NUMERIC') {
            if (!feature.numerical_handling) {
                feature.numerical_handling = "REGULAR";
                feature.rescaling = "AVGSTD"
            }

            if (!feature.missing_handling) {
                feature.missing_handling = 'IMPUTE';
            }

            if (feature.missing_handling === 'IMPUTE' && !["MEAN", "MEDIAN", "CONSTANT"].includes(feature.missing_impute_with)) {
                feature.missing_impute_with = "MEDIAN";
            }

            if (feature.numerical_handling === 'BINARIZE'
                && !["MEAN", "MEDIAN", "CONSTANT"].includes(feature.binarize_threshold_mode)) {
                feature.binarize_threshold_mode = "MEDIAN";
            }

            if (feature.numerical_handling === 'QUANTILE_BIN' && feature.quantile_bin_nb_bins === undefined) {
                feature.quantile_bin_nb_bins = 4;
            }


            if (feature.numerical_handling === 'DATETIME_CYCLICAL' && !feature.datetime_cyclical_periods) {
                feature.datetime_cyclical_periods = [];
            }

            feature.category_handling = undefined;

        } else if (feature.type == "TEXT") {
            if (!feature.text_handling) {
                feature.text_handling = "TOKENIZE_HASHING_SVD";
            }
            if (!feature.hashSize) {
                feature.hashSize = 200000;
            }
            if (!feature.hashSVDSVDLimit) {
                feature.hashSVDSVDLimit = 50000;
            }
            if (!feature.hashSVDSVDComponents) {
                feature.hashSVDSVDComponents = 100;
            }
            if (!feature.minRowsRatio) {
                feature.minRowsRatio = 0.001;
            }
            if (!feature.maxRowsRatio) {
                feature.maxRowsRatio = 0.8;
            }
            if (!feature.maxWords) {
                feature.maxWords = 0;
            }
            if (!feature.ngramMinSize) {
                feature.ngramMinSize = 1;
            }
            if (!feature.ngramMaxSize) {
                feature.ngramMaxSize = 1;
            }
            if (!feature.stopWordsMode) {
                feature.stopWordsMode = "NONE";
            }
            if (!feature.maxSequenceLength) {
                feature.maxSequenceLength = 128;
            }
            if (!feature.sentenceEmbeddingBatchSize) {
                feature.sentenceEmbeddingBatchSize = 32;
            }
        } else if (feature.type == "VECTOR") {
            if (!feature.vector_handling) {
                feature.vector_handling = "UNFOLD";
            }

            if (feature.missing_handling === 'IMPUTE') {
                if (!["MODE", "CONSTANT"].includes(feature.missing_impute_with)) {
                    feature.missing_impute_with = "MODE";
                } else if (feature.missing_impute_with === "CONSTANT" && !feature.impute_constant_value) {
                    feature.impute_constant_value = "0"
                }
            } else if (!feature.missing_handling) {
                feature.missing_handling = "DROP_ROW";
            }
        } else if (feature.type === "IMAGE") {
            // Setting value of Image Handling manually for first selection of Image as type
            // because image type is never guessed by back-end so never set up automatically
            if (!feature.image_handling) {
                if ($scope.isMLBackendType('KERAS')) {
                    feature.image_handling = "CUSTOM";
                } else {
                    feature.image_handling = "EMBEDDING_EXTRACTION";
                }
            }

            if (feature.image_handling === "CUSTOM" && feature.customHandlingCode === "") {
                feature.customHandlingCode = getCustomImageHandlingCode();
            }

            if (!feature.missing_handling) {
                feature.missing_handling = "DROP_ROW";
            }
        }

        if ($scope.isMLBackendType('KERAS')) {
            handleSwitchingToSpecialFeature(feature, oldFeature);
        }
    };

    var datasetColumns = [];
    var datasetLoc = DatasetUtils.getLocFromSmart($stateParams.projectKey, $scope.analysisCoreParams.inputDatasetSmartName);
    DataikuAPI.datasets.get(datasetLoc.projectKey, datasetLoc.name, $stateParams.projectKey)
        .success(function(dataset) {
            datasetColumns = dataset.schema.columns.map(Fn.prop('name'));
            addDataSetColumnId();
        }).error(setErrorInScope.bind($scope));
    var addDataSetColumnId = function() {
        if (datasetColumns.length === 0) {
            return;
        }
        angular.forEach($scope.mlTaskDesign.preprocessing.per_feature, function(feature) {
            feature.datasetColumnId = datasetColumns.indexOf(feature._name);
        });
        // Make sure `filteredMultiSelect` directive is aware of change in way the list of features is sorted
        if ($scope.updateSorted) {
            $scope.updateSorted();
        }
    }
    $scope.$watch('mlTaskDesign.preprocessing.per_feature', addDataSetColumnId);

    $scope.acceptDSSChange = function(feature) {
        Assert.trueish(feature.state.dssWantsToSet, "unexpected call to acceptDSSChange");
        var oldState = feature.state;
        $.each(feature.state.dssWantsToSet, function(k, v){feature[k] = v;});
        feature.state = {
            userModified : false,
            recordedMeaning : oldState.recordedMeaning
        };
    }

    $scope.isMonotonicityCompatible = function(featureSettings) {
        return (featureSettings.type=='NUMERIC' &&
                featureSettings.numerical_handling == 'REGULAR');
    }

    $scope.someMonotonicConstraints = function() {
        return Object.values($scope.uiState.preprocessingPerFeature).some(x => ((x.role == 'INPUT') && (x.type == 'NUMERIC') && (x.numerical_handling == 'REGULAR') && (x.monotonic!='NONE')));
    }

    $scope.warnMonotonicConstraintsIncompatibility = function() {
        if (!$scope.someMonotonicConstraints()) {
            return false;
        } else {
            const algosWithoutMonotonicConstraintSupport = new Set(['gbt_classification', 'gbt_regression',
                'ridge_regression', 'lasso_regression', 'leastsquare_regression', 'sgd_regression', 'sgd_classifier', 'knn', 'logistic_regression',
                'neural_network', 'svc_classifier', 'svm_regression','lars_params', 'deep_neural_network_regression', 'deep_neural_network_classification']);

            $scope.uiState.selectedAlgorithmsWithMonotonicConstraintsIncompatibility = [];
            const modeling = $scope.mlTaskDesign.modeling;
            $scope.base_algorithms[$scope.mlTaskDesign.backendType].forEach(function(x) {
                if((modeling[x.algKey] && modeling[x.algKey].enabled && algosWithoutMonotonicConstraintSupport.has(x.algKey)) ||
                   (modeling.plugin_python && modeling.plugin_python[x.algKey] && modeling.plugin_python[x.algKey].enabled )) {
                    $scope.uiState.selectedAlgorithmsWithMonotonicConstraintsIncompatibility.push(x.name);
                }
            });
            if(modeling.custom_python.some(x => x.enabled)) {
                $scope.uiState.selectedAlgorithmsWithMonotonicConstraintsIncompatibility.push("Custom Python model");
            }
            return $scope.uiState.selectedAlgorithmsWithMonotonicConstraintsIncompatibility.length > 0;
        }
    }

    $scope.groupSet = function(newContent) {
        for(let i in $scope.selection.selectedObjects) {
            let feature = $scope.selection.selectedObjects[i];
            if(PROTECTED_ROLES.includes(feature.role)) {
                continue;
            }
            let modified = false;
            for(let k in newContent) {
                if(feature[k]!==newContent[k]) {
                    feature[k]=newContent[k];
                    modified = true;
                }
            }
            if(modified) {
                if (!feature.state) feature.state = {};
                feature.state.userModified = true;
                $scope.fixupFeatureConfiguration(feature);
            }
        }
    };

    $scope.isGroupSetUseful = function(newContent) {
        for(let i in $scope.selection.selectedObjects) {
            let feature = $scope.selection.selectedObjects[i];
            if(PROTECTED_ROLES.includes(feature.role)) {
                continue;
            }
            for(let k in newContent) {
                if(feature[k]!== newContent[k]) {
                    return true;
                }
            }
        }
        return false;
    };

    $scope.groupCheck = function(newContent) {
        return !$scope.isGroupSetUseful(newContent);
    };

    $scope.supportsImagePreprocessing = function(){
        return $scope.imageHandlingModes.length > 0 ;
    }

    $scope.imputeWithConstant = function() {
        var options = {type: 'text'};
        if ($scope.selection.selectedObjects.some(function(f) { return f.role !== 'TARGET' && f.type === 'NUMERIC'; })) {
            options.type = 'number';    // can only set a numeric constant
        }
        Dialogs.prompt($scope, "Impute with constant " + options.type, "Imputed value", "", options)
            .then(function(value) {
                $scope.groupSet({
                    missing_handling: 'IMPUTE',
                    missing_impute_with: 'CONSTANT',
                    impute_constant_value: options.type === 'number' ? parseFloat(value) : value
                });
            });
    };

    $scope.sendToDeepLearningInput = function() {
        var inputs = $scope.mlTaskDesign.modeling.keras.kerasInputs
                           .filter(function(input) {
                                return !PMLSettings.isSpecialInput(input, $scope.mlTaskDesign.preprocessing.per_feature);
                            }).map(function(input) {
                                return {title: input};
                           });
        Dialogs.select($scope, 'Send to Deep Learning input', 'Please select the input', inputs, inputs[0]).then(function(selectedInput) {
                $scope.selection.selectedObjects.forEach(function(featParams) {
                    if (featParams.role == "TARGET") {
                        return;
                    }
                    if (!PMLSettings.isSpecialFeature(featParams)) {
                        featParams.sendToInput = selectedInput.title;
                    }
                });
            });
    }

    // ONLY NEEDED FOR KERAS BACKEND

    // Goal is to detect when change in Feature settings requires to
    // automatically create/delete Deep Learning Input, for example when
    // switching to Text > Custom preprocessing

    function getNewInputName(featureName, currentInputs) {
        var newInputName = featureName + "_preprocessed";

        if (currentInputs.indexOf(newInputName) == -1) {
            return newInputName;
        } else {
            var i = 1;
            while (true) {
                var suffix = "_" + i;
                if (currentInputs.indexOf(newInputName + suffix) == -1) {
                    return newInputName + suffix
                }
                i += 1;
            }
        }
    }

    function handleSwitchingToSpecialFeature(nv, ov) {
        // Only want to catch when the same feature changes and:
        //   - becomes special or is not anymore
        //   - is special and its role changes (rejected or accepted)

        // First verify that this is the same feature
        if (nv == ov || !nv || !ov || nv._name !== ov._name) {
            return;
        }

        const nvSpecial = PMLSettings.isSpecialFeature(nv);
        const ovSpecial = PMLSettings.isSpecialFeature(ov);

        // Then treat switching INPUT/REJECT case
        if (nv.role !== ov.role && nvSpecial && ovSpecial) {

            // Must create new special input if feature was rejected
            if (nv.role === "INPUT" && ov.role === "REJECT") {
                createNewSpecialInputAndAssignToFeature(nv);
            }
            // Must delete special input if feature was rejected
            if (nv.role === "REJECT" && ov.role === "INPUT") {
                deleteSpecialInputAndSendFeatureToMain(nv)
            }

            return;
        }

        // Finally treat case where feature becomes/is not anymore special
        // Discard cases when "specialty" of feature does not change
        if (nvSpecial === ovSpecial ) {
            return;
        }

        if (nvSpecial) {
            // Must create new input for the special feature
            createNewSpecialInputAndAssignToFeature(nv);
        } else {
            // Must delete Input of special feature and put it in main
            deleteSpecialInputAndSendFeatureToMain(nv);
        }
    }

    function createNewSpecialInputAndAssignToFeature(feature) {
        // Must create new input for the special feature
        var newInputName = getNewInputName(feature._name, $scope.mlTaskDesign.modeling.keras.kerasInputs);

        $scope.mlTaskDesign.modeling.keras.kerasInputs.push(newInputName);
        feature.sendToInput = newInputName;
    }

    function deleteSpecialInputAndSendFeatureToMain(feature) {
        // Must delete Input of special feature and put it in main
        var inputTodelete = feature.sendToInput;

        var inputIndex = $scope.mlTaskDesign.modeling.keras.kerasInputs.indexOf(inputTodelete);
        $scope.mlTaskDesign.modeling.keras.kerasInputs.splice(inputIndex, 1);

        feature.sendToInput = "main";
    }

    function getCustomImageHandlingCode() {
        const importPrefix = VisualMlCodeEnvCompatibility.isEnvAtLeastTensorflow2_2($scope.mlTaskDesign, $scope.codeEnvsCompat) ? "from tensorflow.keras." : "from keras."
        const prepImgCode = importPrefix + "applications.imagenet_utils import preprocess_input\n" +
                            "from dataiku.doctor.deep_learning.keras_utils import load_img\n\n" +
                            "resized_width = 197\n" +
                            "resized_height = 197\n\n" +
                            "# Must return a numpy ndarray representing the image.\n" +
                            "#  - image_file is a file like object\n" +
                            "def preprocess_image(image_file):\n" +
                            "    # This will give you an input shape of (resized_height, resized_width, channels_number)\n"+
                            "    # resized_dims - a tuple (width, height)\n"+
                            "    # channels     - 'L', 'RGB' or 'CMYK'\n"+
                            "    # data_format  - 'channels_last' or 'channels_first'\n"+
                            "    array = load_img(image_file, resized_dims=(resized_width, resized_height), channels='RGB', data_format='channels_last')\n"+
                            "    # Define the actual preprocessing here, for example:\n\n" +
                            "    array = preprocess_input(array, mode='tf')\n"+
                            "    return array\n";
        return prepImgCode;
    }

    $scope.isSpecialFeature = function() {
        var featureData = $scope.selection.selectedObject;
        return PMLSettings.isSpecialFeature(featureData);
    };

    $scope.isNotSpecialInputOrIsSpecialSelection = function(input) {
        var featureData = $scope.selection.selectedObject;
        return PMLSettings.isSpecialFeature(featureData) || !PMLSettings.isSpecialInput(input, $scope.mlTaskDesign.preprocessing.per_feature);
    };

    $scope.setSubsampleFit = function() {
        let deferred = $q.defer();
        let newScope = $scope.$new();

        newScope.uiState = {
            preprocessingFitSampleRatio: $scope.mlTaskDesign.preprocessing.preprocessingFitSampleRatio,
            preprocessingFitSampleSeed: $scope.mlTaskDesign.preprocessing.preprocessingFitSampleSeed
        };

        CreateModalFromTemplate("templates/analysis/prediction/set-subsample-fit-modal.html", 
            newScope,
            null,
            function(scope) {

                scope.acceptDeferred = deferred;

                scope.validate = function () {
                    scope.acceptDeferred.resolve(scope.uiState);
                    scope.dismiss();
                };

                scope.showHideMore = function() {
                    scope.uiState.showMore = !scope.uiState.showMore;
                };

        });
        deferred.promise.then(function(data) {
            $scope.mlTaskDesign.preprocessing.preprocessingFitSampleRatio = data.preprocessingFitSampleRatio;
            $scope.mlTaskDesign.preprocessing.preprocessingFitSampleSeed = data.preprocessingFitSampleSeed;
        });
    }

    $scope.togglePeriod = function(period) {
        const feature = $scope.selection.selectedObject;
        const idx = feature.datetime_cyclical_periods.indexOf(period);
        if (idx > -1) {
            feature.datetime_cyclical_periods.splice(idx, 1);
        } else {
            feature.datetime_cyclical_periods.push(period);
        }
    };
    $scope.selection = {orderQuery: 'datasetColumnId'};

    Object.defineProperty($scope, 'roleIsSelected', {
        get: function() {
            return $scope.feature.role === 'INPUT' || $scope.feature.role === 'INPUT_PAST_ONLY';
        },
        set: function(value) {
            $scope.feature.role = value ? 'INPUT' : 'REJECT';
        }
    });

    $scope.setFeatureRole = function(featureRole) {
        const feature = $scope.selection.filteredObjects.filter(o => o._name === featureRole._name)[0];
        let state;
        if (['INPUT', 'INPUT_PAST_ONLY'].includes(featureRole.role)) {
            state = 'REJECT';
        } else {
            state = 'INPUT';
        }
        if (feature) {
            feature.role = state;
        }
    };


    $scope.hasShiftFromHorizon = function(feature) {
        return $scope.mlTaskDesign.preprocessing.feature_generation.shifts[feature._name] !== undefined
            && $scope.mlTaskDesign.preprocessing.feature_generation.shifts[feature._name].from_horizon
            && $scope.mlTaskDesign.preprocessing.feature_generation.shifts[feature._name].from_horizon.length > 0;
    };

    $scope.handleChangeInputMode = function (feature, newRole) {
        if (feature.role === newRole) {
            return;
        }
        feature.role = newRole;
        if (feature.role === "REJECT") {
            $scope.selection.selectedObject = undefined;
            $scope.selection.selectedObjects = [];
            return;
        } else {
            $scope.selection.selectedObject = feature;
            $scope.selection.selectedObjects = [feature];
        }

        const oldFeature = angular.copy(feature)

        $scope.fixupFeatureConfiguration(feature, oldFeature);
        if ($scope.hasShiftFromHorizon(feature)) {
            CreateModalFromTemplate("/templates/analysis/mlcommon/settings/time-series-reset-shift-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
                newScope.feature = feature;
            });
        }
    };

    $scope.handleMultipleInputChange = function (features, newRole) {

        if ($scope.mlTaskDesign.predictionType !== 'TIMESERIES_FORECAST') {
            $scope.groupSet({role:newRole})
            return;
        }

        let shouldOpenModal = false;
        for (let feature of features.filter(feature => feature.role !== newRole)) {
            const oldFeature = angular.copy(feature)
            feature.role = newRole;
            $scope.fixupFeatureConfiguration(feature, oldFeature);
            if ($scope.hasShiftFromHorizon(feature)) {
                shouldOpenModal = true;
            }
        }
        if (shouldOpenModal) {
            CreateModalFromTemplate("/templates/analysis/mlcommon/settings/time-series-reset-shift-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
                newScope.features = features;
            });
        }
    };

    $scope.handleMultipleTypeChange = function (features, targetType) {

        if ($scope.mlTaskDesign.predictionType !== 'TIMESERIES_FORECAST') {
            $scope.groupSet({type:targetType})
            return;
        }

        let shouldOpenModal = false;
        for (let feature of features.filter(feature => feature.type !== targetType)) {
            feature.type = targetType;
            if ($scope.hasShiftFromHorizon(feature)) {
                shouldOpenModal = true;
            }
        }
        if (shouldOpenModal) {
            CreateModalFromTemplate("/templates/analysis/mlcommon/settings/time-series-reset-shift-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
                newScope.features = features;
            });
        }
    };

});

app.controller("MLTaskOneFeatureController", function($scope, PMLSettings, $stateParams, $filter, DataikuAPI, VisualMlCodeEnvCompatibility, Dialogs) {

    $scope.onVariableTypeChange = function() {
        const featureData = $scope.selection.selectedObject;
        if (featureData.type === 'VECTOR') {
            // Setting value of Vector Handling manually for first selection of Vector as type
            // because vector type is never guessed by back-end so never set up automatically
            if (featureData.vector_handling === undefined) {
                featureData.vector_handling = "UNFOLD";
                featureData.missing_impute_with = 'MODE';
            }
        }

        // update available options list for missing values dropdown depending on new feature type & update the selected one only if needed:
        $scope.setMissingValuesModes(featureData);
        if (!$scope.missingValuesModes.some(mode => mode[0] === featureData.missing_handling)) {
            featureData.missing_handling = $scope.missingValuesModes[0][0];
        }

        if (!$scope.featureSupportsAutoShifts(featureData)) {
            $scope.mlTaskDesign.preprocessing.feature_generation.shifts[featureData._name].from_horizon_mode = 'FIXED';
        }
    };

    $scope.onVariableRoleChange = function() {
        // eslint-disable-next-line no-console
        console.debug("onVariableRoleChange", $scope.feature);
    };

    $scope.$watch('uiState.managedFolderSmartId', function(newValue) {

        if($scope.mlTaskDesign.managedFolderSmartId !== newValue) {
            if (!$scope.mlTaskDesign.managedFolderSmartId){ 
                onChangeManagedFolder(); //First init no confirm
            }
            else{
                const featuresUsingManagedFolder = getFeaturesUsingManagedFolder();
                // current feature may or may not be included in featuresUsingManagedFolder depending on whether settings are saved or not.
                // since we only need to know how many features "beside" current one are using the managedfolder, we remove it from the above list
                const currentFeatureName = $scope.selection.selectedObject._name
                if(featuresUsingManagedFolder.includes(currentFeatureName)){
                    featuresUsingManagedFolder.splice(featuresUsingManagedFolder.indexOf(currentFeatureName), 1);
                }
                if(featuresUsingManagedFolder.length > 0){
                    Dialogs.confirm(
                        $scope, "Changing the image location",
                        `Image location folder is currently used for the following features preprocessings: ${featuresUsingManagedFolder}.
                         Changing the image location will update the current feature preprocessing as well as those of the above features.`
                    ).then(onChangeManagedFolder, onAbortChangeManagedFolder)
                }
                else{
                    onChangeManagedFolder();
                }
            }
        }
    }, true);

    function onChangeManagedFolder() {
        $scope.mlTaskDesign.managedFolderSmartId = $scope.uiState.managedFolderSmartId;
    }
    function onAbortChangeManagedFolder() {
        $scope.uiState.managedFolderSmartId = $scope.mlTaskDesign.managedFolderSmartId;
    }
    function getFeaturesUsingManagedFolder() {
        // return names of features having an active preprocessing using images (Keras excluded)

        if ($scope.isMLBackendType('KERAS')){ 
            return []; // Keras use different managedFolder per features
        } 
        var featuresUsingManagedFolder = [];
        for (const feature of $scope.selection.allObjects) {
            if (feature.role == "INPUT" && feature.type == "IMAGE") {
                featuresUsingManagedFolder.push(feature._name);
            }
        }
        return featuresUsingManagedFolder;
    }

    // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
    $scope.puppeteerHook_elementContentLoaded = true;

    $scope.$watch('selection.selectedObject', function(newValue, oldValue) {
        // Only allow preprocessing of not rejected input features
        if (newValue) {
            $scope.canPreprocess = newValue.role != 'TARGET' && newValue.role != 'REJECT' && newValue.role != 'WEIGHT' && newValue.role != 'TREATMENT';
        }
    }, true);

    let hasFetchedImageEmbeddingModelOptions = false;
    function setImageEmbeddingModelsOptions() {
        if (hasFetchedImageEmbeddingModelOptions) {
            return;
        }

        hasFetchedImageEmbeddingModelOptions = true;
        DataikuAPI.pretrainedModels.getImageEmbeddingModels().success((modelList) => {
            modelList.sort((a, b)  => a.modelFriendlyName.localeCompare(b.modelFriendlyName, 'en', { sensitivity: 'base' }));
            $scope.imageEmbeddingModelsOptions = modelList;
        }).error(setErrorInScope.bind($scope))
    };

    function sortSentenceEmbeddingModels(sentenceEmbeddingModels) {
        sentenceEmbeddingModels.sort((a, b)  => a.modelFriendlyName.localeCompare(b.modelFriendlyName, 'en', { sensitivity: 'base' }));
        return sentenceEmbeddingModels;
    }

    $scope.sentenceEmbeddingModelsOptions = [];
    $scope.sentenceEmbeddingModelsList = [];
    $scope.$watchCollection("scope.mlTaskDesign.envSelection", () => {
        // We need to refresh the pretrained model list when the env selection changes because of the models in the code env resources
        DataikuAPI.pretrainedModels.getSentenceEmbeddingModels($scope.mlTaskDesign.envSelection, $stateParams.projectKey).success((sentenceEmbeddingModels) => {
                $scope.sentenceEmbeddingModelsList = sentenceEmbeddingModels;
                $scope.sentenceEmbeddingModelsOptions = sortSentenceEmbeddingModels($scope.sentenceEmbeddingModelsList);
            }).error(setErrorInScope.bind($scope));
        setImageEmbeddingModelsOptions();
    });
    $scope.getMaxTokensLimit = function() {
        if ($scope.selection.selectedObject == null) {
            return null;
        }

        // Do not limit the max sequence length if no model / an invalid model (e.g. cached from previous env) / a model without known max sequence length is selected (in the latter case the maxTokensLimit attribute will be null)
        const selectedModel = $scope.sentenceEmbeddingModelsList.find(m => m.modelRefId == $scope.selection.selectedObject.sentenceEmbeddingModel);
        if (selectedModel) {
            return selectedModel.maxTokensLimit;
        }
        return null;
    }
    $scope.getTextOverflowWarning = function() {
        if ($scope.selection.selectedObject == null) { return null; }
        const selectedModel = $scope.sentenceEmbeddingModelsList.find(m => m.modelRefId == $scope.selection.selectedObject.sentenceEmbeddingModel);
        if (!selectedModel) {return null; }

        const tokensLimit = $scope.isCodeEnvResourceModelSelected()? $scope.selection.selectedObject.maxSequenceLength : selectedModel.maxTokensLimit;
        const overflowBehavior = selectedModel.modelType == "HUGGINGFACE_TRANSFORMER_LOCAL" ? 'be truncated.': 'fail the training.';
        if (tokensLimit == undefined || selectedModel.maxTokensLimit == undefined) { // custom HF models doesn't always have a token limit in their metadata but it'll be retrieved with maxSequenceLength field when loading the model
            return "Texts longer than 'model.max_seq_length' tokens will " + overflowBehavior;
        }

        return "The selected model supports up to " + selectedModel.maxTokensLimit + " tokens per row, texts longer than "+ tokensLimit + " tokens will " + overflowBehavior;
        
    }

    $scope.checkSentenceEmbeddingModelCompatible = function() {
        $scope.selectedModelType = "";
        // Define as incompatible if no model selected
        if ($scope.selection.selectedObject == null) {
            return false;
        }

        const selectedModel = $scope.sentenceEmbeddingModelsList.find(m => m.modelRefId == $scope.selection.selectedObject.sentenceEmbeddingModel);
        if (selectedModel) {
            return selectedModel.compat;
        }

        return false; // selected model cannot be found anymore (code env or connection changed in between)
    }

    $scope.isCodeEnvResourceModelSelected = function(){
        if( !$scope.selection.selectedObject || !$scope.selection.selectedObject.sentenceEmbeddingModel){
            return false; // no selected model
        }
        return !$scope.selection.selectedObject.isStructuredRef; 
    };

    $scope.enrichSentenceEmbPreprocessingParams = function (selectedModelId) {
        const selectedModel = $scope.sentenceEmbeddingModelsList.find(m => m.modelRefId == selectedModelId);
        if (selectedModel) {
            $scope.selection.selectedObject.isStructuredRef = selectedModel.isStructuredRef;
            $scope.selection.selectedObject.embeddingSize = selectedModel.embeddingSize;
        }
    };

    function getCurrentEnvCompatibility(){
        if (!$scope.mlTaskDesign || !$scope.mlTaskDesign.envSelection) {
            return null;
        }
        return VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
    }
    $scope.isCodeEnvCompatibleWithSentenceEmbedding = function() {
        const envCompat = getCurrentEnvCompatibility();
        return envCompat && envCompat.sentenceEmbedding && envCompat.sentenceEmbedding.compatible;
    };

});

app.component("pretrainedModelSelector", {
    bindings: {
        "modelRefId": "=", // the chosen model ref id
        "onChange": "<", // (optional) callback when selection changed, takes the selected model id as a parameter
        "modelOptions": "<", // list of objects representing the model optional choices -> [ { modelFriendlyName, 'model 1 name', modelRefId: 'model 1 id', modelType: 'HuggingFace local' } , ... ]
        "embeddingType": "<" // (optional) used to diplay which embedding type will be extracted (at the moment 'text' and 'image')
    },
    template: `
<div class="control-group">
    <label class="control-label">Pre-trained model</label>
    <div class="controls">
        <basic-select
            items="$ctrl.modelOptions"
            bind-label="modelFriendlyName"
            bind-value="modelRefId"
            bind-annotation="type"
            bind-disabled="disabled"
            group-by-fn="$ctrl.niceModelType"
            searchable="true"
            actions-box="false"
            placeholder="{{ $ctrl.placeholder }}"
            invalidate-on-ghosts="true"
            ghost-items-tooltip="The selected model is not available anymore."
            multiple="false"
            ng-model="$ctrl.modelRefId"
            ng-change="$ctrl.onChange($ctrl.modelRefId)"
        />
        <div class="help-inline" ng-if="$ctrl.modelLibrary == 'SentenceTransformers'">
            Model used to extract the text embeddings.
        </div>
        <div class="help-inline" ng-if="$ctrl.embeddingType">
            Model used to extract the {{ $ctrl.embeddingType }} embeddings.
        </div>
        <div class="help-inline" ng-if="$ctrl.embeddingType == null">
            Model used to extract the embeddings.
        </div>
    </div>
</div>
`,
    controller: function($filter) {
        const ctrl = this;
        ctrl.$onInit = () => {
            ctrl.modelOptions = ctrl.modelOptions == null ? [] : ctrl.modelOptions;
            ctrl.placeholder = 'Nothing selected';
            ctrl.niceModelType = (modelOption) => {
                return $filter('niceLLMType')(modelOption.modelType);
            };
        };

        ctrl.$onChanges = () => {
            if (ctrl.modelOptions) { // might not be retrieved yet, this function will be called again once it's set
                if (ctrl.modelOptions.length == 0) {
                    ctrl.placeholder = 'No available models.'
                }
            }
        }
    }
});

/**
 * Simple service to retrieve and cache the distribution of each feature.
 * Offers 2 utilities:
 *   * `fetchColumnAnalysisIfNeeded` which return a promise that resolves with the result.
 *      Should be used for initial fetching or when needing to know when the result is here and do something with it
 *   * `analyses`, a {column -> ColumnAnalysis} object which contains the currently cached analyses.
 *      MUST be readonly and used when having the result is not a necessity, i.e. in the HTML templates.
 */
app.service("ColumnAnalysesService", function($stateParams, DataikuAPI) {
    function init(analysisCoreParams, onError) {
        const analyses = {};
        const alphanumMaxResults = 50,
              fullSamplePartitionId = null,
              withFullSampleStatistics = null,
              forceTimePeriodAnalysis = null;

        function fetchColumnAnalysisIfNeeded(column) {
            return new Promise((resolve, _) => {
                if (analyses[column]) {
                    resolve(analyses[column])
                } else {
                    DataikuAPI.shakers.detailedColumnAnalysis(
                        $stateParams.projectKey, analysisCoreParams.projectKey, analysisCoreParams.inputDatasetSmartName,
                        analysisCoreParams.script, null, column, alphanumMaxResults, fullSamplePartitionId,
                        withFullSampleStatistics, forceTimePeriodAnalysis
                        ).then((columnAnalysisResult) => {
                            analyses[column] = columnAnalysisResult.data;
                            resolve(analyses[column]);
                        })
                        .catch(onError);
                }
            })
        }

        return {
            fetchColumnAnalysisIfNeeded,
            analyses
        }
    }
    return {
        init
    }
});

app.component("tooManyDroppedRowsWarning", {
    bindings: {
        columnAnalysis: "<"
    },
    template: `
        <p ng-if="$ctrl.displayWarning" class="mleftright8 mbot8 alert alert-warning">
            Dropping rows with missing values could remove around {{$ctrl.columnAnalysis.alphanumFacet.missing | smartPercentage:1}} of the input data (based on design sample).
        </p>
    `,
    controller: function() {
        const ctrl = this;
        const DROPPED_ROW_WARNING_THRESHOLD = 0.5; // to be kept consistent with DROPPED_ROWS_THRESHOLD in modeling_parameter.py

        ctrl.$onChanges = () => {
            ctrl.displayWarning = ctrl.columnAnalysis
                                  && ctrl.columnAnalysis.alphanumFacet
                                  && ctrl.columnAnalysis.alphanumFacet.missing > DROPPED_ROW_WARNING_THRESHOLD;
        };
    }
});

    app.service("TimeseriesFeatureGenerationService", function() {
        this.cleanUpSettings = function(mlTaskDesign) {
            /*
            Cleans the timeseries shift values of an ML Task for time-aware feature engineering.
            Beware, this function is used in the wide prediction context, not only for timeseries ML tasks.
            */
            if (mlTaskDesign.predictionType !== "TIMESERIES_FORECAST") {
                return;
            }

            for (var featureName in mlTaskDesign.preprocessing.feature_generation.shifts) {
                const toCheckFromForecast = mlTaskDesign.preprocessing.feature_generation.shifts[featureName].from_forecast;
                if (typeof toCheckFromForecast === "string") {
                    mlTaskDesign.preprocessing.feature_generation.shifts[featureName].from_forecast = [...new Set(Array.from(toCheckFromForecast.split(",")).map(x => parseInt(x)).filter(x => !isNaN(x)))];
                }
                const toCheckFromHorizon = mlTaskDesign.preprocessing.feature_generation.shifts[featureName].from_horizon;
                if (typeof toCheckFromHorizon === "string") {
                    mlTaskDesign.preprocessing.feature_generation.shifts[featureName].from_horizon = [...new Set(Array.from(toCheckFromHorizon.split(",")).map(x => parseInt(x)).filter(x => !isNaN(x)))];
                }
            }
        };
    });

    app.controller('FeatureGenerationController', function($scope, CreateModalFromTemplate, TimeseriesFeatureGenerationService, TimeseriesForecastingUtils) {

        $scope.selectedFeaturesContainsPastCovariate = function () {
            return $scope.selection.selectedObjects.some(feature => $scope.isPastOnlyFeature(feature._name));
        };
        $scope.allSelectedFeaturesSupportAutoShift = function () {
            return $scope.selection.selectedObjects.some(feature => !$scope.featureSupportsAutoShifts(feature));
        };

        $scope.setFeatureShifts = function(horizonShifts, forecastShifts, enableHorizonShiftsAuto) {
            $scope.selection.selectedObjects.forEach(function(feature) {
                if (!$scope.mlTaskDesign.preprocessing.feature_generation.shifts[feature._name]) {
                    $scope.mlTaskDesign.preprocessing.feature_generation.shifts[feature._name] = {};
                }
                $scope.mlTaskDesign.preprocessing.feature_generation.shifts[feature._name].from_forecast = forecastShifts || "";
                if (enableHorizonShiftsAuto) {
                    if ($scope.featureSupportsAutoShifts(feature)) {
                        $scope.mlTaskDesign.preprocessing.feature_generation.shifts[feature._name].from_horizon_mode = 'AUTO';
                    } else {
                        // Do not update horizon shifts for feature not supporting auto
                    }
                } else {
                    $scope.mlTaskDesign.preprocessing.feature_generation.shifts[feature._name].from_horizon = horizonShifts || "";
                    $scope.mlTaskDesign.preprocessing.feature_generation.shifts[feature._name].from_horizon_mode = 'FIXED';
                }

            });
            $scope.validateShifts();
        };

        $scope.openShiftsModal = function() {
            CreateModalFromTemplate("/templates/analysis/mlcommon/settings/time-series-edit-feature-shift-modal.html", $scope, null, function(newScope) {
                newScope.save = () => {
                    $scope.setFeatureShifts(newScope.horizonShifts, newScope.forecastShifts, newScope.enableHorizonShiftsAuto);
                    newScope.dismiss();
                }
            }, false, false, true)
        };

        $scope.setFeatureAutoShiftsParams = function (maxSelectedHorizonShifts, minHorizonShiftPastOnly, maxHorizonShiftPastOnly, minHorizonShiftKnownInAdvance, maxHorizonShiftKnownInAdvance) {
            if (!$scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params) {
                $scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params = {};
            }

            $scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params.max_selected_horizon_shifts = maxSelectedHorizonShifts || 1;
            $scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params.min_horizon_shift_past_only = minHorizonShiftPastOnly;
            $scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params.max_horizon_shift_past_only = maxHorizonShiftPastOnly;
            $scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params.min_horizon_shift_known_in_advance = minHorizonShiftKnownInAdvance;
            $scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params.max_horizon_shift_known_in_advance = maxHorizonShiftKnownInAdvance;
        };

        $scope.validateAutoShiftsParams = function () {
            let autoShiftsParams = $scope.mlTaskDesign.preprocessing?.feature_generation?.auto_shifts_params;
            $scope.hasAutoShiftsParamsValidationErrors = autoShiftsParams === undefined
                || typeof autoShiftsParams.max_selected_horizon_shifts !== 'number' || autoShiftsParams.max_selected_horizon_shifts < 1
                || $scope.validatePastOnlyAutoHorizonShiftRange(autoShiftsParams.min_horizon_shift_past_only, autoShiftsParams.max_horizon_shift_past_only) !== ""
                || $scope.validateKnownInAdvanceAutoHorizonShiftRange(autoShiftsParams.min_horizon_shift_known_in_advance, autoShiftsParams.max_horizon_shift_known_in_advance) !== ""
        };

        $scope.openAutoShiftsModal = function () {
            CreateModalFromTemplate("/templates/analysis/mlcommon/settings/time-series-edit-auto-shift-modal.html", $scope, null, function (newScope) {

                if ($scope.mlTaskDesign.preprocessing.feature_generation?.auto_shifts_params) {
                    newScope.maxSelectedHorizonShifts = $scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params.max_selected_horizon_shifts;
                    newScope.minHorizonShiftPastOnly = $scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params.min_horizon_shift_past_only;
                    newScope.maxHorizonShiftPastOnly = $scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params.max_horizon_shift_past_only;
                    newScope.minHorizonShiftKnownInAdvance = $scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params.min_horizon_shift_known_in_advance;
                    newScope.maxHorizonShiftKnownInAdvance = $scope.mlTaskDesign.preprocessing.feature_generation.auto_shifts_params.max_horizon_shift_known_in_advance;
                } else {
                    newScope.maxSelectedHorizonShifts = 3 ;
                    newScope.minHorizonShiftPastOnly = -$scope.mlTaskDesign.predictionLength - 35;
                    newScope.maxHorizonShiftPastOnly = -$scope.mlTaskDesign.predictionLength;
                    newScope.minHorizonShiftKnownInAdvance = -35;
                    newScope.maxHorizonShiftKnownInAdvance = 0;
                }

                newScope.inputPastOnlyHorizonShiftsRangeValidationError = newScope.validatePastOnlyAutoHorizonShiftRange(newScope.minHorizonShiftPastOnly, newScope.maxHorizonShiftPastOnly);
                newScope.inputHorizonShiftsRangeValidationError = newScope.validateKnownInAdvanceAutoHorizonShiftRange(newScope.minHorizonShiftKnownInAdvance, newScope.maxHorizonShiftKnownInAdvance);

                newScope.save = () => {
                    $scope.setFeatureAutoShiftsParams(newScope.maxSelectedHorizonShifts, newScope.minHorizonShiftPastOnly, newScope.maxHorizonShiftPastOnly, newScope.minHorizonShiftKnownInAdvance, newScope.maxHorizonShiftKnownInAdvance);
                    $scope.validateAutoShiftsParams();
                    newScope.dismiss();
                }
            }, false, false, true)
        };

        $scope.setWindowOperations = function(numOperations, catOperations, windowIndex) {
            let selectedAndValidFeatures = $scope.uiState.windows[windowIndex].operations_list
                .filter(el => {
                    return el.$selected && $scope.uiState.windowableFeatures.map(feature => feature._name).includes(el[0]);
                })
            for (let el of selectedAndValidFeatures) {
                if (el[2].type === "NUMERIC") {
                    el[1] = angular.copy(numOperations);
                } else if (el[2].type === "CATEGORY") {
                    el[1] = angular.copy(catOperations);
                }
            }
            $scope.validateWindows();
        }


        $scope.openWindowsMassActionModal = function(windowIndex, window) {
            CreateModalFromTemplate("/templates/analysis/mlcommon/settings/time-series-windows-mass-action-modal.html", $scope, null, function(newScope) {
                let per_feature = $scope.mlTaskDesign.preprocessing.per_feature;
                let selectedFeaturesNumerical = window.operations_list.filter(x => per_feature[x[0]].type === "NUMERIC" && x.$selected && TimeseriesForecastingUtils.isWindowCompatible(per_feature[x[0]]));
                let someDisabledOperationsNumerical = new Set(selectedFeaturesNumerical.map(x => x[1]).flatMap(x => x.filter(y => !y.enabled).map(y => y.operation)));
                let allEnabledOperationsNumerical = (new Set(["MEAN", "MEDIAN", "STD", "MIN", "MAX"])).difference(someDisabledOperationsNumerical);
                let partiallyEnabledOperationsNumerical = (new Set(selectedFeaturesNumerical.map(x => x[1]).flatMap(x => x.filter(y => y.enabled).map(y => y.operation)))).difference(allEnabledOperationsNumerical);
                newScope.operationsNumerical = [
                    {operation: "MEAN", enabled: allEnabledOperationsNumerical.has("MEAN"), partiallyEnabled: partiallyEnabledOperationsNumerical.has("MEAN")},
                    {operation: "MEDIAN", enabled: allEnabledOperationsNumerical.has("MEDIAN"), partiallyEnabled: partiallyEnabledOperationsNumerical.has("MEDIAN")},
                    {operation: "STD", enabled: allEnabledOperationsNumerical.has("STD"), partiallyEnabled: partiallyEnabledOperationsNumerical.has("STD")},
                    {operation: "MIN", enabled: allEnabledOperationsNumerical.has("MIN"), partiallyEnabled: partiallyEnabledOperationsNumerical.has("MIN")},
                    {operation: "MAX", enabled: allEnabledOperationsNumerical.has("MAX"), partiallyEnabled: partiallyEnabledOperationsNumerical.has("MAX")}
                ];
                let selectedFeaturesCategorical = window.operations_list.filter(x => per_feature[x[0]].type === "CATEGORY" && x.$selected && TimeseriesForecastingUtils.isWindowCompatible(per_feature[x[0]]));
                let someDisabledOperationsCategorical = new Set(selectedFeaturesCategorical.map(x => x[1]).flatMap(x => x.filter(y => !y.enabled).map(y => y.operation)));
                let allEnabledOperationsCategorical = (new Set(["FREQUENCY"])).difference(someDisabledOperationsCategorical);
                let partiallyEnabledOperationsCategorical = (new Set(selectedFeaturesCategorical.map(x => x[1]).flatMap(x => x.filter(y => y.enabled).map(y => y.operation)))).difference(allEnabledOperationsCategorical);
                newScope.operationsCategorical = [
                    {operation: "FREQUENCY", enabled: allEnabledOperationsCategorical.has("FREQUENCY"), partiallyEnabled: partiallyEnabledOperationsCategorical.has("FREQUENCY")},
                ];
                newScope.window = window;
                newScope.atLeastOneNumericalSelected = function () {
                    return selectedFeaturesNumerical.length > 0;
                }
                newScope.atLeastOneCategoricalSelected = function () {
                    return selectedFeaturesCategorical.length > 0;
                }
                newScope.save = () => {
                    $scope.setWindowOperations(newScope.operationsNumerical, newScope.operationsCategorical, windowIndex);
                    newScope.dismiss();
                }
            }, false, false, true)
        };

        $scope.openWindowsSettingsModal = function() {
            CreateModalFromTemplate("/templates/analysis/mlcommon/settings/time-series-windows-settings-modal.html", $scope, null, function(newScope) {
                newScope.windows_rescale_numericals = $scope.mlTaskDesign.preprocessing.feature_generation.windows_rescale_numericals;
                newScope.windows_max_categories = $scope.mlTaskDesign.preprocessing.feature_generation.windows_max_categories;

                newScope.save = () => {
                    $scope.mlTaskDesign.preprocessing.feature_generation.windows_rescale_numericals = newScope.windows_rescale_numericals;
                    $scope.mlTaskDesign.preprocessing.feature_generation.windows_max_categories = newScope.windows_max_categories;
                    newScope.dismiss();
                }
            }, false, false, true)
        };

        $scope.hasAnyEnabledOperation = function(window) {
            return window.operations_list.some(
                operations => operations[1].some(
                    operation => operation.enabled
                )
            );
        }

        $scope.getTableSize = function(numberOfElements, rowHeight) {
            return Math.min(numberOfElements, 10) * rowHeight + 10;
        }

        $scope.$on('$destroy', function() {
            // clean settings when leaving the tab
            TimeseriesFeatureGenerationService.cleanUpSettings($scope.mlTaskDesign);
         })
    });

app.component("cyclicalEncodingWarning", {
    bindings: {
        datetimeCyclicalPeriods: '<',
        columnAnalysis: '<'
    },
    template: `
        <div ng-show="$ctrl.warningOnSelectedPeriods" class="doctor-explanation alert-warning mtop8 mright16">
            <span>{{ $ctrl.warningOnSelectedPeriods }}</span>
        </div>
    `,
    controller: function() {
        const $ctrl = this;

        let _prevData = null;
        $ctrl.$doCheck = () => {
            const newData = {
                datetimeCyclicalPeriods: angular.copy($ctrl.datetimeCyclicalPeriods),
                columnAnalysis: angular.copy($ctrl.columnAnalysis)
            }
            if (!angular.equals(_prevData, newData)) {
                _prevData = newData;
                updateWarning();
            }
        };

        function updateWarning() {
            if (!$ctrl.datetimeCyclicalPeriods.length) {
                $ctrl.warningOnSelectedPeriods = "At least one period should be selected to enable cyclical encoding.";
                return;
            }

            const numericalAnalysis = $ctrl.columnAnalysis && $ctrl.columnAnalysis.numericalAnalysis;

            if (!numericalAnalysis) {
                delete $ctrl.warningOnSelectedPeriods;
                return;
            }

            if (numericalAnalysis.relevantPeriods && !numericalAnalysis.relevantPeriods.length) {
                 $ctrl.warningOnSelectedPeriods = "According to the sample data on this feature, cyclical encoding"
                    + " does not seem to yield enough distinct values for any period.";
                return;
            }

            const irrelevantSelectedPeriods = $ctrl.datetimeCyclicalPeriods.filter(period => numericalAnalysis.irrelevantPeriods.includes(period));
            if (!irrelevantSelectedPeriods.length) {
                delete $ctrl.warningOnSelectedPeriods;
            } else {
                $ctrl.warningOnSelectedPeriods = "According to the sample data on this feature, some of the selected"
                    + " periods (" + irrelevantSelectedPeriods.map(period => period.toLowerCase()).join(", ")
                    + ") might not be relevant for the feature.";
            }
        }
    }
});
})();
