(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('LinesDrawer', LinesDrawer);

    // (!) This service previously was in static/dataiku/js/simple_report/curves/lines.js
    function LinesDrawer(Fn, LinesUtils, ChartDimension, ChartAxesUtils, ReferenceLines, ColumnAvailability, ChartCustomMeasures, ChartUsableColumns) {

        return function(g, chartDef, chartHandler, chartData, chartBase, linesData, facetIndex, redraw, isInteractive, iLabelsBoundingBoxes, getRefLineValuesOverride, refLinesContainer) {

            const xDimension = ChartDimension.getGenericDimension(chartDef);
            const emptyBinsMode = xDimension.numParams.emptyBinsMode;
            const xLabels = chartData.getAxisLabels('x');
            const rightYAxis = ChartAxesUtils.getRightYAxis(chartBase.yAxes);
            const leftYAxis = ChartAxesUtils.getLeftYAxis(chartBase.yAxes);
            let xAxis = chartBase.xAxis;

            chartBase.DOMUtils = chartBase.DOMUtils || {};
            chartBase.DOMUtils[facetIndex] = chartBase.DOMUtils[facetIndex] || {};

            const wrappers = LinesUtils.drawWrappers(chartDef, chartBase, linesData, g, isInteractive, redraw, 'wrapper');

            // During interaction, prevent re-drawing the points and remove them from the DOM for performances.
            if (!redraw) {
                chartBase.DOMUtils[facetIndex].points = LinesUtils.drawPoints(chartDef, chartBase, chartData, facetIndex, wrappers, xAxis, xLabels, leftYAxis, rightYAxis, xDimension, emptyBinsMode);
                LinesUtils.addTooltipAndHighlightAndContextualMenuHandlers(chartDef, chartHandler, chartBase, facetIndex, g, wrappers);
                chartBase.DOMUtils[facetIndex].labels = LinesUtils.drawLabels(chartDef, chartBase, chartData, chartHandler.getChartTheme(), facetIndex, wrappers, xAxis, xLabels, leftYAxis, rightYAxis, xDimension, emptyBinsMode, iLabelsBoundingBoxes);
            } else if (isInteractive && !chartBase.DOMUtils[facetIndex].pointsHaveBeenRemoved) {
                if(chartBase.DOMUtils[facetIndex].points){
                    chartBase.DOMUtils[facetIndex].points.remove();
                }
                if(chartBase.DOMUtils[facetIndex].labels){
                    chartBase.DOMUtils[facetIndex].labels.remove();
                }
                chartBase.DOMUtils[facetIndex].pointsHaveBeenRemoved = true;
            }

            const [lineGenerator, lineGs, lineDashGs] = LinesUtils.configureLines(chartDef, chartData, facetIndex, wrappers, chartBase.DOMUtils[facetIndex].lineGenerator, xAxis, leftYAxis, rightYAxis, xDimension, xLabels, emptyBinsMode);

            chartBase.DOMUtils[facetIndex].lineGenerator = lineGenerator;

            // Add thicker, invisible lines to catch mouseover event
            [lineGs, lineDashGs].forEach(lineGs => {
                let hiddenLines = lineGs.selectAll('path.masked');

                if (!redraw) {
                    hiddenLines = hiddenLines.data(function(d) {
                        return [d];
                    });
                    hiddenLines.enter()
                        .insert('path')
                        .attr('class', 'line masked')
                        .attr('fill', 'none')
                        .attr('stroke-width', '10')
                        .attr('stroke', 'transparent');
                    hiddenLines.exit().remove();
                }

                hiddenLines.attr('d', Fn.SELF);
            });

            LinesUtils.drawPaths(chartDef, chartBase, chartData, facetIndex, lineGs, xDimension, xLabels, xAxis, leftYAxis, rightYAxis, emptyBinsMode, redraw, !isInteractive, chartDef.strokeWidth, lineDashGs);

            const isPercentScaleOnYAxis = chartDef.genericMeasures.some(measure => measure.displayAxis === 'axis1' && ChartDimension.isPercentScale([measure]));
            const isPercentScaleOnY2Axis = chartDef.genericMeasures.some(measure => measure.displayAxis !== 'axis1' && ChartDimension.isPercentScale([measure]));
            const leftYFormattingOptions = leftYAxis && ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, leftYAxis.id);
            const rightYFormattingOptions = rightYAxis && ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, rightYAxis.id);

            const dataSpec = chartHandler.getDataSpec();
            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
            ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);

            const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, xAxis, undefined),
                referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures, true, getRefLineValuesOverride);

            const yAxes = [];
            if (leftYAxis) {
                yAxes.push({ ...leftYAxis, isPercentScale: isPercentScaleOnYAxis, formattingOptions: leftYFormattingOptions });
            }
            if (rightYAxis) {
                yAxes.push({ ...rightYAxis, isPercentScale: isPercentScaleOnY2Axis, formattingOptions: rightYFormattingOptions });
            }

            xAxis = { ...xAxis, formattingOptions: xAxis && chartDef.xAxisFormatting };

            ReferenceLines.drawReferenceLines(
                // For a mix chart, if we have lines, we will draw ref lines in the last line wrapper, if we have none - in the bar wrapper.
                wrappers[0].length ? d3.select(wrappers[0][wrappers[0].length-1]) : refLinesContainer,
                chartBase.vizWidth,
                chartBase.vizHeight,
                xAxis,
                yAxes,
                displayedReferenceLines,
                referenceLinesValues
            );

            if (!redraw) {
                const isInteractive = ChartDimension.isInteractiveChart(chartDef);
                const isCroppedChart = ChartAxesUtils.isCroppedChart(chartDef);
                /*
                 * Clip paths to prevent lines from overlapping axis:
                 * - during offline zoom
                 * - or when user chose a custom range which results in the line being cropped (out of visible range)
                 */
                (isInteractive || isCroppedChart) && LinesUtils.clipPaths(chartBase, chartDef, g, wrappers);
            }
        };
    }

})();
